<?php

namespace Limb_Chatbot\Includes\AI_Providers\Gemini\Utilities;

use Limb_Chatbot\Includes\AI_Providers\Gemini\Endpoints\Model_Endpoint;
use Limb_Chatbot\Includes\AI_Providers\Gemini\Gemini;
use Limb_Chatbot\Includes\Data_Objects\AI_Model;
use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Utilities\AI_Model_Utility as Global_Utility;

/**
 * Class AI_Model_Utility
 *
 * Utility class for handling AI Model operations for the Gemini provider.
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Gemini\Utilities
 * @since 1.0.0
 */
class AI_Model_Utility {

	/**
	 * Global AI model utility instance.
	 *
	 * @var Global_Utility
	 * @since 1.0.0
	 */
	public Global_Utility $global_utility;

	/**
	 * Model endpoint instance for API interactions.
	 *
	 * @var Model_Endpoint
	 * @since 1.0.0
	 */
	public Model_Endpoint $endpoint;

	/**
	 * The AI model instance.
	 *
	 * @var AI_Model|null
	 * @since 1.0.0
	 */
	public ?AI_Model $model;

	/**
	 * Config instance, nullable.
	 *
	 * @var Config|null
	 * @since 1.0.0
	 */
	public ?Config $config = null;

	/**
	 * Timeout in seconds for HTTP requests.
	 *
	 * @var int|null
	 * @since 1.0.0
	 */
	public ?int $timeout = 60;

	/**
	 * Constructor.
	 *
	 * @param Global_Utility $global_utility Global utility instance.
	 * @since 1.0.0
	 */
	public function __construct( Global_Utility $global_utility ) {
		$this->global_utility = $global_utility;
		if ( is_null( $this->global_utility->config ) ) {
			$this->config = Config::where( [ 'default' => true, 'related_to' => Gemini::class ] )->first();
		}
		$this->endpoint = new Model_Endpoint( $this );
		$this->model    = $this->global_utility->model;
	}

	/**
	 * Delete the AI model.
	 *
	 * @return bool True on success.
	 * @since 1.0.0
	 *
	 * @throws Exception If model is missing.
	 */
	public function delete(): bool {
		if ( is_null( $this->model ) ) {
			throw new Exception( Error_Codes::DATA_INVALID_AI_MODEL_ARGUMENT, __( 'Model is missing', 'limb-chatbot' ) );
		}

		return $this->endpoint->delete( $this->model );
	}

	/**
	 * List available AI models.
	 *
	 * @return array List of AI models.
	 * @throws Exception
	 * @since 1.0.0
	 */
	public function list() {
		return $this->endpoint->list();
	}

	/**
	 * Retrieve the AI model details.
	 *
	 * @return AI_Model The AI model.
	 *
	 * @throws Exception If model is missing.
	 * @since 1.0.0
	 */
	public function retrieve(): AI_Model {
		if ( is_null( $this->model ) ) {
			throw new Exception( Error_Codes::DATA_INVALID_AI_MODEL_ARGUMENT, __( 'Model is missing', 'limb-chatbot' ) );
		}

		return $this->endpoint->retrieve( $this->model );
	}

	/**
	 * Get request timeout.
	 *
	 * @return int|null Timeout in seconds.
	 * @since 1.0.0
	 */
	public function get_timeout(): ?int {
		return $this->timeout;
	}

	/**
	 * Get the config instance.
	 *
	 * @return Config|null
	 * @since 1.0.0
	 */
	public function get_config(): ?Config {
		return $this->config;
	}
}