<?php

namespace Limb_Chatbot\Includes\AI_Providers\Gemini;

use Limb_Chatbot\Includes\AI_Providers\Gemini\Services\AI_Model_Cost_Service;
use Limb_Chatbot\Includes\Config_Dependent_Interface;
use Limb_Chatbot\Includes\AI_Providers\AI_Provider;
use Limb_Chatbot\Includes\Data_Objects\AI_Model;
use Limb_Chatbot\Includes\Interfaces\File_Upload_Capable_Interface;

/**
 * Class Gemini
 *
 * AI Provider implementation for Google Gemini.
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Gemini
 * @since 1.0.0
 */
class Gemini extends AI_Provider implements Config_Dependent_Interface, File_Upload_Capable_Interface {

	/**
	 * Base URL for Gemini API.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	const API_BASE_URL = 'https://generativelanguage.googleapis.com/';

	/**
	 * Assistant role identifier.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	const ROLE_ASSISTANT = 'model';

	/**
	 * Unique ID for Gemini provider.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public static string $id = 'gemini';

	/**
	 * Human-readable name for the provider.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public static string $name = 'Gemini';

	/**
	 * Gemini constructor.
	 *
	 * Hooks into custom actions to handle model cost and usage tier checking.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'lbaic_before_limits_check_' . self::$id, array( AI_Model_Cost_Service::class, 'before_limits_check' ) );
		add_action( 'lbaic_before_update_usage_' . self::$id, array( AI_Model_Cost_Service::class, 'check_user_tier' ), 10, 3 );
	}

	/**
	 * Checks if the given MIME type is supported for file uploads.
	 *
	 * @param  string  $mime_type  The MIME type to check.
	 * @param  string|null  $purpose  The purpose of the file (optional).
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public function supports_mime_type( string $mime_type, ?string $purpose = null ): bool {
		$supported_mime_types = array(
			// Images
			'image/png',
			'image/jpeg',
			'image/webp',
			// Documents
			'application/pdf',
			'text/plain',
		);

		return in_array( $mime_type, $supported_mime_types, true );
	}

	public function get_default_embedding_model(){
		return AI_Model::find_by_name('gemini-embedding-001');
	}
}