<?php

namespace Limb_Chatbot\Includes\AI_Providers\Deep_Seek\Seeders;

use Exception;
use Limb_Chatbot\Includes\Data_Objects\AI_Model;
use Limb_Chatbot\Includes\AI_Providers\Deep_Seek\Deep_Seek;

/**
 * Class AI_Models_Seeder
 *
 * Seeder class responsible for seeding AI models specific to the Deep Seek AI provider.
 * Creates or updates AI models with predefined configurations and metadata.
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Deep_Seek\Seeders
 * @since 1.0.0
 */
class AI_Models_Seeder {

	/**
	 * Run the seeder to create or update AI models.
	 *
	 * Iterates over a predefined list of models, creates new entries if not present,
	 * or updates existing ones. Also updates associated metadata for each model.
	 *
	 * @return void
	 * @throws Exception
	 * @since 1.0.0
	 */
	public function run() {
		$models = array(
			// DeepSeek-V3.2 Chat Model (non-thinking mode).
			array(
				'label'              => 'DeepSeek-V3.2 Chat',
				'name'               => 'deepseek-chat',
				'ai_provider_id'     => Deep_Seek::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'input_token_limit'  => 131072, // 128K context length.
				'output_token_limit' => 8192,   // 8K max output.
				'modalities'         => array(
					'text'  => true,
					'image' => false,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => false,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => false,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true, // JSON mode support.
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
				),
				'metas'              => array(
					'input_token_cost'            => 0.00000028,   // $0.28/1M tokens (cache miss).
					'output_token_cost'           => 0.00000042,   // $0.42/1M tokens.
					'input_token_cost_cache_hit'  => 0.000000028,  // $0.028/1M tokens (cache hit).
					'input_token_cost_cache_miss' => 0.00000028,   // $0.28/1M tokens (cache miss).
				),
			),
		);

		foreach ( $models as $item ) {
			$metas = $item['metas'];
			unset( $item['metas'] );
			if ( AI_Model::find_by_name( $item['name'] ) ) {
				AI_Model::update( [ 'name' => $item['name'] ], $item );
				$model = AI_Model::where( [ 'name' => $item['name'] ] )->first();
			} else {
				$model = AI_Model::create( $item );
			}
			if ( $model instanceof AI_Model ) {
				foreach ( $metas as $meta_key => $meta ) {
					$model->update_meta( $meta_key, $meta );
				}
			}
		}
	}
}