<?php
/**
 * Model Endpoint for DeepSeek AI Provider.
 *
 * Handles model-related actions for DeepSeek, such as listing available models.
 * DeepSeek API endpoint: GET https://api.deepseek.com/models
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Deep_Seek\Endpoints
 * @since   1.0.0
 */

namespace Limb_Chatbot\Includes\AI_Providers\Deep_Seek\Endpoints;

use Limb_Chatbot\Includes\AI_Providers\Deep_Seek\Endpoints\Model\Model_Response_Handler;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;

/**
 * Class Model_Endpoint
 *
 * Handles model listing operations for the DeepSeek AI provider.
 * Uses the DeepSeek models API endpoint to retrieve available models.
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Deep_Seek\Endpoints
 * @since   1.0.0
 * @version 1.0.9 Class implemented.
 */
class Model_Endpoint extends Deep_Seek_Endpoint {

	/**
	 * Model_Endpoint constructor.
	 *
	 * @param mixed $utility Utility object containing configuration and tools.
	 *
	 * @since 1.0.0
	 */
	public function __construct( $utility ) {
		parent::__construct( $utility );
	}

	/**
	 * Retrieve a specific model.
	 *
	 * DeepSeek API does not support retrieving individual model details.
	 *
	 * @throws Exception Always, because not supported by DeepSeek API.
	 * @since  1.0.9
	 */
	public function retrieve() {
		throw new Exception(
			Error_Codes::NOT_SUPPORTED,
			__( 'DeepSeek API does not support retrieving individual model details', 'limb-chatbot' )
		);
	}

	/**
	 * List all available models for the DeepSeek provider.
	 *
	 * Calls the DeepSeek /models endpoint to get available models.
	 *
	 * @return array List of models with their details.
	 * @throws Exception If the API request fails.
	 * @since  1.0.9
	 */
	public function list(): array {
		$http_client = $this->http_client_factory();

		$response = $http_client->get(
			self::API_BASE_URL . '/models',
			array(
				'headers' => $this->get_header(),
				'timeout' => $this->utility->get_timeout(),
			)
		);

		return ( new Model_Response_Handler( $response, $http_client, $this ) )->get_models();
	}

	/**
	 * Delete a model.
	 *
	 * DeepSeek API does not support model deletion.
	 *
	 * @throws Exception Always, because not supported by DeepSeek API.
	 * @since  1.0.9
	 */
	public function delete() {
		throw new Exception(
			Error_Codes::NOT_SUPPORTED,
			__( 'DeepSeek API does not support model deletion', 'limb-chatbot' )
		);
	}
}

