<?php
namespace Limb_Chatbot\Includes\AI_Providers\Deep_Seek\Endpoints;

use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Services\Endpoint;

/**
 * Class Deep_Seek_Endpoint
 *
 * Base endpoint class for interacting with the Deep Seek API.
 * Handles authentication headers and common API configurations.
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Deep_Seek\Endpoints
 * @since 1.0.0
 */
class Deep_Seek_Endpoint extends Endpoint {

	/**
	 * Base URL for the Deep Seek API.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const API_BASE_URL = 'https://api.deepseek.com';

	/**
	 * Retrieve the HTTP headers required for API requests.
	 *
	 * Checks for API key presence in the config and prepares Authorization header.
	 *
	 * @since 1.0.0
	 *
	 * @param Config|null $config Optional configuration object containing API credentials.
	 * @param string      $beta_version Optional beta version string (currently unused).
	 *
	 * @throws Exception Throws exception if API key is missing or invalid.
	 *
	 * @return array Associative array of HTTP headers for the API request.
	 */
	protected function get_header( ?Config $config = null, string $beta_version = '' ): array {
		$config = $config ?? $this->get_config();
		if ( ! ( $config instanceof Config ) || empty( $config->get_params()['api_key'] ) ) {
			throw new Exception( Error_Codes::AUTHENTICATION_API_KEY_MISSING, __( 'Api key is missing', 'limb-chatbot' ) );
		}

		return array(
			'Authorization' => 'Bearer ' . $config->get_params()['api_key'],
			'Content-Type'  => 'application/json'
		);
	}
}