<?php

namespace Limb_Chatbot\Includes\AI_Providers\Deep_Seek;

use Limb_Chatbot\Includes\Config_Dependent_Interface;
use Limb_Chatbot\Includes\AI_Providers\AI_Provider;
use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\Utilities\AI_Model_Utility;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;

/**
 * Class Deep_Seek
 *
 * Represents the Deep Seek AI provider integration.
 * Implements config dependent interface and registers hooks
 * related to model cost management.
 *
 * @since 1.0.0
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Deep_Seek
 */
class Deep_Seek extends AI_Provider implements Config_Dependent_Interface {
	/**
	 * Unique ID of the Deep Seek provider.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public static string $id = 'deep-seek';

	/**
	 * Display name of the Deep Seek provider.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public static string $name = 'Deep_Seek';

	/**
	 * Deep_Seek constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// No hooks needed - DeepSeek uses flat-rate pricing.
	}

	/**
	 * Validates whether the given configuration contains valid API credentials for DeepSeek.
	 *
	 * Validates by attempting to list available models from the DeepSeek API.
	 * Handles DeepSeek-specific error codes:
	 * - 400: Invalid Format - Invalid request body format
	 * - 401: Authentication Fails - Wrong API key
	 * - 422: Invalid Parameters - Request contains invalid parameters
	 * - 429: Rate Limit Reached - Sending requests too quickly
	 * - 500: Server Error - Server encounters an issue
	 * - 503: Server Overloaded - Server overloaded due to high traffic
	 *
	 * @since 1.0.0
	 *
	 * @param Config $config Configuration object to validate.
	 * @return bool True if valid API credentials; false otherwise.
	 * @throws Exception If authentication fails or other API errors occur.
	 */
	public function is_valid_config( Config $config ): bool {
		try {
			$models = ( new AI_Model_Utility( null, $config, $this->get_id() ) )->list();

			return ! empty( $models );
		} catch ( Exception $e ) {
			// Re-throw authentication and quota errors with clear messages
			$error_code = $e->get_error_code();

			switch ( $error_code ) {
				case Error_Codes::AUTHENTICATION_UNAUTHORIZED:
				case Error_Codes::AUTHENTICATION_API_KEY_MISSING:
					throw new Exception(
						Error_Codes::AUTHENTICATION_UNAUTHORIZED,
						__( 'DeepSeek API authentication failed. Please check your API key.', 'limb-chatbot' ),
						$e->get_error_data()
					);

				case Error_Codes::VALIDATION_INVALID_VALUE:
					throw new Exception(
						Error_Codes::VALIDATION_INVALID_VALUE,
						__( 'DeepSeek API request contains invalid format or parameters.', 'limb-chatbot' ),
						$e->get_error_data()
					);

				case Error_Codes::TECHNICAL_ERROR:
					throw new Exception(
						Error_Codes::TECHNICAL_ERROR,
						__( 'DeepSeek API server error. Please retry after a brief wait.', 'limb-chatbot' ),
						$e->get_error_data()
					);

				default:
					// Re-throw the original exception for other errors
					throw $e;
			}
		}
	}

	public function get_default_embedding_model(){
		return null;
	}
}