<?php

namespace Limb_Chatbot\Includes\AI_Providers\Claude\Seeders;

use Exception;
use Limb_Chatbot\Includes\AI_Providers\Claude\Claude;
use Limb_Chatbot\Includes\Data_Objects\AI_Model;

/**
 * Class AI_Models_Seeder
 *
 * Seeder class responsible for seeding AI models specific to the Claude AI provider.
 * Creates or updates AI models with predefined configurations and metadata.
 *
 * Based on Claude models documentation as of January 2026:
 * - Claude Sonnet 4.5: Best for real-world agents and coding
 * - Claude Haiku 4.5: Fastest model with near-frontier intelligence
 * - Claude Opus 4.5: Premium model with maximum intelligence
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Claude\Seeders
 * @since 1.0.9
 */
class AI_Models_Seeder {

	/**
	 * Run the seeder to create or update AI models.
	 *
	 * Iterates over a predefined list of models, creates new entries if not present,
	 * or updates existing ones. Also updates associated metadata for each model.
	 *
	 * @return void
	 * @throws Exception If seeding fails.
	 * @since 1.0.9
	 */
	public function run() {
		$models = $this->get_models();

		foreach ( $models as $model_data ) {
			$existing = AI_Model::find_by_name( $model_data['name'] );

			if ( $existing ) {
				// Update existing model
				foreach ( $model_data as $key => $value ) {
					if ( $key !== 'metas' && property_exists( $existing, $key ) ) {
						$existing->{$key} = $value;
					}
				}
				$existing->save();
				$model = $existing;
			} else {
				// Create new model
				$model = AI_Model::create( $model_data );
			}

			// Update metas
			if ( ! empty( $model_data['metas'] ) && $model ) {
				foreach ( $model_data['metas'] as $meta_key => $meta_value ) {
					$model->update_meta( $meta_key, $meta_value );
				}
			}
		}
	}

	/**
	 * Get the list of Claude models to seed.
	 *
	 * @return array Array of model configurations.
	 * @since 1.0.9
	 */
	private function get_models(): array {
		return array(
			// Latest Claude 4.5 Models
			array(
				'label'              => 'Claude Sonnet 4.5',
				'name'               => 'claude-sonnet-4-5',
				'ai_provider_id'     => Claude::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 64000,
				'modalities'         => array(
					'text'  => true,
					'image' => true,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => true,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => true,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => true,
					'vision'             => true,
					'pdf_support'        => true,
				),
				'metas'              => array(
					'output_token_cost' => 0.000015, // $15 per MTok
					'input_token_cost'  => 0.000003, // $3 per MTok
				),
			),
			array(
				'label'              => 'Claude Haiku 4.5',
				'name'               => 'claude-haiku-4-5',
				'ai_provider_id'     => Claude::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 64000,
				'modalities'         => array(
					'text'  => true,
					'image' => true,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => true,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => true,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => true,
					'vision'             => true,
					'pdf_support'        => true,
				),
				'metas'              => array(
					'output_token_cost' => 0.000005, // $5 per MTok
					'input_token_cost'  => 0.000001, // $1 per MTok
				),
			),
			array(
				'label'              => 'Claude Opus 4.5',
				'name'               => 'claude-opus-4-5',
				'ai_provider_id'     => Claude::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 64000,
				'modalities'         => array(
					'text'  => true,
					'image' => true,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => true,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => true,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => true,
					'vision'             => true,
					'pdf_support'        => true,
				),
				'metas'              => array(
					'output_token_cost' => 0.000025, // $25 per MTok
					'input_token_cost'  => 0.000005, // $5 per MTok
				),
			),
			// Legacy Claude 4 Models
			array(
				'label'              => 'Claude Opus 4.1',
				'name'               => 'claude-opus-4-1',
				'ai_provider_id'     => Claude::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 32000,
				'modalities'         => array(
					'text'  => true,
					'image' => true,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => true,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => true,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => true,
					'vision'             => true,
					'pdf_support'        => true,
				),
				'metas'              => array(
					'output_token_cost' => 0.000075,
					'input_token_cost'  => 0.000015,
				),
			),
			array(
				'label'              => 'Claude Sonnet 4.0',
				'name'               => 'claude-sonnet-4-0',
				'ai_provider_id'     => Claude::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 64000,
				'modalities'         => array(
					'text'  => true,
					'image' => true,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => true,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => true,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => true,
					'vision'             => true,
					'pdf_support'        => true,
				),
				'metas'              => array(
					'output_token_cost' => 0.000015,
					'input_token_cost'  => 0.000003,
				),
			),
			array(
				'label'              => 'Claude Opus 4.0',
				'name'               => 'claude-opus-4-0',
				'ai_provider_id'     => Claude::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 32000,
				'modalities'         => array(
					'text'  => true,
					'image' => true,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => true,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => true,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => true,
					'vision'             => true,
					'pdf_support'        => true,
				),
				'metas'              => array(
					'output_token_cost' => 0.000075, // $75 per MTok
					'input_token_cost'  => 0.000015, // $15 per MTok
				),
			),
			// Legacy Claude 3 Haiku
			array(
				'label'              => 'Claude 3 Haiku',
				'name'               => 'claude-3-haiku-20240307',
				'ai_provider_id'     => Claude::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 4096,
				'modalities'         => array(
					'text'  => true,
					'image' => true,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => true,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => true,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => false,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => false,
					'vision'             => true,
					'pdf_support'        => false,
				),
				'metas'              => array(
					'output_token_cost' => 0.00000125, // $1.25 per MTok
					'input_token_cost'  => 0.00000025, // $0.25 per MTok
				),
			),
		);
	}
}

