<?php

namespace Limb_Chatbot\Includes\AI_Providers;

use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\Utilities\AI_Model_Utility;
use Limb_Chatbot\Includes\Utilities\Utility;
use Limb_Chatbot\Includes\Services\Helper;

/**
 * Base class representing an AI provider.
 *
 * Provides common functionality to interact with AI provider utilities and configurations.
 *
 * @package Limb_Chatbot\Includes\AI_Providers
 * @since 1.0.0
 */
class AI_Provider {

	/**
	 * Retrieves the unique identifier of the AI provider.
	 *
	 * Uses the static `$id` property of the subclass.
	 *
	 * @since 1.0.0
	 *
	 * @return string AI provider identifier.
	 */
	public function get_id(): string {
		return static::$id;
	}

	/**
	 * Retrieves the display name of the AI provider.
	 *
	 * Uses the static `$name` property of the subclass or returns empty string if not set.
	 *
	 * @since 1.0.0
	 *
	 * @return string AI provider display name.
	 */
	public function get_name(): string {
		return static::$name ?? '';
	}

	/**
	 * Returns an instance of the AI provider utility for the given utility class.
	 *
	 * Constructs the fully qualified class name based on the AI provider namespace and the
	 * utility class short name. Returns an instance if the class exists, otherwise null.
	 *
	 * @since 1.0.0
	 *
	 * @param Utility $utility Utility instance to create AI provider utility for.
	 * @return object|null Instance of the AI provider utility or null if class does not exist.
	 */
	public function get_ai_providers_utility( Utility $utility ) {
		$classname = Helper::get_namespace_name( static::class ) . '\\Utilities\\' . Helper::get_class_shortname( $utility );

		return class_exists( $classname ) ? new $classname( $utility ) : null;
	}

	/**
	 * Validates whether the given configuration contains valid AI models for this provider.
	 *
	 * Uses AI_Model_Utility to list available models for the configuration and provider.
	 *
	 * @since 1.0.0
	 *
	 * @param Config $config Configuration object to validate.
	 * @return bool True if valid models exist; false otherwise.
	 */
	public function is_valid_config( Config $config ): bool {
		$models = ( new AI_Model_Utility( null, $config, $this->get_id() ) )->list();

		return ! empty( $models );
	}

	public function get_default_embedding_model() {
		return null;
	}
}
