import {__, sprintf} from "@wordpress/i18n";
import {handleError} from "../admin/helpers/notifications";
export {isValidEmailFormat} from "./email";

/**
 * Validate value
 *
 * @param {any} value The field value
 * @param {function[]} validations Validations callbacks
 * @param {boolean} showAllErrors Show all errors
 * @return {{valid: boolean}|{valid: boolean, message: string}}
 */
export const validate = (value, validations, showAllErrors = false) => {
    // If no validations specified then any value is valid
    if (!validations?.length) {
        return {
            valid: true,
            message: "No validations callbacks passed."
        }
    }
    try {
        let messages = [];
        // Start validation
        for (const validationCallback of validations) {
            if (typeof validationCallback === 'function') {
                const res = validationCallback(value);
                if (!res.valid) {
                    // Error found
                    messages.push(res.message);

                    if (!showAllErrors) {
                        break;
                    }
                }
            }
        }

        return {
            valid: messages.length === 0,
            message: messages.join('<br/>'),
        };
    } catch (e) {
        handleError(e);
        return {
            valid: false,
            message: e.message || __("The field validation failed.", 'limb-chatbot'),
        };
    }
}

/**
 * Required
 *
 * @param {any} value The field value
 * @param {object} options Additional options
 * @return {string|boolean}
 */
export const required = (value, options = false) => {
    try {
        // Message
        const message = options?.message || __("This field is required.", 'limb-chatbot');
        // Validation
        if (!value) {
            return {
                valid: false,
                message
            };
        } else if (options?.trim) {
            if (!value.trim()) {
                return {
                    valid: false,
                    message
                };
            }
        }
        // Array
        if (Array.isArray(value) && !value.length) {
            return {
                valid: false,
                message
            };
        }

        return {
            valid: true
        }
    } catch (e) {
        handleError(e);
        return {
            valid: false,
            message: e.message || __("The field validation failed.", 'limb-chatbot'),
        };
    }
}

/**
 * Alphanumeric
 *
 * @param {any} value The field value
 * @param {object} options Additional options
 * @return {{valid: boolean, message: (*|string)}|{valid: boolean}|{valid: boolean, message: string}}
 */
export const alphanumeric = (value, options = false) => {
    try {
        // Pattern and message
        let message = options?.message;
        let pattern;
        if (options?.lowercase) {
            pattern = new RegExp('^[a-z0-9]+$');
            if (!message) {
                message = __("Must consist of lower case alphanumeric characters.", 'limb-chatbot')
            }
        } else if (options?.uppercase) {
            pattern = new RegExp('^[A-Z0-9]+$')
            if (!message) {
                message = __("Must consist of upper case alphanumeric characters.", 'limb-chatbot')
            }
        } else {
            pattern = new RegExp('^[a-zA-Z0-9]+$')
            if (!message) {
                message = __("Must consist of alphanumeric characters.", 'limb-chatbot')
            }
        }
        // Validation
        if (!pattern.test(value)) {
            return {
                valid: false,
                message
            };
        }

        return {
            valid: true
        }
    } catch (e) {
        handleError(e);
        return {
            valid: false,
            message: e.message || __("The field validation failed.", 'limb-chatbot'),
        };
    }
}

/**
 * Alphanumeric and special characters
 *
 * @param {any} value The field value
 * @param {string} specials Special characters
 * @param {object} options Additional options
 * @return {{valid: boolean, message: (*|string)}|{valid: boolean}|{valid: boolean, message: string}}
 */
export const alphanumericAndSpec = (value, specials, options = false) => {
    try {
        // Pattern and message
        let message = options?.message;
        // Validation
        const escapedSpecials = specials.replace(/[-/\\^$*+?.()|[\]{}]/g, '\\$&');
        let pattern;
        if (options?.lowercase) {
            pattern = new RegExp(`^[a-z0-9${escapedSpecials}]+$`);
            if (!message) {
                message = sprintf(__("Must consist of lower case alphanumeric characters or '%s'.", 'limb-chatbot'), specials);
            }
        } else if (options?.uppercase) {
            pattern = new RegExp(`^[A-Z0-9${escapedSpecials}]+$`)
            if (!message) {
                message = sprintf(__("Must consist of upper case alphanumeric characters or '%s'.", 'limb-chatbot'), specials);
            }
        } else {
            pattern = new RegExp(`^[a-zA-Z0-9${escapedSpecials}]+$`);
            if (!message) {
                message = sprintf(__("Must consist alphanumeric characters or '%s'.", 'limb-chatbot'), specials);
            }
        }
        if (!pattern.test(value)) {
            return {
                valid: false,
                message
            };
        }

        return {
            valid: true
        }
    } catch (e) {
        handleError(e);
        return {
            valid: false,
            message: e.message || __("The field validation failed.", 'limb-chatbot'),
        };
    }
}