/**
 * Core email validation logic
 * Validates email format using practical regex and additional checks
 *
 * @param {string} value Email value to validate
 * @returns {boolean} True if email format is valid
 */
export const isValidEmailFormat = (value) => {
    if (!value || typeof value !== 'string') {
        return false;
    }

    const trimmedValue = value.trim();

    // Allow empty (caller should handle required validation separately)
    if (trimmedValue === '') {
        return true;
    }

    // Practical email regex - covers 99% of real-world emails
    // Allows: letters, numbers, dots, hyphens, underscores in local part
    // Requires proper domain structure with valid TLD
    const emailRegex = /^[a-zA-Z0-9._-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/;

    if (!emailRegex.test(trimmedValue)) {
        return false;
    }

    // Check for consecutive dots
    if (trimmedValue.includes('..')) {
        return false;
    }

    // Check local part doesn't start or end with dot
    const [localPart, domain] = trimmedValue.split('@');
    if (!localPart || !domain) {
        return false;
    }

    if (localPart.startsWith('.') || localPart.endsWith('.')) {
        return false;
    }

    // Check domain doesn't start or end with dot/hyphen
    if (domain.startsWith('.') || domain.startsWith('-') ||
        domain.endsWith('.') || domain.endsWith('-')) {
        return false;
    }

    return true;
};