import {useContext, useEffect, useRef, useState} from "@wordpress/element";
import {__} from "@wordpress/i18n";
import {ChatBotsContext} from "../../../../contexts/chatbots";
import useClickOutside from "../../../../hooks/use-click-outside";
import {ANIMATION_TYPES, ICON_COLORS_1, TRIGGER_ANIMATION_TYPE, TRIGGER_ANIMATIONS} from "../_data";

export default function Multi({toggleChatbots, leadCaptureBlocking}) {
    const triggerRef = useRef(null);
    const chatbotsContext = useContext(ChatBotsContext);

    const [isChatListOpened, setIsChatListOpened] = useState(false);
    const [triggerBgIcon, setTriggerBgIcon] = useState('squircle');
    const [triggerIcon, setTriggerIcon] = useState('icon');
    // Check if we're in admin preview
    const isAdminPreview = typeof document !== 'undefined' && document.getElementById('lbaic-chatbot-preview') !== null;
    const isChatbotClosed = chatbotsContext.openedChatbotsUUIDs.length === 0;

    useClickOutside([triggerRef], () => setIsChatListOpened(false))

    useEffect(() => {
        // Update background shape
        const shape = chatbotsContext.chatbots[chatbotsContext.activeChatbotIndex].utility?.icon_bg_shape;
        if (shape) {
            setTriggerBgIcon(shape);
        }
        // Update icon
        const icon = chatbotsContext.chatbots[chatbotsContext.activeChatbotIndex].utility?.icon;
        if (icon) {
            setTriggerIcon(icon);
        }
    }, [chatbotsContext.chatbots, chatbotsContext.activeChatbotIndex]);

    const handleChatListOpened = () => {
        setIsChatListOpened(!isChatListOpened);
    }

    const handleTriggerItemClicked = (uuid) => {
        setIsChatListOpened(false);
        if (chatbotsContext.openedChatbotsUUIDs.indexOf(uuid) !== -1) {
            LimbChatbot.Hooks.doAction('lbaic.chatbot.actions.close', uuid);
        } else {
            toggleChatbots([uuid]);
        }
    }

    // Hide trigger in admin preview when any chat is open
    if (isAdminPreview && !isChatbotClosed) {
        return null;
    }

    return (
        <div ref={triggerRef} className={`lbaic-trigger ${TRIGGER_ANIMATIONS[TRIGGER_ANIMATION_TYPE].open}${leadCaptureBlocking ? ' lbaic-trigger-disabled' : ''}`}>
            <button className='lbaic-trigger-in' onClick={handleChatListOpened} aria-label={__("Toggle chatbots", 'limb-chatbot')}>
                <div className='lbaic-trigger-icon'>
                    <svg className='lbaic-trigger-icon-bg' xmlns='http://www.w3.org/2000/svg'>
                        <use href={`#lbaic-trigger-bg-${triggerBgIcon}`} fill='var(--lbaic-trigger-color)'/>
                        <use href={`#lbaic-trigger-bg-${triggerBgIcon}`} className='lbaic-trigger-hover'
                             fill='var(--lbaic-white-100)' fillOpacity='0.2'/>
                        <use href={`#lbaic-trigger-bg-${triggerBgIcon}`} className='lbaic-trigger-active'
                             fill='var(--lbaic-black-100)' fillOpacity='0.2'/>
                    </svg>
                    <svg className={`lbaic-trigger-icon-in${isChatListOpened ? '' : ' lbaic-trigger-icon-main'}`}
                         xmlns='http://www.w3.org/2000/svg'>
                        {
                            isChatListOpened ? (
                                <use
                                    href={`#lbaic-trigger-arrow`}
                                    fill='var(--lbaic-white-100)'/>
                            ) : (
                                ANIMATION_TYPES[triggerIcon] === 1 ? (
                                    <>
                                        <use
                                            href={`#lbaic-${('chatbot-icon-' + triggerIcon + '-shape')}`}
                                            fill={`${ICON_COLORS_1[triggerIcon].shape}`}/>
                                        <use
                                            href={`#lbaic-${('chatbot-icon-' + triggerIcon + '-left')}`}
                                            fill={`${ICON_COLORS_1[triggerIcon].left}`}/>
                                        <use
                                            href={`#lbaic-${('chatbot-icon-' + triggerIcon + '-center')}`}
                                            fill={`${ICON_COLORS_1[triggerIcon].center}`}/>
                                        <use
                                            href={`#lbaic-${('chatbot-icon-' + triggerIcon + '-right')}`}
                                            fill={`${ICON_COLORS_1[triggerIcon].right}`}/>
                                    </>
                                ) : ANIMATION_TYPES[triggerIcon] === 2 ? (
                                    <>
                                        <use
                                            href={`#lbaic-${('chatbot-icon-' + triggerIcon + '-first')}`}
                                            fill='var(--lbaic-white-100)'/>
                                        <use
                                            href={`#lbaic-${('chatbot-icon-' + triggerIcon + '-last')}`}
                                            fill='var(--lbaic-white-100)'/>
                                    </>
                                ) : ANIMATION_TYPES[triggerIcon] === 3 ? (
                                    <>
                                        <use
                                            href={`#lbaic-${('chatbot-icon-' + triggerIcon + '-shape')}`}
                                            fill={`${ICON_COLORS_1[triggerIcon].shape}`}/>
                                        <use
                                            href={`#lbaic-${('chatbot-icon-' + triggerIcon + '-line')}`}
                                            fill={`${ICON_COLORS_1[triggerIcon].center}`}/>
                                        <use
                                            href={`#lbaic-${('chatbot-icon-' + triggerIcon + '-line-2')}`}
                                            fill={`${ICON_COLORS_1[triggerIcon].center}`}/>
                                    </>
                                ) : (
                                    <use
                                        href={`#lbaic-${('chatbot-icon-' + triggerIcon)}`}
                                        fill='var(--lbaic-white-100)'/>
                                )
                            )
                        }
                    </svg>
                </div>
            </button>
            {isChatListOpened &&
                <div className='lbaic-trigger-list'>
                    {chatbotsContext.chatbots.map(chatbot =>
                        <div key={chatbot.uuid} className='lbaic-trigger-item'
                             onClick={() => handleTriggerItemClicked(chatbot.uuid)}>
                            <button className='lbaic-trigger-item-in'
                                    style={{'--lbaic-trigger-item-color': chatbot.utility.color}}>
                                {!!chatbot.utility?.title &&
                                    <span
                                        className='lbaic-trigger-item-label'>{chatbot.utility.title}</span>}
                                <svg className='lbaic-trigger-item-icon' xmlns='http://www.w3.org/2000/svg'>
                                    <use href={`#lbaic-chatbot-icon-${chatbot.utility.icon || 1}`}
                                         fill='var(--lbaic-white-100)'/>
                                </svg>
                            </button>
                        </div>)}
                </div>}
        </div>
    )
}