import TextInput, {DefaultTextarea, validateTextInput} from './text';
import TextareaInput, {validateTextareaInput} from './textarea';
import ButtonsInput, {validateButtonsInput} from './buttons';
import EmailInput, {validateEmailInput} from './email';
import PhoneInput, {validatePhoneInput} from './phone';
import UrlInput, {validateUrlInput} from './url';
import RecommendationInput, {validateRecommendationInput} from './recommendation';

/**
 * Input type components map
 */
export const INPUT_COMPONENTS = {
    text: TextInput,
    textarea: TextareaInput,
    buttons: ButtonsInput,
    email: EmailInput,
    phone: PhoneInput,
    url: UrlInput,
    recommendation: RecommendationInput
};

/**
 * Input validators map
 */
export const INPUT_VALIDATORS = {
    text: validateTextInput,
    textarea: validateTextareaInput,
    buttons: validateButtonsInput,
    email: validateEmailInput,
    phone: validatePhoneInput,
    url: validateUrlInput,
    recommendation: validateRecommendationInput
};

/**
 * Get the appropriate input component for the given type
 *
 * @param {string} inputType The input type
 * @returns {React.Component} The input component
 */
export const getInputComponent = (inputType) => {
    // If no input type is specified, return the default textarea
    if (!inputType) {
        return DefaultTextarea;
    }
    // Otherwise, return the specific input component or default to text input
    return INPUT_COMPONENTS[inputType] || DefaultTextarea;
};

/**
 * Get the appropriate validator for the given type
 *
 * @param {string} inputType The input type
 * @returns {Function} The validator function
 */
export const getInputValidator = (inputType) => {
    return INPUT_VALIDATORS[inputType] || INPUT_VALIDATORS.text;
};

/**
 * Validate input value based on input type
 *
 * @param {string} value Input value
 * @param {string} inputType Input type
 * @returns {boolean} Is valid
 */
export const validateInput = (value, inputType) => {
    const validator = getInputValidator(inputType);
    return validator(value);
};

// Export individual components
export {
    TextInput,
    TextareaInput,
    ButtonsInput,
    DefaultTextarea,
    EmailInput,
    PhoneInput,
    UrlInput,
    RecommendationInput
};
