import {__} from "@wordpress/i18n";
import {isValidEmailFormat} from "../../../../../../validations";

/**
 * EmailInput Component
 * Validates email format
 *
 * @param {Object} props Component props
 * @param {React.Ref} props.textareaRef Textarea ref
 * @param {string} props.text Current text value
 * @param {Function} props.onTextChange Text change handler
 * @param {Function} props.onKeyDown Key down handler
 * @param {boolean} props.isValid Is input valid
 * @param {string} props.placeholder Custom placeholder
 */
export default function EmailInput({
                                       textareaRef,
                                       text,
                                       onTextChange,
                                       onKeyDown,
                                       isValid,
                                       placeholder
                                   }) {
    const hasError = !isValid && text.trim().length > 0;
    const placeholderText = placeholder || __('Enter email address...', 'limb-chatbot');

    return (
        <input
            ref={textareaRef}
            type="email"
            className={`lbaic-textarea${hasError ? ' lbaic-textarea-invalid' : ''}`}
            placeholder={placeholderText}
            value={text}
            onChange={onTextChange}
            onKeyDown={onKeyDown}
            autoComplete="off"
        />
    );
}

/**
 * Validate email input
 * Uses shared email validation utility
 *
 * @param {string} value Input value
 * @returns {boolean} Is valid email
 */
export const validateEmailInput = (value) => {
    // Empty input is "valid" (just can't send)
    if (!value || value.trim().length === 0) {
        return true;
    }

    return isValidEmailFormat(value);
};
