import Tooltip from "../../../tooltip";

/**
 * ButtonsInput Component
 * Displays clickable button options for user selection
 *
 * @param {Object} props Component props
 * @param {Function} props.onButtonClick Button click handler for auto-submit
 * @param {Object} props.config Configuration object containing options array
 */
export default function ButtonsInput({
                                         onButtonClick,
                                         config,
                                     }) {
    // Extract options from config
    const options = config?.options || [];

    // If no options provided, return null
    if (!options || options.length === 0) {
        return null;
    }

    return (
        <div className='lbaic-buttons-input'>
            {options.map((option, index) => {
                const button = (
                    <button
                        key={index}
                        type="button"
                        className='lbaic-button-option'
                        onClick={() => onButtonClick(option.label, option.value)}
                        data-value={option.value}
                    >
                        <span className='lbaic-button-option-label'>{option.label}</span>
                    </button>
                );

                // Wrap with tooltip if tooltip text is provided
                if (option.tooltip) {
                    return (
                        <div className='lbaic-button-option-tooltip-wrapper'>
                            <Tooltip
                                key={index}
                                title={option.tooltip}
                                tooltipPosition="top"
                            >
                                {button}
                            </Tooltip>
                        </div>
                    );
                }

                return button;
            })}
        </div>
    );
}

/**
 * Validate buttons input
 * Always valid since buttons auto-submit on click
 *
 * @param {string} value Input value
 * @returns {boolean} Always returns true
 */
export const validateButtonsInput = (value) => {
    // Always valid - clicking a button is the validation itself
    return true;
};

