import {resolveSelect} from '@wordpress/data';
import {urlSearchParams} from "../../../helpers";
import {handleResponse} from "../../helpers/rest";

/**
 * Get site data
 *
 * @return {Promise<{title: *, description: *}>} Site data
 */
export const GetSiteData = async () => {
    const allData = await resolveSelect('core').getSite();

    return {
        title: allData?.title,
        description: allData?.description,
    }
};

/**
 * Get all user-friendly, public post types
 *
 * @param {object} params Params
 * @return {Promise<object[]>} Array of post type objects
 */
export const GetPostTypes = async (params = {}) => {
    return LimbChatbot?.wp_objects?.post_types || [];
}

/**
 * Get post type schema
 *
 * @param {string} postType Post type
 * @param {object} params Query params
 * @return {Promise<any>}
 */
export const GetPostTypeSchema = async (postType, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${LimbChatbot.rest.url}post_types/${postType}/schema${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': LimbChatbot.rest.nonce
        }
    }).then(handleResponse);
}

/**
 * Get WP posts
 *
 * @param {string} postType Post type
 * @param {object} params Query params
 * @return {Promise<any>}
 */
export const GetPosts = async (postType, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${LimbChatbot.rest.url}post_types/${postType}${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': LimbChatbot.rest.nonce
        }
    }).then(handleResponse);
}

/**
 * Get post
 *
 * @param {string} postType Post type
 * @param {number} postId Post ID
 * @return {Promise<(function(string, string, (number|string)=, (Object|undefined)): function({select: *, dispatch: *, registry: *, resolveSelect: *}): Promise<void>)|*>}
 */
export const GetPost = (postType, postId) => {
    return resolveSelect('core').getEntityRecord(
        'postType',
        postType,
        postId
    );
}

/**
 * Get all user-friendly, public taxonomies
 *
 * @param {object} params Params
 * @return {Promise<object[]>} Array of taxonomy objects
 */
export const GetTaxonomies = async (params = {}) => {
    const allTaxonomies = await resolveSelect('core').getTaxonomies({
        per_page: -1,
        ...params
    }) || [];

    // Filter out internal or hidden taxonomies
    return allTaxonomies.filter(
        (item) => item.visibility?.public && item.visibility?.show_ui && !item.types.includes('wp_template')
    );
}

/**
 * Get term
 *
 * @param {string} taxonomy Taxonomy
 * @param {number} termId Term ID
 * @return {(function(string, string, (number|string)=, (Object|undefined)): function({select: *, dispatch: *, registry: *, resolveSelect: *}): Promise<void>)|*}
 */
export const GetTerm = (taxonomy, termId) => {
    return resolveSelect('core').getEntityRecord(
        'taxonomy',
        taxonomy,
        termId
    );
}

/**
 * Get object groups
 *
 * @param {object} params Query params
 * @return {Promise<*>}
 * @constructor
 */
export const GetObjectGroups = (params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${LimbChatbot.rest.url}object_groups${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': LimbChatbot.rest.nonce
        }
    }).then(handleResponse);
}

/**
 * Get object group
 *
 * @param {string} objectType Object type
 * @param {object} params Query params
 * @return {Promise<*>}
 * @constructor
 */
export const GetObjectGroup = (objectType, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${LimbChatbot.rest.url}object_groups/${objectType}/objects${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': LimbChatbot.rest.nonce
        }
    }).then(handleResponse);
}