import {urlSearchParams} from "../../../helpers";
import {handleResponse} from "../../helpers/rest";

/**
 * Get vectors
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {object} params Query params
 * @return {Promise<any>}
 * @constructor
 */
export const GetVectors = (restUrl, restNonce, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}vectors${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Get vector
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Vector ID
 * @param {object} params Query params
 * @return {Promise<any>}
 * @constructor
 */
export const GetVector = (restUrl, restNonce, id, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}vectors/${id}${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Create vector
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {object} requestData Vector data to create
 * @return {Promise<any>}
 * @constructor
 */
export const CreateVector = (restUrl, restNonce, requestData) => {
    return fetch(`${restUrl}vectors`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Update vector
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Vector ID
 * @param {object} requestData Vector data to update
 * @return {Promise<any>}
 * @constructor
 */
export const UpdateVector = (restUrl, restNonce, id, requestData) => {
    return fetch(`${restUrl}vectors/${id}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Upsert vector
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Vector ID
 * @param {object} requestData Vector data to update
 * @return {Promise<*>}
 * @constructor
 */
export const UpsertVector = (restUrl, restNonce, id, requestData) => {
    return fetch(`${restUrl}vectors/${id}/upsert`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Upsert vectors
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number[]} ids Vectors IDs
 * @param {object} requestData Vector data to update
 * @return {Promise<*>}
 * @constructor
 */
export const UpsertVectors = (restUrl, restNonce, ids, requestData) => {
    return fetch(`${restUrl}vectors/upsert/batch`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify({
            ...requestData,
            vector_ids: ids
        })
    }).then(handleResponse);
}

/**
 * Sync vector
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Vector ID
 * @return {Promise<*>}
 * @constructor
 */
export const SyncVector = (restUrl, restNonce, id) => {
    return fetch(`${restUrl}vectors/${id}/sync`, {
        method: 'POST',
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
    }).then(handleResponse);
}

/**
 * Sync vectors
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number[]} ids Vectors IDs
 * @return {Promise<*>}
 * @constructor
 */
export const BatchSyncVector = (restUrl, restNonce, ids) => {
    return fetch(`${restUrl}vectors/sync/batch`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify({
            vector_ids: ids
        })
    }).then(handleResponse);
}

/**
 * Delete vector
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Vector ID
 * @param {object} params Query params
 * @return {Promise<any>}
 * @constructor
 */
export const DeleteVector = (restUrl, restNonce, id, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}vectors/${id}${query}`, {
        method: 'DELETE',
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}