import {urlSearchParams} from "../../../helpers";
import {handleResponse} from "../../helpers/rest";

/**
 * Get settings
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {object} params Query params
 * @return {Promise<any>}
 */
export const GetSettings = (restUrl, restNonce, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}settings${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Get setting
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {string} key Setting key
 * @return {Promise<any>}
 */
export const GetSetting = (restUrl, restNonce, key) => {
    return fetch(`${restUrl}settings/${key}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Create setting
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {object[]} requestData Settings to create/update
 * @return {Promise<any>}
 */
export const CreateSetting = (restUrl, restNonce, requestData) => {
    return fetch(`${restUrl}settings`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Create settings
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {object[]} requestData Settings to create/update
 * @return {Promise<any>}
 */
export const CreateSettings = (restUrl, restNonce, requestData) => {
    return fetch(`${restUrl}settings/batch`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Create/Update setting
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {string} key Setting key
 * @param {any} value Setting value
 * @return {Promise<any>}
 */
export const UpdateSetting = (restUrl, restNonce, key, value) => {
    return fetch(`${restUrl}settings/${key}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify({value})
    }).then(handleResponse);
}

/**
 * Update/Create/Delete settings
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {object[]} requestData Settings to create/update
 * @param {object} params Query params
 * @return {Promise<any>}
 */
export const UpdateSettings = (restUrl, restNonce, requestData, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}settings/batch${query}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Delete setting
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {string} key Setting key
 * @return {Promise<void>}
 */
export const DeleteSetting = (restUrl, restNonce, key) => {
    return fetch(`${restUrl}settings/${key}`, {
        method: 'DELETE',
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce,
        },
    }).then(handleResponse);
}


/**
 * Delete preview setting
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {object} params Query params
 * @return {Promise<void>}
 */
export const DeleteSettings = (restUrl, restNonce, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}settings${query}`, {
        method: 'DELETE',
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce,
        },
    }).then(handleResponse);
}

/**
 * Delete file
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {string} key Setting key
 * @param {object} requestData Body
 * @return {Promise<void>}
 */
export const DeleteFile = (restUrl, restNonce, key, requestData) => {
    return fetch(`${restUrl}settings/${key}`, {
        method: 'DELETE',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce,
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}