import {urlSearchParams} from "../../../helpers";
import {handleResponse} from "../../helpers/rest";

export const DATASET_TYPE_KNOWLEDGE = 1;
export const DATASET_TYPE_FINE_TUNING = 2;
export const DATASET_TYPE_ACTIONABLE_KNOWLEDGE = 3;

export const DATASET_STATUS_PENDING = 0;        // Created but not started
export const DATASET_STATUS_GENERATING = 1;     // Currently generating content
export const DATASET_STATUS_GENERATED = 2;      // Content generated, ready for indexing
export const DATASET_STATUS_INDEXING = 3;       // Currently indexing vectors
export const DATASET_STATUS_INDEXED = 4;        // Vectors indexed, ready for completion
export const DATASET_STATUS_FAILED = 6;         // Failed at some stage

/**
 * Get datasets
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {object} params Query params
 * @return {Promise<any>}
 * @constructor
 */
export const GetDatasets = (restUrl, restNonce, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}datasets${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Get dataset
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Dataset ID
 * @param {object} params Query params
 * @return {Promise<any>}
 * @constructor
 */
export const GetDataset = (restUrl, restNonce, id, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}datasets/${id}${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Create dataset
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {object} requestData Dataset data to create
 * @return {Promise<any>}
 * @constructor
 */
export const CreateDataset = (restUrl, restNonce, requestData) => {
    return fetch(`${restUrl}datasets`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Update dataset
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Dataset ID
 * @param {object} requestData Dataset data to update
 * @return {Promise<any>}
 * @constructor
 */
export const UpdateDataset = (restUrl, restNonce, id, requestData) => {
    return fetch(`${restUrl}datasets/${id}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Dataset tuning
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {object} requestData Tuning data
 * @return {Promise<*>}
 */
export const TuningDataset = (restUrl, restNonce, requestData) => {
    return fetch(`${restUrl}ai_models/tuning`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Delete dataset
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Dataset ID
 * @return {Promise<any>}
 * @constructor
 */
export const DeleteDataset = (restUrl, restNonce, id) => {
    return fetch(`${restUrl}datasets/${id}`, {
        method: 'DELETE',
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Upload dataset file
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {File} file File
 * @return {Promise<any>}
 * @constructor
 */
export const UploadDatasetFile = (restUrl, restNonce, file) => {
    const formData = new FormData();
    formData.append('file', file);

    return fetch(`${restUrl}datasets/upload`, {
        method: 'POST',
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: formData
    }).then(handleResponse);
}

/**
 * Import dataset file
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {object} requestData Request data
 * @return {Promise<any>}
 * @constructor
 */
export const ImportDatasetFile = (restUrl, restNonce, requestData) => {
    return fetch(`${restUrl}datasets/import`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Regenerate dataset from source content
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Dataset ID
 * @param {string} sourceContent HTML source content
 * @param {string|null} title Dataset title (optional)
 * @return {Promise<any>}
 * @constructor
 */
export const RegenerateDataset = (restUrl, restNonce, id, sourceContent, title = null) => {
    const body = {
        source_content: sourceContent
    };

    // Only include title if provided
    if (title !== null) {
        body.title = title;
    }

    return fetch(`${restUrl}datasets/${id}/regenerate`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(body)
    }).then(handleResponse);
}