import {urlSearchParams} from "../../../helpers";
import {handleResponse} from "../../helpers/rest";

/**
 * Get datasets entry
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} datasetId Dataset ID
 * @param {object} params Query params
 * @return {Promise<any>}
 * @constructor
 */
export const GetDatasetEntries = (restUrl, restNonce, datasetId, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}datasets/${datasetId}/dataset_entries${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Get dataset entry
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Dataset entry ID
 * @param {object} params Query params
 * @return {Promise<any>}
 * @constructor
 */
export const GetDatasetEntry = (restUrl, restNonce, id, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}dataset_entries/${id}${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Create dataset entry
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} datasetId Dataset ID
 * @param {object} requestData Dataset data to create
 * @return {Promise<any>}
 * @constructor
 */
export const CreateDatasetEntry = (restUrl, restNonce, datasetId, requestData) => {
    return fetch(`${restUrl}datasets/${datasetId}/dataset_entries`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Update dataset entry
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Dataset entry ID
 * @param {object} requestData Dataset data to update
 * @return {Promise<any>}
 * @constructor
 */
export const UpdateDatasetEntry = (restUrl, restNonce, id, requestData) => {
    return fetch(`${restUrl}dataset_entries/${id}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Delete dataset entry
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} id Dataset entry ID
 * @return {Promise<any>}
 * @constructor
 */
export const DeleteDatasetEntry = (restUrl, restNonce, id) => {
    return fetch(`${restUrl}dataset_entries/${id}`, {
        method: 'DELETE',
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Delete dataset entries
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} datasetId Dataset ID
 * @param {number[]} ids Dataset entries IDs
 * @return {Promise<any>}
 * @constructor
 */
export const DeleteDatasetEntries = (restUrl, restNonce, datasetId, ids) => {
    const query = urlSearchParams({
        id: ids
    });

    return fetch(`${restUrl}datasets/${datasetId}/dataset_entries/${query}`, {
        method: 'DELETE',
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}