import {urlSearchParams} from "../../../helpers";
import {handleResponse} from "../../helpers/rest";

/**
 * Get action parameters
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} actionId Action ID
 * @param {object} params Query params
 * @return {Promise<any>}
 * @constructor
 */
export const GetActionParameters = (restUrl, restNonce, actionId, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}actions/${actionId}/parameters${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Get action parameter
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} actionId Action ID
 * @param {number} id Action parameter ID
 * @param {object} params Query params
 * @return {Promise<any>}
 * @constructor
 */
export const GetActionParameter = (restUrl, restNonce, actionId, id, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}actions/${actionId}/parameters/${id}${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Create action parameter
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} actionId Action ID
 * @param {object} requestData Action data to create
 * @return {Promise<any>}
 * @constructor
 */
export const CreateActionParameter = (restUrl, restNonce, actionId, requestData) => {
    return fetch(`${restUrl}actions/${actionId}/parameters`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Update action
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} actionId Action ID
 * @param {number} id Action parameter ID
 * @param {object} requestData Action data to update
 * @return {Promise<any>}
 * @constructor
 */
export const UpdateActionParameter = (restUrl, restNonce, actionId, id, requestData) => {
    return fetch(`${restUrl}actions/${actionId}/parameters/${id}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Delete action parameter
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} actionId Action ID
 * @param {number} id Action parameter ID
 * @return {Promise<any>}
 * @constructor
 */
export const DeleteActionParameter = (restUrl, restNonce, actionId, id) => {
    return fetch(`${restUrl}actions/${actionId}/parameters/${id}`, {
        method: 'DELETE',
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}