import {urlSearchParams} from "../../../helpers";
import {handleResponse} from "../../helpers/rest";

/**
 * Get action callbacks
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} actionId Action ID
 * @param {object} params Query params
 * @return {Promise<any>}
 * @constructor
 */
export const GetActionCallbacks = (restUrl, restNonce, actionId, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}actions/${actionId}/callbacks${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Get action callback
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} actionId Action ID
 * @param {number} id Action callback ID
 * @param {object} params Query params
 * @return {Promise<any>}
 * @constructor
 */
export const GetActionCallback = (restUrl, restNonce, actionId, id, params = {}) => {
    const query = urlSearchParams(params);

    return fetch(`${restUrl}actions/${actionId}/callbacks/${id}${query}`, {
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}

/**
 * Create action callback
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} actionId Action ID
 * @param {object} requestData Action data to create
 * @return {Promise<any>}
 * @constructor
 */
export const CreateActionCallback = (restUrl, restNonce, actionId, requestData) => {
    return fetch(`${restUrl}actions/${actionId}/callbacks`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Update action
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} actionId Action ID
 * @param {number} id Action callback ID
 * @param {object} requestData Action data to update
 * @return {Promise<any>}
 * @constructor
 */
export const UpdateActionCallback = (restUrl, restNonce, actionId, id, requestData) => {
    return fetch(`${restUrl}actions/${actionId}/callbacks/${id}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        },
        body: JSON.stringify(requestData)
    }).then(handleResponse);
}

/**
 * Delete action callback
 *
 * @param {string} restUrl Rest url
 * @param {string} restNonce Rest nonce
 * @param {number} actionId Action ID
 * @param {number} id Action callback ID
 * @return {Promise<any>}
 * @constructor
 */
export const DeleteActionCallback = (restUrl, restNonce, actionId, id) => {
    return fetch(`${restUrl}actions/${actionId}/callbacks/${id}`, {
        method: 'DELETE',
        headers: {
            'Accept': 'application/json',
            'X-WP-Nonce': restNonce
        }
    }).then(handleResponse);
}