import {__} from "@wordpress/i18n";
import PopupContainer from "../../container";

/**
 * Get widget item types for popup
 * @param {string} location - Widget location ('home' or 'new_chat')
 * @return {Array} Filtered widget item types
 */
const getWidgetItemTypes = (location) => {
    const allTypes = [
        {
            type: 'text',
            label: __("Text", 'limb-chatbot'),
            icon: 'widget-text',
        },
        {
            type: 'prompt',
            label: __("Prompt", 'limb-chatbot'),
            icon: 'pw',
        },
        {
            type: 'link',
            label: __("Link", 'limb-chatbot'),
            icon: 'link',
        },
        {
            type: 'message',
            label: __("Message", 'limb-chatbot'),
            icon: 'message',
        },
        {
            type: 'chat_history',
            label: __("Chat history", 'limb-chatbot'),
            icon: 'history',
        },
        {
            type: 'lead_capture',
            label: __("Lead capture", 'limb-chatbot'),
            icon: 'lead-capture',
        }
    ];

    // Filter based on location
    if (location === 'home') {
        // Home: Show Prompt, Link, Chat history, Text (exclude Message, Lead capture)
        return allTypes.filter(item => item.type !== 'message' && item.type !== 'lead_capture');
    } else if (location === 'new_chat') {
        // New chat: Show Prompt, Message, Lead capture (exclude Link, Chat history, Text)
        return allTypes.filter(item => item.type === 'prompt' || item.type === 'message' || item.type === 'lead_capture');
    }

    // Default: show all types
    return allTypes;
};

export default function AddWidgetItem({close, onSelect, location}) {
    const widgetItemTypes = getWidgetItemTypes(location);

    const handleSelect = (item) => {
        if (item.disabled) {
            return;
        }
        if (onSelect) {
            onSelect(item.type);
        }
        close();
    };

    return (
        <PopupContainer
            title={__("Add item", 'limb-chatbot')}
            description={__("Choose the type of widget", 'limb-chatbot')}
            close={close}
            headerActions={
                <button
                    className="lbaic-settings-popup-header-action"
                    onClick={close}
                >
                    <svg className='lbaic-settings-popup-header-action-i' xmlns='http://www.w3.org/2000/svg' fill='none'
                         viewBox='0 0 24 24'>
                        <use href='#lbaic-settings-close'/>
                    </svg>
                </button>
            }
            popupClassName="lbaic-settings-cb-pw-popup-add-item"
        >
            <div className="lbaic-settings-mp-popup-body">
                {widgetItemTypes.map((item, index) => (
                    <div
                        key={index}
                        className={`lbaic-settings-mp-popup-body-in${item.disabled ? ' lbaic-settings-pointer-none' : ''}`}
                        onClick={() => handleSelect(item)}
                    >
                        {item.badge}
                        <div
                            className={`lbaic-settings-mp-popup-body-inner${item.disabled ? ' lbaic-settings-disabled' : ''}`}>
                            <svg className='lbaic-settings-mp-popup-icon' xmlns='http://www.w3.org/2000/svg' fill='none'
                                 viewBox='0 0 24 24'>
                                <use href={`#lbaic-settings-${item.icon}`}/>
                            </svg>
                            <span className="lbaic-settings-mp-popup-body-label">{item.label}</span>
                            {item.integration}
                        </div>
                    </div>
                ))}
            </div>
        </PopupContainer>
    );
}