import {useState} from "@wordpress/element";
import {__, sprintf} from '@wordpress/i18n';
import Notice from "../../../../../sections/notice";
import Container from "../../../containers/content/container";
import confirm from "../../../../../../helpers/confirm";
import {handleError, handleSuccess} from "../../../../../../helpers/notifications";
import {DeletePluginData} from "../../../../../../rest/plugin";
import {getNavLinkTo} from "../../../../../../helpers";
import ContentBodyInner from "../../../containers/content/body-inner";

export default function SettingsUninstall({notifications}) {
    const [uninstalling, setUninstalling] = useState(false);

    const handleUninstall = async () => {
        if (uninstalling) {
            return;
        }

        // Set flag to skip chatbot preview on redirect
        window.lbaicSkipChatbotPreviewOnRedirect = true;

        const confirmed = await confirm({
            title: __("Confirm reset to defaults", 'limb-chatbot'),
            description: sprintf(__("Are you sure you want to reset to defaults?%sThis will permanently reset all data related to the plugin.", 'limb-chatbot'), '<br/>', '<br/><br/>'),
            confirmLabel: __("Yes, reset", 'limb-chatbot'),
            cancelLabel: __("Cancel", 'limb-chatbot'),
            style: {
                '--lbaic-settings-popup-max-width': '345px',
            },
            beforeUnmount: (confirmed) => {
                if (!confirmed) {
                    window.lbaicSkipChatbotPreviewOnRedirect = false;
                }
            },
        });

        if (!confirmed) {
            return;
        }

        setUninstalling(true);
        try {
            await DeletePluginData(LimbChatbot.rest.url, LimbChatbot.rest.nonce);
            handleSuccess(notifications.set, {
                title: __("All plugin data has been removed successfully.", 'limb-chatbot'),
            });
            // Set flags to skip unsaved changes warning
            window.lbaicSkipUnsavedChangesWarning = true;
            // Redirect to default chatbot page after successful reset
            window.location.href = getNavLinkTo('chatbot', 'settings');
        } catch (e) {
            handleError(e, notifications.set, {
                title: __("Failed to delete plugin data.", 'limb-chatbot'),
                description: e?.message ? __(e.message, 'limb-chatbot') : __("Please try again.", 'limb-chatbot'),
            });
            setUninstalling(false);
        }
    };

    return (
        <ContentBodyInner>
            <Container>
                <div className="lbaic-settings-column">
                    <div className="lbaic-settings-column-in">
                        <Notice
                            type='warning'
                            desc={__("Click the button below to reset Limb AI Chatbot to its initial state. This will remove all existing data and settings, and the onboarding setup will appear again. All data will be permanently lost and cannot be recovered.", 'limb-chatbot')}
                            dismissible={false}
                        />
                    </div>
                </div>
                <div className='lbaic-settings-column'>
                    <div className="lbaic-settings-column-in">
                        <button
                            onClick={handleUninstall}
                            className={`lbaic-settings-button lbaic-settings-button-center lbaic-settings-button-h-40 lbaic-settings-button-danger${uninstalling ? ' lbaic-settings-button-disabled' : ''}`}
                        >
                            {uninstalling && (
                                <svg className='lbaic-settings-button-i lbaic-settings-loading-circle'
                                     xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 24 24'>
                                    <use href='#lbaic-settings-circle'/>
                                </svg>
                            )}
                            <span className='lbaic-settings-button-label'>{__("Reset to defaults", 'limb-chatbot')}</span>
                        </button>
                    </div>
                </div>
            </Container>
        </ContentBodyInner>
    );
}


