import {__, sprintf} from "@wordpress/i18n";
import Chip from "../../../../../../fields/chip";
import Tooltip from "../../../../../../fields/tooltip";
import {handleError} from "../../../../../../../helpers/notifications";
import {
    DATASET_STATUS_FAILED,
    DATASET_STATUS_GENERATED,
    DATASET_STATUS_GENERATING,
    DATASET_STATUS_INDEXED,
    DATASET_STATUS_INDEXING,
    DATASET_STATUS_PENDING
} from "../../../../../../../rest/datasets";
import {DATASET_PROCESSES_INFO} from "../_data";
import {parseDateString} from "../../../../../../../../helpers";
import {WP_POST_TYPES_ICONS} from "../add-edit/_data";

/**
 * Render row content of the `Name` column
 *
 * @param {object} row Row data
 * @param {number} rowIndex Row index
 * @param {object} processInfo Process info
 * @return {JSX.Element}
 */
export const renderName = (row, rowIndex, processInfo = null) => {
    // Errors/warnings
    const errors = [], warnings = [];
    if (row.metas?.length) {
        for (const meta of row.metas) {
            if (meta.meta_key === 'errors') {
                if (meta.meta_value?.length) {
                    for (const error of meta.meta_value) {
                        // Show only error that isn't critical, critical once already shown with notifications
                        if (error.is_critical) {
                            errors.push(error.message);
                        } else {
                            warnings.push(error.message);
                        }
                    }
                }
                break;
            }
        }
    }

    if (row.status !== DATASET_STATUS_INDEXED) {
        // If is newly created and a process is currently going
        const {preparing, processState} = processInfo || {};
        const isInProcessRow =
            // Some process is currently active
            (
                processState?.status === 'start'
                || processState?.status === 'pause'
            )
            && (
                processState.datasetIds?.includes(row.id) // The process includes the current row
                || ( // Or newly created
                    processState.startedAt
                    && parseDateString(row.updated_at) > processState.startedAt
                )
            );

        // Show errors if it hasn't yet
        if (!isInProcessRow && !errors.length && !warnings.length) {
            if (row.status === DATASET_STATUS_PENDING) {
                warnings.push(sprintf(__("%s hasn't been started or completed.", 'limb-chatbot'), DATASET_PROCESSES_INFO['dataset_generating'].title));
            } else if (row.status === DATASET_STATUS_GENERATING) {
                warnings.push(sprintf(__("%s not complete.", 'limb-chatbot'), DATASET_PROCESSES_INFO['dataset_generating'].title));
            } else if (row.status === DATASET_STATUS_FAILED) {
                errors.push(__("Something went wrong!", 'limb-chatbot'));
            }
        }
    }

    // Notices
    let info = false;
    if (errors.length > 0 || warnings.length > 0) {
        info = <>
            {errors.length > 0 &&
                <Tooltip label={errors.map(item => __(item, 'limb-chatbot')).join('<br/>')}>
                    <svg className='lbaic-settings-tooltip-i lbaic-settings-error-i' xmlns='http://www.w3.org/2000/svg'
                         fill='none'
                         viewBox='0 0 24 24'>
                        <use href='#lbaic-settings-error'/>
                    </svg>
                </Tooltip>}
            {warnings.length > 0 &&
                <Tooltip label={warnings.join('<br/>')}>
                    <svg className='lbaic-settings-tooltip-i lbaic-settings-warning-i'
                         xmlns='http://www.w3.org/2000/svg'
                         fill='none'
                         viewBox='0 0 24 24'>
                        <use href='#lbaic-settings-warning'/>
                    </svg>
                </Tooltip>}
        </>
    }

    try {
        let link;
        if (row.source) {
            switch (row.source_type) {
                case 'cpt':
                    link = row.source_url || row.source_object?.link || row.source_object?.guid;
                    break;
                default:
                    break;
            }
        }

        return <div className='lbaic-settings-table-card-body-item-in lbaic-settings-table-card-body-input-in'>
            {/*Info*/}
            {info && (
                <div className='lbaic-settings-table-card-body-label-actions'>
                    {info}
                </div>
            )}
            {/*Label*/}
            {Boolean(link) ? (
                <span className='lbaic-settings-table-card-body-label lbaic-settings-table-card-body-input-label'>
                    <a href={link} target="_blank" rel="noopener"
                       className='lbaic-settings-table-card-body-input-link' onClick={(e) => e.stopPropagation()}>
                        <span className='lbaic-settings-table-card-body-input-link-in'
                              dangerouslySetInnerHTML={{__html: row.name || __("N/A", 'limb-chatbot')}}/>
                        <svg className='lbaic-settings-table-card-body-i lbaic-settings-table-card-body-input-i'
                             xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 24 24'>
                            <use href='#lbaic-settings-external-arrow' className="lbaic-settings-external-arrow"/>
                            <use href='#lbaic-external-box'/>
                        </svg>
                    </a>
                </span>
            ) : (
                <span
                    className={`lbaic-settings-table-card-body-label lbaic-settings-table-card-body-label-edit${!row.name ? ' lbaic-settings-table-card-body-label-null' : ''}`}
                    dangerouslySetInnerHTML={{__html: row.name || __("N/A", 'limb-chatbot')}}/>
            )}
        </div>;
    } catch (e) {
        handleError(e);
    }

    return <div className='lbaic-settings-table-card-body-item-in lbaic-settings-table-card-body-input-in'>
        {info && (
            <div className='lbaic-settings-table-card-body-label-actions'>
                {info}
            </div>
        )}
        <span
            className="lbaic-settings-table-card-body-label lbaic-settings-table-card-body-label-null">{__("N/A", 'limb-chatbot')}</span>
    </div>;
}

/**
 * Render row content of the `Source` column
 *
 * @param {object} row Row data
 * @return {JSX.Element}
 */
export const renderSource = (row) => {
    let iconName, title;
    switch (row?.source_type) {
        case 'cpt':
            iconName = WP_POST_TYPES_ICONS.includes(row.source_sub_type) ? row.source_sub_type : 'settings';
            title = row.source_sub_type;
            break;
        default:
            break;
    }

    return <Chip label={title} iconName={iconName}/>;
}

/**
 * Render row content of the `Storage` column
 *
 * @param {object} row Row data
 * @return {JSX.Element[]|JSX.Element}
 */
export const renderStorage = (row) => {
    if (row?.storage?.id) {
        return <Chip label={row.storage.vector_db_id || __('local', 'limb-chatbot')}
                     iconName={row.storage.vector_db_id || 'wordpress'}/>;
    } else {
        return <Chip/>;
    }
}

/**
 * Render row content of the `AI` column
 *
 * @param {object} row Row data
 * @param {string|null} kbAIProviderId Knowledge settings AI provider ID
 * @return {JSX.Element}
 */
export const renderAIProvider = (row, kbAIProviderId) => {
    const rowAIProviderId = row?.ai_provider_id;
    const showWarning = kbAIProviderId && rowAIProviderId && rowAIProviderId !== kbAIProviderId;

    return (
        <div className='lbaic-settings-table-card-body-item-in lbaic-settings-table-card-body-ai-in'>
            {showWarning && (
                <Tooltip
                    label={__("This dataset was created using a different embedding model, so it won't be included in knowledge retrieval with the current Knowledge → Settings.", 'limb-chatbot')}
                    width={210}
                >
                    <svg className='lbaic-settings-tooltip-i lbaic-settings-warning-i'
                         xmlns='http://www.w3.org/2000/svg'
                         fill='none'
                         viewBox='0 0 24 24'>
                        <use href='#lbaic-settings-warning'/>
                    </svg>
                </Tooltip>
            )}
            {rowAIProviderId ? (
                <Chip label={rowAIProviderId} iconName={rowAIProviderId}/>
            ) : (
                <Chip/>
            )}
        </div>
    );
}