import {useEffect, useRef, useState} from "@wordpress/element";
import {createHooks} from "@wordpress/hooks";
import {useLocation} from 'react-router-dom';
import Search from "./components/search";
import Header from "./components/header";
import NavList from "./components/list";

if (!LimbChatbot?.Hooks) {
    LimbChatbot.Hooks = createHooks();
}

LimbChatbot.Hooks.addFilter('lbaic.admin.page.settings.dropdownNavLink.classes', 'lbaic/admin/page/settings/dropdown-nav-link/classes', (classes, navItem, params, state) => {
    if (navItem?.disabled) {
        return [...classes, 'lbaic-settings-pointer-none'];
    }
    return classes;
});

LimbChatbot.Hooks.addFilter('lbaic.admin.page.settings.dropdownNavLabel.classes', 'lbaic/admin/page/settings/dropdown-nav-label/classes', (classes, navItem) => {
    if (navItem?.disabled) {
        return [...classes, 'lbaic-settings-disabled'];
    }
    return classes;
});

export default function Nav() {
    const location = useLocation();

    const [isNavOpened, setIsNavOpened] = useState(false);
    const navList = useRef(null);

    useEffect(() => {
        const layout = document.querySelector('.lbaic-settings-nav-layout');

        if (!layout) return;

        const update = () => {
            document.documentElement.style.setProperty(
                '--nav-height-offset',
                `${layout.scrollHeight}px`
            );
        };

        update();

        const resizeObserver = new ResizeObserver(() => {
            update();
        });

        resizeObserver.observe(layout);

        return () => {
            resizeObserver.disconnect();
        };
    }, []);

    useEffect(() => {
        updateNavHeight();

        window.addEventListener('resize', updateNavHeight);
        return () => {
            window.removeEventListener('resize', updateNavHeight);
        };
    }, [isNavOpened]);

    useEffect(() => {
        // Close nav when route changes
        if (isNavOpened) {
            setIsNavOpened(false);
        }
    }, [location]);

    /**
     * Toggle mobile navigation
     *
     * @return {undefined}
     */
    const toggleNav = () => setIsNavOpened(!isNavOpened);

    /**
     * Filter navigation items
     *
     * @param {string} search Searched string
     */
    const filterNavItems = (search) => {
        navList.current.filterItems(search);
    }

    const updateNavHeight = () => {
        const layout = document.querySelector('.lbaic-settings-nav-layout');

        if (!layout) return;

        const height = layout.offsetHeight;

        document.documentElement.style.setProperty(
            '--nav-height-offset',
            `${height}px`
        );
    };

    return (
        <div className={`lbaic-settings-nav-layout${isNavOpened ? ' lbaic-settings-nav-opened' : ''}`}>
            <Header isNavOpened={isNavOpened} toggleNav={toggleNav} />
            <div className='lbaic-settings-nav'>
                <div className="lbaic-settings-nav-overlay" onClick={toggleNav}></div>
                <div className='lbaic-settings-nav-in'>
                    <Search onSearchChange={filterNavItems}/>
                    <NavList ref={navList}/>
                </div>
            </div>
        </div>
    )
}