import {useEffect, useState} from '@wordpress/element';
import {NavLink, useSearchParams} from 'react-router-dom';
import {dropdownNavLinkClasses, getNavLinkTo, getNavLinkToParams, navLinkClasses} from "../../../../../../../../helpers";
import NavListItemHeader from "./header";

export default function NavListItem({data}) {
    const [searchParams, setSearchParams] = useSearchParams();
    const [opened, setOpened] = useState(false);

    useEffect(() => {
        // Open dropdown if one of menu is opened
        if (data.children?.length > 0) {
            const menu = searchParams.get('menu');
            if (!opened) {
                setOpened(data.children?.some(child => child.slug === menu));
            }
        }
    }, [searchParams]);

    /**
     * Toggle navigation menu item
     */
    const toggleNavItem = () => {
        if (data.disabled) {
            return;
        }
        setOpened(!opened);
    }

    /**
     * Handle nav link click
     *
     * @param {Event} e Event
     */
    const handleNavLinkClick = (e) => {
        if (data.disabled) {
            e.preventDefault();
        }
    }

    /**
     * Menu dropdown item
     *
     * @param {object} item Dropdown item data
     * @return {JSX.Element}
     */
    const dropdownItem = (item) => {
        const navLinkInfo = [item.slug];
        // Include first tab slug if exists
        if (item.tabs?.length) {
            navLinkInfo.push(item.tabs[0].slug);
        }
        return <div key={navLinkInfo.join('-')} className='lbaic-settings-nav-dropdown-item'>
            <NavLink to={getNavLinkTo(...navLinkInfo)}
                     className={(state) => dropdownNavLinkClasses(item, getNavLinkToParams(...navLinkInfo), state)}>
                <NavListItemHeader data={item} classPrefix="lbaic-settings-nav-dropdown"/>
            </NavLink>
        </div>
    }

    return <div className={`lbaic-settings-nav-item${opened ? ' opened' : ''}`}>
        {
            'children' in data ?
                <>
                    <button
                        className={`lbaic-settings-nav-item-in${data.disabled ? ' lbaic-settings-pointer-none' : ''}`}
                        onClick={toggleNavItem}
                    >
                        <NavListItemHeader data={data}/>
                        {!data.disabled && (
                            <svg className='lbaic-settings-nav-item-arrow' xmlns='http://www.w3.org/2000/svg'
                                 fill='none'
                                 viewBox='0 0 24 24'>
                                <use href='#lbaic-settings-arrow'/>
                            </svg>
                        )}
                    </button>
                    {data.children?.length > 0 && (
                        <div className='lbaic-settings-nav-dropdown'>
                            <div className='lbaic-settings-nav-dropdown-in'>
                                {data.children.map(dropdownItem)}
                            </div>
                        </div>
                    )}
                </>
                :
                <NavLink
                    to={getNavLinkTo(data.slug)}
                    className={(state) => navLinkClasses(getNavLinkToParams(data.slug), state) + (data.disabled ? ' lbaic-settings-pointer-none' : '')}
                    onClick={handleNavLinkClick}
                >
                    <NavListItemHeader data={data}/>
                </NavLink>
        }
    </div>
}