import {useRef} from "@wordpress/element";
import Copilot from "../../button/copilot";
import {fieldFocus} from "../../../../helpers";
import ContenteditableSimple from "../contenteditable/simple";

export default function DualInput({value1, value2, setValue1, setValue2, className, holderClassName, disabled, isReverse, placeholder1, placeholder2, validate1, validate2, errorMessage1, errorMessage2, autofocusOn, save, cancel, notifications}) {
    const input1Ref = useRef(null);
    const input2Ref = useRef(null);
    const edited = useRef(null);

    /**
     * Update value
     *
     * @param {string} newValue New value
     * @param {function} setValue Set value state
     * @param {function} validate Validation function
     * @param {string} initialValue Initial value
     */
    const updateValue = (newValue, setValue, validate, initialValue) => {
        if (!edited.current) {
            edited.current = true;
        }

        // New value
        if (newValue === initialValue) {
            return;
        }

        setValue(newValue);
        // Validate
        if (typeof validate === 'function') {
            validate(newValue);
        }
    }

    /**
     * Handle Enter and Escape keys click
     *
     * @param e
     */
    const handleKeyDown = (e) => {
        if (e.key === 'Enter' && !e.shiftKey) {
            if (typeof save === 'function') {
                e.preventDefault();
                if (save()) {
                    // Focus the first input
                    fieldFocus(input1Ref.current);
                }
            }
        } else if (e.key === 'Escape') {
            if (typeof cancel === 'function') {
                cancel();
            }
        }
    }

    return (
        <div
            className={`lbaic-settings-dual-input-holder${className ? ' ' + className : ''}${isReverse ? ' lbaic-settings-dual-input-reverse' : ''}${disabled ? ' lbaic-settings-dual-input-disabled' : ''}${errorMessage1 || errorMessage2 ? ' lbaic-settings-dual-input-error' : ''}`}>
            <div className='lbaic-settings-dual-input-holder-in'>
                <div className={`lbaic-settings-dual-input-holder-inner${holderClassName ? ' ' + holderClassName : ''}${value1 ? ' lbaic-settings-dual-input-has-value' : ''}`}>
                    <div className='lbaic-settings-dual-input-children lbaic-settings-dual-input-content-editable-children' tabIndex='-1'>
                        <Copilot text={value1}
                                 generated={(text) => updateValue(text, setValue1, validate1, value1)}
                                 direction='right'
                                 notifications={notifications}/>
                    </div>
                    <ContenteditableSimple
                        editableRef={input1Ref}
                        value={value1}
                        onChange={text => updateValue(text, setValue1, validate1, value1)}
                        className="lbaic-settings-dual-input lbaic-settings-dual-input-content-editable"
                        keyDown={handleKeyDown}
                        autofocus={autofocusOn === 1}/>
                    <div className='lbaic-settings-dual-input-placeholder'>
                        <span className='lbaic-settings-dual-input-placeholder-in'>{placeholder1}</span>
                    </div>
                </div>
                <div className={`lbaic-settings-dual-input-holder-inner${value2 ? ' lbaic-settings-dual-input-has-value' : ''}`}>
                    <div className='lbaic-settings-dual-input-children lbaic-settings-dual-input-content-editable-children' tabIndex='-1'>
                        <Copilot text={value2}
                                 generated={(text) => updateValue(text, setValue2, validate2, value2)}
                                 direction='right'
                                 notifications={notifications}/>
                    </div>
                    <ContenteditableSimple
                        editableRef={input2Ref}
                        value={value2}
                        onChange={text => updateValue(text, setValue2, validate2, value2)}
                        className="lbaic-settings-dual-input lbaic-settings-dual-input-content-editable"
                        keyDown={handleKeyDown}
                        autofocus={autofocusOn === 2}/>
                    <div className='lbaic-settings-dual-input-placeholder'>
                        <span className='lbaic-settings-dual-input-placeholder-in'>{placeholder2}</span>
                    </div>
                </div>
            </div>
            {!!(errorMessage1 || errorMessage2) &&
                <div className='lbaic-settings-dual-input-error-holder'>
                    <div className='lbaic-settings-dual-input-error-holder-in'>
                        <div className='lbaic-settings-dual-input-error-holder-item'>
                            {!!errorMessage1 && <p className='lbaic-settings-dual-input-error-holder-item-in'>{errorMessage1}</p>}
                        </div>
                        <div className='lbaic-settings-dual-input-error-holder-item'>
                            {!!errorMessage2 && <p className='lbaic-settings-dual-input-error-holder-item-in'>{errorMessage2}</p>}
                        </div>
                    </div>
                </div>}
        </div>
    )
}