import {__} from "@wordpress/i18n";
import ButtonPlayPause from "../play-pause";
import Button from "../_";

/**
 * PlayPauseCancel component
 * Combines a play/pause button with a cancel/stop button for process control
 *
 * @param {string} label - Label for the play/pause button
 * @param {string} icon - Icon for the play/pause button
 * @param {string} type - Button type (primary, secondary, danger)
 * @param {boolean} playing - Whether the process is currently playing
 * @param {boolean} paused - Whether the process is currently paused
 * @param {number|boolean} progress - Progress value (0-100) or false
 * @param {Function} play - Function to call when play is clicked
 * @param {Function} pause - Function to call when pause is clicked
 * @param {Function} cancel - Function to call when cancel/stop is clicked
 * @param {boolean} playPauseDisabled - Whether the play/pause button is disabled
 * @param {boolean} cancelDisabled - Whether the cancel/stop button is disabled
 * @param {boolean} canceling - Whether the cancel action is in progress
 * @param {string} cancelLabel - Custom label for cancel button (default: "Cancel" when paused, "Stop" when playing)
 * @param {string} className - Additional CSS classes
 * @param isEndOfActions
 */
export default function ButtonPlayPauseCancel({
                                                  label,
                                                  icon,
                                                  type = "primary",
                                                  playing,
                                                  paused,
                                                  progress,
                                                  play,
                                                  pause,
                                                  cancel,
                                                  playPauseDisabled,
                                                  cancelDisabled,
                                                  canceling,
                                                  cancelLabel,
                                                  className = "",
                                                  isEndOfActions
                                              }) {
    // Default cancel label based on state
    const defaultCancelLabel = paused
        ? __("Cancel", 'limb-chatbot')
        : __("Stop", 'limb-chatbot');

    // Only show cancel button when a process is active (playing or paused)
    const showCancelButton = playing || paused;

    return (
        <>
            <ButtonPlayPause
                type={type}
                label={label}
                icon={icon}
                playing={playing}
                paused={paused}
                progress={progress}
                play={play}
                pause={pause}
                disabled={playPauseDisabled}
                className={`${isEndOfActions && !showCancelButton ? ' lbaic-settings-content-footer-actions-in lbaic-settings-content-footer-actions-end' : ''}${className ? ' ' + className : ''}`}
            />
            {showCancelButton && (
                <Button
                    type="danger"
                    label={cancelLabel || defaultCancelLabel}
                    icon="stop"
                    onClick={cancel}
                    loading={canceling}
                    disabled={cancelDisabled}
                    className={isEndOfActions ? "lbaic-settings-content-footer-actions-in lbaic-settings-content-footer-actions-end" : ''}
                />
            )}
        </>
    );
}

