import {__, sprintf} from "@wordpress/i18n";
import {numberFormat} from "../../../../../helpers";

/**
 * Format cost value with appropriate decimal places
 * If the number would format to "0.00" with default 2 decimals,
 * increase precision to show the actual value
 *
 * @param {number} value The cost value to format
 * @return {string} Formatted cost string
 */
export const formatCostValue = (value) => {
    // Handle zero or invalid values
    if (!value || value === 0) {
        return numberFormat(value);
    }

    // If value is less than 0.01, it would show as "0.00" with 2 decimal places
    // So we need to calculate how many decimal places are needed
    if (value < 0.01 && value > 0) {
        // Use toFixed to avoid scientific notation, then analyze
        // Start with high precision to capture all digits
        let valueStr = value.toFixed(15);

        // Remove trailing zeros to get actual precision needed
        valueStr = valueStr.replace(/0+$/, '');

        // If it still has decimal point, calculate decimal places
        if (valueStr.includes('.')) {
            const decimalPart = valueStr.split('.')[1];
            if (decimalPart) {
                // Count decimal places needed (excluding trailing zeros already removed)
                const neededDecimals = decimalPart.length;

                // Cap at reasonable maximum (10 decimal places) to avoid extremely long numbers
                const decimals = Math.min(neededDecimals, 10);

                return numberFormat(value, decimals);
            }
        }
    }

    // Default formatting (2 decimal places, or more if numberFormat auto-adjusts based on decimal part length)
    return numberFormat(value);
};

export const UTILITIES_HEADER_KEYS = [
    __("Related to", 'limb-chatbot'),
    ''
];

export const UTILITIES = {
    'user': {
        title: __("User", 'limb-chatbot'),
    },
    'guest': {
        title: __("Guest", 'limb-chatbot'),
    },
    'input': {
        title: __("Single message (input)", 'limb-chatbot'),
    },
    'output': {
        title: __("Single message (output)", 'limb-chatbot'),
    },
};

export const CUSTOM_OPTIONS_KEYS_PREFIX = 'lbaic.settings.chatbot.costs-limits.custom-options';

export const LIMITS = {
    unitToken: 1,
    unitMessagePerChat: 2,
    unitNewChat: 4,
    unitCost: 3,
};

export const MAX_COSTS = {
    'user': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: numberFormat(0.1),
            value: 0.1
        },
        {
            label: numberFormat(0.3),
            value: 0.3
        },
        {
            label: numberFormat(0.5),
            value: 0.5
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_costs.user') || '[]').map(value => ({
            label: formatCostValue(value),
            value,
        }))
    ],
    'guest': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: numberFormat(0.01),
            value: 0.01
        },
        {
            label: numberFormat(0.05),
            value: 0.05
        },
        {
            label: numberFormat(0.1),
            value: 0.1
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_costs.guest') || '[]').map(value => ({
            label: formatCostValue(value),
            value,
        }))
    ],
    'input': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: numberFormat(0.001),
            value: 0.001
        },
        {
            label: numberFormat(0.002),
            value: 0.002
        },
        {
            label: numberFormat(0.004),
            value: 0.004
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_costs.input') || '[]').map(value => ({
            label: formatCostValue(value),
            value,
        }))
    ],
    'output': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: numberFormat(0.002),
            value: 0.002
        },
        {
            label: numberFormat(0.004),
            value: 0.004
        },
        {
            label: numberFormat(0.009),
            value: 0.009
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_costs.output') || '[]').map(value => ({
            label: formatCostValue(value),
            value,
        }))
    ],
};

export const MAX_TOKENS = {
    'user': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: numberFormat(8500, false),
            value: 8500
        },
        {
            label: numberFormat(25700, false),
            value: 25700
        },
        {
            label: numberFormat(42800, false),
            value: 42800
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_tokens.user') || '[]').map(value => ({
            label: numberFormat(value, false),
            value,
        }))
    ],
    'guest': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: numberFormat(1600, false),
            value: 1600
        },
        {
            label: numberFormat(4000, false),
            value: 4000
        },
        {
            label: numberFormat(8500, false),
            value: 8500
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_tokens.guest') || '[]').map(value => ({
            label: numberFormat(value, false),
            value,
        }))
    ],
    'input': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: numberFormat(100, false),
            value: 100
        },
        {
            label: numberFormat(200, false),
            value: 200
        },
        {
            label: numberFormat(400, false),
            value: 400
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_tokens.input') || '[]').map(value => ({
            label: numberFormat(value, false),
            value,
        }))
    ],
    'output': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: numberFormat(200, false),
            value: 200
        },
        {
            label: numberFormat(400, false),
            value: 400
        },
        {
            label: numberFormat(800, false),
            value: 800
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_tokens.output') || '[]').map(value => ({
            label: numberFormat(value, false),
            value,
        }))
    ],
};

export const MAX_MESSAGES_PER_CHAT = {
    'user': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: '20',
            value: 20
        },
        {
            label: '50',
            value: 50
        },
        {
            label: '100',
            value: 100
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_messages_per_chat.user') || '[]').map(value => ({
            label: numberFormat(value, false),
            value,
        }))
    ],
    'guest': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: '10',
            value: 10
        },
        {
            label: '20',
            value: 20
        },
        {
            label: '40',
            value: 40
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_messages_per_chat.guest') || '[]').map(value => ({
            label: numberFormat(value, false),
            value,
        }))
    ],
};

export const MAX_NEW_CHATS = {
    'user': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: '5',
            value: 5
        },
        {
            label: '10',
            value: 10
        },
        {
            label: '20',
            value: 20
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_new_chats.user') || '[]').map(value => ({
            label: numberFormat(value, false),
            value,
        }))
    ],
    'guest': [
        {
            label: __("Unlimited", 'limb-chatbot'),
            value: 0
        },
        {
            label: '2',
            value: 2
        },
        {
            label: '5',
            value: 5
        },
        {
            label: '10',
            value: 10
        },
        ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.max_new_chats.guest') || '[]').map(value => ({
            label: numberFormat(value, false),
            value,
        }))
    ],
};

export const PERIODS = [
    {
        label: __("No", 'limb-chatbot'),
        value: 0
    },
    {
        label: sprintf(__("%d days", 'limb-chatbot'), 1),
        value: 24 * 3600 // One day
    },
    {
        label: __("Week", 'limb-chatbot'),
        value: 7 * 24 * 3600
    },
    {
        label: sprintf(__("%d days", 'limb-chatbot'), 15),
        value: 15 * 24 * 3600
    },
    {
        label: __("Month", 'limb-chatbot'),
        value: 30 * 24 * 3600
    },
    ...JSON.parse(localStorage.getItem('lbaic.settings.chatbot.costs-limits.custom-options.period') || '[]').map(value => ({
        label: sprintf(__("%d days", 'limb-chatbot'), value / (24 * 3600)),
        value,
    }))
];

export const AI_PROVIDERS_PRICING = {
    'open-ai': {
        link: "https://platform.openai.com/docs/models",
    },
    'gemini': {
        link: "https://ai.google.dev/gemini-api/docs/pricing",
    },
    'deep-seek': {
        link: "https://api-docs.deepseek.com/quick_start/pricing",
    },
};