<?php
namespace LightSyncPro\Util;

if (!defined('ABSPATH')) exit;

class Image {

  /** Detect if the server can write WebP (Imagick or GD) */
  public static function canMakeWebp(): bool {
    if (class_exists('Imagick')) {
      try {
        $formats = array_map('strtoupper', \Imagick::queryFormats());
        if (in_array('WEBP', $formats, true)) return true;
      } catch (\Throwable $e) {}
    }
    return function_exists('imagewebp'); // GD fallback
  }

  /**
   * Resize to a max bounding box and prefer WebP output.
   *
   * @param string $srcTmp  absolute path to the downloaded temp image (jpg/png/etc.)
   * @param int    $maxDim  max width/height, e.g. 2048
   * @return array{path:string, mime:string}
   */
  public static function prepareForMedia(string $srcTmp, int $maxDim = 2048): array {
    $ft = wp_check_filetype($srcTmp);
    $origMime = $ft['type'] ?? 'image/jpeg';

    $editor = wp_get_image_editor($srcTmp);
    if (is_wp_error($editor)) {
      return ['path' => $srcTmp, 'mime' => $origMime];
    }

    // Resize (no crop)
    $editor->resize($maxDim, $maxDim, false);

    // Prefer WebP
    if (self::canMakeWebp()) {
      $dest = preg_replace('/\.[^.]+$/', '.webp', $srcTmp);
      $saved = $editor->save($dest, 'image/webp');
      if (!is_wp_error($saved) && !empty($saved['path'])) {
        return ['path' => $saved['path'], 'mime' => 'image/webp'];
      }
      // fall back to original format if WebP save fails
    }

    // Save back in original mime/extension
    $saved = $editor->save();
    if (!is_wp_error($saved) && !empty($saved['path'])) {
      $type = wp_check_filetype($saved['path'])['type'] ?? $origMime;
      return ['path' => $saved['path'], 'mime' => $type];
    }

    // Last resort
    return ['path' => $srcTmp, 'mime' => $origMime];
  }
}
