<?php
namespace LightSyncPro\OAuth;

use LightSyncPro\Admin\Admin;
use LightSyncPro\Util\Crypto;
use LightSyncPro\Util\Logger;

if ( ! defined( 'ABSPATH' ) ) exit;


/**
 * OpenRouter OAuth handler for LightSync Pro
 * 
 * Handles broker-based OAuth PKCE flow for OpenRouter AI.
 * 
 * Follows the SAME pattern as Canva/Figma/Dropbox:
 * - Broker handles OAuth and stores the persistent credential (API key)
 * - Plugin receives only a broker JWT during pickup
 * - Plugin calls broker's /token/refresh to get a short-lease API key
 * - Short-lease key is stored as a WordPress transient (auto-expires in 1hr)
 * - When transient expires, plugin requests a fresh key from broker
 * - No long-lived credentials in the WordPress database
 * 
 * Flow:
 * 1. User clicks "Connect OpenRouter" → redirected to broker
 * 2. Broker redirects to openrouter.ai/auth (PKCE)
 * 3. User authenticates → code returned to broker
 * 4. Broker exchanges code for API key, encrypts & stores on broker
 * 5. User redirected back to WP with ?lsp_openrouter_connected=1
 * 6. Plugin picks up broker JWT (no API key in response)
 * 7. Generation: plugin calls broker /token/refresh → gets short-lease key → calls OpenRouter directly
 * 8. Models: plugin → OpenRouter public API (no auth needed)
 */
class OpenRouterOAuth {
    const BROKER_URL = 'https://lightsyncpro.com';

    public static function init() {
        // Handle ?lsp_openrouter_connected=1 callback
        add_action('admin_init', [__CLASS__, 'handle_connection_callback']);
    }

    /**
     * Check if OpenRouter is connected
     */
    public static function is_connected() {
        $o = Admin::get_opt();
        return !empty($o['openrouter_broker_token_enc']);
    }

    /**
     * Get OpenRouter auth URL (starts OAuth PKCE flow via broker)
     */
    public static function auth_url() {
        $state  = wp_create_nonce('lightsync_openrouter_oauth_state');
        $site   = home_url();
        $return = admin_url('admin.php?page=lightsyncpro&source=ai');
        
        return self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_openrouter_connect_start'
             . '&site='   . rawurlencode($site)
             . '&state='  . rawurlencode($state)
             . '&return=' . rawurlencode($return);
    }

    /**
     * Handle the connection callback (?lsp_openrouter_connected=1)
     * Pick up the broker JWT from the broker
     */
    public static function handle_connection_callback() {
        if (!isset($_GET['lsp_openrouter_connected']) || $_GET['lsp_openrouter_connected'] !== '1') {
            return;
        }

        error_log('[LSP OpenRouter] === Connection callback started ===');
        error_log('[LSP OpenRouter] GET params: ' . print_r(array_keys($_GET), true));

        // Verify state nonce
        $state = isset($_GET['state']) ? sanitize_text_field($_GET['state']) : '';
        error_log('[LSP OpenRouter] State param: ' . ($state ? substr($state, 0, 10) . '...' : '(empty)'));
        
        if (!wp_verify_nonce($state, 'lightsync_openrouter_oauth_state')) {
            error_log('[LSP OpenRouter] ✗ Nonce verification FAILED — this is likely the issue');
            error_log('[LSP OpenRouter] Proceeding anyway with pickup (state still passed to broker for its own verification)');
            // Don't return — try the pickup anyway. The broker validates independently.
        } else {
            error_log('[LSP OpenRouter] ✓ Nonce verified OK');
        }

        // Pick up the broker token
        $site = home_url();
        $pickup_url = self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_openrouter_install_pickup'
                    . '&site=' . rawurlencode($site)
                    . '&state=' . rawurlencode($state);

        error_log('[LSP OpenRouter] Pickup URL: ' . preg_replace('/state=[^&]+/', 'state=***', $pickup_url));

        $response = wp_remote_get($pickup_url, [
            'timeout' => 20,
            'headers' => ['Accept' => 'application/json'],
        ]);

        if (is_wp_error($response)) {
            error_log('[LSP OpenRouter] ✗ Pickup HTTP error: ' . $response->get_error_message());
            set_transient('lsp_openrouter_notice', 'error:' . $response->get_error_message(), 60);
            return;
        }

        $http_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        error_log('[LSP OpenRouter] Pickup response: HTTP ' . $http_code . ' — ' . substr($body, 0, 500));

        $data = json_decode($body, true);

        if (empty($data['success']) || empty($data['data']['broker_token'])) {
            $error = $data['data']['error'] ?? ($body ?: 'Unknown error');
            error_log('[LSP OpenRouter] ✗ Pickup failed: ' . $error);
            set_transient('lsp_openrouter_notice', 'error:' . $error, 60);
            return;
        }

        // Encrypt and store the broker token
        $broker_token = $data['data']['broker_token'];
        $encrypted = Crypto::enc($broker_token);

        $opts = [
            'openrouter_broker_token_enc' => $encrypted,
            'openrouter_connected_at'     => time(),
        ];

        // Clean up any legacy stored API key from pre-refresh architecture
        $opts['openrouter_api_key_enc'] = '';

        Admin::set_opt($opts);

        // Clear any stale short-lease key transient
        delete_transient('lsp_openrouter_access_token');

        error_log('[LSP OpenRouter] ✓ Connected — broker token stored, API key stays on broker');
        set_transient('lsp_openrouter_notice', 'success', 60);

        // Redirect to clean URL (removes lsp_openrouter_connected params)
        $clean_url = admin_url('admin.php?page=lightsyncpro&source=ai&lsp_or_done=1');
        wp_safe_redirect($clean_url);
        exit;
    }

    /**
     * Get the decrypted broker token for API calls
     * 
     * @return string|WP_Error Broker JWT or error
     */
    public static function get_broker_token() {
        $o = Admin::get_opt();
        $enc = $o['openrouter_broker_token_enc'] ?? '';
        
        if (!$enc) {
            return new \WP_Error('no_broker', 'OpenRouter not connected. Click Connect to OpenRouter.');
        }

        $broker = Crypto::dec($enc);
        if (!$broker) {
            return new \WP_Error('bad_broker', 'Stored OpenRouter broker token not readable.');
        }

        return $broker;
    }

    /**
     * Get the OpenRouter API key for direct calls (short-lease from broker)
     * 
     * Mirrors the Canva/Figma/Dropbox token refresh pattern:
     * 1. Check WordPress transient (cached short-lease key, ≤1 hour)
     * 2. If expired, call broker's /openrouter/token/refresh endpoint
     * 3. Broker decrypts stored API key and returns it with TTL
     * 4. Store in transient (auto-expires)
     * 
     * Result: no long-lived credentials in the WordPress database.
     *
     * @return string|WP_Error API key or error
     */
    public static function get_api_key() {
        // 1. Check transient first (cached short-lease key)
        $cached = get_transient('lsp_openrouter_access_token');
        if ($cached) {
            return $cached;
        }

        // 2. Get broker token for authentication
        $broker_token = self::get_broker_token();
        if (is_wp_error($broker_token)) {
            return $broker_token;
        }

        // 3. Request fresh short-lease key from broker
        $resp = wp_remote_post(
            self::BROKER_URL . '/wp-json/lsp-broker/v1/openrouter/token/refresh',
            [
                'timeout' => 15,
                'headers' => [
                    'Authorization' => 'Bearer ' . $broker_token,
                    'Accept'        => 'application/json',
                ],
            ]
        );

        if (is_wp_error($resp)) {
            return new \WP_Error('token_refresh_failed', 'Could not refresh OpenRouter key: ' . $resp->get_error_message());
        }

        $http_code = wp_remote_retrieve_response_code($resp);
        $data = json_decode(wp_remote_retrieve_body($resp), true);

        // Handle broker errors
        if ($http_code !== 200 || empty($data['success']) || empty($data['data']['access_token'])) {
            $error = $data['data']['error'] ?? 'Unknown error refreshing OpenRouter key';
            $reconnect = !empty($data['data']['reconnect']);

            Logger::debug('[LSP OpenRouter] Token refresh failed: ' . $error . ($reconnect ? ' (reconnect needed)' : ''));

            return new \WP_Error(
                $reconnect ? 'openrouter_reconnect' : 'token_refresh_failed',
                $error
            );
        }

        // 4. Store as transient — auto-expires, never persists in options table
        $key = $data['data']['access_token'];
        $ttl = (int)($data['data']['expires_in'] ?? 3600);

        // Refresh 60s early to avoid hitting OpenRouter with a stale key
        set_transient('lsp_openrouter_access_token', $key, max($ttl - 60, 300));

        Logger::debug('[LSP OpenRouter] Short-lease key refreshed, TTL=' . $ttl . 's');

        return $key;
    }

    /**
     * Disconnect OpenRouter
     */
    public static function disconnect() {
        // Notify broker to clean up stored credentials
        $broker_token = self::get_broker_token();
        if (!is_wp_error($broker_token)) {
            wp_remote_post(
                self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_openrouter_disconnect',
                [
                    'timeout' => 10,
                    'headers' => [
                        'Authorization' => 'Bearer ' . $broker_token,
                        'Accept'        => 'application/json',
                    ],
                ]
            );
        }

        Admin::set_opt([
            'openrouter_broker_token_enc' => '',
            'openrouter_api_key_enc'      => '',  // Clear any legacy stored key
            'openrouter_connected_at'     => 0,
        ]);

        // Clear short-lease key transient and cached models
        delete_transient('lsp_openrouter_access_token');
        delete_transient('lsp_openrouter_models_v3');

        Logger::debug('[LSP OpenRouter] Disconnected – broker token and transients cleared');
    }

    /**
     * Check connection status with broker
     * 
     * @return array{connected: bool, error?: string}
     */
    public static function status() {
        $broker_token = self::get_broker_token();
        if (is_wp_error($broker_token)) {
            return ['connected' => false, 'error' => $broker_token->get_error_message()];
        }

        $resp = wp_remote_get(
            self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_openrouter_status',
            [
                'timeout' => 10,
                'headers' => [
                    'Authorization' => 'Bearer ' . $broker_token,
                    'Accept'        => 'application/json',
                ],
            ]
        );

        if (is_wp_error($resp)) {
            return ['connected' => false, 'error' => $resp->get_error_message()];
        }

        $data = json_decode(wp_remote_retrieve_body($resp), true);
        
        if (!empty($data['success']) && !empty($data['data']['connected'])) {
            return ['connected' => true];
        }

        return ['connected' => false, 'error' => $data['data']['error'] ?? 'Unknown status'];
    }
}
