<?php
namespace LightSyncPro\OAuth;

use LightSyncPro\Admin\Admin;
use LightSyncPro\Util\Crypto;
use LightSyncPro\Util\Logger;



class OAuth {
    const NS = 'lightsyncpro/v1';
    const CB = 'oauth-callback';

    // NEW: hourly token guard hook
    const CRON_TOKEN_GUARD = 'lightsync_token_guard';

    public static function init(){
        add_action('rest_api_init', [ __CLASS__, 'rest' ] );

        // NEW: ensure the guard is scheduled + handler
        add_action('init', [ __CLASS__, 'ensure_guard_scheduled' ]);
        add_action(self::CRON_TOKEN_GUARD, [ __CLASS__, 'cron_token_guard' ]);
    }

    public static function is_connected(){
        $o = Admin::get_opt();
        return !empty($o['broker_token_enc']);
    }

    public static function redirect_uri(){
        return home_url('/wp-json/'.self::NS.'/'.self::CB);
    }

    public static function auth_url($ignored = null){
        $state  = wp_create_nonce('lightsync_oauth_state');
        $site   = home_url();
        $return = admin_url('admin.php?page=lightsyncpro');
        return 'https://lightsyncpro.com/wp-admin/admin-ajax.php?action=lsp_broker_connect_start'
             . '&site='   . rawurlencode($site)
             . '&state='  . rawurlencode($state)
             . '&return=' . rawurlencode($return);
    }

    public static function rest(){
        register_rest_route(self::NS, '/ping', [
            'methods'=>'GET',
            'permission_callback'=>'__return_true',
            'callback'=>function(){ return ['ok'=>true]; }
        ]);
    }

    /** Keep token valid: refresh if missing or <60s left. */
    public static function ensure_token(){
        $o   = Admin::get_opt();
        $exp = (int)($o['token_expires'] ?? 0);
        $now = time();

        if (!empty($o['access_token']) && ($exp - $now) > 60) {
            return true;
        }

        $enc = $o['broker_token_enc'] ?? '';
        if (!$enc) return new \WP_Error('no_broker', 'Not connected. Click Connect to Adobe.');

        $broker = Crypto::dec($enc);
        if (!$broker) return new \WP_Error('bad_broker', 'Stored broker token not readable.');

        $resp = wp_remote_post(
            'https://lightsyncpro.com/wp-admin/admin-ajax.php?action=lsp_broker_token_refresh',
            [
                'timeout' => 20,
                'headers' => [
                    'Authorization' => 'Bearer ' . $broker,
                    'Accept'        => 'application/json',
                ],
            ]
        );
        if (is_wp_error($resp)) return $resp;

        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);
        $data = json_decode($body, true);

        $ok      = is_array($data) && array_key_exists('success', $data) ? (bool)$data['success'] : true;
        $payload = (is_array($data) && array_key_exists('data', $data)) ? $data['data'] : $data;

        if (!$ok) {
            $msg = (is_array($payload) && !empty($payload['error'])) ? $payload['error'] : 'broker error';
            return new \WP_Error('no_access', 'Broker refused: ' . $msg);
        }

        if (empty($payload['access_token'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                Logger::debug('[LSP] broker refresh code=' . $code . ' body=' . $body);
            }
            return new \WP_Error('no_access', 'Could not obtain access token from broker');
        }

        // Cushion expiry so we refresh a bit early (avoid edge 401s)
        $expires_in = (int)($payload['expires_in'] ?? 3600);
        $cushion    = 30; // seconds
        Admin::set_opt([
            'access_token'  => $payload['access_token'],
            'token_expires' => $now + max(300, $expires_in - $cushion),
        ]);

        return true;
    }

    /** Proactively refresh when <5 minutes remain; otherwise no-op. */
    public static function maybe_refresh() {
        $o   = Admin::get_opt();
        $exp = (int)($o['token_expires'] ?? 0);
        if (!$exp) return self::ensure_token(); // handles not-connected case
        if (($exp - time()) < 300) {
            return self::ensure_token();
        }
        return true;
    }

  public static function disconnect() {

    // Clear auth-related options
    Admin::set_opt([
        'access_token'     => '',
        'token_expires'    => 0,
        'broker_token_enc' => '',
        'client_id'        => '',
        'catalog_id'       => '',
        'album_ids'        => [],
    ]);

    // ✅ Purge cached albums (all catalogs)
    global $wpdb;
    $like = $wpdb->esc_like('_transient_lightsync_albums_') . '%';

   // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->query(
        $wpdb->prepare(
            "DELETE FROM {$wpdb->options}
             WHERE option_name LIKE %s
                OR option_name LIKE %s",
            $like,
            str_replace('_transient_', '_transient_timeout_', $like)
        )
    );

    if (defined('WP_DEBUG') && WP_DEBUG) {
        Logger::debug('[LSP] OAuth::disconnect() – tokens cleared + album cache purged');
    }
}


    public static function headers() {
        $o = Admin::get_opt();
        $apiKey = defined('LIGHTSYNC_ADOBE_CLIENT_ID')
            ? LIGHTSYNC_ADOBE_CLIENT_ID
            : (string) ($o['client_id'] ?? '');

        if ( defined('WP_DEBUG') && WP_DEBUG ) {
            Logger::debug('[LSP] headers apiKey=' . $apiKey);
        }

        return [
            'Authorization' => 'Bearer ' . ($o['access_token'] ?? ''),
            'x-api-key'     => $apiKey,
            'Accept'        => 'application/json',
        ];
    }

    // ---------- NEW: guard scheduler + handler ----------

    public static function ensure_guard_scheduled(){
        if (!wp_next_scheduled(self::CRON_TOKEN_GUARD)) {
            // First run in ~1 minute, then hourly
            wp_schedule_event(time() + 60, 'hourly', self::CRON_TOKEN_GUARD);
        }
    }

    public static function cron_token_guard(){
        $ok = self::maybe_refresh();
        if (is_wp_error($ok) && defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP] token_guard failed: ' . $ok->get_error_message());
        }
    }
}

