<?php
namespace LightSyncPro\OAuth;

use LightSyncPro\Admin\Admin;
use LightSyncPro\Util\Crypto;
use LightSyncPro\Util\Logger;


/**
 * Dropbox OAuth handler for LightSync Pro
 * Handles broker-based OAuth flow and token management for Dropbox
 */
class DropboxOAuth {
    const BROKER_URL = 'https://lightsyncpro.com';
    const CRON_DROPBOX_TOKEN_GUARD = 'lightsync_dropbox_token_guard';

    public static function init() {
        // Handle ?lsp_dropbox_connected=1 callback
        add_action('admin_init', [__CLASS__, 'handle_connection_callback']);
        
        // Token refresh guard
        add_action('init', [__CLASS__, 'ensure_guard_scheduled']);
        add_action(self::CRON_DROPBOX_TOKEN_GUARD, [__CLASS__, 'cron_token_guard']);
    }

    /**
     * Check if Dropbox is connected
     */
    public static function is_connected() {
        $o = Admin::get_opt();
        return !empty($o['dropbox_broker_token_enc']);
    }

    /**
     * Get Dropbox auth URL (starts OAuth flow via broker)
     */
    public static function auth_url() {
        $state  = wp_create_nonce('lightsync_dropbox_oauth_state');
        $site   = home_url();
        $return = admin_url('admin.php?page=lightsyncpro&tab=source');
        
        return self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_dropbox_connect_start'
             . '&site='   . rawurlencode($site)
             . '&state='  . rawurlencode($state)
             . '&return=' . rawurlencode($return);
    }

    /**
     * Handle the connection callback (?lsp_dropbox_connected=1)
     */
    public static function handle_connection_callback() {
        if (!isset($_GET['lsp_dropbox_connected']) || $_GET['lsp_dropbox_connected'] !== '1') {
            return;
        }

        // Verify state nonce
        $state = isset($_GET['state']) ? sanitize_text_field($_GET['state']) : '';
        if (!wp_verify_nonce($state, 'lightsync_dropbox_oauth_state')) {
            Logger::debug('[LSP Dropbox] Invalid state nonce on callback');
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error"><p>Dropbox connection failed: Invalid security token. Please try again.</p></div>';
            });
            return;
        }

        // Pick up the broker token
        $site = home_url();
        $pickup_url = self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_dropbox_install_pickup'
                    . '&site=' . rawurlencode($site)
                    . '&state=' . rawurlencode($state);

        $response = wp_remote_get($pickup_url, [
            'timeout' => 20,
            'headers' => ['Accept' => 'application/json'],
        ]);

        if (is_wp_error($response)) {
            Logger::debug('[LSP Dropbox] Pickup request failed: ' . $response->get_error_message());
            add_action('admin_notices', function() use ($response) {
                echo '<div class="notice notice-error"><p>Dropbox connection failed: ' . esc_html($response->get_error_message()) . '</p></div>';
            });
            return;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (empty($data['success']) || empty($data['data']['broker_token'])) {
            $error = $data['data']['error'] ?? 'Unknown error';
            Logger::debug('[LSP Dropbox] Pickup failed: ' . $error);
            add_action('admin_notices', function() use ($error) {
                echo '<div class="notice notice-error"><p>Dropbox connection failed: ' . esc_html($error) . '</p></div>';
            });
            return;
        }

        // Encrypt and store the broker token
        $broker_token = $data['data']['broker_token'];
        $encrypted = Crypto::enc($broker_token);

        Admin::set_opt([
            'dropbox_broker_token_enc' => $encrypted,
            'dropbox_connected_at'     => time(),
        ]);

        Logger::debug('[LSP Dropbox] Successfully connected and stored broker token');

        add_action('admin_notices', function() {
            echo '<div class="notice notice-success"><p>✓ Successfully connected to Dropbox!</p></div>';
        });
    }

    /**
     * Ensure access token is valid, refresh if needed
     */
    public static function ensure_token() {
        $o   = Admin::get_opt();
        $exp = (int)($o['dropbox_token_expires'] ?? 0);
        $now = time();

        // If we have a valid token with >60s remaining, we're good
        if (!empty($o['dropbox_access_token']) && ($exp - $now) > 60) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                Logger::debug('[LSP Dropbox] ensure_token: using cached token, expires in ' . ($exp - $now) . 's');
            }
            return true;
        }

        // Need to refresh from broker
        $enc = $o['dropbox_broker_token_enc'] ?? '';
        if (!$enc) {
            Logger::debug('[LSP Dropbox] ensure_token: no broker token stored');
            return new \WP_Error('no_broker', 'Dropbox not connected. Click Connect to Dropbox.');
        }

        $broker = Crypto::dec($enc);
        if (!$broker) {
            Logger::debug('[LSP Dropbox] ensure_token: failed to decrypt broker token');
            return new \WP_Error('bad_broker', 'Stored Dropbox broker token not readable.');
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Dropbox] ensure_token: refreshing from broker...');
        }

        $resp = wp_remote_post(
            self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_dropbox_token_refresh',
            [
                'timeout' => 20,
                'headers' => [
                    'Authorization' => 'Bearer ' . $broker,
                    'Accept'        => 'application/json',
                ],
            ]
        );

        if (is_wp_error($resp)) {
            Logger::debug('[LSP Dropbox] ensure_token: broker request failed - ' . $resp->get_error_message());
            return $resp;
        }

        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);
        $data = json_decode($body, true);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Dropbox] ensure_token: broker response code=' . $code);
        }

        $ok      = is_array($data) && array_key_exists('success', $data) ? (bool)$data['success'] : true;
        $payload = (is_array($data) && array_key_exists('data', $data)) ? $data['data'] : $data;

        if (!$ok) {
            $msg = (is_array($payload) && !empty($payload['error'])) ? $payload['error'] : 'broker error';
            Logger::debug('[LSP Dropbox] ensure_token: broker returned error - ' . $msg);
            
            if (!empty($payload['reconnect'])) {
                self::disconnect();
                return new \WP_Error('reconnect_required', 'Dropbox authorization expired. Please reconnect.');
            }
            
            return new \WP_Error('no_access', 'Dropbox broker refused: ' . $msg);
        }

        if (empty($payload['access_token'])) {
            Logger::debug('[LSP Dropbox] ensure_token: no access_token in response, code=' . $code . ' body=' . $body);
            return new \WP_Error('no_access', 'Could not obtain Dropbox access token from broker');
        }

        // Dropbox tokens expire in 4 hours (14400 seconds)
        $expires_in = (int)($payload['expires_in'] ?? 14400);
        $cushion    = 60;
        Admin::set_opt([
            'dropbox_access_token'  => $payload['access_token'],
            'dropbox_token_expires' => $now + max(300, $expires_in - $cushion),
        ]);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Dropbox] ensure_token: got new token, expires in ' . $expires_in . 's');
        }

        return true;
    }

    /**
     * Proactively refresh when <5 minutes remain
     */
    public static function maybe_refresh() {
        $o   = Admin::get_opt();
        $exp = (int)($o['dropbox_token_expires'] ?? 0);
        if (!$exp) return self::ensure_token();
        if (($exp - time()) < 300) {
            return self::ensure_token();
        }
        return true;
    }

    /**
     * Disconnect Dropbox
     */
    public static function disconnect() {
        Admin::set_opt([
            'dropbox_access_token'      => '',
            'dropbox_token_expires'     => 0,
            'dropbox_broker_token_enc'  => '',
            'dropbox_connected_at'      => 0,
            'dropbox_selected_folders'  => [],
        ]);

        global $wpdb;
        $like = $wpdb->esc_like('_transient_lightsync_dropbox_') . '%';
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options}
                 WHERE option_name LIKE %s
                    OR option_name LIKE %s",
                $like,
                str_replace('_transient_', '_transient_timeout_', $like)
            )
        );

        Logger::debug('[LSP Dropbox] Disconnected – tokens cleared + cache purged');
    }

    /**
     * Get headers for Dropbox API calls
     */
    public static function headers() {
        $o = Admin::get_opt();
        return [
            'Authorization' => 'Bearer ' . ($o['dropbox_access_token'] ?? ''),
        ];
    }

    /**
     * Schedule token guard cron
     */
    public static function ensure_guard_scheduled() {
        if (!self::is_connected()) return;
        
        if (!wp_next_scheduled(self::CRON_DROPBOX_TOKEN_GUARD)) {
            wp_schedule_event(time() + 120, 'hourly', self::CRON_DROPBOX_TOKEN_GUARD);
        }
    }

    /**
     * Cron handler for token refresh
     */
    public static function cron_token_guard() {
        $ok = self::maybe_refresh();
        if (is_wp_error($ok) && defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Dropbox] token_guard failed: ' . $ok->get_error_message());
        }
    }

    /**
     * List folder contents from Dropbox
     */
    public static function list_folder($path = '', $cursor = null) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            Logger::debug('[LSP Dropbox] list_folder ensure_token failed: ' . $token_ok->get_error_message());
            return $token_ok;
        }

        if ($cursor) {
            $url = 'https://api.dropboxapi.com/2/files/list_folder/continue';
            $body = ['cursor' => $cursor];
        } else {
            $url = 'https://api.dropboxapi.com/2/files/list_folder';
            $body = [
                'path' => $path ?: '',
                'recursive' => false,
                'include_media_info' => true,
                'include_deleted' => false,
                'include_has_explicit_shared_members' => false,
                'include_mounted_folders' => true,
                'limit' => 500,
            ];
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Dropbox] list_folder calling API: ' . $url . ' with path: "' . ($path ?: '(root)') . '"');
        }

        $response = wp_remote_post($url, [
            'timeout' => 30,
            'headers' => array_merge(self::headers(), [
                'Content-Type' => 'application/json',
            ]),
            'body' => wp_json_encode($body),
        ]);

        if (is_wp_error($response)) {
            Logger::debug('[LSP Dropbox] list_folder wp_remote_post error: ' . $response->get_error_message());
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Dropbox] list_folder failed: ' . $code . ' - ' . $body);
            
            // Parse error for better message
            $error_data = json_decode($body, true);
            $error_msg = 'Dropbox API error: ' . $code;
            if (!empty($error_data['error_summary'])) {
                $error_msg = $error_data['error_summary'];
            } elseif (!empty($error_data['error']['.tag'])) {
                $error_msg = $error_data['error']['.tag'];
            }
            
            return new \WP_Error('dropbox_api_error', $error_msg);
        }

        $data = json_decode($body, true);
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Dropbox] list_folder success, entries: ' . count($data['entries'] ?? []));
        }

        return $data;
    }

    /**
     * Get ALL contents of a folder with pagination
     */
    public static function list_folder_all($path = '') {
        $all_entries = [];
        $cursor = null;
        $has_more = true;

        while ($has_more) {
            $result = self::list_folder($path, $cursor);
            
            if (is_wp_error($result)) {
                if (!empty($all_entries)) {
                    return ['entries' => $all_entries];
                }
                return $result;
            }

            if (!empty($result['entries'])) {
                $all_entries = array_merge($all_entries, $result['entries']);
            }

            $has_more = !empty($result['has_more']);
            $cursor = $result['cursor'] ?? null;
        }

        return ['entries' => $all_entries];
    }

    /**
     * Get file metadata
     */
    public static function get_metadata($path) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $response = wp_remote_post('https://api.dropboxapi.com/2/files/get_metadata', [
            'timeout' => 20,
            'headers' => array_merge(self::headers(), [
                'Content-Type' => 'application/json',
            ]),
            'body' => wp_json_encode([
                'path' => $path,
                'include_media_info' => true,
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Dropbox] get_metadata failed: ' . $code . ' - ' . $body);
            return new \WP_Error('dropbox_api_error', 'Failed to get metadata: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Get folder changes since last check using cursor
     * Much faster than full listing for folders with many files
     * 
     * @param string $path Folder path
     * @param bool $force_full Force a full listing (ignore saved cursor)
     * @return array ['entries' => [...], 'cursor' => '...', 'has_more' => bool, 'is_full_sync' => bool]
     */
    public static function list_folder_changes($path, $force_full = false) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        // Get saved cursor for this path
        $cursors = get_option('lightsync_dropbox_cursors', []);
        $cursor_key = md5($path);
        $saved_cursor = $cursors[$cursor_key] ?? null;
        
        $is_full_sync = false;
        $all_entries = [];
        $new_cursor = null;
        
        if ($saved_cursor && !$force_full) {
            // Try to use saved cursor for incremental sync
            Logger::debug('[LSP Dropbox] Using saved cursor for ' . $path);
            
            $result = self::list_folder_continue($saved_cursor);
            
            if (is_wp_error($result)) {
                $error_msg = $result->get_error_message();
                // Cursor expired or invalid - fall back to full sync
                if (strpos($error_msg, 'reset') !== false || strpos($error_msg, 'cursor') !== false || strpos($error_msg, 'expired') !== false) {
                    Logger::debug('[LSP Dropbox] Cursor expired/invalid, doing full sync');
                    $saved_cursor = null;
                    // Fall through to full sync below
                } else {
                    return $result;
                }
            } else {
                // Cursor worked - collect all changes
                $all_entries = $result['entries'] ?? [];
                $new_cursor = $result['cursor'] ?? null;
                $has_more = !empty($result['has_more']);
                
                // Handle pagination of changes
                while ($has_more && $new_cursor) {
                    $more = self::list_folder_continue($new_cursor);
                    if (is_wp_error($more)) {
                        break;
                    }
                    $all_entries = array_merge($all_entries, $more['entries'] ?? []);
                    $new_cursor = $more['cursor'] ?? null;
                    $has_more = !empty($more['has_more']);
                }
                
                // Save the new cursor
                if ($new_cursor) {
                    $cursors[$cursor_key] = $new_cursor;
                    update_option('lightsync_dropbox_cursors', $cursors, false);
                }
                
                Logger::debug('[LSP Dropbox] Incremental sync found ' . count($all_entries) . ' changed entries');
                
                return [
                    'entries' => $all_entries,
                    'cursor' => $new_cursor,
                    'is_full_sync' => false,
                    'changes_only' => true,
                ];
            }
        }
        
        // Full sync - either no cursor or cursor was invalid
        Logger::debug('[LSP Dropbox] Doing full folder listing for ' . $path);
        $is_full_sync = true;
        
        $result = self::list_folder($path);
        if (is_wp_error($result)) {
            return $result;
        }
        
        $all_entries = $result['entries'] ?? [];
        $new_cursor = $result['cursor'] ?? null;
        $has_more = !empty($result['has_more']);
        
        // Handle pagination
        while ($has_more && $new_cursor) {
            $more = self::list_folder($path, $new_cursor);
            if (is_wp_error($more)) {
                break;
            }
            $all_entries = array_merge($all_entries, $more['entries'] ?? []);
            $new_cursor = $more['cursor'] ?? null;
            $has_more = !empty($more['has_more']);
        }
        
        // Save cursor for next time
        if ($new_cursor) {
            $cursors[$cursor_key] = $new_cursor;
            update_option('lightsync_dropbox_cursors', $cursors, false);
            Logger::debug('[LSP Dropbox] Saved cursor for ' . $path);
        }
        
        return [
            'entries' => $all_entries,
            'cursor' => $new_cursor,
            'is_full_sync' => true,
            'changes_only' => false,
        ];
    }
    
    /**
     * Continue listing from cursor (for changes or pagination)
     */
    public static function list_folder_continue($cursor) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $response = wp_remote_post('https://api.dropboxapi.com/2/files/list_folder/continue', [
            'timeout' => 30,
            'headers' => array_merge(self::headers(), [
                'Content-Type' => 'application/json',
            ]),
            'body' => wp_json_encode(['cursor' => $cursor]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            $error_data = json_decode($body, true);
            $error_msg = 'Dropbox API error: ' . $code;
            if (!empty($error_data['error_summary'])) {
                $error_msg = $error_data['error_summary'];
            }
            return new \WP_Error('dropbox_cursor_error', $error_msg);
        }

        return json_decode($body, true);
    }
    
    /**
     * Clear saved cursor for a folder (forces full sync next time)
     */
    public static function clear_cursor($path) {
        $cursors = get_option('lightsync_dropbox_cursors', []);
        $cursor_key = md5($path);
        unset($cursors[$cursor_key]);
        update_option('lightsync_dropbox_cursors', $cursors, false);
    }
    
    /**
     * Clear all saved cursors
     */
    public static function clear_all_cursors() {
        delete_option('lightsync_dropbox_cursors');
    }

    /**
     * Get thumbnail for an image file
     * Returns base64-encoded image data
     */
    public static function get_thumbnail($path, $size = 'w256h256') {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        // Dropbox thumbnail API uses Dropbox-API-Arg header
        $args = wp_json_encode([
            'resource' => ['.tag' => 'path', 'path' => $path],
            'format'   => ['.tag' => 'jpeg'],
            'size'     => ['.tag' => $size],
            'mode'     => ['.tag' => 'bestfit'],
        ]);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Dropbox] get_thumbnail calling API for: ' . $path);
        }

        $response = wp_remote_post('https://content.dropboxapi.com/2/files/get_thumbnail_v2', [
            'timeout' => 30,
            'headers' => [
                'Authorization'   => 'Bearer ' . (Admin::get_opt()['dropbox_access_token'] ?? ''),
                'Dropbox-API-Arg' => $args,
                'Content-Type'    => 'application/octet-stream',
            ],
            'body' => '',
        ]);

        if (is_wp_error($response)) {
            Logger::debug('[LSP Dropbox] get_thumbnail wp_error: ' . $response->get_error_message());
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Dropbox] get_thumbnail response code: ' . $code . ', body length: ' . strlen($body));
        }

        if ($code !== 200) {
            Logger::debug('[LSP Dropbox] get_thumbnail failed: ' . $code . ' - ' . substr($body, 0, 500));
            return new \WP_Error('dropbox_api_error', 'Thumbnail failed: ' . $code);
        }

        // Check if we got actual image data
        if (strlen($body) < 100) {
            Logger::debug('[LSP Dropbox] get_thumbnail got empty/small response');
            return new \WP_Error('dropbox_api_error', 'Empty thumbnail response');
        }
        
        // Return as base64 data URL
        return 'data:image/jpeg;base64,' . base64_encode($body);
    }

    /**
     * Get large rendition of an image (2048x1536 max)
     * Used as fallback for large files that timeout during full download
     * Returns raw image bytes (not base64)
     */
    public static function get_large_rendition($path) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        // Use largest available size: w2048h1536
        $args = wp_json_encode([
            'resource' => ['.tag' => 'path', 'path' => $path],
            'format'   => ['.tag' => 'jpeg'],
            'size'     => ['.tag' => 'w2048h1536'],
            'mode'     => ['.tag' => 'bestfit'],
        ]);

        Logger::debug('[LSP Dropbox] get_large_rendition calling API for: ' . $path);

        $response = wp_remote_post('https://content.dropboxapi.com/2/files/get_thumbnail_v2', [
            'timeout' => 60, // Shorter timeout since renditions are smaller
            'headers' => [
                'Authorization'   => 'Bearer ' . (Admin::get_opt()['dropbox_access_token'] ?? ''),
                'Dropbox-API-Arg' => $args,
                'Content-Type'    => 'application/octet-stream',
            ],
            'body' => '',
        ]);

        if (is_wp_error($response)) {
            Logger::debug('[LSP Dropbox] get_large_rendition wp_error: ' . $response->get_error_message());
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        Logger::debug('[LSP Dropbox] get_large_rendition response code: ' . $code . ', body length: ' . strlen($body));

        if ($code !== 200) {
            Logger::debug('[LSP Dropbox] get_large_rendition failed: ' . $code . ' - ' . substr($body, 0, 500));
            return new \WP_Error('dropbox_api_error', 'Rendition failed: ' . $code);
        }

        // Check if we got actual image data
        if (strlen($body) < 1000) {
            Logger::debug('[LSP Dropbox] get_large_rendition got empty/small response');
            return new \WP_Error('dropbox_api_error', 'Empty rendition response');
        }
        
        Logger::debug('[LSP Dropbox] get_large_rendition success: ' . strlen($body) . ' bytes');
        
        // Return raw image bytes
        return $body;
    }

    /**
     * Get temporary download link for a file
     */
    public static function get_temporary_link($path) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $response = wp_remote_post('https://api.dropboxapi.com/2/files/get_temporary_link', [
            'timeout' => 20,
            'headers' => array_merge(self::headers(), [
                'Content-Type' => 'application/json',
            ]),
            'body' => wp_json_encode([
                'path' => $path,
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Dropbox] get_temporary_link failed: ' . $code . ' - ' . $body);
            return new \WP_Error('dropbox_api_error', 'Dropbox API error: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Download file content directly
     */
    public static function download($path) {
        Logger::debug('[LSP Dropbox] download() starting for path: ' . $path);
        
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            Logger::debug('[LSP Dropbox] download() ensure_token failed: ' . $token_ok->get_error_message());
            return $token_ok;
        }

        Logger::debug('[LSP Dropbox] download() calling content API...');
        
        $response = wp_remote_post('https://content.dropboxapi.com/2/files/download', [
            'timeout' => 120,
            'headers' => array_merge(self::headers(), [
                'Dropbox-API-Arg' => wp_json_encode(['path' => $path]),
            ]),
        ]);

        if (is_wp_error($response)) {
            Logger::debug('[LSP Dropbox] download() wp_remote_post failed: ' . $response->get_error_message());
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        Logger::debug('[LSP Dropbox] download() HTTP response code: ' . $code);

        if ($code !== 200) {
            $body = wp_remote_retrieve_body($response);
            Logger::debug('[LSP Dropbox] download failed: ' . $code . ' - ' . $body);
            return new \WP_Error('dropbox_download_error', 'Dropbox download error: ' . $code . ' - ' . substr($body, 0, 200));
        }

        $body = wp_remote_retrieve_body($response);
        Logger::debug('[LSP Dropbox] download() success, received ' . strlen($body) . ' bytes');
        
        return $body;
    }

    /**
     * Get current account info
     */
    public static function get_current_account() {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $response = wp_remote_post('https://api.dropboxapi.com/2/users/get_current_account', [
            'timeout' => 15,
            'headers' => self::headers(),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            return new \WP_Error('dropbox_api_error', 'Dropbox API error: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Filter entries to only images
     */
    public static function filter_images($entries) {
        $image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp', 'tiff', 'tif'];
        
        return array_filter($entries, function($entry) use ($image_extensions) {
            if (($entry['.tag'] ?? '') !== 'file') {
                return false;
            }
            
            $name = strtolower($entry['name'] ?? '');
            $ext = pathinfo($name, PATHINFO_EXTENSION);
            
            return in_array($ext, $image_extensions, true);
        });
    }

    /**
     * Filter entries to only folders
     */
    public static function filter_folders($entries) {
        return array_filter($entries, function($entry) {
            return ($entry['.tag'] ?? '') === 'folder';
        });
    }
}
