<?php
namespace LightSyncPro\OAuth;

use LightSyncPro\Admin\Admin;
use LightSyncPro\Util\Crypto;
use LightSyncPro\Util\Logger;


/**
 * Canva OAuth handler for LightSync Pro
 * Handles broker-based OAuth flow and token management for Canva
 */
class CanvaOAuth {
    const BROKER_URL = 'https://lightsyncpro.com';
    const CRON_CANVA_TOKEN_GUARD = 'lightsync_canva_token_guard';

    public static function init() {
        // Handle ?lsp_canva_connected=1 callback
        add_action('admin_init', [__CLASS__, 'handle_connection_callback']);
        
        // Token refresh guard
        add_action('init', [__CLASS__, 'ensure_guard_scheduled']);
        add_action(self::CRON_CANVA_TOKEN_GUARD, [__CLASS__, 'cron_token_guard']);
    }

    /**
     * Check if Canva is connected
     */
    public static function is_connected() {
        $o = Admin::get_opt();
        return !empty($o['canva_broker_token_enc']);
    }

    /**
     * Get Canva auth URL (starts OAuth flow via broker)
     */
    public static function auth_url() {
        $state  = wp_create_nonce('lightsync_canva_oauth_state');
        $site   = home_url();
        $return = admin_url('admin.php?page=lightsyncpro&tab=source');
        
        return self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_canva_connect_start'
             . '&site='   . rawurlencode($site)
             . '&state='  . rawurlencode($state)
             . '&return=' . rawurlencode($return);
    }

    /**
     * Handle the connection callback (?lsp_canva_connected=1)
     * Pick up the broker token from the broker
     */
    public static function handle_connection_callback() {
        if (!isset($_GET['lsp_canva_connected']) || $_GET['lsp_canva_connected'] !== '1') {
            return;
        }

        // Verify state nonce
        $state = isset($_GET['state']) ? sanitize_text_field($_GET['state']) : '';
        if (!wp_verify_nonce($state, 'lightsync_canva_oauth_state')) {
            Logger::debug('[LSP Canva] Invalid state nonce on callback');
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error"><p>Canva connection failed: Invalid security token. Please try again.</p></div>';
            });
            return;
        }

        // Pick up the broker token
        $site = home_url();
        $pickup_url = self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_canva_install_pickup'
                    . '&site=' . rawurlencode($site)
                    . '&state=' . rawurlencode($state);

        $response = wp_remote_get($pickup_url, [
            'timeout' => 20,
            'headers' => ['Accept' => 'application/json'],
        ]);

        if (is_wp_error($response)) {
            Logger::debug('[LSP Canva] Pickup request failed: ' . $response->get_error_message());
            add_action('admin_notices', function() use ($response) {
                echo '<div class="notice notice-error"><p>Canva connection failed: ' . esc_html($response->get_error_message()) . '</p></div>';
            });
            return;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (empty($data['success']) || empty($data['data']['broker_token'])) {
            $error = $data['data']['error'] ?? 'Unknown error';
            Logger::debug('[LSP Canva] Pickup failed: ' . $error);
            add_action('admin_notices', function() use ($error) {
                echo '<div class="notice notice-error"><p>Canva connection failed: ' . esc_html($error) . '</p></div>';
            });
            return;
        }

        // Encrypt and store the broker token
        $broker_token = $data['data']['broker_token'];
        $encrypted = Crypto::enc($broker_token);

        Admin::set_opt([
            'canva_broker_token_enc' => $encrypted,
            'canva_connected_at'     => time(),
        ]);

        Logger::debug('[LSP Canva] Successfully connected and stored broker token');

        // Show success notice
        add_action('admin_notices', function() {
            echo '<div class="notice notice-success"><p>✓ Successfully connected to Canva!</p></div>';
        });
    }

    /**
     * Ensure access token is valid, refresh if needed
     */
    public static function ensure_token() {
        $o   = Admin::get_opt();
        $exp = (int)($o['canva_token_expires'] ?? 0);
        $now = time();

        // If we have a valid token with >60s remaining, we're good
        if (!empty($o['canva_access_token']) && ($exp - $now) > 60) {
            return true;
        }

        // Need to refresh from broker
        $enc = $o['canva_broker_token_enc'] ?? '';
        if (!$enc) {
            return new \WP_Error('no_broker', 'Canva not connected. Click Connect to Canva.');
        }

        $broker = Crypto::dec($enc);
        if (!$broker) {
            return new \WP_Error('bad_broker', 'Stored Canva broker token not readable.');
        }

        $resp = wp_remote_post(
            self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_canva_token_refresh',
            [
                'timeout' => 20,
                'headers' => [
                    'Authorization' => 'Bearer ' . $broker,
                    'Accept'        => 'application/json',
                ],
            ]
        );

        if (is_wp_error($resp)) {
            return $resp;
        }

        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);
        $data = json_decode($body, true);

        $ok      = is_array($data) && array_key_exists('success', $data) ? (bool)$data['success'] : true;
        $payload = (is_array($data) && array_key_exists('data', $data)) ? $data['data'] : $data;

        if (!$ok) {
            $msg = (is_array($payload) && !empty($payload['error'])) ? $payload['error'] : 'broker error';
            return new \WP_Error('no_access', 'Canva broker refused: ' . $msg);
        }

        if (empty($payload['access_token'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                Logger::debug('[LSP Canva] broker refresh code=' . $code . ' body=' . $body);
            }
            return new \WP_Error('no_access', 'Could not obtain Canva access token from broker');
        }

        // Store with cushion for early refresh
        $expires_in = (int)($payload['expires_in'] ?? 3600);
        $cushion    = 30;
        Admin::set_opt([
            'canva_access_token'  => $payload['access_token'],
            'canva_token_expires' => $now + max(300, $expires_in - $cushion),
        ]);

        return true;
    }

    /**
     * Proactively refresh when <5 minutes remain
     */
    public static function maybe_refresh() {
        $o   = Admin::get_opt();
        $exp = (int)($o['canva_token_expires'] ?? 0);
        if (!$exp) return self::ensure_token();
        if (($exp - time()) < 300) {
            return self::ensure_token();
        }
        return true;
    }

    /**
     * Disconnect Canva
     */
    public static function disconnect() {
        Admin::set_opt([
            'canva_access_token'      => '',
            'canva_token_expires'     => 0,
            'canva_broker_token_enc'  => '',
            'canva_connected_at'      => 0,
            'canva_folder_ids'        => [],
        ]);

        // Clear cached designs
        global $wpdb;
        $like = $wpdb->esc_like('_transient_lightsync_canva_') . '%';
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options}
                 WHERE option_name LIKE %s
                    OR option_name LIKE %s",
                $like,
                str_replace('_transient_', '_transient_timeout_', $like)
            )
        );

        Logger::debug('[LSP Canva] Disconnected – tokens cleared + cache purged');
    }

    /**
     * Get headers for Canva API calls
     */
    public static function headers() {
        $o = Admin::get_opt();
        return [
            'Authorization' => 'Bearer ' . ($o['canva_access_token'] ?? ''),
            'Accept'        => 'application/json',
        ];
    }

    /**
     * Schedule token guard cron
     */
    public static function ensure_guard_scheduled() {
        if (!self::is_connected()) return;
        
        if (!wp_next_scheduled(self::CRON_CANVA_TOKEN_GUARD)) {
            wp_schedule_event(time() + 120, 'hourly', self::CRON_CANVA_TOKEN_GUARD);
        }
    }

    /**
     * Cron handler for token refresh
     */
    public static function cron_token_guard() {
        $ok = self::maybe_refresh();
        if (is_wp_error($ok) && defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Canva] token_guard failed: ' . $ok->get_error_message());
        }
    }

    /**
     * Fetch user's Canva designs
     */
    public static function get_designs($continuation = null) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $url = 'https://api.canva.com/rest/v1/designs';
        if ($continuation) {
            $url .= '?continuation=' . urlencode($continuation);
        }

        $response = wp_remote_get($url, [
            'timeout' => 30,
            'headers' => self::headers(),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Canva] get_designs failed: ' . $code . ' - ' . $body);
            return new \WP_Error('canva_api_error', 'Canva API error: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Fetch ALL user's Canva designs with pagination
     * @param int $max_pages Maximum pages to fetch (0 = unlimited, default 10 = ~250 designs)
     */
    public static function get_all_designs($max_pages = 10) {
        $all_designs = [];
        $continuation = null;
        $page = 0;

        do {
            $result = self::get_designs($continuation);
            
            if (is_wp_error($result)) {
                // If we have some designs, return them; otherwise return error
                if (!empty($all_designs)) {
                    return ['items' => $all_designs];
                }
                return $result;
            }

            if (!empty($result['items'])) {
                $all_designs = array_merge($all_designs, $result['items']);
            }

            $continuation = $result['continuation'] ?? null;
            $page++;

            // Safety limit to prevent infinite loops
            if ($max_pages > 0 && $page >= $max_pages) {
                break;
            }

        } while ($continuation);

        return ['items' => $all_designs];
    }

    /**
     * Fetch folders from Canva
     */
    public static function get_folders($continuation = null) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $url = 'https://api.canva.com/rest/v1/folders';
        if ($continuation) {
            $url .= '?continuation=' . urlencode($continuation);
        }

        $response = wp_remote_get($url, [
            'timeout' => 30,
            'headers' => self::headers(),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Canva] get_folders failed: ' . $code . ' - ' . $body);
            return new \WP_Error('canva_api_error', 'Canva API error: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Export a design to get downloadable URL
     */
    public static function export_design($design_id, $format = 'png') {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $url = 'https://api.canva.com/rest/v1/exports';

        $response = wp_remote_post($url, [
            'timeout' => 60,
            'headers' => array_merge(self::headers(), [
                'Content-Type' => 'application/json',
            ]),
            'body' => wp_json_encode([
                'design_id' => $design_id,
                'format' => [
                    'type' => $format,
                ],
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200 && $code !== 202) {
            Logger::debug('[LSP Canva] export_design failed: ' . $code . ' - ' . $body);
            return new \WP_Error('canva_export_error', 'Canva export error: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Get export status/result
     */
    public static function get_export($export_id) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $url = 'https://api.canva.com/rest/v1/exports/' . $export_id;

        $response = wp_remote_get($url, [
            'timeout' => 30,
            'headers' => self::headers(),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Canva] get_export failed: ' . $code . ' - ' . $body);
            return new \WP_Error('canva_api_error', 'Canva API error: ' . $code);
        }

        return json_decode($body, true);
    }
}
