<?php
namespace LightSyncPro\Http;
if (!defined('ABSPATH')) exit;

class Endpoints {

    private static function is_mock() {
        return defined('LIGHTSYNC_DEV_MOCK') && LIGHTSYNC_DEV_MOCK
            && defined('LIGHTSYNC_DEV_BASE') && LIGHTSYNC_DEV_BASE;
    }

    private static function base() {
        if (self::is_mock()) {
            return rtrim(LIGHTSYNC_DEV_BASE, '/');
        }
        return 'https://lr.adobe.io';
    }

    public static function catalogs() {
        return self::base() . (self::is_mock()
            ? '/catalog'
            : '/v2/catalog'
        );
    }

    public static function albums($catalogId) {
        return self::base() . (self::is_mock()
            ? "/catalogs/{$catalogId}/albums"
            : "/v2/catalogs/{$catalogId}/albums"
        );
    }

    /**
     * Album assets endpoint
     *
     * $cursor should be EXACTLY what Adobe returns in links.next.href
     * OR a fully resolved URL (base + href).
     */
    public static function album_assets($catalogId, $albumId, $cursor = null, $limit = null) {

        $base = rtrim(self::base(), '/');

        // Clamp limit (Adobe max is 200)
        $lim = null;
        if (!empty($limit)) {
            $lim = (int) $limit;
            $lim = max(1, min($lim, 200));
        }

        /**
         * Helper: merge / override query params on any URL
         */
        $add_qs = function(string $url, array $add): string {
            $parts = wp_parse_url($url);
            if (!$parts || empty($parts['path'])) return $url;

            $q = [];
            if (!empty($parts['query'])) {
                parse_str($parts['query'], $q);
            }

            foreach ($add as $k => $v) {
                if ($v === null || $v === '') continue;
                $q[$k] = $v;
            }

            $scheme = $parts['scheme'] ?? '';
            $host   = $parts['host'] ?? '';
            $port   = isset($parts['port']) ? ':' . $parts['port'] : '';
            $user   = $parts['user'] ?? '';
            $pass   = isset($parts['pass']) ? ':' . $parts['pass'] : '';
            $pass   = ($user || $pass) ? "{$pass}@" : '';
            $path   = $parts['path'] ?? '';
            $frag   = isset($parts['fragment']) ? '#' . $parts['fragment'] : '';

            $query = http_build_query($q, '', '&');

            if ($scheme && $host) {
                return "{$scheme}://{$user}{$pass}{$host}{$port}{$path}" . ($query ? "?{$query}" : '') . $frag;
            }

            return $url;
        };

        // Always enforce embed + limit
        $must = [
            'embed' => 'asset',
            'limit' => $lim,
        ];

        /**
         * ===========================
         * PAGINATION HANDLING
         * ===========================
         */

        // 1) Full URL from previous run → use directly
        if (is_string($cursor) && preg_match('~^https?://~i', $cursor)) {
            return $add_qs($cursor, $must);
        }

        // 2) Absolute or relative href from links.next.href
        if (is_string($cursor) && $cursor !== '') {
            $href = trim($cursor);

            // Absolute path (starts with "/")
            if (str_starts_with($href, '/')) {
                return $add_qs($base . $href, $must);
            }

            // Relative href ("albums/..." etc)
            return $add_qs($base . '/' . ltrim($href, '/'), $must);
        }

        /**
         * ===========================
         * FIRST PAGE
         * ===========================
         */

        $path = self::is_mock()
            ? "/catalogs/{$catalogId}/albums/{$albumId}/assets"
            : "/v2/catalogs/{$catalogId}/albums/{$albumId}/assets";

        $qs = ['embed' => 'asset'];
        if ($lim !== null) {
            $qs['limit'] = (string) $lim;
        }

        return $base . $path . '?' . http_build_query($qs, '', '&');
    }

    public static function generate_renditions($catalogId, $assetId) {
        return self::base() . (self::is_mock()
            ? "/catalogs/{$catalogId}/assets/{$assetId}/renditions"
            : "/v2/catalogs/{$catalogId}/assets/{$assetId}/renditions"
        );
    }

    public static function rendition($catalogId, $assetId, $size = '2048') {
        return self::base() . (self::is_mock()
            ? "/renditions/{$size}/{$assetId}"
            : "/v2/catalogs/{$catalogId}/assets/{$assetId}/renditions/{$size}"
        );
    }
}
