<?php
namespace LightSyncPro\Http;

use LightSyncPro\OAuth\OAuth;
use LightSyncPro\Util\Logger;

if (!defined('ABSPATH')) exit;

class Client {

    /**
     * GET with proactive refresh + single retry on 401/403.
     */
    public static function get($url, $timeout = 30) {
        // Proactive refresh if token is near expiry
        $pre = OAuth::maybe_refresh();
        if (is_wp_error($pre)) {
            // Fallback: ensure_token (older installs without maybe_refresh)
            $pre = OAuth::ensure_token();
            if (is_wp_error($pre)) return $pre;
        }

        $headers = OAuth::headers();
        // Ensure we always accept JSON by default
        if (empty($headers['Accept'])) {
            $headers['Accept'] = 'application/json';
        }

        $args = ['headers' => $headers, 'timeout' => $timeout];
        $resp = wp_remote_get($url, $args);
        $code = (int) wp_remote_retrieve_response_code($resp);

        if ($code === 401 || $code === 403) {
            // Try one refresh + retry
            $t = OAuth::ensure_token();
            if (is_wp_error($t)) return $t;

            $args['headers'] = OAuth::headers();
            if (empty($args['headers']['Accept'])) {
                $args['headers']['Accept'] = 'application/json';
            }

            $resp = wp_remote_get($url, $args);
            $code = (int) wp_remote_retrieve_response_code($resp);
        }

        // Non-200s: convert to WP_Error with debug details
        if ($code !== 200) {
            $body = wp_remote_retrieve_body($resp);

            if (defined('WP_DEBUG') && WP_DEBUG) {
               Logger::debug('[LSP] GET ' . $url . ' -> ' . $code . ' body=' . $body);
                Logger::debug('[LSP] Sent headers: ' . json_encode($args['headers']));
            }

            // Special case: Lightroom "ResourceNotFoundError" (asset removed)
            if ($code === 404 && $body && strpos($body, '"ResourceNotFoundError"') !== false) {
                return new \WP_Error(
                    'lightsync_asset_missing',
                    'Lightroom reports this asset no longer exists in the catalog.',
                    [
                        'code' => $code,
                        'body' => $body,
                        'url'  => $url,
                    ]
                );
            }

            // General HTTP error
            $msg = $body ?: ('HTTP ' . $code);
            return new \WP_Error(
                'lightsync_http_error',
                'Lightroom API error: ' . $msg,
                [
                    'code' => $code,
                    'body' => $body,
                    'url'  => $url,
                ]
            );
        }

        return $resp;
    }

    /**
     * POST with proactive refresh + single retry on 401/403.
     * We keep responses as-is (e.g., generateRenditions may return 202).
     */
    public static function post($url, $body = null, $timeout = 30){
        // Proactive refresh if token is near expiry
        $pre = OAuth::maybe_refresh();
        if (is_wp_error($pre)) {
            $pre = OAuth::ensure_token();
            if (is_wp_error($pre)) return $pre;
        }

        $headers = OAuth::headers();
        if (empty($headers['Accept'])) {
            $headers['Accept'] = 'application/json';
        }

        $args = [
            'headers' => $headers,
            'timeout' => $timeout,
            'method'  => 'POST',
        ];

        if (!is_null($body)) {
            $args['headers']['Content-Type'] = 'application/json';
            $args['body'] = is_string($body) ? $body : wp_json_encode($body);
        }

        $resp = wp_remote_post($url, $args);
        $code = (int) wp_remote_retrieve_response_code($resp);

        if ($code === 401 || $code === 403) {
            // Try one refresh + retry
            $t = OAuth::ensure_token();
            if (is_wp_error($t)) return $t;

            $args['headers'] = OAuth::headers();
            if (empty($args['headers']['Accept'])) {
                $args['headers']['Accept'] = 'application/json';
            }
            $args['headers']['Content-Type'] = 'application/json';

            $resp = wp_remote_post($url, $args);
            $code = (int) wp_remote_retrieve_response_code($resp);
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP] POST ' . $url . ' -> ' . $code . ' body=' . wp_remote_retrieve_body($resp));
        }

        // Intentionally not converting non-200 to error:
        // generateRenditions etc. may return 202/204 and that's fine.
        return $resp;
    }
}
