<?php
namespace LightSyncPro\Blocks;

if ( ! defined( 'ABSPATH' ) ) exit;

class Gallery_Block {

    public static function init(){
        add_action('init', [ __CLASS__, 'register' ]);
    }

    public static function register(){
        // Editor script (no build tools)
        wp_register_script(
            'lightsyncpro-block-gallery',
            plugins_url('assets/block-gallery.js', LIGHTSYNC_PRO_FILE),
            ['wp-blocks','wp-element','wp-components','wp-block-editor','wp-api-fetch'],
            LIGHTSYNC_PRO_VERSION,
            true
        );

        register_block_type('lightsyncpro/gallery', [
            'render_callback' => [ __CLASS__, 'render' ],
            'attributes' => [
                'album' => ['type'=>'string', 'default'=>''],
                'columns' => ['type'=>'number', 'default'=>3],
                'size' => ['type'=>'string', 'default'=>'medium'],
                'showCaptions' => ['type'=>'boolean', 'default'=>false],
            ],
            'editor_script' => 'lightsyncpro-block-gallery',
            'editor_style'  => 'lightsyncpro-gallery',
        ]);
    }

    public static function render($attrs){
        $album = isset($attrs['album']) ? sanitize_text_field($attrs['album']) : '';
        $cols  = max(1, min(6, intval($attrs['columns'] ?? 3)));
        $size  = in_array($attrs['size'] ?? 'medium', ['thumbnail','medium','large','full']) ? $attrs['size'] : 'medium';
        $show  = !empty($attrs['showCaptions']);

        if (!$album){
            return '<div class="lsp-gallery lsp-note">Select a Lightroom album in block settings.</div>';
        }

        // Accept either slug or raw album ID:
        $term = term_exists($album, 'lightsync_album');
        if ( ! $term ){
            $term = term_exists('lr-'.$album, 'lightsync_album'); // fallback: raw ID
        }
        if ( ! $term ){
            return '<div class="lsp-gallery lsp-note">Album not found: '.esc_html($album).'</div>';
        }
        $term_id = is_array($term) ? intval($term['term_id']) : intval($term);

       $q = new \WP_Query([
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'posts_per_page' => 100,
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
            'tax_query'      => [[
                'taxonomy' => 'lightsync_album',
                'field'    => 'term_id',
                'terms'    => $term_id
            ]]
        ]);

        if (!$q->have_posts()){
            return '<div class="lsp-gallery lsp-note">No images in this album yet.</div>';
        }

        // Assets for grid and lightbox
        wp_enqueue_style('lightsyncpro-gallery', plugins_url('assets/gallery.css', LIGHTSYNC_PRO_FILE), [], LIGHTSYNC_PRO_VERSION);
        wp_enqueue_style('lightsyncpro-lightbox', plugins_url('assets/lightbox.css', LIGHTSYNC_PRO_FILE), [], LIGHTSYNC_PRO_VERSION);
        wp_enqueue_script('lightsyncpro-lightbox', plugins_url('assets/lightbox.js', LIGHTSYNC_PRO_FILE), [], LIGHTSYNC_PRO_VERSION, true);

        $out = '<div class="lsp-gallery cols-'.$cols.'">';
        while( $q->have_posts() ){ $q->the_post();
            $id = get_the_ID();
            $img  = wp_get_attachment_image($id, $size, false, ['loading'=>'lazy']);
            $full = wp_get_attachment_image_src($id, 'full');
            $href = $full ? esc_url($full[0]) : '#';
            $cap  = $show ? wp_get_attachment_caption($id) : '';
            $out .= '<figure class="lsp-item"><a href="'.$href.'" class="lsp-lightbox-link" data-lsp="1">'.$img.'</a>';
            if ($show && $cap){ $out .= '<figcaption>'.esc_html($cap).'</figcaption>'; }
            $out .= '</figure>';
        }
        wp_reset_postdata();
        $out .= '</div>';

        return $out;
    }
}
