<?php
namespace LightSyncPro\AI;

use LightSyncPro\Admin\Admin;
use LightSyncPro\OAuth\OpenRouterOAuth;
use LightSyncPro\Mapping\Mapping;
use LightSyncPro\Util\Image;
use LightSyncPro\Util\Logger;


/**
 * AI Content Generation for LightSync Pro
 * 
 * Handles the full lifecycle of AI-generated assets:
 *   1. Fetch available models from OpenRouter (lightweight, cached)
 *   2. Generate content via OpenRouter (400+ models)
 *   3. Preview in plugin UI (base64 → <img> tag)
 *   4. Commit to WordPress media library with full metadata
 *   5. Tag for future redistribution to other destinations
 * 
 * Architecture:
 * - OpenRouter: Plugin → OpenRouter directly (API key via broker)
 * - Model list: Plugin → OpenRouter public API (cached 1hr)
 * - OAuth: Plugin → Broker → OpenRouter (one-time)
 */
class AIGenerate {

    const BROKER_URL = 'https://lightsyncpro.com';

    /**
     * Initialize AJAX handlers
     */
    public static function init() {
        // Model listing
        add_action('wp_ajax_lsp_ai_get_models', [__CLASS__, 'ajax_get_models']);
        
        // Generation
        add_action('wp_ajax_lsp_ai_generate', [__CLASS__, 'ajax_generate']);
        
        // Commit preview to media library
        add_action('wp_ajax_lsp_ai_commit', [__CLASS__, 'ajax_commit']);

        // Regenerate existing asset
        add_action('wp_ajax_lsp_ai_regenerate', [__CLASS__, 'ajax_regenerate']);

        // Version history and rollback
        add_action('wp_ajax_lsp_ai_versions', [__CLASS__, 'ajax_versions']);
        add_action('wp_ajax_lsp_ai_rollback', [__CLASS__, 'ajax_rollback']);

        // Re-optimize existing asset (convert to WebP in place)
        add_action('wp_ajax_lsp_ai_reoptimize', [__CLASS__, 'ajax_reoptimize']);

        // Browse AI-generated assets
        add_action('wp_ajax_lsp_ai_browse', [__CLASS__, 'ajax_browse']);

        // Push update to destinations (Shopify)
        add_action('wp_ajax_lsp_ai_push_destinations', [__CLASS__, 'ajax_push_destinations']);

        // Background sync (queue + status)
        add_action('wp_ajax_lsp_ai_background_sync', [__CLASS__, 'ajax_background_sync']);
        add_action('wp_ajax_lsp_ai_background_status', [__CLASS__, 'ajax_background_status']);

        // Cron handler for background processing
        add_action('lsp_ai_background_sync_tick', [__CLASS__, 'cron_background_tick']);
    }


    /* ========================================================================
     * AJAX: Get Available Models
     * ======================================================================== */

    public static function ajax_get_models() {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_ajax_nonce');
        if (!current_user_can('upload_files')) {
            wp_send_json_error('Insufficient permissions', 403);
        }

        // Check connection
        if (!OpenRouterOAuth::is_connected()) {
            wp_send_json_error(['error' => 'OpenRouter not connected'], 401);
        }

        @set_time_limit(60);

        // Force refresh clears cache (used by retry button)
        $force_refresh = !empty($_POST['force_refresh']);

        // Clear model cache when plugin version changes or forced
        $cache_ver = get_option('lsp_models_ver', '');
        if ($cache_ver !== LIGHTSYNC_PRO_VERSION || $force_refresh) {
            delete_transient('lsp_openrouter_models_v3');
            update_option('lsp_models_ver', LIGHTSYNC_PRO_VERSION, false);
        }

        // Try cache first
        $cached = get_transient('lsp_openrouter_models_v3');
        if ($cached) {
            wp_send_json_success($cached);
        }

        // Fetch image generation models from OpenRouter (public API — no auth needed)
        $models_data = null;

        // Attempt 1: Category filter (fast, fewer results)
        $resp = wp_remote_get(
            'https://openrouter.ai/api/v1/models?category=image-generation',
            [
                'timeout' => 15,
                'headers' => ['Accept' => 'application/json'],
            ]
        );

        if (!is_wp_error($resp)) {
            $code = wp_remote_retrieve_response_code($resp);
            $body = json_decode(wp_remote_retrieve_body($resp), true);
            if ($code === 200 && !empty($body['data'])) {
                $models_data = $body['data'];
                error_log('[LSP AI] Fetched ' . count($models_data) . ' models via category filter');
            } else {
                error_log('[LSP AI] Category filter returned HTTP ' . $code . ' or empty — trying all models');
            }
        } else {
            error_log('[LSP AI] Category filter error: ' . $resp->get_error_message() . ' — trying all models');
        }

        // Attempt 2: Fetch all models and filter server-side
        if (!$models_data) {
            $resp = wp_remote_get(
                'https://openrouter.ai/api/v1/models',
                [
                    'timeout' => 30,
                    'headers' => ['Accept' => 'application/json'],
                ]
            );

            if (is_wp_error($resp)) {
                error_log('[LSP AI] All models fetch error: ' . $resp->get_error_message());
                wp_send_json_error(['error' => 'Cannot reach OpenRouter API: ' . $resp->get_error_message()], 502);
            }

            $code = wp_remote_retrieve_response_code($resp);
            $body = json_decode(wp_remote_retrieve_body($resp), true);

            if ($code !== 200 || empty($body['data'])) {
                error_log('[LSP AI] All models API returned HTTP ' . $code);
                wp_send_json_error(['error' => 'OpenRouter API error (HTTP ' . $code . ')'], 502);
            }

            // Filter to image-capable models
            $models_data = array_filter($body['data'], function($m) {
                $arch = $m['architecture'] ?? [];
                $out = $arch['output_modalities'] ?? [];
                return in_array('image', $out);
            });
            $models_data = array_values($models_data);
            error_log('[LSP AI] Filtered ' . count($models_data) . ' image models from ' . count($body['data']) . ' total');
        }

        // Normalize format for the plugin UI
        $models = [];
        foreach ($models_data as $m) {
            $id = $m['id'] ?? '';

            // Skip meta/routing models
            if (stripos($id, 'autorouter') !== false || stripos($id, 'auto/') !== false || stripos($id, 'router') !== false) {
                continue;
            }

            $pricing = $m['pricing'] ?? [];

            $models[] = [
                'id'         => $id,
                'name'       => $m['name'] ?? $id,
                'capability' => 'image',
                'is_free'    => (strpos($id, ':free') !== false)
                             || (floatval($pricing['prompt'] ?? 1) <= 0 && floatval($pricing['completion'] ?? 1) <= 0 && floatval($pricing['image'] ?? 1) <= 0),
                'pricing'    => [
                    'prompt'     => $pricing['prompt'] ?? '0',
                    'completion' => $pricing['completion'] ?? '0',
                    'image'      => $pricing['image'] ?? '0',
                ],
                'context_length' => (int)($m['context_length'] ?? 0),
            ];
        }

        $result = ['models' => $models];

        // Cache for 1 hour
        set_transient('lsp_openrouter_models_v3', $result, HOUR_IN_SECONDS);
        wp_send_json_success($result);
    }


    /* ========================================================================
     * AJAX: Generate Content (Preview)
     * ======================================================================== */

    public static function ajax_generate() {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_ajax_nonce');
        if (!current_user_can('upload_files')) {
            wp_send_json_error('Insufficient permissions', 403);
        }

        // Sync gate check
        $gate = \lsp_gate_check();
        if ( is_wp_error( $gate ) ) {
            wp_send_json_error([
                'error' => 'sync_disabled',
                'message' => $gate->get_error_message(),
            ], 403);
        }

        // Bump limits — image gen can take 2-3 min for large models
        @ini_set('memory_limit', '512M');
        @set_time_limit(200);

        // Collect request params
        $model        = sanitize_text_field($_POST['model'] ?? '');
        $prompt       = wp_kses_post($_POST['prompt'] ?? '');
        $type         = sanitize_text_field($_POST['type'] ?? 'image');
        $aspect_ratio = sanitize_text_field($_POST['aspect_ratio'] ?? '1:1');
        $preview      = !empty($_POST['preview']);
        $is_free      = !empty($_POST['is_free']);
        $provider     = sanitize_text_field($_POST['provider'] ?? '');

        if (empty($prompt)) {
            wp_send_json_error(['error' => 'Prompt is required'], 400);
        }

        // ── Route to OpenRouter ──
        if (!OpenRouterOAuth::is_connected()) {
            wp_send_json_error(['error' => 'OpenRouter not connected'], 401);
        }

        // Get API key for DIRECT OpenRouter calls (no broker proxy)
        $api_key = OpenRouterOAuth::get_api_key();
        if (is_wp_error($api_key)) {
            wp_send_json_error(['error' => $api_key->get_error_message()], 401);
        }

        // Build OpenRouter chat completion request
        $or_body = [
            'model'    => $model,
            'stream'   => false,
        ];

        $is_image = ($type === 'image');

        if ($is_image) {
            // Determine if model supports text+image or image-only output
            // Models like Gemini output both; Flux, DALL-E, Recraft, Sourceful are image-only
            $image_only_patterns = ['flux', 'dall-e', 'stable-diffusion', 'recraft', 'sourceful', 'ideogram', 'seedream', 'hidream'];
            $is_image_only = false;
            $model_lower = strtolower($model);
            foreach ($image_only_patterns as $pattern) {
                if (strpos($model_lower, $pattern) !== false) {
                    $is_image_only = true;
                    break;
                }
            }

            // Set modalities based on model capability
            $or_body['modalities'] = $is_image_only ? ['image'] : ['image', 'text'];

            // Minimal max_tokens — we want the IMAGE, not a text essay
            // Image data comes in a separate 'images' field, not counted as tokens
            // This only limits the text portion of the response
            $or_body['max_tokens'] = $is_image_only ? 256 : 512;

            $or_body['image_config'] = [
                'aspect_ratio' => $aspect_ratio,
            ];

            // Preview mode: force 1K for cheaper/faster iteration
            if ($preview) {
                $or_body['image_config']['image_size'] = '1K';
            }

            // Keep prompt lean — aspect ratio is handled by image_config, no need to repeat it
            $or_body['messages'] = [
                ['role' => 'user', 'content' => "Generate an image: " . $prompt],
            ];
        } else {
            $or_body['messages'] = [
                ['role' => 'user', 'content' => $prompt],
            ];
        }

        // Call OpenRouter directly (NOT through broker)
        $resp = wp_remote_post('https://openrouter.ai/api/v1/chat/completions', [
            'timeout' => $is_image ? 180 : 120,
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type'  => 'application/json',
                'HTTP-Referer'  => home_url(),
                'X-Title'       => 'LightSync Pro',
            ],
            'body' => wp_json_encode($or_body),
        ]);

        if (is_wp_error($resp)) {
            wp_send_json_error(['error' => 'OpenRouter request failed: ' . $resp->get_error_message()], 502);
        }

        $status = wp_remote_retrieve_response_code($resp);
        $data   = json_decode(wp_remote_retrieve_body($resp), true);

        // Handle OpenRouter errors
        if ($status !== 200 || !empty($data['error'])) {
            $err_msg = '';
            if (!empty($data['error']['message'])) {
                $err_msg = $data['error']['message'];
            } elseif (!empty($data['error'])) {
                $err_msg = is_string($data['error']) ? $data['error'] : wp_json_encode($data['error']);
            } else {
                $err_msg = 'HTTP ' . $status;
            }

            $code = $status >= 400 && $status < 600 ? $status : 502;
            wp_send_json_error([
                'error' => $err_msg,
                'code'  => ($code === 402 || stripos($err_msg, 'credit') !== false) ? 'insufficient_credits' : '',
            ], $code);
        }

        // Extract text content
        $content = $data['choices'][0]['message']['content'] ?? '';

        // Extract generated images (base64 data URLs)
        $images = [];
        if (!empty($data['choices'][0]['message']['images'])) {
            foreach ($data['choices'][0]['message']['images'] as $idx => $img) {
                $img_url = $img['image_url']['url'] ?? '';
                if (!$img_url) continue;

                $mime = 'image/png';
                $b64  = $img_url;
                if (preg_match('/^data:(image\/[a-z+]+);base64,(.+)$/i', $img_url, $m)) {
                    $mime = $m[1];
                    $b64  = $m[2];
                }

                $images[] = [
                    'index'    => $idx,
                    'mime'     => $mime,
                    'base64'   => $b64,
                    'data_url' => $img_url,
                    'size'     => strlen(base64_decode($b64)),
                ];
            }
        }

        $content_type = 'text';
        if (!empty($images) && !empty($content)) {
            $content_type = 'mixed';
        } elseif (!empty($images)) {
            $content_type = 'image';
        }

        $usage = $data['usage'] ?? [];

        wp_send_json_success([
            'content_type'  => $content_type,
            'content'       => $content,
            'images'        => $images,
            'model'         => $data['model'] ?? $model,
            'finish_reason' => $data['choices'][0]['finish_reason'] ?? 'unknown',
            'is_preview'    => $preview,
            'is_free'       => $is_free || (strpos($model, ':free') !== false),
            'usage'         => [
                'prompt_tokens'     => (int)($usage['prompt_tokens'] ?? 0),
                'completion_tokens' => (int)($usage['completion_tokens'] ?? 0),
                'total_tokens'      => (int)($usage['total_tokens'] ?? 0),
                'cost'              => $usage['cost'] ?? null,
            ],
        ]);
    }


    /* ========================================================================
     * AJAX: Commit Generated Image to Media Library
     * ======================================================================== */

    public static function ajax_commit() {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_ajax_nonce');
        if (!current_user_can('upload_files')) {
            wp_send_json_error('Insufficient permissions', 403);
        }

        // Sync gate check
        $gate = \lsp_gate_check();
        if ( is_wp_error( $gate ) ) {
            wp_send_json_error([
                'error' => 'sync_disabled',
                'message' => $gate->get_error_message(),
            ], 403);
        }

        // Detect if POST was truncated by post_max_size
        if (empty($_POST) && $_SERVER['CONTENT_LENGTH'] > 0) {
            $max = ini_get('post_max_size');
            wp_send_json_error([
                'error' => "Image data exceeded server's post_max_size ({$max}). Increase post_max_size in php.ini or contact your host.",
            ], 413);
        }

        @ini_set('memory_limit', '512M');
        @set_time_limit(120);

        $base64       = $_POST['base64'] ?? '';
        $mime          = sanitize_text_field($_POST['mime'] ?? 'image/png');
        $prompt        = wp_kses_post($_POST['prompt'] ?? '');
        $model         = sanitize_text_field($_POST['model'] ?? '');
        $aspect_ratio  = sanitize_text_field($_POST['aspect_ratio'] ?? '1:1');
        $is_free       = !empty($_POST['is_free']);
        $filename      = sanitize_file_name($_POST['filename'] ?? '');
        $destination   = sanitize_text_field($_POST['destination'] ?? 'wp');

        if (empty($base64)) {
            wp_send_json_error(['error' => 'No image data provided'], 400);
        }

        // Decode base64
        $image_data = base64_decode($base64);
        if (!$image_data) {
            wp_send_json_error(['error' => 'Invalid base64 data'], 400);
        }

        // Generate filename from prompt if not provided
        if (empty($filename)) {
            $slug = sanitize_title(wp_trim_words($prompt, 8, ''));
            $ext  = self::mime_to_ext($mime);
            $filename = 'ai-' . ($slug ?: 'generated') . '-' . time() . '.' . $ext;
        }

        // Write to temp file
        if (!function_exists('wp_tempnam')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        $tmp = wp_tempnam($filename);
        file_put_contents($tmp, $image_data);

        // Run through the LightSync image optimization pipeline
        // Same pipeline that processes Lightroom, Canva, Figma images
        $maxDim = (int) Admin::get_opt('max_dimension', 2048);
        $prepared = Image::prepareForMedia($tmp, $maxDim);

        if (empty($prepared['path']) || !file_exists($prepared['path'])) {
            @unlink($tmp);
            wp_send_json_error(['error' => 'Image optimization failed'], 500);
        }

        // Update filename extension to match the optimized format (e.g. .png → .webp)
        $new_ext = self::mime_to_ext($prepared['mime']);
        $old_ext = pathinfo($filename, PATHINFO_EXTENSION);
        if ($new_ext !== $old_ext) {
            $filename = preg_replace('/\.' . preg_quote($old_ext, '/') . '$/', '.' . $new_ext, $filename);
        }

        // Import into WordPress media library
        $result = self::import_to_media_library(
            $prepared['path'],
            $prepared['mime'],
            $filename,
            $prompt
        );

        // Clean up temp files
        @unlink($tmp);
        if ($prepared['path'] !== $tmp) {
            @unlink($prepared['path']);
        }

        if (is_wp_error($result)) {
            wp_send_json_error(['error' => $result->get_error_message()], 500);
        }

        $attachment_id = $result;

        // Tag with AI metadata
        self::save_ai_metadata($attachment_id, [
            'prompt'       => $prompt,
            'model'        => $model,
            'aspect_ratio' => $aspect_ratio,
            'is_free'      => $is_free,
            'version'      => 1,
        ]);

        // Save mapping (same system as Lightroom/Canva/Figma)
        $ai_asset_id = 'ai_' . $attachment_id . '_' . time();
        Mapping::save_wp_mapping('openrouter', $ai_asset_id, $attachment_id, [
            'source' => 'openrouter',
        ]);

        // Run AVIF conversion if enabled
        $avif_result = null;
        $attachment_path = get_attached_file($attachment_id);
        if ($attachment_path && class_exists('\LightSyncPro\LightSync_Compress')) {
            $avif = \LightSyncPro\LightSync_Compress::convert_to_avif($attachment_path);
            if (!empty($avif['ok'])) {
                $avif_result = [
                    'path'       => $avif['dst'],
                    'bytes_in'   => $avif['bytes_in'],
                    'bytes_out'  => $avif['bytes_out'],
                    'savings'    => round((1 - $avif['bytes_out'] / max(1, $avif['bytes_in'])) * 100, 1),
                ];
            }
        }

        // Get the final attachment data
        $url = wp_get_attachment_url($attachment_id);
        $meta = wp_get_attachment_metadata($attachment_id);
        $final_path = get_attached_file($attachment_id);

        // Sync to Shopify if destination includes it
        $shopify_result = null;
        if (in_array($destination, ['shopify', 'both'], true)) {
            if (class_exists('\LightSyncPro\Shopify\Shopify') && $final_path && file_exists($final_path)) {
                $shopify_bytes = file_get_contents($final_path);
                $source_id = 'ai_' . $attachment_id;
                $alt_text = wp_trim_words($prompt, 12, '');
                
                $s_result = \LightSyncPro\Shopify\Shopify::upload_file(
                    $shopify_bytes,
                    basename($final_path),
                    $source_id,
                    'openrouter',
                    $alt_text
                );

                if (is_wp_error($s_result)) {
                    $shopify_result = ['error' => $s_result->get_error_message()];
                } else {
                    $shopify_result = [
                        'file_id'  => $s_result['file_id'] ?? '',
                        'file_url' => $s_result['file_url'] ?? '',
                        'updated'  => $s_result['updated'] ?? false,
                    ];
                    // Track destination mapping
                    $dest_map = get_post_meta($attachment_id, '_lightsync_destination_map', true) ?: [];
                    $dest_map['shopify'] = [
                        'file_id'        => $s_result['file_id'] ?? '',
                        'file_url'       => $s_result['file_url'] ?? '',
                        'version_pushed' => 1,
                        'pushed_at'      => gmdate('c'),
                    ];
                    update_post_meta($attachment_id, '_lightsync_destination_map', $dest_map);
                }
            } else {
                $shopify_result = ['error' => 'Shopify not connected'];
            }
        }

        // Track destination on the attachment
        $destinations = ['wordpress'];
        if (in_array($destination, ['shopify', 'both'], true)) {
            $destinations[] = 'shopify';
        }
        update_post_meta($attachment_id, '_lightsync_destinations', $destinations);

        // Activity log
        $short_model = basename(str_replace('/', '-', $model));
        $dest_label = 'WordPress';
        if ($destination === 'both') $dest_label = 'WordPress + Shopify';
        elseif ($destination === 'shopify') $dest_label = 'WordPress + Shopify';
        $activity_msg = sprintf('AI Generated → %s: "%s" (%s)', $dest_label, wp_trim_words($prompt, 8, '…'), $short_model);
        if ($shopify_result && !empty($shopify_result['error'])) {
            $activity_msg .= ' [Shopify failed: ' . $shopify_result['error'] . ']';
            Admin::add_activity($activity_msg, 'warning', 'ai');
        } else {
            Admin::add_activity($activity_msg, 'success', 'ai');
        }

        wp_send_json_success([
            'attachment_id' => $attachment_id,
            'url'           => $url,
            'filename'      => basename($final_path),
            'width'         => $meta['width'] ?? 0,
            'height'        => $meta['height'] ?? 0,
            'filesize'      => filesize($final_path),
            'mime'          => get_post_mime_type($attachment_id),
            'avif'          => $avif_result,
            'shopify'       => $shopify_result,
            'destination'   => $destination,
            'edit_url'      => admin_url('post.php?post=' . $attachment_id . '&action=edit'),
        ]);
    }


    /* ========================================================================
     * AJAX: Regenerate Existing Asset (Update in Place)
     * ======================================================================== */

    public static function ajax_regenerate() {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_ajax_nonce');
        if (!current_user_can('upload_files')) {
            wp_send_json_error('Insufficient permissions', 403);
        }

        // Sync gate check
        $gate = \lsp_gate_check();
        if ( is_wp_error( $gate ) ) {
            wp_send_json_error([
                'error' => 'sync_disabled',
                'message' => $gate->get_error_message(),
            ], 403);
        }

        // Detect if POST was truncated by post_max_size
        if (empty($_POST) && $_SERVER['CONTENT_LENGTH'] > 0) {
            $max = ini_get('post_max_size');
            wp_send_json_error([
                'error' => "Image data exceeded server's post_max_size ({$max}). Increase post_max_size in php.ini or contact your host.",
            ], 413);
        }

        // Bump memory/time for large image processing
        @ini_set('memory_limit', '512M');
        @set_time_limit(120);

        $attachment_id = (int) ($_POST['attachment_id'] ?? 0);
        $base64        = $_POST['base64'] ?? '';
        $prompt        = wp_kses_post($_POST['prompt'] ?? '');
        $model         = sanitize_text_field($_POST['model'] ?? '');
        $is_free       = !empty($_POST['is_free']);

        if (!$attachment_id || empty($base64)) {
            wp_send_json_error(['error' => 'Missing attachment_id or image data'], 400);
        }

        // Verify this is an AI-generated asset
        $source = get_post_meta($attachment_id, '_lightsync_source', true);
        if ($source !== 'openrouter') {
            wp_send_json_error(['error' => 'This asset was not AI-generated'], 400);
        }

        // Decode base64
        $image_data = base64_decode($base64);
        if (!$image_data) {
            wp_send_json_error(['error' => 'Invalid base64 data'], 400);
        }

        // Get current file path
        $current_path = get_attached_file($attachment_id);
        if (!$current_path) {
            wp_send_json_error(['error' => 'Attachment file not found'], 404);
        }

        // Write new image to temp, process through pipeline
        $tmp = wp_tempnam(basename($current_path));
        file_put_contents($tmp, $image_data);

        $maxDim = (int) Admin::get_opt('max_dimension', 2048);
        $prepared = Image::prepareForMedia($tmp, $maxDim);

        if (empty($prepared['path']) || !file_exists($prepared['path'])) {
            @unlink($tmp);
            wp_send_json_error(['error' => 'Image optimization failed'], 500);
        }

        // ── Backup current version before overwriting ──
        // Preserve paid images so users can rollback to any previous version
        $current_version = max(1, (int) get_post_meta($attachment_id, '_lightsync_ai_version', true));
        $versions_dir = dirname($current_path) . '/lsp-versions/' . $attachment_id;
        if (!is_dir($versions_dir)) {
            wp_mkdir_p($versions_dir);
        }
        $backup_ext = pathinfo($current_path, PATHINFO_EXTENSION);
        $backup_filename = 'v' . $current_version . '.' . $backup_ext;
        $backup_path = $versions_dir . '/' . $backup_filename;
        if (file_exists($current_path) && !file_exists($backup_path)) {
            copy($current_path, $backup_path);
        }

        // Overwrite the existing file
        $dir = dirname($current_path);
        $new_path = $dir . '/' . basename($current_path);
        
        // If format changed (e.g. PNG → WebP), update the extension
        $new_ext = self::mime_to_ext($prepared['mime']);
        $old_ext = pathinfo($current_path, PATHINFO_EXTENSION);
        if ($new_ext !== $old_ext) {
            $new_path = preg_replace('/\.' . preg_quote($old_ext, '/') . '$/', '.' . $new_ext, $current_path);
        }

        copy($prepared['path'], $new_path);
        @unlink($tmp);
        if ($prepared['path'] !== $tmp) {
            @unlink($prepared['path']);
        }

        // Update attachment file path if it changed
        if ($new_path !== $current_path) {
            update_attached_file($attachment_id, $new_path);
            @unlink($current_path);  // Remove old file
        }

        // Regenerate WordPress thumbnails
        if (!function_exists('wp_generate_attachment_metadata')) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }
        $meta = wp_generate_attachment_metadata($attachment_id, $new_path);
        wp_update_attachment_metadata($attachment_id, $meta);

        // Update MIME type if changed
        wp_update_post([
            'ID'             => $attachment_id,
            'post_mime_type' => $prepared['mime'],
        ]);

        // Increment version and update AI metadata
        $version = (int) get_post_meta($attachment_id, '_lightsync_ai_version', true);
        $version = max(1, $version) + 1;

        self::save_ai_metadata($attachment_id, [
            'prompt'  => $prompt,
            'model'   => $model,
            'is_free' => $is_free,
            'version' => $version,
        ]);

        // Update version history — add backup_path to existing entry or append new one
        $history = get_post_meta($attachment_id, '_lightsync_ai_version_history', true) ?: [];
        $found_idx = null;
        foreach ($history as $idx => $h) {
            if (($h['version'] ?? 0) === $current_version) {
                $found_idx = $idx;
                break;
            }
        }
        if ($found_idx !== null) {
            // Update existing entry with backup path
            $history[$found_idx]['backup_path'] = $backup_path;
            $history[$found_idx]['checksum'] = hash('sha256', $image_data);
        } else {
            // Append new entry
            $history[] = [
                'version'     => $current_version,
                'prompt'      => get_post_meta($attachment_id, '_lightsync_ai_prompt', true) ?: $prompt,
                'model'       => get_post_meta($attachment_id, '_lightsync_ai_model', true) ?: $model,
                'created_at'  => get_post_meta($attachment_id, '_lightsync_ai_generated_at', true) ?: gmdate('c'),
                'checksum'    => hash('sha256', $image_data),
                'backup_path' => $backup_path,
            ];
        }
        update_post_meta($attachment_id, '_lightsync_ai_version_history', $history);

        // AVIF conversion
        $avif_result = null;
        if (class_exists('\LightSyncPro\LightSync_Compress')) {
            $avif = \LightSyncPro\LightSync_Compress::convert_to_avif($new_path);
            if (!empty($avif['ok'])) {
                $avif_result = [
                    'bytes_in'  => $avif['bytes_in'],
                    'bytes_out' => $avif['bytes_out'],
                    'savings'   => round((1 - $avif['bytes_out'] / max(1, $avif['bytes_in'])) * 100, 1),
                ];
            }
        }

        // Check which destinations are stale
        $dest_map = get_post_meta($attachment_id, '_lightsync_destination_map', true) ?: [];
        $stale_destinations = [];
        foreach ($dest_map as $dest_key => $mapping) {
            if (($mapping['version_pushed'] ?? 0) < $version) {
                $stale_destinations[] = $dest_key;
            }
        }

        // Activity log
        $stale_str = count($stale_destinations) ? ' (' . count($stale_destinations) . ' destination(s) need update)' : '';
        Admin::add_activity(
            sprintf('AI Regenerated v%d: "%s"%s', $version, wp_trim_words($prompt, 8, '…'), $stale_str),
            'info',
            'ai'
        );

        wp_send_json_success([
            'attachment_id'      => $attachment_id,
            'version'            => $version,
            'url'                => self::cache_bust_url(wp_get_attachment_url($attachment_id), $attachment_id),
            'width'              => $meta['width'] ?? 0,
            'height'             => $meta['height'] ?? 0,
            'filesize'           => filesize($new_path),
            'avif'               => $avif_result,
            'stale_destinations' => $stale_destinations,
        ]);
    }


    /* ========================================================================
     * AJAX: Get version history for an AI asset
     * ======================================================================== */

    public static function ajax_versions() {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_ajax_nonce');
        if (!current_user_can('upload_files')) {
            wp_send_json_error('Insufficient permissions', 403);
        }

        $attachment_id = (int) ($_POST['attachment_id'] ?? 0);
        if (!$attachment_id) {
            wp_send_json_error(['error' => 'Missing attachment_id'], 400);
        }

        $history = get_post_meta($attachment_id, '_lightsync_ai_version_history', true) ?: [];
        $current_version = (int) get_post_meta($attachment_id, '_lightsync_ai_version', true);

        // Add thumbnail URLs for versions that have backup files
        $upload_dir = wp_upload_dir();
        foreach ($history as &$entry) {
            $entry['has_backup'] = !empty($entry['backup_path']) && file_exists($entry['backup_path']);
            if ($entry['has_backup']) {
                // Generate a URL for the backup file
                $relative = str_replace($upload_dir['basedir'], '', $entry['backup_path']);
                $entry['backup_url'] = $upload_dir['baseurl'] . $relative;
            }
        }
        unset($entry);

        wp_send_json_success([
            'attachment_id'   => $attachment_id,
            'current_version' => $current_version,
            'history'         => $history,
        ]);
    }


    /* ========================================================================
     * AJAX: Rollback to a previous version
     * ======================================================================== */

    public static function ajax_rollback() {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_ajax_nonce');
        if (!current_user_can('upload_files')) {
            wp_send_json_error(['error' => 'Insufficient permissions']);
        }

        @ini_set('memory_limit', '512M');
        @set_time_limit(120);

        $attachment_id  = (int) ($_POST['attachment_id'] ?? 0);
        $target_version = (int) ($_POST['version'] ?? 0);

        if (!$attachment_id || !$target_version) {
            wp_send_json_error(['error' => 'Missing attachment_id or version']);
        }

        // Verify this is an AI asset
        $source = get_post_meta($attachment_id, '_lightsync_source', true);
        if ($source !== 'openrouter') {
            wp_send_json_error(['error' => 'Not an AI asset (source: ' . $source . ')']);
        }

        // Find the target version in history — prefer entry with valid backup_path
        $history = get_post_meta($attachment_id, '_lightsync_ai_version_history', true) ?: [];
        $target_entry = null;
        foreach ($history as $entry) {
            if (($entry['version'] ?? 0) === $target_version) {
                // Keep overwriting — last match with backup_path wins
                if (!empty($entry['backup_path'])) {
                    $target_entry = $entry;
                } elseif ($target_entry === null) {
                    $target_entry = $entry; // fallback to entry without backup
                }
            }
        }

        if (!$target_entry || empty($target_entry['backup_path']) || !file_exists($target_entry['backup_path'])) {
            wp_send_json_error(['error' => 'Version backup not found — cannot rollback']);
        }

        $backup_path = $target_entry['backup_path'];
        $current_path = get_attached_file($attachment_id);

        // Backup the CURRENT version before rolling back (so they can undo the undo)
        $current_version = (int) get_post_meta($attachment_id, '_lightsync_ai_version', true);
        $versions_dir = dirname($current_path) . '/lsp-versions/' . $attachment_id;
        if (!is_dir($versions_dir)) {
            wp_mkdir_p($versions_dir);
        }
        $cur_ext = pathinfo($current_path, PATHINFO_EXTENSION);
        $cur_backup = $versions_dir . '/v' . $current_version . '.' . $cur_ext;
        if (file_exists($current_path) && !file_exists($cur_backup)) {
            copy($current_path, $cur_backup);
            // Update or add current version in history with backup_path
            $found_current = false;
            foreach ($history as $idx => $h) {
                if (($h['version'] ?? 0) === $current_version) {
                    $history[$idx]['backup_path'] = $cur_backup;
                    $found_current = true;
                    break;
                }
            }
            if (!$found_current) {
                $history[] = [
                    'version'     => $current_version,
                    'prompt'      => get_post_meta($attachment_id, '_lightsync_ai_prompt', true),
                    'model'       => get_post_meta($attachment_id, '_lightsync_ai_model', true),
                    'created_at'  => get_post_meta($attachment_id, '_lightsync_ai_generated_at', true) ?: gmdate('c'),
                    'backup_path' => $cur_backup,
                ];
            }
            update_post_meta($attachment_id, '_lightsync_ai_version_history', $history);
        }

        // Restore the backup file
        $restore_ext = pathinfo($backup_path, PATHINFO_EXTENSION);
        $old_ext = pathinfo($current_path, PATHINFO_EXTENSION);
        $new_path = $current_path;

        if ($restore_ext !== $old_ext) {
            $new_path = preg_replace('/\.' . preg_quote($old_ext, '/') . '$/', '.' . $restore_ext, $current_path);
        }

        copy($backup_path, $new_path);

        if ($new_path !== $current_path) {
            update_attached_file($attachment_id, $new_path);
            @unlink($current_path);
        }

        // Regenerate thumbnails
        if (!function_exists('wp_generate_attachment_metadata')) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }
        $meta = wp_generate_attachment_metadata($attachment_id, $new_path);
        wp_update_attachment_metadata($attachment_id, $meta);

        // Update MIME type
        $mime = wp_check_filetype($new_path)['type'] ?: 'image/webp';
        wp_update_post([
            'ID'             => $attachment_id,
            'post_mime_type' => $mime,
        ]);

        // Restore metadata from the target version
        $rollback_version = $current_version + 1;
        self::save_ai_metadata($attachment_id, [
            'prompt'  => $target_entry['prompt'] ?? '',
            'model'   => $target_entry['model'] ?? '',
            'is_free' => false,
            'version' => $rollback_version,
        ]);

        // Flag destinations as stale
        $dest_map = get_post_meta($attachment_id, '_lightsync_destination_map', true) ?: [];
        $stale_destinations = array_keys($dest_map);

        Admin::add_activity(
            sprintf('AI Rolled back to v%d (now v%d): "%s"', $target_version, $rollback_version, wp_trim_words($target_entry['prompt'] ?? '', 8, '…')),
            'info',
            'ai'
        );

        wp_send_json_success([
            'attachment_id'      => $attachment_id,
            'version'            => $rollback_version,
            'restored_from'      => $target_version,
            'url'                => self::cache_bust_url(wp_get_attachment_url($attachment_id), $attachment_id),
            'width'              => $meta['width'] ?? 0,
            'height'             => $meta['height'] ?? 0,
            'filesize'           => filesize($new_path),
            'stale_destinations' => $stale_destinations,
        ]);
    }


    /* ========================================================================
     * AJAX: Re-optimize existing AI asset (convert to WebP in place)
     * Same attachment ID, same Shopify mapping — just better format/compression
     * ======================================================================== */

    public static function ajax_reoptimize() {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_ajax_nonce');
        if (!current_user_can('upload_files')) {
            wp_send_json_error('Insufficient permissions', 403);
        }

        @ini_set('memory_limit', '512M');
        @set_time_limit(120);

        $attachment_id = (int) ($_POST['attachment_id'] ?? 0);
        if (!$attachment_id) {
            wp_send_json_error(['error' => 'Missing attachment_id'], 400);
        }

        $current_path = get_attached_file($attachment_id);
        if (!$current_path || !file_exists($current_path)) {
            wp_send_json_error(['error' => 'Attachment file not found'], 404);
        }

        $current_mime = get_post_mime_type($attachment_id);

        // Already WebP? Nothing to do
        if ($current_mime === 'image/webp') {
            wp_send_json_success([
                'attachment_id' => $attachment_id,
                'already_webp'  => true,
                'message'       => 'Already optimized as WebP',
            ]);
        }

        // Run through prepareForMedia (handles resize + WebP conversion)
        $maxDim = (int) Admin::get_opt('max_dimension', 2048);
        $prepared = Image::prepareForMedia($current_path, $maxDim);

        if (empty($prepared['path']) || !file_exists($prepared['path'])) {
            wp_send_json_error(['error' => 'Image optimization failed'], 500);
        }

        // If format didn't change, nothing to update
        if ($prepared['mime'] === $current_mime && $prepared['path'] === $current_path) {
            wp_send_json_success([
                'attachment_id' => $attachment_id,
                'already_webp'  => true,
                'message'       => 'WebP not available on this server — image unchanged',
            ]);
        }

        $old_size = filesize($current_path);

        // Update the file
        $new_ext = self::mime_to_ext($prepared['mime']);
        $old_ext = pathinfo($current_path, PATHINFO_EXTENSION);
        $new_path = $current_path;

        if ($new_ext !== $old_ext) {
            $new_path = preg_replace('/\.' . preg_quote($old_ext, '/') . '$/', '.' . $new_ext, $current_path);
        }

        // Copy optimized file into place
        if ($prepared['path'] !== $new_path) {
            copy($prepared['path'], $new_path);
        }
        // Clean up temp file
        if ($prepared['path'] !== $current_path && $prepared['path'] !== $new_path) {
            @unlink($prepared['path']);
        }

        // Remove old file if extension changed
        if ($new_path !== $current_path) {
            @unlink($current_path);
            update_attached_file($attachment_id, $new_path);
        }

        // Regenerate thumbnails
        if (!function_exists('wp_generate_attachment_metadata')) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }
        $meta = wp_generate_attachment_metadata($attachment_id, $new_path);
        wp_update_attachment_metadata($attachment_id, $meta);

        // Update MIME type
        wp_update_post([
            'ID'             => $attachment_id,
            'post_mime_type' => $prepared['mime'],
        ]);

        $new_size = filesize($new_path);
        $savings = $old_size > 0 ? round((1 - $new_size / $old_size) * 100, 1) : 0;

        // Flag destinations as stale (file changed)
        $version = (int) get_post_meta($attachment_id, '_lightsync_ai_version', true);
        $dest_map = get_post_meta($attachment_id, '_lightsync_destination_map', true) ?: [];
        $stale_destinations = [];
        foreach ($dest_map as $dest_key => $mapping) {
            if (($mapping['version_pushed'] ?? 0) < $version) {
                $stale_destinations[] = $dest_key;
            }
        }

        // AVIF conversion
        $avif_result = null;
        if (class_exists('\LightSyncPro\LightSync_Compress')) {
            $avif = \LightSyncPro\LightSync_Compress::convert_to_avif($new_path);
            if (!empty($avif['ok'])) {
                $avif_result = [
                    'bytes_in'  => $avif['bytes_in'],
                    'bytes_out' => $avif['bytes_out'],
                    'savings'   => round((1 - $avif['bytes_out'] / max(1, $avif['bytes_in'])) * 100, 1),
                ];
            }
        }

        Admin::add_activity(
            sprintf('AI Re-optimized #%d: %s → %s (%s%% savings)', $attachment_id, strtoupper($old_ext), strtoupper($new_ext), $savings),
            'info',
            'ai'
        );

        wp_send_json_success([
            'attachment_id'      => $attachment_id,
            'url'                => self::cache_bust_url(wp_get_attachment_url($attachment_id), $attachment_id),
            'old_format'         => $current_mime,
            'new_format'         => $prepared['mime'],
            'old_size'           => $old_size,
            'new_size'           => $new_size,
            'savings'            => $savings,
            'avif'               => $avif_result,
            'width'              => $meta['width'] ?? 0,
            'height'             => $meta['height'] ?? 0,
            'stale_destinations' => $stale_destinations,
        ]);
    }


    /* ========================================================================
     * AJAX: Browse AI-Generated Assets
     * ======================================================================== */

    public static function ajax_browse() {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_ajax_nonce');
        if (!current_user_can('upload_files')) {
            wp_send_json_error('Insufficient permissions', 403);
        }

        $page     = max(1, (int) ($_POST['page'] ?? 1));
        $per_page = max(1, min(100, (int) ($_POST['per_page'] ?? 40)));

        $query = new \WP_Query([
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'posts_per_page' => $per_page,
            'paged'          => $page,
            'meta_query'     => [
                [
                    'key'   => '_lightsync_source',
                    'value' => 'openrouter',
                ],
            ],
            'orderby' => 'date',
            'order'   => 'DESC',
        ]);

        $items = [];
        foreach ($query->posts as $post) {
            $meta = wp_get_attachment_metadata($post->ID);
            $thumb = wp_get_attachment_image_url($post->ID, 'medium');

            $history = get_post_meta($post->ID, '_lightsync_ai_version_history', true) ?: [];

            $items[] = [
                'id'           => $post->ID,
                'url'          => self::cache_bust_url(wp_get_attachment_url($post->ID), $post->ID),
                'thumbnail'    => self::cache_bust_url($thumb ?: wp_get_attachment_url($post->ID), $post->ID),
                'filename'     => basename(get_attached_file($post->ID)),
                'width'        => $meta['width'] ?? 0,
                'height'       => $meta['height'] ?? 0,
                'prompt'       => get_post_meta($post->ID, '_lightsync_ai_prompt', true),
                'model'        => get_post_meta($post->ID, '_lightsync_ai_model', true),
                'version'      => (int) get_post_meta($post->ID, '_lightsync_ai_version', true),
                'has_history'  => count($history) > 0,
                'mime'         => get_post_mime_type($post->ID),
                'is_free'      => (bool) get_post_meta($post->ID, '_lightsync_ai_is_free', true),
                'generated_at' => get_post_meta($post->ID, '_lightsync_ai_generated_at', true),
                'destinations' => self::get_destination_summary($post->ID),
                'date'         => $post->post_date,
            ];
        }

        wp_send_json_success([
            'items'       => $items,
            'total'       => $query->found_posts,
            'total_pages' => $query->max_num_pages,
            'page'        => $page,
        ]);
    }


    /* ========================================================================
     * AJAX: Push Update to Destinations (Shopify file update in-place)
     * ======================================================================== */

    public static function ajax_push_destinations() {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_ajax_nonce');
        if (!current_user_can('upload_files')) {
            wp_send_json_error('Insufficient permissions', 403);
        }

        $attachment_id = (int) ($_POST['attachment_id'] ?? 0);
        if (!$attachment_id) {
            wp_send_json_error(['error' => 'Missing attachment_id'], 400);
        }

        // Verify this is an AI-generated asset
        $source = get_post_meta($attachment_id, '_lightsync_source', true);
        if ($source !== 'openrouter') {
            wp_send_json_error(['error' => 'This asset was not AI-generated'], 400);
        }

        $version   = (int) get_post_meta($attachment_id, '_lightsync_ai_version', true);
        $dest_map  = get_post_meta($attachment_id, '_lightsync_destination_map', true) ?: [];
        $force_dest = sanitize_text_field($_POST['force_dest'] ?? '');

        // If force_dest is set, ensure the destination exists in the map
        if ($force_dest && in_array($force_dest, ['shopify', 'both'], true)) {
            if (!isset($dest_map['shopify'])) {
                $dest_map['shopify'] = [
                    'file_id'        => '',
                    'file_url'       => '',
                    'version_pushed' => 0,
                    'pushed_at'      => '',
                ];
            }
        }

        if (empty($dest_map)) {
            wp_send_json_error(['error' => 'No destinations configured — select Shopify or Both as destination first'], 400);
        }

        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            wp_send_json_error(['error' => 'Attachment file not found'], 404);
        }

        $bytes    = file_get_contents($file_path);
        $filename = basename($file_path);
        $prompt   = get_post_meta($attachment_id, '_lightsync_ai_prompt', true);
        $alt_text = wp_trim_words($prompt, 12, '');
        $results  = [];

        foreach ($dest_map as $dest_key => $mapping) {
            // Skip if already current
            if (($mapping['version_pushed'] ?? 0) >= $version) {
                $results[$dest_key] = ['skipped' => true, 'already_current' => true];
                continue;
            }

            if ($dest_key === 'shopify') {
                if (!class_exists('\LightSyncPro\Shopify\Shopify')) {
                    $results[$dest_key] = ['error' => 'Shopify module not available'];
                    continue;
                }

                $source_id = 'ai_' . $attachment_id;

                // Use upload_file which handles update-in-place via fileUpdate mutation
                // when it detects an existing mapping (same source_id + different checksum)
                $s_result = \LightSyncPro\Shopify\Shopify::upload_file(
                    $bytes,
                    $filename,
                    $source_id,
                    'openrouter',
                    $alt_text
                );

                if (is_wp_error($s_result)) {
                    $results[$dest_key] = ['error' => $s_result->get_error_message()];
                } else {
                    // Update destination map with new version
                    $dest_map[$dest_key]['file_id']        = $s_result['file_id'] ?? $mapping['file_id'];
                    $dest_map[$dest_key]['file_url']        = $s_result['file_url'] ?? '';
                    $dest_map[$dest_key]['version_pushed']  = $version;
                    $dest_map[$dest_key]['pushed_at']        = gmdate('c');

                    $results[$dest_key] = [
                        'file_id'    => $s_result['file_id'] ?? '',
                        'updated'    => $s_result['updated'] ?? false,
                        'id_changed' => $s_result['id_changed'] ?? false,
                    ];
                }
            }
        }

        // Save updated destination map
        update_post_meta($attachment_id, '_lightsync_destination_map', $dest_map);

        // Update sync timestamp
        update_post_meta($attachment_id, '_lightsync_last_synced_at', gmdate('c'));
        update_post_meta($attachment_id, '_lightsync_last_sync_kind', 'ai-sync');

        // Activity log
        $success_count = 0;
        $fail_count = 0;
        foreach ($results as $r) {
            if (!empty($r['error'])) $fail_count++;
            elseif (empty($r['skipped'])) $success_count++;
        }
        if ($success_count > 0 || $fail_count > 0) {
            $push_prompt = wp_trim_words($prompt, 6, '…');
            if ($fail_count > 0) {
                Admin::add_activity(
                    sprintf('AI Sync v%d → Shopify: "%s" (failed)', $version, $push_prompt),
                    'error',
                    'ai'
                );
            } else {
                Admin::add_activity(
                    sprintf('AI Sync v%d → Shopify: "%s"', $version, $push_prompt),
                    'success',
                    'ai'
                );
            }
        }

        wp_send_json_success([
            'attachment_id' => $attachment_id,
            'version'       => $version,
            'results'       => $results,
        ]);
    }


    /* ========================================================================
     * Helper: Import image to WordPress media library
     * ======================================================================== */

    private static function import_to_media_library($file_path, $mime, $filename, $title = '') {
        if (!function_exists('wp_insert_attachment')) {
            require_once ABSPATH . 'wp-admin/includes/post.php';
        }
        if (!function_exists('wp_generate_attachment_metadata')) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }

        // Move to uploads directory
        $upload_dir = wp_upload_dir();
        $dest_path  = $upload_dir['path'] . '/' . $filename;

        // Ensure unique filename
        $dest_path = wp_unique_filename($upload_dir['path'], $filename);
        $dest_path = $upload_dir['path'] . '/' . $dest_path;

        if (!copy($file_path, $dest_path)) {
            return new \WP_Error('copy_failed', 'Failed to copy image to uploads directory');
        }

        // Create attachment post
        $attachment = [
            'post_mime_type' => $mime,
            'post_title'     => $title ?: pathinfo($filename, PATHINFO_FILENAME),
            'post_content'   => '',
            'post_status'    => 'inherit',
        ];

        $attachment_id = wp_insert_attachment($attachment, $dest_path);

        if (is_wp_error($attachment_id)) {
            @unlink($dest_path);
            return $attachment_id;
        }

        // Generate metadata (thumbnails, sizes, etc.)
        $meta = wp_generate_attachment_metadata($attachment_id, $dest_path);
        wp_update_attachment_metadata($attachment_id, $meta);

        return $attachment_id;
    }


    /* ========================================================================
     * Helper: Save AI metadata on attachment
     * ======================================================================== */

    private static function save_ai_metadata(int $attachment_id, array $data) {
        // Core source tracking (same key all sources use)
        update_post_meta($attachment_id, '_lightsync_source', 'openrouter');

        // AI-specific metadata
        if (!empty($data['prompt'])) {
            update_post_meta($attachment_id, '_lightsync_ai_prompt', $data['prompt']);
        }
        if (!empty($data['model'])) {
            update_post_meta($attachment_id, '_lightsync_ai_model', $data['model']);
        }
        if (!empty($data['aspect_ratio'])) {
            update_post_meta($attachment_id, '_lightsync_ai_aspect_ratio', $data['aspect_ratio']);
        }
        
        update_post_meta($attachment_id, '_lightsync_ai_is_free', !empty($data['is_free']));
        update_post_meta($attachment_id, '_lightsync_ai_version', (int) ($data['version'] ?? 1));

        // Timestamp
        if (empty(get_post_meta($attachment_id, '_lightsync_ai_generated_at', true))) {
            update_post_meta($attachment_id, '_lightsync_ai_generated_at', gmdate('c'));
        }

        // Sync tracking
        update_post_meta($attachment_id, '_lightsync_last_synced_at', gmdate('c'));
        update_post_meta($attachment_id, '_lightsync_last_sync_kind', 'ai-generate');

        // Initialize version history if first version
        $version = (int) ($data['version'] ?? 1);
        if ($version === 1) {
            update_post_meta($attachment_id, '_lightsync_ai_version_history', [
                [
                    'version'    => 1,
                    'prompt'     => $data['prompt'] ?? '',
                    'model'      => $data['model'] ?? '',
                    'created_at' => gmdate('c'),
                ],
            ]);
        }
    }


    /* ========================================================================
     * Helper: Get destination sync summary for an asset
     * ======================================================================== */

    private static function get_destination_summary(int $attachment_id): array {
        $dest_map = get_post_meta($attachment_id, '_lightsync_destination_map', true);
        if (!is_array($dest_map) || empty($dest_map)) {
            return [];
        }

        $version = (int) get_post_meta($attachment_id, '_lightsync_ai_version', true);
        $summary = [];

        foreach ($dest_map as $dest_key => $mapping) {
            $summary[] = [
                'key'        => $dest_key,
                'site_id'    => $mapping['site_id'] ?? '',
                'is_current' => ($mapping['version_pushed'] ?? 0) >= $version,
                'pushed_at'  => $mapping['pushed_at'] ?? '',
            ];
        }

        return $summary;
    }


    /* ========================================================================
     * Helper: MIME type to file extension
     * ======================================================================== */

    private static function mime_to_ext(string $mime): string {
        $map = [
            'image/png'  => 'png',
            'image/jpeg' => 'jpg',
            'image/webp' => 'webp',
            'image/avif' => 'avif',
            'image/gif'  => 'gif',
            'image/svg+xml' => 'svg',
        ];
        return $map[$mime] ?? 'png';
    }


    /* ========================================================================
     * Helper: Cache-bust an image URL using file modification time
     * Prevents browser/CDN from serving stale images after regenerate/reoptimize
     * ======================================================================== */

    private static function cache_bust_url(string $url, int $attachment_id = 0): string {
        $ts = 0;
        if ($attachment_id) {
            $path = get_attached_file($attachment_id);
            if ($path && file_exists($path)) {
                $ts = filemtime($path);
            }
        }
        if (!$ts) $ts = time();
        $sep = (strpos($url, '?') !== false) ? '&' : '?';
        return $url . $sep . 'lsp=' . $ts;
    }


    /* ========================================================================
     * AJAX: Start Background Sync (queue IDs for cron processing)
     * ======================================================================== */

    public static function ajax_background_sync() {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_ajax_nonce');
        if (!current_user_can('upload_files')) {
            wp_send_json_error('Insufficient permissions', 403);
        }

        $ids  = array_map('intval', $_POST['attachment_ids'] ?? []);
        $dest = sanitize_text_field($_POST['dest'] ?? 'shopify');

        if (empty($ids)) {
            wp_send_json_error(['error' => 'No images selected'], 400);
        }

        // Store queue in transient
        $queue = [
            'ids'       => $ids,
            'dest'      => $dest,
            'total'     => count($ids),
            'completed' => 0,
            'errors'    => [],
            'status'    => 'running',
            'started'   => time(),
        ];
        set_transient('lsp_ai_bg_sync', $queue, HOUR_IN_SECONDS);

        // Schedule first tick immediately
        if (!wp_next_scheduled('lsp_ai_background_sync_tick')) {
            wp_schedule_single_event(time(), 'lsp_ai_background_sync_tick');
        }

        // Also try to process one immediately for responsiveness
        self::process_one_background_item();

        wp_send_json_success(['queued' => count($ids)]);
    }


    /* ========================================================================
     * AJAX: Check Background Sync Status
     * ======================================================================== */

    public static function ajax_background_status() {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_ajax_nonce');

        $queue = get_transient('lsp_ai_bg_sync');
        if (!$queue) {
            wp_send_json_success([
                'status'    => 'idle',
                'total'     => 0,
                'completed' => 0,
            ]);
            return;
        }

        wp_send_json_success([
            'status'    => $queue['status'] ?? 'running',
            'total'     => $queue['total'] ?? 0,
            'completed' => $queue['completed'] ?? 0,
            'errors'    => count($queue['errors'] ?? []),
        ]);
    }


    /* ========================================================================
     * Cron: Process background sync tick
     * ======================================================================== */

    public static function cron_background_tick() {
        self::process_one_background_item();
    }

    private static function process_one_background_item() {
        $queue = get_transient('lsp_ai_bg_sync');
        if (!$queue || ($queue['status'] ?? '') === 'complete') return;

        $ids       = $queue['ids'] ?? [];
        $dest      = $queue['dest'] ?? 'shopify';
        $completed = $queue['completed'] ?? 0;

        if ($completed >= count($ids)) {
            $queue['status'] = 'complete';
            set_transient('lsp_ai_bg_sync', $queue, 5 * MINUTE_IN_SECONDS);
            return;
        }

        $attachment_id = $ids[$completed];

        // Verify this is an AI-generated asset
        $source = get_post_meta($attachment_id, '_lightsync_source', true);
        if ($source !== 'openrouter') {
            $queue['errors'][] = ['id' => $attachment_id, 'error' => 'Not an AI image'];
            $queue['completed'] = $completed + 1;
            set_transient('lsp_ai_bg_sync', $queue, HOUR_IN_SECONDS);
            // Schedule next tick
            wp_schedule_single_event(time() + 2, 'lsp_ai_background_sync_tick');
            return;
        }

        $version   = (int) get_post_meta($attachment_id, '_lightsync_ai_version', true);
        $dest_map  = get_post_meta($attachment_id, '_lightsync_destination_map', true) ?: [];

        // Ensure shopify mapping exists
        if (in_array($dest, ['shopify', 'both'], true) && !isset($dest_map['shopify'])) {
            $dest_map['shopify'] = [
                'file_id'        => '',
                'file_url'       => '',
                'version_pushed' => 0,
                'pushed_at'      => '',
            ];
        }

        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            $queue['errors'][] = ['id' => $attachment_id, 'error' => 'File not found'];
            $queue['completed'] = $completed + 1;
            set_transient('lsp_ai_bg_sync', $queue, HOUR_IN_SECONDS);
            wp_schedule_single_event(time() + 2, 'lsp_ai_background_sync_tick');
            return;
        }

        $bytes    = file_get_contents($file_path);
        $filename = basename($file_path);
        $prompt   = get_post_meta($attachment_id, '_lightsync_ai_prompt', true);
        $alt_text = wp_trim_words($prompt, 15, '…');

        // Push to Shopify
        $pushed = false;
        if (isset($dest_map['shopify']) && class_exists('\LightSyncPro\Shopify\Shopify')) {
            $source_id = 'ai_' . $attachment_id;
            $s_result = \LightSyncPro\Shopify\Shopify::upload_file(
                $bytes, $filename, $source_id, 'openrouter', $alt_text
            );

            if (is_wp_error($s_result)) {
                $queue['errors'][] = ['id' => $attachment_id, 'error' => $s_result->get_error_message()];
            } else {
                $dest_map['shopify']['file_id']        = $s_result['file_id'] ?? '';
                $dest_map['shopify']['file_url']        = $s_result['file_url'] ?? '';
                $dest_map['shopify']['version_pushed']  = $version ?: 1;
                $dest_map['shopify']['pushed_at']        = gmdate('c');
                $pushed = true;
            }
        }

        if ($pushed) {
            update_post_meta($attachment_id, '_lightsync_destination_map', $dest_map);
            update_post_meta($attachment_id, '_lightsync_last_synced_at', gmdate('c'));
            update_post_meta($attachment_id, '_lightsync_last_sync_kind', 'ai-bg-sync');
        }

        $queue['completed'] = $completed + 1;

        if ($queue['completed'] >= count($ids)) {
            $queue['status'] = 'complete';
            set_transient('lsp_ai_bg_sync', $queue, 5 * MINUTE_IN_SECONDS);

            // Activity log
            $err_count = count($queue['errors'] ?? []);
            $ok_count  = $queue['completed'] - $err_count;
            Admin::add_activity(
                sprintf('AI Background Sync Complete: %d image(s) → Shopify%s', $ok_count, $err_count ? " ({$err_count} failed)" : ''),
                $err_count ? 'warning' : 'success',
                'ai'
            );
        } else {
            set_transient('lsp_ai_bg_sync', $queue, HOUR_IN_SECONDS);
            // Schedule next tick in 2 seconds (avoid rate limiting)
            wp_schedule_single_event(time() + 2, 'lsp_ai_background_sync_tick');
        }
    }
}
