/* global jQuery, ajaxurl */
jQuery(function ($) {
  "use strict";

  // ----------------------------
  // Minimal modal (no dependencies)
  // ----------------------------
  function ensureModal() {
    if ($("#lsp-relink-modal").length) return;

    const css = `
#lsp-relink-modal{position:fixed;inset:0;z-index:999999;display:none}
#lsp-relink-modal .lsp-backdrop{position:absolute;inset:0;background:rgba(0,0,0,.55)}
#lsp-relink-modal .lsp-panel{position:relative;max-width:760px;width:calc(100% - 40px);max-height:calc(100% - 40px);margin:20px auto;background:#fff;border-radius:14px;box-shadow:0 20px 60px rgba(0,0,0,.35);display:flex;flex-direction:column;overflow:hidden}
#lsp-relink-modal .lsp-head{padding:14px 16px;border-bottom:1px solid rgba(0,0,0,.08);display:flex;align-items:center;justify-content:space-between;gap:12px}
#lsp-relink-modal .lsp-title{font-size:14px;font-weight:600}
#lsp-relink-modal .lsp-close{border:0;background:transparent;font-size:22px;line-height:1;cursor:pointer;padding:0 6px}
#lsp-relink-modal .lsp-body{padding:14px 16px;display:flex;flex-direction:column;gap:12px;overflow:auto}
#lsp-relink-modal .lsp-meta{font-size:12px;opacity:.85}
#lsp-relink-modal .lsp-search{display:flex;gap:8px;align-items:center}
#lsp-relink-modal input[type="search"]{width:100%;padding:10px 12px;border:1px solid rgba(0,0,0,.15);border-radius:10px;font-size:13px}
#lsp-relink-modal .lsp-list{
  border:1px solid rgba(0,0,0,.10);
  border-radius:12px;
  overflow:auto;
  max-height:360px; /* ✅ scrollable list */
}
#lsp-relink-modal .lsp-row{display:flex;justify-content:space-between;gap:14px;align-items:flex-start;padding:10px 12px;border-bottom:1px solid rgba(0,0,0,.08)}
#lsp-relink-modal .lsp-row:last-child{border-bottom:0}
#lsp-relink-modal .lsp-left{display:flex;flex-direction:column;gap:2px}
#lsp-relink-modal .lsp-name{font-size:13px;font-weight:600}
#lsp-relink-modal .lsp-sub{font-size:12px;opacity:.8}
#lsp-relink-modal .lsp-btn{border:1px solid rgba(0,0,0,.18);background:#f6f7f7;border-radius:10px;padding:8px 10px;font-size:12px;cursor:pointer}
#lsp-relink-modal .lsp-btn:disabled{opacity:.6;cursor:not-allowed}
#lsp-relink-modal .lsp-foot{padding:12px 16px;border-top:1px solid rgba(0,0,0,.08);display:flex;justify-content:space-between;align-items:center;gap:12px}
#lsp-relink-modal .lsp-status{font-size:12px;opacity:.9}
#lsp-relink-modal .lsp-pill{display:inline-flex;gap:6px;align-items:center;font-size:11px;border:1px solid rgba(0,0,0,.15);border-radius:999px;padding:2px 8px;opacity:.85}
#lsp-relink-modal .lsp-muted{opacity:.65}
#lsp-relink-modal .lsp-head-left{display:flex;align-items:center;gap:10px}
#lsp-relink-modal .lsp-logo{width:18px;height:18px;display:inline-flex;align-items:center;justify-content:center}
#lsp-relink-modal .lsp-logo svg{width:18px;height:18px;display:block}
#lsp-relink-modal .lsp-head-left{
  display:flex;
  align-items:center;
  gap:10px;
}

#lsp-relink-modal .lsp-logo img{
  display:block;
  width:18px;
  height:18px;
}


    `;

    $("head").append("<style id='lsp-relink-css'>" + css + "</style>");

    const html = `
<div id="lsp-relink-modal" aria-hidden="true">
  <div class="lsp-backdrop"></div>
  <div class="lsp-panel" role="dialog" aria-modal="true">
    <div class="lsp-head">
  <div class="lsp-head-left">
    <span class="lsp-logo">
     <img
    src="${window.LIGHTSYNCPRO && LIGHTSYNCPRO.pluginUrl ? LIGHTSYNCPRO.pluginUrl + '/assets/lsp-icon.svg' : ''}"
    alt="LightSync Pro"
    width="18"
    height="18"
    decoding="async"
  />
    </span>
    <div class="lsp-title">Relink to another version</div>
  </div>
  <button class="lsp-close" type="button" aria-label="Close">×</button>
</div>
    <div class="lsp-body">
    <div class="lsp-muted lsp-tip">
  Tip: Switching versions is non-destructive. Previous images are safely archived.
</div>
      <div class="lsp-meta"></div>
      <div class="lsp-search">
        <input type="search" class="lsp-q" placeholder="Search by title or asset ID…" autocomplete="off" />
        <span class="lsp-pill"><span class="lsp-count">0</span><span class="lsp-muted">shown</span></span>
      </div>
      <div class="lsp-list"></div>
    </div>
    <div class="lsp-foot">
      <div class="lsp-status lsp-muted">Ready.</div>
      <button class="lsp-btn lsp-cancel" type="button">Close</button>
    </div>
  </div>
</div>
    `;
    $("body").append(html);

    // Close handlers
    $(document).on("click", "#lsp-relink-modal .lsp-close, #lsp-relink-modal .lsp-cancel, #lsp-relink-modal .lsp-backdrop", function () {
      closeModal();
    });

    // ESC to close
    $(document).on("keydown", function (e) {
      if (e.key === "Escape" && $("#lsp-relink-modal").is(":visible")) closeModal();
    });
  }

  function openModal() {
    ensureModal();
    $("#lsp-relink-modal").fadeIn(120).attr("aria-hidden", "false");
    $("body").addClass("lsp-modal-open");
  }

  function closeModal() {
    $("#lsp-relink-modal").fadeOut(120).attr("aria-hidden", "true");
    $("body").removeClass("lsp-modal-open");
  }

  function setStatus(msg, muted) {
    const $s = $("#lsp-relink-modal .lsp-status");
    $s.text(msg || "");
    $s.toggleClass("lsp-muted", !!muted);
  }

  // ----------------------------
  // Rendering + client-side filter
  // ----------------------------
  function renderList(items, currentAssetId) {
    const $list = $("#lsp-relink-modal .lsp-list");
    $list.empty();

    if (!items || !items.length) {
        $list.append(
            `<div class="lsp-row"><div class="lsp-left"><div class="lsp-name">No candidates found.</div><div class="lsp-sub">Try another search.</div></div></div>`
        );
        $("#lsp-relink-modal .lsp-count").text("0");
        return;
    }

    items.forEach(function (it) {
        const title = (it.title || it.name || it.fileName || it.filename || "").trim();
        const updated = (it.updated || it.updated_label || it.date || it.modified || "").trim();
        const aid = (it.asset_id || it.id || "").trim();
        const isImported = !!it.is_imported;
        const isCurrent = !!it.is_current;
        const existingAttId = it.attachment_id || 0;

        const name = title ? title : "(No title)";
        const sub = `${aid}${updated ? " • " + updated : ""}`;

        // Build badges
        let badges = '';
        if (isCurrent) {
            badges += '<span class="lsp-pill" style="margin-left:6px;background:#d1fae5;color:#065f46;border-color:#a7f3d0;">Current</span>';
        } else if (isImported) {
            badges += '<span class="lsp-pill" style="margin-left:6px;background:#fef3c7;color:#92400e;border-color:#fde68a;">In Library</span>';
        }

        // Button text based on state
        let btnText = 'Switch';
        let btnDisabled = isCurrent;
        if (isImported && !isCurrent) {
            btnText = 'Replace';
        }

        $list.append(`
            <div class="lsp-row" data-aid="${escapeHtml(aid)}" data-title="${escapeHtml(title.toLowerCase())}" data-imported="${isImported ? '1' : '0'}" data-att="${existingAttId}">
                <div class="lsp-left">
                    <div class="lsp-name">${escapeHtml(name)}${badges}</div>
                    <div class="lsp-sub">${escapeHtml(sub)}</div>
                </div>
                <div>
                    <button type="button" class="lsp-btn lsp-pick" ${btnDisabled ? "disabled" : ""}>${btnText}</button>
                </div>
            </div>
        `);
    });

    $("#lsp-relink-modal .lsp-count").text(String(items.length));
}

  function applyFilter(allItems, q) {
    q = (q || "").toLowerCase().trim();
    if (!q) return allItems;

    return allItems.filter(function (it) {
      const aid = (it.asset_id || it.id || "").toLowerCase();
      const title = (it.title || it.name || it.fileName || it.filename || "").toLowerCase();
      return aid.includes(q) || title.includes(q);
    });
  }

  function escapeHtml(str) {
    return String(str || "")
      .replaceAll("&", "&amp;")
      .replaceAll("<", "&lt;")
      .replaceAll(">", "&gt;")
      .replaceAll('"', "&quot;")
      .replaceAll("'", "&#039;");
  }

  // ----------------------------
  // Click: Relink (open + fetch)
  // ----------------------------
  let STATE = {
    attachmentId: 0,
    nonce: "",
    currentAsset: "",
    allCandidates: [],
    source: "lightroom", // 'lightroom', 'canva', 'figma', or 'dropbox'
  };

  // Important: prevent default so links don't turn red / add hashes
  $(document).on("click", "a.lsp-relink, button.lsp-relink", function (e) {
    e.preventDefault();
    e.stopPropagation();

    const $el = $(this);
    const att = parseInt($el.data("att"), 10) || 0;
    const nonce = String($el.data("nonce") || "");
    const asset = String($el.data("asset") || "");
    const source = String($el.data("source") || "lightroom");

    if (!att || !nonce) return;

    STATE.attachmentId = att;
    STATE.nonce = nonce;
    STATE.currentAsset = asset;
    STATE.allCandidates = [];
    STATE.source = source;

    openModal();
    
    // Update modal title based on source
    if (source === "canva") {
      $("#lsp-relink-modal .lsp-title").text("Canva Design Versions");
      $("#lsp-relink-modal .lsp-tip").text("To update this design, make changes in Canva and resync from the LightSync Pro dashboard.");
    } else if (source === "figma") {
      $("#lsp-relink-modal .lsp-title").text("Figma Frame Versions");
      $("#lsp-relink-modal .lsp-tip").text("To update this frame, edit it in Figma and resync from the LightSync Pro dashboard.");
    } else if (source === "dropbox") {
      $("#lsp-relink-modal .lsp-title").text("Dropbox File Versions");
      $("#lsp-relink-modal .lsp-tip").text("To update this file, replace it in Dropbox and resync from the LightSync Pro dashboard.");
    } else {
      $("#lsp-relink-modal .lsp-title").text("Relink to another Lightroom version");
      $("#lsp-relink-modal .lsp-tip").text("Tip: Switching versions is non-destructive. Previous images are safely archived.");
    }
    
    $("#lsp-relink-modal .lsp-q").val("").trigger("input");
    $("#lsp-relink-modal .lsp-meta").text("Loading candidates…");
    setStatus("Fetching assets…", true);
    $("#lsp-relink-modal .lsp-list").html(
      `<div class="lsp-row"><div class="lsp-left"><div class="lsp-name">Loading…</div><div class="lsp-sub">Please wait</div></div></div>`
    );
    $("#lsp-relink-modal .lsp-count").text("0");

    // Different handling for Canva vs Lightroom vs Figma
    if (source === "canva") {
      // For Canva, show explanation that versioning works differently
      $("#lsp-relink-modal .lsp-meta").text("Canva designs sync as snapshots.");
      $("#lsp-relink-modal .lsp-list").html(
        `<div class="lsp-row">
          <div class="lsp-left">
            <div class="lsp-name">Canva Version History</div>
            <div class="lsp-sub">Canva designs don't have album-based versioning like Lightroom.</div>
          </div>
        </div>
        <div class="lsp-row">
          <div class="lsp-left">
            <div class="lsp-name" style="color:#0f172a;">How to update this design:</div>
            <div class="lsp-sub">
              1. Edit the design in Canva<br>
              2. Go to LightSync Pro → Canva Designs<br>
              3. Select this design and click Sync Now
            </div>
          </div>
        </div>
        <div class="lsp-row">
          <div class="lsp-left">
            <div class="lsp-name" style="color:#0f172a;">The updated version will replace this image</div>
            <div class="lsp-sub">Previous versions are kept in WordPress Trash for 30 days.</div>
          </div>
        </div>`
      );
      setStatus("Canva designs sync on-demand.", true);
      return;
    }

    if (source === "figma") {
      // For Figma, show explanation similar to Canva
      $("#lsp-relink-modal .lsp-meta").text("Figma frames sync as snapshots.");
      $("#lsp-relink-modal .lsp-list").html(
        `<div class="lsp-row">
          <div class="lsp-left">
            <div class="lsp-name">Figma Version History</div>
            <div class="lsp-sub">Figma frames are exported as point-in-time snapshots.</div>
          </div>
        </div>
        <div class="lsp-row">
          <div class="lsp-left">
            <div class="lsp-name" style="color:#0f172a;">How to update this frame:</div>
            <div class="lsp-sub">
              1. Edit the frame in Figma<br>
              2. Go to LightSync Pro → Figma Frames<br>
              3. Select this frame and click Sync Now
            </div>
          </div>
        </div>
        <div class="lsp-row">
          <div class="lsp-left">
            <div class="lsp-name" style="color:#0f172a;">The updated version will replace this image</div>
            <div class="lsp-sub">Previous versions are kept in WordPress Trash for 30 days.</div>
          </div>
        </div>`
      );
      setStatus("Figma frames sync on-demand.", true);
      return;
    }

    if (source === "dropbox") {
      // For Dropbox, show explanation similar to Canva/Figma
      $("#lsp-relink-modal .lsp-meta").text("Dropbox files sync as snapshots.");
      $("#lsp-relink-modal .lsp-list").html(
        `<div class="lsp-row">
          <div class="lsp-left">
            <div class="lsp-name">Dropbox Version History</div>
            <div class="lsp-sub">Dropbox files are imported as point-in-time copies.</div>
          </div>
        </div>
        <div class="lsp-row">
          <div class="lsp-left">
            <div class="lsp-name" style="color:#0f172a;">How to update this file:</div>
            <div class="lsp-sub">
              1. Update or replace the file in Dropbox<br>
              2. Go to LightSync Pro → Dropbox<br>
              3. Navigate to the file and click Sync Now
            </div>
          </div>
        </div>
        <div class="lsp-row">
          <div class="lsp-left">
            <div class="lsp-name" style="color:#0f172a;">The updated version will replace this image</div>
            <div class="lsp-sub">Previous versions are kept in WordPress Trash for 30 days.</div>
          </div>
        </div>`
      );
      setStatus("Dropbox files sync on-demand.", true);
      return;
    }

    // Lightroom - fetch candidates from album
    $.post(ajaxurl, {
      action: "lightsync_relink_candidates",
      nonce: nonce,
      attachment_id: att,
    })
      .done(function (resp) {
        if (!resp || !resp.success) {
          const msg = resp && resp.data && resp.data.error ? resp.data.error : "Unknown error.";
          setStatus("Error: " + msg, false);
          $("#lsp-relink-modal .lsp-meta").text("Could not load candidates.");
          $("#lsp-relink-modal .lsp-list").html(
            `<div class="lsp-row"><div class="lsp-left"><div class="lsp-name">Failed to load.</div><div class="lsp-sub">${escapeHtml(
              msg
            )}</div></div></div>`
          );
          return;
        }

        const d = resp.data || {};
        const albumId = d.album_id || "";
        const catalogId = d.catalog_id || "";
        const current = d.current_asset || STATE.currentAsset || "";
        const candidates = Array.isArray(d.candidates) ? d.candidates : [];

        STATE.currentAsset = current;
        STATE.allCandidates = candidates;

        $("#lsp-relink-modal .lsp-meta").text(
          `Album-scoped list • Catalog: ${catalogId} • Album: ${albumId} • Candidates: ${candidates.length}`
        );

        const filtered = applyFilter(STATE.allCandidates, "");
        renderList(filtered, STATE.currentAsset);
        setStatus("Pick a version to switch to (same album).", true);
      })
      .fail(function (jqXHR, textStatus, errorThrown) {
        const statusCode = jqXHR.status || "N/A";
        const errMsg = errorThrown || textStatus || "Unknown error";
        console.error("LSP Relink candidates failed:", statusCode, errMsg, jqXHR.responseText);
        setStatus("Request failed: " + statusCode + " " + errMsg, false);
        $("#lsp-relink-modal .lsp-meta").text("Could not load candidates.");
        $("#lsp-relink-modal .lsp-list").html(
          `<div class="lsp-row"><div class="lsp-left"><div class="lsp-name">Request failed (${statusCode})</div><div class="lsp-sub">${escapeHtml(errMsg)} — Check browser console for details.</div></div></div>`
        );
      });
  });

  // ----------------------------
  // Search input: instant filter
  // ----------------------------
  $(document).on("input", "#lsp-relink-modal .lsp-q", function () {
    const q = $(this).val();
    const filtered = applyFilter(STATE.allCandidates, q);
    renderList(filtered, STATE.currentAsset);
    setStatus(filtered.length ? "Showing filtered results." : "No matches.", true);
  });

  // ----------------------------
  // Click: Switch to selected asset
  // ----------------------------
$(document).on("click", "#lsp-relink-modal .lsp-pick", function () {
    const $btn = $(this);
    const $row = $btn.closest(".lsp-row");
    const newAsset = String($row.data("aid") || "");
    const isImported = $row.data("imported") === "1" || $row.data("imported") === 1;
    const existingAttId = parseInt($row.data("att"), 10) || 0;

    if (!STATE.attachmentId || !STATE.nonce || !newAsset) return;

    // Confirm if switching to an already-imported version
    if (isImported && existingAttId && existingAttId !== STATE.attachmentId) {
        const confirmMsg = 
            'This version already exists in your Media Library (Attachment #' + existingAttId + ').\n\n' +
            'Switching will update the current attachment to use this Lightroom asset. ' +
            'The other attachment (#' + existingAttId + ') will remain unchanged.\n\n' +
            'Continue?';
        if (!confirm(confirmMsg)) {
            return;
        }
    }

    $btn.prop("disabled", true).text("Switching…");
    setStatus("Switching attachment source…", true);

    $.post(ajaxurl, {
        action: "lightsync_switch_source",
        nonce: STATE.nonce,
        attachment_id: STATE.attachmentId,
        new_asset_id: newAsset,
    })
   .done(function (resp) {
    if (!resp || !resp.success) {
        const msg = resp && resp.data && resp.data.detail ? resp.data.detail : "Switch failed.";
        setStatus("Error: " + msg, false);
        $btn.prop("disabled", false).text("Switch");
        return;
    }

    const d = resp.data || {};
    STATE.currentAsset = d.new_asset_id || newAsset;

    // Build status message
    let statusMsg = "Switched successfully.";
    if (d.unlinked_count && d.unlinked_count > 0) {
        statusMsg += " (" + d.unlinked_count + " duplicate link" + (d.unlinked_count > 1 ? "s" : "") + " removed)";
    }
    
    if (d.avif_primary === true) {
        statusMsg = "Switched. AVIF generated (WebP backup kept).";
    } else if (d.avif_primary === false) {
        statusMsg = "Switched. Using WebP.";
    }
    
    setStatus(statusMsg, true);
    

        // Best-effort: update the column badge/label without refresh
        try {
          const $tr = $("#post-" + STATE.attachmentId);
          if ($tr.length) {
            $tr.find(".lsp-relink").attr("data-asset", STATE.currentAsset).data("asset", STATE.currentAsset);

            const kind = (d.kind || "SWITCH").toString();
            const label = (d.label || "").toString();

            $tr.find("td.lightsync_synced, td.column-lsp_synced")
              .find("span")
              .each(function () {
                const t = $(this).text().trim();
                if (t === "RELINK" || t === "SWITCH" || t === "SYNC" || t === "UPDATE" || t === "IMPORT") {
                  $(this).text(kind);
                }
              });

            if (label) {
              $tr.find("td.lightsync_synced, td.column-lsp_synced")
                .find("span")
                .each(function () {
                  const t = $(this).text();
                  if (t && t.match(/\d{4}|am|pm|:/i)) {
                    $(this).text(label);
                    return false;
                  }
                });
            }
          }
        } catch (e) {}

        // Update list to mark "Current"
     // Update candidates data to reflect the switch
STATE.allCandidates = STATE.allCandidates.map(function(item) {
    const aid = item.asset_id || item.id || "";
    
    // Mark the new asset as current
    if (aid === STATE.currentAsset) {
        item.is_current = true;
        item.is_imported = true;
        item.attachment_id = STATE.attachmentId;
    }
    
    // If this was the unlinked attachment, mark it as no longer imported
    if (item.attachment_id === existingAttId && aid !== STATE.currentAsset && d.unlinked_count > 0) {
        item.is_imported = false;
        item.attachment_id = 0;
    }
    
    return item;
});

// Re-render the list with updated data
const q = $("#lsp-relink-modal .lsp-q").val();
const filtered = applyFilter(STATE.allCandidates, q);
renderList(filtered, STATE.currentAsset);

        // keep open
      })
      .fail(function (jqXHR, textStatus, errorThrown) {
        const statusCode = jqXHR.status || "N/A";
        const errMsg = errorThrown || textStatus || "Unknown error";
        console.error("LSP Switch failed:", statusCode, errMsg, jqXHR.responseText);
        setStatus("Request failed: " + statusCode + " " + errMsg, false);
        $btn.prop("disabled", false).text("Switch");
      });
  });

  // Prevent list-view row click from hijacking our links/buttons
  $(document).on("click", "a.lsp-unlink", function (e) {
    e.preventDefault();
    e.stopPropagation();
  });
});