(function ($) {
  /* ---------- small helpers ---------- */
  function param(name){
    var m = new RegExp('[?&]'+name+'=([^&]+)').exec(window.location.search);
    return m ? decodeURIComponent(m[1].replace(/\+/g,' ')) : '';
  }

  function api(action, data){
    data = data || {};
    data.action = action;
    data._ajax_nonce = LIGHTSYNCPRO.nonce;
    return $.post(LIGHTSYNCPRO.ajaxurl, data, null, 'json');
  }

  function showError(where, msg){
    try {
      var $out = (where && $(where).length) ? $(where) : $('#lsp-log');
      $out.show().append(
        $('<div class="lsp-error"/>').text(typeof msg==='string' ? msg : (msg && msg.message) || 'Unknown error')
      );
    } catch(e){ alert(msg); }
  }

  async function fetchJSON(url, opts){
    const r = await fetch(url, opts || {});
    const j = await r.json().catch(()=>({}));
    if (!r.ok) throw new Error(j.error || ('HTTP '+r.status));
    return j;
  }

  function percent(n,d){ n=Number(n||0); d=Number(d||0); if(!d) return 0; return Math.min(100, Math.round((n/d)*100)); }

  function escapeHtml(str){
    if (!str) return '';
    return String(str).replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/"/g,'&quot;');
  }

  function updateBar(barSel, valSel, used, max){
    used = Number(used||0); max = Number(max||0);
    const pct = percent(used, max);
    $(barSel).css('width', pct + '%');
    $(valSel).text( used + '/' + (max||0) );
  }

  // Optional: catch hard JS errors and surface them in the UI
  window.addEventListener('error', function(e){
    try { $('#lsp-log').show().append($('<div class="lsp-error"/>').text('JS error: ' + e.message)); } catch(_) {}
  });

  /* ---------- 1) broker pickup on return ---------- */
  function pickup(){
    var connected = param('lsp_connected');
    var state = param('state');
    if (connected !== '1' || !state) return;

    console.log('[LSP] Pickup: starting broker token retrieval...');

    $.getJSON(LIGHTSYNCPRO.broker_pickup, { state: state })
      .done(function(resp){
        console.log('[LSP] Pickup: broker response', resp);
        
        if (!resp || !resp.success || !resp.data || !resp.data.broker_token){
          console.error('[LSP] Pickup failed - invalid response', resp);
          cleanupUrl();
          return;
        }
        
        var brokerToken = resp.data.broker_token;
        var clientId    = resp.data.client_id || '';
        
        console.log('[LSP] Pickup: saving broker token...');

        $.post(LIGHTSYNCPRO.ajaxurl, {
          action:       'lightsyncpro_ajax_save_broker',
          _ajax_nonce:  LIGHTSYNCPRO.nonce,
          broker_token: brokerToken,
          client_id:    clientId
        })
        .done(function(saveResp){
          console.log('[LSP] Pickup: save response', saveResp);
          if (saveResp && saveResp.success) {
            console.log('[LSP] Pickup: success, reloading page...');
            // Force reload to show connected state
            var url = new URL(window.location.href);
            url.searchParams.delete('lsp_connected');
            url.searchParams.delete('state');
            window.location.replace(url.toString());
          } else {
            console.error('[LSP] Pickup: save failed', saveResp);
            cleanupUrl();
          }
        })
        .fail(function(xhr, status, err){
          console.error('[LSP] Pickup: save AJAX failed', status, err);
          cleanupUrl();
        });
      })
      .fail(function(xhr, status, err){
        console.error('[LSP] Pickup: broker pickup failed', status, err);
        cleanupUrl();
      });
      
    function cleanupUrl(){
      var url = new URL(window.location.href);
      url.searchParams.delete('lsp_connected');
      url.searchParams.delete('state');
      window.history.replaceState({}, document.title, url.toString());
    }
  }

  /* ---------- 2) populate catalogs/albums (robust) ---------- */
  window.lastCatalogs = window.lastCatalogs || [];

  function fillAlbumsForSelected(){
    var $cat = $('#lsp-catalog'), $alb = $('#lsp-albums'), $grid = $('#lsp-albums-grid');
    var cid = $cat.val();
    $alb.empty();
    
    var c = (window.lastCatalogs || []).find(function(x){ return x.id === cid; });
    var albums = (c && c.albums) ? c.albums : [];
    var savedAlbums = (window.LIGHTSYNCPRO && LIGHTSYNCPRO.saved && LIGHTSYNCPRO.saved.albums) || [];
    var schedules = (window.LIGHTSYNCPRO && LIGHTSYNCPRO.saved && LIGHTSYNCPRO.saved.schedules) || {};
    var destinations = (window.LIGHTSYNCPRO && LIGHTSYNCPRO.saved && LIGHTSYNCPRO.saved.destinations) || {};
    var syncStatus = (window.LIGHTSYNCPRO && LIGHTSYNCPRO.saved && LIGHTSYNCPRO.saved.sync_status) || {};
    
    
    console.log('[LSP Debug] Destinations from server:', destinations);
    console.log('[LSP Debug] Sync status from server:', syncStatus);
    
    
    // Populate hidden select for form compatibility
    albums.forEach(function(a){
      var isSelected = savedAlbums.indexOf(a.id) !== -1;
      $('<option/>').val(a.id).text(a.name || a.id).prop('selected', isSelected).appendTo($alb);
    });
    
    // Render album cards
    if (!$grid.length) return;
    
    if (!albums.length) {
      $grid.html('<div style="grid-column:1/-1;padding:32px;text-align:center;color:#6b7280;font-size:13px;background:#f8fafc;border:2px dashed #e2e8f0;border-radius:12px;">No albums found in this catalog</div>');
      return;
    }
    
    var html = '';
    albums.forEach(function(a) {
      var isSelected = savedAlbums.indexOf(a.id) !== -1;
      var albumName = a.name || a.id;
      var currentSchedule = schedules[a.id] || 'off';
      var currentDests = destinations[a.id] || ['wordpress']; // Default to WordPress
      var albumSync = syncStatus[a.id] || { count: 0, last_sync: null, wp: false };
      
      // Determine current destination value for dropdown
        if (currentDests.indexOf('wordpress') !== -1) {
      destValue = 'wp';
      }
      console.log('[LSP Debug] Album', a.id, 'currentDests:', currentDests, 'destValue:', destValue);
      var coverId = a.cover_id || '';
      var coverUrl = a.cover_url || ''; // Pre-cached URL from server
      
      // If we have a cached cover URL, use it directly
      var thumbStyle = 'aspect-ratio:4/3;display:flex;align-items:center;justify-content:center;background-size:cover;background-position:center;position:relative;';
      if (coverUrl) {
        thumbStyle += 'background-image:url(' + coverUrl + ');';
      } else {
        thumbStyle += 'background:linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%);';
      }
      
      // Sync status badge - matches Canva/Figma/Dropbox glassmorphism style
      var syncBadgeHtml = '';
      var albumUpdated = a.updated || null; // Album's last modified time from Lightroom
      var lastSynced = albumSync.last_sync || null; // Our last sync time
      var needsUpdate = albumUpdated && lastSynced && albumUpdated > lastSynced;
      var isSynced = albumSync.count > 0;
      
      if (isSynced) {
        // Build destination icons based on where album was ACTUALLY synced (not just settings)
        // Use explicit #fff stroke instead of currentColor to avoid inheritance issues
        var destIconsHtml = '';
        var wpIcon = '<svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><rect x="3" y="3" width="7" height="7" rx="1"/><rect x="14" y="3" width="7" height="7" rx="1"/><rect x="3" y="14" width="7" height="7" rx="1"/><rect x="14" y="14" width="7" height="7" rx="1"/></svg>';
        var shopifyIcon = '<svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/><line x1="3" y1="6" x2="21" y2="6"/><path d="M16 10a4 4 0 01-8 0"/></svg>';
        
        // Use actual sync status (where images were synced) not just destination settings
        if (albumSync.wp) destIconsHtml += wpIcon;
        if (albumSync.shopify) destIconsHtml += shopifyIcon;
        if (!destIconsHtml) destIconsHtml = wpIcon; // Fallback to WP if somehow neither
        
        if (needsUpdate) {
          // Orange "needs update" badge with refresh icon
          syncBadgeHtml = '<div class="lsp-sync-badge lsp-sync-update" style="position:absolute;top:6px;left:6px;background:rgba(245,158,11,0.85);backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);color:#fff;font-size:10px;font-weight:500;padding:4px 8px;border-radius:6px;display:flex;align-items:center;gap:4px;box-shadow:0 2px 8px rgba(0,0,0,0.15);border:1px solid rgba(255,255,255,0.2);z-index:2;" title="Album updated in Lightroom - sync to get latest"><svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2.5" style="margin-right:2px;"><path d="M23 4v6h-6M1 20v-6h6"/><path d="M3.51 9a9 9 0 0114.85-3.36L23 10M1 14l4.64 4.36A9 9 0 0020.49 15"/></svg>' + destIconsHtml + '</div>';
        } else {
          // Green "synced" badge with checkmark
          syncBadgeHtml = '<div class="lsp-sync-badge lsp-sync-done" style="position:absolute;top:6px;left:6px;background:rgba(16,185,129,0.85);backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);color:#fff;font-size:10px;font-weight:500;padding:4px 8px;border-radius:6px;display:flex;align-items:center;gap:4px;box-shadow:0 2px 8px rgba(0,0,0,0.15);border:1px solid rgba(255,255,255,0.2);z-index:2;" title="' + albumSync.count + ' images synced"><svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="3" style="margin-right:2px;"><path d="M20 6L9 17l-5-5"/></svg>' + destIconsHtml + '</div>';
        }
      }
      
      // Destination display
      var destHtml = '';

      
      html += '<div class="lsp-album-card' + (isSelected ? ' selected' : '') + '" data-album-id="' + escapeHtml(a.id) + '" data-cover-id="' + escapeHtml(coverId) + '" data-cover-loaded="' + (coverUrl ? '1' : '0') + '" style="position:relative;border:2px solid ' + (isSelected ? '#dc2626' : '#e2e8f0') + ';border-radius:10px;overflow:hidden;cursor:pointer;transition:all 0.15s;background:#fff;">' +
        '<div class="lsp-album-card-thumb" style="' + thumbStyle + '">' +
          (coverUrl ? '' : '<svg class="lsp-album-placeholder" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="' + (isSelected ? '#dc2626' : '#9ca3af') + '" stroke-width="1.5"><rect x="3" y="3" width="18" height="18" rx="2"/><circle cx="8.5" cy="8.5" r="1.5"/><path d="M21 15l-5-5L5 21"/></svg>') +
          syncBadgeHtml +
        '</div>' +
        '<div style="padding:10px 12px;border-top:1px solid #f1f5f9;">' +
          '<div style="font-size:12px;font-weight:600;color:#374151;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;" title="' + escapeHtml(albumName) + '">' + escapeHtml(albumName) + '</div>' +

          destHtml +
        '</div>' +
        (isSelected ? '<div class="lsp-album-checkmark" style="position:absolute;top:8px;right:8px;width:24px;height:24px;background:#dc2626;border-radius:50%;display:flex;align-items:center;justify-content:center;"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="3"><path d="M20 6L9 17l-5-5"/></svg></div>' : '') +
      '</div>';
    });
    
    $grid.html(html);
    
    // Update count badge
    $('#lsp-pick-count').text('Albums: ' + savedAlbums.length);
    
    renderAlbumSchedules($alb.val() || []);
    
    // Update Lightroom tab indicator and command bar

    
    // Lazy load only uncached album covers
    loadAlbumCovers(cid);
  }
  
  // Lazy load album cover thumbnails (only for uncached ones)
  function loadAlbumCovers(catalogId) {
    var $cards = $('.lsp-album-card[data-cover-loaded="0"]');
    
    $cards.each(function() {
      var $card = $(this);
      var albumId = $card.data('album-id');
      var coverId = $card.data('cover-id') || '';
      
      $.post(ajaxurl, {
        action: 'lsp_get_album_cover',
        _wpnonce: LIGHTSYNCPRO.nonce,
        catalog_id: catalogId,
        album_id: albumId,
        cover_id: coverId
      }, function(resp) {
        if (resp.success && resp.data && resp.data.url) {
          var $thumb = $card.find('.lsp-album-card-thumb');
          $thumb.css('background-image', 'url(' + resp.data.url + ')');
          $thumb.find('.lsp-album-placeholder').fadeOut(200);
          $card.attr('data-cover-loaded', '1');
        }
      });
    });
  }

  // Handle album card click
  $(document).on('click', '.lsp-album-card', function(e) {
    // Don't toggle if clicking the schedule dropdown or destination dropdown
    if ($(e.target).hasClass('lsp-card-destination')) return;
    
    var $card = $(this);
    var albumId = $card.data('album-id');
    var $select = $('#lsp-albums');
    var $option = $select.find('option[value="' + albumId + '"]');
    var isSelected = $card.hasClass('selected');
    var schedules = (window.LIGHTSYNCPRO && LIGHTSYNCPRO.saved && LIGHTSYNCPRO.saved.schedules) || {};
    var destinations = (window.LIGHTSYNCPRO && LIGHTSYNCPRO.saved && LIGHTSYNCPRO.saved.destinations) || {};
    var currentSchedule = schedules[albumId] || 'off';
    
    
    // Toggle selection
    if (isSelected) {
      $card.removeClass('selected')
           .css('border-color', '#e2e8f0');
      $card.find('.lsp-album-checkmark').remove();
      $card.find('.lsp-card-destination').remove();
      $card.find('.lsp-album-placeholder').attr('stroke', '#9ca3af');
      $option.prop('selected', false);
    } else {
      $card.addClass('selected')
           .css('border-color', '#dc2626');
      $card.find('.lsp-album-placeholder').attr('stroke', '#dc2626');
      // Add checkmark
      $card.append('<div class="lsp-album-checkmark" style="position:absolute;top:8px;right:8px;width:24px;height:24px;background:#dc2626;border-radius:50%;display:flex;align-items:center;justify-content:center;"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="3"><path d="M20 6L9 17l-5-5"/></svg></div>');

      $option.prop('selected', true);
    }
    
    // Trigger change event on hidden select to save
    $select.trigger('change');
  });

  // Handle schedule dropdown change on card
  // Hover effect for album cards
  $(document).on('mouseenter', '.lsp-album-card:not(.selected)', function() {
    $(this).css('border-color', '#dc2626').css('box-shadow', '0 4px 12px rgba(220,38,38,0.15)');
  });
  $(document).on('mouseleave', '.lsp-album-card:not(.selected)', function() {
    $(this).css('border-color', '#e2e8f0').css('box-shadow', 'none');
  });

  // Handle destination dropdown change on card
  $(document).on('change', '.lsp-card-destination', function(e) {
    e.stopPropagation();
    var $dropdown = $(this);
    var albumId = $dropdown.data('album-id');
    var destValue = $dropdown.val();
    
    // Convert dropdown value to destinations array
    var destinations = [];
    destinations = ['wordpress'];
    
    console.log('[LSP Debug] Saving destination for album', albumId, ':', destValue, '->', destinations);
    
    // Visual feedback
    $dropdown.css('opacity', '0.7');
    
    // Save via AJAX
    $.post(ajaxurl, {
      action: 'lsp_save_album_destinations',
      _wpnonce: LIGHTSYNCPRO.nonce,
      album_id: albumId,
      destinations: JSON.stringify(destinations)
    }, function(resp) {
      console.log('[LSP Debug] Save response:', resp);
      $dropdown.css('opacity', '1');
      if (resp.success) {
        // Update local cache
        if (!LIGHTSYNCPRO.saved.destinations) LIGHTSYNCPRO.saved.destinations = {};
        LIGHTSYNCPRO.saved.destinations[albumId] = destinations;
        console.log('[LSP Debug] Updated local cache:', LIGHTSYNCPRO.saved.destinations);
        
        // Flash green to confirm
        $dropdown.css('border-color', '#22c55e');
        setTimeout(function() {
          $dropdown.css('border-color', '#d1d5db');
        }, 600);
      } else {
        // Flash red on error
        $dropdown.css('border-color', '#ef4444');
        setTimeout(function() {
          $dropdown.css('border-color', '#d1d5db');
        }, 600);
      }
    }).fail(function(xhr, status, error) {
      console.error('[LSP Debug] Save failed:', status, error);
      $dropdown.css('opacity', '1');
      $dropdown.css('border-color', '#ef4444');
      setTimeout(function() {
        $dropdown.css('border-color', '#d1d5db');
      }, 600);
    });
  });

  async function postJsonWithRetry(url, formData, tries = 6) {
    let attempt = 0;
    while (attempt < tries) {
      attempt++;
      try {
        const res = await fetch(url, { method: "POST", body: formData, credentials: "same-origin" });

        // Retry gateway/proxy errors
        if (res.status === 502 || res.status === 504 || res.status === 520) {
          const wait = Math.min(8000, 500 * Math.pow(2, attempt - 1));
          await new Promise(r => setTimeout(r, wait));
          continue;
        }

        // Normal failure: bubble it up
        if (!res.ok) {
          const text = await res.text().catch(() => "");
          throw new Error(`HTTP ${res.status}: ${text.slice(0, 200)}`);
        }

        return await res.json();

      } catch (e) {
        // Network error / aborted request → retry
        const wait = Math.min(8000, 500 * Math.pow(2, attempt - 1));
        if (attempt >= tries) throw e;
        await new Promise(r => setTimeout(r, wait));
      }
    }
    throw new Error("Failed after retries");
  }

  function fillCatalogsSelected(payload){
    var catalogs = (payload && Array.isArray(payload.catalogs)) ? payload.catalogs : [];

    if (!catalogs.length) {
      $('#lsp-albums-grid').html('<div style="grid-column:1/-1;padding:32px;text-align:center;color:#6b7280;font-size:13px;background:#f8fafc;border:2px dashed #e2e8f0;border-radius:12px;">No catalogs found. Connect to Adobe Lightroom first.</div>');
      return;
    }

    window.lastCatalogs = catalogs;

    var $cat = $('#lsp-catalog'), $alb = $('#lsp-albums');
    $cat.empty();
    $alb.empty();

    catalogs.forEach(function(c){
      $('<option/>').val(c.id).text(c.name || c.id).appendTo($cat);
    });

    var selCat = (payload.selected && payload.selected.catalog) ||
                 (window.LIGHTSYNCPRO && LIGHTSYNCPRO.saved && LIGHTSYNCPRO.saved.catalog) || '';
    if (selCat) $cat.val(selCat);

    // Update saved albums before rendering cards
    var savedAlbums = (payload.selected && payload.selected.albums) ||
                      (window.LIGHTSYNCPRO && LIGHTSYNCPRO.saved && LIGHTSYNCPRO.saved.albums) || [];
    if (window.LIGHTSYNCPRO && LIGHTSYNCPRO.saved) {
      LIGHTSYNCPRO.saved.albums = savedAlbums;
    }

    fillAlbumsForSelected();
  }

  function refreshLists(){
    $('#lsp-log').empty();
    api('lightsyncpro_ajax_list', {})
      .done(function(resp){
        if (!resp || !resp.success){
          var data = resp && resp.data;
          var msg  = (data && (data.message || data)) || 'Failed to load catalogs';
          showError('#lsp-log', msg);
          if (data && data.needReconnect){
            $('#lsp-reconnect-banner').show();
          }
          return;
        }
        fillCatalogsSelected(resp.data || {});
      })
      .fail(function(xhr){
        showError('#lsp-log', xhr && xhr.responseText || 'Network error');
      });
  }

  /* ---------- 3) Schedules UI & save selection ---------- */
  function scheduleOptionsHtml(selected){
    var opts = [
      {v:'off',        t:'Off'},
      {v:'15m',        t:'Every 15 minutes'},
      {v:'hourly',     t:'Hourly'},
      {v:'twicedaily', t:'Twice daily'},
      {v:'daily',      t:'Daily'}
    ];
    var html = '<select class="lsp-sched-select">';
    opts.forEach(function(o){
      var sel = (o.v === selected ? ' selected' : '');
      html += '<option value="'+o.v+'"'+sel+'>'+o.t+'</option>';
    });
    html += '</select>';
    return html;
  }

  function renderAlbumSchedules(selectedAlbums){
    var $wrap = $('#lsp-album-schedules');
    if (!$wrap.length) return;
    var schedules = (LIGHTSYNCPRO.saved && LIGHTSYNCPRO.saved.schedules) || {};
    var syncTarget = LIGHTSYNCPRO.sync_target || 'wp';
    
    $wrap.empty();
    if (!selectedAlbums || !selectedAlbums.length){
      $wrap.hide();
      return;
    }
    
    // Add destination header
    var destLabels = {
      'wp': 'WordPress Media',
                };
    var destClasses = {
      'wp': 'lsp-badge-wp',
      'both': 'lsp-badge-both'
    };
    var destLabel = destLabels[syncTarget] || 'WordPress Media';
    var destClass = destClasses[syncTarget] || 'lsp-badge-wp';
    
    var header = '<div class="lsp-sync-dest-header" style="margin-bottom:12px;padding:10px 14px;background:#f8fafc;border-radius:10px;border:1px solid #e2e8f0;">' +
      '<span style="font-weight:600;color:#475569;">Syncing to:</span>' +
      '<span class="badge ' + destClass + '" id="lsp-sync-dest-badge" style="margin-left:8px;">' + destLabel + '</span>' +
    '</div>';
    $wrap.append(header);
    
    selectedAlbums.forEach(function(id){
      var label = id;
      var name = '';
      if (window.lastCatalogs){
        var catId = $('#lsp-catalog').val();
        var cat = window.lastCatalogs.find(function(c){ return c.id === catId; });
        if (cat && Array.isArray(cat.albums)){
          var a = cat.albums.find(function(x){ return x.id === id; });
          name = (a && a.name) || '';
        }
      }
      label = name || id;
      var current = schedules[id] || 'off';
      var row =
        '<div class="panel" data-album="'+id+'">' +
         '<div class="row" style="justify-content:space-between">' +
          '<div style="font-weight:800">'+ label +'</div>' +
          '<div class="lsp-sched-ctrl">'+ scheduleOptionsHtml(current).replace('<select','<select id="sched-'+id+'"') +'</div>' +
           '</div>' +
        '</div>';
      $wrap.append(row);
    });
    $wrap.show();
  }

  // Update sync destination badge when radio changes
$(document).on('change', 'input[name="lightsyncpro_settings[sync_target]"]', function(){
    var val = $(this).val();
    var labels = {
      'wp': 'WordPress Media',
                };
    var classes = {
      'wp': 'lsp-badge-wp',
      'both': 'lsp-badge-both'
    };
    
    var $badge = $('#lsp-sync-dest-badge');
    if ($badge.length) {
      $badge.text(labels[val] || 'WordPress Media')
            .removeClass('lsp-badge-wp lsp-badge-both')
            .addClass(classes[val] || 'lsp-badge-wp');
    }
    
    // Update LIGHTSYNCPRO so re-renders use correct value
    LIGHTSYNCPRO.sync_target = val;
    
    // Update helper sidebar destination display
    var colors = { 'wp': '#0073aa', 'both': '#8b5cf6' };
    var $helperDest = $('#lsp-lightroom-content aside.help [data-lr-dest]');
    if ($helperDest.length) {
      $helperDest.css('border-left-color', colors[val] || '#0073aa')
                 .css('background', (colors[val] || '#0073aa') + '15')
                 .find('[data-lr-dest-label]').text(labels[val] || 'WordPress Media').css('color', colors[val] || '#0073aa');
    }
});

  /* ---------- 4) estimate ---------- */
  function estimate(){
    var catalog = $('#lsp-catalog').val();
    var albums = $('#lsp-albums').val() || [];
    if (!catalog || albums.length===0){ showError('#lsp-log','Select at least one album to estimate import.'); return; }

    $('#lsp-estimate-out').text('Estimating…').show();
    api('lightsyncpro_ajax_estimate', { catalog: catalog, albums: albums })
      .done(function(resp){
        if (!resp || !resp.success){
          $('#lsp-estimate-out').hide();
          showError('#lsp-log', (resp && resp.data) || 'Estimate failed');
          return;
        }
        var total = (resp.data && resp.data.total) || 0;
        $('#lsp-estimate-out').text('~ ' + total + ' assets').show();
      })
      .fail(function(xhr){
        $('#lsp-estimate-out').hide();
        showError('#lsp-log', xhr && xhr.responseText || 'Network error');
      });
  }


function updatePhotosUsageUI(data) {
  if (!data) return;
  const root = document.getElementById('lsp-photos-usage');
  if (!root) return;

  const val = root.querySelector('.photos-val');
  const bar = root.querySelector('.photos-bar');

  if (val && typeof data.label !== 'undefined') val.textContent = data.label;
  if (bar && typeof data.pct !== 'undefined')  bar.style.width = data.pct + '%';
}


  /* ---------- 6) License & Usage panel ---------- */
  async function loadUsage(){
    const base = (LIGHTSYNCPRO.broker_base || '').replace(/\/+$/,'');
    const key  = (LIGHTSYNCPRO.license_key || '').trim();
    if (!base || !key) return;

    const v = await fetchJSON(base + '/wp-json/lsp-broker/v1/license/validate', {
      method:'POST',
      headers:{'Content-Type':'application/json'},
      body: JSON.stringify({ license_key: key, site_url: location.origin })
    });
    const plan = v.plan || 'free';
    const caps = v.caps || {max_sites:1,max_albums:3,max_automations:1};
    const usage = v.usage || {};

    $('#lsp-plan').text(plan.toUpperCase());

    const albumsSelected = (LIGHTSYNCPRO.saved && LIGHTSYNCPRO.saved.albums) ? LIGHTSYNCPRO.saved.albums.length : 0;

    const a = await fetchJSON(base + '/wp-json/lsp-broker/v1/activations/' + encodeURIComponent(key));
    const activations = Array.isArray(a.activations) ? a.activations : [];
    const sitesUsed = activations.length;

    // Filter to only show current site
    const currentSite = location.origin;
    const currentSiteActivation = activations.find(row => {
      const rowSite = (row.site_url || '').replace(/\/$/, '');
      const thisSite = currentSite.replace(/\/$/, '');
      return rowSite === thisSite || rowSite.includes(location.hostname);
    });

    const $tbody = $('#lsp-activations tbody').empty();
    if (!currentSiteActivation){
      $tbody.append('<tr><td colspan="4">This site is not yet activated.</td></tr>');
    } else {
      const row = currentSiteActivation;
      const tr = $('<tr>');
      tr.append('<td>'+ (row.site_url||'—') +' <span style="color:#10b981;font-size:11px;">(this site)</span></td>');
      tr.append('<td>'+ (row.adobe_email||'—') +'</td>');
      tr.append('<td>'+ (row.last_seen_at||'—') +'</td>');

      const $btn = $('<button class="button-link-delete" type="button">Deactivate This Site</button>');
      $btn.on('click', async ()=>{
        if (!confirm('Deactivate this site from your license? You can reactivate later if seats are available.')) return;
        $btn.prop('disabled', true).text('Removing…');
        try{
          await fetchJSON(base + '/wp-json/lsp-broker/v1/deactivate', {
            method:'POST',
            headers:{'Content-Type':'application/json'},
            body: JSON.stringify({ license: key, site: row.site_url })
          });
          tr.remove();
          $tbody.append('<tr><td colspan="4">Site deactivated. Refresh to reactivate.</td></tr>');
          const newUsed = Math.max(0, sitesUsed - 1);
          updateBar('#sites-bar', '#sites-val', newUsed, caps.max_sites);
        }catch(e){
          alert(e.message||'Error');
          $btn.prop('disabled', false).text('Deactivate This Site');
        }
      });

      tr.append($('<td>').append($btn));
      $tbody.append(tr);
    }

    updateBar('#sites-bar',  '#sites-val',  sitesUsed,      caps.max_sites);
    updateBar('#albums-bar', '#albums-val', albumsSelected, caps.max_albums);
    updateBar('#autos-bar',  '#autos-val',  (usage.automations_used||0), caps.max_automations);

    // ✅ Photos usage (if broker provides it)
const photosUsed = usage.photos_month_count || usage.photos_used || usage.photos_used_month || 0;
const photosMax  = caps.max_photos_month || caps.photos_month || caps.max_photos || 0;
if (document.getElementById('photos-val') && document.getElementById('photos-bar')) {
  updateBar('#photos-bar', '#photos-val', photosUsed, photosMax);
}


    api('lightsyncpro_ajax_usage_get', {}).done(function(r){
      if (r && r.success) updatePhotosUsageUI(r.data);
    });

  }

  /* ---------- 7) Upgrade (Pro/Agency) ---------- */
  async function upgrade(plan){
    try{
      const base = (LIGHTSYNCPRO.broker_base || '').replace(/\/+$/,'');
      const res = await fetch(base + '/wp-json/lsp-broker/v1/stripe/checkout', {
        method:'POST',
        headers:{'Content-Type':'application/json'},
        body: JSON.stringify({ plan: plan, email: (LIGHTSYNCPRO.admin_email || '') })
      });
      const j = await res.json();
      if (j.url) window.location = j.url; else alert(j.error || 'Checkout error');
    }catch(e){ alert(e.message || 'Network error'); }
  }

  /* ---------- 8) Diag helper ---------- */
  function runDiag(){
    $.post(LIGHTSYNCPRO.ajaxurl, {
      action: 'lightsyncpro_ajax_diag',
      _ajax_nonce: LIGHTSYNCPRO.nonce
    }, null, 'json')
    .done(function(resp){
      console.log('LSP DIAG:', resp);
      if (!resp || !resp.success) { alert('Diag error: ' + (resp && resp.data)); return; }
      alert('Diag ok. Open console for details.');
    })
    .fail(function(xhr){ alert('Diag failed: ' + (xhr && xhr.responseText)); });
  }

  /* ---------- 9) init on admin page ---------- */
  $(function(){
    pickup();

    if ($('#lsp-refresh').length){
      $('#lsp-refresh').on('click', function(e){ e.preventDefault(); refreshLists(); });
      $('#lsp-catalog').on('change', function(){ fillAlbumsForSelected(); });
      // Save handled by admin-inline.js
      $('#lsp-estimate').on('click', function(e){ e.preventDefault(); estimate(); });


      refreshLists();
    }

    if ($('#lsp-usage').length){
      $('#lsp-usage-refresh').on('click', function(e){ e.preventDefault(); loadUsage().catch(()=>{}); });
      loadUsage().catch(err => console.warn('usage load failed:', err));
    }

    $('#lsp-disconnect-adobe').on('click', function(e){
      e.preventDefault();

      if (!confirm('This will disconnect LightSync Pro from your Adobe account via the broker. You will need to reconnect to sync again. Continue?')) {
        return;
      }

      api('lightsyncpro_disconnect')
        .done(function(res){
          if (res && res.success) {
            alert((res.data && res.data.message) || 'Disconnected from Adobe.');
            window.location.reload();
          } else {
            showError('#lsp-log', (res && res.data) || 'Unknown error during disconnect.');
          }
        })
        .fail(function(xhr){
          showError('#lsp-log', xhr && xhr.responseText || 'Request failed during disconnect.');
        });
    });

    $('#lsp-upgrade-pro').on('click', function(e){ e.preventDefault(); upgrade('pro'); });
    $('#lsp-upgrade-agency').on('click', function(e){ e.preventDefault(); upgrade('agency'); });
  });

})(jQuery);

/* ---------- 10) Misc standalone bindings ---------- */
jQuery(function($){
  $('#lsp-cron-check').on('click', function(e){
    e.preventDefault();
    $('#lsp-cron-out').text('Checking…').show();
    $.post(LIGHTSYNCPRO.ajaxurl, {
      action: 'lightsyncpro_ajax_cron_check',
      _ajax_nonce: LIGHTSYNCPRO.nonce
    }, null, 'json')
    .done(function(resp){
      if(!resp || !resp.success){ $('#lsp-cron-out').hide(); alert(resp && resp.data || 'Error'); return; }
      console.table(resp.data.rows);
      const due = resp.data.rows.filter(r=>r.due_now).map(r=>r.album);
      $('#lsp-cron-out').text(due.length ? ('Due now: '+due.join(', ')) : 'Nothing due right now');
    })
    .fail(function(xhr){ $('#lsp-cron-out').hide(); alert(xhr && xhr.responseText || 'Network error'); });
  });

  $('.lsp-pills').on('click', '.lsp-pill', function(){
    const $btn = $(this), pane = $btn.data('pane');
    $btn.addClass('active').siblings().removeClass('active');
    const $wrap = $btn.closest('.lsp-card-body');
    $wrap.find('[data-pane-id]').hide();
    $wrap.find('[data-pane-id="'+pane+'"]').show();
  });
});

/* ===== LightSync Pro: Progress + Log helper ===== */
(function () {
  function $(sel) { return document.querySelector(sel); }

  var progress = $('#lsp-progress') || (function () {
    var wrap = document.createElement('div');
    wrap.id = 'lsp-progress';
    wrap.className = 'lsp-progress';
    wrap.style.display = 'none';
    wrap.innerHTML = [
      '<div class="lsp-ring" role="progressbar" aria-valuemin="0" aria-valuemax="100" aria-valuenow="0"></div>',
      '<div class="lsp-center"><div class="lsp-percent">0%</div></div>'
    ].join('');
    var host = document.querySelector('.wrap.lightsyncpro .lsp-card-body') || document.body;
    host.appendChild(wrap);
    return wrap;
  })();

  var logEl = $('#lsp-log') || (function () {
    var el = document.createElement('div');
    el.id = 'lsp-log';
    el.className = 'lsp-log';
    progress.appendChild(el);
    return el;
  })();

  if (!logEl.style.maxHeight) {
    logEl.style.maxHeight = '220px';
    logEl.style.overflow = 'auto';
    logEl.style.fontFamily = 'ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", monospace';
    logEl.style.fontSize = '12px';
    logEl.style.border = '1px solid #e5e7eb';
    logEl.style.borderRadius = '8px';
    logEl.style.padding = '10px';
    logEl.style.background = '#fff';
    logEl.style.marginTop = '10px';
  }

  function setPct(pct) {
    pct = Math.max(0, Math.min(100, Math.floor(pct)));
    var ring = progress.querySelector('.lsp-ring');
    var percent = progress.querySelector('.lsp-percent');
    if (percent) percent.textContent = pct + '%';
    if (ring) {
      ring.style.setProperty('--p', pct);
      ring.setAttribute('aria-valuenow', String(pct));
      ring.classList.remove('indeterminate');
    }
  }

  function setIndeterminate(on) {
    var ring = progress.querySelector('.lsp-ring');
    if (ring) ring.classList.toggle('indeterminate', !!on);
  }

  function show() {
    progress.style.display = '';
    logEl.style.display = 'block';
  }

  function hide() {
    setIndeterminate(false);
    progress.style.display = 'none';
  }

  function clearLog() {
    logEl.innerHTML = '';
  }

  function append(msg) {
    if (!msg && msg !== 0) return;
    var line = document.createElement('div');
    try {
      var t = '[' + new Date().toLocaleTimeString() + '] ';
      line.textContent = t + String(msg);
    } catch (e) {
      line.textContent = String(msg);
    }
    logEl.appendChild(line);
    logEl.scrollTop = logEl.scrollHeight;
  }

  window.lspProgress = {
    show: show,
    hide: hide,
    set: setPct,
    indeterminate: setIndeterminate,
    log: append,
    clear: clearLog
  };

  if (!window.lspUpdateProgress) {
    window.lspUpdateProgress = function (data) {
      if (data && typeof data.pct === 'number') setPct(data.pct);
    };
  }
})();

/* ---------- 11) Upgrade CTA (generic) ---------- */
jQuery(function($){
  $(document).on('click', '.lsp-upgrade-btn', async function(e){
    e.preventDefault();
    const plan = $(this).data('plan');
    if (!plan) { alert('No plan specified.'); return; }
    const email = (window.LIGHTSYNCPRO && LIGHTSYNCPRO.admin_email) || '';
    if (!email) {
      if(!confirm('No admin email detected. Continue without prefilled email?')) return;
    }
    // Trigger the existing button handler
    const btn = document.getElementById('lsp-upgrade-' + plan);
    if (btn) $(btn).trigger('click');
  });
});

/* ---------- 12) Light Multi-Select (albums) ---------- */
(function($){
  function LightMultiSelect(selectEl, opts){
    if (selectEl._lspMulti) return selectEl._lspMulti; // guard
    this.sel = selectEl;
    this.opts = Object.assign({
      placeholder: 'Type to filter…',
      noResultsText: 'No results',
      selectAllText: 'Select all',
      clearAllText: 'Clear',
      maxSelection: null
    }, opts || {});
    this.state = { open: false, filter: '' };
    this.build();
    this.bind();
    console.log('[LightSyncPro] MultiSelect attached to #'+(selectEl.id||'(no id)'));
  }

  LightMultiSelect.prototype.build = function(){
    const sel = this.sel;

    const existingWrap = sel.closest('.lsp-ms');
    if (existingWrap) {
      this.wrap = existingWrap;
      this.control   = existingWrap.querySelector('.ms-control');
      this.badgeWrap = existingWrap.querySelector('.ms-badges');
      this.input     = existingWrap.querySelector('.ms-input');
      this.dropdown  = existingWrap.querySelector('.ms-dropdown');
      this.list      = existingWrap.querySelector('.ms-list');
      this.btnAll    = existingWrap.querySelector('.ms-actions .button:first-child');
      this.btnClear  = existingWrap.querySelector('.ms-actions .button:last-child');
      this.headLeft  = existingWrap.querySelector('.ms-head > div:first-child');
      this.rebuildList();
      this.refreshBadges();
      this.refreshDisabledState();
      return;
    }

    sel.classList.add('hidden'); // hide original

    const wrap = document.createElement('div');
    wrap.className = 'lsp-ms';
    sel.parentNode.insertBefore(wrap, sel);
    wrap.appendChild(sel);

    const control = document.createElement('div'); control.className = 'ms-control';
    wrap.appendChild(control);

    const badgeWrap = document.createElement('div'); badgeWrap.className = 'ms-badges';
    const input = document.createElement('input'); input.type='text'; input.className='ms-input'; input.placeholder=this.opts.placeholder;
    control.appendChild(badgeWrap); control.appendChild(input);

    const dd = document.createElement('div'); dd.className='ms-dropdown hidden';
    const head = document.createElement('div'); head.className='ms-head';
    const headLeft = document.createElement('div'); headLeft.textContent='Select Albums'; headLeft.style.fontWeight='600';
    const actions = document.createElement('div'); actions.className='ms-actions';
    const btnAll = document.createElement('button'); btnAll.type='button'; btnAll.className='button'; btnAll.textContent=this.opts.selectAllText;
    const btnClear = document.createElement('button'); btnClear.type='button'; btnClear.className='button'; btnClear.textContent=this.opts.clearAllText;
    actions.appendChild(btnAll); actions.appendChild(btnClear); head.appendChild(headLeft); head.appendChild(actions);
    const list = document.createElement('div'); list.className='ms-list';
    dd.appendChild(head); dd.appendChild(list); wrap.appendChild(dd);

    this.wrap=wrap; this.control=control; this.badgeWrap=badgeWrap; this.input=input;
    this.dropdown=dd; this.list=list; this.btnAll=btnAll; this.btnClear=btnClear; this.headLeft=headLeft;

    this.rebuildList();
    this.refreshBadges();
    this.refreshDisabledState();
  };

  LightMultiSelect.prototype.refreshDisabledState = function(){
    this.wrap.classList.toggle('disabled', !!this.sel.disabled);
  };
  LightMultiSelect.prototype.open   = function(){ this.dropdown.classList.remove('hidden'); this.state.open=true; };
  LightMultiSelect.prototype.close  = function(){ this.dropdown.classList.add('hidden'); this.state.open=false; this.input.value=''; this.state.filter=''; this.filterList(); };
  LightMultiSelect.prototype.count  = function(){ return Array.from(this.sel.options).filter(o=>o.selected).length; };
  LightMultiSelect.prototype._ok    = function(opt){ return !this.opts.maxSelection || this.count()<this.opts.maxSelection || opt.selected; };
  LightMultiSelect.prototype._pass  = function(opt){ if(!this.state.filter) return true; return (opt.textContent||opt.label||'').toLowerCase().includes(this.state.filter); };
  LightMultiSelect.prototype._emit  = function(){ this.sel.dispatchEvent(new Event('change',{bubbles:true})); };

  LightMultiSelect.prototype.bind = function(){
    const self=this;

    this.control.addEventListener('mousedown', function(){ if(self.wrap.classList.contains('disabled')) return; self.open(); setTimeout(()=>self.input.focus(),0); });
    document.addEventListener('mousedown', function(e){ if(!self.wrap.contains(e.target)) self.close(); });

    this.input.addEventListener('input', function(){ self.state.filter=this.value.toLowerCase(); self.filterList(); });
    this.input.addEventListener('keydown', function(e){ if(e.key==='Escape') self.close(); });

    this.btnAll.addEventListener('click', function(){
      Array.from(self.sel.options).forEach(opt=>{
        if(!opt.disabled && self._pass(opt) && self._ok(opt)) opt.selected=true;
      });
      self._emit(); self.refreshBadges(); self.filterList(); self.input.focus();
    });
    this.btnClear.addEventListener('click', function(){
      Array.from(self.sel.options).forEach(opt=>opt.selected=false);
      self._emit(); self.refreshBadges(); self.filterList(); self.input.focus();
    });

    const mo = new MutationObserver(()=>{ self.rebuildList(); self.refreshBadges(); });
    mo.observe(this.sel, { childList:true });

    this.sel.addEventListener('lsp:optionsUpdated', ()=>{ self.rebuildList(); self.refreshBadges(); });

    this.sel._lspMulti=this;
  };

  LightMultiSelect.prototype.rebuildList = function(){
    const self=this, opts=Array.from(this.sel.options);
    this.list.innerHTML='';
    if(!opts.length){
      const empty=document.createElement('div'); empty.className='ms-empty'; empty.textContent='No albums…';
      this.list.appendChild(empty);
      return;
    }

    opts.forEach(function(opt){
      const item=document.createElement('div'); item.className='ms-item'; item.tabIndex=0;
      const cb=document.createElement('input'); cb.type='checkbox'; cb.checked=!!opt.selected;
      const label=document.createElement('div'); label.textContent=opt.textContent||opt.label||opt.value;
      item.appendChild(cb); item.appendChild(label); self.list.appendChild(item);

      const toggle=function(){
        if(opt.disabled) return;
        if(!opt.selected && !self._ok(opt)){ item.animate([{transform:'scale(1)'},{transform:'scale(.98)'},{transform:'scale(1)'}],{duration:160}); return; }
        opt.selected=!opt.selected; cb.checked=opt.selected; self._emit(); self.refreshBadges(); self._updateHeadCount();
      };
      item.addEventListener('click', toggle);
      item.addEventListener('keydown', function(e){ if(e.key===' '||e.key==='Enter'){ e.preventDefault(); toggle(); }});
    });

    this.filterList();
    this._updateHeadCount();
  };

  LightMultiSelect.prototype._updateHeadCount = function(){
    if(this.headLeft){ const n=this.count(); this.headLeft.textContent = n?`Select Albums (${n} selected)`:'Select Albums'; }
  };

  LightMultiSelect.prototype.refreshBadges = function(){
    const badges = [];
    Array.from(this.sel.options).forEach(function(o){
      if (o.selected) badges.push(o.textContent || o.label || o.value);
    });
    this.badgeWrap.innerHTML = '';
    badges.forEach(function(t){
      const b = document.createElement('span');
      b.className = 'ms-badge';
      b.textContent = t;
      this.badgeWrap.appendChild(b);
    }, this);
    this._updateHeadCount();
  };

  LightMultiSelect.prototype.filterList = function(){
    const q=this.state.filter;
    let visible=0;
    Array.from(this.list.children).forEach(c=>{
      if(!c.classList.contains('ms-item')) return;
      const show = !q || c.textContent.toLowerCase().includes(q);
      c.style.display = show ? '' : 'none';
      if(show) visible++;
    });
    if(!visible){
      if(!this._emptyNode){ const e=document.createElement('div'); e.className='ms-empty'; e.textContent=this.opts.noResultsText; this._emptyNode=e; }
      if(!this._emptyNode.isConnected) this.list.appendChild(this._emptyNode);
    } else if(this._emptyNode && this._emptyNode.isConnected){ this._emptyNode.remove(); }

    const options=Array.from(this.sel.options);
    Array.from(this.list.querySelectorAll('.ms-item')).forEach((item,i)=>{
      const cb=item.querySelector('input[type="checkbox"]'); const opt=options[i];
      if(cb && opt) cb.checked=!!opt.selected;
    });
  };

  $(function(){
    console.log('[LightSyncPro] Admin JS loaded');

    const plan = (window.LIGHTSYNCPRO && (LIGHTSYNCPRO.plan || 'free')) || 'free';
    const isFree = (plan === 'free');
    const maxSel = isFree ? 1 : null;

    function initWhenReady(){
      const targets = document.querySelectorAll('select[multiple][data-enhance="multiselect"]');
      if(!targets.length){ setTimeout(initWhenReady, 150); return; }
      targets.forEach(function(sel){
        if(sel._lspMulti) return;
        new LightMultiSelect(sel, { maxSelection: maxSel });
      });
    }
    initWhenReady();
  });
})(jQuery);

/* ---------- 13) Side nav highlight on scroll ---------- */
(function($){
  const $links = $('.lsp-side .lsp-nav a');
  if (!$links.length) return;

  const $targets = $links.map(function(){
    const id = $(this).attr('href');
    try { return $(id); } catch(e){ return null; }
  }).get().filter(Boolean);

  function onScroll(){
    const fromTop = window.scrollY + 120;
    let activeId = null;

    for (const $t of $targets){
      if ($t.length && $t.offset().top <= fromTop) activeId = $t.attr('id');
    }
    if (!activeId) return;
    $links.removeClass('active').filter('[href="#'+activeId+'"]').addClass('active');
  }

  $(window).on('scroll', onScroll);
  onScroll();
})(jQuery);

/* ---------- 14) Source Tabs (Lightroom/Canva) ---------- */
(function($){
  'use strict';

  // Reset all source loaded flags when page is restored from bfcache
  // (e.g., after window.location.reload() from sync completion)
  // bfcache preserves JS state but gives fresh DOM — loaded flags must reset
  window.addEventListener('pageshow', function(e) {
    if (e.persisted) {
      console.log('[LightSync] Page restored from bfcache, resetting all source flags');
      window.lspCanvaLoaded = false;
      window.lspFigmaLoaded = false;
      window.lspDropboxLoaded = false;
      // AI flags reset in lspAI IIFE's own pageshow handler

      // Re-init whichever tab is active
      var $active = $('.lsp-source-content.active');
      if ($active.length) {
        var id = $active.attr('id') || '';
        if (id.indexOf('canva') !== -1) loadCanvaDesigns();
        else if (id.indexOf('figma') !== -1) loadFigmaData();
        else if (id.indexOf('dropbox') !== -1) loadDropboxFolder('');
      }
    }
  });

  // Tab switching
  $('.lsp-source-tab').on('click', function(){
    if ($(this).prop('disabled')) return;
    
    const source = $(this).data('source');
    const aiTab = $(this).data('ai-tab');
    
    // Handle AI tab switching (separate from source tabs)
    if (aiTab) {
      // Update active AI tab
      $('[data-ai-tab]').removeClass('active');
      $(this).addClass('active');
      
      // Show/hide AI content
      $('.lsp-ai-tab-content').hide();
      $('#lsp-ai-' + aiTab + '-content').show();
      return;
    }
    
    if (!source) return;

    // Update active tab
    $('.lsp-source-tab').removeClass('active');
    $(this).addClass('active');

    // Show/hide content
    $('.lsp-source-content').removeClass('active');
    $('#lsp-' + source + '-content').addClass('active');

    // Update URL without reload
    const url = new URL(window.location.href);
    url.searchParams.set('source', source);
    window.history.replaceState({}, '', url);

    // Update nav visibility based on source
    updateNavForSource(source);

    // Update command bar for source
    updateCommandBarForSource(source);

    // Load Canva designs if switching to Canva
    if (source === 'canva' && !window.lspCanvaLoaded) {
      loadCanvaDesigns();
    }

    // Load Figma data if switching to Figma
    if (source === 'figma' && !window.lspFigmaLoaded) {
      loadFigmaData();
    }

    // Load Dropbox data if switching to Dropbox
    if (source === 'dropbox' && !window.lspDropboxLoaded) {
      loadDropboxFolder('');
    }

    // Load AI data if switching to AI
    if (source === 'ai') {
      console.log('[LightSync] AI tab clicked, lspAI defined=' + (typeof lspAI !== 'undefined'));
      if (typeof lspAI !== 'undefined') lspAI.init();
    }
  });

  // Update nav sidebar based on active source
  function updateNavForSource(source) {
    const $syncNav = $('a[href="#lsp-sync"]').parent();
    const $namingNav = $('a[href="#lsp-naming"]').parent();
    const $syncSection = $('#lsp-sync');
    const $namingSection = $('#lsp-naming');
    const $pickTitle = $('#lsp-pick-title');
    const $pickCount = $('#lsp-pick-count');
    const $navPick = $('#lsp-nav-pick a');
    const $kpiLabel = $('#lsp-kpi-albums-label');
    const $kpiValue = $('#lsp-kpi-albums-value');

    if (source === 'canva') {
      // Hide Lightroom-specific nav items and sections for Canva
      $syncNav.hide();
      $namingNav.hide();
      $syncSection.hide();
      $namingSection.hide();
      
      // Update nav text and href
      if ($navPick.length) {
        $navPick.text('Canva Designs');
        $navPick.attr('href', '#lsp-canva-pick');
      }
      // Update mobile nav
      $('#lsp-mobile-nav-pick').attr('href', '#lsp-canva-pick');
      $('.lsp-mobile-nav-pick-label').text('Designs');
      
      // Update header
      if ($pickTitle.length) $pickTitle.text('Canva Designs');
      
      // Update count - will be set properly when designs load
      const canvaCount = window.lspCanvaDesigns ? window.lspCanvaDesigns.length : 0;
      if ($pickCount.length) $pickCount.text('Designs: ' + canvaCount);
      
      // Update KPI
      if ($kpiLabel.length) $kpiLabel.text('Designs');
      if ($kpiValue.length) $kpiValue.text(canvaCount);
    } else if (source === 'figma') {
      // Hide Lightroom-specific nav items and sections for Figma
      $syncNav.hide();
      $namingNav.hide();
      $syncSection.hide();
      $namingSection.hide();
      
      // Update nav text and href
      if ($navPick.length) {
        $navPick.text('Figma Frames');
        $navPick.attr('href', '#lsp-figma-pick');
      }
      // Update mobile nav
      $('#lsp-mobile-nav-pick').attr('href', '#lsp-figma-pick');
      $('.lsp-mobile-nav-pick-label').text('Frames');
      
      // Update header
      if ($pickTitle.length) $pickTitle.text('Figma Frames');
      
      // Update count
      const figmaCount = window.lspFigmaAllFrames ? window.lspFigmaAllFrames.length : 0;
      if ($pickCount.length) $pickCount.text('Frames: ' + figmaCount);
      
      // Update KPI
      if ($kpiLabel.length) $kpiLabel.text('Frames');
      if ($kpiValue.length) $kpiValue.text(figmaCount);
    } else if (source === 'dropbox') {
      // Hide Lightroom-specific nav items and sections for Dropbox
      $syncNav.hide();
      $namingNav.hide();
      $syncSection.hide();
      $namingSection.hide();
      
      // Update nav text and href
      if ($navPick.length) {
        $navPick.text('Dropbox Folders');
        $navPick.attr('href', '#lsp-dropbox-pick');
      }
      // Update mobile nav
      $('#lsp-mobile-nav-pick').attr('href', '#lsp-dropbox-pick');
      $('.lsp-mobile-nav-pick-label').text('Folders');
      
      // Update header
      if ($pickTitle.length) $pickTitle.text('Dropbox Folders');
      
      // Update count
      const dropboxCount = window.lspDropboxFolders ? window.lspDropboxFolders.length : 0;
      if ($pickCount.length) $pickCount.text('Folders: ' + dropboxCount);
      
      // Update KPI
      if ($kpiLabel.length) $kpiLabel.text('Folders');
      if ($kpiValue.length) $kpiValue.text(dropboxCount);
    } else if (source === 'ai') {
      // Hide Lightroom-specific nav items and sections for AI
      $syncNav.hide();
      $namingNav.hide();
      $syncSection.hide();
      $namingSection.hide();
      
      // Update nav text and href
      if ($navPick.length) {
        $navPick.text('AI Generate');
        $navPick.attr('href', '#lsp-ai-content');
      }
      // Update mobile nav
      $('#lsp-mobile-nav-pick').attr('href', '#lsp-ai-content');
      $('.lsp-mobile-nav-pick-label').text('AI');
      
      // Update header
      if ($pickTitle.length) $pickTitle.text('AI Generate');
      
      // Update KPI
      if ($kpiLabel.length) $kpiLabel.text('AI Images');
      if ($kpiValue.length) $kpiValue.text(window.lspAiCount || 0);
    } else {
      // Lightroom (default) — restore nav
      $syncNav.show();
      $namingNav.show();
      $syncSection.show();
      $namingSection.show();
      if ($navPick.length) {
        $navPick.text('Pick Albums');
        $navPick.attr('href', '#lsp-pick');
      }
      $('#lsp-mobile-nav-pick').attr('href', '#lsp-pick');
      $('.lsp-mobile-nav-pick-label').text('Albums');
      if ($pickTitle.length) $pickTitle.text('Pick Albums');
      const albumCount = $pickCount.data('album-count') || 0;
      if ($pickCount.length) $pickCount.text('Albums: ' + albumCount);
      if ($kpiLabel.length) $kpiLabel.text('Albums');
      if ($kpiValue.length) $kpiValue.text(albumCount);
    }
  }

  // Update command bar based on active source
  function updateCommandBarForSource(source) {
    const $estimateBtn = $('#lsp-estimate');
    let $canvaSelectionEl = $('#lsp-canva-commandbar-selection');
    let $figmaSelectionEl = $('#lsp-figma-commandbar-selection');
    
    if (source === 'canva') {
      // Hide estimate button
      $estimateBtn.hide();
      
      // Hide Figma, Dropbox, AI selections
      if ($figmaSelectionEl.length) $figmaSelectionEl.hide();
      $('#lsp-figma-sep').hide();
      let $dropboxSelectionEl = $('#lsp-dropbox-commandbar-selection');
      if ($dropboxSelectionEl.length) $dropboxSelectionEl.hide();
      $('#lsp-dropbox-sep').hide();
      $('#lsp-ai-commandbar-selection, #lsp-ai-sep').hide();
      
      // Show Canva selection count in command bar
      if (!$canvaSelectionEl.length) {
        // Create the selection indicator if it doesn't exist
        $estimateBtn.after('<span class="separator" id="lsp-canva-sep" style="display:none;">·</span><span id="lsp-canva-commandbar-selection" class="pill" style="display:none;"><span class="lsp-selection-text">0 designs selected</span></span>');
        $canvaSelectionEl = $('#lsp-canva-commandbar-selection');
      }
      $canvaSelectionEl.show();
      $('#lsp-canva-sep').show();
      updateCanvaCommandBarSelection();
    } else if (source === 'figma') {
      // Hide estimate button and Canva/Dropbox/AI selection
      $estimateBtn.hide();
      if ($canvaSelectionEl.length) $canvaSelectionEl.hide();
      $('#lsp-canva-sep').hide();
      let $dropboxSelectionEl = $('#lsp-dropbox-commandbar-selection');
      if ($dropboxSelectionEl.length) $dropboxSelectionEl.hide();
      $('#lsp-dropbox-sep').hide();
      $('#lsp-ai-commandbar-selection, #lsp-ai-sep').hide();
      
      // Show Figma selection count in command bar
      if (!$figmaSelectionEl.length) {
        $estimateBtn.after('<span class="separator" id="lsp-figma-sep" style="display:none;">·</span><span id="lsp-figma-commandbar-selection" class="pill" style="display:none;"><span class="lsp-selection-text">0 frames selected</span></span>');
        $figmaSelectionEl = $('#lsp-figma-commandbar-selection');
      }
      $figmaSelectionEl.show();
      $('#lsp-figma-sep').show();
      updateFigmaCommandBarSelection();
    } else if (source === 'dropbox') {
      // Hide estimate button and other source selections for Dropbox
      $estimateBtn.hide();
      if ($canvaSelectionEl.length) $canvaSelectionEl.hide();
      $('#lsp-canva-sep').hide();
      if ($figmaSelectionEl.length) $figmaSelectionEl.hide();
      $('#lsp-figma-sep').hide();
      $('#lsp-ai-commandbar-selection, #lsp-ai-sep').hide();
      $('#lsp-shutterstock-commandbar-selection, #lsp-shutterstock-sep').hide();
      
      // Show Dropbox selection count in command bar
      let $dropboxSelectionEl = $('#lsp-dropbox-commandbar-selection');
      if (!$dropboxSelectionEl.length) {
        $estimateBtn.after('<span class="separator" id="lsp-dropbox-sep" style="display:none;">·</span><span id="lsp-dropbox-commandbar-selection" class="pill" style="display:none;"><span class="lsp-selection-text">0 files selected</span></span>');
        $dropboxSelectionEl = $('#lsp-dropbox-commandbar-selection');
      }
      $dropboxSelectionEl.show();
      $('#lsp-dropbox-sep').show();
      updateDropboxCommandBarSelection();
    } else if (source === 'shutterstock') {
      // Hide estimate button and other source selections for Shutterstock
      $estimateBtn.hide();
      if ($canvaSelectionEl.length) $canvaSelectionEl.hide();
      $('#lsp-canva-sep').hide();
      if ($figmaSelectionEl.length) $figmaSelectionEl.hide();
      $('#lsp-figma-sep').hide();
      $('#lsp-dropbox-commandbar-selection, #lsp-dropbox-sep').hide();
      $('#lsp-ai-commandbar-selection, #lsp-ai-sep').hide();
      
      // Show Shutterstock selection count in command bar
      let $shutterstockSelectionEl = $('#lsp-shutterstock-commandbar-selection');
      if (!$shutterstockSelectionEl.length) {
        $estimateBtn.after('<span class="separator" id="lsp-shutterstock-sep" style="display:none;">·</span><span id="lsp-shutterstock-commandbar-selection" class="pill" style="display:none;"><span class="lsp-selection-text">0 images selected</span></span>');
        $shutterstockSelectionEl = $('#lsp-shutterstock-commandbar-selection');
      }
      $shutterstockSelectionEl.show();
      $('#lsp-shutterstock-sep').show();
      if (typeof updateShutterstockCommandBarSelection === 'function') {
        updateShutterstockCommandBarSelection();
      }
    } else if (source === 'ai') {
      // Hide all other selection counts and estimate for AI
      $estimateBtn.hide();
      if ($canvaSelectionEl.length) $canvaSelectionEl.hide();
      $('#lsp-canva-sep').hide();
      if ($figmaSelectionEl.length) $figmaSelectionEl.hide();
      $('#lsp-figma-sep').hide();
      let $dropboxSelectionEl2 = $('#lsp-dropbox-commandbar-selection');
      if ($dropboxSelectionEl2.length) $dropboxSelectionEl2.hide();
      $('#lsp-dropbox-sep').hide();
      $('#lsp-shutterstock-commandbar-selection, #lsp-shutterstock-sep').hide();

      // Show AI selection count in command bar
      let $aiSelectionEl = $('#lsp-ai-commandbar-selection');
      if (!$aiSelectionEl.length) {
        $estimateBtn.after('<span class="separator" id="lsp-ai-sep" style="display:none;">·</span><span id="lsp-ai-commandbar-selection" class="pill" style="display:none;"><span class="lsp-selection-text">0 images selected</span></span>');
      }
      updateAiSelectionUI();
    } else {
      // Show estimate for Lightroom
      $estimateBtn.show();
      
      // Hide Canva, Figma, Dropbox, Shutterstock, AI selection counts
      if ($canvaSelectionEl.length) $canvaSelectionEl.hide();
      $('#lsp-canva-sep').hide();
      if ($figmaSelectionEl.length) $figmaSelectionEl.hide();
      $('#lsp-figma-sep').hide();
      let $dropboxSelectionEl = $('#lsp-dropbox-commandbar-selection');
      if ($dropboxSelectionEl.length) $dropboxSelectionEl.hide();
      $('#lsp-dropbox-sep').hide();
      $('#lsp-shutterstock-commandbar-selection, #lsp-shutterstock-sep').hide();
      $('#lsp-ai-commandbar-selection, #lsp-ai-sep').hide();
    }
  }

  // Update the command bar selection count for Canva
  function updateCanvaCommandBarSelection() {
    const count = window.lspCanvaSelected ? window.lspCanvaSelected.size : 0;
    const $el = $('#lsp-canva-commandbar-selection .lsp-selection-text');
    if ($el.length) {
      $el.text(count + ' design' + (count !== 1 ? 's' : '') + ' selected');
    }
    // Also update the selection count in the panel
    const $panelCount = $('#lsp-canva-selection-count');
    if ($panelCount.length) {
      if (count > 0) {
        $panelCount.text(count + ' design' + (count !== 1 ? 's' : '') + ' selected');
      } else {
        $panelCount.text('');
      }
    }
  }

  // Update the command bar selection count for Figma
  function updateFigmaCommandBarSelection() {
    const count = window.lspFigmaSelectedFrames ? window.lspFigmaSelectedFrames.length : 0;
    const $el = $('#lsp-figma-commandbar-selection .lsp-selection-text');
    if ($el.length) {
      $el.text(count + ' frame' + (count !== 1 ? 's' : '') + ' selected');
    }
    // Also update the selection count in the panel
    const $panelCount = $('#lsp-figma-selection-count');
    if ($panelCount.length) {
      if (count > 0) {
        $panelCount.text(count + ' frame' + (count !== 1 ? 's' : '') + ' selected');
      } else {
        $panelCount.text('');
      }
    }
    // Update the main frames count badge
    const total = window.lspFigmaAllFrames ? window.lspFigmaAllFrames.length : 0;
    $('#lsp-figma-count').text('Frames: ' + total);
  }

  // Update the command bar selection count for Dropbox
  function updateDropboxCommandBarSelection() {
    const count = window.lspDropboxSelectedFiles ? window.lspDropboxSelectedFiles.length : 0;
    const $el = $('#lsp-dropbox-commandbar-selection .lsp-selection-text');
    if ($el.length) {
      $el.text(count + ' file' + (count !== 1 ? 's' : '') + ' selected');
    }
    // Also update the selection count in the panel
    const $panelCount = $('#lsp-dropbox-selection-count');
    if ($panelCount.length) {
      if (count > 0) {
        $panelCount.text(count + ' file' + (count !== 1 ? 's' : '') + ' selected');
      } else {
        $panelCount.text('');
      }
    }
    // Update the main files count badge
    const total = window.lspDropboxAllFiles ? window.lspDropboxAllFiles.length : 0;
    $('#lsp-dropbox-count').text('Files: ' + total);
  }

  // Expose to window
  window.updateCanvaCommandBarSelection = updateCanvaCommandBarSelection;
  window.updateFigmaCommandBarSelection = updateFigmaCommandBarSelection;
  window.updateDropboxCommandBarSelection = updateDropboxCommandBarSelection;

  // Initialize nav state on page load
  $(function(){
    // Store original album count
    const $pickCount = $('#lsp-pick-count');
    if ($pickCount.length) {
      const countText = $pickCount.text();
      const match = countText.match(/(\d+)/);
      if (match) {
        $pickCount.data('album-count', parseInt(match[1], 10));
      }
    }
    
    const activeSource = $('.lsp-source-tab.active').data('source') || 'lightroom';
    updateNavForSource(activeSource);
    updateCommandBarForSource(activeSource);
  });

  // Canva state
  window.lspCanvaLoaded = false;
  window.lspCanvaDesigns = [];
  window.lspCanvaSelected = new Set();
  window.lspCanvaSyncedIds = new Set(); // Track already synced designs (for backward compat)
  window.lspCanvaSyncedTimes = {}; // Track sync timestamps: {design_id: timestamp}

  // Load Canva designs
  function loadCanvaDesigns(){
    const $grid = $('#lsp-canva-designs');
    if (!$grid.length) return;

    $grid.html('<div style="text-align:center;padding:40px;color:#64748b;"><div class="spinner is-active" style="float:none;margin:0 auto 12px;"></div><p>Loading your Canva designs...</p></div>');

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_canva_get_designs',
        _wpnonce: LIGHTSYNCPRO.nonce
      },
      success: function(resp){
        if (resp.success && resp.data && resp.data.items) {
          window.lspCanvaDesigns = resp.data.items;
          window.lspCanvaLoaded = true;
          
          // Load synced status
          loadSyncedStatus(function(){
            renderCanvaDesigns(filterAndSortDesigns());
            updateCanvaCount(resp.data.items.length);
          });
        } else {
          const err = (resp.data && resp.data.error) ? resp.data.error : 'Failed to load designs';
          $grid.html('<div style="text-align:center;padding:40px;color:#ef4444;"><p>⚠️ ' + err + '</p><button class="btn ghost" id="lsp-canva-retry">Retry</button></div>');
        }
      },
      error: function(xhr, status, error){
        $grid.html('<div style="text-align:center;padding:40px;color:#ef4444;"><p>⚠️ Connection error: ' + error + '</p><button class="btn ghost" id="lsp-canva-retry">Retry</button></div>');
      }
    });
  }

  // Load which designs are already synced
  function loadSyncedStatus(callback){
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_canva_get_synced',
        _wpnonce: LIGHTSYNCPRO.nonce
      },
      success: function(resp){
        if (resp.success && resp.data) {
          // New format: {design_id: {time: timestamp, dest: 'wp'}}
          if (typeof resp.data === 'object' && !Array.isArray(resp.data)) {
            window.lspCanvaSyncedData = resp.data; // Full data with dest
            window.lspCanvaSyncedTimes = {};
            Object.keys(resp.data).forEach(function(id) {
              // Handle both old format (number) and new format (object)
              if (typeof resp.data[id] === 'object') {
                window.lspCanvaSyncedTimes[id] = resp.data[id].time;
              } else {
                window.lspCanvaSyncedTimes[id] = resp.data[id];
              }
            });
            window.lspCanvaSyncedIds = new Set(Object.keys(resp.data));
          } else {
            // Old format: array of IDs
            window.lspCanvaSyncedIds = new Set(resp.data);
          }
        }
        if (callback) callback();
      },
      error: function(){
        if (callback) callback();
      }
    });
  }

  // Filter and sort designs
  function filterAndSortDesigns(){
    let designs = [...window.lspCanvaDesigns];
    
    // Search filter
    const search = ($('#lsp-canva-search').val() || '').toLowerCase().trim();
    if (search) {
      designs = designs.filter(function(d){
        return (d.title || '').toLowerCase().includes(search);
      });
    }

    // Sort
    const sort = $('#lsp-canva-sort').val() || 'newest';
    designs.sort(function(a, b){
      if (sort === 'name') {
        return (a.title || '').localeCompare(b.title || '');
      } else if (sort === 'oldest') {
        return (a.created_at || 0) - (b.created_at || 0);
      } else {
        return (b.created_at || 0) - (a.created_at || 0);
      }
    });

    return designs;
  }

  // Expose to window for admin-sync.js
  window.filterAndSortDesigns = filterAndSortDesigns;

  // Render Canva designs grid
  function renderCanvaDesigns(designs){
    const $grid = $('#lsp-canva-designs');
    
    if (!designs || !designs.length) {
      const search = $('#lsp-canva-search').val();
      if (search) {
        $grid.html('<div style="text-align:center;padding:40px;color:#64748b;"><p>No designs match "' + escapeHtml(search) + '"</p></div>');
      } else {
        $grid.html('<div style="text-align:center;padding:40px;color:#64748b;"><p>No designs found in your Canva account.</p><p style="font-size:13px;">Create some designs in Canva, then refresh.</p></div>');
      }
      return;
    }

    let html = '';
    designs.forEach(function(design){
      const thumb = design.thumbnail && design.thumbnail.url ? design.thumbnail.url : '';
      const name = design.title || 'Untitled';
      const created = design.created_at ? new Date(design.created_at * 1000).toLocaleDateString() : '';
      const selected = window.lspCanvaSelected.has(design.id) ? 'selected' : '';
      const synced = window.lspCanvaSyncedIds.has(design.id);
      
      // Check if design was updated since last sync
      let hasUpdate = false;
      if (synced && design.updated_at && window.lspCanvaSyncedTimes[design.id]) {
        const designUpdated = design.updated_at; // Unix timestamp from Canva
        const syncedAt = window.lspCanvaSyncedTimes[design.id];
        hasUpdate = designUpdated > syncedAt;
      }

      html += '<div class="lsp-canva-design-card ' + selected + (synced ? ' synced' : '') + (hasUpdate ? ' has-update' : '') + '" data-id="' + design.id + '">';
      
      // Build destination icons (same SVGs as destination selector)
      var destIconsHtml = '';
      if (synced && window.lspCanvaSyncedData && window.lspCanvaSyncedData[design.id]) {
        var dest = window.lspCanvaSyncedData[design.id].dest || 'wp';
        destIconsHtml = '<span style="display:inline-flex;gap:3px;margin-left:4px;vertical-align:middle;">';
        if (dest === 'wp' || dest === 'both') {
          destIconsHtml += '<span title="Synced to WordPress" style="display:flex;"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><rect x="3" y="3" width="7" height="7" rx="1"/><rect x="14" y="3" width="7" height="7" rx="1"/><rect x="3" y="14" width="7" height="7" rx="1"/><rect x="14" y="14" width="7" height="7" rx="1"/></svg></span>';
        }
        if (dest === 'shopify' || dest === 'both') {
          destIconsHtml += '<span title="Synced to Shopify" style="display:flex;"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/><line x1="3" y1="6" x2="21" y2="6"/><path d="M16 10a4 4 0 01-8 0"/></svg></span>';
        }
        destIconsHtml += '</span>';
      }
      
      // Show appropriate badge with destination icons (upper left) - Glassmorphism style
      if (hasUpdate) {
        html += '<div class="lsp-sync-badge lsp-sync-update" style="position:absolute;top:6px;left:6px;background:rgba(245,158,11,0.85);backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);color:#fff;font-size:10px;font-weight:500;padding:4px 8px;border-radius:6px;display:flex;align-items:center;gap:2px;box-shadow:0 2px 8px rgba(0,0,0,0.15);border:1px solid rgba(255,255,255,0.2);z-index:2;" title="Updated in Canva - click to resync"><svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" style="margin-right:2px;"><path d="M23 4v6h-6M1 20v-6h6"/><path d="M3.51 9a9 9 0 0114.85-3.36L23 10M1 14l4.64 4.36A9 9 0 0020.49 15"/></svg>' + destIconsHtml + '</div>';
      } else if (synced) {
        html += '<div class="lsp-sync-badge lsp-sync-done" style="position:absolute;top:6px;left:6px;background:rgba(16,185,129,0.85);backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);color:#fff;font-size:10px;font-weight:500;padding:4px 8px;border-radius:6px;display:flex;align-items:center;gap:2px;box-shadow:0 2px 8px rgba(0,0,0,0.15);border:1px solid rgba(255,255,255,0.2);z-index:2;" title="Synced"><svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" style="margin-right:2px;"><path d="M20 6L9 17l-5-5"/></svg>' + destIconsHtml + '</div>';
      }
      
      if (thumb) {
        html += '<img class="lsp-canva-design-thumb" src="' + thumb + '" alt="' + escapeHtml(name) + '" loading="lazy">';
      } else {
        html += '<div class="lsp-canva-design-thumb" style="display:flex;align-items:center;justify-content:center;font-size:32px;">🎨</div>';
      }
      html += '<div class="lsp-canva-design-info">';
      html += '<div class="lsp-canva-design-name" title="' + escapeHtml(name) + '">' + escapeHtml(name) + '</div>';
      if (created) {
        html += '<div class="lsp-canva-design-meta">' + created + (hasUpdate ? ' <span style="color:#f59e0b;font-weight:500;">• Updated</span>' : '') + '</div>';
      }
      html += '</div></div>';
    });

    $grid.html(html);

    // Click to select/deselect
    $grid.find('.lsp-canva-design-card').on('click', function(){
      const id = $(this).data('id');
      if (window.lspCanvaSelected.has(id)) {
        window.lspCanvaSelected.delete(id);
        $(this).removeClass('selected');
      } else {
        window.lspCanvaSelected.add(id);
        $(this).addClass('selected');
      }
      updateSyncButton();
    });
  }

  // Expose to window for admin-sync.js
  window.renderCanvaDesigns = renderCanvaDesigns;

  // Search/filter handlers
  $('#lsp-canva-search').on('input', debounce(function(){
    renderCanvaDesigns(filterAndSortDesigns());
  }, 300));

  $('#lsp-canva-sort').on('change', function(){
    renderCanvaDesigns(filterAndSortDesigns());
  });

  function debounce(fn, delay){
    let timer;
    return function(){
      const args = arguments;
      clearTimeout(timer);
      timer = setTimeout(function(){ fn.apply(this, args); }, delay);
    };
  }

  function updateCanvaCount(count){
    $('#lsp-canva-count').text('Designs: ' + count);
    // Also update header count when on Canva tab
    const activeSource = $('.lsp-source-tab.active').data('source');
    if (activeSource === 'canva') {
      $('#lsp-pick-count').text('Designs: ' + count);
      // Update KPI as well
      $('#lsp-kpi-albums-label').text('Designs');
      $('#lsp-kpi-albums-value').text(count);
    }
  }

  function updateSyncButton(){
    // Update selection count display in panel
    const count = window.lspCanvaSelected.size;
    const $countEl = $('#lsp-canva-selection-count');
    if ($countEl.length) {
      if (count > 0) {
        $countEl.text(count + ' design' + (count > 1 ? 's' : '') + ' selected');
      } else {
        $countEl.text('');
      }
    }
    
    // Also update command bar selection count
    if (typeof window.updateCanvaCommandBarSelection === 'function') {
      window.updateCanvaCommandBarSelection();
    }
  }

  // Expose to window
  window.updateSyncButton = updateSyncButton;



  function escapeHtml(str){
    if (!str) return '';
    return str.replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/"/g,'&quot;');
  }

  // Refresh designs
  $(document).on('click', '#lsp-canva-refresh, #lsp-canva-retry', function(){
    window.lspCanvaLoaded = false;
    window.lspCanvaSelected.clear();
    updateSyncButton();
    loadCanvaDesigns();
  });

  // Disconnect Canva
  $('#lsp-canva-disconnect').on('click', function(){
    if (!confirm('Disconnect from Canva? You can reconnect anytime.')) return;

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_canva_disconnect',
        _wpnonce: LIGHTSYNCPRO.nonce
      },
      success: function(resp){
        if (resp.success) {
          location.reload();
        } else {
          alert('Failed to disconnect: ' + (resp.data || 'Unknown error'));
        }
      }
    });
  });

  // Save Canva sync target preference
  $('input[name="lsp_canva_sync_target"]').on('change', function(){
    const target = $(this).val();
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_canva_save_target',
        _wpnonce: LIGHTSYNCPRO.nonce,
        target: target
      }
    });
  });

  // Auto-load Canva if starting on Canva tab
  $(function(){
    if ($('#lsp-canva-content.active').length && !window.lspCanvaLoaded) {
      loadCanvaDesigns();
    }
    // Auto-load AI if starting on AI tab
    if ($('#lsp-ai-content.active').length) {
      console.log('[LightSync] AI tab active on page load, initializing...');
      if (typeof lspAI !== 'undefined') lspAI.init();
    }

    // Failsafe: if AI tab is visible but models never loaded, retry after 5s
    setTimeout(function() {
      if ($('#lsp-ai-content.active').length && typeof lspAI !== 'undefined') {
        if (!window.lspAiState || !window.lspAiState.modelsLoaded) {
          console.log('[LightSync] AI failsafe: models not loaded after 5s, retrying...');
          lspAI.init();
        }
      }
    }, 5000);
  });

  /* ==================== FIGMA HANDLERS ==================== */

  // Figma state
  window.lspFigmaLoaded = false;
  window.lspFigmaFiles = {};
  window.lspFigmaAllFrames = [];
  window.lspFigmaSelectedFrames = [];
  window.lspFigmaSyncedInfo = {};

  // Load all Figma files and their frames
  function loadFigmaData() {
    const $grid = $('#lsp-figma-frames');
    if (!$grid.length) return;

    console.log('[LSP Figma] loadFigmaData called');
    $grid.html('<div style="text-align:center;padding:40px;color:#64748b;grid-column:1/-1;"><div class="spinner is-active" style="float:none;margin:0 auto 12px;"></div><p>Loading files from Figma...</p></div>');

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_figma_get_files',
        _wpnonce: LIGHTSYNCPRO.nonce
      },
      success: function(response) {
        console.log('[LSP Figma] get_files response:', response);
        
        if (response.success) {
          window.lspFigmaFiles = {};
          window.lspFigmaAllFrames = [];
          
          const files = response.data.files || [];
          console.log('[LSP Figma] Files returned:', files.length, files);
          
          files.forEach(function(f) {
            window.lspFigmaFiles[f.key] = f;
          });
          
          window.lspFigmaLoaded = true;
          
          // Populate file filter dropdown
          updateFigmaFileFilter();
          
          // Load frames for all files
          loadAllFigmaFrames();
        } else {
          const err = (response.data && response.data.error) ? response.data.error : 'Failed to load files';
          console.error('[LSP Figma] Error:', err);
          $grid.html('<div style="text-align:center;padding:40px;color:#ef4444;grid-column:1/-1;"><p>⚠️ ' + err + '</p><button class="btn ghost" id="lsp-figma-retry">Retry</button></div>');
        }
      },
      error: function(xhr, status, error) {
        console.error('[LSP Figma] AJAX error:', status, error);
        $grid.html('<div style="text-align:center;padding:40px;color:#ef4444;grid-column:1/-1;"><p>⚠️ Connection error: ' + error + '</p><button class="btn ghost" id="lsp-figma-retry">Retry</button></div>');
      }
    });
  }

  // Show notice banner
  function showFigmaNotice(message, type, showReset) {
    $('#lsp-figma-notice').remove();
    
    var bgColor = type === 'warning' ? '#fef3c7' : '#eff6ff';
    var borderColor = type === 'warning' ? '#fcd34d' : '#93c5fd';
    var textColor = type === 'warning' ? '#92400e' : '#1e40af';
    var icon = type === 'warning' ? '⚠️' : 'ℹ️';
    
    var resetBtn = showReset ? '<button type="button" id="lsp-figma-clear-rate-limit" style="margin-left:8px;padding:4px 10px;background:#fff;border:1px solid ' + borderColor + ';border-radius:4px;cursor:pointer;font-size:12px;">Reset</button>' : '';
    
    var html = '<div id="lsp-figma-notice" style="margin-bottom:12px;padding:10px 14px;background:' + bgColor + ';border:1px solid ' + borderColor + ';border-radius:8px;font-size:13px;color:' + textColor + ';display:flex;align-items:center;gap:8px;">' +
      '<span>' + icon + '</span>' +
      '<span id="lsp-figma-notice-text">' + message + '</span>' +
      resetBtn +
      '<button type="button" onclick="$(this).parent().remove()" style="margin-left:auto;background:none;border:none;cursor:pointer;opacity:0.7;font-size:16px;">×</button>' +
    '</div>';
    
    $('#lsp-figma-frames').before(html);
  }

  // Handle clear rate limit button
  $(document).on('click', '#lsp-figma-clear-rate-limit', function() {
    var $btn = $(this);
    $btn.prop('disabled', true).text('Clearing...');
    
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_figma_clear_rate_limit',
        _wpnonce: LIGHTSYNCPRO.nonce
      },
      success: function(response) {
        if (response.success) {
          $('#lsp-figma-notice').remove();
          if (window.lspFigmaCountdown) {
            clearInterval(window.lspFigmaCountdown);
          }
          loadFigmaData();
        } else {
          $btn.prop('disabled', false).text('Reset');
          alert(response.data?.error || 'Failed to clear');
        }
      },
      error: function() {
        $btn.prop('disabled', false).text('Reset');
        alert('Connection error');
      }
    });
  });

  // Rate limit countdown
  function startRateLimitCountdown(seconds) {
    if (window.lspFigmaCountdown) {
      clearInterval(window.lspFigmaCountdown);
    }
    
    var remaining = seconds;
    window.lspFigmaCountdown = setInterval(function() {
      remaining--;
      if (remaining <= 0) {
        clearInterval(window.lspFigmaCountdown);
        $('#lsp-figma-notice-text').text('Rate limit expired. You can refresh now.');
        $('#lsp-figma-notice').css({background: '#dcfce7', borderColor: '#86efac'}).find('span:first').text('✓');
      } else {
        $('#lsp-figma-notice-text').text('Rate limit reached. Showing cached data. Try again in ' + remaining + 's');
      }
    }, 1000);
  }

  // Load frames from all files
  function loadAllFigmaFrames() {
    const fileKeys = Object.keys(window.lspFigmaFiles);
    const includeNested = $('#lsp-figma-show-nested').is(':checked');
    
    // Reset flags and synced info
    window.lspFigmaRateLimited = false;
    window.lspFigmaCached = false;
    window.lspFigmaSyncedInfo = {};
    
    // Remove any existing notices
    $('#lsp-figma-notice').remove();
    
    console.log('[LSP Figma] Loading frames for files:', fileKeys);
    
    if (fileKeys.length === 0) {
      console.log('[LSP Figma] No files to load');
      renderFigmaFrames();
      return;
    }

    // Show loading state
    $('#lsp-figma-frames').html('<div style="text-align:center;padding:40px;color:#64748b;grid-column:1/-1;"><div class="spinner is-active" style="float:none;margin:0 auto 12px;"></div><p>Loading ' + (includeNested ? 'all elements' : 'frames') + ' from ' + fileKeys.length + ' file(s)...</p></div>');

    window.lspFigmaAllFrames = [];
    var loadedCount = 0;
    var errors = [];

    fileKeys.forEach(function(fileKey) {
      console.log('[LSP Figma] Fetching frames for file:', fileKey);
      
      $.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
          action: 'lsp_figma_get_frames',
          _wpnonce: LIGHTSYNCPRO.nonce,
          file_key: fileKey,
          with_thumbnails: 1, // Fetch thumbnails with frames
          include_nested: includeNested ? 1 : 0
        },
        success: function(response) {
          console.log('[LSP Figma] Response for ' + fileKey + ':', response);
          
          if (response.success) {
            const frames = response.data.frames || [];
            const fileName = window.lspFigmaFiles[fileKey]?.name || 'Unknown';
            
            // Track if we're using cached/rate-limited data
            if (response.data.rate_limited) {
              window.lspFigmaRateLimited = true;
              window.lspFigmaRetryAfter = response.data.retry_after || 60;
            }
            if (response.data.cached) {
              window.lspFigmaCached = true;
            }
            
            console.log('[LSP Figma] Got ' + frames.length + ' frames from ' + fileName + (response.data.cached ? ' (cached)' : ''));
            
            frames.forEach(function(frame) {
              window.lspFigmaAllFrames.push({
                ...frame,
                file_key: fileKey,
                file_name: fileName
              });
            });
            
            // Store synced info for this file's frames
            var synced = response.data.synced || {};
            Object.keys(synced).forEach(function(nodeId) {
              window.lspFigmaSyncedInfo[nodeId] = synced[nodeId];
            });
          } else {
            const errMsg = (response.data && response.data.error) ? response.data.error : 'Unknown error';
            console.error('[LSP Figma] Error loading frames for ' + fileKey + ':', errMsg);
            errors.push(fileKey + ': ' + errMsg);
          }
          loadedCount++;
          if (loadedCount === fileKeys.length) {
            console.log('[LSP Figma] All files loaded. Total frames:', window.lspFigmaAllFrames.length);
            if (errors.length > 0) {
              console.error('[LSP Figma] Errors:', errors);
            }
            renderFigmaFrames();
            updateFigmaCommandBarSelection();
            
            // Show notices
            if (window.lspFigmaRateLimited) {
              var retryAfter = window.lspFigmaRetryAfter || 60;
              showFigmaNotice('Rate limit reached. Showing cached data. Try again in ' + retryAfter + ' seconds.', 'warning', true);
              
              // Start countdown
              startRateLimitCountdown(retryAfter);
            } else if (window.lspFigmaCached) {
              showFigmaNotice('Showing cached data (30 min). Click Refresh to update.', 'info', false);
            }
            
            // Show error toast if any failures
            if (errors.length > 0 && window.lspFigmaAllFrames.length === 0) {
              $('#lsp-figma-frames').html(
                '<div style="text-align:center;padding:40px;color:#ef4444;grid-column:1/-1;">' +
                '<p>⚠️ Failed to load frames</p>' +
                '<p style="font-size:12px;color:#64748b;margin-top:8px;">' + errors.join('<br>') + '</p>' +
                '<button class="btn ghost" style="margin-top:12px;" onclick="loadFigmaData()">Retry</button>' +
                '</div>'
              );
            }
          }
        },
        error: function(xhr, status, error) {
          console.error('[LSP Figma] AJAX error for ' + fileKey + ':', status, error, xhr.responseText);
          errors.push(fileKey + ': ' + (error || status));
          loadedCount++;
          if (loadedCount === fileKeys.length) {
            renderFigmaFrames();
            updateFigmaCommandBarSelection();
            
            // Show error if all failed
            if (window.lspFigmaAllFrames.length === 0) {
              $('#lsp-figma-frames').html(
                '<div style="text-align:center;padding:40px;color:#ef4444;grid-column:1/-1;">' +
                '<p>⚠️ Failed to load frames</p>' +
                '<p style="font-size:12px;color:#64748b;margin-top:8px;">' + errors.join('<br>') + '</p>' +
                '<button class="btn ghost" style="margin-top:12px;" onclick="loadFigmaData()">Retry</button>' +
                '</div>'
              );
            }
          }
        }
      });
    });
  }

  // Load frames for a single file (when clicking file card)
  function loadFigmaFrames(fileKey, fileName) {
    const includeNested = $('#lsp-figma-show-nested').is(':checked');
    
    // Reset flags and synced info
    window.lspFigmaRateLimited = false;
    window.lspFigmaCached = false;
    window.lspFigmaSyncedInfo = {};
    
    // Remove any existing notices
    $('#lsp-figma-notice').remove();
    
    console.log('[LSP Figma] Loading frames for single file:', fileKey, fileName);

    // Show loading state
    $('#lsp-figma-frames').html('<div style="text-align:center;padding:40px;color:#64748b;grid-column:1/-1;"><div class="spinner is-active" style="float:none;margin:0 auto 12px;"></div><p>Loading ' + (includeNested ? 'all elements' : 'frames') + ' from "' + escapeHtml(fileName) + '"...</p></div>');

    window.lspFigmaAllFrames = [];

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_figma_get_frames',
        _wpnonce: LIGHTSYNCPRO.nonce,
        file_key: fileKey,
        with_thumbnails: 1,
        include_nested: includeNested ? 1 : 0
      },
      success: function(response) {
        console.log('[LSP Figma] Single file response for ' + fileKey + ':', response);
        
        if (response.success) {
          const frames = response.data.frames || [];
          
          // Track if we're using cached/rate-limited data
          if (response.data.rate_limited) {
            window.lspFigmaRateLimited = true;
            window.lspFigmaRetryAfter = response.data.retry_after || 60;
          }
          if (response.data.cached) {
            window.lspFigmaCached = true;
          }
          
          console.log('[LSP Figma] Got ' + frames.length + ' frames from ' + fileName + (response.data.cached ? ' (cached)' : ''));
          
          frames.forEach(function(frame) {
            window.lspFigmaAllFrames.push({
              ...frame,
              file_key: fileKey,
              file_name: fileName
            });
          });
          
          // Store synced info for this file's frames
          var synced = response.data.synced || {};
          Object.keys(synced).forEach(function(nodeId) {
            window.lspFigmaSyncedInfo[nodeId] = synced[nodeId];
          });
          
          // Update file filter to show this file selected
          $('#lsp-figma-file-filter').val(fileKey);
          
          renderFigmaFrames();
          updateFigmaCommandBarSelection();
          
          // Show notices
          if (window.lspFigmaRateLimited) {
            showFigmaNotice('Rate limit reached. Showing cached data.', 'warning', true);
          } else if (window.lspFigmaCached) {
            showFigmaNotice('Showing cached data (30 min). Click Refresh to update.', 'info', false);
          }
        } else {
          const errMsg = (response.data && response.data.error) ? response.data.error : 'Unknown error';
          console.error('[LSP Figma] Error loading frames for ' + fileKey + ':', errMsg);
          $('#lsp-figma-frames').html(
            '<div style="text-align:center;padding:40px;color:#ef4444;grid-column:1/-1;">' +
            '<p>⚠️ Failed to load frames</p>' +
            '<p style="font-size:12px;color:#64748b;margin-top:8px;">' + escapeHtml(errMsg) + '</p>' +
            '</div>'
          );
        }
      },
      error: function(xhr, status, error) {
        console.error('[LSP Figma] AJAX error:', status, error);
        $('#lsp-figma-frames').html('<div style="text-align:center;padding:40px;color:#ef4444;grid-column:1/-1;"><p>⚠️ Connection error</p></div>');
      }
    });
  }

  // Update file filter dropdown
  function updateFigmaFileFilter() {
    const $filter = $('#lsp-figma-file-filter');
    var html = '<option value="all">All Files</option>';
    
    Object.keys(window.lspFigmaFiles).forEach(function(key) {
      const file = window.lspFigmaFiles[key];
      html += '<option value="' + key + '">' + escapeHtml(file.name) + '</option>';
    });
    
    $filter.html(html);
    
    // Also update files list
    updateFigmaFilesList();
  }

  // Render visual files grid with thumbnails
  function updateFigmaFilesList() {
    const $grid = $('#lsp-figma-files-grid');
    const $empty = $('#lsp-figma-files-empty');
    const files = Object.values(window.lspFigmaFiles || {});
    
    if (files.length === 0) {
      $grid.hide();
      $empty.show();
      return;
    }
    
    $empty.hide();
    $grid.show().empty();
    
    files.forEach(function(file) {
      var thumbnailStyle = file.thumbnail 
        ? 'background-image:url(' + escapeHtml(file.thumbnail) + ');background-size:cover;background-position:center;'
        : 'background:#e2e8f0;display:flex;align-items:center;justify-content:center;';
      
      var cardHtml = 
        '<div class="lsp-figma-file-card" data-key="' + escapeHtml(file.key) + '" style="position:relative;border:2px solid #e2e8f0;border-radius:10px;overflow:hidden;cursor:pointer;transition:all 0.15s;">' +
          '<div style="aspect-ratio:4/3;' + thumbnailStyle + '">' +
            (file.thumbnail ? '' : '<svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="#94a3b8" stroke-width="1.5"><path d="M5 5.5A3.5 3.5 0 0 1 8.5 2H12v7H8.5A3.5 3.5 0 0 1 5 5.5z"/><path d="M12 2h3.5a3.5 3.5 0 1 1 0 7H12V2z"/><path d="M12 12.5a3.5 3.5 0 1 1 7 0 3.5 3.5 0 1 1-7 0z"/><path d="M5 19.5A3.5 3.5 0 0 1 8.5 16H12v3.5a3.5 3.5 0 1 1-7 0z"/><path d="M5 12.5A3.5 3.5 0 0 1 8.5 9H12v7H8.5A3.5 3.5 0 0 1 5 12.5z"/></svg>') +
          '</div>' +
          '<div style="padding:8px 10px;background:#fff;border-top:1px solid #f1f5f9;">' +
            '<div style="font-size:12px;font-weight:500;color:#374151;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;" title="' + escapeHtml(file.name) + '">' + escapeHtml(file.name) + '</div>' +
          '</div>' +
          '<button type="button" class="lsp-figma-remove-file" data-key="' + escapeHtml(file.key) + '" style="position:absolute;top:6px;right:6px;width:22px;height:22px;background:rgba(0,0,0,0.6);border:none;border-radius:50%;cursor:pointer;display:none;align-items:center;justify-content:center;" title="Remove file">' +
            '<svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><path d="M18 6L6 18M6 6l12 12"/></svg>' +
          '</button>' +
        '</div>';
      $grid.append(cardHtml);
    });
    
    // Add hover styles via CSS class behavior
    $grid.find('.lsp-figma-file-card').hover(
      function() { 
        $(this).css('border-color', '#2563eb').find('.lsp-figma-remove-file').css('display', 'flex'); 
      },
      function() { 
        $(this).css('border-color', '#e2e8f0').find('.lsp-figma-remove-file').css('display', 'none'); 
      }
    );
  }

  // Handle click on file card to load frames
  $(document).on('click', '.lsp-figma-file-card', function(e) {
    if ($(e.target).closest('.lsp-figma-remove-file').length) return; // Don't trigger if clicking remove button
    
    var fileKey = $(this).data('key');
    var file = window.lspFigmaFiles[fileKey];
    if (!file) return;
    
    // Visual feedback - highlight selected card
    $('.lsp-figma-file-card').css('border-color', '#e2e8f0');
    $(this).css('border-color', '#2563eb');
    
    // Load frames for this file
    loadFigmaFrames(fileKey, file.name);
  });

  // Show/hide add URL panel
  $(document).on('click', '#lsp-figma-show-add-url', function() {
    $('#lsp-figma-add-url-panel').slideDown(200);
    $('#lsp-figma-file-url').focus();
  });
  
  $(document).on('click', '#lsp-figma-hide-add-url', function() {
    $('#lsp-figma-add-url-panel').slideUp(200);
    $('#lsp-figma-file-url').val('');
  });

  // Handle remove file click
  $(document).on('click', '.lsp-figma-remove-file', function() {
    var fileKey = $(this).data('key');
    var fileName = window.lspFigmaFiles[fileKey]?.name || fileKey;
    
    if (!confirm('Remove "' + fileName + '" from LightSync Pro?\n\nThis won\'t delete anything from Figma, only disconnects this file.')) {
      return;
    }
    
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_figma_remove_file',
        _wpnonce: LIGHTSYNCPRO.nonce,
        file_key: fileKey
      },
      success: function(response) {
        if (response.success) {
          // Remove from local state
          delete window.lspFigmaFiles[fileKey];
          
          // Remove frames for this file
          window.lspFigmaAllFrames = (window.lspFigmaAllFrames || []).filter(function(f) {
            return f.file_key !== fileKey;
          });
          
          // Clear any selections for this file
          window.lspFigmaSelectedFrames = (window.lspFigmaSelectedFrames || []).filter(function(id) {
            var frame = (window.lspFigmaAllFrames || []).find(function(f) { return f.id === id; });
            return !frame || frame.file_key !== fileKey;
          });
          
          // Update UI
          updateFigmaFileFilter();
          renderFigmaFrames();
          updateFigmaCommandBarSelection();
        } else {
          alert(response.data?.error || 'Failed to remove file');
        }
      },
      error: function() {
        alert('Connection error');
      }
    });
  });

  // Render frames grid
  function renderFigmaFrames() {
    const frames = window.lspFigmaAllFrames || [];
    const selected = window.lspFigmaSelectedFrames || [];
    const $grid = $('#lsp-figma-frames');
    const search = ($('#lsp-figma-search').val() || '').toLowerCase().trim();
    const fileFilter = $('#lsp-figma-file-filter').val() || 'all';
    const typeFilter = $('#lsp-figma-type-filter').val() || 'all';
    const showNested = $('#lsp-figma-show-nested').is(':checked');

    // Filter frames
    const filtered = frames.filter(function(frame) {
      if (fileFilter !== 'all' && frame.file_key !== fileFilter) return false;
      if (typeFilter !== 'all' && frame.type !== typeFilter) return false;
      if (search && !frame.name.toLowerCase().includes(search) && !frame.file_name.toLowerCase().includes(search)) return false;
      return true;
    });

    if (frames.length === 0) {
      $grid.html('<div style="text-align:center;padding:40px;color:#64748b;grid-column:1/-1;"><p>No frames yet. Add a Figma file URL above to get started.</p></div>');
      return;
    }

    if (filtered.length === 0) {
      $grid.html('<div style="text-align:center;padding:40px;color:#64748b;grid-column:1/-1;"><p>No frames match your filter.</p></div>');
      return;
    }

    // Add styles if not present
    if (!$('#lsp-figma-styles').length) {
      $('head').append('<style id="lsp-figma-styles">' +
        '.lsp-figma-frame-card { position:relative;background:#fff;border:2px solid #e2e8f0;border-radius:8px;cursor:pointer;transition:all 0.2s;overflow:hidden; }' +
        '.lsp-figma-frame-card:hover { border-color:#2563eb;box-shadow:0 4px 12px rgba(0,0,0,0.1); }' +
        '.lsp-figma-frame-card.selected { border-color:#2563eb;background:#eff6ff; }' +
        '.lsp-figma-frame-thumb { height:100px;background:#f1f5f9;display:flex;align-items:center;justify-content:center;overflow:hidden; }' +
        '.lsp-figma-frame-thumb img { width:100%;height:100%;object-fit:contain;background:#f8fafc; }' +
        '.lsp-figma-type-badge { font-size:9px;padding:2px 6px;border-radius:4px;background:#e2e8f0;color:#64748b;text-transform:uppercase;letter-spacing:0.5px; }' +
        '.lsp-figma-type-badge.component { background:#dbeafe;color:#1d4ed8; }' +
        '.lsp-figma-type-badge.instance { background:#f3e8ff;color:#7c3aed; }' +
        '.lsp-figma-type-badge.vector { background:#dcfce7;color:#16a34a; }' +
        '.lsp-figma-type-badge.text { background:#fef3c7;color:#b45309; }' +
        '.lsp-figma-type-badge.group { background:#fee2e2;color:#dc2626; }' +
      '</style>');
    }

    // Type badge helper
    function getTypeBadgeClass(type) {
      if (type === 'COMPONENT' || type === 'COMPONENT_SET') return 'component';
      if (type === 'INSTANCE') return 'instance';
      if (type === 'VECTOR' || type === 'BOOLEAN_OPERATION') return 'vector';
      if (type === 'TEXT') return 'text';
      if (type === 'GROUP') return 'group';
      return '';
    }

    var html = '';
    filtered.forEach(function(frame) {
      const isSelected = selected.includes(frame.id);
      const syncInfo = window.lspFigmaSyncedInfo && window.lspFigmaSyncedInfo[frame.id];
      const isSynced = !!syncInfo;
      const needsUpdate = syncInfo && syncInfo.needs_update;
      const syncDest = syncInfo && syncInfo.dest ? syncInfo.dest : 'wp';
      const thumbUrl = frame.thumbnail || frame.thumbnail_url;
      const hasParent = frame.parent && showNested;
      const typeBadgeClass = getTypeBadgeClass(frame.type);
      
      // Build destination icons (same SVGs as destination selector)
      var destIconsHtml = '';
      if (isSynced) {
        destIconsHtml = '<span style="display:inline-flex;gap:3px;margin-left:4px;vertical-align:middle;">';
        if (syncDest === 'wp' || syncDest === 'both') {
          destIconsHtml += '<span title="Synced to WordPress" style="display:flex;"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><rect x="3" y="3" width="7" height="7" rx="1"/><rect x="14" y="3" width="7" height="7" rx="1"/><rect x="3" y="14" width="7" height="7" rx="1"/><rect x="14" y="14" width="7" height="7" rx="1"/></svg></span>';
        }
        if (syncDest === 'shopify' || syncDest === 'both') {
          destIconsHtml += '<span title="Synced to Shopify" style="display:flex;"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/><line x1="3" y1="6" x2="21" y2="6"/><path d="M16 10a4 4 0 01-8 0"/></svg></span>';
        }
        destIconsHtml += '</span>';
      }
      
      html += '<div class="lsp-figma-frame-card' + (isSelected ? ' selected' : '') + (isSynced ? ' synced' : '') + (needsUpdate ? ' needs-update' : '') + '" data-id="' + frame.id + '" data-file="' + frame.file_key + '" title="' + escapeHtml(frame.name) + (hasParent ? ' (in ' + escapeHtml(frame.parent) + ')' : '') + '">';
      html += '<div class="lsp-figma-frame-thumb">';
      if (thumbUrl) {
        html += '<img src="' + thumbUrl + '" alt="' + escapeHtml(frame.name) + '" loading="lazy">';
      } else {
        html += '<span style="font-size:10px;color:#94a3b8;text-transform:uppercase;">' + (frame.type || 'FRAME') + '</span>';
      }
      html += '</div>';
      html += '<div style="padding:8px;">';
      html += '<div style="font-size:12px;font-weight:500;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">' + escapeHtml(frame.name) + '</div>';
      if (hasParent) {
        html += '<div style="font-size:10px;color:#64748b;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">↳ ' + escapeHtml(frame.parent) + '</div>';
      }
      html += '<div style="display:flex;align-items:center;gap:6px;margin-top:4px;">';
      html += '<span class="lsp-figma-type-badge ' + typeBadgeClass + '">' + (frame.type || 'FRAME') + '</span>';
      if (!hasParent) {
        html += '<span style="font-size:10px;color:#94a3b8;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">' + escapeHtml(frame.file_name) + '</span>';
      }
      html += '</div>';
      html += '</div>';
      if (isSelected) {
        html += '<div class="lsp-frame-selected-badge" style="position:absolute;top:4px;right:4px;width:20px;height:20px;background:#2563eb;border-radius:50%;display:flex;align-items:center;justify-content:center;"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="3"><path d="M20 6L9 17l-5-5"/></svg></div>';
      }
      // Sync status badges with destination icons (upper left) - Glassmorphism style
      if (needsUpdate) {
        html += '<div class="lsp-sync-badge lsp-sync-update" style="position:absolute;top:6px;left:6px;background:rgba(245,158,11,0.85);backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);color:#fff;font-size:10px;font-weight:500;padding:4px 8px;border-radius:6px;display:flex;align-items:center;gap:2px;box-shadow:0 2px 8px rgba(0,0,0,0.15);border:1px solid rgba(255,255,255,0.2);z-index:2;" title="File has been updated in Figma"><svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" style="margin-right:2px;"><path d="M23 4v6h-6M1 20v-6h6"/><path d="M3.51 9a9 9 0 0114.85-3.36L23 10M1 14l4.64 4.36A9 9 0 0020.49 15"/></svg>' + destIconsHtml + '</div>';
      } else if (isSynced) {
        html += '<div class="lsp-sync-badge lsp-sync-done" style="position:absolute;top:6px;left:6px;background:rgba(16,185,129,0.85);backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);color:#fff;font-size:10px;font-weight:500;padding:4px 8px;border-radius:6px;display:flex;align-items:center;gap:2px;box-shadow:0 2px 8px rgba(0,0,0,0.15);border:1px solid rgba(255,255,255,0.2);z-index:2;" title="Synced"><svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" style="margin-right:2px;"><path d="M20 6L9 17l-5-5"/></svg>' + destIconsHtml + '</div>';
      }
      html += '</div>';
    });

    $grid.html(html);

    // Update count
    $('#lsp-figma-selection-count').text(filtered.length + ' element' + (filtered.length !== 1 ? 's' : '') + ' shown');

    // Click handler for frame selection
    $grid.find('.lsp-figma-frame-card').on('click', function() {
      const $card = $(this);
      const frameId = $card.data('id');
      
      if ($card.hasClass('selected')) {
        $card.removeClass('selected');
        $card.find('.lsp-frame-selected-badge').remove();
        window.lspFigmaSelectedFrames = window.lspFigmaSelectedFrames.filter(function(id) { return id !== frameId; });
      } else {
        $card.addClass('selected');
        $card.append('<div class="lsp-frame-selected-badge" style="position:absolute;top:4px;right:4px;width:20px;height:20px;background:#2563eb;border-radius:50%;display:flex;align-items:center;justify-content:center;"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="3"><path d="M20 6L9 17l-5-5"/></svg></div>');
        window.lspFigmaSelectedFrames.push(frameId);
      }
      
      updateFigmaCommandBarSelection();
    });
  }

  // Search handler
  $('#lsp-figma-search').on('input', debounce(function() {
    renderFigmaFrames();
  }, 300));

  // File filter handler
  $('#lsp-figma-file-filter').on('change', function() {
    renderFigmaFrames();
  });

  // Nested elements toggle
  $('#lsp-figma-show-nested').on('change', function() {
    const showNested = $(this).is(':checked');
    
    // Show/hide type filter
    if (showNested) {
      $('#lsp-figma-type-filter').show();
    } else {
      $('#lsp-figma-type-filter').hide().val('all');
    }
    
    // Clear selections when toggling
    window.lspFigmaSelectedFrames = [];
    
    // Reload frames with new setting
    loadAllFigmaFrames();
  });

  // Type filter handler
  $('#lsp-figma-type-filter').on('change', function() {
    renderFigmaFrames();
  });

  // Select all/none
  $('#lsp-figma-select-all').on('click', function() {
    const fileFilter = $('#lsp-figma-file-filter').val() || 'all';
    const typeFilter = $('#lsp-figma-type-filter').val() || 'all';
    const search = ($('#lsp-figma-search').val() || '').toLowerCase().trim();
    
    window.lspFigmaSelectedFrames = (window.lspFigmaAllFrames || [])
      .filter(function(frame) {
        if (fileFilter !== 'all' && frame.file_key !== fileFilter) return false;
        if (typeFilter !== 'all' && frame.type !== typeFilter) return false;
        if (search && !frame.name.toLowerCase().includes(search)) return false;
        return true;
      })
      .map(function(f) { return f.id; });
    
    renderFigmaFrames();
    updateFigmaCommandBarSelection();
  });

  $('#lsp-figma-select-none').on('click', function() {
    window.lspFigmaSelectedFrames = [];
    renderFigmaFrames();
    updateFigmaCommandBarSelection();
  });

  // Add file
  $('#lsp-figma-add-file-btn').on('click', function() {
    const $input = $('#lsp-figma-file-url');
    const url = $input.val().trim();
    if (!url) return;

    const $btn = $(this);
    $btn.prop('disabled', true).text('Adding...');

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_figma_add_file',
        _wpnonce: LIGHTSYNCPRO.nonce,
        file_url: url
      },
      success: function(response) {
        $btn.prop('disabled', false).html('<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right:4px;"><path d="M12 5v14M5 12h14"/></svg>Add');
        if (response.success) {
          $input.val('');
          // Hide add URL panel
          $('#lsp-figma-add-url-panel').slideUp(200);
          // Reload all data
          loadFigmaData();
        } else {
          alert(response.data.error || 'Failed to add file');
        }
      },
      error: function() {
        $btn.prop('disabled', false).html('<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right:4px;"><path d="M12 5v14M5 12h14"/></svg>Add');
        alert('Connection error');
      }
    });
  });

  // Refresh
  $('#lsp-figma-refresh').on('click', function() {
    loadFigmaData();
  });

  // Check for Updates button
  $('#lsp-figma-check-updates').on('click', function() {
    var $btn = $(this);
    var originalHtml = $btn.html();
    
    $btn.prop('disabled', true).html('<span class="spinner is-active" style="float:none;margin:0 6px 0 0;"></span>Checking...');
    
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_figma_check_updates',
        _wpnonce: LIGHTSYNCPRO.nonce
      },
      success: function(response) {
        $btn.prop('disabled', false).html(originalHtml);
        
        if (response.success) {
          var data = response.data;
          
          // Log debug info to console
          console.log('[LSP Figma] Check for Updates debug:', data.debug);
          
          if (data.frames_need_update > 0) {
            // Show notification with update info
            var msg = data.message + '\n\nFiles with updates:\n';
            Object.keys(data.updates).forEach(function(fileKey) {
              var update = data.updates[fileKey];
              msg += '• ' + update.name + ' (' + update.frames_need_update + ' frame(s))\n';
            });
            msg += '\nRefresh the page to see "Needs Update" badges.';
            alert(msg);
            
            // Reload to show updated badges
            loadFigmaData();
          } else {
            // Show floating success notification
            var notifHtml = 
              '<div id="lsp-figma-updates-notif" style="position:fixed;bottom:20px;right:20px;background:#fff;border-radius:12px;padding:16px 20px;box-shadow:0 8px 30px rgba(0,0,0,.18);z-index:100000;max-width:340px;border-left:4px solid #10b981;">' +
                '<div style="display:flex;align-items:flex-start;gap:12px;">' +
                  '<div style="width:32px;height:32px;background:#d1fae5;border-radius:8px;display:flex;align-items:center;justify-content:center;flex-shrink:0;">' +
                    '<span style="color:#10b981;font-size:18px;">✓</span>' +
                  '</div>' +
                  '<div style="flex:1;">' +
                    '<strong style="display:block;color:#1e293b;font-size:14px;margin-bottom:4px;">All up to date</strong>' +
                    '<span style="font-size:13px;color:#64748b;">No updates available for synced frames.</span>' +
                  '</div>' +
                  '<button onclick="this.parentNode.parentNode.remove()" style="background:none;border:none;color:#94a3b8;cursor:pointer;font-size:18px;padding:0;line-height:1;">×</button>' +
                '</div>' +
              '</div>';
            document.body.insertAdjacentHTML('beforeend', notifHtml);
            setTimeout(function() {
              var el = document.getElementById('lsp-figma-updates-notif');
              if (el) el.remove();
            }, 4000);
          }
        } else {
          alert('Error: ' + (response.data.error || 'Unknown error'));
        }
      },
      error: function() {
        $btn.prop('disabled', false).html(originalHtml);
        alert('Connection error');
      }
    });
  });

  // Load thumbnails button
  $('#lsp-figma-load-thumbs').on('click', function() {
    var $btn = $(this);
    var fileKeys = Object.keys(window.lspFigmaFiles || {});
    
    if (fileKeys.length === 0) {
      alert('No files loaded yet');
      return;
    }
    
    $btn.prop('disabled', true).html('<span class="spinner is-active" style="float:none;margin:0 6px 0 0;"></span>Loading...');
    
    var loadedCount = 0;
    var allThumbnails = {};
    
    fileKeys.forEach(function(fileKey) {
      $.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
          action: 'lsp_figma_get_frames',
          _wpnonce: LIGHTSYNCPRO.nonce,
          file_key: fileKey,
          with_thumbnails: 1,
          include_nested: $('#lsp-figma-show-nested').is(':checked') ? 1 : 0
        },
        success: function(response) {
          if (response.success && response.data.frames) {
            response.data.frames.forEach(function(frame) {
              if (frame.thumbnail) {
                allThumbnails[frame.id] = frame.thumbnail;
              }
            });
          }
          loadedCount++;
          if (loadedCount === fileKeys.length) {
            // Update existing frames with thumbnails
            (window.lspFigmaAllFrames || []).forEach(function(frame) {
              if (allThumbnails[frame.id]) {
                frame.thumbnail = allThumbnails[frame.id];
              }
            });
            renderFigmaFrames();
            $btn.prop('disabled', false).html('<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right:6px;"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"/><circle cx="8.5" cy="8.5" r="1.5"/><polyline points="21 15 16 10 5 21"/></svg>Load Previews');
            showFigmaNotice('Previews loaded for ' + Object.keys(allThumbnails).length + ' frames', 'info', false);
          }
        },
        error: function() {
          loadedCount++;
          if (loadedCount === fileKeys.length) {
            renderFigmaFrames();
            $btn.prop('disabled', false).html('<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right:6px;"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"/><circle cx="8.5" cy="8.5" r="1.5"/><polyline points="21 15 16 10 5 21"/></svg>Load Previews');
            showFigmaNotice('Some previews failed to load. Try again in a minute.', 'warning', false);
          }
        }
      });
    });
  });

  // Retry handler (for error states)
  $(document).on('click', '#lsp-figma-retry', function() {
    loadFigmaData();
  });

  // Disconnect
  $('#lsp-figma-disconnect').on('click', function() {
    if (!confirm('Disconnect from Figma? Your synced images will remain in WordPress.')) return;
    
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_figma_disconnect',
        _wpnonce: LIGHTSYNCPRO.nonce
      },
      success: function(response) {
        if (response.success) {
          location.reload();
        }
      }
    });
  });

  // Disconnect Dropbox
  $('#lsp-dropbox-disconnect').on('click', function() {
    if (!confirm('Disconnect from Dropbox? Your synced images will remain in WordPress.')) return;
    
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_dropbox_disconnect',
        _wpnonce: LIGHTSYNCPRO.nonce
      },
      success: function(response) {
        if (response.success) {
          location.reload();
        }
      }
    });
  });

  // OpenRouter AI disconnect (bottom section button)
  $('#lsp-ai-disconnect-btn').on('click', function() {
    if (!confirm('Disconnect from OpenRouter? Your AI-generated images will remain in WordPress.')) return;
    
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_openrouter_disconnect',
        _wpnonce: LIGHTSYNCPRO.nonce
      },
      success: function(response) {
        if (response.success) {
          location.reload();
        }
      }
    });
  });

  // Disconnect Shutterstock
  $('#lsp-shutterstock-disconnect').on('click', function() {
    if (!confirm('Disconnect from Shutterstock? Your synced images will remain in WordPress.')) return;
    
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_shutterstock_disconnect',
        _wpnonce: LIGHTSYNCPRO.nonce
      },
      success: function(response) {
        if (response.success) {
          location.reload();
        }
      }
    });
  });



  // Save sync target
  $('input[name="lsp_figma_sync_target"]').on('change', function() {
    const target = $(this).val();
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_figma_save_target',
        _wpnonce: LIGHTSYNCPRO.nonce,
        target: target
      }
    });
  });

  // Auto-load Figma if starting on Figma tab
  $(function(){
    if ($('#lsp-figma-content.active').length && !window.lspFigmaLoaded) {
      loadFigmaData();
      // Also load teams for browser
      loadFigmaTeams();
    }
  });

  // Expose Figma functions for use by admin-sync.js
  window.loadFigmaData = loadFigmaData;
  window.renderFigmaFrames = renderFigmaFrames;

  /* ==================== END FIGMA HANDLERS ==================== */

  /* ==================== DROPBOX HANDLERS ==================== */

  // Dropbox state
  window.lspDropboxLoaded = false;
  window.lspDropboxCurrentPath = '';
  window.lspDropboxFolders = [];
  window.lspDropboxImages = [];
  window.lspDropboxAllFiles = [];
  window.lspDropboxSelectedFiles = [];
  window.lspDropboxSynced = {};

  /**
   * Load Dropbox folder contents
   */
  function loadDropboxFolder(path) {
    const $content = $('#lsp-dropbox-browser-content');
    
    console.log('[LSP Dropbox] Loading folder:', path || '(root)');
    
    $content.html(`
      <div style="text-align:center;padding:40px;color:#64748b;grid-column:1/-1;">
        <div class="spinner is-active" style="float:none;margin:0 auto 12px;"></div>
        <p>Loading${path ? ' folder...' : ' your Dropbox...'}</p>
      </div>
    `);

    // Load synced file IDs first (only on initial load)
    const loadSynced = !window.lspDropboxLoaded ? 
      $.post(ajaxurl, {
        action: 'lsp_dropbox_get_synced',
        _wpnonce: LIGHTSYNCPRO.nonce
      }).then(r => {
        if (r.success) window.lspDropboxSynced = r.data || {};
      }).catch(() => {}) : Promise.resolve();

    loadSynced.then(() => {
      $.post(ajaxurl, {
        action: 'lsp_dropbox_list_folder',
        path: path,
        _wpnonce: LIGHTSYNCPRO.nonce
      })
      .done(function(r) {
        console.log('[LSP Dropbox] AJAX response:', r);
        
        if (!r.success) {
          $content.html(`
            <div style="text-align:center;padding:40px;color:#dc2626;grid-column:1/-1;">
              <p>Error: ${r.data?.error || 'Failed to load folder'}</p>
              <button type="button" class="btn ghost" onclick="loadDropboxFolder('${path}')">Try Again</button>
            </div>
          `);
          return;
        }

        window.lspDropboxLoaded = true;
        window.lspDropboxCurrentPath = path;
        window.lspDropboxFolders = r.data.folders || [];
        window.lspDropboxImages = r.data.images || [];
        
        // Update all files count
        window.lspDropboxAllFiles = window.lspDropboxImages;

        renderDropboxContent();
        updateDropboxBreadcrumb(path);
        updateDropboxCommandBarSelection();
        
        // Update count badge
        const total = window.lspDropboxFolders.length + window.lspDropboxImages.length;
        $('#lsp-dropbox-count').text('Items: ' + total);
      })
      .fail(function(xhr, status, error) {
        console.error('Dropbox AJAX error:', status, error, xhr.responseText);
        let errorMsg = 'Connection failed';
        try {
          const resp = JSON.parse(xhr.responseText);
          if (resp.data?.error) errorMsg = resp.data.error;
        } catch(e) {}
        
        $content.html(`
          <div style="text-align:center;padding:40px;color:#dc2626;grid-column:1/-1;">
            <p>Error: ${errorMsg}</p>
            <p style="font-size:12px;color:#94a3b8;margin-top:8px;">${status}: ${error}</p>
            <button type="button" class="btn ghost" style="margin-top:12px;" onclick="loadDropboxFolder('${path}')">Try Again</button>
          </div>
        `);
      });
    });
  }

  /**
   * Render Dropbox folders and images
   */
  function renderDropboxContent() {
    const $content = $('#lsp-dropbox-browser-content');
    const folders = window.lspDropboxFolders || [];
    const images = window.lspDropboxImages || [];
    const synced = window.lspDropboxSynced || {};
    const selected = window.lspDropboxSelectedFiles || [];
    const searchTerm = ($('#lsp-dropbox-search').val() || '').toLowerCase();

    let html = '';

    // Filter folders
    const filteredFolders = folders.filter(f => 
      !searchTerm || f.name.toLowerCase().includes(searchTerm)
    );

    // Filter images
    const filteredImages = images.filter(img => 
      !searchTerm || img.name.toLowerCase().includes(searchTerm)
    );

    // Render folders first
    filteredFolders.forEach(folder => {
      html += `
        <div class="lsp-dropbox-folder-card" data-path="${folder.path}" 
             style="background:#f8fafc;border:1px solid #e2e8f0;border-radius:10px;padding:16px;cursor:pointer;transition:all 0.15s;text-align:center;">
          <svg width="40" height="40" viewBox="0 0 24 24" fill="#0891b2" stroke="none" style="margin:0 auto 8px;display:block;">
            <path d="M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"/>
          </svg>
          <div style="font-size:13px;font-weight:500;color:#374151;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;" title="${folder.name}">
            ${folder.name}
          </div>
        </div>
      `;
    });

    // Render images with thumbnail placeholders
    filteredImages.forEach(img => {
      try {
        const isSelected = selected.includes(img.id);
        const syncData = synced && synced.hasOwnProperty && synced.hasOwnProperty(img.id) ? synced[img.id] : null;
        // Handle both old format (number) and new format (object with time/dest)
        const isSynced = !!syncData;
        const syncedAt = syncData ? (typeof syncData === 'object' ? Number(syncData.time) : Number(syncData)) : 0;
        const syncDest = syncData && typeof syncData === 'object' ? (syncData.dest || 'wp') : 'wp';
        const syncedDate = (isSynced && syncedAt > 0) ? new Date(syncedAt * 1000).toLocaleDateString() : '';
        
        // Check if file was modified after last sync (safely parse date)
        let modifiedAt = 0;
        if (img.modified && typeof img.modified === 'string' && img.modified.length > 0) {
          const parsed = Date.parse(img.modified);
          if (!isNaN(parsed)) {
            modifiedAt = parsed / 1000;
          }
        }
        
        // Only mark as updated if:
        // 1. File is synced
        // 2. Modified time is MORE than 60 seconds after sync time (buffer for timing differences)
        // 3. Sync was not within the last 60 seconds (grace period for recent syncs)
        const now = Math.floor(Date.now() / 1000);
        const syncAge = now - syncedAt;
        const isRecentSync = syncedAt > 0 && syncAge < 60; // Synced within last minute
        const modifiedAfterSync = modifiedAt > 0 && syncedAt > 0 && (modifiedAt - syncedAt) > 60;
        const isUpdated = isSynced && !isRecentSync && modifiedAfterSync;
        
        // Build destination icons (same SVGs as destination selector)
        let destIconsHtml = '';
        if (isSynced) {
          destIconsHtml = '<span style="display:inline-flex;gap:3px;margin-left:4px;vertical-align:middle;">';
          if (syncDest === 'wp' || syncDest === 'both') {
            destIconsHtml += '<span title="Synced to WordPress" style="display:flex;"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><rect x="3" y="3" width="7" height="7" rx="1"/><rect x="14" y="3" width="7" height="7" rx="1"/><rect x="3" y="14" width="7" height="7" rx="1"/><rect x="14" y="14" width="7" height="7" rx="1"/></svg></span>';
          }
          if (syncDest === 'shopify' || syncDest === 'both') {
            destIconsHtml += '<span title="Synced to Shopify" style="display:flex;"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/><line x1="3" y1="6" x2="21" y2="6"/><path d="M16 10a4 4 0 01-8 0"/></svg></span>';
          }
          destIconsHtml += '</span>';
        }
        
        // Build sync badge for upper left - Glassmorphism style
        let syncBadge = '';
        if (isUpdated) {
          syncBadge = `<div class="lsp-sync-badge lsp-sync-update" style="position:absolute;top:6px;left:6px;background:rgba(245,158,11,0.85);backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);color:#fff;font-size:10px;font-weight:500;padding:4px 8px;border-radius:6px;display:flex;align-items:center;gap:2px;box-shadow:0 2px 8px rgba(0,0,0,0.15);border:1px solid rgba(255,255,255,0.2);z-index:2;" title="Updated since last sync"><svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" style="margin-right:2px;"><path d="M23 4v6h-6M1 20v-6h6"/><path d="M3.51 9a9 9 0 0114.85-3.36L23 10M1 14l4.64 4.36A9 9 0 0020.49 15"/></svg>${destIconsHtml}</div>`;
        } else if (isSynced) {
          syncBadge = `<div class="lsp-sync-badge lsp-sync-done" style="position:absolute;top:6px;left:6px;background:rgba(16,185,129,0.85);backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);color:#fff;font-size:10px;font-weight:500;padding:4px 8px;border-radius:6px;display:flex;align-items:center;gap:2px;box-shadow:0 2px 8px rgba(0,0,0,0.15);border:1px solid rgba(255,255,255,0.2);z-index:2;" title="Synced ${syncedDate}"><svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" style="margin-right:2px;"><path d="M20 6L9 17l-5-5"/></svg>${destIconsHtml}</div>`;
        }
        
        // Border color based on state
        let borderColor = '#e2e8f0';
        if (isSelected) borderColor = '#0891b2';
        else if (isUpdated) borderColor = '#f59e0b';
        
        html += `
          <div class="lsp-dropbox-image-card ${isSelected ? 'selected' : ''}" 
               data-id="${img.id}" data-path="${img.path}" data-name="${img.name}" data-synced="${isSynced ? '1' : '0'}"
               style="position:relative;background:#fff;border:2px solid ${borderColor};border-radius:10px;overflow:hidden;cursor:pointer;transition:all 0.15s;">
            ${isUpdated ? '<div style="position:absolute;top:0;left:0;right:0;height:3px;background:linear-gradient(90deg,#f59e0b,#fbbf24);z-index:1;"></div>' : ''}
            ${syncBadge}
            <div class="lsp-dropbox-thumb" data-path="${img.path}" style="aspect-ratio:1;background:#f1f5f9;display:flex;align-items:center;justify-content:center;overflow:hidden;">
              <div class="lsp-thumb-placeholder">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="#cbd5e1" stroke-width="1.5">
                  <rect x="3" y="3" width="18" height="18" rx="2"/>
                  <circle cx="8.5" cy="8.5" r="1.5"/>
                  <path d="M21 15l-5-5L5 21"/>
                </svg>
              </div>
            </div>
            <div style="padding:8px;">
              <div style="font-size:12px;font-weight:500;color:#374151;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;" title="${img.name}">
                ${img.name}
              </div>
            </div>
            <div class="lsp-dropbox-check" style="position:absolute;top:8px;right:8px;width:22px;height:22px;border-radius:50%;background:${isSelected ? '#0891b2' : 'rgba(255,255,255,0.9)'};border:2px solid ${isSelected ? '#0891b2' : '#d1d5db'};display:flex;align-items:center;justify-content:center;">
              ${isSelected ? '<svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="3"><path d="M20 6L9 17l-5-5"/></svg>' : ''}
            </div>
          </div>
        `;
      } catch (e) {
        console.error('[LSP Dropbox] Error rendering image:', img, e);
      }
    });

    if (!filteredFolders.length && !filteredImages.length) {
      html = `
        <div style="text-align:center;padding:40px;color:#64748b;grid-column:1/-1;">
          <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="#d1d5db" stroke-width="1.5" style="margin:0 auto 12px;">
            <rect x="3" y="3" width="18" height="18" rx="2"/>
            <circle cx="8.5" cy="8.5" r="1.5"/>
            <path d="M21 15l-5-5L5 21"/>
          </svg>
          <p style="margin:0;">${searchTerm ? 'No matches found' : 'This folder is empty'}</p>
        </div>
      `;
    }

    $content.html(html);

    // Bind folder click
    $content.find('.lsp-dropbox-folder-card').on('click', function() {
      const path = $(this).data('path');
      loadDropboxFolder(path);
    });

    // Bind image click for selection
    $content.find('.lsp-dropbox-image-card').on('click', function(e) {
      const $card = $(this);
      const id = $card.data('id');
      const idx = window.lspDropboxSelectedFiles.indexOf(id);
      
      if (idx > -1) {
        window.lspDropboxSelectedFiles.splice(idx, 1);
        $card.removeClass('selected');
        $card.css('border-color', '#e2e8f0');
        $card.find('.lsp-dropbox-check').html('').css({background: 'rgba(255,255,255,0.9)', borderColor: '#d1d5db'});
      } else {
        window.lspDropboxSelectedFiles.push(id);
        $card.addClass('selected');
        $card.css('border-color', '#0891b2');
        $card.find('.lsp-dropbox-check').html('<svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="3"><path d="M20 6L9 17l-5-5"/></svg>').css({background: '#0891b2', borderColor: '#0891b2'});
      }
      
      updateDropboxCommandBarSelection();
    });

    // Load thumbnails
    loadDropboxThumbnails();
  }

  /**
   * Load thumbnails for visible images
   */
  function loadDropboxThumbnails() {
    const $thumbs = $('.lsp-dropbox-thumb').not('.loaded').not('.loading');
    
    console.log('[LSP Dropbox] Loading thumbnails, found:', $thumbs.length);
    
    if ($thumbs.length === 0) return;
    
    // Load thumbnails in batches to avoid overwhelming the server
    let loaded = 0;
    const maxConcurrent = 4;
    
    $thumbs.each(function() {
      if (loaded >= maxConcurrent) return false; // Break loop
      
      const $thumb = $(this);
      const path = $thumb.data('path');
      
      if (!path) {
        console.log('[LSP Dropbox] Thumb missing path');
        return;
      }
      
      console.log('[LSP Dropbox] Loading thumbnail for:', path);
      
      $thumb.addClass('loading');
      // Add loading spinner
      $thumb.find('.lsp-thumb-placeholder').html('<div class="spinner is-active" style="float:none;margin:0;"></div>');
      loaded++;
      
      $.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
          action: 'lsp_dropbox_get_thumbnail',
          path: path,
          _wpnonce: LIGHTSYNCPRO.nonce
        },
        timeout: 30000
      })
      .done(function(r) {
        console.log('[LSP Dropbox] Thumbnail response for', path, r);
        if (r.success && r.data && r.data.thumbnail) {
          $thumb.addClass('loaded').removeClass('loading');
          $thumb.html('<img src="' + r.data.thumbnail + '" style="width:100%;height:100%;object-fit:cover;">');
        } else {
          console.log('[LSP Dropbox] Thumbnail failed:', r);
          $thumb.addClass('loaded error').removeClass('loading');
          $thumb.find('.lsp-thumb-placeholder').html('<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="#cbd5e1" stroke-width="1.5"><rect x="3" y="3" width="18" height="18" rx="2"/><circle cx="8.5" cy="8.5" r="1.5"/><path d="M21 15l-5-5L5 21"/></svg>');
        }
        
        // Load more thumbnails
        setTimeout(loadDropboxThumbnails, 100);
      })
      .fail(function(xhr, status, error) {
        console.error('[LSP Dropbox] Thumbnail AJAX failed:', status, error, xhr.responseText);
        $thumb.addClass('loaded error').removeClass('loading');
        $thumb.find('.lsp-thumb-placeholder').html('<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="#cbd5e1" stroke-width="1.5"><rect x="3" y="3" width="18" height="18" rx="2"/><circle cx="8.5" cy="8.5" r="1.5"/><path d="M21 15l-5-5L5 21"/></svg>');
        setTimeout(loadDropboxThumbnails, 100);
      });
    });
  }

  /**
   * Update breadcrumb navigation
   */
  function updateDropboxBreadcrumb(path) {
    const $breadcrumb = $('#lsp-dropbox-breadcrumb');
    
    let html = `
      <span class="lsp-dropbox-crumb" data-path="" style="cursor:pointer;color:#0891b2;font-weight:500;">
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="vertical-align:middle;margin-right:4px;">
          <path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/>
          <polyline points="9 22 9 12 15 12 15 22"/>
        </svg>
        Dropbox
      </span>
    `;

    if (path) {
      const parts = path.split('/').filter(Boolean);
      let currentPath = '';
      
      parts.forEach((part, i) => {
        currentPath += '/' + part;
        const isLast = i === parts.length - 1;
        html += `
          <span style="color:#94a3b8;margin:0 4px;">›</span>
          <span class="lsp-dropbox-crumb" data-path="${currentPath}" 
                style="cursor:pointer;color:${isLast ? '#374151' : '#0891b2'};${isLast ? 'font-weight:500;' : ''}">
            ${part}
          </span>
        `;
      });
    }

    $breadcrumb.html(html);

    // Bind breadcrumb clicks
    $breadcrumb.find('.lsp-dropbox-crumb').on('click', function() {
      const crumbPath = $(this).data('path');
      loadDropboxFolder(crumbPath);
    });
  }

  // Dropbox search
  $('#lsp-dropbox-search').on('input', function() {
    renderDropboxContent();
  });

  // Dropbox refresh
  $('#lsp-dropbox-refresh').on('click', function() {
    loadDropboxFolder(window.lspDropboxCurrentPath || '');
  });

  // Select all images
  $('#lsp-dropbox-select-all').on('click', function() {
    window.lspDropboxSelectedFiles = window.lspDropboxImages.map(img => img.id);
    renderDropboxContent();
    updateDropboxCommandBarSelection();
  });

  // Select none
  $('#lsp-dropbox-select-none').on('click', function() {
    window.lspDropboxSelectedFiles = [];
    renderDropboxContent();
    updateDropboxCommandBarSelection();
  });

  // Save Dropbox sync target
  $(document).on('change', 'input[name="lsp_dropbox_sync_target"]', function() {
    $.post(ajaxurl, {
      action: 'lsp_dropbox_save_target',
      target: $(this).val(),
      _wpnonce: LIGHTSYNCPRO.nonce
    });
  });



  // Load Dropbox on page load if already on Dropbox tab
  $(function() {
    if ($('#lsp-dropbox-content').hasClass('active') && !window.lspDropboxLoaded) {
      loadDropboxFolder('');
    }
  });

  // Expose Dropbox functions
  window.loadDropboxFolder = loadDropboxFolder;
  window.renderDropboxContent = renderDropboxContent;
  window.loadDropboxThumbnails = loadDropboxThumbnails;

  /* ==================== END DROPBOX HANDLERS ==================== */

  /* ==================== SHUTTERSTOCK HANDLERS ==================== */

  // Shutterstock state
  window.lspShutterstockLoaded = false;
  window.lspShutterstockImages = [];
  window.lspShutterstockSelectedFiles = [];
  window.lspShutterstockSynced = {};
  window.lspShutterstockPage = 1;
  window.lspShutterstockHasMore = false;

  function loadShutterstockLicenses(page, append) {
    page = page || 1;
    append = append || false;
    
    if (page === 1 && !append) {
      $('#lsp-shutterstock-loading').show();
      $('#lsp-shutterstock-grid').hide();
      $('#lsp-shutterstock-empty').hide();
      $('#lsp-shutterstock-controls').hide();
      $('#lsp-shutterstock-dest-section').hide();
    }

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_shutterstock_get_licenses',
        _wpnonce: LIGHTSYNCPRO.nonce,
        page: page,
        per_page: 20
      },
      success: function(response) {
        $('#lsp-shutterstock-loading').hide();

        if (!response.success) {
          $('#lsp-shutterstock-status').text('Error: ' + (response.data && response.data.error ? response.data.error : 'Unknown error'));
          return;
        }

        var data = response.data;
        
        if (append) {
          window.lspShutterstockImages = window.lspShutterstockImages.concat(data.images);
        } else {
          window.lspShutterstockImages = data.images;
          window.lspShutterstockSynced = data.synced || {};
          window.lspShutterstockDebug = data.debug || {};
        }
        
        window.lspShutterstockPage = data.page;
        window.lspShutterstockHasMore = data.has_more;
        window.lspShutterstockLoaded = true;

        if (window.lspShutterstockImages.length === 0) {
          $('#lsp-shutterstock-empty').show();
          $('#lsp-shutterstock-status').text('No licensed images found');
          $('#lsp-shutterstock-controls').hide();
          $('#lsp-shutterstock-filters').hide();
        } else {
          renderShutterstockGrid();
          $('#lsp-shutterstock-grid').show();
          $('#lsp-shutterstock-controls').css('display', 'flex');
          $('#lsp-shutterstock-dest-section').show();
          $('#lsp-shutterstock-filters').show();
          $('#lsp-shutterstock-total-display').text(data.total_count);
          $('#lsp-shutterstock-status').text(data.total_count + ' licensed image' + (data.total_count !== 1 ? 's' : ''));
        }

        // Show/hide pagination
        if (data.has_more) {
          $('#lsp-shutterstock-pagination').show();
        } else {
          $('#lsp-shutterstock-pagination').hide();
        }
        
        updateShutterstockCommandBarSelection();
      },
      error: function() {
        $('#lsp-shutterstock-loading').hide();
        $('#lsp-shutterstock-status').text('Error loading licenses');
      }
    });
  }

  function renderShutterstockGrid() {
    var $grid = $('#lsp-shutterstock-grid');
    var allImages = window.lspShutterstockImages || [];
    var synced = window.lspShutterstockSynced || {};
    var selected = window.lspShutterstockSelectedFiles || [];
    
    // Get filter values
    var searchTerm = ($('#lsp-shutterstock-search').val() || '').toLowerCase().trim();
    var filterStatus = $('#lsp-shutterstock-filter-status').val() || 'all';
    var sortOrder = $('#lsp-shutterstock-sort').val() || 'newest';
    
    // Filter images
    var images = allImages.filter(function(img) {
      // Search filter
      if (searchTerm) {
        var desc = (img.description || '').toLowerCase();
        var id = String(img.id || '').toLowerCase();
        if (desc.indexOf(searchTerm) === -1 && id.indexOf(searchTerm) === -1) {
          return false;
        }
      }
      
      // Status filter
      var syncData = synced[img.id];
      var isSynced = !!syncData;
      var syncDest = syncData && typeof syncData === 'object' ? (syncData.dest || 'wp') : 'wp';
      
      if (filterStatus === 'unsynced' && isSynced) return false;
      if (filterStatus === 'synced' && !isSynced) return false;
      if (filterStatus === 'wp' && !(isSynced && (syncDest === 'wp' || syncDest === 'both'))) return false;
      if (filterStatus === 'shopify' && !(isSynced && (syncDest === 'shopify' || syncDest === 'both'))) return false;
      
      return true;
    });
    
    // Sort images
    images.sort(function(a, b) {
      if (sortOrder === 'newest') {
        return (b.download_date || '').localeCompare(a.download_date || '');
      } else if (sortOrder === 'oldest') {
        return (a.download_date || '').localeCompare(b.download_date || '');
      } else if (sortOrder === 'az') {
        return (a.description || '').localeCompare(b.description || '');
      } else if (sortOrder === 'za') {
        return (b.description || '').localeCompare(a.description || '');
      }
      return 0;
    });
    
    // Update filter info
    var hasFilters = searchTerm || filterStatus !== 'all';
    if (hasFilters) {
      $('#lsp-shutterstock-filter-info').show();
      $('#lsp-shutterstock-filtered-count').text(images.length);
    } else {
      $('#lsp-shutterstock-filter-info').hide();
    }
    
    // Show/hide no results message
    if (images.length === 0 && allImages.length > 0) {
      $grid.hide();
      $('#lsp-shutterstock-no-results').show();
      $('#lsp-shutterstock-controls').hide();
      return;
    } else {
      $('#lsp-shutterstock-no-results').hide();
      $grid.show();
      $('#lsp-shutterstock-controls').css('display', 'flex');
    }

    // DEBUG: Show what we got
    var debugInfo = 'Images: ' + images.length + ' (filtered from ' + allImages.length + ')';
    if (images.length > 0) {
      var firstImg = images[0];
      debugInfo += ' | First ID: ' + firstImg.id;
      debugInfo += ' | Has thumb: ' + (firstImg.thumbnail ? 'YES (' + firstImg.thumbnail.substring(0, 50) + '...)' : 'NO');
    }
    console.log('[LSP Shutterstock] ' + debugInfo);

    var html = '';
    
    html += '<div class="lsp-shutterstock-browser-content" style="display:grid;grid-template-columns:repeat(auto-fill,minmax(140px,1fr));gap:12px;">';

    images.forEach(function(img) {
      try {
        var isSelected = selected.indexOf(String(img.id)) > -1;
        var syncData = synced[img.id] || null;
        var isSynced = !!syncData;
        var syncedAt = syncData ? (typeof syncData === 'object' ? Number(syncData.time) : Number(syncData)) : 0;
        var syncDest = syncData && typeof syncData === 'object' ? (syncData.dest || 'wp') : 'wp';
        var syncedDate = (isSynced && syncedAt > 0) ? new Date(syncedAt * 1000).toLocaleDateString() : '';
        
        // Check if license was updated (redownloaded) - for Shutterstock we don't track updates yet
        var isUpdated = false;
        
        // Build destination icons
        var destIconsHtml = '';
        if (isSynced) {
          destIconsHtml = '<span style="display:inline-flex;gap:3px;margin-left:4px;vertical-align:middle;">';
          if (syncDest === 'wp' || syncDest === 'both') {
            destIconsHtml += '<span title="Synced to WordPress" style="display:flex;"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><rect x="3" y="3" width="7" height="7" rx="1"/><rect x="14" y="3" width="7" height="7" rx="1"/><rect x="3" y="14" width="7" height="7" rx="1"/><rect x="14" y="14" width="7" height="7" rx="1"/></svg></span>';
          }
          if (syncDest === 'shopify' || syncDest === 'both') {
            destIconsHtml += '<span title="Synced to Shopify" style="display:flex;"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/><line x1="3" y1="6" x2="21" y2="6"/><path d="M16 10a4 4 0 01-8 0"/></svg></span>';
          }
          destIconsHtml += '</span>';
        }
        
        // Build sync badge - Glassmorphism style matching Dropbox
        var syncBadge = '';
        if (isUpdated) {
          syncBadge = '<div class="lsp-sync-badge lsp-sync-update" style="position:absolute;top:6px;left:6px;background:rgba(245,158,11,0.85);backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);color:#fff;font-size:10px;font-weight:500;padding:4px 8px;border-radius:6px;display:flex;align-items:center;gap:2px;box-shadow:0 2px 8px rgba(0,0,0,0.15);border:1px solid rgba(255,255,255,0.2);z-index:2;" title="Updated since last sync"><svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" style="margin-right:2px;"><path d="M23 4v6h-6M1 20v-6h6"/><path d="M3.51 9a9 9 0 0114.85-3.36L23 10M1 14l4.64 4.36A9 9 0 0020.49 15"/></svg>' + destIconsHtml + '</div>';
        } else if (isSynced) {
          syncBadge = '<div class="lsp-sync-badge lsp-sync-done" style="position:absolute;top:6px;left:6px;background:rgba(16,185,129,0.85);backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);color:#fff;font-size:10px;font-weight:500;padding:4px 8px;border-radius:6px;display:flex;align-items:center;gap:2px;box-shadow:0 2px 8px rgba(0,0,0,0.15);border:1px solid rgba(255,255,255,0.2);z-index:2;" title="Synced ' + syncedDate + '"><svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" style="margin-right:2px;"><path d="M20 6L9 17l-5-5"/></svg>' + destIconsHtml + '</div>';
        }
        
        // Selection checkbox
        var checkHtml = '<div class="lsp-shutterstock-check" style="position:absolute;top:8px;right:8px;width:22px;height:22px;border-radius:50%;border:2px solid ' + (isSelected ? '#ee2d24' : '#d1d5db') + ';background:' + (isSelected ? '#ee2d24' : 'rgba(255,255,255,0.9)') + ';display:flex;align-items:center;justify-content:center;transition:all 0.15s;z-index:3;">';
        if (isSelected) {
          checkHtml += '<svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="3"><path d="M20 6L9 17l-5-5"/></svg>';
        }
        checkHtml += '</div>';
        
        // Border color based on state
        var borderColor = '#e2e8f0';
        if (isSelected) borderColor = '#ee2d24';
        else if (isUpdated) borderColor = '#f59e0b';
        
        html += '<div class="lsp-shutterstock-image-card ' + (isSelected ? 'selected' : '') + '" ' +
               'data-id="' + img.id + '" data-license-id="' + img.license_id + '" data-synced="' + (isSynced ? '1' : '0') + '" ' +
               'data-description="' + (img.description || '').replace(/"/g, '&quot;') + '" ' +
               'style="position:relative;background:#fff;border:2px solid ' + borderColor + ';border-radius:10px;overflow:hidden;cursor:pointer;transition:all 0.15s;">';
        html += syncBadge;
        html += checkHtml;
        html += '<div class="lsp-shutterstock-thumb" style="aspect-ratio:1;background:#f1f5f9;display:flex;align-items:center;justify-content:center;overflow:hidden;">';
        if (img.thumbnail) {
          console.log('[LSP Shutterstock] Image', img.id, 'thumbnail:', img.thumbnail.substring(0, 80));
          html += '<img src="' + img.thumbnail + '" alt="" style="width:100%;height:100%;object-fit:cover;" loading="lazy" onerror="console.error(\'[LSP] Image load error:\', this.src);this.style.display=\'none\';this.parentNode.innerHTML=\'<svg width=32 height=32 viewBox=0 0 24 24 fill=none stroke=#cbd5e1 stroke-width=1.5><rect x=3 y=3 width=18 height=18 rx=2/><circle cx=8.5 cy=8.5 r=1.5/><path d=M21 15l-5-5L5 21/></svg>\'">';
        } else {
          console.log('[LSP Shutterstock] Image', img.id, 'has NO thumbnail');
          html += '<svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="#cbd5e1" stroke-width="1.5"><rect x="3" y="3" width="18" height="18" rx="2"/><circle cx="8.5" cy="8.5" r="1.5"/><path d="M21 15l-5-5L5 21"/></svg>';
        }
        html += '</div>';
        html += '<div style="padding:8px;font-size:11px;color:#64748b;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;" title="' + (img.description || '').replace(/"/g, '&quot;') + '">' + (img.description || 'Image ' + img.id).substring(0, 30) + '</div>';
        html += '</div>';
      } catch (e) {
        console.error('[LSP Shutterstock] Error rendering image:', img, e);
      }
    });

    html += '</div>';
    $grid.html(html);

    // Bind image click for selection
    $grid.find('.lsp-shutterstock-image-card').on('click', function() {
      var $card = $(this);
      var id = String($card.data('id'));
      var idx = window.lspShutterstockSelectedFiles.indexOf(id);
      
      if (idx > -1) {
        window.lspShutterstockSelectedFiles.splice(idx, 1);
        $card.removeClass('selected');
        $card.css('border-color', '#e2e8f0');
        $card.find('.lsp-shutterstock-check').html('').css({background: 'rgba(255,255,255,0.9)', borderColor: '#d1d5db'});
      } else {
        window.lspShutterstockSelectedFiles.push(id);
        $card.addClass('selected');
        $card.css('border-color', '#ee2d24');
        $card.find('.lsp-shutterstock-check').html('<svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="3"><path d="M20 6L9 17l-5-5"/></svg>').css({background: '#ee2d24', borderColor: '#ee2d24'});
      }
      
      updateShutterstockCommandBarSelection();
    });
  }

  // Update the command bar selection count for Shutterstock
  function updateShutterstockCommandBarSelection() {
    var count = window.lspShutterstockSelectedFiles ? window.lspShutterstockSelectedFiles.length : 0;
    
    // Update command bar pill
    var $commandbarEl = $('#lsp-shutterstock-commandbar-selection .lsp-selection-text');
    if ($commandbarEl.length) {
      $commandbarEl.text(count + ' image' + (count !== 1 ? 's' : '') + ' selected');
    }
    
    // Update inline selection count below grid
    var $selectionCount = $('#lsp-shutterstock-selection-count');
    if ($selectionCount.length) {
      if (count > 0) {
        $selectionCount.text(count + ' image' + (count !== 1 ? 's' : '') + ' selected');
      } else {
        $selectionCount.text('');
      }
    }
  }

  // Select All (only visible/filtered images that aren't synced)
  $(document).on('click', '#lsp-shutterstock-select-all', function() {
    var synced = window.lspShutterstockSynced || {};
    var allImages = window.lspShutterstockImages || [];
    
    // Get current filter values
    var searchTerm = ($('#lsp-shutterstock-search').val() || '').toLowerCase().trim();
    var filterStatus = $('#lsp-shutterstock-filter-status').val() || 'all';
    
    // Filter images same as renderShutterstockGrid
    var filteredImages = allImages.filter(function(img) {
      if (searchTerm) {
        var desc = (img.description || '').toLowerCase();
        var id = String(img.id || '').toLowerCase();
        if (desc.indexOf(searchTerm) === -1 && id.indexOf(searchTerm) === -1) {
          return false;
        }
      }
      var syncData = synced[img.id];
      var isSynced = !!syncData;
      var syncDest = syncData && typeof syncData === 'object' ? (syncData.dest || 'wp') : 'wp';
      if (filterStatus === 'unsynced' && isSynced) return false;
      if (filterStatus === 'synced' && !isSynced) return false;
      if (filterStatus === 'wp' && !(isSynced && (syncDest === 'wp' || syncDest === 'both'))) return false;
      if (filterStatus === 'shopify' && !(isSynced && (syncDest === 'shopify' || syncDest === 'both'))) return false;
      return true;
    });
    
    // Select all filtered images that aren't synced
    window.lspShutterstockSelectedFiles = filteredImages
      .filter(function(img) { return !synced[img.id]; })
      .map(function(img) { return String(img.id); });
    renderShutterstockGrid();
    updateShutterstockCommandBarSelection();
  });

  // Select None
  $(document).on('click', '#lsp-shutterstock-select-none', function() {
    window.lspShutterstockSelectedFiles = [];
    renderShutterstockGrid();
    updateShutterstockCommandBarSelection();
  });

  // Refresh
  $(document).on('click', '#lsp-shutterstock-refresh', function() {
    window.lspShutterstockSelectedFiles = [];
    // Reset filters
    $('#lsp-shutterstock-search').val('');
    $('#lsp-shutterstock-filter-status').val('all');
    $('#lsp-shutterstock-sort').val('newest');
    loadShutterstockLicenses(1);
  });

  // Search with debounce
  var shutterstockSearchTimer = null;
  $(document).on('input', '#lsp-shutterstock-search', function() {
    clearTimeout(shutterstockSearchTimer);
    shutterstockSearchTimer = setTimeout(function() {
      renderShutterstockGrid();
    }, 200);
  });

  // Filter and sort change
  $(document).on('change', '#lsp-shutterstock-filter-status, #lsp-shutterstock-sort', function() {
    renderShutterstockGrid();
  });

  // Clear filters
  $(document).on('click', '#lsp-shutterstock-clear-filters', function() {
    $('#lsp-shutterstock-search').val('');
    $('#lsp-shutterstock-filter-status').val('all');
    $('#lsp-shutterstock-sort').val('newest');
    renderShutterstockGrid();
  });

  // Load More
  $(document).on('click', '#lsp-shutterstock-load-more', function() {
    var $btn = $(this);
    $btn.prop('disabled', true).text('Loading...');
    loadShutterstockLicenses(window.lspShutterstockPage + 1, true);
    $btn.prop('disabled', false).text('Load More');
  });

  // Auto-load Shutterstock if starting on Shutterstock tab
  $(function(){
    if ($('#lsp-shutterstock-content.active').length && !window.lspShutterstockLoaded) {
      loadShutterstockLicenses();
    }
  });

  // Load when switching to Shutterstock tab
  $(document).on('click', '.lsp-source-tab[data-source="shutterstock"]', function() {
    setTimeout(function() {
      if (!window.lspShutterstockLoaded) {
        loadShutterstockLicenses();
      }
    }, 100);
  });

  // Expose functions
  window.loadShutterstockLicenses = loadShutterstockLicenses;
  window.renderShutterstockGrid = renderShutterstockGrid;
  window.updateShutterstockCommandBarSelection = updateShutterstockCommandBarSelection;

  /* ==================== END SHUTTERSTOCK HANDLERS ==================== */

  // Save Shutterstock sync target
  $('input[name="lsp_shutterstock_sync_target"]').on('change', function() {
    const target = $(this).val();
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_shutterstock_save_target',
        _wpnonce: LIGHTSYNCPRO.nonce,
        target: target
      }
    });
  });

  // Destination card selection
  $(document).on('click', '.lsp-dest-card:not(.disabled)', function(){
    var $card = $(this);
    var $container = $card.closest('.lsp-dest-cards');
    var $radio = $card.find('input[type="radio"]');
    var radioValue = $radio.val();
    
    // If Hub is selected, show site selector modal first
    if (radioValue === 'hub') {
      // Determine which source this is for
      var source = 'lightroom';
      if ($card.attr('id')) {
        if ($card.attr('id').indexOf('canva') !== -1) source = 'canva';
        else if ($card.attr('id').indexOf('figma') !== -1) source = 'figma';
        else if ($card.attr('id').indexOf('dropbox') !== -1) source = 'dropbox';
      }
      
      // Show Hub site selector modal
      showHubSiteSelectorModal(source, function(selectedSites) {
        if (selectedSites && selectedSites.length > 0) {
          // Store selected sites
          window.lspHubSelectedSites = selectedSites;
          
          // Now select the Hub card
          $container.find('.lsp-dest-card').removeClass('selected');
          $card.addClass('selected');
          $radio.prop('checked', true).trigger('change');
          
          // Update subtitle to show selected count
          $card.find('.lsp-dest-sub').text(selectedSites.length + ' site' + (selectedSites.length !== 1 ? 's' : '') + ' selected');
        }
        // If cancelled, don't select Hub
      });
      return;
    }
    
    // Update visual selection
    $container.find('.lsp-dest-card').removeClass('selected');
    $card.addClass('selected');
    
    // Check the radio
    $radio.prop('checked', true).trigger('change');
  });
  
  // Hub Site Selector Modal
  function showHubSiteSelectorModal(source, callback) {
    // Remove existing modal
    $('#lsp-hub-site-modal').remove();
    
    // Get Hub sites from lspData
    var hubSites = (lspData && lspData.hub && lspData.hub.sites) || [];
    
    if (hubSites.length === 0) {
      alert('No Hub sites connected. Add sites in the Hub tab first.');
      callback(null);
      return;
    }
    
    // Build site list HTML
    var sitesHtml = '';
    hubSites.forEach(function(site) {
      var icon = '🌐';
      sitesHtml += '<label class="lsp-hub-site-item" style="display:flex;align-items:center;padding:12px 16px;border:1px solid #e2e8f0;border-radius:10px;margin-bottom:8px;cursor:pointer;transition:all 0.15s;">' +
        '<input type="checkbox" value="' + site.id + '" checked style="margin-right:12px;width:18px;height:18px;accent-color:#ef4444;">' +
        '<span style="font-size:18px;margin-right:10px;">' + icon + '</span>' +
        '<span style="flex:1;">' +
          '<strong style="display:block;font-size:14px;color:#1e293b;">' + (site.site_name || site.site_url) + '</strong>' +
          '<span style="font-size:12px;color:#64748b;">' + site.site_url + '</span>' +
        '</span>' +
      '</label>';
    });
    
    var modalHtml = 
      '<div id="lsp-hub-site-modal" style="position:fixed;inset:0;z-index:100001;display:flex;align-items:center;justify-content:center;background:rgba(15,23,42,.6);backdrop-filter:blur(4px);">' +
        '<div style="background:#fff;border-radius:20px;padding:0;max-width:480px;width:90%;max-height:80vh;overflow:hidden;box-shadow:0 25px 60px rgba(0,0,0,.3);">' +
          '<div style="padding:24px 28px 16px;border-bottom:1px solid #e2e8f0;">' +
            '<h3 style="margin:0 0 4px;font-size:18px;font-weight:600;color:#1e293b;">Select Hub Destinations</h3>' +
            '<p style="margin:0;font-size:14px;color:#64748b;">Choose which sites to distribute to</p>' +
          '</div>' +
          '<div style="padding:20px 28px;max-height:400px;overflow-y:auto;">' +
            '<div style="margin-bottom:12px;">' +
              '<label style="display:flex;align-items:center;cursor:pointer;">' +
                '<input type="checkbox" id="lsp-hub-select-all" checked style="margin-right:8px;width:16px;height:16px;accent-color:#ef4444;">' +
                '<span style="font-size:13px;font-weight:500;color:#64748b;">Select All</span>' +
              '</label>' +
            '</div>' +
            '<div id="lsp-hub-sites-list">' + sitesHtml + '</div>' +
          '</div>' +
          '<div style="padding:16px 28px 24px;border-top:1px solid #e2e8f0;display:flex;justify-content:space-between;align-items:center;">' +
            '<span id="lsp-hub-selected-count" style="font-size:13px;color:#64748b;">' + hubSites.length + ' selected</span>' +
            '<div>' +
              '<button type="button" id="lsp-hub-cancel" class="button" style="margin-right:8px;">Cancel</button>' +
              '<button type="button" id="lsp-hub-confirm" class="button button-primary" style="background:#ef4444;border-color:#ef4444;">Confirm</button>' +
            '</div>' +
          '</div>' +
        '</div>' +
      '</div>';
    
    $('body').append(modalHtml);
    
    // Select All toggle
    $('#lsp-hub-select-all').on('change', function() {
      $('#lsp-hub-sites-list input[type="checkbox"]').prop('checked', $(this).prop('checked'));
      updateHubSelectedCount();
    });
    
    // Individual checkbox change
    $('#lsp-hub-sites-list').on('change', 'input[type="checkbox"]', function() {
      updateHubSelectedCount();
    });
    
    function updateHubSelectedCount() {
      var total = $('#lsp-hub-sites-list input[type="checkbox"]').length;
      var checked = $('#lsp-hub-sites-list input[type="checkbox"]:checked').length;
      $('#lsp-hub-selected-count').text(checked + ' selected');
      $('#lsp-hub-select-all').prop('checked', checked === total);
    }
    
    // Cancel button
    $('#lsp-hub-cancel').on('click', function() {
      $('#lsp-hub-site-modal').remove();
      callback(null);
    });
    
    // Click outside to cancel
    $('#lsp-hub-site-modal').on('click', function(e) {
      if (e.target === this) {
        $(this).remove();
        callback(null);
      }
    });
    
    // Confirm button
    $('#lsp-hub-confirm').on('click', function() {
      var selected = [];
      $('#lsp-hub-sites-list input[type="checkbox"]:checked').each(function() {
        selected.push($(this).val());
      });
      
      if (selected.length === 0) {
        alert('Please select at least one destination site.');
        return;
      }
      
      $('#lsp-hub-site-modal').remove();
      callback(selected);
    });
  }



  // ============================================
  // HELPER TOGGLE
  // ============================================
  $('#lsp-helpers-toggle').on('click', function() {
    var $btn = $(this);
    var isHidden = $btn.attr('data-hidden') === '1';
    var newState = !isHidden;
    
    // Toggle class on wrap
    if (newState) {
      $('.wrap.lightsyncpro').addClass('lsp-helpers-hidden');
    } else {
      $('.wrap.lightsyncpro').removeClass('lsp-helpers-hidden');
    }
    
    // Update button state
    $btn.attr('data-hidden', newState ? '1' : '0');
    $btn.find('.lsp-helpers-label').text(newState ? 'Show Helpers' : 'Hide Helpers');
    
    // Save preference
    $.post(LIGHTSYNCPRO.ajaxurl, {
      action: 'lsp_toggle_helpers',
      _wpnonce: LIGHTSYNCPRO.nonce,
      hidden: newState ? 1 : 0
    });
  });
  
  // Apply initial state
  if ($('#lsp-helpers-toggle').attr('data-hidden') === '1') {
    $('.wrap.lightsyncpro').addClass('lsp-helpers-hidden');
  }

  // ============================================
  // GUIDED TOUR
  // ============================================
  var tourSteps = [
    {
      target: '#lsp-sources',
      title: 'Source Tabs',
      content: 'Switch between Lightroom, Canva, Figma, and Dropbox. Each source has its own connection settings and sync options.'
    },
    {
      target: '.lsp-dest-cards',
      title: 'Sync Destination',
      content: 'Your images sync directly to the WordPress Media Library. The destination icons on synced items show where they\'ve been synced.'
    },
    {
      target: '.lsp-sync-badge, .lsp-canva-synced-badge, .lsp-figma-frame-card.synced',
      title: 'Sync Status Badges',
      content: 'Green ✓ = synced and up to date. Orange ⟳ = source file has changed since last sync. Click to re-sync and get the latest version.'
    },
    {
      target: 'aside.help',
      title: 'Helper Sidebars',
      content: 'Each section has a helper sidebar with tips and the current sync destination. You can hide these anytime using the "Hide Helpers" button above.'
    },
        {
      target: '#lsp-avif-section, #lsp-compression-settings',
      title: 'Image Optimization',
      content: 'Convert images to modern formats like AVIF and WebP for faster page loads. Compression settings apply to all synced images automatically.'
    }
  ];
  
  var currentStep = 0;
  var $overlay = null;
  var $popup = null;
  
  function findTourTarget(selector) {
    // Try multiple selectors (comma-separated fallbacks)
    var selectors = selector.split(',').map(function(s) { return s.trim(); });
    for (var i = 0; i < selectors.length; i++) {
      var $el = $(selectors[i]).filter(':visible').first();
      if ($el.length) return $el;
    }
    return null;
  }
  
  function showTourStep(index) {
    if (index < 0 || index >= tourSteps.length) {
      endTour();
      return;
    }
    
    currentStep = index;
    var step = tourSteps[index];
    var $target = findTourTarget(step.target);
    
    // Create overlay if needed
    if (!$overlay) {
      $overlay = $('<div class="lsp-tour-overlay"></div>').appendTo('body');
    }
    
    // Remove previous highlight
    $('.lsp-tour-highlight').removeClass('lsp-tour-highlight');
    
    // Highlight target if found
    if ($target && $target.length) {
      $target.addClass('lsp-tour-highlight');
      
      // Scroll into view
      $('html, body').animate({
        scrollTop: $target.offset().top - 100
      }, 300);
    }
    
    // Build progress dots
    var progressHtml = '<div class="lsp-tour-progress">';
    for (var i = 0; i < tourSteps.length; i++) {
      var cls = i < index ? 'done' : (i === index ? 'active' : '');
      progressHtml += '<span class="' + cls + '"></span>';
    }
    progressHtml += '</div>';
    
    // Build popup
    var popupHtml = 
      '<div class="lsp-tour-popup">' +
        progressHtml +
        '<h4><span class="step-badge">' + (index + 1) + '/' + tourSteps.length + '</span> ' + step.title + '</h4>' +
        '<p>' + step.content + '</p>' +
        '<div class="lsp-tour-actions">' +
          '<button type="button" class="lsp-tour-skip">Skip Tour</button>' +
          (index > 0 ? '<button type="button" class="lsp-tour-prev">← Back</button>' : '') +
          '<button type="button" class="lsp-tour-next">' + (index === tourSteps.length - 1 ? 'Finish' : 'Next →') + '</button>' +
        '</div>' +
      '</div>';
    
    // Remove old popup
    if ($popup) $popup.remove();
    
    // Add new popup
    $popup = $(popupHtml).appendTo('body');
    
    // Bind events
    $popup.find('.lsp-tour-skip').on('click', endTour);
    $popup.find('.lsp-tour-prev').on('click', function() { showTourStep(currentStep - 1); });
    $popup.find('.lsp-tour-next').on('click', function() { showTourStep(currentStep + 1); });
    
    // Always center popup on screen (fixed position)
    $popup.css({
      left: '50%',
      top: '50%',
      transform: 'translate(-50%, -50%)',
      position: 'fixed'
    });
  }
  
  function endTour() {
    // Remove UI elements
    if ($overlay) { $overlay.remove(); $overlay = null; }
    if ($popup) { $popup.remove(); $popup = null; }
    $('.lsp-tour-highlight').removeClass('lsp-tour-highlight');
    
    // Mark tour as completed
    $.post(LIGHTSYNCPRO.ajaxurl, {
      action: 'lsp_tour_complete',
      _wpnonce: LIGHTSYNCPRO.nonce
    });
  }
  
  // Start tour button
  $('#lsp-tour-btn').on('click', function() {
    currentStep = 0;
    showTourStep(0);
  });
  
  // Close tour on Escape
  $(document).on('keydown', function(e) {
    if (e.key === 'Escape' && $overlay) {
      endTour();
    }
  });

  // ========== Mobile Bottom Nav ==========
  $(function() {
    const $mobileNav = $('#lsp-mobile-nav');
    const $moreBtn = $('#lsp-mobile-more-btn');
    const $moreMenu = $('#lsp-mobile-more-menu');
    
    if (!$mobileNav.length) return;
    
    // Toggle more menu
    $moreBtn.on('click', function(e) {
      e.stopPropagation();
      $moreMenu.toggleClass('open');
    });
    
    // Close more menu when clicking outside
    $(document).on('click', function(e) {
      if (!$(e.target).closest('.lsp-mobile-nav-more, .lsp-mobile-more-menu').length) {
        $moreMenu.removeClass('open');
      }
    });
    
    // Close more menu when clicking a link inside it
    $moreMenu.on('click', 'a', function() {
      $moreMenu.removeClass('open');
    });
    
    // Update active state on scroll
    function updateMobileNavActive() {
      const scrollPos = $(window).scrollTop() + 150;
      const sections = {
        'sources': $('#lsp-sources'),
        'pick': $('#lsp-pick, #lsp-canva-pick, #lsp-figma-pick, #lsp-dropbox-pick').filter(':visible').first(),
        'settings': $('#lsp-avif, #lsp-naming').filter(':visible').first(),
        'ai': $('#lsp-ai')
      };
      
      let activeSection = 'sources';
      
      for (const [key, $section] of Object.entries(sections)) {
        if ($section.length && $section.offset().top <= scrollPos) {
          activeSection = key;
        }
      }
      
      $('.lsp-mobile-nav-item').removeClass('active');
      $(`.lsp-mobile-nav-item[data-section="${activeSection}"]`).addClass('active');
    }
    
    $(window).on('scroll', updateMobileNavActive);
    updateMobileNavActive();
    
    // Handle nav item clicks
    $('.lsp-mobile-nav-item[href]').on('click', function() {
      $('.lsp-mobile-nav-item').removeClass('active');
      $(this).addClass('active');
    });
  });

})(jQuery);

/* ==================== AI GENERATE MODULE ==================== */
var lspAI = (function($) {
  'use strict';

  // State
  var state = {
    models: [],
    modelsLoaded: false,
    currentPreview: null,   // { base64, mime, prompt, model, is_free, aspect_ratio }
    generating: false,
    committing: false,
    browsePage: 1,
    browseTotal: 0
  };

  window.lspAiLoaded = false;
  window.lspAiCount = 0;

  // Reset flags on page load — handles bfcache restoration after window.location.reload()
  // Without this, bfcache preserves lspAiBound=true from before reload,
  // so init() skips bindEvents() and the fresh DOM has no event handlers.
  window.addEventListener('pageshow', function(e) {
    if (e.persisted) {
      console.log('[LightSync AI] Page restored from bfcache, resetting state');
      window.lspAiBound = false;
      window.lspAiLoaded = false;
      state.modelsLoaded = false;
      state.modelsLoading = false;
      // Re-init if AI tab is active
      if (document.querySelector('#lsp-ai-content.active')) {
        init();
      }
    }
  });

  // Expose for admin-sync.js to use with Sync Now button
  window.lspAiState = state;
  window.lspAiLoadBrowse = function(page) { loadBrowse(page || state.browsePage || 1); };

  /* ---------- Public init ---------- */
  function init() {
    console.log('[LightSync AI] init() called, bound=' + !!window.lspAiBound + ', modelsLoaded=' + state.modelsLoaded);
    try {
      if (!window.lspAiBound) {
        window.lspAiBound = true;
        bindEvents();
      }
      // Always load models if they haven't successfully loaded
      if (!state.modelsLoaded) {
        loadModels();
      }
      // Always refresh browse on tab visit (cheap query, ensures fresh data)
      loadBrowse(state.browsePage || 1);
      window.lspAiLoaded = true;
    } catch(e) {
      console.error('[LightSync AI] init() error:', e);
    }
  }

  /* ---------- Event binding ---------- */
  function bindEvents() {
    // Generate
    $('#lsp-ai-generate-btn').on('click', doGenerate);

    // Commit
    $('#lsp-ai-commit-btn').on('click', doCommit);

    // Regenerate (from preview — same prompt, new result)
    $('#lsp-ai-regenerate-btn').on('click', doGenerate);

    // Discard preview
    $('#lsp-ai-discard-btn').on('click', discardPreview);

    // Generate another (after commit)
    $('#lsp-ai-new-btn').on('click', resetToForm);

    // Disconnect
    $('#lsp-ai-disconnect-btn').on('click', doDisconnect);

    // Model change → update cost badge
    $(document).on('change', '#lsp-ai-model', updateCostBadge);

    // Enter key in prompt → generate
    $('#lsp-ai-prompt').on('keydown', function(e) {
      if (e.key === 'Enter' && !e.shiftKey) {
        e.preventDefault();
        doGenerate();
      }
    });

    // AI destination card clicks
    $(document).on('click', '.lsp-dest-card[data-ai-dest]', function() {
      if ($(this).hasClass('disabled')) return;
      $('.lsp-dest-card[data-ai-dest]').removeClass('selected');
      $(this).addClass('selected');
      $(this).find('input[type=radio]').prop('checked', true);
      updateCommitButtonText();
    });

    // AI grid: Select All
    $('#lsp-ai-select-all').on('click', function() {
      state.aiSelectedIds = state.aiBrowseItems.map(function(item) { return item.id; });
      $('.lsp-ai-thumb').addClass('selected');
      updateAiSelectionUI();
    });

    // AI grid: Select None
    $('#lsp-ai-select-none').on('click', function() {
      state.aiSelectedIds = [];
      $('.lsp-ai-thumb').removeClass('selected');
      updateAiSelectionUI();
    });
  }

  function getSelectedDest() {
    return $('input[name="lsp_ai_dest"]:checked').val() || 'wp';
  }

  function updateCommitButtonText() {
    var dest = getSelectedDest();
    var $btn = $('#lsp-ai-commit-btn');
    if (dest === 'both') {
      $btn.text('✓ Use This — Save to WP + Shopify');
    } else if (dest === 'shopify') {
      $btn.text('✓ Use This — Save to Shopify');
    } else {
      $btn.text('✓ Use This — Save to Media Library');
    }
  }

  /* ========================================================================
   * Models
   * ======================================================================== */

  /* ---------- Model Loading Retry UI ---------- */
  function showRetryButton(reason) {
    var $container = $('#lsp-ai-generate-form');
    if (!$container.length || $('#lsp-ai-retry-bar').length) return;
    $container.prepend(
      '<div id="lsp-ai-retry-bar" style="display:flex;align-items:center;gap:10px;padding:10px 14px;margin-bottom:14px;background:rgba(245,158,11,.08);border:1px solid rgba(245,158,11,.2);border-radius:10px">' +
        '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#f59e0b" stroke-width="2"><circle cx="12" cy="12" r="10"/><path d="M12 8v4"/><path d="M12 16h.01"/></svg>' +
        '<span style="font-size:12px;color:#92400e;flex:1">' + (reason || 'Models failed to load') + '</span>' +
        '<button type="button" id="lsp-ai-retry-btn" class="btn ghost" style="font-size:11px;padding:4px 12px;white-space:nowrap">' +
          '<svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" style="margin-right:4px"><path d="M21 2v6h-6"/><path d="M3 12a9 9 0 0 1 15-6.7L21 8"/></svg>' +
          'Retry' +
        '</button>' +
      '</div>'
    );
    $('#lsp-ai-retry-btn').on('click', function() {
      hideRetryButton();
      state.modelsLoaded = false;
      state.modelsLoading = false;
      loadModels(true); // force refresh = clear server cache
    });
  }

  function hideRetryButton() {
    $('#lsp-ai-retry-bar').remove();
  }

  function loadModels(forceRefresh) {
    console.log('[LightSync AI] loadModels() called, loading=' + state.modelsLoading + ', connected=' + LIGHTSYNCPRO.openrouter_connected);
    if (state.modelsLoading) return; // Prevent duplicate calls
    var $sel = $('#lsp-ai-model');
    console.log('[LightSync AI] #lsp-ai-model found=' + $sel.length);
    if (!$sel.length) return; // Element not in DOM yet
    $sel.html('<option value="">Loading models...</option>');

    // If OpenRouter not connected, skip fetching models
    if (!LIGHTSYNCPRO.openrouter_connected) {
      $sel.html('<option value="">No providers connected</option>');
      updateCostBadge();
      return;
    }

    state.modelsLoading = true;

    // Show retry button after 10 seconds if still loading
    var retryTimer = setTimeout(function() {
      if (state.modelsLoading) {
        $sel.html('<option value="">Still loading models...</option>');
        showRetryButton('Models taking longer than expected');
      }
    }, 10000);

    var postData = {
      action: 'lsp_ai_get_models',
      _ajax_nonce: LIGHTSYNCPRO.nonce
    };
    if (forceRefresh) postData.force_refresh = 1;

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      timeout: 45000,
      data: postData,
      success: function(resp) {
        clearTimeout(retryTimer);
        state.modelsLoading = false;
        hideRetryButton();
        try {
        console.log('[LightSync] Models response:', resp);
        if (!resp.success) {
          var errMsg = (resp.data && resp.data.error) || 'Failed to load models';
          console.error('[LightSync] Models error:', errMsg);
          $sel.html('<option value="">' + errMsg + '</option>');
          showRetryButton(errMsg);
          return;
        }

        var models = resp.data.models || resp.data || [];
        console.log('[LightSync] Raw models count:', models.length);
        // Filter to image-capable models only (this is the image generation UI)
        models = models.filter(function(m) {
          return m.capability === 'image';
        });
        console.log('[LightSync] Image models count:', models.length);
        state.models = models;
        state.modelsLoaded = true;

        // Sort by estimated cost (cheapest first), then by name
        models.sort(function(a, b) {
          var aCost = estimateCostRaw(a);
          var bCost = estimateCostRaw(b);
          if (aCost !== bCost) return aCost - bCost;
          return (a.name || a.id).localeCompare(b.name || b.id);
        });

        var html = '';
        var currentTier = '';

        models.forEach(function(m) {
          var free = isFreeModel(m);
          var raw = estimateCostRaw(m);
          var tier = '';
          if (free) tier = 'free';
          else if (raw < 0.01) tier = 'budget';
          else if (raw < 0.05) tier = 'standard';
          else tier = 'premium';

          var tierLabels = {
            'free':    '✓ Free Models',
            'budget':  '💰 Budget — Under $0.01/image',
            'standard': '💰 Standard — Under $0.05/image',
            'premium': '💎 Premium — $0.05+/image'
          };

          if (tier !== currentTier) {
            if (currentTier) html += '</optgroup>';
            html += '<optgroup label="' + tierLabels[tier] + '">';
            currentTier = tier;
          }

          var label = m.name || m.id;
          var costLabel = '';
          if (free) {
            // Only append if name doesn't already indicate free
            if (label.indexOf('Free') === -1 && label.indexOf('free') === -1) {
              label += ' ✓ Free';
            }
            costLabel = 'Free';
          } else {
            costLabel = estimateCost(m);
            if (costLabel) label += '  (' + costLabel + ')';
          }

          html += '<option value="' + esc(m.id) + '" data-free="' + (free ? '1' : '0') + '" data-provider="openrouter" data-cost="' + esc(costLabel) + '">' + esc(label) + '</option>';
        });

        if (currentTier) html += '</optgroup>';

        if (!html) {
          html = '<option value="">No image models available</option>';
        }

        $sel.html(html);

        // Auto-select best default: prefer free model if available, otherwise first (cheapest)
        var freeOpt = $sel.find('option[data-free="1"]').first();
        if (freeOpt.length) {
          $sel.val(freeOpt.val());
        }
        // First option with a value is already selected by default (cheapest due to sort)

        updateCostBadge();
        } catch(e) {
          console.error('[LightSync] Model load error:', e);
          $sel.html('<option value="">Error loading models — check console</option>');
          showRetryButton('Error processing model list');
        }
      },
      error: function(xhr, status) {
        clearTimeout(retryTimer);
        state.modelsLoading = false;
        console.error('[LightSync] Models AJAX error:', status, xhr.status, xhr.responseText);
        var msg = status === 'timeout' ? 'Model list timed out' : 'Failed to load models (HTTP ' + xhr.status + ')';
        $sel.html('<option value="">' + msg + '</option>');
        showRetryButton(msg);
      }
    });
  }

  function isFreeModel(m) {
    // OpenRouter convention: truly free endpoints have ':free' suffix
    if (typeof m.id === 'string' && m.id.indexOf(':free') !== -1) return true;
    // Explicit flag from broker
    if (m.is_free) return true;
    // Check pricing — free if ALL pricing fields are 0 or negative
    if (m.pricing) {
      var prompt = parseFloat(m.pricing.prompt || 0);
      var completion = parseFloat(m.pricing.completion || 0);
      var image = parseFloat(m.pricing.image || 0);
      return prompt <= 0 && completion <= 0 && image <= 0;
    }
    return false;
  }

  function estimateCostRaw(m) {
    if (isFreeModel(m)) return 0;
    if (!m.pricing) return 0;

    var prompt = parseFloat(m.pricing.prompt || 0);
    var completion = parseFloat(m.pricing.completion || 0);
    var image = parseFloat(m.pricing.image || 0);

    // Clamp negatives (some promotional models have negative pricing)
    if (prompt < 0) prompt = 0;
    if (completion < 0) completion = 0;
    if (image < 0) image = 0;

    // Per-image pricing (flat rate per image generation)
    if (image > 0) {
      // OpenRouter image pricing: if > 1, it's per-million-tokens format
      if (image > 1) {
        return (image / 1000000) * 1290;
      }
      return image;
    }

    // Token-based estimate: ~20 input tokens (lean prompt) + ~1290 output tokens (image)
    // OpenRouter pricing is per-token (tiny decimals like 0.000005)
    var est = (prompt * 20) + (completion * 1290);

    // Sanity check: if estimate > $1/image, pricing is likely per-million format
    if (est > 1 && (prompt > 0.001 || completion > 0.001)) {
      est = (prompt / 1000000 * 20) + (completion / 1000000 * 1290);
    }

    return est;
  }

  function estimateCost(m) {
    // Prefer broker's pre-formatted label (but not if it says Free for a paid model)
    if (m.cost_label && m.cost_label !== 'Free' && !isFreeModel(m)) return m.cost_label;
    if (isFreeModel(m)) return '';
    var est = estimateCostRaw(m);
    if (est === 0) return '';
    if (est < 0.001) return '< $0.001/image';
    if (est < 0.01) return '~$' + est.toFixed(4).replace(/0+$/, '') + '/image';
    if (est < 1) return '~$' + est.toFixed(3).replace(/0+$/, '') + '/image';
    return '~$' + est.toFixed(2) + '/image';
  }

  var CREDITS_MSG = 'OpenRouter credits required. Add credits at openrouter.ai/credits to start generating — images cost as little as $0.01 each.';
  var CREDITS_HTML = 'OpenRouter credits required to generate images.<br><a href="https://openrouter.ai/credits" target="_blank" style="color:var(--lsp-primary, #ff5757);font-weight:700;text-decoration:underline">Add credits at OpenRouter →</a><br><span style="opacity:.7">Images cost as little as $0.01 each. No subscriptions.</span>';

  function parseAjaxError(xhr, textStatus) {
    var serverMsg = '';
    var serverCode = '';
    try {
      var json = JSON.parse(xhr.responseText);
      serverMsg = (json.data && json.data.error) || '';
      serverCode = (json.data && json.data.code) || '';
    } catch(e) {}
    if (xhr.status === 402 || serverCode === 'insufficient_credits') {
      return { msg: CREDITS_MSG, html: CREDITS_HTML, isCredits: true };
    }
    if (serverMsg) return { msg: serverMsg, html: serverMsg, isCredits: false };
    if (textStatus === 'timeout') return { msg: 'Request timed out — try again or choose a different model.', html: '', isCredits: false };
    return { msg: 'Network error (status: ' + (xhr.status || textStatus) + '). Check your server error log.', html: '', isCredits: false };
  }

  function updateCostBadge() {
    var $sel = $('#lsp-ai-model');
    var $badge = $('#lsp-ai-cost-label');
    var opt = $sel.find(':selected');

    if (!opt.length || !opt.val()) { $badge.hide(); return; }

    var free = opt.data('free') === 1 || opt.data('free') === '1';
    if (free) {
      $badge.text('Free ✓').css({background: 'rgba(34,197,94,.15)', color: '#16a34a'}).show();
    } else {
      var costText = opt.data('cost') || '';
      if (costText) {
        $badge.text(costText).css({background: 'rgba(249,115,22,.12)', color: '#ea580c'}).show();
      } else {
        $badge.text('Paid').css({background: 'rgba(249,115,22,.12)', color: '#ea580c'}).show();
      }
    }
  }

  function findModel(id) {
    for (var i = 0; i < state.models.length; i++) {
      if (state.models[i].id === id) return state.models[i];
    }
    return null;
  }

  /* ========================================================================
   * Generate
   * ======================================================================== */

  function doGenerate() {
    if (state.generating) return;

    var prompt = $.trim($('#lsp-ai-prompt').val());
    if (!prompt) {
      $('#lsp-ai-prompt').focus();
      return;
    }

    var model = $('#lsp-ai-model').val();
    if (!model) {
      alert('Please select a model.');
      return;
    }

    var aspect = $('#lsp-ai-aspect').val();
    var opt = $('#lsp-ai-model').find(':selected');
    var isFree = opt.data('free') === 1 || opt.data('free') === '1';
    var provider = opt.data('provider') || 'openrouter';

    state.generating = true;
    $('#lsp-ai-generate-btn').prop('disabled', true);
    $('#lsp-ai-generating').show();
    $('#lsp-ai-preview').hide();
    $('#lsp-ai-committed').hide();

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_ai_generate',
        _ajax_nonce: LIGHTSYNCPRO.nonce,
        model: model,
        prompt: prompt,
        type: 'image',
        aspect_ratio: aspect,
        is_free: isFree ? 1 : 0,
        provider: provider,
        preview: 0
      },
      timeout: 200000,  // 200s — image gen can be slow
      success: function(resp) {
        state.generating = false;
        $('#lsp-ai-generate-btn').prop('disabled', false);
        $('#lsp-ai-generating').hide();

        if (!resp.success) {
          var err = resp.data && resp.data.error || 'Generation failed';
          if (resp.data && resp.data.code === 'insufficient_credits') {
            showError(CREDITS_HTML);
          } else {
            showError(err);
          }
          return;
        }

        // Extract image from response
        var data = resp.data;
        var imageB64 = null;
        var imageMime = 'image/png';

        // OpenRouter returns content array with image blocks
        if (data.choices && data.choices.length) {
          var content = data.choices[0].message && data.choices[0].message.content;
          if (Array.isArray(content)) {
            for (var i = 0; i < content.length; i++) {
              if (content[i].type === 'image_url' && content[i].image_url) {
                var url = content[i].image_url.url || '';
                // data:image/png;base64,...
                var match = url.match(/^data:(image\/[^;]+);base64,(.+)$/);
                if (match) {
                  imageMime = match[1];
                  imageB64 = match[2];
                }
              }
            }
          }
        }

        // Fallback: direct base64 fields
        if (!imageB64 && data.base64) {
          imageB64 = data.base64;
          imageMime = data.mime || 'image/png';
        }

        // Fallback: broker returns images array
        if (!imageB64 && data.images && data.images.length) {
          imageB64 = data.images[0].base64;
          imageMime = data.images[0].mime || 'image/png';
        }

        // Fallback: image_url at top level
        if (!imageB64 && data.image_url) {
          var m2 = data.image_url.match(/^data:(image\/[^;]+);base64,(.+)$/);
          if (m2) {
            imageMime = m2[1];
            imageB64 = m2[2];
          }
        }

        if (!imageB64) {
          showError('No image returned. The model may not support image generation. Try a different model.');
          return;
        }

        // Store preview state
        state.currentPreview = {
          base64: imageB64,
          mime: imageMime,
          prompt: prompt,
          model: model,
          is_free: isFree,
          aspect_ratio: aspect
        };

        // Render preview
        showPreview(imageB64, imageMime);
      },
      error: function(xhr, status) {
        state.generating = false;
        $('#lsp-ai-generate-btn').prop('disabled', false);
        $('#lsp-ai-generating').hide();
        var err = parseAjaxError(xhr, status);
        showError(err.html || err.msg);
      }
    });
  }

  function showPreview(b64, mime) {
    var $preview = $('#lsp-ai-preview');
    var $img = $('#lsp-ai-preview-image');

    var src = 'data:' + mime + ';base64,' + b64;
    $img.html('<img src="' + src + '" style="max-width:100%;border-radius:6px;display:block" alt="AI Generated Preview">');

    $preview.show();
    updateCommitButtonText();
    // Scroll to preview
    $preview[0].scrollIntoView({ behavior: 'smooth', block: 'nearest' });
  }

  function discardPreview() {
    state.currentPreview = null;
    $('#lsp-ai-preview').hide();
    $('#lsp-ai-preview-image').html('');
  }

  /* ========================================================================
   * Commit (Save to Media Library)
   * ======================================================================== */

  function doCommit() {
    if (state.committing || !state.currentPreview) return;

    state.committing = true;
    var $btn = $('#lsp-ai-commit-btn');
    var dest = getSelectedDest();
    $btn.prop('disabled', true).text('Saving...');

    var p = state.currentPreview;

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      timeout: 120000,
      data: {
        action: 'lsp_ai_commit',
        _ajax_nonce: LIGHTSYNCPRO.nonce,
        base64: p.base64,
        mime: p.mime,
        prompt: p.prompt,
        model: p.model,
        aspect_ratio: p.aspect_ratio,
        is_free: p.is_free ? 1 : 0,
        destination: dest
      },
      success: function(resp) {
        state.committing = false;
        updateCommitButtonText();
        $btn.prop('disabled', false);

        if (!resp.success) {
          showError(resp.data && resp.data.error || 'Failed to save');
          return;
        }

        var d = resp.data;

        // Show committed state
        $('#lsp-ai-preview').hide();

        var details = ' — ' + d.filename;
        if (d.width && d.height) details += ' (' + d.width + '×' + d.height + ')';
        var filesize = formatBytes(d.filesize || 0);
        details += ', ' + filesize;
        if (d.avif) {
          details += ' | AVIF: ' + formatBytes(d.avif.bytes_out) + ' (' + d.avif.savings + '% savings)';
        }

        // Show destination sync status
        var destMsg = '';
        if (d.shopify) {
          if (d.shopify.error) {
            destMsg = '<br><span style="color:#dc2626">⚠ Shopify: ' + esc(d.shopify.error) + '</span>';
          } else {
            destMsg = '<br><span style="color:#22c55e">✓ Also synced to Shopify</span>';
          }
        }

        $('#lsp-ai-committed-details').html(details + destMsg);
        // Update banner title based on destination
        var bannerTitle = '✓ Saved to Media Library';
        if (dest === 'both' && d.shopify && !d.shopify.error) {
          bannerTitle = '✓ Saved to Media Library + Shopify';
        } else if (dest === 'shopify' && d.shopify && !d.shopify.error) {
          bannerTitle = '✓ Saved to Media Library + Shopify';
        }
        $('#lsp-ai-committed').find('strong').first().text(bannerTitle);
        $('#lsp-ai-edit-link').attr('href', d.edit_url || '#');
        $('#lsp-ai-committed').show();

        // Refresh the browse grid
        state.currentPreview = null;
        loadBrowse(1);
      },
      error: function(xhr, textStatus) {
        state.committing = false;
        $btn.prop('disabled', false);
        updateCommitButtonText();
        var err = parseAjaxError(xhr, textStatus);
        showError(err.html || err.msg);
      }
    });
  }

  /* ========================================================================
   * Browse AI Generated Assets (Figma-style grid with multi-select)
   * ======================================================================== */

  // Selection state
  state.aiSelectedIds = [];
  state.aiBrowseItems = [];

  function loadBrowse(page) {
    var $grid = $('#lsp-ai-grid');
    var $empty = $('#lsp-ai-grid-empty');
    console.log('[LightSync AI] loadBrowse() called, #lsp-ai-grid found=' + $grid.length);

    if (!$grid.length) return;

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      timeout: 30000,
      data: {
        action: 'lsp_ai_browse',
        _ajax_nonce: LIGHTSYNCPRO.nonce,
        page: page || 1,
        per_page: 40
      },
      success: function(resp) {
        console.log('[LightSync AI] loadBrowse response:', resp ? resp.success : 'null');
        if (!resp || !resp.success) {
          var err = (resp && resp.data && resp.data.error) || 'Failed to load images';
          console.error('[LightSync AI] Browse error:', err);
          $empty.html('Could not load images — <a href="#" onclick="if(typeof lspAI!==\'undefined\')lspAI.init();return false;">retry</a>').show();
          return;
        }

        var d = resp.data;
        state.browsePage = d.page;
        state.browseTotal = d.total;
        state.aiBrowseItems = d.items || [];
        window.lspAiCount = d.total;

        if (!d.items || !d.items.length) {
          $grid.html('');
          $empty.show();
          $('#lsp-ai-batch-bar').hide();
          return;
        }

        $empty.hide();
        renderAiGrid(d.items);

        // Pagination
        var pHtml = '';
        if (d.total_pages > 1) {
          pHtml += '<div style="grid-column:1/-1;text-align:center;padding:12px">';
          if (d.page > 1) {
            pHtml += '<button type="button" class="button lsp-ai-page-btn" data-page="' + (d.page - 1) + '">← Prev</button> ';
          }
          pHtml += '<span style="color:var(--lsp-subtext, #666);margin:0 8px;font-size:12px">Page ' + d.page + ' of ' + d.total_pages + '</span>';
          if (d.page < d.total_pages) {
            pHtml += ' <button type="button" class="button lsp-ai-page-btn" data-page="' + (d.page + 1) + '">Next →</button>';
          }
          pHtml += '</div>';
          $grid.append(pHtml);
        }

        // Bind pagination
        $grid.find('.lsp-ai-page-btn').on('click', function() {
          loadBrowse($(this).data('page'));
        });
      },
      error: function(xhr, status) {
        console.error('[LightSync AI] Browse AJAX error:', status, xhr.status);
        $empty.html('Failed to load images (' + status + ') — <a href="#" onclick="if(typeof lspAI!==\'undefined\')lspAI.init();return false;">retry</a>').show();
      }
    });
  }

  function renderAiGrid(items) {
    var $grid = $('#lsp-ai-grid');
    var html = '';

    // Inject styles once
    if (!$('#lsp-ai-grid-styles').length) {
      $('head').append('<style id="lsp-ai-grid-styles">' +
        '.lsp-ai-thumb{position:relative;cursor:pointer;border-radius:12px;overflow:hidden;border:2px solid rgba(0,0,0,.06);background:#f8fafc;transition:all .2s ease}' +
        '.lsp-ai-thumb:hover{border-color:var(--lsp-primary, #ff5757);box-shadow:0 4px 16px rgba(255,87,87,.12)}' +
        '.lsp-ai-thumb.selected{border-color:var(--lsp-primary, #ff5757);background:rgba(255,87,87,.04)}' +
        '.lsp-ai-thumb .lsp-ai-select-badge{position:absolute;top:6px;right:6px;width:22px;height:22px;background:var(--lsp-primary, #ff5757);border-radius:50%;display:flex;align-items:center;justify-content:center;opacity:0;transform:scale(.5);transition:all .2s ease;z-index:3}' +
        '.lsp-ai-thumb.selected .lsp-ai-select-badge{opacity:1;transform:scale(1)}' +
        '.lsp-ai-thumb .lsp-ai-expand-btn{position:absolute;bottom:32px;right:6px;height:28px;background:rgba(15,23,42,.65);backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);border:1px solid rgba(255,255,255,.2);border-radius:8px;display:flex;align-items:center;justify-content:center;gap:4px;cursor:pointer;opacity:0;transform:translateY(4px);transition:all .2s ease;z-index:3;padding:0 8px;color:#fff;font-size:10px;font-weight:600;font-family:inherit;letter-spacing:.3px}' +
        '.lsp-ai-thumb .lsp-ai-expand-btn .lsp-expand-label{display:none}' +
        '.lsp-ai-thumb:hover .lsp-ai-expand-btn{opacity:1;transform:translateY(0)}' +
        '.lsp-ai-thumb:hover .lsp-ai-expand-btn .lsp-expand-label{display:inline}' +
        '.lsp-ai-thumb .lsp-ai-expand-btn:hover{background:rgba(15,23,42,.85);transform:scale(1.04)}' +
        '.lsp-ai-sync-badge{position:absolute;top:6px;left:6px;backdrop-filter:blur(8px);-webkit-backdrop-filter:blur(8px);color:#fff;font-size:10px;font-weight:500;padding:4px 8px;border-radius:6px;display:flex;align-items:center;gap:3px;box-shadow:0 2px 8px rgba(0,0,0,.15);border:1px solid rgba(255,255,255,.2);z-index:2}' +
        '.lsp-ai-sync-badge.synced{background:rgba(16,185,129,.85)}' +
        '.lsp-ai-sync-badge.stale{background:rgba(245,158,11,.85)}' +
        '.lsp-ai-sync-badge.wp-only{background:rgba(37,99,235,.75)}' +
        '.lsp-ai-sync-badge.ai-wp{background:rgba(249,115,22,.8)}' +
        '#lsp-ai-versions-grid{scrollbar-width:thin;scrollbar-color:rgba(0,0,0,.15) transparent}' +
        '#lsp-ai-versions-grid::-webkit-scrollbar{height:4px}' +
        '#lsp-ai-versions-grid::-webkit-scrollbar-thumb{background:rgba(0,0,0,.15);border-radius:4px}' +
        '.lsp-version-card .button{transition:all .15s}' +
        '.lsp-version-card .button:hover{background:var(--lsp-primary, #ff5757);color:#fff;border-color:var(--lsp-primary, #ff5757)}' +
      '</style>');
    }

    items.forEach(function(item) {
      var isSelected = state.aiSelectedIds.indexOf(item.id) !== -1;

      // Build glassmorphism sync badge — AI images are always in WP
      var syncBadge = '';
      var hasShopify = false;
      var hasStale = false;
      if (item.destinations && item.destinations.length) {
        item.destinations.forEach(function(dest) {
          if (dest.key === 'shopify') hasShopify = true;
          if (!dest.is_current) hasStale = true;
        });
      }

      // WP icon (grid squares)
      var wpIcon = '<svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="7" height="7" rx="1"/><rect x="14" y="3" width="7" height="7" rx="1"/><rect x="3" y="14" width="7" height="7" rx="1"/><rect x="14" y="14" width="7" height="7" rx="1"/></svg>';
      // Shopify icon (bag)
      var shopIcon = '<svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/><line x1="3" y1="6" x2="21" y2="6"/><path d="M16 10a4 4 0 01-8 0"/></svg>';

      if (hasStale) {
        // Stale — needs update (amber)
        syncBadge = '<div class="lsp-ai-sync-badge stale" title="Needs update">';
        syncBadge += '<svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M23 4v6h-6M1 20v-6h6"/><path d="M3.51 9a9 9 0 0114.85-3.36L23 10M1 14l4.64 4.36A9 9 0 0020.49 15"/></svg>';
        syncBadge += wpIcon;
        if (hasShopify) syncBadge += shopIcon;
        syncBadge += '</div>';
      } else if (hasShopify) {
        // Synced to WP + Shopify (green)
        syncBadge = '<div class="lsp-ai-sync-badge synced" title="Synced to WP + Shopify">';
        syncBadge += '<svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3"><path d="M20 6L9 17l-5-5"/></svg>';
        syncBadge += wpIcon + shopIcon;
        syncBadge += '</div>';
      } else {
        // WP only (orange — AI source color)
        syncBadge = '<div class="lsp-ai-sync-badge ai-wp" title="In WordPress Media Library">';
        syncBadge += wpIcon;
        syncBadge += '</div>';
      }

      html += '<div class="lsp-ai-thumb' + (isSelected ? ' selected' : '') + '" data-id="' + item.id + '">';
      html += '<img src="' + esc(item.thumbnail) + '" style="width:100%;aspect-ratio:1;object-fit:cover;display:block" loading="lazy" alt="">';

      // Glassmorphism sync badge (top-left)
      html += syncBadge;

      // Selection check badge (top-right) — always in DOM, shown via CSS
      html += '<div class="lsp-ai-select-badge"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="3"><path d="M20 6L9 17l-5-5"/></svg></div>';

      // Expand/detail icon (bottom-right, appears on hover) — glassmorphism with label
      html += '<button type="button" class="lsp-ai-expand-btn" data-id="' + item.id + '" title="View details, regenerate &amp; version history">';
      html += '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><circle cx="12" cy="12" r="10"/><path d="M12 16v-4"/><path d="M12 8h.01"/></svg>';
      html += '<span class="lsp-expand-label">Details</span>';
      html += '</button>';

      // Info bar
      html += '<div style="padding:6px 10px;font-size:11px;color:var(--lsp-subtext, #64748b);white-space:nowrap;overflow:hidden;text-overflow:ellipsis">';
      if (item.is_free) html += '<span style="color:var(--lsp-success, #22c55e);font-weight:700">Free</span> · ';
      html += 'v' + (item.version || 1);
      if (item.has_history) html += ' <span style="display:inline-flex;align-items:center;gap:2px;background:rgba(99,102,241,.1);color:#6366f1;font-size:9px;font-weight:700;padding:1px 5px;border-radius:4px;vertical-align:middle;letter-spacing:.3px" title="Has version history — click expand to view &amp; restore"><svg width="10" height="10" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M3 12a9 9 0 1 0 9-9"/><path d="M3 3v6h6"/><path d="M12 7v5l3 3"/></svg> HISTORY</span>';
      html += '</div>';
      html += '</div>';
    });

    $grid.html(html);

    // Bind click handlers
    $grid.find('.lsp-ai-thumb').on('click', function(e) {
      // Ignore if they clicked the expand button
      if ($(e.target).closest('.lsp-ai-expand-btn').length) return;
      // Single click = toggle selection
      toggleAiSelect($(this).data('id'));
    });

    // Expand button → open detail drawer
    $grid.find('.lsp-ai-expand-btn').on('click', function(e) {
      e.stopPropagation();
      openDetail($(this).data('id'), state.aiBrowseItems);
    });

    updateAiSelectionUI();
  }

  function toggleAiSelect(id) {
    var idx = state.aiSelectedIds.indexOf(id);
    if (idx !== -1) {
      state.aiSelectedIds.splice(idx, 1);
    } else {
      state.aiSelectedIds.push(id);
    }

    // Update card UI
    var $card = $('.lsp-ai-thumb[data-id="' + id + '"]');
    $card.toggleClass('selected', state.aiSelectedIds.indexOf(id) !== -1);

    updateAiSelectionUI();
  }

  function updateAiSelectionUI() {
    var count = state.aiSelectedIds.length;
    var $count = $('#lsp-ai-selection-count');
    var $batchBar = $('#lsp-ai-batch-bar');
    var $batchCount = $('#lsp-ai-batch-count');

    if (count > 0) {
      $count.text(count + ' image' + (count !== 1 ? 's' : '') + ' selected');
      $batchCount.text(count + ' image' + (count !== 1 ? 's' : '') + ' selected');
      $batchBar.css('display', 'flex');
    } else {
      $count.text('');
      $batchBar.hide();
    }

    // Update command bar pill if present
    var $cmdPill = $('#lsp-ai-commandbar-selection .lsp-selection-text');
    if ($cmdPill.length) {
      if (count > 0) {
        $cmdPill.text(count + ' image' + (count !== 1 ? 's' : '') + ' selected');
        $('#lsp-ai-commandbar-selection, #lsp-ai-sep').show();
      } else {
        $('#lsp-ai-commandbar-selection, #lsp-ai-sep').hide();
      }
    }
  }

  /* ========================================================================
   * Detail Panel (click on thumbnail)
   * ======================================================================== */

  function openDetail(attachmentId, items) {
    var item = null;
    for (var i = 0; i < items.length; i++) {
      if (items[i].id === attachmentId) { item = items[i]; break; }
    }
    if (!item) return;

    // Remove existing detail panel
    $('#lsp-ai-detail-panel').remove();

    var destHtml = '';
    var hasStale = false;

    // Always show WordPress — AI images are always in Media Library
    destHtml += '<div style="display:flex;align-items:center;gap:8px;padding:8px 12px;background:rgba(249,115,22,.1);border-radius:10px;font-size:12px;color:#f97316;font-weight:600">';
    destHtml += '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>';
    destHtml += '<span>WordPress</span>';
    destHtml += '<span style="font-weight:400;opacity:.8;margin-left:auto">Current</span>';
    destHtml += '</div>';

    if (item.destinations && item.destinations.length) {
      item.destinations.forEach(function(d) {
        if (d.key === 'wordpress') return; // Already shown above
        var color = d.is_current ? 'var(--lsp-success)' : 'var(--lsp-warning)';
        var bgColor = d.is_current ? 'rgba(34,197,94,.12)' : 'rgba(245,158,11,.12)';
        var label = d.is_current ? 'Current' : 'Needs update';
        var icon = d.is_current
          ? '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>'
          : '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 9v4"/><circle cx="12" cy="17" r="1"/><path d="M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/></svg>';
        destHtml += '<div style="display:flex;align-items:center;gap:8px;padding:8px 12px;background:' + bgColor + ';border-radius:10px;font-size:12px;color:' + color + ';font-weight:600">';
        destHtml += icon + ' ';
        destHtml += '<span style="text-transform:capitalize">' + esc(d.key) + '</span>';
        destHtml += '<span style="font-weight:400;opacity:.8;margin-left:auto">' + label + '</span>';
        destHtml += '</div>';
        if (!d.is_current) hasStale = true;
      });
      if (hasStale) {
        destHtml += '<button type="button" id="lsp-ai-detail-push" class="button button-primary" data-id="' + item.id + '" style="margin-top:10px;width:100%;text-align:center">↑ Sync to Destinations</button>';
        destHtml += '<span id="lsp-ai-detail-push-state" style="display:none;text-align:center;font-size:12px;color:var(--lsp-subtext);margin-top:8px;padding:8px"><span class="spinner is-active" style="float:none;margin:0 4px 0 0;width:14px;height:14px"></span>Syncing...</span>';
      }
    }

    // Build panel
    var panel = '<div id="lsp-ai-detail-panel" class="wrap lightsyncpro" style="position:fixed;top:0;right:0;bottom:0;width:440px;max-width:92vw;background:var(--lsp-card, #fff);box-shadow:-8px 0 40px rgba(15,23,42,0.12);z-index:100100;overflow-y:auto;animation:lspSlideIn 0.25s cubic-bezier(.4,0,.2,1);font-family:Montserrat,system-ui,-apple-system,sans-serif">';
    panel += '<style>@keyframes lspSlideIn{from{transform:translateX(100%);opacity:.8}to{transform:translateX(0);opacity:1}}</style>';

    // Header bar
    panel += '<div style="position:sticky;top:0;z-index:2;background:var(--lsp-card, #fff);padding:16px 24px;border-bottom:1px solid rgba(0,0,0,.06);display:flex;justify-content:space-between;align-items:center">';
    panel += '<h3 style="margin:0;font-size:16px;font-weight:800;letter-spacing:.2px;color:#0b1220">Image Details</h3>';
    panel += '<button type="button" id="lsp-ai-detail-close" style="background:rgba(0,0,0,.04);border:none;width:32px;height:32px;border-radius:10px;font-size:16px;cursor:pointer;color:var(--lsp-subtext);display:flex;align-items:center;justify-content:center;transition:background .15s">✕</button>';
    panel += '</div>';

    // Content
    panel += '<div style="padding:24px">';

    // Image with rounded corners
    panel += '<div style="border-radius:14px;overflow:hidden;margin-bottom:20px;box-shadow:0 4px 16px rgba(15,23,42,.08)">';
    panel += '<img src="' + esc(item.url) + '" style="width:100%;display:block" alt="">';
    panel += '</div>';

    // Prompt field
    panel += '<div style="margin-bottom:20px">';
    panel += '<label style="display:block;font-weight:700;font-size:11px;text-transform:uppercase;letter-spacing:.5px;margin-bottom:6px;color:var(--lsp-subtext)">Prompt</label>';
    panel += '<textarea id="lsp-ai-detail-prompt" style="width:100%;min-height:80px;border:none;background:rgba(0,0,0,.04);border-radius:10px;padding:10px 12px;font-size:13px;resize:vertical;font-family:inherit;transition:all .15s">' + esc(item.prompt || '') + '</textarea>';
    panel += '</div>';

    // Info grid - sub-card style
    panel += '<div style="display:grid;grid-template-columns:1fr 1fr;gap:1px;background:rgba(0,0,0,.06);border-radius:12px;overflow:hidden;margin-bottom:20px">';
    panel += '<div style="background:var(--lsp-card, #fff);padding:12px 14px"><div style="font-size:10px;text-transform:uppercase;letter-spacing:.5px;color:var(--lsp-subtext);font-weight:600;margin-bottom:2px">Model</div><div style="font-size:12px;font-weight:700;color:var(--lsp-text)">' + esc(shortModelName(item.model)) + '</div></div>';
    panel += '<div style="background:var(--lsp-card, #fff);padding:12px 14px"><div style="font-size:10px;text-transform:uppercase;letter-spacing:.5px;color:var(--lsp-subtext);font-weight:600;margin-bottom:2px">Version</div><div style="font-size:12px;font-weight:700;color:var(--lsp-text)">v' + (item.version || 1) + '</div></div>';
    panel += '<div style="background:var(--lsp-card, #fff);padding:12px 14px"><div style="font-size:10px;text-transform:uppercase;letter-spacing:.5px;color:var(--lsp-subtext);font-weight:600;margin-bottom:2px">Size</div><div style="font-size:12px;font-weight:700;color:var(--lsp-text)">' + (item.width || '?') + ' × ' + (item.height || '?') + '</div></div>';
    panel += '<div style="background:var(--lsp-card, #fff);padding:12px 14px"><div style="font-size:10px;text-transform:uppercase;letter-spacing:.5px;color:var(--lsp-subtext);font-weight:600;margin-bottom:2px">Created</div><div style="font-size:12px;font-weight:700;color:var(--lsp-text)">' + formatDate(item.generated_at || item.date) + '</div></div>';
    panel += '</div>';

    // Destinations section
    panel += '<div style="margin-bottom:20px">';
    panel += '<label style="display:block;font-weight:700;font-size:11px;text-transform:uppercase;letter-spacing:.5px;margin-bottom:8px;color:var(--lsp-subtext)">Sync Status</label>';
    panel += '<div style="display:flex;flex-direction:column;gap:6px">' + destHtml + '</div>';
    panel += '</div>';

    // Version History section (only if asset has history)
    panel += '<div id="lsp-ai-detail-versions" style="margin-bottom:20px;display:none">';
    panel += '<label style="display:block;font-weight:700;font-size:11px;text-transform:uppercase;letter-spacing:.5px;margin-bottom:8px;color:var(--lsp-subtext)">Version History</label>';
    panel += '<div id="lsp-ai-versions-grid" style="display:flex;gap:8px;overflow-x:auto;padding:4px 0"></div>';
    panel += '<div id="lsp-ai-versions-loading" style="text-align:center;padding:12px;color:var(--lsp-subtext);font-size:12px"><span class="spinner is-active" style="float:none;margin:0 4px 0 0;width:14px;height:14px"></span> Loading versions...</div>';
    panel += '</div>';

    // Model selector for regeneration (cloned from main form, synced)
    panel += '<div style="margin-bottom:16px">';
    panel += '<label style="display:block;font-weight:700;font-size:11px;text-transform:uppercase;letter-spacing:.5px;margin-bottom:6px;color:var(--lsp-subtext)">Model for Regeneration</label>';
    panel += '<select id="lsp-ai-detail-model" style="width:100%;border:none;background:rgba(0,0,0,.04);border-radius:10px;padding:10px 12px;font-size:13px;font-family:inherit;cursor:pointer"></select>';
    panel += '<div id="lsp-ai-detail-cost" style="display:none;margin-top:6px;font-size:12px;padding:4px 10px;border-radius:6px;background:rgba(34,197,94,.1);color:#16a34a;font-weight:600"></div>';
    panel += '</div>';

    // Aspect ratio for regeneration
    panel += '<div style="margin-bottom:20px">';
    panel += '<label style="display:block;font-weight:700;font-size:11px;text-transform:uppercase;letter-spacing:.5px;margin-bottom:6px;color:var(--lsp-subtext)">Aspect Ratio</label>';
    panel += '<select id="lsp-ai-detail-aspect" style="width:100%;border:none;background:rgba(0,0,0,.04);border-radius:10px;padding:10px 12px;font-size:13px;font-family:inherit;cursor:pointer">';
    panel += '<option value="1:1">1:1 — Square</option>';
    panel += '<option value="16:9">16:9 — Wide Landscape</option>';
    panel += '<option value="9:16">9:16 — Tall Portrait</option>';
    panel += '<option value="4:3">4:3 — Standard Landscape</option>';
    panel += '<option value="3:4">3:4 — Standard Portrait</option>';
    panel += '<option value="3:2">3:2 — Photo Landscape</option>';
    panel += '</select>';
    panel += '</div>';

    // Action buttons - LSP style
    panel += '<div style="display:flex;gap:8px;flex-wrap:wrap">';
    panel += '<button type="button" id="lsp-ai-detail-regen" class="btn primary" data-id="' + item.id + '" style="flex:1;text-align:center">✨ Regenerate</button>';
    panel += '<a href="' + esc(item.url) + '" target="_blank" class="btn ghost" style="flex:1;text-align:center">Open Full Size ↗</a>';
    panel += '</div>';

    // Re-optimize button (only show if not already WebP)
    var isWebP = item.mime === 'image/webp';
    if (!isWebP) {
      panel += '<div style="margin-top:10px">';
      panel += '<button type="button" id="lsp-ai-detail-reoptimize" class="btn ghost" data-id="' + item.id + '" style="width:100%;text-align:center;font-size:12px">';
      panel += '⚡ Convert to WebP — Same ID, smaller file';
      panel += '</button>';
      panel += '<div id="lsp-ai-reoptimize-state" style="display:none;margin-top:8px;text-align:center;padding:12px;background:rgba(0,0,0,.03);border-radius:10px">';
      panel += '<span class="spinner is-active" style="float:none;margin:0 4px 0 0;width:14px;height:14px"></span>';
      panel += '<span style="color:var(--lsp-subtext);font-size:12px;font-weight:600">Optimizing...</span>';
      panel += '</div>';
      panel += '</div>';
    }

    // Regenerating state
    panel += '<div id="lsp-ai-detail-regen-state" style="display:none;margin-top:16px;text-align:center;padding:20px;background:rgba(0,0,0,.03);border-radius:12px">';
    panel += '<span class="spinner is-active" style="float:none;margin:0 auto 8px;display:block"></span>';
    panel += '<div style="color:var(--lsp-subtext);font-size:13px;font-weight:600">Generating new version...</div>';
    panel += '</div>';

    // Side-by-side preview
    panel += '<div id="lsp-ai-detail-compare" style="display:none;margin-top:20px">';
    panel += '<label style="display:block;font-weight:700;font-size:11px;text-transform:uppercase;letter-spacing:.5px;margin-bottom:10px;color:var(--lsp-subtext)">Compare Versions</label>';
    panel += '<div style="display:grid;grid-template-columns:1fr 1fr;gap:10px">';
    panel += '<div><div style="font-size:10px;font-weight:600;text-transform:uppercase;letter-spacing:.5px;color:var(--lsp-subtext);margin-bottom:4px;text-align:center">Current · v' + (item.version || 1) + '</div><img src="' + esc(item.url) + '" style="width:100%;border-radius:10px;border:2px solid rgba(0,0,0,.06)" alt="Current"></div>';
    panel += '<div><div style="font-size:10px;font-weight:600;text-transform:uppercase;letter-spacing:.5px;color:var(--lsp-primary, #ff5757);margin-bottom:4px;text-align:center">New Preview</div><img id="lsp-ai-detail-new-img" src="" style="width:100%;border-radius:10px;border:2px solid var(--lsp-primary, #ff5757)" alt="New"></div>';
    panel += '</div>';
    panel += '<div style="display:flex;gap:8px;margin-top:12px">';
    panel += '<button type="button" id="lsp-ai-detail-save-new" class="button button-primary" data-id="' + item.id + '" style="flex:1;text-align:center">✓ Save New Version</button>';
    panel += '<button type="button" id="lsp-ai-detail-discard-new" class="button" style="flex:1;text-align:center">✕ Discard</button>';
    panel += '</div>';
    panel += '</div>';

    panel += '</div>'; // end content padding
    panel += '</div>'; // end panel

    // Backdrop with blur
    var backdrop = '<div id="lsp-ai-detail-backdrop" style="position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(15,23,42,0.25);backdrop-filter:blur(2px);-webkit-backdrop-filter:blur(2px);z-index:100099;animation:lspFadeIn .2s ease"></div>';
    backdrop += '<style>@keyframes lspFadeIn{from{opacity:0}to{opacity:1}}</style>';

    $('body').append(backdrop + panel);

    // Populate model dropdown from main form, pre-select this image's model
    var $mainModel = $('#lsp-ai-model');
    var $detailModel = $('#lsp-ai-detail-model');
    if ($mainModel.length && $detailModel.length) {
      $detailModel.html($mainModel.html());
      // Try to select the model this image was generated with
      var currentModel = item.model || '';
      if (currentModel && $detailModel.find('option[value="' + currentModel + '"]').length) {
        $detailModel.val(currentModel);
      } else {
        // Fall back to whatever the main form has selected
        $detailModel.val($mainModel.val());
      }
    }

    // Pre-select aspect ratio based on image dimensions
    var $detailAspect = $('#lsp-ai-detail-aspect');
    if (item.width && item.height && $detailAspect.length) {
      var ratio = item.width / item.height;
      if (Math.abs(ratio - 1) < 0.05) $detailAspect.val('1:1');
      else if (Math.abs(ratio - 16/9) < 0.1) $detailAspect.val('16:9');
      else if (Math.abs(ratio - 9/16) < 0.1) $detailAspect.val('9:16');
      else if (Math.abs(ratio - 4/3) < 0.1) $detailAspect.val('4:3');
      else if (Math.abs(ratio - 3/4) < 0.1) $detailAspect.val('3:4');
      else if (Math.abs(ratio - 3/2) < 0.1) $detailAspect.val('3:2');
    }

    // Bind close
    $('#lsp-ai-detail-close, #lsp-ai-detail-backdrop').on('click', closeDetail);

    // Update detail cost badge on model change
    function updateDetailCost() {
      var $opt = $('#lsp-ai-detail-model').find(':selected');
      var $badge = $('#lsp-ai-detail-cost');
      if (!$opt.length || !$opt.val()) { $badge.hide(); return; }
      var free = $opt.data('free') === 1 || $opt.data('free') === '1';
      if (free) {
        $badge.text('Estimated: Free ✓').css({background: 'rgba(34,197,94,.1)', color: '#16a34a'}).show();
      } else {
        var cost = $opt.data('cost') || 'Paid';
        $badge.html('Estimated: <strong>' + esc(cost) + '</strong>').css({background: 'rgba(249,115,22,.1)', color: '#ea580c'}).show();
      }
    }
    $('#lsp-ai-detail-model').on('change', updateDetailCost);
    updateDetailCost();

    // Bind regenerate
    $('#lsp-ai-detail-regen').on('click', function() {
      doDetailRegenerate(item);
    });

    // Bind push to destinations
    $('#lsp-ai-detail-push').on('click', function() {
      doDetailPush(item.id);
    });

    // Bind re-optimize
    $('#lsp-ai-detail-reoptimize').on('click', function() {
      var $btn = $(this);
      $btn.prop('disabled', true);
      $('#lsp-ai-reoptimize-state').show();

      $.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
          action: 'lsp_ai_reoptimize',
          _ajax_nonce: LIGHTSYNCPRO.nonce,
          attachment_id: item.id
        },
        success: function(resp) {
          $('#lsp-ai-reoptimize-state').hide();
          if (!resp.success) {
            alert(resp.data && resp.data.error || 'Optimization failed');
            $btn.prop('disabled', false);
            return;
          }
          var d = resp.data;
          if (d.already_webp) {
            $btn.replaceWith('<div style="text-align:center;font-size:12px;color:var(--lsp-success);font-weight:600;padding:8px">✓ ' + (d.message || 'Already WebP') + '</div>');
            return;
          }
          // Show success with savings
          var savingsText = d.savings > 0 ? ' — ' + d.savings + '% smaller' : '';
          $btn.replaceWith('<div style="text-align:center;font-size:12px;color:var(--lsp-success);font-weight:600;padding:8px">✓ Converted to WebP' + savingsText + '</div>');

          // Update main image in panel
          $('#lsp-ai-detail-panel img:first').attr('src', d.url + '?t=' + Date.now());

          // Refresh the grid
          loadBrowse(state.browsePage || 1);
        },
        error: function() {
          $('#lsp-ai-reoptimize-state').hide();
          alert('Re-optimization request failed.');
          $btn.prop('disabled', false);
        }
      });
    });

    // ESC to close
    $(document).on('keydown.lspAiDetail', function(e) {
      if (e.key === 'Escape') closeDetail();
    });

    // Load version history
    if (item.has_history || (item.version && item.version > 1)) {
      $('#lsp-ai-detail-versions').show();
      $.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
          action: 'lsp_ai_versions',
          _ajax_nonce: LIGHTSYNCPRO.nonce,
          attachment_id: item.id
        },
        success: function(resp) {
          $('#lsp-ai-versions-loading').hide();
          if (!resp.success || !resp.data.history || !resp.data.history.length) {
            $('#lsp-ai-detail-versions').hide();
            return;
          }

          var versions = resp.data.history;
          var currentVer = resp.data.current_version;
          var html = '';

          // Show each version as a small thumbnail card
          versions.forEach(function(v) {
            var isCurrent = v.version === currentVer;
            var hasBackup = v.has_backup && v.backup_url;
            var thumbUrl = hasBackup ? v.backup_url : '';
            var borderColor = isCurrent ? 'var(--lsp-primary, #ff5757)' : 'rgba(0,0,0,.1)';
            var opacity = hasBackup || isCurrent ? '1' : '0.4';

            html += '<div class="lsp-version-card" style="flex:0 0 100px;text-align:center;opacity:' + opacity + '">';
            html += '<div style="width:100px;height:100px;border-radius:10px;overflow:hidden;border:2px solid ' + borderColor + ';margin-bottom:6px;background:#f1f5f9;position:relative">';
            if (thumbUrl) {
              html += '<img src="' + esc(thumbUrl) + '?t=' + Date.now() + '" style="width:100%;height:100%;object-fit:cover" alt="v' + v.version + '">';
            } else if (isCurrent) {
              html += '<img src="' + esc(item.url) + '?t=' + Date.now() + '" style="width:100%;height:100%;object-fit:cover" alt="Current">';
            } else {
              html += '<div style="display:flex;align-items:center;justify-content:center;height:100%;color:#94a3b8;font-size:11px">No backup</div>';
            }
            if (isCurrent) {
              html += '<div style="position:absolute;bottom:0;left:0;right:0;background:var(--lsp-primary, #ff5757);color:#fff;font-size:9px;font-weight:700;padding:2px 0;text-align:center">CURRENT</div>';
            }
            html += '</div>';
            html += '<div style="font-size:11px;font-weight:700;color:var(--lsp-text)">v' + v.version + '</div>';
            html += '<div style="font-size:10px;color:var(--lsp-subtext);margin-bottom:4px">' + formatDate(v.created_at) + '</div>';
            if (hasBackup && !isCurrent) {
              html += '<button type="button" class="button lsp-ai-rollback-btn" data-id="' + item.id + '" data-version="' + v.version + '" style="font-size:10px;padding:2px 8px;min-height:24px;line-height:1.2">Restore</button>';
            }
            html += '</div>';
          });

          $('#lsp-ai-versions-grid').html(html);

          // Bind rollback buttons
          $('#lsp-ai-versions-grid').on('click', '.lsp-ai-rollback-btn', function() {
            var $btn = $(this);
            var targetVersion = $btn.data('version');
            var attachId = $btn.data('id');

            if (!confirm('Restore version ' + targetVersion + '? Your current image will be backed up so you can switch back.')) {
              return;
            }

            $btn.prop('disabled', true).text('Restoring...');

            $.ajax({
              url: ajaxurl,
              type: 'POST',
              data: {
                action: 'lsp_ai_rollback',
                _ajax_nonce: LIGHTSYNCPRO.nonce,
                attachment_id: attachId,
                version: targetVersion
              },
              success: function(resp) {
                if (!resp.success) {
                  alert(resp.data && resp.data.error || 'Rollback failed');
                  $btn.prop('disabled', false).text('Restore');
                  return;
                }

                // Update the main image in the panel
                var d = resp.data;
                $('#lsp-ai-detail-panel img:first').attr('src', d.url);

                // Refresh the grid and re-open the panel
                loadBrowse(state.browsePage || 1);

                // Close and re-open to refresh everything
                closeDetail();

                // Brief delay then show success
                setTimeout(function() {
                  // Find updated item and re-open
                  var $thumb = $('.lsp-ai-thumb[data-id="' + attachId + '"]');
                  if ($thumb.length) {
                    $thumb.find('.lsp-ai-expand-btn').trigger('click');
                  }
                }, 600);
              },
              error: function(xhr) {
                var msg = 'Rollback request failed. Please try again.';
                try {
                  var r = JSON.parse(xhr.responseText);
                  if (r && r.data && r.data.error) msg = r.data.error;
                  else if (r && typeof r.data === 'string') msg = r.data;
                } catch(e) {}
                console.error('[LightSync AI] Rollback error:', xhr.status, xhr.responseText);
                alert(msg);
                $btn.prop('disabled', false).text('Restore');
              }
            });
          });
        },
        error: function() {
          $('#lsp-ai-versions-loading').html('<span style="color:#dc2626">Failed to load version history</span>');
        }
      });
    }
  }

  function closeDetail() {
    $('#lsp-ai-detail-panel, #lsp-ai-detail-backdrop').remove();
    $(document).off('keydown.lspAiDetail');
  }

  /* ========================================================================
   * Regenerate from Detail Panel (Side-by-Side)
   * ======================================================================== */

  function doDetailRegenerate(item) {
    var prompt = $.trim($('#lsp-ai-detail-prompt').val());
    if (!prompt) return;

    // Use the detail panel's own model selector
    var model = $('#lsp-ai-detail-model').val() || $('#lsp-ai-model').val();
    if (!model) {
      alert('Select a model to regenerate with.');
      return;
    }

    var aspect = $('#lsp-ai-detail-aspect').val() || '1:1';
    var $opt = $('#lsp-ai-detail-model').find(':selected');
    var isFree = $opt.data('free') === 1 || $opt.data('free') === '1';
    var provider = $opt.data('provider') || 'openrouter';

    $('#lsp-ai-detail-regen').prop('disabled', true);
    $('#lsp-ai-detail-regen-state').show();
    $('#lsp-ai-detail-compare').hide();

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_ai_generate',
        _ajax_nonce: LIGHTSYNCPRO.nonce,
        model: model,
        prompt: prompt,
        type: 'image',
        aspect_ratio: aspect,
        is_free: isFree ? 1 : 0,
        provider: provider,
        preview: 0
      },
      timeout: 200000,
      success: function(resp) {
        $('#lsp-ai-detail-regen').prop('disabled', false);
        $('#lsp-ai-detail-regen-state').hide();

        if (!resp.success) {
          alert(resp.data && resp.data.error || 'Generation failed');
          return;
        }

        // Extract image (same logic as doGenerate)
        var data = resp.data;
        var imageB64 = null;
        var imageMime = 'image/png';

        if (data.choices && data.choices.length) {
          var content = data.choices[0].message && data.choices[0].message.content;
          if (Array.isArray(content)) {
            for (var i = 0; i < content.length; i++) {
              if (content[i].type === 'image_url' && content[i].image_url) {
                var url = content[i].image_url.url || '';
                var match = url.match(/^data:(image\/[^;]+);base64,(.+)$/);
                if (match) {
                  imageMime = match[1];
                  imageB64 = match[2];
                }
              }
            }
          }
        }
        if (!imageB64 && data.base64) { imageB64 = data.base64; imageMime = data.mime || 'image/png'; }
        if (!imageB64 && data.images && data.images.length) { imageB64 = data.images[0].base64; imageMime = data.images[0].mime || 'image/png'; }
        if (!imageB64 && data.image_url) {
          var m2 = data.image_url.match(/^data:(image\/[^;]+);base64,(.+)$/);
          if (m2) { imageMime = m2[1]; imageB64 = m2[2]; }
        }

        if (!imageB64) {
          alert('No image returned. Try a different model.');
          return;
        }

        // Store for save
        state.regenPreview = {
          base64: imageB64,
          mime: imageMime,
          prompt: prompt,
          model: model,
          is_free: isFree,
          attachment_id: item.id
        };

        // Show side-by-side
        $('#lsp-ai-detail-new-img').attr('src', 'data:' + imageMime + ';base64,' + imageB64);
        $('#lsp-ai-detail-compare').show();

        // Bind save new version
        $('#lsp-ai-detail-save-new').off('click').on('click', function() {
          doSaveNewVersion(item.id);
        });

        // Bind discard new
        $('#lsp-ai-detail-discard-new').off('click').on('click', function() {
          state.regenPreview = null;
          $('#lsp-ai-detail-compare').hide();
        });
      },
      error: function(xhr, textStatus) {
        $('#lsp-ai-detail-regen').prop('disabled', false);
        $('#lsp-ai-detail-regen-state').hide();
        var err = parseAjaxError(xhr, textStatus);
        alert(err.msg);
      }
    });
  }

  function doSaveNewVersion(attachmentId) {
    if (!state.regenPreview) return;

    var $btn = $('#lsp-ai-detail-save-new');
    $btn.prop('disabled', true).text('Saving...');

    var p = state.regenPreview;

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      timeout: 120000,
      data: {
        action: 'lsp_ai_regenerate',
        _ajax_nonce: LIGHTSYNCPRO.nonce,
        attachment_id: attachmentId,
        base64: p.base64,
        prompt: p.prompt,
        model: p.model,
        is_free: p.is_free ? 1 : 0
      },
      success: function(resp) {
        $btn.prop('disabled', false).text('✓ Save New Version');

        if (!resp.success) {
          alert(resp.data && resp.data.error || 'Failed to save');
          return;
        }

        var d = resp.data;
        state.regenPreview = null;

        // Hide the compare view
        $('#lsp-ai-detail-compare').hide();

        // Update the destinations section to show stale + sync button
        if (d.stale_destinations && d.stale_destinations.length) {
          var newDestHtml = '';
          // Always show WP as current (orange)
          newDestHtml += '<div style="display:flex;align-items:center;gap:8px;padding:8px 12px;background:rgba(249,115,22,.1);border-radius:10px;font-size:12px;color:#f97316;font-weight:600">';
          newDestHtml += '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>';
          newDestHtml += '<span>WordPress</span>';
          newDestHtml += '<span style="font-weight:400;opacity:.8;margin-left:auto">v' + d.version + ' saved</span>';
          newDestHtml += '</div>';
          d.stale_destinations.forEach(function(dest) {
            if (dest === 'wordpress') return;
            newDestHtml += '<div style="display:flex;align-items:center;gap:8px;padding:8px 12px;background:rgba(245,158,11,.12);border-radius:10px;font-size:12px;color:var(--lsp-warning, #f59e0b);font-weight:600">';
            newDestHtml += '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 9v4"/><circle cx="12" cy="17" r="1"/><path d="M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/></svg>';
            newDestHtml += '<span style="text-transform:capitalize">' + esc(dest) + '</span>';
            newDestHtml += '<span style="font-weight:400;opacity:.8;margin-left:auto">v' + d.version + ' available</span>';
            newDestHtml += '</div>';
          });
          newDestHtml += '<button type="button" id="lsp-ai-detail-push" class="button button-primary" data-id="' + attachmentId + '" style="margin-top:10px;width:100%;text-align:center">↑ Sync v' + d.version + ' to Destinations</button>';
          newDestHtml += '<span id="lsp-ai-detail-push-state" style="display:none;text-align:center;font-size:12px;color:var(--lsp-subtext, #6b7280);margin-top:8px;padding:8px"><span class="spinner is-active" style="float:none;margin:0 4px 0 0;width:14px;height:14px"></span>Syncing...</span>';

          // Replace the sync status section content
          var $syncSection = $('#lsp-ai-detail-panel').find('label:contains("SYNC STATUS"), label:contains("Sync Status")').parent();
          $syncSection.find('div, button, span').not('label').remove();
          $syncSection.append('<div style="display:flex;flex-direction:column;gap:6px">' + newDestHtml + '</div>');

          // Bind the new push button
          $('#lsp-ai-detail-push').on('click', function() {
            doDetailPush(attachmentId);
          });
        }

        // Update the main image in the detail panel
        if (d.url) {
          $('#lsp-ai-detail-panel img:first').attr('src', d.url + '?v=' + d.version);
        }

        // Show inline success
        var $compare = $('#lsp-ai-detail-compare');
        $compare.html('<div style="padding:14px 16px;background:rgba(34,197,94,.1);border-radius:12px;text-align:center"><span style="font-weight:700;font-size:13px;color:var(--lsp-success, #22c55e)">✓ Updated to v' + d.version + '</span></div>').show();

        loadBrowse(state.browsePage);
      },
      error: function(xhr, textStatus) {
        $btn.prop('disabled', false).text('✓ Save New Version');
        var err = parseAjaxError(xhr, textStatus);
        alert(err.msg);
      }
    });
  }

  /* ========================================================================
   * Sync to Destinations (Shopify file update in-place)
   * ======================================================================== */

  function doDetailPush(attachmentId) {
    var $btn = $('#lsp-ai-detail-push');
    var $state = $('#lsp-ai-detail-push-state');

    $btn.prop('disabled', true).hide();
    $state.show();

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_ai_push_destinations',
        _ajax_nonce: LIGHTSYNCPRO.nonce,
        attachment_id: attachmentId
      },
      success: function(resp) {
        $state.hide();

        if (!resp.success) {
          $btn.prop('disabled', false).show();
          alert(resp.data && resp.data.error || 'Sync failed');
          return;
        }

        var d = resp.data;
        var resultHtml = '';

        // Show results for each destination
        if (d.results) {
          for (var key in d.results) {
            var r = d.results[key];
            if (r.error) {
              resultHtml += '<div style="display:flex;align-items:center;gap:8px;padding:8px 12px;background:rgba(239,68,68,.1);border-radius:10px;font-size:12px;color:var(--lsp-danger, #ef4444);font-weight:600">';
              resultHtml += '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg>';
              resultHtml += '<span style="text-transform:capitalize">' + esc(key) + '</span>';
              resultHtml += '<span style="font-weight:400;opacity:.8;margin-left:auto">' + esc(r.error) + '</span>';
              resultHtml += '</div>';
            } else {
              resultHtml += '<div style="display:flex;align-items:center;gap:8px;padding:8px 12px;background:rgba(34,197,94,.12);border-radius:10px;font-size:12px;color:var(--lsp-success, #22c55e);font-weight:600">';
              resultHtml += '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>';
              resultHtml += '<span style="text-transform:capitalize">' + esc(key) + '</span>';
              resultHtml += '<span style="font-weight:400;opacity:.8;margin-left:auto">' + (r.id_changed ? 'Replaced' : 'Updated') + '</span>';
              resultHtml += '</div>';
            }
          }
        }

        // Replace push button area with results
        $btn.replaceWith(resultHtml);

        loadBrowse(state.browsePage);
      },
      error: function() {
        $state.hide();
        $btn.prop('disabled', false).show();
        alert('Network error. Please try again.');
      }
    });
  }

  /* ========================================================================
   * Disconnect
   * ======================================================================== */

  function doDisconnect() {
    if (!confirm('Disconnect from OpenRouter? You can reconnect anytime.')) return;

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_openrouter_disconnect',
        _wpnonce: LIGHTSYNCPRO.nonce
      },
      success: function(resp) {
        if (resp.success) {
          location.reload();
        } else {
          alert('Failed to disconnect: ' + (resp.data || 'Unknown error'));
        }
      }
    });
  }

  /* ========================================================================
   * Reset / UI helpers
   * ======================================================================== */

  function resetToForm() {
    state.currentPreview = null;
    $('#lsp-ai-committed').hide();
    $('#lsp-ai-preview').hide();
    $('#lsp-ai-preview-image').html('');
    $('#lsp-ai-generate-form').show();
    $('#lsp-ai-prompt').val('').focus();
  }

  function showError(msg) {
    // Insert error above the generate form
    var $err = $('#lsp-ai-error');
    if (!$err.length) {
      $('#lsp-ai-generate-form').before('<div id="lsp-ai-error" class="lsp-notice lsp-notice-error" style="margin-bottom:12px"></div>');
      $err = $('#lsp-ai-error');
    }
    $err.html(msg).show();
    setTimeout(function() { $err.fadeOut(300); }, 8000);
  }

  function formatBytes(bytes) {
    if (!bytes) return '0 B';
    if (bytes < 1024) return bytes + ' B';
    if (bytes < 1048576) return (bytes / 1024).toFixed(1) + ' KB';
    return (bytes / 1048576).toFixed(1) + ' MB';
  }

  function formatDate(iso) {
    if (!iso) return '—';
    try {
      var d = new Date(iso);
      return d.toLocaleDateString();
    } catch (e) {
      return iso;
    }
  }

  function shortModelName(id) {
    if (!id) return '—';
    // "google/gemini-2.0-flash-exp:free" → "Gemini 2.0 Flash"
    var parts = id.split('/');
    var name = parts[parts.length - 1];
    name = name.replace(/:free$/, '').replace(/-exp$/, '');
    return name;
  }

  function esc(str) {
    if (!str) return '';
    var div = document.createElement('div');
    div.textContent = str;
    return div.innerHTML;
  }

  /* ---------- Public API ---------- */
  return {
    init: init
  };

})(jQuery);