/* global LIGHTSYNCPRO, ajaxurl, lspProgress, lspShowProgressModal, lspModal */

// ===== Manual Sync Batch Handler =====
document.addEventListener('DOMContentLoaded', function() {
  var btn = document.getElementById('lsp-start');
  var logEl = document.getElementById('lsp-log');
  var albumsEl = document.getElementById('lsp-albums');
  var catalogEl = document.getElementById('lsp-catalog');

  if (!btn || !window.lspProgress) return;

  // ============================================
  // Background Sync Status Polling
  // ============================================
  var backgroundPollInterval = null;

  function startBackgroundStatusPolling() {
    if (window._lspBackgroundPoll) {
      clearInterval(window._lspBackgroundPoll);
    }

    // Update command bar - disable sync button
    if (btn) {
      btn.disabled = true;
      btn.innerHTML = '<span class="dashicons dashicons-clock"></span> Syncing...';
    }
    
    // Add cancel button to command bar if not already present
    var commandBar = document.querySelector('.commandbar .pillrow:last-child');
    if (commandBar && !document.getElementById('lsp-cancel-background')) {
      var cancelBtn = document.createElement('button');
      cancelBtn.type = 'button';
      cancelBtn.id = 'lsp-cancel-background';
      cancelBtn.className = 'btn ghost';
      cancelBtn.innerHTML = '<span class="dashicons dashicons-no-alt"></span> Cancel';
      cancelBtn.addEventListener('click', cancelBackgroundSync);
      commandBar.appendChild(cancelBtn);
    }

    // Create floating progress card (like Canva)
    var existingCard = document.getElementById('lsp-lr-bg-progress');
    if (existingCard) existingCard.remove();
    
    var progressHtml = 
      '<div id="lsp-lr-bg-progress" style="position:fixed;bottom:20px;right:20px;background:#fff;border-radius:16px;padding:20px 24px;box-shadow:0 8px 30px rgba(0,0,0,.18);z-index:100000;min-width:320px;max-width:400px;">' +
        '<div style="display:flex;align-items:center;gap:14px;margin-bottom:12px;">' +
          '<div style="width:44px;height:44px;border-radius:12px;display:grid;place-items:center;background:rgba(59,130,246,.12);border:1px solid rgba(59,130,246,.25);">' +
            '<span class="dashicons dashicons-format-gallery" style="font-size:20px;color:#3b82f6;animation:none;"></span>' +
          '</div>' +
          '<div style="flex:1;">' +
            '<strong style="display:block;color:#1e293b;font-size:15px;">Syncing Lightroom</strong>' +
            '<span id="lsp-lr-bg-album" style="font-size:13px;color:#64748b;">Starting...</span>' +
          '</div>' +
          '<button type="button" id="lsp-lr-bg-cancel" style="background:none;border:none;color:#94a3b8;cursor:pointer;font-size:20px;padding:4px;" title="Cancel sync">×</button>' +
        '</div>' +
        '<div style="margin-bottom:10px;">' +
          '<div style="display:flex;justify-content:space-between;margin-bottom:6px;">' +
            '<span id="lsp-lr-bg-status" style="font-size:13px;color:#64748b;">Preparing...</span>' +
            '<span id="lsp-lr-bg-pct" style="font-size:13px;font-weight:600;color:#3b82f6;">0%</span>' +
          '</div>' +
          '<div style="height:8px;background:#e5e7eb;border-radius:4px;overflow:hidden;">' +
            '<div id="lsp-lr-bg-bar" style="height:100%;width:0%;background:linear-gradient(90deg,#3b82f6,#60a5fa);transition:width 0.4s ease;border-radius:4px;"></div>' +
          '</div>' +
        '</div>' +
        '<div id="lsp-lr-bg-log" style="max-height:100px;overflow-y:auto;font-family:ui-monospace,monospace;font-size:11px;background:rgba(0,0,0,0.03);border-radius:8px;padding:8px 10px;color:#475569;"></div>' +
      '</div>';
    
    document.body.insertAdjacentHTML('beforeend', progressHtml);
    
    // Bind cancel button in floating card
    document.getElementById('lsp-lr-bg-cancel').addEventListener('click', function() {
      cancelBackgroundSync();
    });
    
    // Track last album to only log on changes
    var lastAlbumIndex = -1;
    var lastAlbumName = '';

    function logToCard(msg) {
      var logEl = document.getElementById('lsp-lr-bg-log');
      if (logEl) {
        var time = new Date().toLocaleTimeString();
        logEl.innerHTML += '<div>[' + time + '] ' + msg + '</div>';
        logEl.scrollTop = logEl.scrollHeight;
      }
    }

    function updateCardProgress(pct, albumName, albumInfo, elapsed) {
      var barEl = document.getElementById('lsp-lr-bg-bar');
      var pctEl = document.getElementById('lsp-lr-bg-pct');
      var albumEl = document.getElementById('lsp-lr-bg-album');
      var statusEl = document.getElementById('lsp-lr-bg-status');
      
      if (barEl) barEl.style.width = pct + '%';
      if (pctEl) pctEl.textContent = pct + '%';
      if (albumEl) albumEl.textContent = albumName || 'Processing...';
      if (statusEl) statusEl.textContent = albumInfo + (elapsed ? ' · ' + elapsed : '');
    }

    logToCard('Background sync started...');

    // Track progress to detect stalls
    var lastPct = 0;
    var stallCount = 0;
    var maxStallPolls = 20; // ~60 seconds of no progress
    var errorCount = 0;
    var maxErrors = 5;

    function poll() {
      var body = new URLSearchParams();
      body.set('action', 'lightsyncpro_ajax_background_status');
      body.set('_ajax_nonce', LIGHTSYNCPRO.nonce);

      fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      })
      .then(function(r) { 
        if (!r.ok) throw new Error('HTTP ' + r.status);
        return r.json(); 
      })
      .then(function(json) {
        // Reset error count on successful response
        errorCount = 0;
        
        if (!json.success || !json.data) {
          logToCard('⚠ Invalid response from server');
          return;
        }

        var d = json.data;

        if (!d.running) {
          // Sync complete
          clearInterval(window._lspBackgroundPoll);
          window._lspBackgroundPoll = null;

          // Remove cancel button from command bar
          var cancelBtn = document.getElementById('lsp-cancel-background');
          if (cancelBtn) cancelBtn.remove();

          // Re-enable sync button
          if (btn) {
            btn.disabled = false;
            btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';
          }

          // Update floating card to show completion
          updateCardProgress(100, 'Complete!', 'All albums synced', '');
          logToCard('✓ Background sync complete!');
          
          var barEl = document.getElementById('lsp-lr-bg-bar');
          if (barEl) barEl.style.background = 'linear-gradient(90deg,#10b981,#34d399)';
          
          // Also update the inline progress if visible
          lspProgress.set(100);
          var pctEl = document.querySelector('#lsp-progress .lsp-percent');
          if (pctEl) pctEl.textContent = '100%';

          // Remove floating card after delay
          setTimeout(function() {
            var card = document.getElementById('lsp-lr-bg-progress');
            if (card) card.remove();
            lspProgress.hide();
            // Refresh to show updated activity
            window.location.reload();
          }, 3000);
          return;
        }

        // Update progress display
        var total = d.total_albums || 1;
        var current = (d.current_album_index || 0);
        var albumName = d.current_album_name || ('Album ' + (current + 1));
        var pct = d.progress_pct || 0;
        
        // Fallback calculation if server doesn't provide good pct
        if (pct === 0 && total > 0) {
          pct = Math.round(((current / total) * 100) + ((1 / total) * 50));
          pct = Math.max(5, Math.min(95, pct));
        }

        var albumInfo = 'Album ' + (current + 1) + ' of ' + total;
        var elapsedStr = d.elapsed || '';
        
        updateCardProgress(pct, albumName, albumInfo, elapsedStr);

        // Also update inline progress ring
        lspProgress.set(pct);
        var inlinePctEl = document.querySelector('#lsp-progress .lsp-percent');
        if (inlinePctEl) inlinePctEl.textContent = pct + '%';

        // Only log when album changes
        if (current !== lastAlbumIndex || albumName !== lastAlbumName) {
          if (lastAlbumIndex >= 0 && lastAlbumName) {
            logToCard('✓ ' + lastAlbumName);
          }
          logToCard('Syncing: ' + albumName);
          lastAlbumIndex = current;
          lastAlbumName = albumName;
          stallCount = 0; // Reset stall counter on album change
        }
        
        // Detect stalls - if progress hasn't changed for a while
        if (pct === lastPct && current === lastAlbumIndex) {
          stallCount++;
          if (stallCount >= maxStallPolls) {
            logToCard('⚠ Sync appears stalled. You may need to refresh and retry.');
            var statusEl = document.getElementById('lsp-lr-bg-status');
            if (statusEl) statusEl.textContent = 'May be stalled - check Activity log';
          }
        } else {
          stallCount = 0;
        }
        lastPct = pct;
      })
      .catch(function(err) {
        console.error('Status poll error:', err);
        errorCount++;
        logToCard('⚠ Connection error: ' + (err.message || err));
        
        if (errorCount >= maxErrors) {
          logToCard('❌ Too many errors. Sync may have failed.');
          var statusEl = document.getElementById('lsp-lr-bg-status');
          if (statusEl) statusEl.textContent = 'Connection errors - check console';
        }
      });
    }

    // Poll immediately, then every 3 seconds (faster than before)
    poll();
    window._lspBackgroundPoll = setInterval(poll, 3000);
  }

  function stopBackgroundStatusPolling() {
    if (backgroundPollInterval) {
      clearInterval(backgroundPollInterval);
      backgroundPollInterval = null;
    }
    if (window._lspBackgroundPoll) {
      clearInterval(window._lspBackgroundPoll);
      window._lspBackgroundPoll = null;
    }
  }

  // ============================================
  // Cancel Background Sync
  // ============================================
  function cancelBackgroundSync() {
    if (!confirm('Are you sure you want to cancel the background sync? Progress will be lost.')) {
      return;
    }

    var body = new URLSearchParams();
    body.set('action', 'lightsyncpro_ajax_cancel_background');
    body.set('_ajax_nonce', LIGHTSYNCPRO.nonce);

    fetch(LIGHTSYNCPRO.ajaxurl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: body.toString()
    })
    .then(function(r) { return r.json(); })
    .then(function(json) {
      // Stop polling
      stopBackgroundStatusPolling();

      // Remove cancel button from command bar
      var cancelBtn = document.getElementById('lsp-cancel-background');
      if (cancelBtn) cancelBtn.remove();

      // Remove floating progress card
      var floatingCard = document.getElementById('lsp-lr-bg-progress');
      if (floatingCard) floatingCard.remove();

      // Hide inline progress
      lspProgress.hide();

      // Re-enable sync button
      if (btn) {
        btn.disabled = false;
        btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';
      }

      if (json.success) {
        lspShowToast('Background sync cancelled', 'warning');
        log('✗ Background sync cancelled');
      } else {
        lspShowToast('Failed to cancel sync', 'error');
      }
    })
    .catch(function(err) {
      console.error('Cancel error:', err);
      lspShowToast('Failed to cancel sync', 'error');
    });
  }

  // ============================================
  // Sync Choice Modal
  // ============================================
  function showSyncChoiceModal(estimatedCount, onChoice) {
    // Remove existing modal if any
    var existing = document.getElementById('lsp-sync-choice-modal');
    if (existing) existing.remove();

    var modalHtml = 
      '<div id="lsp-sync-choice-modal" class="lsp-modal" aria-hidden="false" style="display:block;position:fixed;inset:0;z-index:10060;">' +
        '<div class="lsp-modal-backdrop" data-lsp-close style="position:absolute;inset:0;background:rgba(15,23,42,.55);backdrop-filter:blur(6px);"></div>' +
        '<div class="lsp-modal-card" style="position:absolute;left:50%;top:18%;transform:translateX(-50%);width:min(520px,calc(100% - 28px));background:#fff;border:1px solid #e5e7eb;border-radius:18px;box-shadow:0 30px 80px rgba(15,23,42,.35);padding:20px;">' +
          '<div class="lsp-modal-head" style="display:flex;gap:12px;align-items:center;margin-bottom:16px;">' +
            '<div style="width:40px;height:40px;border-radius:12px;display:grid;place-items:center;background:rgba(59,130,246,.14);border:1px solid rgba(59,130,246,.3);color:#1d4ed8;">' +
              '<span class="dashicons dashicons-cloud-upload" style="font-size:20px;"></span>' +
            '</div>' +
            '<h3 style="margin:0;font-size:17px;font-weight:700;">Choose Sync Method</h3>' +
          '</div>' +
          '<div class="lsp-modal-body">' +
            '<p style="margin:0 0 16px;color:#64748b;">' +
              'Up to <strong style="color:#0f172a;">' + estimatedCount + ' images</strong> to check. ' +
              'Already-synced images will be skipped automatically.' +
            '</p>' +
            '<div style="display:grid;gap:12px;">' +
              '<label class="lsp-sync-option" data-mode="foreground" style="display:flex;gap:12px;padding:14px;border:2px solid #3b82f6;border-radius:12px;cursor:pointer;background:#eff6ff;transition:all .15s;">' +
                '<input type="radio" name="sync_mode" value="foreground" checked style="margin-top:3px;accent-color:#3b82f6;">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Sync Now (Foreground)</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Watch progress in real-time. Keep this tab open until complete.' +
                  '</span>' +
                '</div>' +
              '</label>' +
              '<label class="lsp-sync-option" data-mode="background" style="display:flex;gap:12px;padding:14px;border:2px solid #e5e7eb;border-radius:12px;cursor:pointer;background:#fff;transition:all .15s;">' +
                '<input type="radio" name="sync_mode" value="background" style="margin-top:3px;accent-color:#3b82f6;">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Sync in Background</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Runs via WordPress cron. You can close this tab safely.' +
                  '</span>' +
                '</div>' +
              '</label>' +
            '</div>' +
          '</div>' +
          '<div class="lsp-modal-actions" style="display:flex;gap:10px;justify-content:flex-end;margin-top:20px;">' +
            '<button type="button" class="btn ghost" data-lsp-close style="padding:10px 18px;border-radius:10px;border:1px solid #e5e7eb;background:#fff;cursor:pointer;font-weight:600;">Cancel</button>' +
            '<button type="button" class="btn primary" id="lsp-sync-choice-confirm" style="padding:10px 18px;border-radius:10px;border:none;background:#3b82f6;color:#fff;cursor:pointer;font-weight:600;">Start Sync</button>' +
          '</div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    var modal = document.getElementById('lsp-sync-choice-modal');
    var options = modal.querySelectorAll('.lsp-sync-option');
    var radios = modal.querySelectorAll('input[name="sync_mode"]');

    // Highlight selected option on change
    radios.forEach(function(radio) {
      radio.addEventListener('change', function() {
        options.forEach(function(opt) {
          if (opt.querySelector('input').checked) {
            opt.style.borderColor = '#3b82f6';
            opt.style.background = '#eff6ff';
          } else {
            opt.style.borderColor = '#e5e7eb';
            opt.style.background = '#fff';
          }
        });
      });
    });

    // Hover effects
    options.forEach(function(opt) {
      opt.addEventListener('mouseenter', function() {
        if (!opt.querySelector('input').checked) {
          opt.style.borderColor = '#93c5fd';
          opt.style.background = '#f8fafc';
        }
      });
      opt.addEventListener('mouseleave', function() {
        if (!opt.querySelector('input').checked) {
          opt.style.borderColor = '#e5e7eb';
          opt.style.background = '#fff';
        }
      });
    });

    // Close handlers
    modal.querySelectorAll('[data-lsp-close]').forEach(function(el) {
      el.addEventListener('click', function() {
        modal.remove();
        onChoice(null);
      });
    });

    // ESC key to close
    function handleEsc(e) {
      if (e.key === 'Escape') {
        modal.remove();
        document.removeEventListener('keydown', handleEsc);
        onChoice(null);
      }
    }
    document.addEventListener('keydown', handleEsc);

    // Confirm handler
    var confirmBtn = document.getElementById('lsp-sync-choice-confirm');
    if (confirmBtn) {
      confirmBtn.addEventListener('click', function() {
        var checked = modal.querySelector('input[name="sync_mode"]:checked');
        var mode = checked ? checked.value : 'foreground';
        modal.remove();
        document.removeEventListener('keydown', handleEsc);
        onChoice(mode);
      });
    }
  }

  // ============================================
  // Start Background Sync
  // ============================================
  function startBackgroundSync(catalog, albums) {
    log('Starting background sync...');
    
    // Get current sync target from radio
    var lrSyncTargetRadio = document.querySelector('input[name$="[sync_target]"]:checked');
    var syncTarget = lrSyncTargetRadio ? lrSyncTargetRadio.value : 'wp';
    
    // Disable button immediately
    if (btn) {
      btn.disabled = true;
      btn.innerHTML = '<span class="dashicons dashicons-clock"></span> Starting...';
    }

    var body = new URLSearchParams();
    body.set('action', 'lightsyncpro_ajax_batch');
    body.set('_ajax_nonce', LIGHTSYNCPRO.nonce);
    body.set('catalog', catalog);
    albums.forEach(function(a) {
      body.append('albums[]', a);
    });
    body.set('background', '1');
    body.set('sync_target', syncTarget); // Pass sync target explicitly

    fetch(LIGHTSYNCPRO.ajaxurl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: body.toString()
    })
    .then(function(r) { 
      // Check if response is ok before parsing JSON
      if (!r.ok) {
        throw new Error('Server returned ' + r.status);
      }
      return r.text(); 
    })
    .then(function(text) {
      // Try to parse JSON, handle non-JSON responses
      var json;
      try {
        // Remove any BOM or whitespace before JSON
        text = text.trim();
        if (text.charAt(0) !== '{' && text.charAt(0) !== '[') {
          // Find the start of JSON
          var jsonStart = text.indexOf('{');
          if (jsonStart > 0) {
            console.warn('Found non-JSON prefix:', text.substring(0, jsonStart));
            text = text.substring(jsonStart);
          }
        }
        json = JSON.parse(text);
      } catch (e) {
        console.error('JSON parse error:', e, 'Response:', text.substring(0, 500));
        throw new Error('Invalid server response');
      }
      return json;
    })
    .then(function(json) {
      console.log('Background sync response:', json);

      // Check for success - WordPress returns {success: true, data: {...}}
      if (json.success && json.data) {
        var msg = json.data.message || 'Background sync started for ' + (json.data.albums_queued || albums.length) + ' album(s)';
        log(msg);
        lspShowToast(msg, 'success');

        // Start polling - this will show the progress UI
        startBackgroundStatusPolling();
      } else {
        // Restore button
        if (btn) {
          btn.disabled = false;
          btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';
        }
        
        var errMsg = (json.data && json.data.error) ? json.data.error : 'Failed to start background sync';
        log('Error: ' + errMsg);
        lspShowToast(errMsg, 'error');
      }
    })
    .catch(function(err) {
      console.error('Background sync error:', err);
      
      // Even on error, check if sync actually started (server might have queued it before error)
      setTimeout(function() {
        var checkBody = new URLSearchParams();
        checkBody.set('action', 'lightsyncpro_ajax_background_status');
        checkBody.set('_ajax_nonce', LIGHTSYNCPRO.nonce);
        
        fetch(LIGHTSYNCPRO.ajaxurl, {
          method: 'POST',
          headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
          body: checkBody.toString()
        })
        .then(function(r) { return r.json(); })
        .then(function(statusJson) {
          if (statusJson.success && statusJson.data && statusJson.data.running) {
            // Sync DID start despite the error response
            log('Background sync is running');
            lspShowToast('Background sync started', 'success');
            startBackgroundStatusPolling();
          } else {
            // Really failed
            if (btn) {
              btn.disabled = false;
              btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';
            }
            log('Error: ' + (err.message || err));
            lspShowToast('Failed to start background sync', 'error');
          }
        })
        .catch(function() {
          // Status check failed too
          if (btn) {
            btn.disabled = false;
            btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';
          }
          log('Error: ' + (err.message || err));
          lspShowToast('Failed to start background sync', 'error');
        });
      }, 1000);
    });
  }

  // ============================================
  // Toast Helper
  // ============================================
  function lspShowToast(message, type) {
    var el = document.getElementById('lsp-toast');
    if (!el) return;

    el.innerHTML = '';

    var icon = document.createElement('div');
    icon.className = 'lsp-toast-icon';

    var svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
    svg.setAttribute('viewBox', '0 0 24 24');

    if (type === 'error') {
      svg.innerHTML = '<path d="M12 9v4m0 4h.01M12 3l9 18H3l9-18z" fill="none" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
    } else if (type === 'warning') {
      svg.innerHTML = '<path d="M12 9v4m0 4h.01M12 3l9 18H3l9-18z" fill="none" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
    } else {
      svg.innerHTML = '<path d="M5 13l4 4L19 7" fill="none" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
    }

    icon.appendChild(svg);

    var text = document.createElement('div');
    text.textContent = message || 'Done.';

    el.appendChild(icon);
    el.appendChild(text);

    el.className = 'lsp-toast';
    if (type) el.classList.add('lsp-toast-' + type);

    el.classList.remove('is-visible');
    void el.offsetWidth;
    el.classList.add('is-visible');

    clearTimeout(el._lspHideTimer);
    el._lspHideTimer = setTimeout(function() {
      el.classList.remove('is-visible');
    }, 4500);
  }

  // ============================================
  // Logging Helper
  // ============================================
  function log(msg) {
    if (!logEl) return;
    var line = document.createElement('div');
    line.textContent = '[' + new Date().toLocaleTimeString() + '] ' + msg;
    logEl.appendChild(line);
    logEl.scrollTop = logEl.scrollHeight;
  }

  function qs(el) {
    return el ? el.value : '';
  }

  function selectedVals(select) {
    return Array.from(select ? select.options : [])
      .filter(function(o) { return o.selected; })
      .map(function(o) { return o.value; });
  }

  // ============================================
  // Main Sync Button Handler
  // ============================================
  btn.addEventListener('click', function(e) {
    e.preventDefault();

    // Check which source is active
    var activeSource = 'lightroom';
    var canvaTab = document.querySelector('.lsp-source-tab[data-source="canva"].active');
    var figmaTab = document.querySelector('.lsp-source-tab[data-source="figma"].active');
    var dropboxTab = document.querySelector('.lsp-source-tab[data-source="dropbox"].active');
    var shutterstockTab = document.querySelector('.lsp-source-tab[data-source="shutterstock"].active');
    var aiTab = document.querySelector('.lsp-source-tab[data-source="ai"].active');
    if (canvaTab) {
      activeSource = 'canva';
    } else if (figmaTab) {
      activeSource = 'figma';
    } else if (dropboxTab) {
      activeSource = 'dropbox';
    } else if (shutterstockTab) {
      activeSource = 'shutterstock';
    } else if (aiTab) {
      activeSource = 'ai';
    }

    // Handle Canva sync
    if (activeSource === 'canva') {
      triggerCanvaSync();
      return;
    }

    // Handle Figma sync
    if (activeSource === 'figma') {
      triggerFigmaSync();
      return;
    }

    // Handle Dropbox sync
    if (activeSource === 'dropbox') {
      triggerDropboxSync();
      return;
    }

    // Handle Shutterstock sync
    if (activeSource === 'shutterstock') {
      triggerShutterstockSync();
      return;
    }

    // Handle AI sync to destinations
    if (activeSource === 'ai') {
      triggerAiSync();
      return;
    }

    // Handle Lightroom sync (existing logic)
    var catalog = qs(catalogEl);
    var albums = selectedVals(albumsEl);

    if (!catalog || !albums.length) {
      if (typeof window.lspModal === 'function') {
        window.lspModal({
          title: 'No Albums Selected',
          body: 'Please select at least one album to sync.',
          type: 'warning'
        });
      }
      return;
    }

    // Show a small loading indicator while estimating
    btn.disabled = true;
    btn.innerHTML = '<span class="dashicons dashicons-update" style="animation:spin 1s linear infinite;"></span> Estimating...';

    var estimateBody = new URLSearchParams();
    estimateBody.set('action', 'lightsyncpro_ajax_estimate');
    estimateBody.set('_ajax_nonce', LIGHTSYNCPRO.nonce);
    estimateBody.set('catalog', catalog);
    albums.forEach(function(a) {
      estimateBody.append('albums[]', a);
    });

    fetch(LIGHTSYNCPRO.ajaxurl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: estimateBody.toString()
    })
    .then(function(r) { return r.json(); })
    .then(function(json) {
      // Restore button
      btn.disabled = false;
      btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';

      var ok = json && json.success;
      var total = ok ? ((json.data && json.data.total) || 0) : 0;

      if (total > 0) {
        // Check if Hub is the selected destination
        var lrSyncTargetRadio = document.querySelector('input[name$="[sync_target]"]:checked');
        var lrSyncTarget = lrSyncTargetRadio ? lrSyncTargetRadio.value : 'wp';
        
        if (lrSyncTarget === 'hub') {
          // For Hub, use the same sync flow - backend handles it
          // Show choice modal with Hub-specific messaging
          showSyncChoiceModal(total, function(mode) {
            if (mode === 'background') {
              startBackgroundSync(catalog, albums);
            } else if (mode === 'foreground') {
              startForegroundSync(catalog, albums, total);
            }
          });
        } else {
          // Always show choice modal when there are images to process
          showSyncChoiceModal(total, function(mode) {
            if (mode === 'background') {
              startBackgroundSync(catalog, albums);
            } else if (mode === 'foreground') {
              startForegroundSync(catalog, albums, total);
            }
            // null = cancelled, do nothing
          });
        }
      } else {
        // Zero images - show a message or just run foreground to confirm
        if (typeof window.lspModal === 'function') {
          window.lspModal({
            title: 'No Images Found',
            body: 'The selected albums appear to be empty or could not be read.',
            type: 'warning'
          });
        } else {
          lspShowToast('No images found in selected albums.', 'warning');
        }
      }
    })
    .catch(function(err) {
      // Restore button
      btn.disabled = false;
      btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';

      // Estimate failed - still show choice modal with unknown count
      showSyncChoiceModal('?', function(mode) {
        if (mode === 'background') {
          startBackgroundSync(catalog, albums);
        } else if (mode === 'foreground') {
          startForegroundSync(catalog, albums, null);
        }
      });
    });
  });

  // ============================================
  // Canva Sync Handler
  // ============================================
  function triggerCanvaSync() {
    // Get selected Canva designs
    var selectedIds = window.lspCanvaSelected ? Array.from(window.lspCanvaSelected) : [];

    if (!selectedIds.length) {
      if (typeof window.lspModal === 'function') {
        window.lspModal({
          title: 'No Designs Selected',
          body: 'Please select at least one Canva design to sync.',
          type: 'warning'
        });
      }
      return;
    }

    // Show choice modal for foreground/background
    showCanvaSyncChoiceModal(selectedIds.length, function(mode) {
      if (mode === 'background') {
        startCanvaBackgroundSync(selectedIds);
      } else if (mode === 'foreground') {
        startCanvaForegroundSync(selectedIds);
      }
      // null = cancelled, do nothing
    });
  }

  // Show sync choice modal for Canva (matches Lightroom style)
  function showCanvaSyncChoiceModal(count, callback) {
    // Remove existing modal if any
    var existing = document.getElementById('lsp-canva-sync-choice-modal');
    if (existing) existing.remove();

    var modalHtml = 
      '<div id="lsp-canva-sync-choice-modal" class="lsp-modal" aria-hidden="false" style="display:block;position:fixed;inset:0;z-index:10060;">' +
        '<div class="lsp-modal-backdrop" data-lsp-close style="position:absolute;inset:0;background:rgba(15,23,42,.55);backdrop-filter:blur(6px);"></div>' +
        '<div class="lsp-modal-card" style="position:absolute;left:50%;top:18%;transform:translateX(-50%);width:min(520px,calc(100% - 28px));background:#fff;border:1px solid #e5e7eb;border-radius:18px;box-shadow:0 30px 80px rgba(15,23,42,.35);padding:20px;">' +
          '<div class="lsp-modal-head" style="display:flex;gap:12px;align-items:center;margin-bottom:16px;">' +
            '<div style="width:40px;height:40px;border-radius:12px;display:grid;place-items:center;background:rgba(139,92,246,.14);border:1px solid rgba(139,92,246,.3);color:#7c3aed;">' +
              '<span class="dashicons dashicons-art" style="font-size:20px;"></span>' +
            '</div>' +
            '<h3 style="margin:0;font-size:17px;font-weight:700;">Sync Canva Designs</h3>' +
          '</div>' +
          '<div class="lsp-modal-body">' +
            '<p style="margin:0 0 16px;color:#64748b;">' +
              '<strong style="color:#0f172a;">' + count + ' design' + (count > 1 ? 's' : '') + '</strong> selected to sync. ' +
              'Multi-page designs will create separate images.' +
            '</p>' +
            '<div style="display:grid;gap:12px;">' +
              '<label class="lsp-sync-option" data-mode="foreground" style="display:flex;gap:12px;padding:14px;border:2px solid #8b5cf6;border-radius:12px;cursor:pointer;background:#f5f3ff;transition:all .15s;">' +
                '<input type="radio" name="canva_sync_mode" value="foreground" checked style="margin-top:3px;accent-color:#8b5cf6;">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Sync Now (Foreground)</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Watch progress in real-time. Keep this tab open until complete.' +
                  '</span>' +
                '</div>' +
              '</label>' +
              '<label class="lsp-sync-option" data-mode="background" style="display:flex;gap:12px;padding:14px;border:2px solid #e5e7eb;border-radius:12px;cursor:pointer;background:#fff;transition:all .15s;">' +
                '<input type="radio" name="canva_sync_mode" value="background" style="margin-top:3px;accent-color:#8b5cf6;">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Sync in Background</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Runs via WordPress cron. You can close this tab safely.' +
                  '</span>' +
                '</div>' +
              '</label>' +
            '</div>' +
          '</div>' +
          '<div class="lsp-modal-actions" style="display:flex;gap:10px;justify-content:flex-end;margin-top:20px;">' +
            '<button type="button" class="btn ghost" data-lsp-close style="padding:10px 18px;border-radius:10px;border:1px solid #e5e7eb;background:#fff;cursor:pointer;font-weight:600;">Cancel</button>' +
            '<button type="button" class="btn primary" id="lsp-canva-sync-choice-confirm" style="padding:10px 18px;border-radius:10px;border:none;background:#8b5cf6;color:#fff;cursor:pointer;font-weight:600;">Start Sync</button>' +
          '</div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    var modal = document.getElementById('lsp-canva-sync-choice-modal');
    var options = modal.querySelectorAll('.lsp-sync-option');
    var radios = modal.querySelectorAll('input[name="canva_sync_mode"]');

    // Highlight selected option on change
    radios.forEach(function(radio) {
      radio.addEventListener('change', function() {
        options.forEach(function(opt) {
          if (opt.querySelector('input').checked) {
            opt.style.borderColor = '#8b5cf6';
            opt.style.background = '#f5f3ff';
          } else {
            opt.style.borderColor = '#e5e7eb';
            opt.style.background = '#fff';
          }
        });
      });
    });

    // Hover effects
    options.forEach(function(opt) {
      opt.addEventListener('mouseenter', function() {
        if (!opt.querySelector('input').checked) {
          opt.style.borderColor = '#c4b5fd';
          opt.style.background = '#f8fafc';
        }
      });
      opt.addEventListener('mouseleave', function() {
        if (!opt.querySelector('input').checked) {
          opt.style.borderColor = '#e5e7eb';
          opt.style.background = '#fff';
        }
      });
    });

    // Close handlers
    modal.querySelectorAll('[data-lsp-close]').forEach(function(el) {
      el.addEventListener('click', function() {
        modal.remove();
        callback(null);
      });
    });

    // Confirm handler
    document.getElementById('lsp-canva-sync-choice-confirm').addEventListener('click', function() {
      var selected = modal.querySelector('input[name="canva_sync_mode"]:checked');
      var mode = selected ? selected.value : 'foreground';
      modal.remove();
      callback(mode);
    });
  }

  // Foreground sync for Canva - with progress like Lightroom
  function startCanvaForegroundSync(selectedIds) {
    // Get current sync target
    var syncTargetRadio = document.querySelector('input[name="lsp_canva_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    
    // Hub uses the same sync flow as WordPress - backend handles it
    var destText = syncTarget === 'shopify' ? 'Shopify' : (syncTarget === 'both' ? 'WordPress + Shopify' : 'WordPress');

    // Show syncing state
    btn.disabled = true;
    btn.innerHTML = '<span class="dashicons dashicons-update" style="animation:spin 1s linear infinite;"></span> Syncing...';

    // Create floating progress modal
    var existingModal = document.getElementById('lsp-canva-progress-modal');
    if (existingModal) existingModal.remove();

    var modalHtml = 
      '<div id="lsp-canva-progress-modal" style="position:fixed;inset:0;z-index:100000;display:flex;align-items:center;justify-content:center;background:rgba(15,23,42,.6);backdrop-filter:blur(4px);">' +
        '<div style="background:#fff;border-radius:20px;padding:32px 40px;max-width:420px;width:90%;box-shadow:0 25px 60px rgba(0,0,0,.3);text-align:center;">' +
          '<div style="margin-bottom:20px;">' +
            '<div id="lsp-canva-progress-ring" style="position:relative;width:120px;height:120px;margin:0 auto;">' +
              '<svg viewBox="0 0 100 100" style="transform:rotate(-90deg);width:100%;height:100%;">' +
                '<circle cx="50" cy="50" r="45" fill="none" stroke="#e5e7eb" stroke-width="8"/>' +
                '<circle id="lsp-canva-progress-circle" cx="50" cy="50" r="45" fill="none" stroke="#8b5cf6" stroke-width="8" stroke-linecap="round" stroke-dasharray="283" stroke-dashoffset="283" style="transition:stroke-dashoffset 0.3s ease;"/>' +
              '</svg>' +
              '<div id="lsp-canva-progress-pct" style="position:absolute;inset:0;display:flex;align-items:center;justify-content:center;font-size:28px;font-weight:700;color:#1e293b;">0%</div>' +
            '</div>' +
          '</div>' +
          '<h3 id="lsp-canva-progress-title" style="margin:0 0 8px;font-size:18px;font-weight:600;color:#1e293b;">Syncing to ' + destText + '</h3>' +
          '<p id="lsp-canva-progress-sub" style="margin:0 0 16px;font-size:14px;color:#64748b;">Preparing export...</p>' +
          '<div id="lsp-canva-progress-log" style="max-height:150px;overflow-y:auto;text-align:left;font-family:ui-monospace,monospace;font-size:11px;background:#f8fafc;border:1px solid #e2e8f0;border-radius:8px;padding:10px;color:#475569;"></div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    var modal = document.getElementById('lsp-canva-progress-modal');
    var progressCircle = document.getElementById('lsp-canva-progress-circle');
    var progressPct = document.getElementById('lsp-canva-progress-pct');
    var progressSub = document.getElementById('lsp-canva-progress-sub');
    var progressLog = document.getElementById('lsp-canva-progress-log');

    function setProgress(pct) {
      pct = Math.max(0, Math.min(100, Math.round(pct)));
      var offset = 283 - (283 * pct / 100);
      progressCircle.style.strokeDashoffset = offset;
      progressPct.textContent = pct + '%';
    }

    function logMessage(msg) {
      var time = new Date().toLocaleTimeString();
      progressLog.innerHTML += '<div>[' + time + '] ' + msg + '</div>';
      progressLog.scrollTop = progressLog.scrollHeight;
    }

    logMessage('Starting Canva sync for ' + selectedIds.length + ' design(s)...');

    var total = selectedIds.length;
    var completed = 0;
    var syncedIds = [];
    var errors = [];
    var totalPages = 0;
    var totalWpCount = 0;
    
    var totalUpdated = 0;
    var lastSyncTarget = 'wp';

    // Visual progress animation
    var visualPct = 0;
    var targetPct = 0;
    var ticker = null;

    function easeTowardTarget() {
      var delta = targetPct - visualPct;
      if (Math.abs(delta) < 0.3) {
        visualPct = targetPct;
        setProgress(visualPct);
        return;
      }
      var step = Math.max(0.4, Math.abs(delta) * 0.12);
      visualPct = visualPct + (delta > 0 ? step : -step);
      if ((delta > 0 && visualPct > targetPct) || (delta < 0 && visualPct < targetPct)) {
        visualPct = targetPct;
      }
      setProgress(visualPct);
    }

    function startTicker() {
      if (!ticker) ticker = setInterval(easeTowardTarget, 100);
    }

    function stopTicker() {
      if (ticker) {
        clearInterval(ticker);
        ticker = null;
      }
    }

    startTicker();

    // Process designs one at a time
    function processNext(index) {
      if (index >= selectedIds.length) {
        // All done
        stopTicker();
        targetPct = 100;
        visualPct = 100;
        setProgress(100);

        btn.disabled = false;
        btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';

        // Build appropriate message based on sync target
        // Use the intended target (destText) since that's what we attempted
        var actionText = totalUpdated > 0 ? 'updated' : 'synced';
        var msg = 'Successfully ' + actionText + ' ' + syncedIds.length + ' design(s)';
        if (totalPages > syncedIds.length) {
          msg += ' (' + totalPages + ' pages total)';
        }
        msg += ' to ' + destText + '!';

        logMessage('✓ ' + msg);
        progressSub.textContent = 'Sync complete!';
        progressSub.style.color = '#10b981';

        // Update last sync timestamp in UI
        var lastSyncEl = document.getElementById('lsp-last-sync');
        if (lastSyncEl) {
          lastSyncEl.textContent = 'Just now';
          lastSyncEl.classList.add('lsp-flash');
          setTimeout(function() { lastSyncEl.classList.remove('lsp-flash'); }, 1200);
        }

        // Mark as synced and clear selection
        if (window.lspCanvaSyncedIds) {
          syncedIds.forEach(function(id) { window.lspCanvaSyncedIds.add(id); });
        }
        if (window.lspCanvaSyncedTimes) {
          var now = Math.floor(Date.now() / 1000);
          syncedIds.forEach(function(id) { window.lspCanvaSyncedTimes[id] = now; });
        }
        if (window.lspCanvaSelected) {
          window.lspCanvaSelected.clear();
        }

        // Re-render designs grid
        if (typeof window.renderCanvaDesigns === 'function') {
          window.renderCanvaDesigns(window.filterAndSortDesigns ? window.filterAndSortDesigns() : window.lspCanvaDesigns);
        }

        // Update command bar selection
        if (typeof window.updateCanvaCommandBarSelection === 'function') {
          window.updateCanvaCommandBarSelection();
        }

        // Close modal after delay and refresh page
        setTimeout(function() { 
          if (modal) modal.remove();
          
          // Show celebration for first successful Canva sync
          if (window.lspShowCelebration) {
            var firstDesignName = selectedIds.length > 0 ? getDesignName(selectedIds[0]) : '';
            var shown = window.lspShowCelebration('canva', firstDesignName, totalPages || syncedIds.length);
            if (shown) return; // Celebration handles the reload
          }
          
          // Refresh page to reload everything
          window.location.reload();
        }, 2500);
        return;
      }

      var designId = selectedIds[index];
      var designName = getDesignName(designId);
      
      logMessage('Exporting: ' + designName + '...');
      progressSub.textContent = 'Exporting ' + (index + 1) + ' of ' + total + ': ' + designName;

      // Update target progress (give some progress for starting export)
      targetPct = Math.min(95, ((index + 0.3) / total) * 100);

      var body = new URLSearchParams();
      body.set('action', 'lsp_canva_sync_single');
      body.set('_wpnonce', LIGHTSYNCPRO.nonce);
      body.set('design_id', designId);
      body.set('sync_target', syncTarget);

      fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      })
      .then(function(r) { return r.json(); })
      .then(function(json) {
        completed++;
        
        if (json.success) {
          syncedIds.push(designId);
          var pages = json.data.pages || 1;
          totalPages += pages;
          
          // Track sync counts
          if (json.data.wp_count) totalWpCount += json.data.wp_count;
          
          if (json.data.sync_target) lastSyncTarget = json.data.sync_target;
          
          // Track updates - check actual updates, not just re-syncs
          var wasUpdated = json.data.updated || json.data.wp_updated > 0;
          var wasUnchanged = json.data.unchanged;
          if (wasUpdated) {
            totalUpdated++;
          }
          
          // Build appropriate action text
          var actionText;
          if (wasUnchanged) {
            actionText = 'Unchanged';
          } else if (wasUpdated) {
            actionText = 'Updated';
          } else {
            actionText = 'Synced';
          }
          logMessage('✓ ' + actionText + ': ' + designName + (pages > 1 ? ' (' + pages + ' pages)' : ''));
        } else {
          var err = (json.data && json.data.error) ? json.data.error : 'Failed';
          errors.push(designName + ': ' + err);
          logMessage('✗ Failed: ' + designName + ' - ' + err);
        }

        // Update progress
        targetPct = Math.min(95, (completed / total) * 100);

        // Process next design
        processNext(index + 1);
      })
      .catch(function(err) {
        completed++;
        errors.push(designName + ': Connection error');
        logMessage('✗ Error: ' + designName);
        
        // Continue with next design
        targetPct = Math.min(95, (completed / total) * 100);
        processNext(index + 1);
      });
    }

    // Helper to get design name from ID
    function getDesignName(id) {
      if (window.lspCanvaDesigns) {
        for (var i = 0; i < window.lspCanvaDesigns.length; i++) {
          if (window.lspCanvaDesigns[i].id === id) {
            return window.lspCanvaDesigns[i].title || 'Untitled';
          }
        }
      }
      return 'Design';
    }

    // Start processing
    processNext(0);
  }

  // Background sync for Canva
  function startCanvaBackgroundSync(selectedIds) {
    // Get current sync target
    var syncTargetRadio = document.querySelector('input[name="lsp_canva_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    
    btn.disabled = true;
    btn.innerHTML = '<span class="dashicons dashicons-clock"></span> Starting...';

    var body = new URLSearchParams();
    body.set('action', 'lsp_canva_background_sync');
    body.set('_wpnonce', LIGHTSYNCPRO.nonce);
    body.set('sync_target', syncTarget);
    selectedIds.forEach(function(id) {
      body.append('design_ids[]', id);
    });

    fetch(LIGHTSYNCPRO.ajaxurl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: body.toString()
    })
    .then(function(r) { return r.json(); })
    .then(function(json) {
      if (json.success) {
        // Progress indicator handles feedback - no toast needed
        
        // Clear selection
        if (window.lspCanvaSelected) {
          window.lspCanvaSelected.clear();
        }
        if (typeof window.updateCanvaCommandBarSelection === 'function') {
          window.updateCanvaCommandBarSelection();
        }
        if (typeof window.renderCanvaDesigns === 'function') {
          window.renderCanvaDesigns(window.filterAndSortDesigns ? window.filterAndSortDesigns() : window.lspCanvaDesigns);
        }

        // Start polling for status
        startCanvaBackgroundPolling();
      } else {
        var err = (json.data && json.data.error) ? json.data.error : 'Unknown error';
        lspShowToast('Failed to start background sync: ' + err, 'error');
        btn.disabled = false;
        btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';
      }
    })
    .catch(function(err) {
      lspShowToast('Connection error starting background sync', 'error');
      btn.disabled = false;
      btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';
    });
  }

  // Poll for Canva background sync status
  function startCanvaBackgroundPolling() {
    btn.disabled = true;
    
    // Get current sync target for display
    var syncTargetRadio = document.querySelector('input[name="lsp_canva_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    var destText = syncTarget === 'shopify' ? 'Shopify' : (syncTarget === 'both' ? 'WordPress + Shopify' : 'WordPress');

    // Create a simple progress indicator
    var progressHtml = 
      '<div id="lsp-canva-bg-progress" style="position:fixed;bottom:20px;right:20px;background:#fff;border-radius:12px;padding:16px 20px;box-shadow:0 4px 20px rgba(0,0,0,.15);z-index:10000;min-width:280px;">' +
        '<div style="display:flex;align-items:center;gap:12px;margin-bottom:8px;">' +
          '<div class="dashicons dashicons-update" style="animation:spin 1s linear infinite;color:#8b5cf6;"></div>' +
          '<strong style="color:#1e293b;">Syncing to ' + destText + '</strong>' +
        '</div>' +
        '<div id="lsp-canva-bg-status" style="font-size:13px;color:#64748b;">Starting...</div>' +
        '<div style="margin-top:10px;height:6px;background:#e5e7eb;border-radius:3px;overflow:hidden;">' +
          '<div id="lsp-canva-bg-bar" style="height:100%;width:0%;background:#8b5cf6;transition:width 0.3s;"></div>' +
        '</div>' +
      '</div>';
    
    document.body.insertAdjacentHTML('beforeend', progressHtml);

    var pollCount = 0;
    var maxPolls = 200; // Safety limit

    var pollInterval = setInterval(function() {
      pollCount++;
      
      if (pollCount > maxPolls) {
        clearInterval(pollInterval);
        finishBackgroundSync('Sync timed out', true);
        return;
      }

      var body = new URLSearchParams();
      body.set('action', 'lsp_canva_background_status');
      body.set('_wpnonce', LIGHTSYNCPRO.nonce);

      fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      })
      .then(function(r) { return r.json(); })
      .then(function(json) {
        if (!json.success) {
          clearInterval(pollInterval);
          finishBackgroundSync('Error checking status', true);
          return;
        }

        var data = json.data || {};
        var synced = data.synced || 0;
        var total = data.total || 0;
        var current = data.current || 0;
        var running = data.running;
        var totalPages = data.total_pages || synced;

        // Update progress UI
        var statusEl = document.getElementById('lsp-canva-bg-status');
        var barEl = document.getElementById('lsp-canva-bg-bar');
        
        if (statusEl && total > 0) {
          statusEl.textContent = 'Synced ' + synced + ' of ' + total + ' designs...';
        }
        if (barEl && total > 0) {
          var pct = Math.round((current / total) * 100);
          barEl.style.width = pct + '%';
        }

        if (!running) {
          // Sync complete
          clearInterval(pollInterval);

          // Build destination message based on sync_target
          var destText = 'WordPress';
          // sync target is always WordPress in free version

          var msg = 'Synced ' + synced + ' design(s)';
          if (totalPages > synced) {
            msg += ' (' + totalPages + ' pages)';
          }
          msg += ' to ' + destText;
          
          // Update last sync timestamp in UI
          var lastSyncEl = document.getElementById('lsp-last-sync');
          if (lastSyncEl) {
            lastSyncEl.textContent = 'Just now';
            lastSyncEl.classList.add('lsp-flash');
            setTimeout(function() { lastSyncEl.classList.remove('lsp-flash'); }, 1200);
          }

          finishBackgroundSync(msg, synced === 0);

          // Mark as synced
          if (data.design_ids && window.lspCanvaSyncedIds) {
            data.design_ids.forEach(function(id) { 
              window.lspCanvaSyncedIds.add(id); 
            });
          }
          if (data.design_ids && window.lspCanvaSyncedTimes) {
            var now = Math.floor(Date.now() / 1000);
            data.design_ids.forEach(function(id) { 
              window.lspCanvaSyncedTimes[id] = now; 
            });
          }
          
          // Re-render
          if (typeof window.renderCanvaDesigns === 'function') {
            window.renderCanvaDesigns(window.filterAndSortDesigns ? window.filterAndSortDesigns() : window.lspCanvaDesigns);
          }
        }
      })
      .catch(function() {
        // Don't stop on network errors, just continue polling
      });
    }, 2000); // Poll every 2 seconds (each poll processes one design)
  }

  function finishBackgroundSync(message, isError) {
    btn.disabled = false;
    btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';
    
    // Update progress UI
    var progressEl = document.getElementById('lsp-canva-bg-progress');
    if (progressEl) {
      var statusEl = document.getElementById('lsp-canva-bg-status');
      var barEl = document.getElementById('lsp-canva-bg-bar');
      
      if (statusEl) {
        statusEl.textContent = message;
        statusEl.style.color = isError ? '#ef4444' : '#10b981';
      }
      if (barEl) {
        barEl.style.width = '100%';
        barEl.style.background = isError ? '#ef4444' : '#10b981';
      }
      
      // Remove after delay and refresh page
      setTimeout(function() {
        if (progressEl.parentNode) {
          progressEl.parentNode.removeChild(progressEl);
        }
        // Refresh page to reload everything
        window.location.reload();
      }, 2500);
    } else {
      // No progress element, just refresh after toast
      setTimeout(function() {
        window.location.reload();
      }, 2500);
    }
  }

  // ============================================
  // Figma Sync Handler
  // ============================================
  function triggerFigmaSync() {
    // Get selected Figma frames
    var selectedIds = window.lspFigmaSelectedFrames ? window.lspFigmaSelectedFrames.slice() : [];

    if (!selectedIds.length) {
      if (typeof window.lspModal === 'function') {
        window.lspModal({
          title: 'No Frames Selected',
          body: 'Please select at least one Figma frame to sync.',
          type: 'warning'
        });
      }
      return;
    }

    // Show choice modal for foreground/background
    showFigmaSyncChoiceModal(selectedIds.length, function(mode) {
      if (mode === 'foreground') {
        startFigmaForegroundSync(selectedIds);
      } else if (mode === 'background') {
        startFigmaBackgroundSync(selectedIds);
      }
      // null = cancelled, do nothing
    });
  }

  // Show sync choice modal for Figma (matches Canva style)
  function showFigmaSyncChoiceModal(count, callback) {
    // Remove existing modal if any
    var existing = document.getElementById('lsp-figma-sync-choice-modal');
    if (existing) existing.remove();

    // Get export format for display
    var formatEl = document.getElementById('lsp-figma-export-format');
    var format = formatEl ? formatEl.value.toUpperCase() : 'PNG';
    
    // Check if nested mode is active for terminology
    var isNested = document.getElementById('lsp-figma-show-nested') && document.getElementById('lsp-figma-show-nested').checked;
    var itemLabel = isNested ? 'element' : 'frame';
    var itemLabelPlural = isNested ? 'elements' : 'frames';

    var modalHtml = 
      '<div id="lsp-figma-sync-choice-modal" class="lsp-modal" aria-hidden="false" style="display:block;position:fixed;inset:0;z-index:10060;">' +
        '<div class="lsp-modal-backdrop" data-lsp-close style="position:absolute;inset:0;background:rgba(15,23,42,.55);backdrop-filter:blur(6px);"></div>' +
        '<div class="lsp-modal-card" style="position:absolute;left:50%;top:18%;transform:translateX(-50%);width:min(520px,calc(100% - 28px));background:#fff;border:1px solid #e5e7eb;border-radius:18px;box-shadow:0 30px 80px rgba(15,23,42,.35);padding:20px;">' +
          '<div class="lsp-modal-head" style="display:flex;gap:12px;align-items:center;margin-bottom:16px;">' +
            '<div style="width:40px;height:40px;border-radius:12px;display:grid;place-items:center;background:rgba(37,99,235,.14);border:1px solid rgba(37,99,235,.3);color:#2563eb;">' +
              '<span class="dashicons dashicons-art" style="font-size:20px;"></span>' +
            '</div>' +
            '<h3 style="margin:0;font-size:17px;font-weight:700;">Sync from Figma</h3>' +
          '</div>' +
          '<div class="lsp-modal-body">' +
            '<p style="margin:0 0 16px;color:#64748b;">' +
              '<strong style="color:#0f172a;">' + count + ' ' + (count > 1 ? itemLabelPlural : itemLabel) + '</strong> selected to export as <strong style="color:#0f172a;">' + format + '</strong>.' +
            '</p>' +
            '<div style="display:grid;gap:12px;">' +
              '<label class="lsp-sync-option" data-mode="foreground" style="display:flex;gap:12px;padding:14px;border:2px solid #2563eb;border-radius:12px;cursor:pointer;background:#eff6ff;transition:all .15s;">' +
                '<input type="radio" name="figma_sync_mode" value="foreground" checked style="margin-top:3px;accent-color:#2563eb;">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Sync Now</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Export and import immediately. Keep this tab open until complete.' +
                  '</span>' +
                '</div>' +
              '</label>' +
              '<label class="lsp-sync-option" data-mode="background" style="display:flex;gap:12px;padding:14px;border:2px solid #e2e8f0;border-radius:12px;cursor:pointer;background:#fff;transition:all .15s;">' +
                '<input type="radio" name="figma_sync_mode" value="background" style="margin-top:3px;accent-color:#2563eb;">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Background Sync</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Queue for background processing. You can close this tab.' +
                  '</span>' +
                '</div>' +
              '</label>' +
            '</div>' +
          '</div>' +
          '<div class="lsp-modal-actions" style="display:flex;gap:10px;justify-content:flex-end;margin-top:20px;">' +
            '<button type="button" class="btn ghost" data-lsp-close style="padding:10px 18px;border-radius:10px;border:1px solid #e5e7eb;background:#fff;cursor:pointer;font-weight:600;">Cancel</button>' +
            '<button type="button" class="btn primary" id="lsp-figma-sync-choice-confirm" style="padding:10px 18px;border-radius:10px;border:none;background:#2563eb;color:#fff;cursor:pointer;font-weight:600;">Start Sync</button>' +
          '</div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    var modal = document.getElementById('lsp-figma-sync-choice-modal');
    
    // Style radio selections
    var options = modal.querySelectorAll('.lsp-sync-option');
    options.forEach(function(opt) {
      opt.addEventListener('click', function() {
        options.forEach(function(o) {
          o.style.borderColor = '#e2e8f0';
          o.style.background = '#fff';
        });
        this.style.borderColor = '#2563eb';
        this.style.background = '#eff6ff';
        this.querySelector('input').checked = true;
      });
    });

    // Close handlers
    modal.querySelectorAll('[data-lsp-close]').forEach(function(el) {
      el.addEventListener('click', function() {
        modal.remove();
        callback(null);
      });
    });

    // Confirm handler
    document.getElementById('lsp-figma-sync-choice-confirm').addEventListener('click', function() {
      var selected = modal.querySelector('input[name="figma_sync_mode"]:checked');
      var mode = selected ? selected.value : 'foreground';
      modal.remove();
      callback(mode);
    });

    // ESC to close
    var escHandler = function(e) {
      if (e.key === 'Escape') {
        modal.remove();
        document.removeEventListener('keydown', escHandler);
        callback(null);
      }
    };
    document.addEventListener('keydown', escHandler);
  }

  function startFigmaForegroundSync(selectedIds) {
    // Get export settings
    var format = document.getElementById('lsp-figma-export-format');
    var scale = document.getElementById('lsp-figma-export-scale');
    var syncTargetRadio = document.querySelector('input[name="lsp_figma_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    
    // Hub uses the same sync flow as WordPress - backend handles it
    var destText = syncTarget === 'shopify' ? 'Shopify' : (syncTarget === 'both' ? 'WordPress + Shopify' : 'WordPress');

    // Show syncing state
    btn.disabled = true;
    btn.innerHTML = '<span class="dashicons dashicons-update" style="animation:spin 1s linear infinite;"></span> Syncing...';

    // Create floating progress modal (like Canva)
    var existingModal = document.getElementById('lsp-figma-progress-modal');
    if (existingModal) existingModal.remove();

    var modalHtml = 
      '<div id="lsp-figma-progress-modal" style="position:fixed;inset:0;z-index:100000;display:flex;align-items:center;justify-content:center;background:rgba(15,23,42,.6);backdrop-filter:blur(4px);">' +
        '<div style="background:#fff;border-radius:20px;padding:32px 40px;max-width:420px;width:90%;box-shadow:0 25px 60px rgba(0,0,0,.3);text-align:center;">' +
          '<div style="margin-bottom:20px;">' +
            '<div id="lsp-figma-progress-ring" style="position:relative;width:120px;height:120px;margin:0 auto;">' +
              '<svg viewBox="0 0 100 100" style="transform:rotate(-90deg);width:100%;height:100%;">' +
                '<circle cx="50" cy="50" r="45" fill="none" stroke="#e5e7eb" stroke-width="8"/>' +
                '<circle id="lsp-figma-progress-circle" cx="50" cy="50" r="45" fill="none" stroke="#2563eb" stroke-width="8" stroke-linecap="round" stroke-dasharray="283" stroke-dashoffset="283" style="transition:stroke-dashoffset 0.3s ease;"/>' +
              '</svg>' +
              '<div id="lsp-figma-progress-pct" style="position:absolute;inset:0;display:flex;align-items:center;justify-content:center;font-size:28px;font-weight:700;color:#1e293b;">0%</div>' +
            '</div>' +
          '</div>' +
          '<h3 id="lsp-figma-progress-title" style="margin:0 0 8px;font-size:18px;font-weight:600;color:#1e293b;">Syncing to ' + destText + '</h3>' +
          '<p id="lsp-figma-progress-sub" style="margin:0 0 16px;font-size:14px;color:#64748b;">Preparing export...</p>' +
          '<div id="lsp-figma-progress-log" style="max-height:150px;overflow-y:auto;text-align:left;font-family:ui-monospace,monospace;font-size:11px;background:#f8fafc;border:1px solid #e2e8f0;border-radius:8px;padding:10px;color:#475569;"></div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    var modal = document.getElementById('lsp-figma-progress-modal');
    var progressCircle = document.getElementById('lsp-figma-progress-circle');
    var progressPct = document.getElementById('lsp-figma-progress-pct');
    var progressTitle = document.getElementById('lsp-figma-progress-title');
    var progressSub = document.getElementById('lsp-figma-progress-sub');
    var progressLog = document.getElementById('lsp-figma-progress-log');

    function setProgress(pct) {
      pct = Math.max(0, Math.min(100, Math.round(pct)));
      var offset = 283 - (283 * pct / 100);
      progressCircle.style.strokeDashoffset = offset;
      progressPct.textContent = pct + '%';
    }

    function logMessage(msg) {
      var time = new Date().toLocaleTimeString();
      progressLog.innerHTML += '<div>[' + time + '] ' + msg + '</div>';
      progressLog.scrollTop = progressLog.scrollHeight;
    }

    logMessage('Starting Figma sync for ' + selectedIds.length + ' element(s)...');

    // Group frames by file_key for efficient export
    var framesByFile = {};
    var allFrames = window.lspFigmaAllFrames || [];
    selectedIds.forEach(function(frameId) {
      var frame = allFrames.find(function(f) { return f.id === frameId; });
      if (frame) {
        if (!framesByFile[frame.file_key]) {
          framesByFile[frame.file_key] = { frames: [], fileName: frame.file_name };
        }
        framesByFile[frame.file_key].frames.push(frameId);
      }
    });

    var total = selectedIds.length;
    var completed = 0;
    var errors = [];

    // Visual progress animation
    var visualPct = 0;
    var targetPct = 0;
    var ticker = null;

    function easeTowardTarget() {
      var delta = targetPct - visualPct;
      if (Math.abs(delta) < 0.3) {
        visualPct = targetPct;
        setProgress(visualPct);
        return;
      }
      var step = Math.max(0.4, Math.abs(delta) * 0.12);
      visualPct = visualPct + (delta > 0 ? step : -step);
      if ((delta > 0 && visualPct > targetPct) || (delta < 0 && visualPct < targetPct)) {
        visualPct = targetPct;
      }
      setProgress(visualPct);
    }

    function startTicker() {
      if (!ticker) ticker = setInterval(easeTowardTarget, 100);
    }

    function stopTicker() {
      if (ticker) {
        clearInterval(ticker);
        ticker = null;
      }
    }

    startTicker();

    // Process files sequentially
    var fileKeys = Object.keys(framesByFile);
    var currentFileIndex = 0;

    function processNextFile() {
      if (currentFileIndex >= fileKeys.length) {
        // All done
        stopTicker();
        targetPct = 100;
        visualPct = 100;
        setProgress(100);

        btn.disabled = false;
        btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';

        var success = errors.length === 0;
        progressCircle.style.stroke = success ? '#10b981' : '#f59e0b';
        progressTitle.textContent = success ? 'Sync Complete!' : 'Sync Complete with Errors';
        progressSub.textContent = completed + ' of ' + total + ' element(s) synced to ' + destText;

        logMessage(success ? '✓ All elements synced successfully' : '⚠ ' + errors.length + ' error(s) occurred');

        // Add close button
        var closeBtn = document.createElement('button');
        closeBtn.textContent = 'Close';
        closeBtn.style.cssText = 'margin-top:16px;padding:10px 24px;background:#2563eb;color:#fff;border:none;border-radius:8px;font-weight:600;cursor:pointer;';
        closeBtn.onclick = function() {
          modal.remove();
          // Refresh page to update header, activity log, and synced badges
          window.location.reload();
        };
        progressLog.parentNode.appendChild(closeBtn);

        return;
      }

      var fileKey = fileKeys[currentFileIndex];
      var fileData = framesByFile[fileKey];
      var frameIds = fileData.frames;
      var fileName = fileData.fileName || fileKey;

      logMessage('Exporting from ' + fileName + ' (' + frameIds.length + ' elements)...');
      progressSub.textContent = 'Exporting from ' + fileName + '...';

      var body = new URLSearchParams();
      body.set('action', 'lsp_figma_sync_frames');
      body.set('_wpnonce', LIGHTSYNCPRO.nonce);
      body.set('file_key', fileKey);
      frameIds.forEach(function(id) {
        body.append('frame_ids[]', id);
      });
      body.set('format', format ? format.value : 'png');
      body.set('scale', scale ? scale.value : '2');
      body.set('sync_target', syncTarget);

      fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      })
      .then(function(r) { return r.json(); })
      .then(function(json) {
        if (json.success) {
          var syncedCount = json.data.synced || frameIds.length;
          completed += syncedCount;
          logMessage('✓ Synced ' + syncedCount + ' element(s) from ' + fileName);
          
          // Update synced info with destination
          var dest = syncTarget === 'both' ? 'both' : (syncTarget === 'shopify' ? 'shopify' : 'wp');
          frameIds.forEach(function(id) {
            window.lspFigmaSyncedInfo[id] = {
              attachment_id: null,
              synced_at: new Date().toISOString(),
              needs_update: false,
              dest: dest
            };
          });
          // Re-render grid to show sync badges
          if (typeof window.renderFigmaFrames === 'function') {
            window.renderFigmaFrames();
          }
        } else {
          errors.push(fileName + ': ' + (json.data?.error || 'Unknown error'));
          logMessage('✗ Error: ' + (json.data?.error || 'Unknown error'));
        }
        
        // Update progress
        targetPct = Math.round((completed / total) * 100);
        progressSub.textContent = completed + ' of ' + total + ' synced...';
        
        // Process next file
        currentFileIndex++;
        processNextFile();
      })
      .catch(function(err) {
        console.error('Figma sync error:', err);
        errors.push(fileName + ': ' + err.message);
        logMessage('✗ Error: ' + err.message);
        currentFileIndex++;
        processNextFile();
      });
    }

    // Start processing
    processNextFile();
  }

  // Background sync for Figma (with progress bar like Canva)
  function startFigmaBackgroundSync(selectedIds) {
    var format = document.getElementById('lsp-figma-export-format');
    var scale = document.getElementById('lsp-figma-export-scale');
    var syncTargetRadio = document.querySelector('input[name="lsp_figma_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    var destText = syncTarget === 'shopify' ? 'Shopify' : (syncTarget === 'both' ? 'WordPress + Shopify' : 'WordPress');

    // Group frames by file_key
    var framesByFile = {};
    var allFrames = window.lspFigmaAllFrames || [];
    selectedIds.forEach(function(frameId) {
      var frame = allFrames.find(function(f) { return f.id === frameId; });
      if (frame) {
        if (!framesByFile[frame.file_key]) {
          framesByFile[frame.file_key] = [];
        }
        framesByFile[frame.file_key].push({
          id: frameId,
          name: frame.name,
          file_key: frame.file_key
        });
      }
    });

    // Create progress bar UI (like Canva)
    var existingProgress = document.getElementById('lsp-figma-bg-progress');
    if (existingProgress) existingProgress.remove();

    var progressHtml = 
      '<div id="lsp-figma-bg-progress" style="position:fixed;bottom:20px;right:20px;background:#fff;border-radius:12px;padding:16px 20px;box-shadow:0 4px 20px rgba(0,0,0,.15);z-index:10000;min-width:280px;">' +
        '<div style="display:flex;align-items:center;gap:12px;margin-bottom:8px;">' +
          '<div class="dashicons dashicons-update" style="animation:spin 1s linear infinite;color:#2563eb;"></div>' +
          '<strong style="color:#1e293b;">Syncing to ' + destText + '</strong>' +
        '</div>' +
        '<div id="lsp-figma-bg-status" style="font-size:13px;color:#64748b;">Queuing items...</div>' +
        '<div style="margin-top:10px;height:6px;background:#e5e7eb;border-radius:3px;overflow:hidden;">' +
          '<div id="lsp-figma-bg-bar" style="height:100%;width:0%;background:#2563eb;transition:width 0.3s;"></div>' +
        '</div>' +
      '</div>';
    
    document.body.insertAdjacentHTML('beforeend', progressHtml);

    // Queue each file's frames for background processing
    var fileKeys = Object.keys(framesByFile);
    var queuedCount = 0;

    fileKeys.forEach(function(fileKey) {
      var frames = framesByFile[fileKey];
      
      jQuery.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
          action: 'lsp_figma_queue_sync',
          _wpnonce: LIGHTSYNCPRO.nonce,
          file_key: fileKey,
          frame_ids: frames.map(function(f) { return f.id; }),
          format: format ? format.value : 'png',
          scale: scale ? scale.value : '2',
          sync_target: syncTarget
        },
        success: function(response) {
          queuedCount++;
          if (queuedCount === fileKeys.length) {
            // All queued, start polling for progress
            startFigmaBackgroundPolling(selectedIds.length, destText);
          }
        },
        error: function() {
          queuedCount++;
          if (queuedCount === fileKeys.length) {
            finishFigmaBackgroundSync('Error queuing items', true);
          }
        }
      });
    });
  }

  function startFigmaBackgroundPolling(totalItems, destText) {
    var pollCount = 0;
    var maxPolls = 300; // 5 minutes max

    var pollInterval = setInterval(function() {
      pollCount++;
      
      if (pollCount > maxPolls) {
        clearInterval(pollInterval);
        finishFigmaBackgroundSync('Sync timed out', true);
        return;
      }

      jQuery.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
          action: 'lsp_figma_background_status',
          _wpnonce: LIGHTSYNCPRO.nonce
        },
        success: function(response) {
          if (!response.success) {
            return;
          }

          var data = response.data || {};
          var synced = data.synced || 0;
          var total = data.total || totalItems;
          var running = data.running;

          // Update progress UI
          var statusEl = document.getElementById('lsp-figma-bg-status');
          var barEl = document.getElementById('lsp-figma-bg-bar');
          
          if (statusEl && total > 0) {
            statusEl.textContent = 'Synced ' + synced + ' of ' + total + ' element(s)...';
          }
          if (barEl && total > 0) {
            var pct = Math.round((synced / total) * 100);
            barEl.style.width = pct + '%';
          }

          if (!running) {
            // Sync complete
            clearInterval(pollInterval);
            
            var msg = 'Synced ' + synced + ' element(s) to ' + destText;
            finishFigmaBackgroundSync(msg, false);
          }
        },
        error: function() {
          // Ignore polling errors, keep trying
        }
      });
    }, 1000); // Poll every second
  }

  function finishFigmaBackgroundSync(message, isError) {
    var progressEl = document.getElementById('lsp-figma-bg-progress');
    if (progressEl) {
      var statusEl = document.getElementById('lsp-figma-bg-status');
      var barEl = document.getElementById('lsp-figma-bg-bar');
      var spinnerEl = progressEl.querySelector('.dashicons-update');
      
      if (spinnerEl) {
        spinnerEl.style.animation = 'none';
        spinnerEl.className = isError ? 'dashicons dashicons-warning' : 'dashicons dashicons-yes';
        spinnerEl.style.color = isError ? '#ef4444' : '#10b981';
      }
      if (statusEl) {
        statusEl.textContent = message;
        statusEl.style.color = isError ? '#ef4444' : '#10b981';
      }
      if (barEl) {
        barEl.style.width = '100%';
        barEl.style.background = isError ? '#ef4444' : '#10b981';
      }
      
      // Remove after delay and refresh page (like Canva)
      setTimeout(function() {
        if (progressEl.parentNode) {
          progressEl.parentNode.removeChild(progressEl);
        }
        // Refresh page to update header, activity log, and synced badges
        window.location.reload();
      }, 2500);
    } else {
      // No progress element, just refresh after short delay
      setTimeout(function() {
        window.location.reload();
      }, 2500);
    }
  }

  // ============================================
  // Dropbox Sync Handler
  // ============================================
  function triggerDropboxSync() {
    // Get selected Dropbox files
    var selectedIds = window.lspDropboxSelectedFiles ? window.lspDropboxSelectedFiles.slice() : [];

    if (!selectedIds.length) {
      if (typeof window.lspModal === 'function') {
        window.lspModal({
          title: 'No Files Selected',
          body: 'Please select at least one Dropbox file to sync.',
          type: 'warning'
        });
      }
      return;
    }

    // Show choice modal for foreground/background
    showDropboxSyncChoiceModal(selectedIds.length, function(mode) {
      if (mode === 'foreground') {
        startDropboxForegroundSync(selectedIds);
      } else if (mode === 'background') {
        startDropboxBackgroundSync(selectedIds);
      }
      // null = cancelled, do nothing
    });
  }

  function showDropboxSyncChoiceModal(fileCount, callback) {
    // Get sync target
    var syncTargetRadio = document.querySelector('input[name="lsp_dropbox_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    var destText = 'WordPress';

    var existing = document.getElementById('lsp-dropbox-sync-choice-modal');
    if (existing) existing.remove();

    // Dropbox icon SVG
    var dropboxIcon = '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 2L6 6l6 4-6 4 6 4 6-4-6-4 6-4-6-4z"/><path d="M6 14l6 4 6-4"/></svg>';

    var modalHtml = 
      '<div id="lsp-dropbox-sync-choice-modal" class="lsp-modal" aria-hidden="false" style="display:block;position:fixed;inset:0;z-index:10060;">' +
        '<div class="lsp-modal-backdrop" data-lsp-close style="position:absolute;inset:0;background:rgba(15,23,42,.55);backdrop-filter:blur(6px);"></div>' +
        '<div class="lsp-modal-card" style="position:absolute;left:50%;top:18%;transform:translateX(-50%);width:min(520px,calc(100% - 28px));background:#fff;border:1px solid #e5e7eb;border-radius:18px;box-shadow:0 30px 80px rgba(15,23,42,.35);padding:20px;">' +
          '<div class="lsp-modal-head" style="display:flex;gap:12px;align-items:center;margin-bottom:16px;">' +
            '<div style="width:40px;height:40px;border-radius:12px;display:grid;place-items:center;background:rgba(8,145,178,.14);border:1px solid rgba(8,145,178,.3);color:#0891b2;">' +
              dropboxIcon +
            '</div>' +
            '<h3 style="margin:0;font-size:17px;font-weight:700;">Choose Sync Method</h3>' +
          '</div>' +
          '<div class="lsp-modal-body">' +
            '<p style="margin:0 0 16px;color:#64748b;">' +
              '<strong style="color:#0f172a;">' + fileCount + ' file(s)</strong> to sync to <strong style="color:#0f172a;">' + destText + '</strong>. ' +
              'Already-synced images will be skipped automatically.' +
            '</p>' +
            '<div style="display:grid;gap:12px;">' +
              '<label class="lsp-sync-option" data-mode="background" style="display:flex;gap:12px;padding:14px;border:2px solid #0891b2;border-radius:12px;cursor:pointer;background:rgba(8,145,178,.08);transition:all .15s;">' +
                '<input type="radio" name="dropbox_sync_mode" value="background" checked style="margin-top:3px;accent-color:#0891b2;">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Sync in Background (Recommended)</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Reliable sync that handles large files and slow connections. Progress shown in corner.' +
                  '</span>' +
                '</div>' +
              '</label>' +
              '<label class="lsp-sync-option" data-mode="foreground" style="display:flex;gap:12px;padding:14px;border:2px solid #e5e7eb;border-radius:12px;cursor:pointer;background:#fff;transition:all .15s;">' +
                '<input type="radio" name="dropbox_sync_mode" value="foreground" style="margin-top:3px;accent-color:#0891b2;">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Sync Now (Foreground)</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Real-time progress. May timeout on mobile or slow connections.' +
                  '</span>' +
                '</div>' +
              '</label>' +
            '</div>' +
          '</div>' +
          '<div class="lsp-modal-actions" style="display:flex;gap:10px;justify-content:flex-end;margin-top:20px;">' +
            '<button type="button" class="btn ghost" data-lsp-close style="padding:10px 18px;border-radius:10px;border:1px solid #e5e7eb;background:#fff;cursor:pointer;font-weight:600;">Cancel</button>' +
            '<button type="button" class="btn primary" id="lsp-dropbox-sync-choice-confirm" style="padding:10px 18px;border-radius:10px;border:none;background:#0891b2;color:#fff;cursor:pointer;font-weight:600;">Start Sync</button>' +
          '</div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    var modal = document.getElementById('lsp-dropbox-sync-choice-modal');
    var options = modal.querySelectorAll('.lsp-sync-option');
    var radios = modal.querySelectorAll('input[name="dropbox_sync_mode"]');

    // Highlight selected option on change
    radios.forEach(function(radio) {
      radio.addEventListener('change', function() {
        options.forEach(function(opt) {
          if (opt.querySelector('input').checked) {
            opt.style.borderColor = '#0891b2';
            opt.style.background = 'rgba(8,145,178,.08)';
          } else {
            opt.style.borderColor = '#e5e7eb';
            opt.style.background = '#fff';
          }
        });
      });
    });

    // Hover effects
    options.forEach(function(opt) {
      opt.addEventListener('mouseenter', function() {
        if (!opt.querySelector('input').checked) {
          opt.style.borderColor = '#67e8f9';
          opt.style.background = '#f8fafc';
        }
      });
      opt.addEventListener('mouseleave', function() {
        if (!opt.querySelector('input').checked) {
          opt.style.borderColor = '#e5e7eb';
          opt.style.background = '#fff';
        }
      });
    });

    // Close handlers
    modal.querySelectorAll('[data-lsp-close]').forEach(function(el) {
      el.addEventListener('click', function() {
        modal.remove();
        callback(null);
      });
    });

    // ESC key to close
    function handleEsc(e) {
      if (e.key === 'Escape') {
        modal.remove();
        document.removeEventListener('keydown', handleEsc);
        callback(null);
      }
    }
    document.addEventListener('keydown', handleEsc);

    // Confirm handler
    var confirmBtn = document.getElementById('lsp-dropbox-sync-choice-confirm');
    if (confirmBtn) {
      confirmBtn.addEventListener('click', function() {
        var checked = modal.querySelector('input[name="dropbox_sync_mode"]:checked');
        var mode = checked ? checked.value : 'foreground';
        modal.remove();
        document.removeEventListener('keydown', handleEsc);
        callback(mode);
      });
    }
  }

  function startDropboxForegroundSync(selectedIds) {
    var syncTargetRadio = document.querySelector('input[name="lsp_dropbox_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    
    // Hub uses the same sync flow as WordPress - backend handles it
    var destText = syncTarget === 'shopify' ? 'Shopify' : (syncTarget === 'both' ? 'WordPress + Shopify' : 'WordPress');

    // Build file info from our stored data
    var files = [];
    var images = window.lspDropboxImages || [];
    selectedIds.forEach(function(id) {
      var img = images.find(function(i) { return i.id === id; });
      if (img) {
        files.push({
          id: img.id,
          path: img.path,
          name: img.name
        });
      }
    });

    if (!files.length) {
      alert('Could not find file info for selected files');
      return;
    }

    // Show syncing state on button
    btn.disabled = true;
    btn.innerHTML = '<span class="dashicons dashicons-update" style="animation:spin 1s linear infinite;"></span> Syncing...';

    // Dropbox icon SVG (matching the source tab icon)
    var dropboxIcon = '<svg width="22" height="22" viewBox="0 0 24 24" fill="none" stroke="#0891b2" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 2L6 6l6 4-6 4 6 4 6-4-6-4 6-4-6-4z"/><path d="M6 14l6 4 6-4"/></svg>';

    // Create progress modal with Dropbox branding
    var existingModal = document.getElementById('lsp-dropbox-progress-modal');
    if (existingModal) existingModal.remove();

    var modalHtml = 
      '<div id="lsp-dropbox-progress-modal" style="position:fixed;inset:0;z-index:100000;display:flex;align-items:center;justify-content:center;background:rgba(15,23,42,.6);backdrop-filter:blur(4px);">' +
        '<div style="background:#fff;border-radius:20px;padding:32px 40px;max-width:420px;width:90%;box-shadow:0 25px 60px rgba(0,0,0,.3);text-align:center;">' +
          '<div style="margin-bottom:20px;">' +
            '<div id="lsp-dropbox-progress-ring" style="position:relative;width:120px;height:120px;margin:0 auto;">' +
              '<svg viewBox="0 0 100 100" style="transform:rotate(-90deg);width:100%;height:100%;">' +
                '<circle cx="50" cy="50" r="45" fill="none" stroke="#e5e7eb" stroke-width="8"/>' +
                '<circle id="lsp-dropbox-progress-circle" cx="50" cy="50" r="45" fill="none" stroke="#0891b2" stroke-width="8" stroke-linecap="round" stroke-dasharray="283" stroke-dashoffset="283" style="transition:stroke-dashoffset 0.3s ease;"/>' +
              '</svg>' +
              '<div id="lsp-dropbox-progress-pct" style="position:absolute;inset:0;display:flex;align-items:center;justify-content:center;font-size:28px;font-weight:700;color:#1e293b;">0%</div>' +
            '</div>' +
          '</div>' +
          '<div style="display:flex;align-items:center;justify-content:center;gap:8px;margin-bottom:8px;">' +
            dropboxIcon +
            '<h3 id="lsp-dropbox-progress-title" style="margin:0;font-size:18px;font-weight:600;color:#1e293b;">Syncing Dropbox</h3>' +
          '</div>' +
          '<p id="lsp-dropbox-progress-sub" style="margin:0 0 16px;font-size:14px;color:#64748b;">Preparing ' + files.length + ' file(s) → ' + destText + '</p>' +
          '<div id="lsp-dropbox-progress-log" style="max-height:150px;overflow-y:auto;text-align:left;font-family:ui-monospace,monospace;font-size:11px;background:#f8fafc;border:1px solid #e2e8f0;border-radius:8px;padding:10px;color:#475569;"></div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    var modal = document.getElementById('lsp-dropbox-progress-modal');
    var progressCircle = document.getElementById('lsp-dropbox-progress-circle');
    var progressPct = document.getElementById('lsp-dropbox-progress-pct');
    var progressSub = document.getElementById('lsp-dropbox-progress-sub');
    var progressLog = document.getElementById('lsp-dropbox-progress-log');

    var total = files.length;
    var synced = 0;
    var skipped = 0;
    var errors = 0;
    var syncedIds = [];
    var firstFileName = files[0] ? files[0].name : 'file';

    // Smooth progress animation
    var visualPct = 0;
    var targetPct = 0;
    var ticker = null;

    function setProgress(pct) {
      pct = Math.max(0, Math.min(100, Math.round(pct)));
      var offset = 283 - (283 * pct / 100);
      progressCircle.style.strokeDashoffset = offset;
      progressPct.textContent = pct + '%';
    }

    function easeTowardTarget() {
      var delta = targetPct - visualPct;
      if (Math.abs(delta) < 0.3) {
        visualPct = targetPct;
        setProgress(visualPct);
        return;
      }
      var step = Math.max(0.4, Math.abs(delta) * 0.12);
      visualPct = visualPct + (delta > 0 ? step : -step);
      if ((delta > 0 && visualPct > targetPct) || (delta < 0 && visualPct < targetPct)) {
        visualPct = targetPct;
      }
      setProgress(visualPct);
    }

    function startTicker() {
      if (!ticker) ticker = setInterval(easeTowardTarget, 100);
    }

    function stopTicker() {
      if (ticker) { clearInterval(ticker); ticker = null; }
    }

    function logMessage(msg) {
      var time = new Date().toLocaleTimeString();
      progressLog.innerHTML += '<div>[' + time + '] ' + msg + '</div>';
      progressLog.scrollTop = progressLog.scrollHeight;
    }

    function finishSync() {
      stopTicker();
      targetPct = 100;
      setProgress(100);
      
      // Update last sync timestamp in header
      var lastSyncEl = document.getElementById('lsp-dropbox-last-sync');
      if (lastSyncEl) {
        lastSyncEl.textContent = 'Just now';
      }
      
      // Also update global last sync header
      var globalLastSync = document.getElementById('lsp-last-sync-value');
      if (globalLastSync) {
        globalLastSync.textContent = 'Just now';
      }
      
      var actualNewSyncs = synced; // Files that were actually newly synced (not skipped)
      var totalProcessed = synced + skipped;
      
      if (totalProcessed > 0) {
        // Build result message
        var msg = '';
        if (synced > 0) {
          msg = '✓ Synced ' + synced + ' new file(s)';
        }
        if (skipped > 0) {
          msg += (msg ? ', ' : '') + skipped + ' already synced';
        }
        if (errors > 0) {
          msg += ', ' + errors + ' failed';
        }
        logMessage(msg);
        
        // Update subtitle
        if (synced > 0) {
          progressSub.textContent = synced + ' file(s) synced to ' + destText;
          progressSub.style.color = '#10b981';
          progressCircle.style.stroke = '#10b981';
        } else if (skipped > 0) {
          progressSub.textContent = 'All files already synced';
          progressSub.style.color = '#64748b';
          progressCircle.style.stroke = '#64748b';
        }
        
        // Only show celebration for actual NEW syncs (not skipped)
        if (actualNewSyncs > 0 && typeof window.lspShowCelebration === 'function') {
          window.lspShowCelebration('dropbox', firstFileName, actualNewSyncs);
        }
        
        // Only update KPIs for actual new syncs
        if (actualNewSyncs > 0) {
          var photoCountEl = document.getElementById('lsp-kpi-photos-value');
          if (photoCountEl) {
            var currentCount = parseInt(photoCountEl.textContent.replace(/,/g, ''), 10) || 0;
            photoCountEl.textContent = (currentCount + actualNewSyncs).toLocaleString();
          }
          
          var monthCountEl = document.getElementById('lsp-kpi-month-value');
          if (monthCountEl) {
            var monthCount = parseInt(monthCountEl.textContent.replace(/,/g, ''), 10) || 0;
            monthCountEl.textContent = (monthCount + actualNewSyncs).toLocaleString();
          }
        }
      } else if (errors > 0) {
        // Only errors, no successes
        logMessage('✗ Sync failed: ' + errors + ' file(s) could not be synced');
        progressSub.textContent = 'Failed - see log for details';
        progressSub.style.color = '#ef4444';
        progressCircle.style.stroke = '#ef4444';
      }

      // Clear selection
      window.lspDropboxSelectedFiles = [];
      
      // Mark files as synced in UI with current UTC timestamp
      var now = Math.floor(Date.now() / 1000);
      if (!window.lspDropboxSynced) {
        window.lspDropboxSynced = {};
      }
      syncedIds.forEach(function(id) {
        window.lspDropboxSynced[id] = now;
      });

      // Close modal and reload page to get fresh data
      setTimeout(function() {
        modal.remove();
        btn.disabled = false;
        btn.innerHTML = '<span class="dashicons dashicons-update"></span> Sync Now';
        
        // Reload page to get fresh synced data, activity, and timestamps
        window.location.reload();
      }, (synced > 0 || skipped > 0) ? 2500 : 4000);
    }

    function syncNextFile(index) {
      if (index >= files.length) {
        finishSync();
        return;
      }

      var file = files[index];
      targetPct = ((index + 0.3) / total) * 100;
      progressSub.textContent = 'Syncing ' + file.name + ' (' + (index + 1) + '/' + total + ')';
      logMessage('Downloading ' + file.name + '...');

      var body = new FormData();
      body.set('action', 'lsp_dropbox_sync_single');
      body.set('_wpnonce', LIGHTSYNCPRO.nonce);
      body.set('file_id', file.id);
      body.set('file_path', file.path);
      body.set('file_name', file.name);
      body.set('sync_target', syncTarget);

      fetch(ajaxurl, {
        method: 'POST',
        body: body
      })
      .then(function(r) { return r.json(); })
      .then(function(resp) {
        targetPct = ((index + 1) / total) * 100;
        
        if (resp.success && resp.data && (resp.data.wp_id || resp.data.hub_synced || resp.data.shopify_id || resp.data.shopify_skipped)) {
          syncedIds.push(file.id);
          var outputName = resp.data.file_name || file.name;
          
          // Check if this was a skip (already synced) or actual new sync
          if (resp.data.wp_skipped && !resp.data.shopify_id) {
            skipped++;
            logMessage('⊘ ' + file.name + ' (already synced)');
          } else if (resp.data.shopify_skipped && !resp.data.wp_id) {
            skipped++;
            logMessage('⊘ ' + file.name + ' (already on Shopify)');
          } else {
            synced++;
            logMessage('✓ ' + file.name + ' → ' + outputName);
          }
        } else {
          errors++;
          var errMsg = (resp.data && resp.data.error) ? resp.data.error : 'Unknown error';
          logMessage('✗ ' + file.name + ': ' + errMsg);
        }
        syncNextFile(index + 1);
      })
      .catch(function(err) {
        targetPct = ((index + 1) / total) * 100;
        errors++;
        logMessage('✗ ' + file.name + ': Network error');
        syncNextFile(index + 1);
      });
    }

    logMessage('Starting Dropbox sync for ' + total + ' file(s) to ' + destText);
    startTicker();
    syncNextFile(0);
  }

  // Quick sync for small batches - processes files one by one with corner progress
  function startDropboxQuickSync(selectedIds) {
    var syncTargetRadio = document.querySelector('input[name="lsp_dropbox_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    var destText = syncTarget === 'shopify' ? 'Shopify' : (syncTarget === 'both' ? 'WordPress + Shopify' : 'WordPress');

    // Build file info from our stored data
    var files = [];
    var images = window.lspDropboxImages || [];
    
    selectedIds.forEach(function(id) {
      var img = images.find(function(i) { return i.id === id; });
      if (img) {
        files.push({
          id: img.id,
          path: img.path,
          name: img.name
        });
      }
    });

    if (!files.length) {
      alert('Could not find file info for selected files');
      return;
    }

    var total = files.length;
    var synced = 0;
    var errors = [];

    // Show syncing state on button
    btn.disabled = true;
    btn.innerHTML = '<span class="dashicons dashicons-update" style="animation:spin 1s linear infinite;"></span> Syncing...';

    // Dropbox icon
    var dropboxIcon = '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#0891b2" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="flex-shrink:0;"><path d="M12 2L6 6l6 4-6 4 6 4 6-4-6-4 6-4-6-4z"/><path d="M6 14l6 4 6-4"/></svg>';

    // Show progress indicator
    var existingProgress = document.getElementById('lsp-dropbox-bg-progress');
    if (existingProgress) existingProgress.remove();

    var progressHtml = 
      '<div id="lsp-dropbox-bg-progress" style="position:fixed;bottom:20px;right:20px;background:#fff;border-radius:12px;padding:16px 20px;box-shadow:0 4px 20px rgba(0,0,0,.15);z-index:10000;min-width:280px;">' +
        '<div style="display:flex;align-items:center;gap:10px;margin-bottom:8px;">' +
          dropboxIcon +
          '<span style="font-weight:600;color:#1e293b;">Syncing Dropbox</span>' +
          '<span class="dashicons dashicons-update" style="animation:spin 1s linear infinite;color:#0891b2;margin-left:auto;font-size:16px;"></span>' +
        '</div>' +
        '<div id="lsp-dropbox-bg-status" style="font-size:13px;color:#64748b;">Starting sync to ' + destText + '...</div>' +
        '<div id="lsp-dropbox-bg-filename" style="font-size:11px;color:#94a3b8;margin-top:4px;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;"></div>' +
        '<div style="height:4px;background:#e5e7eb;border-radius:2px;margin-top:10px;overflow:hidden;">' +
          '<div id="lsp-dropbox-bg-bar" style="height:100%;width:0%;background:#0891b2;transition:width 0.3s;"></div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', progressHtml);

    // Process files sequentially
    function processNext(index) {
      if (index >= files.length) {
        // All done!
        var msg = 'Synced ' + synced + ' of ' + total + ' file(s) to ' + destText;
        if (errors.length > 0) {
          msg += ' (' + errors.length + ' failed)';
        }
        finishDropboxBackgroundSync(msg, errors.length > 0 && synced === 0);
        
        // Mark files as synced in UI
        var now = Math.floor(Date.now() / 1000);
        files.forEach(function(f) {
          if (window.lspDropboxSynced) {
            window.lspDropboxSynced[f.id] = { time: now, dest: syncTarget };
          }
        });
        
        // Clear selection
        window.lspDropboxSelectedFiles = [];
        return;
      }

      var file = files[index];
      var pct = Math.round((index / total) * 100);

      // Update progress
      var statusEl = document.getElementById('lsp-dropbox-bg-status');
      var filenameEl = document.getElementById('lsp-dropbox-bg-filename');
      var barEl = document.getElementById('lsp-dropbox-bg-bar');
      
      if (statusEl) statusEl.textContent = 'Syncing ' + (index + 1) + ' of ' + total + '...';
      if (filenameEl) filenameEl.textContent = file.name;
      if (barEl) barEl.style.width = pct + '%';

      // Sync this file
      jQuery.ajax({
        url: ajaxurl,
        type: 'POST',
        timeout: 180000, // 3 minute timeout per file
        data: {
          action: 'lsp_dropbox_sync_single',
          _wpnonce: LIGHTSYNCPRO.nonce,
          file_id: file.id,
          file_path: file.path,
          file_name: file.name,
          sync_target: syncTarget
        },
        success: function(response) {
          if (response.success) {
            synced++;
          } else {
            errors.push(file.name + ': ' + (response.data && response.data.error ? response.data.error : 'Unknown error'));
          }
          // Process next file
          processNext(index + 1);
        },
        error: function(xhr, status, error) {
          errors.push(file.name + ': ' + (status === 'timeout' ? 'Timeout - file may be too large' : error));
          // Continue with next file
          processNext(index + 1);
        }
      });
    }

    // Start processing
    processNext(0);
  }

  function startDropboxBackgroundSync(selectedIds) {
    // Use quick sync for ≤10 files (faster, immediate feedback)
    // Use background queue for >10 files (safer for bulk imports)
    if (selectedIds.length <= 10) {
      startDropboxQuickSync(selectedIds);
      return;
    }

    var syncTargetRadio = document.querySelector('input[name="lsp_dropbox_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    var destText = 'WordPress';

    // Build file info from our stored data
    var fileIds = [];
    var filePaths = [];
    var fileNames = [];
    var images = window.lspDropboxImages || [];
    
    selectedIds.forEach(function(id) {
      var img = images.find(function(i) { return i.id === id; });
      if (img) {
        fileIds.push(img.id);
        filePaths.push(img.path);
        fileNames.push(img.name);
      }
    });

    if (!fileIds.length) {
      alert('Could not find file info for selected files');
      return;
    }

    // Show syncing state on button
    btn.disabled = true;
    btn.innerHTML = '<span class="dashicons dashicons-update" style="animation:spin 1s linear infinite;"></span> Syncing...';

    // Dropbox icon
    var dropboxIcon = '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#0891b2" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="flex-shrink:0;"><path d="M12 2L6 6l6 4-6 4 6 4 6-4-6-4 6-4-6-4z"/><path d="M6 14l6 4 6-4"/></svg>';

    // Show background progress indicator with icon
    var existingProgress = document.getElementById('lsp-dropbox-bg-progress');
    if (existingProgress) existingProgress.remove();

    var progressHtml = 
      '<div id="lsp-dropbox-bg-progress" style="position:fixed;bottom:20px;right:20px;background:#fff;border-radius:12px;padding:16px 20px;box-shadow:0 4px 20px rgba(0,0,0,.15);z-index:10000;min-width:280px;">' +
        '<div style="display:flex;align-items:center;gap:10px;margin-bottom:8px;">' +
          dropboxIcon +
          '<span style="font-weight:600;color:#1e293b;">Syncing Dropbox</span>' +
          '<span class="dashicons dashicons-update" style="animation:spin 1s linear infinite;color:#0891b2;margin-left:auto;font-size:16px;"></span>' +
        '</div>' +
        '<div id="lsp-dropbox-bg-status" style="font-size:13px;color:#64748b;">Queuing ' + fileIds.length + ' file(s) → ' + destText + '</div>' +
        '<div style="height:4px;background:#e5e7eb;border-radius:2px;margin-top:10px;overflow:hidden;">' +
          '<div id="lsp-dropbox-bg-bar" style="height:100%;width:0%;background:#0891b2;transition:width 0.3s;"></div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', progressHtml);

    // Queue files for background sync
    var body = new FormData();
    body.set('action', 'lsp_dropbox_sync_files');
    body.set('_wpnonce', LIGHTSYNCPRO.nonce);
    body.set('sync_target', syncTarget);
    fileIds.forEach(function(id) { body.append('file_ids[]', id); });
    filePaths.forEach(function(p) { body.append('file_paths[]', p); });
    fileNames.forEach(function(n) { body.append('file_names[]', n); });

    fetch(ajaxurl, {
      method: 'POST',
      body: body
    })
    .then(function(r) { return r.json(); })
    .then(function(resp) {
      if (!resp.success) {
        finishDropboxBackgroundSync('Failed to queue: ' + (resp.data && resp.data.error ? resp.data.error : 'Unknown error'), true);
        return;
      }

      // Clear selection
      window.lspDropboxSelectedFiles = [];
      
      // Update status
      var statusEl = document.getElementById('lsp-dropbox-bg-status');
      if (statusEl) {
        statusEl.textContent = 'Queued ' + fileIds.length + ' file(s). Processing...';
      }
      
      // Start polling for background status
      startDropboxBackgroundPolling(fileIds.length, destText);
    })
    .catch(function(err) {
      finishDropboxBackgroundSync('Network error: ' + err.message, true);
    });
  }

  function startDropboxBackgroundPolling(totalItems, destText) {
    var pollCount = 0;
    var maxPolls = 600; // 10 minutes max
    var isProcessing = false; // Prevent parallel processing calls

    var pollInterval = setInterval(function() {
      pollCount++;
      
      if (pollCount > maxPolls) {
        clearInterval(pollInterval);
        finishDropboxBackgroundSync('Sync timed out', true);
        return;
      }

      // First, get status (lightweight)
      jQuery.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
          action: 'lsp_dropbox_background_status',
          _wpnonce: LIGHTSYNCPRO.nonce
        },
        success: function(response) {
          if (!response.success) {
            return;
          }

          var data = response.data || {};
          var synced = data.synced || 0;
          var total = data.total || totalItems;
          var running = data.running;
          var needsProcess = data.needs_process;

          // Update progress UI
          var statusEl = document.getElementById('lsp-dropbox-bg-status');
          var barEl = document.getElementById('lsp-dropbox-bg-bar');
          
          if (statusEl && total > 0) {
            statusEl.textContent = 'Synced ' + synced + ' of ' + total + ' file(s)...';
          }
          if (barEl && total > 0) {
            var pct = Math.round((synced / total) * 100);
            barEl.style.width = pct + '%';
          }

          if (!running && !needsProcess) {
            // Sync complete
            clearInterval(pollInterval);
            
            var msg = 'Synced ' + synced + ' file(s) to ' + destText;
            finishDropboxBackgroundSync(msg, false);
            
            // Mark files as synced in UI
            if (data.file_ids && window.lspDropboxSynced) {
              var now = Math.floor(Date.now() / 1000);
              data.file_ids.forEach(function(id) {
                window.lspDropboxSynced[id] = now;
              });
            }
          } else if (needsProcess && !isProcessing) {
            // Trigger processing of next file
            isProcessing = true;
            jQuery.ajax({
              url: ajaxurl,
              type: 'POST',
              timeout: 180000, // 3 minute timeout for processing
              data: {
                action: 'lsp_dropbox_process_next',
                _wpnonce: LIGHTSYNCPRO.nonce
              },
              success: function(processResp) {
                isProcessing = false;
                if (processResp.success && processResp.data) {
                  var pdata = processResp.data;
                  if (pdata.processed && statusEl) {
                    statusEl.textContent = 'Synced: ' + pdata.file_name;
                  } else if (pdata.error && statusEl) {
                    statusEl.textContent = 'Error: ' + pdata.file_name + ' - ' + pdata.error;
                  }
                }
              },
              error: function(xhr, status, error) {
                isProcessing = false;
                // Processing timed out or failed - the item was already removed from queue
                // Continue polling, it will retry or show error
                var statusEl = document.getElementById('lsp-dropbox-bg-status');
                if (statusEl) {
                  statusEl.textContent = 'Processing... (retrying)';
                }
              }
            });
          }
        },
        error: function() {
          // Ignore polling errors, keep trying
        }
      });
    }, 2000); // Poll every 2 seconds
  }

  function finishDropboxBackgroundSync(message, isError) {
    var progressEl = document.getElementById('lsp-dropbox-bg-progress');
    if (progressEl) {
      var statusEl = document.getElementById('lsp-dropbox-bg-status');
      var filenameEl = document.getElementById('lsp-dropbox-bg-filename');
      var barEl = document.getElementById('lsp-dropbox-bg-bar');
      var spinnerEl = progressEl.querySelector('.dashicons-update');
      
      if (spinnerEl) {
        spinnerEl.style.animation = 'none';
        spinnerEl.className = isError ? 'dashicons dashicons-warning' : 'dashicons dashicons-yes';
        spinnerEl.style.color = isError ? '#ef4444' : '#10b981';
      }
      if (statusEl) {
        statusEl.textContent = message;
        statusEl.style.color = isError ? '#ef4444' : '#10b981';
      }
      if (filenameEl) {
        filenameEl.textContent = ''; // Clear filename on finish
      }
      if (barEl) {
        barEl.style.width = '100%';
        barEl.style.background = isError ? '#ef4444' : '#10b981';
      }
      
      // Log error to console so user can see it
      if (isError) {
        console.error('[LSP Dropbox Sync] Error:', message);
      }
      
      // Wait longer on error so user can read the message
      var delay = isError ? 6000 : 2500;
      
      // Remove after delay and refresh page
      setTimeout(function() {
        if (progressEl.parentNode) {
          progressEl.parentNode.removeChild(progressEl);
        }
        // Refresh page to update header, activity log, and synced badges
        window.location.reload();
      }, delay);
    } else {
      // No progress element, just refresh after short delay
      setTimeout(function() {
        window.location.reload();
      }, 2500);
    }
  }

  // ============================================
  // Close Progress Modal Helper
  // ============================================
  function lspCloseProgressModal() {
    var m = document.getElementById('lsp-modal');
    if (m) m.setAttribute('aria-hidden', 'true');
    document.body.classList.remove('lsp-modal-open');
  }

  function lspCloseAndRestoreProgress() {
    lspCloseProgressModal();

    var wrap = document.getElementById('lsp-progress');
    var dock = document.getElementById('lsp-progress-dock');
    if (wrap && dock) {
      dock.appendChild(wrap);
      wrap.style.display = 'none';
    }
  }

  // ============================================
  // Foreground Sync (Existing Logic)
  // ============================================
  function startForegroundSync(catalog, albums, total) {
    // Get current sync target from radio
    var lrSyncTargetRadio = document.querySelector('input[name$="[sync_target]"]:checked');
    var syncTarget = lrSyncTargetRadio ? lrSyncTargetRadio.value : 'wp';
    
    // Create floating progress modal (like Canva)
    var existingModal = document.getElementById('lsp-lr-progress-modal');
    if (existingModal) existingModal.remove();

    var modalHtml = 
      '<div id="lsp-lr-progress-modal" style="position:fixed;inset:0;z-index:100000;display:flex;align-items:center;justify-content:center;background:rgba(15,23,42,.6);backdrop-filter:blur(4px);">' +
        '<div style="background:#fff;border-radius:20px;padding:32px 40px;max-width:460px;width:90%;box-shadow:0 25px 60px rgba(0,0,0,.3);text-align:center;">' +
          '<div style="margin-bottom:20px;">' +
            '<div id="lsp-lr-progress-ring" style="position:relative;width:120px;height:120px;margin:0 auto;">' +
              '<svg viewBox="0 0 100 100" style="transform:rotate(-90deg);width:100%;height:100%;">' +
                '<circle cx="50" cy="50" r="45" fill="none" stroke="#e5e7eb" stroke-width="8"/>' +
                '<circle id="lsp-lr-progress-circle" cx="50" cy="50" r="45" fill="none" stroke="#3b82f6" stroke-width="8" stroke-linecap="round" stroke-dasharray="283" stroke-dashoffset="283" style="transition:stroke-dashoffset 0.3s ease;"/>' +
              '</svg>' +
              '<div id="lsp-lr-progress-pct" style="position:absolute;inset:0;display:flex;align-items:center;justify-content:center;font-size:28px;font-weight:700;color:#1e293b;">0%</div>' +
            '</div>' +
          '</div>' +
          '<h3 id="lsp-lr-progress-title" style="margin:0 0 8px;font-size:18px;font-weight:600;color:#1e293b;">Syncing Lightroom</h3>' +
          '<p id="lsp-lr-progress-sub" style="margin:0 0 16px;font-size:14px;color:#64748b;">' + (total ? 'Checking ' + total + ' photos...' : 'Starting sync...') + '</p>' +
          '<div id="lsp-lr-progress-log" style="max-height:150px;overflow-y:auto;text-align:left;font-family:ui-monospace,monospace;font-size:11px;background:#f8fafc;border:1px solid #e2e8f0;border-radius:8px;padding:10px;color:#475569;"></div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    // Store references for batch runner
    window._lspLrModal = {
      modal: document.getElementById('lsp-lr-progress-modal'),
      circle: document.getElementById('lsp-lr-progress-circle'),
      pct: document.getElementById('lsp-lr-progress-pct'),
      sub: document.getElementById('lsp-lr-progress-sub'),
      log: document.getElementById('lsp-lr-progress-log')
    };

    runBatches(catalog, albums, total, syncTarget);
  }

  // Helper to update the floating modal progress
  function updateLrModalProgress(pct, subText) {
    if (!window._lspLrModal) return;
    
    pct = Math.max(0, Math.min(100, Math.round(pct)));
    var offset = 283 - (283 * pct / 100);
    
    if (window._lspLrModal.circle) {
      window._lspLrModal.circle.style.strokeDashoffset = offset;
    }
    if (window._lspLrModal.pct) {
      window._lspLrModal.pct.textContent = pct + '%';
    }
    if (subText && window._lspLrModal.sub) {
      window._lspLrModal.sub.textContent = subText;
    }
  }

  function logToLrModal(msg) {
    if (!window._lspLrModal || !window._lspLrModal.log) return;
    var time = new Date().toLocaleTimeString();
    window._lspLrModal.log.innerHTML += '<div>[' + time + '] ' + msg + '</div>';
    window._lspLrModal.log.scrollTop = window._lspLrModal.log.scrollHeight;
  }

  function closeLrModal(success, celebrateData) {
    if (!window._lspLrModal || !window._lspLrModal.modal) return;
    
    // Update to completion state
    updateLrModalProgress(100, success ? 'Sync complete!' : 'Sync finished');
    
    if (window._lspLrModal.circle) {
      window._lspLrModal.circle.style.stroke = success ? '#10b981' : '#f59e0b';
    }
    if (window._lspLrModal.sub) {
      window._lspLrModal.sub.style.color = success ? '#10b981' : '#f59e0b';
    }
    
    // Remove after delay
    setTimeout(function() {
      if (window._lspLrModal && window._lspLrModal.modal) {
        window._lspLrModal.modal.remove();
      }
      window._lspLrModal = null;
      
      // Show celebration for first successful sync
      if (success && celebrateData && window.lspShowCelebration) {
        var shown = window.lspShowCelebration('lightroom', celebrateData.albumName, celebrateData.count);
        if (shown) return; // Celebration handles the reload
      }
      
      // Refresh page to reload everything
      window.location.reload();
    }, 2500);
  }

  // ============================================
  // Batch Runner (Foreground)
  // ============================================
  function runBatches(catalog, albums, total, syncTarget) {
    syncTarget = syncTarget || 'wp';
    var doneReal = 0;
    var metaCum = 0;
    var pendCur = 0;
    var idx = 0;
    var cursor = '';
    var firstAlbumName = ''; // Track first album name for celebration

    var batchSz = (LIGHTSYNCPRO && LIGHTSYNCPRO.saved && LIGHTSYNCPRO.saved.batch) ?
      parseInt(LIGHTSYNCPRO.saved.batch, 10) : 200;

    var visualPct = 0;
    var targetPct = 0;
    var ticker = null;

    function easeTowardTarget() {
      var delta = targetPct - visualPct;
      if (Math.abs(delta) < 0.2) {
        visualPct = targetPct;
        updateLrModalProgress(visualPct);
        return;
      }
      var step = Math.max(0.35, Math.abs(delta) * 0.12);
      visualPct = visualPct + (delta > 0 ? step : -step);
      // clamp around target
      if ((delta > 0 && visualPct > targetPct) || (delta < 0 && visualPct < targetPct)) {
        visualPct = targetPct;
      }
      updateLrModalProgress(visualPct);
    }


    function startTicker() {
      if (!ticker) ticker = setInterval(easeTowardTarget, 120);
    }

    function stopTicker() {
      if (ticker) {
        clearInterval(ticker);
        ticker = null;
      }
    }

    function blendedPercent() {
      if (!total || total <= 0) {
        var activityBoost = Math.max(0, Math.min(25, Math.round((metaCum + (pendCur * 2)) / 10)));
        return Math.min(85, 60 + activityBoost);
      }
      var visNumerator = doneReal + 0.6 * metaCum + 0.25 * Math.max(0, pendCur);
      var visDenominator = total + 0.6 * metaCum + 0.25 * Math.max(0, pendCur);
      var pct = Math.round((visNumerator / Math.max(1, visDenominator)) * 100);
      if (pendCur > 0) pct = Math.min(pct, 95);
      return Math.max(0, Math.min(100, pct));
    }

    function updateTargetsAfterServer() {
      var blend = blendedPercent();
      targetPct = Math.max(targetPct, blend);
    }

    // Ensure progress dock exists
    (function ensureProgressDockOnce() {
      var wrap = document.getElementById('lsp-progress');
      if (!wrap) return;
      if (document.getElementById('lsp-progress-dock')) return;

      var dock = document.createElement('div');
      dock.id = 'lsp-progress-dock';
      wrap.parentNode.insertBefore(dock, wrap);
    })();

    function next() {
      var body = new URLSearchParams();
      body.set('action', 'lightsyncpro_ajax_batch');
      body.set('_ajax_nonce', LIGHTSYNCPRO.nonce);
      body.set('catalog', catalog);
      albums.forEach(function(a) {
        body.append('albums[]', a);
      });
      body.set('cursor', cursor || '');
      body.set('index', String(idx));
      body.set('batch', String(batchSz));
      body.set('sync_target', syncTarget); // Pass sync target explicitly

      var inFlightCeil = (pendCur > 0) ? 95 : 90;
      targetPct = Math.min(Math.max(targetPct, visualPct + 1), inFlightCeil);
      startTicker();

      fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      })
      .then(function(r) { return r.text(); })
      .then(function(text) {
        try {
          return JSON.parse(text);
        } catch (e) {
          var preview = text.slice(0, 300).replace(/\s+/g, ' ').trim();
          throw new Error('Non-JSON response: ' + preview);
        }
      })
      .then(function(json) {
        if (!json || json.success !== true) {
          throw new Error((json && json.data) ? json.data : 'Batch call failed');
        }
        var data = json.data || {};

        // Check for quota hit FIRST
        if (data.hit_cap && data.error_message) {
          stopTicker();
          closeLrModal(false);

          if (typeof window.lspModal === 'function') {
            window.lspModal({
              title: 'Monthly Photo Limit Reached',
              body: '<p style="margin-bottom:12px;">' + data.error_message + '</p>' +
                '<p>Need more? Upgrade to sync up to <strong>5,000 photos/month</strong> (Pro) or <strong>unlimited</strong> (Agency).</p>',
              type: 'error',
              primaryText: 'Upgrade Plan',
              primaryHref: data.upgrade_url || 'https://lightsyncpro.com/pricing'
            });
          } else {
            lspShowToast(data.error_message, 'error');
          }
          return;
        }

        // Show warning toast if approaching limit
        if (data.warning_low_quota) {
          lspShowToast(data.warning_low_quota, 'warning');
        }

        var realDelta = (typeof data.processed === 'number' && Number.isFinite(data.processed)) ?
          data.processed :
          (
            ((data.imported && data.imported.length) ? data.imported.length : 0) +
            ((data.updated && data.updated.length) ? data.updated.length : 0) +
            (data.skipped ? parseInt(data.skipped, 10) : 0)
          );
        if (!Number.isFinite(realDelta) || realDelta < 0) realDelta = 0;
        doneReal += realDelta;

        var metaN = (data.meta_updated && data.meta_updated.length) ? data.meta_updated.length : 0;
        metaCum += metaN;
        pendCur = (typeof data.renditions_pending === 'number') ? data.renditions_pending : pendCur;

        if (data.message) logToLrModal(data.message);

        cursor = (data.cursor || data.next_cursor || '');
        if (typeof data.index === 'number') idx = data.index;

        if (!total || total <= 0) {
          total = Math.max(doneReal + Math.floor(metaCum * 0.6), 1);
        }
        
        // If we've processed more than estimated, update the total
        if (doneReal > total) {
          total = doneReal + (cursor ? 5 : 0); // Add buffer if more pages coming
        }

        updateTargetsAfterServer();
        
        // Update sub text with progress
        var subText = 'Processed ' + doneReal + (total ? ' of ~' + total : '') + ' photos...';
        if (window._lspLrModal && window._lspLrModal.sub) {
          window._lspLrModal.sub.textContent = subText;
        }

        if (data.done_all || idx >= albums.length) {
          // Update last sync timestamp
          if (data.lightsync_last_sync_ts) {
            var el = document.getElementById('lsp-last-sync');
            if (el) {
              el.textContent = 'Just now';
              el.classList.add('lsp-flash');
              setTimeout(function() { el.classList.remove('lsp-flash'); }, 1200);
            }
          }

          targetPct = (pendCur > 0) ? Math.max(targetPct, 95) : 100;
          startTicker();

          setTimeout(function() {
            stopTicker();
            
            logToLrModal('✓ Sync complete - ' + doneReal + ' photos processed');
            closeLrModal(true, { albumName: firstAlbumName, count: doneReal });
          }, 700);
          return;
        }

        next();
      })
      .catch(function(err) {
        stopTicker();
        logToLrModal('✗ Error: ' + (err && err.message ? err.message : err));
        closeLrModal(false);
      });
    }

    // Initialize progress state
    logToLrModal('Starting sync...');
    if (!total || total <= 0) {
      targetPct = 60;
    } else {
      visualPct = 1;
      targetPct = 5;
      updateLrModalProgress(visualPct);
    }
    startTicker();
    next();
  }

  // ============================================
  // Shutterstock Sync Handler
  // ============================================
  function triggerShutterstockSync() {
    // Get selected Shutterstock files
    var selectedIds = window.lspShutterstockSelectedFiles ? window.lspShutterstockSelectedFiles.slice() : [];

    if (!selectedIds.length) {
      if (typeof window.lspModal === 'function') {
        window.lspModal({
          title: 'No Images Selected',
          body: 'Please select at least one Shutterstock image to sync.',
          type: 'warning'
        });
      }
      return;
    }

    // Show choice modal for foreground/background
    showShutterstockSyncChoiceModal(selectedIds.length, function(mode) {
      if (mode === 'foreground') {
        startShutterstockForegroundSync(selectedIds);
      } else if (mode === 'background') {
        startShutterstockBackgroundSync(selectedIds);
      }
      // null = cancelled, do nothing
    });
  }

  function showShutterstockSyncChoiceModal(fileCount, callback) {
    // Get sync target
    var syncTargetRadio = document.querySelector('input[name="lsp_shutterstock_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    var destText = 'WordPress';
    if (syncTarget === 'shopify') destText = 'Shopify';
    else if (syncTarget === 'both') destText = 'WordPress + Shopify';

    var existing = document.getElementById('lsp-shutterstock-sync-choice-modal');
    if (existing) existing.remove();

    // Shutterstock icon SVG
    var shutterstockIcon = '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="18" height="18" rx="2"/><path d="M9 9h6v6H9z"/><path d="M9 3v6"/><path d="M15 15v6"/></svg>';

    var modalHtml = 
      '<div id="lsp-shutterstock-sync-choice-modal" class="lsp-modal" aria-hidden="false" style="display:block;position:fixed;inset:0;z-index:10060;">' +
        '<div class="lsp-modal-backdrop" data-lsp-close style="position:absolute;inset:0;background:rgba(15,23,42,.55);backdrop-filter:blur(6px);"></div>' +
        '<div class="lsp-modal-card" style="position:absolute;left:50%;top:18%;transform:translateX(-50%);width:min(520px,calc(100% - 28px));background:#fff;border:1px solid #e5e7eb;border-radius:18px;box-shadow:0 30px 80px rgba(15,23,42,.35);padding:20px;">' +
          '<div class="lsp-modal-head" style="display:flex;gap:12px;align-items:center;margin-bottom:16px;">' +
            '<div style="width:40px;height:40px;border-radius:12px;display:grid;place-items:center;background:rgba(238,45,36,.14);border:1px solid rgba(238,45,36,.3);color:#ee2d24;">' +
              shutterstockIcon +
            '</div>' +
            '<h3 style="margin:0;font-size:17px;font-weight:700;">Choose Sync Method</h3>' +
          '</div>' +
          '<div class="lsp-modal-body">' +
            '<p style="margin:0 0 16px;color:#64748b;">' +
              '<strong style="color:#0f172a;">' + fileCount + ' image(s)</strong> to sync to <strong style="color:#0f172a;">' + destText + '</strong>. ' +
              'Already-synced images will be skipped automatically.' +
            '</p>' +
            '<div style="display:grid;gap:12px;">' +
              '<label class="lsp-sync-option" data-mode="background" style="display:flex;gap:12px;padding:14px;border:2px solid #ee2d24;border-radius:12px;cursor:pointer;background:rgba(238,45,36,.08);transition:all .15s;">' +
                '<input type="radio" name="shutterstock_sync_mode" value="background" checked style="margin-top:3px;accent-color:#ee2d24;">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Sync in Background (Recommended)</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Reliable sync that handles large files and slow connections. Progress shown in corner.' +
                  '</span>' +
                '</div>' +
              '</label>' +
              '<label class="lsp-sync-option" data-mode="foreground" style="display:flex;gap:12px;padding:14px;border:2px solid #e5e7eb;border-radius:12px;cursor:pointer;background:#fff;transition:all .15s;">' +
                '<input type="radio" name="shutterstock_sync_mode" value="foreground" style="margin-top:3px;accent-color:#ee2d24;">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Sync Now (Foreground)</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Real-time progress. May timeout on mobile or slow connections.' +
                  '</span>' +
                '</div>' +
              '</label>' +
            '</div>' +
          '</div>' +
          '<div class="lsp-modal-actions" style="display:flex;gap:10px;justify-content:flex-end;margin-top:20px;">' +
            '<button type="button" class="btn ghost" data-lsp-close style="padding:10px 18px;border-radius:10px;border:1px solid #e5e7eb;background:#fff;cursor:pointer;font-weight:600;">Cancel</button>' +
            '<button type="button" class="btn primary" id="lsp-shutterstock-sync-choice-confirm" style="padding:10px 18px;border-radius:10px;border:none;background:#ee2d24;color:#fff;cursor:pointer;font-weight:600;">Start Sync</button>' +
          '</div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    var modal = document.getElementById('lsp-shutterstock-sync-choice-modal');
    var options = modal.querySelectorAll('.lsp-sync-option');
    var radios = modal.querySelectorAll('input[name="shutterstock_sync_mode"]');

    // Highlight selected option on change
    radios.forEach(function(radio) {
      radio.addEventListener('change', function() {
        options.forEach(function(opt) {
          if (opt.querySelector('input').checked) {
            opt.style.borderColor = '#ee2d24';
            opt.style.background = 'rgba(238,45,36,.08)';
          } else {
            opt.style.borderColor = '#e5e7eb';
            opt.style.background = '#fff';
          }
        });
      });
    });

    // Hover effects
    options.forEach(function(opt) {
      opt.addEventListener('mouseenter', function() {
        if (!opt.querySelector('input').checked) {
          opt.style.borderColor = '#f87171';
          opt.style.background = '#fef2f2';
        }
      });
      opt.addEventListener('mouseleave', function() {
        if (!opt.querySelector('input').checked) {
          opt.style.borderColor = '#e5e7eb';
          opt.style.background = '#fff';
        }
      });
    });

    // Close handlers
    modal.querySelectorAll('[data-lsp-close]').forEach(function(el) {
      el.addEventListener('click', function() {
        modal.remove();
        callback(null);
      });
    });

    // Confirm
    document.getElementById('lsp-shutterstock-sync-choice-confirm').addEventListener('click', function() {
      var selectedMode = modal.querySelector('input[name="shutterstock_sync_mode"]:checked').value;
      modal.remove();
      callback(selectedMode);
    });
  }

  function startShutterstockForegroundSync(selectedIds) {
    var syncTargetRadio = document.querySelector('input[name="lsp_shutterstock_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    var destText = 'WordPress';
    if (syncTarget === 'shopify') destText = 'Shopify';
    else if (syncTarget === 'both') destText = 'WordPress + Shopify';

    var images = window.lspShutterstockImages || [];
    var total = selectedIds.length;
    var synced = 0;
    var errors = [];
    var queue = selectedIds.slice();

    // Show progress UI
    var progressEl = document.getElementById('lsp-shutterstock-progress');
    var progressTextEl = document.getElementById('lsp-shutterstock-progress-text');
    var progressPercentEl = document.getElementById('lsp-shutterstock-progress-percent');
    var progressBarEl = document.getElementById('lsp-shutterstock-progress-bar');
    
    if (progressEl) progressEl.style.display = 'block';

    function updateProgress() {
      var done = synced + errors.length;
      var pct = total > 0 ? Math.round((done / total) * 100) : 0;
      if (progressTextEl) progressTextEl.textContent = 'Syncing ' + done + ' of ' + total + '...';
      if (progressPercentEl) progressPercentEl.textContent = pct + '%';
      if (progressBarEl) progressBarEl.style.width = pct + '%';
    }

    function syncNext() {
      if (queue.length === 0) {
        // Done
        var msg = synced + ' image(s) synced to ' + destText;
        if (errors.length > 0) {
          msg += ' (' + errors.length + ' failed)';
        }
        if (progressTextEl) progressTextEl.textContent = msg;
        if (progressPercentEl) progressPercentEl.textContent = '100%';
        if (progressBarEl) progressBarEl.style.width = '100%';

        setTimeout(function() {
          if (progressEl) progressEl.style.display = 'none';
          // Clear selection and reload
          window.lspShutterstockSelectedFiles = [];
          if (typeof window.loadShutterstockLicenses === 'function') {
            window.loadShutterstockLicenses(1);
          }
          if (typeof window.updateShutterstockCommandBarSelection === 'function') {
            window.updateShutterstockCommandBarSelection();
          }
        }, 2000);
        return;
      }

      updateProgress();

      var id = queue.shift();
      var img = images.find(function(i) { return String(i.id) === String(id); });
      if (!img) {
        syncNext();
        return;
      }

      jQuery.ajax({
        url: ajaxurl,
        type: 'POST',
        timeout: 120000, // 2 minute timeout per image
        data: {
          action: 'lsp_shutterstock_sync',
          _wpnonce: LIGHTSYNCPRO.nonce,
          license_id: img.license_id,
          image_id: img.id,
          description: img.description || ''
        },
        success: function(response) {
          if (response.success) {
            synced++;
            // Update local synced state
            if (!window.lspShutterstockSynced) window.lspShutterstockSynced = {};
            window.lspShutterstockSynced[img.id] = {
              time: Math.floor(Date.now() / 1000),
              dest: syncTarget
            };
          } else {
            errors.push(img.id);
          }
          syncNext();
        },
        error: function() {
          errors.push(img.id);
          syncNext();
        }
      });
    }

    syncNext();
  }

  function startShutterstockBackgroundSync(selectedIds) {
    var syncTargetRadio = document.querySelector('input[name="lsp_shutterstock_sync_target"]:checked');
    var syncTarget = syncTargetRadio ? syncTargetRadio.value : 'wp';
    var destText = 'WordPress';
    if (syncTarget === 'shopify') destText = 'Shopify';
    else if (syncTarget === 'both') destText = 'WordPress + Shopify';

    var images = window.lspShutterstockImages || [];
    
    // Build file data for background job
    var fileData = selectedIds.map(function(id) {
      var img = images.find(function(i) { return String(i.id) === String(id); });
      return img ? {
        id: img.id,
        license_id: img.license_id,
        description: img.description || ''
      } : null;
    }).filter(function(f) { return f !== null; });

    // Create floating progress card
    var existingCard = document.getElementById('lsp-shutterstock-bg-progress');
    if (existingCard) existingCard.remove();

    var progressHtml = 
      '<div id="lsp-shutterstock-bg-progress" style="position:fixed;bottom:20px;right:20px;background:#fff;border-radius:16px;padding:20px 24px;box-shadow:0 8px 30px rgba(0,0,0,.18);z-index:100000;min-width:320px;max-width:400px;">' +
        '<div style="display:flex;align-items:center;gap:14px;margin-bottom:12px;">' +
          '<div style="width:44px;height:44px;border-radius:12px;display:grid;place-items:center;background:rgba(238,45,36,.12);border:1px solid rgba(238,45,36,.25);">' +
            '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#ee2d24" stroke-width="2"><rect x="3" y="3" width="18" height="18" rx="2"/><path d="M9 9h6v6H9z"/></svg>' +
          '</div>' +
          '<div style="flex:1;">' +
            '<div style="font-weight:600;color:#0f172a;font-size:14px;">Shutterstock Sync</div>' +
            '<div id="lsp-shutterstock-bg-status" style="font-size:12px;color:#64748b;">Starting sync to ' + destText + '...</div>' +
          '</div>' +
          '<button id="lsp-shutterstock-bg-close" style="background:none;border:none;cursor:pointer;padding:4px;color:#94a3b8;display:none;">' +
            '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M18 6L6 18M6 6l12 12"/></svg>' +
          '</button>' +
        '</div>' +
        '<div style="height:6px;background:#fee2e2;border-radius:4px;overflow:hidden;">' +
          '<div id="lsp-shutterstock-bg-bar" style="height:100%;background:#ee2d24;width:0%;transition:width 0.3s;"></div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', progressHtml);

    // For Shutterstock, we do foreground sync but with a floating progress card
    // (since we don't have a background queue system for Shutterstock yet)
    var total = fileData.length;
    var synced = 0;
    var errors = [];
    var queue = fileData.slice();

    function updateFloatingProgress() {
      var done = synced + errors.length;
      var pct = total > 0 ? Math.round((done / total) * 100) : 0;
      var statusEl = document.getElementById('lsp-shutterstock-bg-status');
      var barEl = document.getElementById('lsp-shutterstock-bg-bar');
      if (statusEl) statusEl.textContent = 'Synced ' + synced + ' of ' + total + ' to ' + destText;
      if (barEl) barEl.style.width = pct + '%';
    }

    function syncNextBg() {
      if (queue.length === 0) {
        // Done
        var statusEl = document.getElementById('lsp-shutterstock-bg-status');
        var closeBtn = document.getElementById('lsp-shutterstock-bg-close');
        var barEl = document.getElementById('lsp-shutterstock-bg-bar');
        
        var msg = '✓ Synced ' + synced + ' image(s)';
        if (errors.length > 0) msg += ' (' + errors.length + ' failed)';
        if (statusEl) statusEl.textContent = msg;
        if (barEl) barEl.style.width = '100%';
        if (closeBtn) {
          closeBtn.style.display = 'block';
          closeBtn.addEventListener('click', function() {
            document.getElementById('lsp-shutterstock-bg-progress')?.remove();
          });
        }

        // Auto-close after 5 seconds
        setTimeout(function() {
          var card = document.getElementById('lsp-shutterstock-bg-progress');
          if (card) card.remove();
        }, 5000);

        // Clear selection and reload
        window.lspShutterstockSelectedFiles = [];
        if (typeof window.loadShutterstockLicenses === 'function') {
          window.loadShutterstockLicenses(1);
        }
        if (typeof window.updateShutterstockCommandBarSelection === 'function') {
          window.updateShutterstockCommandBarSelection();
        }
        return;
      }

      updateFloatingProgress();

      var file = queue.shift();
      
      jQuery.ajax({
        url: ajaxurl,
        type: 'POST',
        timeout: 120000,
        data: {
          action: 'lsp_shutterstock_sync',
          _wpnonce: LIGHTSYNCPRO.nonce,
          license_id: file.license_id,
          image_id: file.id,
          description: file.description
        },
        success: function(response) {
          if (response.success) {
            synced++;
            // Update local synced state
            if (!window.lspShutterstockSynced) window.lspShutterstockSynced = {};
            window.lspShutterstockSynced[file.id] = {
              time: Math.floor(Date.now() / 1000),
              dest: syncTarget
            };
          } else {
            errors.push(file.id);
          }
          syncNextBg();
        },
        error: function() {
          errors.push(file.id);
          syncNextBg();
        }
      });
    }

    syncNextBg();
  }

  // ============================================
  // AI Sync to Destinations Handler
  // ============================================
  function triggerAiSync() {
    var selectedIds = (window.lspAiState && window.lspAiState.aiSelectedIds) ? window.lspAiState.aiSelectedIds.slice() : [];

    if (!selectedIds.length) {
      if (typeof window.lspModal === 'function') {
        window.lspModal({
          title: 'No Images Selected',
          body: 'Please select at least one AI-generated image to sync.',
          type: 'warning'
        });
      }
      return;
    }

    var destRadio = document.querySelector('input[name="lsp_ai_dest"]:checked');
    var dest = destRadio ? destRadio.value : 'wp';

    if (dest === 'wp') {
      if (typeof window.lspModal === 'function') {
        window.lspModal({
          title: 'Already in WordPress',
          body: 'Selected images are already in your WordPress Media Library. Choose <strong>Shopify</strong> or <strong>Both</strong> as the destination to sync to Shopify.',
          type: 'info'
        });
      }
      return;
    }

    var destLabel = dest === 'both' ? 'WordPress + Shopify' : 'Shopify';
    var count = selectedIds.length;

    showAiSyncChoiceModal(count, destLabel, function(mode) {
      if (mode === 'foreground') {
        startAiForegroundSync(selectedIds, dest, destLabel);
      } else if (mode === 'background') {
        startAiBackgroundSync(selectedIds, dest, destLabel);
      }
    });
  }

  // Sync choice modal — matches Canva/Figma exactly
  function showAiSyncChoiceModal(count, destLabel, callback) {
    var existing = document.getElementById('lsp-ai-sync-choice-modal');
    if (existing) existing.remove();

    var accentColor = '#e11d48'; // rose/coral to match LightSync Pro brand
    var accentBg    = 'rgba(225,29,72,.12)';
    var accentLight = 'rgba(225,29,72,.06)';

    var modalHtml = 
      '<div id="lsp-ai-sync-choice-modal" class="lsp-modal" aria-hidden="false" style="display:block;position:fixed;inset:0;z-index:10060;">' +
        '<div class="lsp-modal-backdrop" data-lsp-close style="position:absolute;inset:0;background:rgba(15,23,42,.55);backdrop-filter:blur(6px);"></div>' +
        '<div class="lsp-modal-card" style="position:absolute;left:50%;top:18%;transform:translateX(-50%);width:min(520px,calc(100% - 28px));background:#fff;border:1px solid #e5e7eb;border-radius:18px;box-shadow:0 30px 80px rgba(15,23,42,.35);padding:20px;">' +
          '<div class="lsp-modal-head" style="display:flex;gap:12px;align-items:center;margin-bottom:16px;">' +
            '<div style="width:40px;height:40px;border-radius:12px;display:grid;place-items:center;background:' + accentBg + ';border:1px solid rgba(225,29,72,.25);color:' + accentColor + ';">' +
              '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5"><path d="M12 2a4 4 0 0 1 4 4v1a1 1 0 0 0 1 1h1a4 4 0 0 1 0 8h-1a1 1 0 0 0-1 1v1a4 4 0 0 1-8 0v-1a1 1 0 0 0-1-1H6a4 4 0 0 1 0-8h1a1 1 0 0 0 1-1V6a4 4 0 0 1 4-4z"/><circle cx="12" cy="12" r="2"/></svg>' +
            '</div>' +
            '<h3 style="margin:0;font-size:17px;font-weight:700;">Sync AI Images</h3>' +
          '</div>' +
          '<div class="lsp-modal-body">' +
            '<p style="margin:0 0 16px;color:#64748b;">' +
              '<strong style="color:#0f172a;">' + count + ' image' + (count > 1 ? 's' : '') + '</strong> selected to sync to <strong style="color:#0f172a;">' + destLabel + '</strong>.' +
            '</p>' +
            '<div style="display:grid;gap:12px;">' +
              '<label class="lsp-sync-option" data-mode="foreground" style="display:flex;gap:12px;padding:14px;border:2px solid ' + accentColor + ';border-radius:12px;cursor:pointer;background:' + accentLight + ';transition:all .15s;">' +
                '<input type="radio" name="ai_sync_mode" value="foreground" checked style="margin-top:3px;accent-color:' + accentColor + ';">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Sync Now (Foreground)</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Watch progress in real-time. Keep this tab open until complete.' +
                  '</span>' +
                '</div>' +
              '</label>' +
              '<label class="lsp-sync-option" data-mode="background" style="display:flex;gap:12px;padding:14px;border:2px solid #e5e7eb;border-radius:12px;cursor:pointer;background:#fff;transition:all .15s;">' +
                '<input type="radio" name="ai_sync_mode" value="background" style="margin-top:3px;accent-color:' + accentColor + ';">' +
                '<div>' +
                  '<strong style="display:block;margin-bottom:4px;color:#0f172a;">Sync in Background</strong>' +
                  '<span style="color:#64748b;font-size:13px;line-height:1.4;">' +
                    'Runs via WordPress cron. You can close this tab safely.' +
                  '</span>' +
                '</div>' +
              '</label>' +
            '</div>' +
          '</div>' +
          '<div class="lsp-modal-actions" style="display:flex;gap:10px;justify-content:flex-end;margin-top:20px;">' +
            '<button type="button" class="btn ghost" data-lsp-close style="padding:10px 18px;border-radius:10px;border:1px solid #e5e7eb;background:#fff;cursor:pointer;font-weight:600;">Cancel</button>' +
            '<button type="button" class="btn primary" id="lsp-ai-sync-choice-confirm" style="padding:10px 18px;border-radius:10px;border:none;background:' + accentColor + ';color:#fff;cursor:pointer;font-weight:600;">Start Sync</button>' +
          '</div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    var modal = document.getElementById('lsp-ai-sync-choice-modal');
    var options = modal.querySelectorAll('.lsp-sync-option');
    var radios = modal.querySelectorAll('input[name="ai_sync_mode"]');

    radios.forEach(function(radio) {
      radio.addEventListener('change', function() {
        options.forEach(function(opt) {
          if (opt.querySelector('input').checked) {
            opt.style.borderColor = accentColor;
            opt.style.background = accentLight;
          } else {
            opt.style.borderColor = '#e5e7eb';
            opt.style.background = '#fff';
          }
        });
      });
    });

    options.forEach(function(opt) {
      opt.addEventListener('mouseenter', function() {
        if (!opt.querySelector('input').checked) {
          opt.style.borderColor = '#fda4af';
          opt.style.background = '#f8fafc';
        }
      });
      opt.addEventListener('mouseleave', function() {
        if (!opt.querySelector('input').checked) {
          opt.style.borderColor = '#e5e7eb';
          opt.style.background = '#fff';
        }
      });
    });

    modal.querySelectorAll('[data-lsp-close]').forEach(function(el) {
      el.addEventListener('click', function() {
        modal.remove();
        callback(null);
      });
    });

    document.getElementById('lsp-ai-sync-choice-confirm').addEventListener('click', function() {
      var selected = modal.querySelector('input[name="ai_sync_mode"]:checked');
      var mode = selected ? selected.value : 'foreground';
      modal.remove();
      callback(mode);
    });
  }

  // Foreground sync with progress ring — matches Canva exactly
  function startAiForegroundSync(ids, dest, destLabel) {
    btn.disabled = true;
    btn.innerHTML = '<span class="dashicons dashicons-update" style="animation:spin 1s linear infinite;"></span> Syncing...';

    var existingModal = document.getElementById('lsp-ai-progress-modal');
    if (existingModal) existingModal.remove();

    var modalHtml = 
      '<div id="lsp-ai-progress-modal" style="position:fixed;inset:0;z-index:100000;display:flex;align-items:center;justify-content:center;background:rgba(15,23,42,.6);backdrop-filter:blur(4px);">' +
        '<div style="background:#fff;border-radius:20px;padding:32px 40px;max-width:420px;width:90%;box-shadow:0 25px 60px rgba(0,0,0,.3);text-align:center;">' +
          '<div style="margin-bottom:20px;">' +
            '<div style="position:relative;width:120px;height:120px;margin:0 auto;">' +
              '<svg viewBox="0 0 100 100" style="transform:rotate(-90deg);width:100%;height:100%;">' +
                '<circle cx="50" cy="50" r="45" fill="none" stroke="#e5e7eb" stroke-width="8"/>' +
                '<circle id="lsp-ai-progress-circle" cx="50" cy="50" r="45" fill="none" stroke="#e11d48" stroke-width="8" stroke-linecap="round" stroke-dasharray="283" stroke-dashoffset="283" style="transition:stroke-dashoffset 0.3s ease;"/>' +
              '</svg>' +
              '<div id="lsp-ai-progress-pct" style="position:absolute;inset:0;display:flex;align-items:center;justify-content:center;font-size:28px;font-weight:700;color:#1e293b;">0%</div>' +
            '</div>' +
          '</div>' +
          '<h3 style="margin:0 0 8px;font-size:18px;font-weight:600;color:#1e293b;">Syncing to ' + destLabel + '</h3>' +
          '<p id="lsp-ai-progress-sub" style="margin:0 0 16px;font-size:14px;color:#64748b;">Preparing...</p>' +
          '<div id="lsp-ai-progress-log" style="max-height:150px;overflow-y:auto;text-align:left;font-family:ui-monospace,monospace;font-size:11px;background:#f8fafc;border:1px solid #e2e8f0;border-radius:8px;padding:10px;color:#475569;"></div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    var progressCircle = document.getElementById('lsp-ai-progress-circle');
    var progressPct    = document.getElementById('lsp-ai-progress-pct');
    var progressSub    = document.getElementById('lsp-ai-progress-sub');
    var progressLog    = document.getElementById('lsp-ai-progress-log');

    function setProgress(pct) {
      pct = Math.max(0, Math.min(100, Math.round(pct)));
      progressCircle.style.strokeDashoffset = 283 - (283 * pct / 100);
      progressPct.textContent = pct + '%';
    }

    function logMsg(msg) {
      var time = new Date().toLocaleTimeString();
      progressLog.innerHTML += '<div>[' + time + '] ' + msg + '</div>';
      progressLog.scrollTop = progressLog.scrollHeight;
    }

    // Smooth easing animation
    var visualPct = 0, targetPct = 0, ticker = null;
    function easeToTarget() {
      var delta = targetPct - visualPct;
      if (Math.abs(delta) < 0.3) { visualPct = targetPct; setProgress(visualPct); if (ticker && targetPct >= 100) { clearInterval(ticker); ticker = null; } return; }
      visualPct += delta * 0.15;
      setProgress(visualPct);
    }
    ticker = setInterval(easeToTarget, 30);

    logMsg('Starting AI sync for ' + ids.length + ' image(s) to ' + destLabel + '...');

    var total = ids.length;
    var completed = 0;
    var errors = [];

    function pushNext() {
      if (completed >= total) {
        targetPct = 100;
        setTimeout(function() {
          var modal = document.getElementById('lsp-ai-progress-modal');

          if (errors.length) {
            progressSub.textContent = 'Completed with ' + errors.length + ' error(s)';
            logMsg('⚠ ' + errors.length + ' of ' + total + ' failed');
          } else {
            progressSub.textContent = 'All images synced successfully!';
            logMsg('✓ All ' + total + ' images synced');
          }

          // Show close button
          var closeBtn = document.createElement('button');
          closeBtn.textContent = 'Done';
          closeBtn.style.cssText = 'margin-top:16px;padding:10px 24px;border-radius:10px;border:none;background:#e11d48;color:#fff;cursor:pointer;font-weight:600;font-size:14px;';
          closeBtn.addEventListener('click', function() {
            modal.remove();
          });
          modal.querySelector('div > div').appendChild(closeBtn);

          btn.disabled = false;
          btn.innerHTML = 'Sync Now';

          if (window.lspAiState) window.lspAiState.aiSelectedIds = [];
          if (typeof window.lspAiLoadBrowse === 'function') window.lspAiLoadBrowse();
        }, 500);
        return;
      }

      targetPct = ((completed + 0.3) / total) * 100;
      progressSub.textContent = 'Image ' + (completed + 1) + ' of ' + total + '...';
      logMsg('Syncing image ' + (completed + 1) + '/' + total + '...');

      var body = new URLSearchParams();
      body.set('action', 'lsp_ai_push_destinations');
      body.set('_ajax_nonce', LIGHTSYNCPRO.nonce);
      body.set('attachment_id', ids[completed]);
      body.set('force_dest', dest);

      fetch(LIGHTSYNCPRO.ajaxurl || ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      })
      .then(function(r) { return r.json(); })
      .then(function(json) {
        completed++;
        targetPct = (completed / total) * 100;
        if (!json.success) {
          var err = (json.data && json.data.error) || 'Unknown error';
          errors.push({ id: ids[completed - 1], error: err });
          logMsg('✕ Image ' + completed + ': ' + err);
        } else {
          logMsg('✓ Image ' + completed + ': synced');
        }
        pushNext();
      })
      .catch(function() {
        completed++;
        errors.push({ id: ids[completed - 1], error: 'Network error' });
        logMsg('✕ Image ' + completed + ': network error');
        pushNext();
      });
    }

    pushNext();
  }

  // Background sync — queue IDs and let cron process
  function startAiBackgroundSync(ids, dest, destLabel) {
    btn.disabled = true;
    btn.innerHTML = '<span class="dashicons dashicons-clock"></span> Starting...';

    var body = new URLSearchParams();
    body.set('action', 'lsp_ai_background_sync');
    body.set('_ajax_nonce', LIGHTSYNCPRO.nonce);
    body.set('dest', dest);
    ids.forEach(function(id) {
      body.append('attachment_ids[]', id);
    });

    fetch(LIGHTSYNCPRO.ajaxurl || ajaxurl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: body.toString()
    })
    .then(function(r) { return r.json(); })
    .then(function(json) {
      if (json.success) {
        if (window.lspAiState) window.lspAiState.aiSelectedIds = [];
        if (typeof window.lspAiLoadBrowse === 'function') window.lspAiLoadBrowse();
        startAiBackgroundPolling(destLabel);
      } else {
        var err = (json.data && json.data.error) ? json.data.error : 'Unknown error';
        lspShowToast('Failed to start background sync: ' + err, 'error');
        btn.disabled = false;
        btn.innerHTML = 'Sync Now';
      }
    })
    .catch(function() {
      lspShowToast('Connection error starting background sync', 'error');
      btn.disabled = false;
      btn.innerHTML = 'Sync Now';
    });
  }

  function startAiBackgroundPolling(destLabel) {
    btn.disabled = true;
    btn.innerHTML = '<span class="dashicons dashicons-clock"></span> Syncing...';

    // Floating progress card (bottom-right, like Lightroom background sync)
    var existingCard = document.getElementById('lsp-ai-bg-progress');
    if (existingCard) existingCard.remove();

    var cardHtml = 
      '<div id="lsp-ai-bg-progress" style="position:fixed;bottom:20px;right:20px;background:#fff;border-radius:16px;padding:20px 24px;box-shadow:0 8px 30px rgba(0,0,0,.18);z-index:100000;min-width:320px;max-width:400px;">' +
        '<div style="display:flex;align-items:center;gap:14px;margin-bottom:12px;">' +
          '<div style="width:44px;height:44px;border-radius:12px;display:grid;place-items:center;background:rgba(225,29,72,.12);border:1px solid rgba(225,29,72,.25);">' +
            '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#e11d48" stroke-width="1.5"><path d="M12 2a4 4 0 0 1 4 4v1a1 1 0 0 0 1 1h1a4 4 0 0 1 0 8h-1a1 1 0 0 0-1 1v1a4 4 0 0 1-8 0v-1a1 1 0 0 0-1-1H6a4 4 0 0 1 0-8h1a1 1 0 0 0 1-1V6a4 4 0 0 1 4-4z"/><circle cx="12" cy="12" r="2"/></svg>' +
          '</div>' +
          '<div style="flex:1;">' +
            '<strong style="display:block;color:#1e293b;font-size:15px;">Syncing AI Images</strong>' +
            '<span id="lsp-ai-bg-dest" style="font-size:13px;color:#64748b;">→ ' + destLabel + '</span>' +
          '</div>' +
        '</div>' +
        '<div style="margin-bottom:6px;display:flex;justify-content:space-between;">' +
          '<span id="lsp-ai-bg-status" style="font-size:13px;color:#64748b;">Starting...</span>' +
          '<span id="lsp-ai-bg-pct" style="font-size:13px;font-weight:600;color:#e11d48;">0%</span>' +
        '</div>' +
        '<div style="height:6px;background:#f1f5f9;border-radius:999px;overflow:hidden;">' +
          '<div id="lsp-ai-bg-bar" style="height:100%;width:0%;background:linear-gradient(90deg,#e11d48,#ff5757);border-radius:999px;transition:width 0.4s ease;"></div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', cardHtml);

    var pollTimer = setInterval(function() {
      var body = new URLSearchParams();
      body.set('action', 'lsp_ai_background_status');
      body.set('_ajax_nonce', LIGHTSYNCPRO.nonce);

      fetch(LIGHTSYNCPRO.ajaxurl || ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      })
      .then(function(r) { return r.json(); })
      .then(function(json) {
        if (!json.success || !json.data) return;
        var d = json.data;
        var pct = d.total > 0 ? Math.round((d.completed / d.total) * 100) : 0;

        var statusEl = document.getElementById('lsp-ai-bg-status');
        var pctEl = document.getElementById('lsp-ai-bg-pct');
        var barEl = document.getElementById('lsp-ai-bg-bar');

        if (statusEl) statusEl.textContent = d.completed + ' of ' + d.total + ' synced';
        if (pctEl)    pctEl.textContent = pct + '%';
        if (barEl)    barEl.style.width = pct + '%';

        if (d.status === 'complete' || d.completed >= d.total) {
          clearInterval(pollTimer);
          btn.disabled = false;
          btn.innerHTML = 'Sync Now';

          if (statusEl) statusEl.textContent = 'Complete! ' + d.completed + ' image' + (d.completed !== 1 ? 's' : '') + ' synced.';
          if (pctEl) pctEl.textContent = '100%';
          if (barEl) barEl.style.width = '100%';

          lspShowToast('✓ Background sync complete: ' + d.completed + ' AI image(s) synced', 'success');

          setTimeout(function() {
            var card = document.getElementById('lsp-ai-bg-progress');
            if (card) card.remove();
          }, 4000);

          if (typeof window.lspAiLoadBrowse === 'function') window.lspAiLoadBrowse();
        }
      });
    }, 3000);
  }

  // ============================================
  // Check for Running Background Sync on Page Load
  // ============================================
  (function checkBackgroundSyncOnLoad() {
    var body = new URLSearchParams();
    body.set('action', 'lightsyncpro_ajax_background_status');
    body.set('_ajax_nonce', LIGHTSYNCPRO.nonce);

    fetch(LIGHTSYNCPRO.ajaxurl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: body.toString()
    })
    .then(function(r) { return r.json(); })
    .then(function(json) {
      if (json.success && json.data && json.data.running) {
        // Show a toast to indicate background sync is running
        lspShowToast('Background sync in progress...', 'success');
        // Start polling - this will also show the progress UI and disable button
        startBackgroundStatusPolling();
      }
    })
    .catch(function() {
      // Silent fail
    });
  })();

  // ============================================
  // AI Insights - Analyze Button Handler
  // ============================================
  var aiBtn = document.getElementById('lsp-ai-analyze');
  if (aiBtn) {
    aiBtn.addEventListener('click', function(e) {
      e.preventDefault();
      
      // Check which source is active
      var activeSource = 'lightroom';
      var canvaTab = document.querySelector('.lsp-source-tab[data-source="canva"].active');
      if (canvaTab) {
        activeSource = 'canva';
      }

      if (activeSource === 'canva') {
        triggerCanvaAiAnalyze();
      } else {
        triggerLightroomAiAnalyze();
      }
    });
  }

  function triggerLightroomAiAnalyze() {
    // Get selected albums
    var albums = selectedVals(albumsEl);
    if (!albums.length) {
      lspShowToast('Please select at least one album to analyze', 'warning');
      return;
    }

    // For now, analyze the first album's images
    // TODO: Get actual image URLs from the album
    lspShowToast('AI analysis for Lightroom coming soon!', 'info');
  }

  function triggerCanvaAiAnalyze() {
    // Get selected Canva designs
    if (!window.lspCanvaSelected || window.lspCanvaSelected.size === 0) {
      lspShowToast('Please select designs to analyze', 'warning');
      return;
    }

    var selectedIds = Array.from(window.lspCanvaSelected);
    
    // Get thumbnail URLs for selected designs
    var imageUrls = [];
    var imageIds = [];
    
    if (window.lspCanvaDesigns) {
      window.lspCanvaDesigns.forEach(function(design) {
        if (window.lspCanvaSelected.has(design.id)) {
          if (design.thumbnail && design.thumbnail.url) {
            imageUrls.push(design.thumbnail.url);
            imageIds.push(design.id);
          }
        }
      });
    }

    if (imageUrls.length === 0) {
      lspShowToast('No thumbnails available for selected designs', 'warning');
      return;
    }

    // Show analyzing state
    aiBtn.disabled = true;
    aiBtn.innerHTML = '<span class="dashicons dashicons-lightbulb" style="animation:pulse 1s ease-in-out infinite;"></span> Analyzing...';

    // Show modal
    showAiAnalysisModal(imageUrls, imageIds, 'canva');
  }

  function showAiAnalysisModal(imageUrls, imageIds, source) {
    // Remove existing modal
    var existing = document.getElementById('lsp-ai-modal');
    if (existing) existing.remove();

    var modalHtml = 
      '<div id="lsp-ai-modal" class="lsp-modal" style="display:block;position:fixed;inset:0;z-index:100000;background:rgba(15,23,42,.6);backdrop-filter:blur(4px);">' +
        '<div style="position:absolute;top:50%;left:50%;transform:translate(-50%,-50%);background:#fff;border-radius:20px;width:90%;max-width:800px;max-height:85vh;overflow:hidden;box-shadow:0 25px 60px rgba(0,0,0,.3);">' +
          '<div style="padding:24px 28px;border-bottom:1px solid #e2e8f0;display:flex;align-items:center;justify-content:space-between;">' +
            '<div>' +
              '<h2 style="margin:0;font-size:20px;font-weight:700;color:#1e293b;">✨ AI Image Analysis</h2>' +
              '<p style="margin:4px 0 0;font-size:13px;color:#64748b;">Analyzing ' + imageUrls.length + ' image(s)</p>' +
            '</div>' +
            '<button type="button" id="lsp-ai-modal-close" style="background:none;border:none;font-size:24px;color:#94a3b8;cursor:pointer;padding:4px;">×</button>' +
          '</div>' +
          '<div id="lsp-ai-modal-content" style="padding:24px 28px;overflow-y:auto;max-height:calc(85vh - 160px);">' +
            '<div style="text-align:center;padding:40px;">' +
              '<div class="spinner is-active" style="float:none;margin:0 auto 16px;"></div>' +
              '<p style="color:#64748b;margin:0;">Sending images to AI for analysis...</p>' +
              '<p style="color:#94a3b8;font-size:12px;margin:8px 0 0;">This may take 10-30 seconds</p>' +
            '</div>' +
          '</div>' +
          '<div id="lsp-ai-modal-footer" style="padding:16px 28px;border-top:1px solid #e2e8f0;display:none;justify-content:space-between;align-items:center;background:#f8fafc;">' +
            '<div id="lsp-ai-modal-summary" style="font-size:13px;color:#64748b;"></div>' +
            '<div style="display:flex;gap:12px;">' +
              '<button type="button" id="lsp-ai-apply-alt" class="btn ghost" style="display:none;">Apply Alt Text</button>' +
              '<button type="button" id="lsp-ai-apply-all" class="btn primary" style="display:none;">Apply All Recommendations</button>' +
            '</div>' +
          '</div>' +
        '</div>' +
      '</div>';

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    // Close handler
    document.getElementById('lsp-ai-modal-close').addEventListener('click', closeAiModal);
    document.getElementById('lsp-ai-modal').addEventListener('click', function(e) {
      if (e.target === this) closeAiModal();
    });

    // Make API call
    var body = new URLSearchParams();
    body.set('action', 'lsp_ai_analyze');
    body.set('_wpnonce', LIGHTSYNCPRO.nonce);
    body.set('source', source);
    imageUrls.forEach(function(url) { body.append('image_urls[]', url); });
    imageIds.forEach(function(id) { body.append('image_ids[]', id); });

    fetch(LIGHTSYNCPRO.ajaxurl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: body.toString()
    })
    .then(function(r) { return r.json(); })
    .then(function(json) {
      if (!json.success) {
        showAiError(json.data ? json.data.error : 'Analysis failed');
        return;
      }
      showAiResults(json.data.analysis, imageUrls, imageIds, source);
    })
    .catch(function(err) {
      showAiError('Connection error: ' + err.message);
    });
  }

  function showAiError(message) {
    var content = document.getElementById('lsp-ai-modal-content');
    if (content) {
      content.innerHTML = 
        '<div style="text-align:center;padding:40px;">' +
          '<div style="width:60px;height:60px;border-radius:50%;background:#fef2f2;display:flex;align-items:center;justify-content:center;margin:0 auto 16px;">' +
            '<span class="dashicons dashicons-warning" style="color:#ef4444;font-size:28px;"></span>' +
          '</div>' +
          '<h3 style="margin:0 0 8px;color:#1e293b;">Analysis Failed</h3>' +
          '<p style="color:#64748b;margin:0;">' + escapeHtml(message) + '</p>' +
        '</div>';
    }
    resetAiButton();
  }

  function showAiResults(analysis, imageUrls, imageIds, source) {
    var content = document.getElementById('lsp-ai-modal-content');
    var footer = document.getElementById('lsp-ai-modal-footer');
    
    if (!content || !analysis) {
      showAiError('Invalid response from AI');
      return;
    }

    var images = analysis.images || [];
    var heroPick = analysis.hero_pick;
    var summary = analysis.summary || '';

    // Build results HTML
    var html = '';

    // Hero recommendation
    if (heroPick !== null && heroPick !== undefined && images[heroPick]) {
      var hero = images[heroPick];
      html += 
        '<div style="background:linear-gradient(135deg,#faf5ff,#fdf4ff);border:2px solid #d8b4fe;border-radius:16px;padding:20px;margin-bottom:24px;">' +
          '<div style="display:flex;align-items:center;gap:8px;margin-bottom:12px;">' +
            '<span style="font-size:20px;">⭐</span>' +
            '<strong style="color:#7c3aed;">Recommended Hero Image</strong>' +
          '</div>' +
          '<div style="display:flex;gap:16px;align-items:flex-start;">' +
            '<img src="' + (hero.url || imageUrls[heroPick]) + '" style="width:120px;height:80px;object-fit:cover;border-radius:8px;border:2px solid #fff;box-shadow:0 2px 8px rgba(0,0,0,.1);">' +
            '<div style="flex:1;">' +
              '<div style="font-size:14px;color:#1e293b;margin-bottom:4px;"><strong>Quality:</strong> ' + (hero.quality_score || '-') + '/10 | <strong>Hero Score:</strong> ' + (hero.hero_score || '-') + '/10</div>' +
              '<div style="font-size:13px;color:#64748b;margin-bottom:8px;">' + escapeHtml(hero.alt_text || '') + '</div>' +
              (hero.tags ? '<div style="display:flex;gap:6px;flex-wrap:wrap;">' + hero.tags.map(function(t) { return '<span style="background:#e9d5ff;color:#7c3aed;font-size:11px;padding:2px 8px;border-radius:99px;">' + escapeHtml(t) + '</span>'; }).join('') + '</div>' : '') +
            '</div>' +
          '</div>' +
        '</div>';
    }

    // All images grid
    html += '<h4 style="margin:0 0 16px;font-size:15px;color:#1e293b;">All Images</h4>';
    html += '<div style="display:grid;grid-template-columns:repeat(auto-fill,minmax(200px,1fr));gap:16px;">';

    images.forEach(function(img, idx) {
      var isHero = idx === heroPick;
      html += 
        '<div class="lsp-ai-result-card" data-index="' + idx + '" style="background:#fff;border:1px solid ' + (isHero ? '#d8b4fe' : '#e2e8f0') + ';border-radius:12px;overflow:hidden;' + (isHero ? 'box-shadow:0 0 0 2px #d8b4fe;' : '') + '">' +
          '<div style="position:relative;">' +
            '<img src="' + (img.url || imageUrls[idx]) + '" style="width:100%;height:120px;object-fit:cover;">' +
            (isHero ? '<span style="position:absolute;top:8px;right:8px;background:#7c3aed;color:#fff;font-size:10px;padding:2px 8px;border-radius:99px;">⭐ Hero</span>' : '') +
            '<span style="position:absolute;bottom:8px;left:8px;background:rgba(0,0,0,.7);color:#fff;font-size:11px;padding:2px 8px;border-radius:4px;">' + (img.suggested_use || 'gallery') + '</span>' +
          '</div>' +
          '<div style="padding:12px;">' +
            '<div style="display:flex;justify-content:space-between;margin-bottom:8px;">' +
              '<span style="font-size:12px;color:#64748b;">Quality: <strong style="color:#1e293b;">' + (img.quality_score || '-') + '</strong>/10</span>' +
              '<span style="font-size:12px;color:#64748b;">Hero: <strong style="color:#1e293b;">' + (img.hero_score || '-') + '</strong>/10</span>' +
            '</div>' +
            '<div style="font-size:12px;color:#475569;line-height:1.4;margin-bottom:8px;min-height:48px;">' + escapeHtml(img.alt_text || 'No alt text generated') + '</div>' +
            (img.tags ? '<div style="display:flex;gap:4px;flex-wrap:wrap;">' + img.tags.slice(0,3).map(function(t) { return '<span style="background:#f1f5f9;color:#64748b;font-size:10px;padding:2px 6px;border-radius:4px;">' + escapeHtml(t) + '</span>'; }).join('') + '</div>' : '') +
            (img.duplicate_group !== null && img.duplicate_group !== undefined ? '<div style="margin-top:8px;font-size:11px;color:#f59e0b;">⚠️ Possible duplicate (group ' + img.duplicate_group + ')</div>' : '') +
          '</div>' +
        '</div>';
    });

    html += '</div>';

    // Summary
    if (summary) {
      html += '<div style="margin-top:24px;padding:16px;background:#f8fafc;border-radius:12px;font-size:13px;color:#64748b;">' +
        '<strong style="color:#1e293b;">Summary:</strong> ' + escapeHtml(summary) +
      '</div>';
    }

    content.innerHTML = html;

    // Show footer with apply buttons
    footer.style.display = 'flex';
    document.getElementById('lsp-ai-modal-summary').textContent = images.length + ' images analyzed';
    document.getElementById('lsp-ai-apply-alt').style.display = 'inline-flex';
    document.getElementById('lsp-ai-apply-all').style.display = 'inline-flex';

    // Store analysis data for apply
    window._lspAiAnalysis = { images: images, imageUrls: imageUrls, imageIds: imageIds, source: source };

    // Apply handlers
    document.getElementById('lsp-ai-apply-alt').addEventListener('click', function() {
      applyAiRecommendations('alt_text');
    });
    document.getElementById('lsp-ai-apply-all').addEventListener('click', function() {
      applyAiRecommendations('all');
    });

    resetAiButton();
  }

  function applyAiRecommendations(actionType) {
    var data = window._lspAiAnalysis;
    if (!data || !data.images) {
      lspShowToast('No analysis data available', 'error');
      return;
    }

    // Note: For Canva, we need to first sync the images to get attachment IDs
    // For now, show a message about this limitation
    lspShowToast('Apply functionality requires synced images. Sync first, then re-analyze.', 'info');
    closeAiModal();
  }

  function closeAiModal() {
    var modal = document.getElementById('lsp-ai-modal');
    if (modal) modal.remove();
    resetAiButton();
  }

  function resetAiButton() {
    if (aiBtn) {
      aiBtn.disabled = false;
      aiBtn.innerHTML = '<span class="dashicons dashicons-lightbulb"></span> AI Analyze';
    }
  }

  function escapeHtml(str) {
    if (!str) return '';
    var div = document.createElement('div');
    div.textContent = str;
    return div.innerHTML;
  }

  // ============================================
  // Performance Insights Refresh Handler
  // ============================================
  var refreshStatsBtn = document.getElementById('lsp-refresh-stats');
  if (refreshStatsBtn) {
    refreshStatsBtn.addEventListener('click', function(e) {
      e.preventDefault();
      
      var btn = this;
      var icon = btn.querySelector('.dashicons');
      if (icon) icon.style.animation = 'lsp-spin 1s linear infinite';
      btn.disabled = true;

      var data = new FormData();
      data.append('action', 'lsp_get_performance_stats');
      data.append('_wpnonce', LIGHTSYNCPRO.nonce);

      fetch(LIGHTSYNCPRO.ajax, { method: 'POST', body: data })
        .then(function(r) { return r.json(); })
        .then(function(json) {
          if (icon) icon.style.animation = '';
          btn.disabled = false;

          if (!json.success || !json.data) {
            lspShowToast('Failed to load insights', 'error');
            return;
          }

          var insights = json.data;
          var panel = document.getElementById('lsp-performance-insights');
          if (!panel) return;

          // Update footer stats
          var imgEl = document.getElementById('lsp-stat-images');
          var impEl = document.getElementById('lsp-stat-impressions');
          var clickEl = document.getElementById('lsp-stat-clicks');
          
          if (imgEl) imgEl.textContent = (insights.total_images || 0).toLocaleString();
          if (impEl) impEl.textContent = (insights.total_impressions || 0).toLocaleString();
          if (clickEl) clickEl.textContent = (insights.total_clicks || 0).toLocaleString();

          // Build suggestions HTML
          var suggestionsEl = document.getElementById('lsp-suggestions');
          if (insights.suggestions && insights.suggestions.length > 0) {
            var suggestionsHtml = '';
            insights.suggestions.forEach(function(s) {
              var bgColor = s.type === 'success' ? '#f0fdf4' : (s.type === 'warning' ? '#fffbeb' : '#f0f9ff');
              var borderColor = s.type === 'success' ? '#bbf7d0' : (s.type === 'warning' ? '#fde68a' : '#bae6fd');
              suggestionsHtml += '<div style="padding:12px 16px;background:' + bgColor + ';border:1px solid ' + borderColor + ';border-radius:8px;margin-bottom:8px;display:flex;align-items:flex-start;gap:10px;">' +
                '<span style="font-size:18px;line-height:1.2;">' + escapeHtml(s.icon) + '</span>' +
                '<span style="font-size:13px;color:#334155;line-height:1.5;">' + escapeHtml(s.text) + '</span>' +
                '</div>';
            });
            if (suggestionsEl) {
              suggestionsEl.innerHTML = suggestionsHtml;
              suggestionsEl.style.display = 'block';
            }
          } else if (suggestionsEl) {
            suggestionsEl.style.display = 'none';
          }

          // Update winners section - just show toast for now, full rebuild is complex
          // The page will show updated data on next full load
          
          lspShowToast('Insights refreshed', 'success');
        })
        .catch(function(err) {
          if (icon) icon.style.animation = '';
          btn.disabled = false;
          lspShowToast('Failed to refresh insights', 'error');
          console.error('Insights refresh error:', err);
        });
    });
  }

  // ============================================
  // First Sync Celebration
  // ============================================
  
  /**
   * Show celebration modal for first successful sync
   * @param {string} source - 'lightroom' or 'canva'
   * @param {string} itemName - Name of the album/design that was synced
   * @param {number} count - Number of images synced
   */
  window.lspShowCelebration = function(source, itemName, count) {
    // Check if already celebrated
    if (source === 'lightroom' && LIGHTSYNCPRO.celebrated_lightroom) return false;
    if (source === 'canva' && LIGHTSYNCPRO.celebrated_canva) return false;
    if (source === 'figma' && LIGHTSYNCPRO.celebrated_figma) return false;
    if (source === 'dropbox' && LIGHTSYNCPRO.celebrated_dropbox) return false;
    
    var sourceLabels = {
      'lightroom': 'Lightroom',
      'canva': 'Canva',
      'figma': 'Figma',
      'dropbox': 'Dropbox'
    };
    var itemTypes = {
      'lightroom': 'album',
      'canva': 'design',
      'figma': 'design',
      'dropbox': 'file'
    };
    
    var sourceLabel = sourceLabels[source] || source;
    var itemType = itemTypes[source] || 'item';
    var headline = itemName ? ('"' + itemName + '" is live!') : ('Your first ' + itemType + ' is live!');
    var subtext = count ? (count + ' image' + (count !== 1 ? 's' : '') + ' synced to your site') : 'Successfully synced to your site';
    
    // Social share text based on source
    var shareTexts = {
      'lightroom': {
        twitter: 'Just synced my Lightroom albums straight to WordPress with @LightSyncPro — no more manual exports! 🎉 #photography #WordPress',
        linkedin: 'Excited to streamline my workflow! LightSync Pro now syncs my Lightroom library directly to my website. Game changer for photographers.'
      },
      'canva': {
        twitter: 'My Canva designs now auto-publish to my website with LightSync Pro ⚡ #design #WordPress',
        linkedin: 'Just connected Canva to my WordPress site with LightSync Pro. Designs now sync automatically — huge time saver!'
      },
      'figma': {
        twitter: 'Figma to WordPress in seconds with @LightSyncPro! My designs are now live on my site 🚀 #design #WordPress',
        linkedin: 'Just connected Figma to my WordPress site with LightSync Pro. Design updates sync automatically — perfect for my workflow!'
      },
      'dropbox': {
        twitter: 'My Dropbox photos now sync directly to WordPress with @LightSyncPro! 📸 #photography #WordPress',
        linkedin: 'Just streamlined my photo workflow with LightSync Pro. Dropbox images sync automatically to my WordPress site!'
      }
    };
    
    var twitterText = shareTexts[source] ? shareTexts[source].twitter : shareTexts.lightroom.twitter;
    var linkedInText = shareTexts[source] ? shareTexts[source].linkedin : shareTexts.lightroom.linkedin;
    
    var twitterUrl = 'https://twitter.com/intent/tweet?text=' + encodeURIComponent(twitterText);
    var linkedInUrl = 'https://www.linkedin.com/sharing/share-offsite/?url=' + encodeURIComponent('https://lightsyncpro.com');
    var facebookUrl = 'https://www.facebook.com/sharer/sharer.php?quote=' + encodeURIComponent(linkedInText) + '&u=' + encodeURIComponent('https://lightsyncpro.com');
    
    // Create modal
    var modal = document.createElement('div');
    modal.className = 'lsp-celebration-modal';
    modal.innerHTML = 
      '<div class="lsp-celebration-backdrop"></div>' +
      '<div class="lsp-celebration-content">' +
        '<div class="lsp-celebration-confetti"></div>' +
        '<div class="lsp-celebration-icon">🎉</div>' +
        '<h2 class="lsp-celebration-headline">' + headline + '</h2>' +
        '<p class="lsp-celebration-subtext">' + subtext + '</p>' +
        '<p class="lsp-celebration-share-label">Share your win</p>' +
        '<div class="lsp-celebration-share-buttons">' +
          '<a href="' + twitterUrl + '" target="_blank" rel="noopener" class="lsp-share-btn lsp-share-twitter" title="Share on X/Twitter">' +
            '<svg width="18" height="18" viewBox="0 0 24 24" fill="currentColor"><path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"/></svg>' +
          '</a>' +
          '<a href="' + linkedInUrl + '" target="_blank" rel="noopener" class="lsp-share-btn lsp-share-linkedin" title="Share on LinkedIn">' +
            '<svg width="18" height="18" viewBox="0 0 24 24" fill="currentColor"><path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z"/></svg>' +
          '</a>' +
          '<a href="' + facebookUrl + '" target="_blank" rel="noopener" class="lsp-share-btn lsp-share-facebook" title="Share on Facebook">' +
            '<svg width="18" height="18" viewBox="0 0 24 24" fill="currentColor"><path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/></svg>' +
          '</a>' +
        '</div>' +
        '<button class="lsp-celebration-close">Continue to dashboard</button>' +
      '</div>';
    
    // Add styles
    var style = document.createElement('style');
    style.textContent = 
      '.lsp-celebration-modal { position: fixed; inset: 0; z-index: 999999; display: flex; align-items: center; justify-content: center; }' +
      '.lsp-celebration-backdrop { position: absolute; inset: 0; background: rgba(0,0,0,0.7); backdrop-filter: blur(4px); }' +
      '.lsp-celebration-content { position: relative; background: linear-gradient(135deg, #1e1b4b 0%, #312e81 100%); border-radius: 20px; padding: 48px; text-align: center; max-width: 420px; width: 90%; box-shadow: 0 25px 50px -12px rgba(0,0,0,0.5); animation: lsp-celebrate-pop 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275); }' +
      '@keyframes lsp-celebrate-pop { 0% { transform: scale(0.8); opacity: 0; } 100% { transform: scale(1); opacity: 1; } }' +
      '.lsp-celebration-icon { font-size: 64px; margin-bottom: 16px; animation: lsp-bounce 0.6s ease infinite alternate; }' +
      '@keyframes lsp-bounce { 0% { transform: translateY(0); } 100% { transform: translateY(-10px); } }' +
      '.lsp-celebration-headline { color: #fff; font-size: 24px; font-weight: 700; margin: 0 0 8px; }' +
      '.lsp-celebration-subtext { color: #a5b4fc; font-size: 15px; margin: 0 0 24px; }' +
      '.lsp-celebration-share-label { color: #818cf8; font-size: 12px; text-transform: uppercase; letter-spacing: 1px; margin: 0 0 12px; }' +
      '.lsp-celebration-share-buttons { display: flex; gap: 12px; justify-content: center; margin-bottom: 28px; }' +
      '.lsp-share-btn { display: flex; align-items: center; justify-content: center; width: 44px; height: 44px; border-radius: 12px; color: #fff; transition: transform 0.2s, box-shadow 0.2s; }' +
      '.lsp-share-btn:hover { transform: translateY(-2px); box-shadow: 0 4px 12px rgba(0,0,0,0.3); }' +
      '.lsp-share-twitter { background: #000; }' +
      '.lsp-share-linkedin { background: #0077b5; }' +
      '.lsp-share-facebook { background: #1877f2; }' +
      '.lsp-celebration-close { background: #6366f1; color: #fff; border: none; padding: 14px 32px; border-radius: 10px; font-size: 15px; font-weight: 600; cursor: pointer; transition: background 0.2s; }' +
      '.lsp-celebration-close:hover { background: #4f46e5; }' +
      '.lsp-celebration-confetti { position: absolute; inset: 0; pointer-events: none; overflow: hidden; border-radius: 20px; }' +
      '.lsp-confetti-piece { position: absolute; width: 10px; height: 10px; top: -10px; animation: lsp-confetti-fall 3s ease-out forwards; }' +
      '@keyframes lsp-confetti-fall { 0% { transform: translateY(0) rotate(0deg); opacity: 1; } 100% { transform: translateY(500px) rotate(720deg); opacity: 0; } }';
    document.head.appendChild(style);
    document.body.appendChild(modal);
    
    // Create confetti
    var confettiContainer = modal.querySelector('.lsp-celebration-confetti');
    var colors = ['#6366f1', '#8b5cf6', '#ec4899', '#f59e0b', '#10b981', '#3b82f6'];
    for (var i = 0; i < 50; i++) {
      var piece = document.createElement('div');
      piece.className = 'lsp-confetti-piece';
      piece.style.left = Math.random() * 100 + '%';
      piece.style.background = colors[Math.floor(Math.random() * colors.length)];
      piece.style.animationDelay = Math.random() * 0.5 + 's';
      piece.style.borderRadius = Math.random() > 0.5 ? '50%' : '2px';
      confettiContainer.appendChild(piece);
    }
    
    // Close handler
    var closeBtn = modal.querySelector('.lsp-celebration-close');
    var backdrop = modal.querySelector('.lsp-celebration-backdrop');
    
    function closeCelebration() {
      modal.remove();
      style.remove();
      
      // Mark as celebrated via AJAX
      var body = new URLSearchParams();
      body.set('action', 'lsp_dismiss_celebration');
      body.set('_wpnonce', LIGHTSYNCPRO.nonce);
      body.set('source', source);
      fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      });
      
      // Update local flag so it doesn't show again this session
      if (source === 'lightroom') LIGHTSYNCPRO.celebrated_lightroom = 1;
      if (source === 'canva') LIGHTSYNCPRO.celebrated_canva = 1;
      if (source === 'dropbox') LIGHTSYNCPRO.celebrated_dropbox = 1;
      if (source === 'figma') LIGHTSYNCPRO.celebrated_figma = 1;
      
      // Reload page
      window.location.reload();
    }
    
    closeBtn.addEventListener('click', closeCelebration);
    backdrop.addEventListener('click', closeCelebration);
    
    return true; // Celebration shown
  };

  // ============================================
  // Hub Site Selector Modal
  // ============================================
  
  window.lspHubSelectedSites = [];
  window.lspPendingHubSync = null;
  
  // Open Hub site selector when clicking Hub destination card
  document.querySelectorAll('[id^="lsp-dest-hub-"]').forEach(function(hubCard) {
    hubCard.addEventListener('click', function(e) {
      e.preventDefault();
      e.stopPropagation();
      
      var modal = document.getElementById('lsp-hub-selector-modal');
      if (!modal) {
        console.log('Hub selector modal not found');
        return;
      }
      
      // Clear previous selection when opening modal
      window.lspHubSelectedSites = [];
      
      // Determine source type from card ID
      var sourceType = 'lightroom';
      var cardId = this.id;
      console.log('Hub card clicked:', cardId);
      if (cardId.includes('canva')) sourceType = 'canva';
      else if (cardId.includes('figma')) sourceType = 'figma';
      else if (cardId.includes('dropbox')) sourceType = 'dropbox';
      
      console.log('Source type:', sourceType);
      
      window.lspPendingHubSync = { sourceType: sourceType };
      
      // Show modal
      modal.style.display = 'block';
      modal.setAttribute('aria-hidden', 'false');
      
      // Pre-check all sites and update visual styling
      modal.querySelectorAll('input[name="hub_sites[]"]').forEach(function(cb) {
        cb.checked = true;
        var label = cb.closest('label');
        if (label) {
          label.style.borderColor = '#2563eb';
          label.style.background = '#eff6ff';
        }
      });
      
      // Add change handler to update styling (once per modal open)
      if (!modal._hubHandlersAttached) {
        modal._hubHandlersAttached = true;
        modal.addEventListener('change', function(e) {
          if (e.target.name === 'hub_sites[]') {
            var label = e.target.closest('label');
            if (label) {
              if (e.target.checked) {
                label.style.borderColor = '#2563eb';
                label.style.background = '#eff6ff';
              } else {
                label.style.borderColor = '#e2e8f0';
                label.style.background = '#f8fafc';
              }
            }
          }
        });
      }
    });
  });
  
  // Confirm Hub site selection
  var hubConfirmBtn = document.getElementById('lsp-hub-confirm-sites');
  if (hubConfirmBtn) {
    hubConfirmBtn.addEventListener('click', function() {
      var modal = document.getElementById('lsp-hub-selector-modal');
      var selectedSites = [];
      
      // Debug: log all checkboxes
      var allCheckboxes = modal.querySelectorAll('input[name="hub_sites[]"]');
      console.log('Total checkboxes found:', allCheckboxes.length);
      allCheckboxes.forEach(function(cb, i) {
        console.log('Checkbox ' + i + ': value=' + cb.value + ', checked=' + cb.checked);
      });
      
      modal.querySelectorAll('input[name="hub_sites[]"]:checked').forEach(function(cb) {
        var siteId = parseInt(cb.value);
        // Prevent duplicates
        if (selectedSites.indexOf(siteId) === -1) {
          selectedSites.push(siteId);
        }
      });
      
      console.log('Selected sites (deduplicated):', selectedSites);
      
      if (selectedSites.length === 0) {
        alert('Please select at least one destination site.');
        return;
      }
      
      // Clear and set fresh - make absolutely sure it's a new array
      window.lspHubSelectedSites = [];
      for (var i = 0; i < selectedSites.length; i++) {
        window.lspHubSelectedSites.push(selectedSites[i]);
      }
      
      console.log('window.lspHubSelectedSites is now:', window.lspHubSelectedSites, 'length:', window.lspHubSelectedSites.length);
      
      // Save to server for backend sync flows
      jQuery.ajax({
        url: ajaxurl,
        method: 'POST',
        data: {
          action: 'lsp_save_hub_sites',
          nonce: lspData.nonce,
          site_ids: window.lspHubSelectedSites
        },
        success: function(r) {
          console.log('Hub sites saved to server:', r);
        },
        error: function(e) {
          console.error('Failed to save Hub sites:', e);
        }
      });
      
      // Update the Hub card text to show selected count instead of total
      var selectedCount = window.lspHubSelectedSites.length;
      document.querySelectorAll('[id^="lsp-dest-hub-"] .lsp-dest-sub').forEach(function(span) {
        span.textContent = selectedCount + ' site' + (selectedCount !== 1 ? 's' : '') + ' selected';
      });
      
      console.log('Updated Hub card text to show ' + selectedCount + ' selected');
      
      // Close modal
      modal.style.display = 'none';
      modal.setAttribute('aria-hidden', 'true');
      
      // Select the Hub radio for the appropriate source
      var pending = window.lspPendingHubSync;
      if (pending) {
        var radioName = 'lightsyncpro_settings[sync_target]'; // Lightroom default
        if (pending.sourceType === 'canva') radioName = 'lsp_canva_sync_target';
        else if (pending.sourceType === 'figma') radioName = 'lsp_figma_sync_target';
        else if (pending.sourceType === 'dropbox') radioName = 'lsp_dropbox_sync_target';
        
        console.log('Looking for radio:', radioName, 'value=hub');
        var radio = document.querySelector('input[name="' + radioName + '"][value="hub"]');
        console.log('Found radio:', radio);
        if (radio) {
          radio.checked = true;
          // Dispatch change event so any listeners are notified
          radio.dispatchEvent(new Event('change', { bubbles: true }));
          
          // Update visual selection on cards
          var card = radio.closest('.lsp-dest-card');
          if (card) {
            var cards = card.parentElement;
            if (cards) {
              cards.querySelectorAll('.lsp-dest-card').forEach(function(c) { 
                c.classList.remove('selected'); 
              });
            }
            card.classList.add('selected');
            console.log('Hub card selected:', card);
          }
        } else {
          console.log('Radio not found for', radioName);
        }
        
        // If we had pending assets to sync, start the distribution now
        if (pending.assetIds && pending.assetIds.length > 0) {
          window.startHubDistribution(pending.assetIds, pending.sourceType, pending.btn);
        }
      }
      
      console.log('Hub sites selected:', selectedSites);
    });
  }
  
  // Close Hub modal handlers
  var hubModal = document.getElementById('lsp-hub-selector-modal');
  if (hubModal) {
    hubModal.querySelectorAll('[data-lsp-close]').forEach(function(el) {
      el.addEventListener('click', function() {
        hubModal.style.display = 'none';
        hubModal.setAttribute('aria-hidden', 'true');
      });
    });
    // Click on backdrop to close
    hubModal.querySelector('.lsp-modal-backdrop')?.addEventListener('click', function() {
      hubModal.style.display = 'none';
      hubModal.setAttribute('aria-hidden', 'true');
    });
  }

  // Hub Distribution Function
  window.startHubDistribution = function(assetIds, sourceType, btn) {
    console.log('startHubDistribution called', {assetIds: assetIds, sourceType: sourceType});
    
    var hubSites = window.lspHubSelectedSites || [];
    console.log('hubSites from window.lspHubSelectedSites:', hubSites, 'length:', hubSites.length);
    
    // Debug: Show what's actually in the array
    if (hubSites.length > 0) {
      console.log('Site IDs that will be used:', JSON.stringify(hubSites));
    }
    
    if (!hubSites.length) {
      // Open the selector modal instead of showing an error
      var modal = document.getElementById('lsp-hub-selector-modal');
      if (modal) {
        window.lspPendingHubSync = { sourceType: sourceType, assetIds: assetIds, btn: btn };
        modal.style.display = 'block';
        modal.setAttribute('aria-hidden', 'false');
        return;
      }
      alert('Please select destination sites from the Hub selector first.');
      return;
    }
    
    if (!assetIds || !assetIds.length) {
      alert('No assets selected.');
      return;
    }
    
    // Show progress modal
    var existingModal = document.getElementById('lsp-hub-progress-modal');
    if (existingModal) existingModal.remove();
    
    var modalHtml = 
      '<div id="lsp-hub-progress-modal" style="position:fixed;inset:0;z-index:100000;display:flex;align-items:center;justify-content:center;background:rgba(15,23,42,.6);backdrop-filter:blur(4px);">' +
        '<div style="background:#fff;border-radius:20px;padding:32px 40px;max-width:480px;width:90%;box-shadow:0 25px 60px rgba(0,0,0,.3);text-align:center;">' +
          '<div style="margin-bottom:20px;">' +
            '<div id="lsp-hub-progress-ring" style="position:relative;width:120px;height:120px;margin:0 auto;">' +
              '<svg viewBox="0 0 100 100" style="transform:rotate(-90deg);width:100%;height:100%;">' +
                '<circle cx="50" cy="50" r="45" fill="none" stroke="#e5e7eb" stroke-width="8"/>' +
                '<circle id="lsp-hub-progress-circle" cx="50" cy="50" r="45" fill="none" stroke="#ef4444" stroke-width="8" stroke-linecap="round" stroke-dasharray="283" stroke-dashoffset="283" style="transition:stroke-dashoffset 0.3s ease;"/>' +
              '</svg>' +
              '<div id="lsp-hub-progress-pct" style="position:absolute;inset:0;display:flex;align-items:center;justify-content:center;font-size:28px;font-weight:700;color:#1e293b;">0%</div>' +
            '</div>' +
          '</div>' +
          '<h3 id="lsp-hub-progress-title" style="margin:0 0 8px;font-size:18px;font-weight:600;color:#1e293b;">Distributing to ' + hubSites.length + ' site' + (hubSites.length !== 1 ? 's' : '') + '</h3>' +
          '<p id="lsp-hub-progress-sub" style="margin:0 0 16px;font-size:14px;color:#64748b;">Starting distribution...</p>' +
          '<div id="lsp-hub-progress-log" style="max-height:150px;overflow-y:auto;text-align:left;font-family:ui-monospace,monospace;font-size:11px;background:#f8fafc;border:1px solid #e2e8f0;border-radius:8px;padding:10px;color:#475569;"></div>' +
        '</div>' +
      '</div>';
    
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    var progressCircle = document.getElementById('lsp-hub-progress-circle');
    var progressPct = document.getElementById('lsp-hub-progress-pct');
    var progressSub = document.getElementById('lsp-hub-progress-sub');
    var progressLog = document.getElementById('lsp-hub-progress-log');
    
    function setProgress(pct) {
      pct = Math.max(0, Math.min(100, Math.round(pct)));
      var offset = 283 - (283 * pct / 100);
      progressCircle.setAttribute('stroke-dashoffset', offset);
      progressPct.textContent = pct + '%';
    }
    
    function addLog(msg) {
      var time = new Date().toLocaleTimeString();
      progressLog.innerHTML += '<div>[' + time + '] ' + msg + '</div>';
      progressLog.scrollTop = progressLog.scrollHeight;
    }
    
    // Disable button during distribution
    if (btn) {
      btn.disabled = true;
      btn.innerHTML = '<span class="dashicons dashicons-update" style="animation:spin 1s linear infinite;"></span> Distributing...';
    }
    
    addLog('Starting Hub distribution...');
    addLog('Assets: ' + assetIds.length + ', Sites: ' + hubSites.length);
    
    // Derive source_id from context (catalog for Lightroom, or use 'default')
    var sourceId = '';
    if (sourceType === 'lightroom' && typeof LIGHTSYNCPRO !== 'undefined' && LIGHTSYNCPRO.catalog) {
      sourceId = LIGHTSYNCPRO.catalog;
    } else if (sourceType === 'canva' && lspData.options && lspData.options.canva_folder_id) {
      sourceId = lspData.options.canva_folder_id;
    } else if (sourceType === 'figma' && lspData.options && lspData.options.figma_file_key) {
      sourceId = lspData.options.figma_file_key;
    } else if (sourceType === 'dropbox' && lspData.options && lspData.options.dropbox_folder) {
      sourceId = lspData.options.dropbox_folder;
    } else {
      sourceId = 'default';
    }
    
    // Create distribution job
    jQuery.ajax({
      url: ajaxurl,
      method: 'POST',
      data: {
        action: 'lsp_hub_create_distribution',
        nonce: lspData.nonce,
        asset_ids: assetIds,
        site_ids: hubSites,
        source_type: sourceType,
        source_id: sourceId
      },
      success: function(response) {
        console.log('AJAX success response:', response);
        if (response.success) {
          var jobId = response.data.job_id;
          var total = response.data.total;
          addLog('Job created: ' + total + ' distributions');
          addLog('Job ID: ' + jobId);
          pollHubJobStatus(jobId, total, setProgress, addLog, progressSub, btn);
        } else {
          var errMsg = (response.data && response.data.error) ? response.data.error : 'Unknown error';
          console.error('AJAX error:', errMsg);
          addLog('Error: ' + errMsg);
          setTimeout(function() {
            document.getElementById('lsp-hub-progress-modal')?.remove();
            if (btn) {
              btn.disabled = false;
              btn.innerHTML = '<span class="dashicons dashicons-cloud-upload"></span> Sync Selected';
            }
          }, 3000);
        }
      },
      error: function(xhr, status, error) {
        console.error('AJAX network error:', status, error, xhr.responseText);
        addLog('Network error: ' + (error || status));
        addLog('Response: ' + (xhr.responseText || '').substring(0, 200));
        setTimeout(function() {
          document.getElementById('lsp-hub-progress-modal')?.remove();
          if (btn) {
            btn.disabled = false;
            btn.innerHTML = '<span class="dashicons dashicons-cloud-upload"></span> Sync Selected';
          }
        }, 5000);
      }
    });
  };
  
  function pollHubJobStatus(jobId, total, setProgress, addLog, progressSub, btn) {
    jQuery.ajax({
      url: ajaxurl,
      method: 'POST',
      data: {
        action: 'lsp_hub_job_status',
        nonce: lspData.nonce,
        job_id: jobId
      },
      success: function(response) {
        if (response.success && response.data) {
          var data = response.data;
          var stats = data.stats || {};
          var completed = parseInt(stats.completed || 0);
          var failed = parseInt(stats.failed || 0);
          var done = completed + failed;
          var pct = total > 0 ? Math.round((done / total) * 100) : 0;
          
          setProgress(pct);
          progressSub.textContent = 'Completed ' + done + ' of ' + total + (failed > 0 ? ' (' + failed + ' failed)' : '');
          addLog('Phase: ' + (data.phase || 'processing') + ' - ' + pct + '%');
          
          if (data.is_complete) {
            addLog('Distribution complete!');
            addLog('✅ Successful: ' + completed);
            if (failed > 0) addLog('❌ Failed: ' + failed);
            
            setTimeout(function() {
              document.getElementById('lsp-hub-progress-modal')?.remove();
              if (btn) {
                btn.disabled = false;
                btn.innerHTML = '<span class="dashicons dashicons-cloud-upload"></span> Sync Selected';
              }
              // Clear selected sites after distribution
              window.lspHubSelectedSites = [];
            }, 2000);
          } else {
            setTimeout(function() {
              pollHubJobStatus(jobId, total, setProgress, addLog, progressSub, btn);
            }, 2000);
          }
        } else {
          addLog('Waiting for job to start...');
          setTimeout(function() {
            pollHubJobStatus(jobId, total, setProgress, addLog, progressSub, btn);
          }, 2000);
        }
      },
      error: function(xhr, status, error) {
        addLog('Poll error: ' + (error || status));
        setTimeout(function() {
          pollHubJobStatus(jobId, total, setProgress, addLog, progressSub, btn);
        }, 3000);
      }
    });
  }

});