/* global LIGHTSYNCPRO, ajaxurl, lspProgress */

// ========== Auto-Save Helper (Global) ==========
var lspAutoSave = (function() {
  // Add CSS animation for spinner
  document.addEventListener('DOMContentLoaded', function() {
    if (!document.getElementById('lsp-autosave-styles')) {
      var style = document.createElement('style');
      style.id = 'lsp-autosave-styles';
      style.textContent = '@keyframes lsp-spin { from { transform: rotate(0deg); } to { transform: rotate(360deg); } }';
      document.head.appendChild(style);
    }
  });
  
  return function(action, data, feedbackEl, callback) {
    if (!window.LIGHTSYNCPRO) return Promise.reject('LIGHTSYNCPRO not defined');
    
    // Show saving indicator
    if (feedbackEl) {
      feedbackEl.innerHTML = '<span style="display:inline-block;animation:lsp-spin 0.8s linear infinite;">↻</span>';
      feedbackEl.style.display = 'inline';
      feedbackEl.style.color = '#64748b';
      feedbackEl.style.marginLeft = '8px';
      feedbackEl.style.fontSize = '12px';
    }
    
    var body = new URLSearchParams();
    body.set('action', action);
    body.set('_ajax_nonce', LIGHTSYNCPRO.nonce);
    for (var key in data) {
      if (data.hasOwnProperty(key)) {
        body.set(key, data[key]);
      }
    }
    
    return fetch(LIGHTSYNCPRO.ajaxurl, {
      method: 'POST',
      credentials: 'same-origin',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: body.toString()
    })
    .then(function(r) { return r.json(); })
    .then(function(resp) {
      if (feedbackEl) {
        if (resp.success) {
          feedbackEl.innerHTML = '✓';
          feedbackEl.style.color = '#22c55e';
          setTimeout(function() { feedbackEl.style.display = 'none'; }, 1500);
        } else {
          feedbackEl.innerHTML = '✗';
          feedbackEl.style.color = '#ef4444';
          setTimeout(function() { feedbackEl.style.display = 'none'; }, 3000);
        }
      }
      if (callback) callback(resp);
      return resp;
    })
    .catch(function(err) {
      if (feedbackEl) {
        feedbackEl.innerHTML = '✗';
        feedbackEl.style.color = '#ef4444';
        setTimeout(function() { feedbackEl.style.display = 'none'; }, 3000);
      }
      console.error('[LSP] Auto-save failed:', err);
    });
  };
})();

// ===== Wipe on Deactivate Confirmation =====
document.addEventListener('DOMContentLoaded', function() {
  var cb = document.querySelector('input[name="lightsyncpro_settings[wipe_on_deactivate]"]');
  if (!cb) return;
  cb.addEventListener('change', function(e) {
    if (cb.checked) {
      var ok = confirm(
        'Enabling this means ALL LightSync Pro data (settings, catalog, albums, usage, cron) ' +
        'will be deleted from the database whenever the plugin is deactivated.\n\n' +
        'Are you sure you want to enable this?'
      );
      if (!ok) {
        cb.checked = false;
      }
    }
  });
});

// ===== Keep Original Filename Badge Toggle =====
document.addEventListener('DOMContentLoaded', function() {
  const checkbox = document.getElementById('lsp-keep-original');
  const badge = document.getElementById('lsp-name-mode-badge');

  if (!checkbox || !badge) return;

  function updateBadge() {
    const isOriginal = checkbox.checked;
    badge.textContent = isOriginal ? 'Original' : 'Custom';
    badge.title = badge.textContent;
    badge.classList.toggle('is-original', isOriginal);
    badge.classList.toggle('is-custom', !isOriginal);
  }

  checkbox.addEventListener('change', updateBadge);
  updateBadge();
});

// ===== Toast & Modal Helpers =====
document.addEventListener('DOMContentLoaded', function() {
  function lspToast(message, type) {
    var el = document.getElementById('lsp-toast');
    if (!el) return;

    el.innerHTML = '';

    var icon = document.createElement('div');
    icon.className = 'lsp-toast-icon';

    var svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
    svg.setAttribute('viewBox', '0 0 24 24');
    svg.innerHTML = (type === 'error')
      ? '<path d="M12 9v4m0 4h.01M12 3l9 18H3l9-18z" fill="none" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>'
      : '<path d="M5 13l4 4L19 7" fill="none" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';

    icon.appendChild(svg);

    var text = document.createElement('div');
    text.textContent = message || 'Done.';

    el.appendChild(icon);
    el.appendChild(text);

    el.classList.remove('is-visible');
    void el.offsetWidth;
    el.classList.add('is-visible');

    clearTimeout(el._lspHideTimer);
    el._lspHideTimer = setTimeout(function() {
      el.classList.remove('is-visible');
    }, 2800);
  }

  window.lspToast = lspToast;

  function lspModal(opts) {
    opts = opts || {};
    var m = document.getElementById('lsp-modal');
    if (!m) return;

    if (!m._lspPortaled) {
      document.body.appendChild(m);
      m._lspPortaled = true;
    }

    var titleEl = document.getElementById('lsp-modal-title');
    var bodyEl = document.getElementById('lsp-modal-body');
    var iconEl = document.getElementById('lsp-modal-icon');
    var primary = document.getElementById('lsp-modal-primary');

    if (titleEl) titleEl.textContent = opts.title || 'Notice';
    if (bodyEl) bodyEl.innerHTML = opts.body || '';
    if (iconEl) iconEl.textContent = (opts.type === 'error') ? '!' : 'i';

    if (primary) {
      if (opts.primaryText && opts.primaryHref) {
        primary.style.display = '';
        primary.textContent = opts.primaryText;
        primary.setAttribute('href', opts.primaryHref);
      } else {
        primary.style.display = 'none';
      }
    }

    document.body.classList.add('lsp-modal-open');
    m.setAttribute('aria-hidden', 'false');

    m.querySelectorAll('[data-lsp-close]').forEach(function(btn) {
      btn.addEventListener('click', closeOnce, { once: true });
    });

    function closeOnce() {
      m.setAttribute('aria-hidden', 'true');
    }

    function onKey(e) {
      if (e.key === 'Escape') {
        closeOnce();
        document.removeEventListener('keydown', onKey);
      }
    }
    document.addEventListener('keydown', onKey);
  }

  window.lspModal = lspModal;

  function selectedVals(select) {
    return Array.from(select ? select.options : [])
      .filter(function(o) { return o.selected; })
      .map(function(o) { return o.value; });
  }

  function getSelection() {
    var catalogEl = document.getElementById('lsp-catalog');
    var albumsEl = document.getElementById('lsp-albums');
    var catalog = catalogEl ? (catalogEl.value || '') : '';
    var albums = selectedVals(albumsEl);
    return { catalog: catalog, albums: albums };
  }

  function requireAlbums(actionLabel) {
    var sel = getSelection();
    if (!sel.catalog || !sel.albums.length) {
      lspModal({
        title: actionLabel + ' requires an album',
        body: '<p><strong>Select at least one album</strong> to continue.</p>' +
          '<p>Go to <em>Catalog & Albums</em> and pick the album(s) you want LightSync Pro to sync.</p>',
        type: 'info',
        primaryText: 'Select Albums',
        primaryHref: '#lsp-pick'
      });
      return null;
    }
    return sel;
  }

  // Wire Estimate button
  var estimateBtn = document.getElementById('lsp-estimate');
  var estimateOut = document.getElementById('lsp-estimate-out');

  if (estimateBtn) {
    estimateBtn.addEventListener('click', function(e) {
      e.preventDefault();

      var sel = requireAlbums('Estimate Import');
      if (!sel) return;

      estimateBtn.disabled = true;

      var body = new URLSearchParams();
      body.set('action', 'lightsyncpro_ajax_estimate');
      body.set('_ajax_nonce', (window.LIGHTSYNCPRO && LIGHTSYNCPRO.nonce) ? LIGHTSYNCPRO.nonce : '');
      body.set('catalog', sel.catalog);
      sel.albums.forEach(function(a) { body.append('albums[]', a); });

      fetch((window.LIGHTSYNCPRO && LIGHTSYNCPRO.ajaxurl) ? LIGHTSYNCPRO.ajaxurl : ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      })
      .then(function(r) { return r.json(); })
      .then(function(json) {
        if (!json || json.success !== true) {
          var msg = (json && json.data && (json.data.message || json.data)) ? (json.data.message || json.data) : 'Estimate failed.';
          lspToast(String(msg), 'error');
          return;
        }

        var total = (json.data && typeof json.data.total !== 'undefined') ? parseInt(json.data.total, 10) : 0;
        if (!Number.isFinite(total)) total = 0;

        if (estimateOut) {
          estimateOut.style.display = '';
          estimateOut.textContent = ' — ' + total + ' photos';
        }

        lspToast('Estimated ' + total + ' photos', 'success');
      })
      .catch(function(err) {
        lspToast('Estimate error: ' + (err && err.message ? err.message : err), 'error');
      })
      .finally(function() {
        estimateBtn.disabled = false;
      });
    });
  }

});

// ===== Save Button Handler =====
document.addEventListener('DOMContentLoaded', function() {
  (function waitForConfig(retries) {
    if (window.LIGHTSYNCPRO) return initSave();
    if ((retries || 0) > 120) return;
    setTimeout(function() { waitForConfig((retries || 0) + 1); }, 30);
  })();

  function isOk(resp) {
    if (!resp || typeof resp !== 'object') return false;
    if (resp.success === true) return true;
    if (resp.ok === true) return true;
    if (resp.saved === true) return true;
    if (resp.data && (resp.data.saved === true)) return true;
    return false;
  }

  function initSave() {
    var saveBtn = document.getElementById('lsp-save');
    var savedMsg = document.getElementById('lsp-saved');
    var form = document.querySelector('form.lsp-card-group');
    var catalogEl = document.getElementById('lsp-catalog');
    var albumsEl = document.getElementById('lsp-albums');
    var schedulesWrap = document.getElementById('lsp-album-schedules');

    // Create feedback element for form auto-save
    var formFeedback = document.createElement('span');
    formFeedback.id = 'lsp-form-feedback';
    formFeedback.style.display = 'none';
    formFeedback.style.fontWeight = '500';
    formFeedback.style.marginLeft = '8px';

    function saveOptions(feedbackEl) {
      if (!form) return Promise.resolve({ ok: true, skipped: true });
      var fd = new FormData(form);

      var body = new URLSearchParams();
      body.set('action', 'lightsyncpro_ajax_save_options');
      body.set('_ajax_nonce', LIGHTSYNCPRO.nonce);

      for (var pair of fd.entries()) {
        var k = pair[0];
        if (k === 'action' || k === 'submit' || k === '_wp_http_referer') continue;
        body.append(k, pair[1]);
      }

      body.set('action', 'lightsyncpro_ajax_save_options');

      // Show saving indicator
      if (feedbackEl) {
        feedbackEl.innerHTML = '<span style="display:inline-block;animation:lsp-spin 0.8s linear infinite;">↻</span>';
        feedbackEl.style.display = 'inline';
        feedbackEl.style.color = '#64748b';
      }

      return fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      }).then(function(r) { return r.json(); })
      .then(function(resp) {
        if (feedbackEl) {
          if (resp.success || resp.ok || resp.saved) {
            feedbackEl.innerHTML = '✓';
            feedbackEl.style.color = '#22c55e';
          } else {
            feedbackEl.innerHTML = '✗';
            feedbackEl.style.color = '#ef4444';
          }
          setTimeout(function() { feedbackEl.style.display = 'none'; }, 1500);
        }
        return resp;
      });
    }

    // Auto-save form fields on change (except album selection)
    if (form) {
      var autoSaveDebounce;
      
      // Find a good place to show feedback
      var namingSection = form.querySelector('section');
      if (namingSection) {
        var sectionHeader = namingSection.querySelector('h2, h3, .section-title');
        if (sectionHeader) sectionHeader.appendChild(formFeedback);
      }
      
      form.querySelectorAll('input[type="text"], input[type="number"], textarea').forEach(function(el) {
        // Skip catalog/album selectors
        if (el.id === 'lsp-catalog' || el.id === 'lsp-albums') return;
        el.addEventListener('blur', function() {
          saveOptions(formFeedback);
        });
      });
      
      form.querySelectorAll('input[type="checkbox"], input[type="radio"], select').forEach(function(el) {
        // Skip catalog/album selectors and schedule selects
        if (el.id === 'lsp-catalog' || el.id === 'lsp-albums') return;
        if (el.closest('#lsp-album-schedules')) return;
        el.addEventListener('change', function() {
          saveOptions(formFeedback);
        });
      });
    }

    // Auto-save album selection when it changes
    function autoSaveSelection() {
      if (!catalogEl && !albumsEl && !schedulesWrap) return;
      
      var body = new URLSearchParams();
      body.set('action', 'lightsyncpro_ajax_save_selection');
      body.set('_ajax_nonce', LIGHTSYNCPRO.nonce);

      if (catalogEl && catalogEl.value) body.set('catalog', catalogEl.value);
      if (albumsEl) {
        selectedVals(albumsEl).forEach(function(a) { body.append('albums[]', a); });
      }
      var schedules = collectSchedules();
      Object.keys(schedules).forEach(function(aid) {
        body.append('schedules[' + aid + ']', schedules[aid]);
      });

      return fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      }).then(function(r) { return r.json(); });
    }

    // Auto-save on catalog/album/schedule change
    if (catalogEl) {
      catalogEl.addEventListener('change', function() {
        autoSaveSelection();
      });
    }
    if (albumsEl) {
      albumsEl.addEventListener('change', function() {
        autoSaveSelection();
      });
    }
    if (schedulesWrap) {
      schedulesWrap.addEventListener('change', function() {
        autoSaveSelection();
      }, true);
    }
  }
});

// ===== Scrollspy Navigation =====
document.addEventListener('DOMContentLoaded', function() {
  const links = Array.from(document.querySelectorAll('#nav a'));
  const targets = links.map(a => {
    const href = a.getAttribute('href');
    if (!href || !href.startsWith('#')) return null;
    try { return document.querySelector(href); } catch (e) { return null; }
  }).filter(Boolean);

  function onScroll() {
    const fromTop = window.scrollY + 150;
    let activeId = targets[0]?.id;
    for (const t of targets) {
      if (t.offsetTop <= fromTop) activeId = t.id;
    }
    links.forEach(a => {
      const href = a.getAttribute('href');
      if (href && href.startsWith('#')) {
        a.classList.toggle('active', href === '#' + activeId);
      }
    });
  }
  window.addEventListener('scroll', onScroll, { passive: true });
  onScroll();
});




// ===== AI Settings (separate block to ensure it always runs) =====
document.addEventListener('DOMContentLoaded', function() {
  if (!window.LIGHTSYNCPRO) {
    console.warn('LSP: LIGHTSYNCPRO not defined');
    return;
  }

  // ========== AI Tab Switching ==========
  var aiTabs = document.querySelectorAll('[data-ai-tab]');
  aiTabs.forEach(function(tab) {
    tab.addEventListener('click', function(e) {
      e.preventDefault();
      var targetTab = this.getAttribute('data-ai-tab');
      if (!targetTab) return;
      
      // Update active tab
      aiTabs.forEach(function(t) { t.classList.remove('active'); });
      this.classList.add('active');
      
      // Show/hide content
      document.querySelectorAll('.lsp-ai-tab-content').forEach(function(c) { 
        c.style.display = 'none'; 
      });
      var targetContent = document.getElementById('lsp-ai-' + targetTab + '-content');
      if (targetContent) {
        targetContent.style.display = 'block';
      }
    });
  });

  // ========== AI Provider Dropdown Toggle ==========
  var aiProvider = document.getElementById('lsp-ai-provider');
  var aiAnthropicRow = document.getElementById('lsp-ai-anthropic-row');
  var aiOpenaiRow = document.getElementById('lsp-ai-openai-row');
  var aiFeaturesRow = document.getElementById('lsp-ai-features-row');
  var aiStatus = document.getElementById('lsp-ai-status');
  var aiStatusText = document.getElementById('lsp-ai-status-text');

  if (aiProvider) {
    aiProvider.addEventListener('change', function() {
      var val = this.value;
      if (aiAnthropicRow) aiAnthropicRow.style.display = val === 'anthropic' ? '' : 'none';
      if (aiOpenaiRow) aiOpenaiRow.style.display = val === 'openai' ? '' : 'none';
      if (aiFeaturesRow) aiFeaturesRow.style.display = val ? '' : 'none';
      if (aiStatus) aiStatus.style.display = val ? 'block' : 'none';
      if (aiStatusText) aiStatusText.textContent = val ? 'AI configured with ' + val.charAt(0).toUpperCase() + val.slice(1) : '';
    });
  }

  // ========== AI Settings (Auto-Save) ==========
  var aiPerfToggle = document.getElementById('lsp-ai-performance-toggle');
  var aiAnthropicKey = document.getElementById('lsp-ai-anthropic-key');
  var aiOpenaiKey = document.getElementById('lsp-ai-openai-key');
  var aiAutoAlt = document.getElementById('lsp-ai-auto-alt');
  
  // Create feedback elements
  var aiPerfFeedback = document.createElement('span');
  aiPerfFeedback.id = 'lsp-ai-perf-feedback';
  aiPerfFeedback.style.display = 'none';
  aiPerfFeedback.style.fontWeight = '500';
  
  var aiVisualFeedback = document.createElement('span');
  aiVisualFeedback.id = 'lsp-ai-visual-feedback';
  aiVisualFeedback.style.display = 'none';
  aiVisualFeedback.style.fontWeight = '500';
  
  // Unified AI save function
  function saveAiSettings(feedbackEl, callback) {
    var data = {
      ai_performance_optimize: aiPerfToggle && aiPerfToggle.checked ? '1' : '0',
      ai_provider: aiProvider ? aiProvider.value : '',
      ai_anthropic_key: aiAnthropicKey ? aiAnthropicKey.value : '',
      ai_openai_key: aiOpenaiKey ? aiOpenaiKey.value : '',
      ai_auto_alt: aiAutoAlt && aiAutoAlt.checked ? '1' : '0'
    };
    lspAutoSave('lightsyncpro_ajax_save_ai', data, feedbackEl, callback);
  }
  
  // Update status tabs after save
  function updateAiStatusTabs(resp) {
    if (!resp || !resp.data) return;
    
    // Update performance tab
    var perfTab = document.querySelector('[data-ai-tab="performance"] .lsp-source-status');
    if (perfTab) {
      perfTab.className = 'lsp-source-status ' + (resp.data.performance ? 'connected' : '');
      perfTab.textContent = resp.data.performance ? '✓ Tracking' : 'Not enabled';
    }
    
    // Update visual tab
    var visualTab = document.querySelector('[data-ai-tab="visual"] .lsp-source-status');
    if (visualTab) {
      visualTab.className = 'lsp-source-status ' + (resp.data.provider ? 'connected' : '');
      visualTab.textContent = resp.data.provider ? '✓ ' + resp.data.provider.charAt(0).toUpperCase() + resp.data.provider.slice(1) : 'Optional';
    }
    
    // Update status box
    if (aiStatus) aiStatus.style.display = resp.data.provider ? 'block' : 'none';
    if (aiStatusText) aiStatusText.textContent = resp.data.provider ? 'AI configured with ' + resp.data.provider.charAt(0).toUpperCase() + resp.data.provider.slice(1) : '';
  }
  
  // Performance toggle - auto-save
  if (aiPerfToggle) {
    var perfLabel = aiPerfToggle.closest('label');
    if (perfLabel) perfLabel.after(aiPerfFeedback);
    
    aiPerfToggle.addEventListener('change', function() {
      saveAiSettings(aiPerfFeedback, updateAiStatusTabs);
    });
  }
  
  // Provider dropdown - auto-save
  if (aiProvider) {
    aiProvider.addEventListener('change', function() {
      var val = this.value;
      if (aiAnthropicRow) aiAnthropicRow.style.display = val === 'anthropic' ? '' : 'none';
      if (aiOpenaiRow) aiOpenaiRow.style.display = val === 'openai' ? '' : 'none';
      if (aiFeaturesRow) aiFeaturesRow.style.display = val ? '' : 'none';
      saveAiSettings(aiVisualFeedback, updateAiStatusTabs);
    });
  }
  
  // API keys - save on blur
  if (aiAnthropicKey) {
    aiAnthropicKey.addEventListener('blur', function() {
      saveAiSettings(aiVisualFeedback, updateAiStatusTabs);
    });
  }
  if (aiOpenaiKey) {
    aiOpenaiKey.addEventListener('blur', function() {
      saveAiSettings(aiVisualFeedback, updateAiStatusTabs);
    });
  }
  
  // Auto-alt toggle
  if (aiAutoAlt) {
    aiAutoAlt.addEventListener('change', function() {
      saveAiSettings(aiVisualFeedback, updateAiStatusTabs);
    });
  }

  // ========== Export CSV ==========
  var exportCsvBtn = document.getElementById('lsp-export-csv');
  if (exportCsvBtn) {
    exportCsvBtn.addEventListener('click', function() {
      exportCsvBtn.disabled = true;
      exportCsvBtn.innerHTML = '<span class="dashicons dashicons-update spin"></span> Exporting...';

      var body = new URLSearchParams();
      body.set('action', 'lsp_export_insights_csv');
      body.set('_wpnonce', LIGHTSYNCPRO.nonce);

      fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        credentials: 'same-origin',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      })
      .then(function(r) { return r.json(); })
      .then(function(resp) {
        exportCsvBtn.disabled = false;
        exportCsvBtn.innerHTML = '<span class="dashicons dashicons-download" style="font-size:14px;line-height:1.5;margin-right:4px;"></span> Export CSV';
        if (resp.success) {
          // Download the CSV
          var blob = new Blob([resp.data.csv], { type: 'text/csv' });
          var url = window.URL.createObjectURL(blob);
          var a = document.createElement('a');
          a.href = url;
          a.download = resp.data.filename;
          document.body.appendChild(a);
          a.click();
          window.URL.revokeObjectURL(url);
          document.body.removeChild(a);
        } else {
          alert('Export failed: ' + (resp.data && resp.data.error ? resp.data.error : 'Unknown error'));
        }
      })
      .catch(function(err) {
        exportCsvBtn.disabled = false;
        exportCsvBtn.innerHTML = '<span class="dashicons dashicons-download" style="font-size:14px;line-height:1.5;margin-right:4px;"></span> Export CSV';
        alert('Export failed: ' + err.message);
      });
    });
  }

  // ========== Clear Tracking Data ==========
  var clearTrackingBtn = document.getElementById('lsp-clear-tracking');
  if (clearTrackingBtn) {
    clearTrackingBtn.addEventListener('click', function() {
      if (!confirm('Are you sure you want to clear ALL tracking data?\\n\\nThis will delete all impressions, clicks, and performance metrics. This action cannot be undone.')) {
        return;
      }

      clearTrackingBtn.disabled = true;
      clearTrackingBtn.innerHTML = '<span class="dashicons dashicons-update spin"></span> Clearing...';

      var body = new URLSearchParams();
      body.set('action', 'lsp_clear_tracking_data');
      body.set('_wpnonce', LIGHTSYNCPRO.nonce);

      fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        credentials: 'same-origin',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      })
      .then(function(r) { return r.json(); })
      .then(function(resp) {
        clearTrackingBtn.disabled = false;
        clearTrackingBtn.innerHTML = '<span class="dashicons dashicons-trash" style="font-size:14px;line-height:1.5;margin-right:4px;"></span> Clear All Data';
        if (resp.success) {
          alert('Tracking data cleared successfully. ' + resp.data.deleted + ' records removed.');
          window.location.reload();
        } else {
          alert('Clear failed: ' + (resp.data && resp.data.error ? resp.data.error : 'Unknown error'));
        }
      })
      .catch(function(err) {
        clearTrackingBtn.disabled = false;
        clearTrackingBtn.innerHTML = '<span class="dashicons dashicons-trash" style="font-size:14px;line-height:1.5;margin-right:4px;"></span> Clear All Data';
        alert('Clear failed: ' + err.message);
      });
    });
  }

  // ========== Data Retention (Auto-Save) ==========
  var retentionSelect = document.getElementById('lsp-tracking-retention');
  
  if (retentionSelect) {
    var retentionFeedback = document.createElement('span');
    retentionFeedback.style.display = 'none';
    retentionFeedback.style.fontWeight = '500';
    retentionSelect.after(retentionFeedback);
    
    retentionSelect.addEventListener('change', function() {
      var data = {
        tracking_retention_days: retentionSelect.value,
        ai_performance_optimize: aiPerfToggle && aiPerfToggle.checked ? '1' : '0',
        ai_provider: aiProvider ? aiProvider.value : '',
        ai_anthropic_key: document.getElementById('lsp-ai-anthropic-key') ? document.getElementById('lsp-ai-anthropic-key').value : '',
        ai_openai_key: document.getElementById('lsp-ai-openai-key') ? document.getElementById('lsp-ai-openai-key').value : ''
      };
      lspAutoSave('lightsyncpro_ajax_save_ai', data, retentionFeedback);
    });
  }

  // ========== A/B Testing ==========
  function loadAbTests() {
    var listEl = document.getElementById('lsp-ab-tests-list');
    if (!listEl) return;

    var body = new URLSearchParams();
    body.set('action', 'lsp_ab_test_list');
    body.set('_wpnonce', LIGHTSYNCPRO.nonce);

    fetch(LIGHTSYNCPRO.ajaxurl, {
      method: 'POST',
      credentials: 'same-origin',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: body.toString()
    })
    .then(function(r) { return r.json(); })
    .then(function(resp) {
      if (!resp.success || !resp.data.tests || resp.data.tests.length === 0) {
        listEl.innerHTML = '<div style="text-align:center;padding:16px;color:#a16207;font-size:13px;">No A/B tests yet. Create one below to compare image performance.</div>';
        return;
      }

      var html = '';
      resp.data.tests.forEach(function(test) {
        var winnerIcon = '';
        var winnerLabel = '';
        if (test.winner === 'a') {
          winnerIcon = '🏆';
          winnerLabel = ' (Winner)';
        } else if (test.winner === 'b') {
          winnerIcon = '🏆';
          winnerLabel = ' (Winner)';
        }

        var confidenceColor = test.confidence === 'high' ? '#22c55e' : (test.confidence === 'medium' ? '#eab308' : '#94a3b8');
        var confidenceLabel = test.confidence === 'high' ? 'High confidence' : (test.confidence === 'medium' ? 'Medium confidence' : 'Collecting data...');

        html += '<div style="padding:12px;background:#fff;border:1px solid #fde68a;border-radius:8px;margin-bottom:10px;">';
        html += '<div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:10px;">';
        html += '<strong style="color:#92400e;">' + test.name + '</strong>';
        html += '<button type="button" class="lsp-ab-delete btn ghost" data-test-id="' + test.id + '" style="padding:2px 8px;font-size:11px;color:#dc2626;">Delete</button>';
        html += '</div>';
        
        html += '<div class="lsp-ab-result-grid" style="display:grid;grid-template-columns:1fr 1fr;gap:12px;font-size:12px;">';
        
        // Image A
        html += '<div style="padding:10px;background:' + (test.winner === 'a' ? '#f0fdf4' : '#f8fafc') + ';border-radius:6px;border:1px solid ' + (test.winner === 'a' ? '#bbf7d0' : '#e2e8f0') + ';">';
        if (test.image_a.thumb) {
          html += '<img src="' + test.image_a.thumb + '" style="width:40px;height:40px;object-fit:cover;border-radius:4px;float:left;margin-right:8px;">';
        }
        html += '<div><strong>A: ' + test.image_a.title + '</strong>' + (test.winner === 'a' ? ' ' + winnerIcon : '') + '</div>';
        html += '<div style="color:#64748b;">' + test.image_a.impressions + ' views · ' + test.image_a.clicks + ' clicks</div>';
        html += '<div style="font-weight:600;color:' + (test.winner === 'a' ? '#22c55e' : '#64748b') + ';">' + test.image_a.ctr + '% CTR</div>';
        html += '</div>';
        
        // Image B
        html += '<div style="padding:10px;background:' + (test.winner === 'b' ? '#f0fdf4' : '#f8fafc') + ';border-radius:6px;border:1px solid ' + (test.winner === 'b' ? '#bbf7d0' : '#e2e8f0') + ';">';
        if (test.image_b.thumb) {
          html += '<img src="' + test.image_b.thumb + '" style="width:40px;height:40px;object-fit:cover;border-radius:4px;float:left;margin-right:8px;">';
        }
        html += '<div><strong>B: ' + test.image_b.title + '</strong>' + (test.winner === 'b' ? ' ' + winnerIcon : '') + '</div>';
        html += '<div style="color:#64748b;">' + test.image_b.impressions + ' views · ' + test.image_b.clicks + ' clicks</div>';
        html += '<div style="font-weight:600;color:' + (test.winner === 'b' ? '#22c55e' : '#64748b') + ';">' + test.image_b.ctr + '% CTR</div>';
        html += '</div>';
        
        html += '</div>';
        
        // Confidence indicator
        html += '<div style="margin-top:10px;font-size:11px;color:' + confidenceColor + ';">';
        html += '<span style="display:inline-block;width:8px;height:8px;background:' + confidenceColor + ';border-radius:50%;margin-right:6px;"></span>';
        html += confidenceLabel;
        if (test.lift > 0 && test.confidence !== 'low') {
          html += ' · ' + test.lift + '% lift';
        }
        html += '</div>';
        
        html += '</div>';
      });

      listEl.innerHTML = html;

      // Attach delete handlers
      listEl.querySelectorAll('.lsp-ab-delete').forEach(function(btn) {
        btn.addEventListener('click', function() {
          if (!confirm('Delete this A/B test?')) return;
          
          var testId = this.getAttribute('data-test-id');
          var delBody = new URLSearchParams();
          delBody.set('action', 'lsp_ab_test_delete');
          delBody.set('_wpnonce', LIGHTSYNCPRO.nonce);
          delBody.set('test_id', testId);

          fetch(LIGHTSYNCPRO.ajaxurl, {
            method: 'POST',
            credentials: 'same-origin',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: delBody.toString()
          })
          .then(function(r) { return r.json(); })
          .then(function(resp) {
            if (resp.success) {
              loadAbTests();
            } else {
              alert('Delete failed');
            }
          });
        });
      });
    })
    .catch(function(err) {
      listEl.innerHTML = '<div style="text-align:center;padding:16px;color:#dc2626;font-size:13px;">Failed to load tests</div>';
    });
  }

  // Load A/B tests on page load
  loadAbTests();

  // ========== A/B Test Media Library Picker ==========
  
  // Helper to update preview for a target (a or b)
  function updateAbPreview(target, attachment) {
    var previewEl = document.getElementById('lsp-ab-preview-' + target);
    var thumbEl = document.getElementById('lsp-ab-thumb-' + target);
    var titleEl = document.getElementById('lsp-ab-title-' + target);
    var infoEl = document.getElementById('lsp-ab-info-' + target);
    var inputEl = document.getElementById('lsp-ab-image-' + target);

    if (!previewEl) return;

    if (attachment) {
      var thumbUrl = attachment.sizes && attachment.sizes.thumbnail 
        ? attachment.sizes.thumbnail.url 
        : attachment.url;
      
      thumbEl.src = thumbUrl;
      titleEl.textContent = attachment.title || attachment.filename || 'Untitled';
      infoEl.textContent = attachment.width + ' × ' + attachment.height + ' · ID: ' + attachment.id;
      inputEl.value = attachment.id;
      previewEl.style.display = 'block';
    } else {
      previewEl.style.display = 'none';
      thumbEl.src = '';
      titleEl.textContent = '';
      infoEl.textContent = '';
    }
  }

  // Open media library picker
  function openAbMediaPicker(target) {
    // Check if wp.media is available
    if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
      alert('Media library not available. Please enter the attachment ID manually.');
      return;
    }

    // Save scroll position before opening modal
    var scrollPos = window.scrollY || window.pageYOffset;

    var frame = wp.media({
      title: 'Select Image ' + target.toUpperCase() + ' for A/B Test',
      button: { text: 'Use This Image' },
      library: { type: 'image' },
      multiple: false
    });

    frame.on('select', function() {
      var attachment = frame.state().get('selection').first().toJSON();
      updateAbPreview(target, attachment);
      // Restore scroll position after selection
      setTimeout(function() {
        window.scrollTo(0, scrollPos);
      }, 50);
    });

    frame.on('close', function() {
      // Restore scroll position when modal closes (even without selection)
      setTimeout(function() {
        window.scrollTo(0, scrollPos);
      }, 50);
    });

    frame.open();
  }

  // Use event delegation for picker and clear buttons (more reliable)
  document.addEventListener('click', function(e) {
    // Handle picker button clicks
    var pickerBtn = e.target.closest('.lsp-ab-picker');
    if (pickerBtn) {
      e.preventDefault();
      var target = pickerBtn.getAttribute('data-target');
      if (target) {
        openAbMediaPicker(target);
      }
      return;
    }

    // Handle clear button clicks
    var clearBtn = e.target.closest('.lsp-ab-clear');
    if (clearBtn) {
      e.preventDefault();
      var target = clearBtn.getAttribute('data-target');
      if (target) {
        var inputEl = document.getElementById('lsp-ab-image-' + target);
        if (inputEl) inputEl.value = '';
        updateAbPreview(target, null);
      }
      return;
    }
  });

  // Handle manual ID entry - fetch attachment details
  ['a', 'b'].forEach(function(target) {
    var inputEl = document.getElementById('lsp-ab-image-' + target);
    if (!inputEl) return;

    var debounceTimer;
    inputEl.addEventListener('input', function() {
      var id = this.value.trim();
      
      clearTimeout(debounceTimer);
      
      if (!id || isNaN(id)) {
        updateAbPreview(target, null);
        return;
      }

      // Debounce the API call
      debounceTimer = setTimeout(function() {
        // Fetch attachment details via WordPress REST API
        fetch(LIGHTSYNCPRO.ajaxurl + '?action=query-attachments&query[p]=' + id + '&_wpnonce=' + LIGHTSYNCPRO.nonce, {
          method: 'POST',
          credentials: 'same-origin'
        })
        .then(function(r) { return r.json(); })
        .then(function(attachments) {
          if (attachments && attachments.length > 0) {
            var att = attachments[0];
            updateAbPreview(target, {
              id: att.id,
              title: att.title,
              filename: att.filename,
              url: att.url,
              width: att.width,
              height: att.height,
              sizes: att.sizes
            });
          } else {
            updateAbPreview(target, null);
          }
        })
        .catch(function() {
          updateAbPreview(target, null);
        });
      }, 500);
    });
  });

  // Create A/B test
  var abCreateBtn = document.getElementById('lsp-ab-create');
  if (abCreateBtn) {
    abCreateBtn.addEventListener('click', function() {
      var name = document.getElementById('lsp-ab-name').value.trim();
      var imageA = document.getElementById('lsp-ab-image-a').value.trim();
      var imageB = document.getElementById('lsp-ab-image-b').value.trim();
      var position = document.getElementById('lsp-ab-position').value;

      if (!name || !imageA || !imageB) {
        alert('Please fill in all required fields');
        return;
      }

      abCreateBtn.disabled = true;
      abCreateBtn.textContent = 'Creating...';

      var body = new URLSearchParams();
      body.set('action', 'lsp_ab_test_create');
      body.set('_wpnonce', LIGHTSYNCPRO.nonce);
      body.set('name', name);
      body.set('image_a', imageA);
      body.set('image_b', imageB);
      body.set('position', position);

      fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        credentials: 'same-origin',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      })
      .then(function(r) { return r.json(); })
      .then(function(resp) {
        abCreateBtn.disabled = false;
        abCreateBtn.textContent = 'Start A/B Test';
        if (resp.success) {
          document.getElementById('lsp-ab-name').value = '';
          document.getElementById('lsp-ab-image-a').value = '';
          document.getElementById('lsp-ab-image-b').value = '';
          // Clear previews
          updateAbPreview('a', null);
          updateAbPreview('b', null);
          loadAbTests();
        } else {
          alert('Create failed: ' + (resp.data && resp.data.error ? resp.data.error : 'Unknown error'));
        }
      })
      .catch(function(err) {
        abCreateBtn.disabled = false;
        abCreateBtn.textContent = 'Start A/B Test';
        alert('Create failed: ' + err.message);
      });
    });
  }

  /* ==================== SHOPIFY CONNECT/DISCONNECT ==================== */
  (function() {
    function post(action, extra) {
      var body = new URLSearchParams();
      body.set('action', action);
      body.set('_wpnonce', LIGHTSYNCPRO.nonce);
      if (extra) {
        Object.keys(extra).forEach(function(k) { body.set(k, extra[k]); });
      }
      return fetch(LIGHTSYNCPRO.ajaxurl, {
        method: 'POST',
        credentials: 'same-origin',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString()
      }).then(function(r) { return r.json(); });
    }

    var globalConnectedCard = document.getElementById('lsp-shopify-global-connected');
    var globalDisconnectedCard = document.getElementById('lsp-shopify-global-disconnected');
    var globalConnectBtn = document.getElementById('lsp-shopify-global-connect');
    var globalDisconnectBtn = document.getElementById('lsp-shopify-global-disconnect');

    function updateGlobalShopifyUI(connected) {
      if (connected) {
        if (globalDisconnectedCard) globalDisconnectedCard.style.display = 'none';
        if (globalConnectedCard) globalConnectedCard.style.display = 'flex';
      } else {
        if (globalDisconnectedCard) globalDisconnectedCard.style.display = 'flex';
        if (globalConnectedCard) globalConnectedCard.style.display = 'none';
      }
    }

    if (globalConnectBtn) {
      globalConnectBtn.addEventListener('click', function(e) {
        e.preventDefault();
        
        var state = 'lightsync_' + Math.random().toString(36).slice(2) + Date.now();
        var baseAdmin = LIGHTSYNCPRO.ajaxurl.replace('admin-ajax.php', 'admin.php?page=lightsyncpro');
        var returnUrl = baseAdmin + '&lsp_shopify_connected=1&state=' + encodeURIComponent(state);
        
        var url = LIGHTSYNCPRO.shopify_start +
          '&site=' + encodeURIComponent(window.location.origin + '/') +
          '&state=' + encodeURIComponent(state) +
          '&return=' + encodeURIComponent(returnUrl);
        
        window.location.href = url;
      });
    }

    if (globalDisconnectBtn) {
      globalDisconnectBtn.addEventListener('click', function(e) {
        e.preventDefault();
        
        if (!confirm('Disconnect from Shopify?\n\nYour synced files will remain in Shopify.')) {
          return;
        }
        
        post('lightsync_shopify_disconnect', {})
          .then(function(json) {
            if (json && json.success) {
              updateGlobalShopifyUI(false);
              if (window.lspToast) lspToast('Shopify disconnected', 'success');
              setTimeout(function() { window.location.reload(); }, 500);
            } else {
              var msg = (json && json.data && (json.data.message || json.data.error))
                ? (json.data.message || json.data.error)
                : 'Disconnect failed.';
              alert(msg);
            }
          })
          .catch(function(err) {
            alert('Disconnect error: ' + (err && err.message ? err.message : err));
          });
      });
    }

    // Also handle the "Click to connect" card in destination selectors
    $(document).on('click', '#lsp-dest-shopify-connect', function(e) {
      e.preventDefault();
      
      var state = 'lightsync_' + Math.random().toString(36).slice(2) + Date.now();
      var baseAdmin = LIGHTSYNCPRO.ajaxurl.replace('admin-ajax.php', 'admin.php?page=lightsyncpro');
      var returnUrl = baseAdmin + '&lsp_shopify_connected=1&state=' + encodeURIComponent(state);
      
      var url = LIGHTSYNCPRO.shopify_start +
        '&site=' + encodeURIComponent(window.location.origin + '/') +
        '&state=' + encodeURIComponent(state) +
        '&return=' + encodeURIComponent(returnUrl);
      
      window.location.href = url;
    });

    // Listen for OAuth callback via postMessage (popup flow)
    window.addEventListener('message', function(e) {
      if (e.data && e.data.type === 'lightsync_shopify_connected' && e.data.shop_domain) {
        updateGlobalShopifyUI(true);
        if (window.lspToast) lspToast('Shopify connected!', 'success');
      }
    });
  })();

// ========== AI Generate Module ==========
var lspAI = (function($) {
  'use strict';

  var state = {
    initialized: false,
    models: [],
    generating: false,
    preview: null,        // { base64, mime, model, prompt, aspect_ratio, is_free }
    browsePage: 1,
    browseTotal: 0
  };

  // ---- Init (called once when AI tab first shown) ----
  function init() {
    if (state.initialized) return;
    state.initialized = true;

    loadModels();
    loadBrowser();
    bindEvents();
  }

  // ---- Bind all DOM events ----
  function bindEvents() {
    // Generate button
    $(document).on('click', '#lsp-ai-generate-btn', handleGenerate);

    // Commit (save to media library)
    $(document).on('click', '#lsp-ai-commit-btn', handleCommit);

    // Regenerate (same prompt, new result)
    $(document).on('click', '#lsp-ai-regenerate-btn', function() {
      handleGenerate();
    });

    // Discard preview
    $(document).on('click', '#lsp-ai-discard-btn', function() {
      state.preview = null;
      $('#lsp-ai-preview').hide();
      $('#lsp-ai-generate-form').show();
    });

    // New generation after commit
    $(document).on('click', '#lsp-ai-new-btn', function() {
      state.preview = null;
      $('#lsp-ai-committed').hide();
      $('#lsp-ai-preview').hide();
      $('#lsp-ai-generate-form').show();
      $('#lsp-ai-prompt').val('').focus();
    });

    // Disconnect
    $(document).on('click', '#lsp-ai-disconnect-btn', function() {
      if (!confirm('Disconnect from OpenRouter? You can reconnect anytime.')) return;
      $.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
          action: 'lsp_openrouter_disconnect',
          _wpnonce: LIGHTSYNCPRO.nonce
        },
        success: function(resp) {
          if (resp.success) location.reload();
          else alert('Failed to disconnect: ' + (resp.data && resp.data.error || 'Unknown error'));
        }
      });
    });

    // Model dropdown change — update cost badge
    $(document).on('change', '#lsp-ai-model', function() {
      updateCostBadge($(this).val());
    });

    // Grid item click — open detail / regenerate
    $(document).on('click', '.lsp-ai-grid-item', function() {
      var id = $(this).data('id');
      openAssetDetail(id);
    });

    // Ctrl/Cmd+Enter to generate
    $(document).on('keydown', '#lsp-ai-prompt', function(e) {
      if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
        e.preventDefault();
        handleGenerate();
      }
    });
  }

  // ---- Load models from broker ----
  function loadModels() {
    var $sel = $('#lsp-ai-model');
    if (!$sel.length) return;

    $sel.html('<option value="">Loading models...</option>');

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_ai_get_models',
        _ajax_nonce: LIGHTSYNCPRO.nonce
      },
      success: function(resp) {
        if (!resp.success || !resp.data || !resp.data.models) {
          $sel.html('<option value="">Failed to load models</option>');
          return;
        }

        state.models = resp.data.models;
        var html = '';

        // Group: free models first
        var free = state.models.filter(function(m) { return m.is_free; });
        var paid = state.models.filter(function(m) { return !m.is_free; });

        if (free.length) {
          html += '<optgroup label="Free Models">';
          free.forEach(function(m) {
            html += '<option value="' + esc(m.id) + '" data-free="1" data-cost="0">'
                  + esc(m.name) + ' — Free ✓</option>';
          });
          html += '</optgroup>';
        }
        if (paid.length) {
          html += '<optgroup label="Paid Models">';
          paid.forEach(function(m) {
            var cost = m.cost_estimate ? ' ~$' + m.cost_estimate : '';
            html += '<option value="' + esc(m.id) + '" data-free="0" data-cost="' + esc(m.cost_estimate || '') + '">'
                  + esc(m.name) + cost + '</option>';
          });
          html += '</optgroup>';
        }

        if (!html) {
          html = '<option value="">No image models available</option>';
        }

        $sel.html(html);
        updateCostBadge($sel.val());
      },
      error: function() {
        $sel.html('<option value="">Error loading models</option>');
      }
    });
  }

  // ---- Update cost badge next to generate button ----
  function updateCostBadge(modelId) {
    var $badge = $('#lsp-ai-cost-label');
    if (!$badge.length) return;

    var opt = $('#lsp-ai-model option:selected');
    if (!opt.length) { $badge.hide(); return; }

    var isFree = opt.data('free') === 1 || opt.data('free') === '1';

    if (isFree) {
      $badge.text('Free ✓').removeClass('lsp-badge-warning').addClass('lsp-badge-success').show();
    } else {
      var cost = opt.data('cost');
      $badge.text(cost ? '~$' + cost + '/image' : 'Paid')
            .removeClass('lsp-badge-success').addClass('lsp-badge-warning').show();
    }
  }

  // ---- Generate preview ----
  function handleGenerate() {
    if (state.generating) return;

    var prompt = $.trim($('#lsp-ai-prompt').val());
    if (!prompt) {
      $('#lsp-ai-prompt').focus();
      return;
    }

    var model  = $('#lsp-ai-model').val();
    var aspect = $('#lsp-ai-aspect').val();
    var isFree = $('#lsp-ai-model option:selected').data('free') === 1;

    state.generating = true;
    $('#lsp-ai-generate-btn').prop('disabled', true);
    $('#lsp-ai-generating').show();
    $('#lsp-ai-committed').hide();

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_ai_generate',
        _ajax_nonce: LIGHTSYNCPRO.nonce,
        model: model,
        prompt: prompt,
        type: 'image',
        aspect_ratio: aspect,
        preview: 0
      },
      timeout: 180000,  // 3 min timeout for generation
      success: function(resp) {
        state.generating = false;
        $('#lsp-ai-generate-btn').prop('disabled', false);
        $('#lsp-ai-generating').hide();

        if (!resp.success) {
          var err = resp.data && resp.data.error || 'Generation failed';
          if (resp.data && resp.data.code === 'insufficient_credits') {
            err += '<br><a href="https://openrouter.ai/credits" target="_blank" class="button button-small" style="margin-top:8px">Add credits →</a>';
          }
          showNotice('#lsp-ai-generate-form', err, 'error');
          return;
        }

        // Extract image data from response
        var data = resp.data;
        var base64 = '';
        var mime = 'image/png';

        // OpenRouter returns images in content blocks
        if (data.choices && data.choices.length) {
          var content = data.choices[0].message && data.choices[0].message.content;
          if (Array.isArray(content)) {
            for (var i = 0; i < content.length; i++) {
              if (content[i].type === 'image_url' && content[i].image_url) {
                // data:image/png;base64,xxxxx
                var url = content[i].image_url.url || '';
                var match = url.match(/^data:([^;]+);base64,(.+)$/);
                if (match) {
                  mime = match[1];
                  base64 = match[2];
                  break;
                }
              }
            }
          }
        }

        // Fallback: broker may normalize the response
        if (!base64 && data.image_base64) {
          base64 = data.image_base64;
          mime = data.image_mime || mime;
        }

        if (!base64) {
          showNotice('#lsp-ai-generate-form', 'No image was returned. Try a different model or prompt.', 'error');
          return;
        }

        // Store preview state
        state.preview = {
          base64: base64,
          mime: mime,
          model: model,
          prompt: prompt,
          aspect_ratio: aspect,
          is_free: isFree
        };

        // Render preview
        var imgSrc = 'data:' + mime + ';base64,' + base64;
        $('#lsp-ai-preview-image').html(
          '<img src="' + imgSrc + '" style="max-width:100%;height:auto;border-radius:8px;display:block" alt="AI Generated Preview">'
        );
        $('#lsp-ai-preview').show();

        // Scroll to preview
        $('html, body').animate({ scrollTop: $('#lsp-ai-preview').offset().top - 60 }, 300);
      },
      error: function(xhr) {
        state.generating = false;
        $('#lsp-ai-generate-btn').prop('disabled', false);
        $('#lsp-ai-generating').hide();

        var msg = 'Generation request failed.';
        if (xhr.status === 402) {
          msg = 'Insufficient OpenRouter credits.';
        } else if (xhr.status === 0) {
          msg = 'Request timed out. Try a different model.';
        }
        showNotice('#lsp-ai-generate-form', msg, 'error');
      }
    });
  }

  // ---- Commit preview to media library ----
  function handleCommit() {
    if (!state.preview || !state.preview.base64) return;

    var $btn = $('#lsp-ai-commit-btn');
    $btn.prop('disabled', true).text('Saving...');

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_ai_commit',
        _ajax_nonce: LIGHTSYNCPRO.nonce,
        base64: state.preview.base64,
        mime: state.preview.mime,
        prompt: state.preview.prompt,
        model: state.preview.model,
        aspect_ratio: state.preview.aspect_ratio,
        is_free: state.preview.is_free ? 1 : 0
      },
      success: function(resp) {
        $btn.prop('disabled', false).text('✓ Use This — Save to Media Library');

        if (!resp.success) {
          showNotice('#lsp-ai-preview', resp.data && resp.data.error || 'Save failed', 'error');
          return;
        }

        var d = resp.data;
        var details = ' — ' + d.filename;
        if (d.width && d.height) details += ' (' + d.width + '×' + d.height + ')';
        if (d.filesize) details += ', ' + formatBytes(d.filesize);
        if (d.avif && d.avif.savings) details += ', AVIF saved ' + d.avif.savings + '%';

        $('#lsp-ai-committed-details').text(details);
        $('#lsp-ai-edit-link').attr('href', d.edit_url);

        // Show committed state, hide preview
        $('#lsp-ai-preview').hide();
        $('#lsp-ai-committed').show();

        // Update KPI count
        window.lspAiCount = (window.lspAiCount || 0) + 1;
        var $kpiVal = $('#lsp-kpi-albums-value');
        if ($kpiVal.length && $('.lsp-source-tab.active').data('source') === 'ai') {
          $kpiVal.text(window.lspAiCount);
        }

        // Refresh browser grid
        loadBrowser();

        // Clear preview state
        state.preview = null;
      },
      error: function() {
        $btn.prop('disabled', false).text('✓ Use This — Save to Media Library');
        showNotice('#lsp-ai-preview', 'Network error saving image.', 'error');
      }
    });
  }

  // ---- Load AI asset browser grid ----
  function loadBrowser() {
    var $grid = $('#lsp-ai-grid');
    var $empty = $('#lsp-ai-grid-empty');
    if (!$grid.length) return;

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_ai_browse',
        _ajax_nonce: LIGHTSYNCPRO.nonce,
        page: state.browsePage,
        per_page: 40
      },
      success: function(resp) {
        if (!resp.success || !resp.data) return;

        var items = resp.data.items || [];
        state.browseTotal = resp.data.total || 0;
        window.lspAiCount = state.browseTotal;

        if (!items.length) {
          $grid.empty();
          $empty.show();
          return;
        }

        $empty.hide();
        var html = '';

        items.forEach(function(item) {
          var thumb = item.thumbnail || item.url;
          var modelLabel = (item.model || '').split('/').pop();
          var versionBadge = item.version > 1 ? '<span class="lsp-ai-version-badge">v' + item.version + '</span>' : '';
          var destDots = '';
          if (item.destinations && item.destinations.length) {
            item.destinations.forEach(function(d) {
              var color = d.is_current ? '#46b450' : '#f0b849';
              destDots += '<span style="display:inline-block;width:6px;height:6px;border-radius:50%;background:' + color + ';margin-right:2px" title="' + esc(d.key) + (d.is_current ? ' ✓' : ' stale') + '"></span>';
            });
          }

          html += '<div class="lsp-ai-grid-item" data-id="' + item.id + '" '
               +  'style="position:relative;cursor:pointer;border-radius:6px;overflow:hidden;background:#f5f5f5;aspect-ratio:1">'
               +  '<img src="' + esc(thumb) + '" style="width:100%;height:100%;object-fit:cover" loading="lazy" alt="">'
               +  '<div style="position:absolute;bottom:0;left:0;right:0;background:linear-gradient(transparent,rgba(0,0,0,.7));padding:4px 6px;color:#fff;font-size:10px;line-height:1.3">'
               +  '<div style="white-space:nowrap;overflow:hidden;text-overflow:ellipsis">' + esc(item.prompt || '') + '</div>'
               +  '<div style="display:flex;justify-content:space-between;align-items:center;margin-top:2px">'
               +  '<span style="opacity:.8">' + esc(modelLabel) + '</span>'
               +  '<span>' + versionBadge + destDots + '</span>'
               +  '</div></div></div>';
        });

        $grid.html(html);

        // Add pagination if needed
        if (resp.data.total_pages > 1) {
          var pageHtml = '<div style="grid-column:1/-1;text-align:center;padding:8px 0">';
          if (state.browsePage > 1) {
            pageHtml += '<button type="button" class="button button-small lsp-ai-page-btn" data-page="' + (state.browsePage - 1) + '">← Prev</button> ';
          }
          pageHtml += '<span style="color:#666;font-size:12px;margin:0 8px">Page ' + state.browsePage + ' of ' + resp.data.total_pages + '</span>';
          if (state.browsePage < resp.data.total_pages) {
            pageHtml += ' <button type="button" class="button button-small lsp-ai-page-btn" data-page="' + (state.browsePage + 1) + '">Next →</button>';
          }
          pageHtml += '</div>';
          $grid.append(pageHtml);

          $(document).off('click', '.lsp-ai-page-btn').on('click', '.lsp-ai-page-btn', function() {
            state.browsePage = $(this).data('page');
            loadBrowser();
          });
        }
      }
    });
  }

  // ---- Open asset detail modal (for regeneration) ----
  function openAssetDetail(attachmentId) {
    // Find item in current browse data
    var $item = $('.lsp-ai-grid-item[data-id="' + attachmentId + '"]');
    if (!$item.length) return;

    var src = $item.find('img').attr('src');

    // Simple detail overlay
    var overlayHtml = '<div id="lsp-ai-detail-overlay" style="position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,.7);z-index:100010;display:flex;align-items:center;justify-content:center;padding:20px">'
      + '<div style="background:#fff;border-radius:12px;max-width:700px;width:100%;max-height:90vh;overflow-y:auto;padding:24px;position:relative">'
      + '<button type="button" id="lsp-ai-detail-close" style="position:absolute;top:12px;right:12px;background:none;border:none;font-size:20px;cursor:pointer;color:#666">✕</button>'
      + '<img src="' + esc(src) + '" style="width:100%;border-radius:8px;margin-bottom:16px" alt="">'
      + '<div id="lsp-ai-detail-meta" style="color:#666;font-size:12px;margin-bottom:12px">Loading...</div>'
      + '<div style="display:flex;gap:8px">'
      + '<a href="' + LIGHTSYNCPRO.adminurl + 'post.php?post=' + attachmentId + '&action=edit" class="button" target="_blank">Edit in Media Library →</a>'
      + '</div></div></div>';

    $('body').append(overlayHtml);

    // Close handlers
    $(document).on('click', '#lsp-ai-detail-close, #lsp-ai-detail-overlay', function(e) {
      if (e.target === this || e.target.id === 'lsp-ai-detail-close') {
        $('#lsp-ai-detail-overlay').remove();
      }
    });
    $(document).on('keydown.aidetail', function(e) {
      if (e.key === 'Escape') {
        $('#lsp-ai-detail-overlay').remove();
        $(document).off('keydown.aidetail');
      }
    });

    // Load asset metadata via browse endpoint for this item
    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        action: 'lsp_ai_browse',
        _ajax_nonce: LIGHTSYNCPRO.nonce,
        page: 1,
        per_page: 1,
        attachment_id: attachmentId
      },
      success: function(resp) {
        if (!resp.success) return;
        var items = resp.data.items || [];
        // Find our item
        var item = null;
        for (var i = 0; i < items.length; i++) {
          if (items[i].id === attachmentId) { item = items[i]; break; }
        }
        if (!item && items.length) item = items[0];
        if (!item) return;

        var metaHtml = '<div style="margin-bottom:8px"><strong>Prompt:</strong> ' + esc(item.prompt || 'N/A') + '</div>'
          + '<div style="margin-bottom:8px"><strong>Model:</strong> ' + esc(item.model || 'N/A') + '</div>'
          + '<div style="margin-bottom:8px"><strong>Version:</strong> ' + (item.version || 1)
          + (item.is_free ? ' <span class="lsp-badge lsp-badge-success" style="font-size:10px">Free</span>' : '') + '</div>'
          + '<div style="margin-bottom:8px"><strong>Generated:</strong> ' + esc(item.generated_at || item.date || '') + '</div>';

        if (item.destinations && item.destinations.length) {
          metaHtml += '<div><strong>Synced to:</strong> ';
          item.destinations.forEach(function(d) {
            metaHtml += '<span style="display:inline-block;padding:2px 6px;background:' + (d.is_current ? '#e6f6e6' : '#fff3cd') + ';border-radius:3px;font-size:11px;margin-right:4px">'
              + esc(d.key) + (d.is_current ? ' ✓' : ' ⚠ stale') + '</span>';
          });
          metaHtml += '</div>';
        }

        $('#lsp-ai-detail-meta').html(metaHtml);
      }
    });
  }

  // ---- Utility: show notice ----
  function showNotice(after, message, type) {
    var $existing = $(after).siblings('.lsp-ai-notice');
    $existing.remove();

    var bgColor = type === 'error' ? '#fef2f2' : '#f0fdf4';
    var borderColor = type === 'error' ? '#fca5a5' : '#86efac';
    var textColor = type === 'error' ? '#991b1b' : '#166534';

    var $notice = $('<div class="lsp-ai-notice" style="padding:10px 14px;border-radius:6px;font-size:13px;margin:8px 0;'
      + 'background:' + bgColor + ';border:1px solid ' + borderColor + ';color:' + textColor + '">'
      + message + '</div>');

    $(after).after($notice);
    setTimeout(function() { $notice.fadeOut(300, function() { $(this).remove(); }); }, 8000);
  }

  // ---- Utility: escape HTML ----
  function esc(str) {
    if (!str) return '';
    var d = document.createElement('div');
    d.appendChild(document.createTextNode(str));
    return d.innerHTML;
  }

  // ---- Utility: format bytes ----
  function formatBytes(bytes) {
    if (!bytes) return '0 B';
    var k = 1024;
    var sizes = ['B', 'KB', 'MB', 'GB'];
    var i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
  }

  // Public API
  return { init: init };

})(jQuery);
});