<?php
/**
 * Plugin Name: LightSync Pro – Image Pipeline for Lightroom, Canva, Figma, Dropbox, Shutterstock & AI
 * Plugin URI: https://lightsyncpro.com
 * Description: 6-source image pipeline for WordPress & Shopify. Sync from Lightroom, Canva, Figma, Dropbox, and Shutterstock — or generate with 400+ AI models. Update once, sync everywhere.
 * Version: 2.1.4
 * Author: Tag Team Design
 * Author URI: https://tagteamdesign.com
 * Requires PHP: 7.4
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: lightsyncpro
 * Domain Path: /languages
 */

if ( ! defined( 'ABSPATH' ) ) exit;

// Memory optimization for sync operations
if (defined('DOING_AJAX') && DOING_AJAX) {
    $current_limit = ini_get('memory_limit');
    $current_bytes = wp_convert_hr_to_bytes($current_limit);
    $target_bytes = 768 * 1024 * 1024;
    if ($current_bytes < $target_bytes) {
        @ini_set('memory_limit', '768M');
    }
    if (function_exists('gc_enable')) {
        gc_enable();
    }
}

// Prevent conflict with paid Pro version
$pro_plugin_file = dirname( plugin_dir_path( __FILE__ ) ) . '/lightsyncpro-pro/lightsyncpro-pro.php';
if ( file_exists( $pro_plugin_file ) ) {
    $active_plugins = (array) get_option( 'active_plugins', [] );
    $pro_is_active = in_array( 'lightsyncpro-pro/lightsyncpro-pro.php', $active_plugins, true );
    if ( ! $pro_is_active && is_multisite() ) {
        $network_plugins = (array) get_site_option( 'active_sitewide_plugins', [] );
        $pro_is_active = isset( $network_plugins['lightsyncpro-pro/lightsyncpro-pro.php'] );
    }
    if ( $pro_is_active ) {
        add_action( 'admin_notices', function() {
            echo '<div class="notice notice-warning"><p><strong>LightSync Pro (Free)</strong> has been disabled because the paid version is active. You can safely delete the free version.</p></div>';
        });
        return;
    }
}

if ( ! defined( 'LIGHTSYNC_PRO' ) )         define( 'LIGHTSYNC_PRO', 'lightsyncpro' );
if ( ! defined( 'LIGHTSYNC_VERSION' ) )      define( 'LIGHTSYNC_VERSION', '2.1.4' );
if ( ! defined( 'LIGHTSYNC_PRO_VERSION' ) )  define( 'LIGHTSYNC_PRO_VERSION', '2.1.4' );
if ( ! defined( 'LIGHTSYNC_PRO_NAME' ) )     define( 'LIGHTSYNC_PRO_NAME', 'LightSync Pro' );
if ( ! defined( 'LIGHTSYNC_PRO_SLUG' ) )     define( 'LIGHTSYNC_PRO_SLUG', 'lightsyncpro' );
if ( ! defined( 'LIGHTSYNC_PRO_DIR' ) )      define( 'LIGHTSYNC_PRO_DIR', plugin_dir_path( __FILE__ ) );
if ( ! defined( 'LIGHTSYNC_PRO_URL' ) )      define( 'LIGHTSYNC_PRO_URL', plugin_dir_url( __FILE__ ) );
if ( ! defined( 'LIGHTSYNC_PRO_BASENAME' ) ) define( 'LIGHTSYNC_PRO_BASENAME', plugin_basename( __FILE__ ) );
if ( ! defined( 'LIGHTSYNC_PRO_FILE' ) )     define( 'LIGHTSYNC_PRO_FILE', __FILE__ );

// Remote sources config URL
if ( ! defined( 'LSP_SOURCES_CONFIG_URL' ) ) define( 'LSP_SOURCES_CONFIG_URL', 'https://lightsyncpro.com/api/sources.json' );

// Sync gate endpoint
if ( ! defined( 'LSP_SYNC_GATE_URL' ) ) define( 'LSP_SYNC_GATE_URL', 'https://lightsyncpro.com/wp-json/lsp-broker/v1/sync-gate' );

/**
 * Sync Gate - Check if syncing is allowed for this site.
 * Pings the broker once every 24 hours to verify sync permission.
 * Fails open (allows sync) if broker is unreachable to avoid breaking user experience.
 * 
 * @return array [ 'allowed' => bool, 'message' => string ]
 */
function lsp_check_sync_gate() {
    $transient_key = 'lsp_sync_gate_status';
    $cached = get_transient( $transient_key );
    
    // Return cached result if valid
    if ( $cached !== false && isset( $cached['allowed'] ) ) {
        return $cached;
    }
    
    // Build request payload
    $site_url = get_site_url();
    $site_hash = md5( $site_url ); // Privacy-friendly identifier
    
    $response = wp_remote_post( LSP_SYNC_GATE_URL, [
        'timeout'   => 10,
        'sslverify' => true,
        'headers'   => [ 'Content-Type' => 'application/json' ],
        'body'      => wp_json_encode( [
            'site_url'  => $site_url,
            'site_hash' => $site_hash,
            'version'   => LIGHTSYNC_VERSION,
            'php'       => PHP_VERSION,
            'wp'        => get_bloginfo( 'version' ),
        ] ),
    ] );
    
    // Fail open - if broker unreachable, allow syncing
    if ( is_wp_error( $response ) ) {
        $result = [ 'allowed' => true, 'message' => '' ];
        // Cache for 1 hour if failed (retry sooner)
        set_transient( $transient_key, $result, HOUR_IN_SECONDS );
        return $result;
    }
    
    $code = wp_remote_retrieve_response_code( $response );
    $body = wp_remote_retrieve_body( $response );
    $data = json_decode( $body, true );
    
    // Valid response from broker
    if ( $code === 200 && is_array( $data ) && isset( $data['allowed'] ) ) {
        $result = [
            'allowed' => (bool) $data['allowed'],
            'message' => sanitize_text_field( $data['message'] ?? '' ),
        ];
        // Cache for 24 hours on success
        set_transient( $transient_key, $result, DAY_IN_SECONDS );
        return $result;
    }
    
    // Unexpected response - fail open
    $result = [ 'allowed' => true, 'message' => '' ];
    set_transient( $transient_key, $result, HOUR_IN_SECONDS );
    return $result;
}

/**
 * Check sync gate and return WP_Error if blocked.
 * Use this at the start of sync handlers.
 * 
 * @return true|WP_Error True if allowed, WP_Error if blocked
 */
function lsp_gate_check() {
    $gate = lsp_check_sync_gate();
    if ( ! $gate['allowed'] ) {
        $message = $gate['message'] ?: 'Sync is currently disabled for this site. Please contact support.';
        return new \WP_Error( 'sync_disabled', $message );
    }
    return true;
}

/**
 * Clear sync gate cache (useful for testing or after admin action)
 */
function lsp_clear_sync_gate() {
    delete_transient( 'lsp_sync_gate_status' );
}

/**
 * Get source availability from remote config (cached)
 */
function lsp_get_source_status( $source ) {
    $defaults = [
        'lightroom'  => [ 'enabled' => true, 'message' => '' ],
        'canva'      => [ 'enabled' => false, 'message' => 'Coming Soon - Awaiting API approval' ],
        'figma'      => [ 'enabled' => true, 'message' => '' ],
        'dropbox'    => [ 'enabled' => true, 'message' => '' ],
        'openrouter' => [ 'enabled' => true, 'message' => '' ],
    ];
    $config = get_transient( 'lsp_sources_config' );
    if ( false === $config ) {
        $response = wp_remote_get( LSP_SOURCES_CONFIG_URL, [ 'timeout' => 5, 'sslverify' => true ] );
        if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200 ) {
            $body = wp_remote_retrieve_body( $response );
            $data = json_decode( $body, true );
            if ( $data && isset( $data['sources'] ) ) {
                $config = $data['sources'];
                set_transient( 'lsp_sources_config', $config, 6 * HOUR_IN_SECONDS );
            }
        }
        if ( ! $config ) {
            $config = $defaults;
            set_transient( 'lsp_sources_config', $config, 30 * MINUTE_IN_SECONDS );
        }
    }
    if ( isset( $config[ $source ] ) ) {
        return [
            'enabled' => (bool) ( $config[ $source ]['enabled'] ?? true ),
            'message' => (string) ( $config[ $source ]['message'] ?? '' ),
        ];
    }
    return $defaults[ $source ] ?? [ 'enabled' => true, 'message' => '' ];
}

function lsp_is_source_enabled( $source ) {
    $status = lsp_get_source_status( $source );
    return $status['enabled'];
}

function lsp_clear_sources_cache() {
    delete_transient( 'lsp_sources_config' );
}

/**
 * Brand configuration — always LightSync Pro for free version
 */
function lsp_get_brand( $force_refresh = false ) {
    static $brand = null;
    if ( $brand !== null && ! $force_refresh ) return $brand;
    $brand = [
        'name'            => 'LightSync Pro',
        'slug'            => 'lightsyncpro',
        'logo_white'      => LIGHTSYNC_PRO_URL . 'assets/lsp-logo-white.svg',
        'logo_dark'       => LIGHTSYNC_PRO_URL . 'assets/lsp-logo-dark.svg',
        'logo_gradient'   => LIGHTSYNC_PRO_URL . 'assets/lsp-icon.svg',
        'url'             => 'https://lightsyncpro.com',
        'docs_url'        => 'https://lightsyncpro.com/docs/',
        'support_url'     => 'https://lightsyncpro.com/support/',
        'is_enterprise'   => false,
    ];
    return $brand;
}

function lsp_is_enterprise() { return false; }

add_action( 'admin_init', function() {
    if ( isset( $_GET['lsp_clear_sources'] ) && current_user_can( 'manage_options' ) ) {
        lsp_clear_sources_cache();
        add_action( 'admin_notices', function() {
            echo '<div class="notice notice-success is-dismissible"><p>LightSync Pro: Sources config cache cleared.</p></div>';
        });
    }
});

add_filter('wp_editor_set_quality', function($q){ return 82; }, 10, 1);
add_filter('big_image_size_threshold', fn() => 2560);

// Autoloader
spl_autoload_register( function( $class ){
    if ( class_exists( $class, false ) ) return;
    $prefix = 'LightSyncPro\\';
    if ( strpos( $class, $prefix ) !== 0 ) return;
    $rel  = substr( $class, strlen( $prefix ) );
    $rel  = str_replace( '\\', '/', $rel );
    $dir  = strtolower( dirname( $rel ) );
    $dir  = ($dir === '.' ? '' : $dir . '/' );
    $base = basename( $rel );
    $slug = strtolower( str_replace( '_', '-', $base ) );
    $candidates = [
        LIGHTSYNC_PRO_DIR . 'includes/' . $dir . 'class-' . $slug . '.php',
        LIGHTSYNC_PRO_DIR . 'includes/' . strtolower( $rel ) . '.php',
        LIGHTSYNC_PRO_DIR . 'includes/' . $rel . '.php',
    ];
    foreach ( $candidates as $file ) {
        if ( file_exists( $file ) ) { require_once $file; return; }
    }
});

// Core requires
require_once LIGHTSYNC_PRO_DIR . 'includes/http/class-endpoints.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/compress/class-avif-php.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/http/class-client.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/oauth/class-oauth.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/oauth/class-canva-oauth.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/oauth/class-figma-oauth.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/oauth/class-dropbox-oauth.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/oauth/class-openrouter-oauth.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/oauth/class-shutterstock-oauth.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/ai/class-ai-generate.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/shopify/class-shopify.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/mapping/class-mapping.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/util/class-adobe.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/sync/class-sync.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/admin/class-admin.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/admin/class-weekly-digest.php';
require_once LIGHTSYNC_PRO_DIR . 'includes/util/helpers.php';

// Boot
add_action( 'plugins_loaded', function(){
    \LightSyncPro\OAuth\OAuth::init();
    \LightSyncPro\OAuth\CanvaOAuth::init();
    \LightSyncPro\OAuth\FigmaOAuth::init();
    \LightSyncPro\OAuth\DropboxOAuth::init();
    \LightSyncPro\OAuth\OpenRouterOAuth::init();
    \LightSyncPro\OAuth\ShutterstockOAuth::init();
    \LightSyncPro\AI\AIGenerate::init();
    \LightSyncPro\Admin\Admin::init();
    \LightSyncPro\Admin\WeeklyDigest::init();
    \LightSyncPro\Sync\Sync::init();
    if ( class_exists('\LightSyncPro\Blocks\Gallery_Block') ) \LightSyncPro\Blocks\Gallery_Block::init();
});

add_filter('upload_mimes', function($m){
    $m['webp'] = 'image/webp';
    return $m;
});

add_action('delete_attachment', function($post_id){
    $post_id = (int) $post_id;
    if (class_exists('\LightSyncPro\Mapping\Mapping')) {
        \LightSyncPro\Mapping\Mapping::delete_wp_mappings($post_id);
        return;
    }
    $asset_id = (string) get_post_meta($post_id, '_lightsync_asset_id', true);
    if (!$asset_id) return;
    if (class_exists('\LightSyncPro\Sync\Sync') && method_exists('\LightSyncPro\Sync\Sync','handle_attachment_deleted')) {
        \LightSyncPro\Sync\Sync::handle_attachment_deleted($post_id);
        return;
    }
    $map = (array) get_option('lightsync_asset_map', []);
    if (isset($map[$asset_id])) {
        $mapped_att = is_array($map[$asset_id])
            ? (int) ($map[$asset_id]['attachment_id'] ?? 0)
            : (int) $map[$asset_id];
        if ($mapped_att === $post_id) {
            unset($map[$asset_id]);
            update_option('lightsync_asset_map', $map, false);
        }
    }
}, 10, 1);
