<?php
namespace LightSyncPro\Sync;

use LightSyncPro\Admin\Admin;
use LightSyncPro\OAuth\OAuth;
use LightSyncPro\Http\Client;
use LightSyncPro\Util\Logger;
use LightSyncPro\Util\Adobe;
use LightSyncPro\Http\Endpoints;
use LightSyncPro\Util\Image;
use LightSyncPro\LightSync_Compress;


// phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_error_log

class Sync {

    // Cron hook names
    private const CRON_HOOK_SYNC   = 'lightsyncpro_sync_cron';
    private const CRON_HOOK_RETRY  = 'lightsync_retry_rendition';

    public static function init(){
        add_action('init', [ __CLASS__, 'register_taxonomy' ] );
        add_action('init', [__CLASS__, 'sweep_pending_renditions']);
        add_action('init', [__CLASS__, 'register_media_library_ui']);
        add_action('wp_ajax_lightsync_relink_attachment', [__CLASS__, 'ajax_relink_attachment']);
        add_action('wp_ajax_lightsync_unlink_attachment', [__CLASS__, 'ajax_unlink_attachment']);
        add_action('wp_ajax_lightsync_relink_candidates', [__CLASS__, 'ajax_relink_candidates']);
        add_action('wp_ajax_lightsync_switch_source',     [__CLASS__, 'ajax_switch_source']);
        add_action('delete_attachment', [__CLASS__, 'handle_attachment_deleted'], 10, 1);



      add_filter('get_terms_args', function($args, $taxonomies){
    // Only touch our taxonomy
    if (!in_array('lightsync_album', (array)$taxonomies, true)) {
        return $args;
    }

    // Must be an array
    if (!is_array($args)) {
        return $args;
    }

    // Current catalog scope
    $cat = self::get_catalog_id(); // <- not current_catalog_id()
    if (!$cat) {
        return $args;
    }

    // Normalize meta_query to an array
    $mq = [];
    if (isset($args['meta_query']) && is_array($args['meta_query'])) {
        $mq = $args['meta_query'];
    }

   $mq[] = [
        'key'     => '_lightsync_catalog_id',
        'value'   => $cat,
        'compare' => '=',
    ];

    // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
    $args['meta_query'] = $mq;
    return $args;
}, 10, 2);



        // Run a sync tick
        add_action(self::CRON_HOOK_SYNC, [__CLASS__, 'cron_sync'], 10, 3);

        // Handle a single asset rendition retry
        add_action(self::CRON_HOOK_RETRY, function($catalog_id, $album_id, $asset_id){

            $tok = \LightSyncPro\OAuth\OAuth::maybe_refresh();
            if (is_wp_error($tok)) {
                \LightSyncPro\Util\Logger::debug('[LSP] retry_rendition token refresh failed: ' . $tok->get_error_message());
                // try again soon – avoids losing the retry entirely
                if (!wp_next_scheduled(self::CRON_HOOK_RETRY, [$catalog_id, $album_id, $asset_id])) {
                    wp_schedule_single_event(time() + 300, self::CRON_HOOK_RETRY, [$catalog_id, $album_id, $asset_id]);
                }
                return;
            }

            // lightweight re-pull of that one asset membership and re-import
            $albs = self::get_albums($catalog_id);
            $album_name = '';
            if (!is_wp_error($albs)){
                foreach (($albs['resources'] ?? []) as $a){
                    if (!empty($a['id']) && $a['id'] === $album_id){
                        $album_name = $a['payload']['name'] ?? '';
                        break;
                    }
                }
            }

           $url  = "https://lr.adobe.io/v2/catalogs/{$catalog_id}/assets/{$asset_id}";
$resp = self::http_get_strict($url, 'json', ['timeout' => 20]);
if (!is_wp_error($resp)) {
    $body = wp_remote_retrieve_body($resp);
    $body = preg_replace('/^\s*while\s*\(\s*1\s*\)\s*\{\s*\}\s*/', '', (string)$body);
    $doc  = json_decode($body, true);

    if (is_array($doc)) {
        // Wrap into the shape import_or_update expects
        $res = [
            'asset'   => [ 'id' => $asset_id, 'payload' => ($doc['payload'] ?? []) ],
            'payload' => ($doc['payload'] ?? []),
            'updated' => ($doc['updated'] ?? ''),
        ];
        self::import_or_update($catalog_id, $album_id, $album_name, $res);
    }
}

        }, 10, 3);
    }

    private static function get_catalog_id(){
        return \LightSyncPro\Admin\Admin::get_opt('catalog_id');
    }
    private static function get_current_album(){
        return \LightSyncPro\Admin\Admin::get_opt('album_id');
    }

    private static function get_cached_album_name(string $catalog_id, string $album_id): string {
    $key = 'lightsync_album_name_' . md5($catalog_id . ':' . $album_id);
    $name = get_transient($key);
    if (is_string($name) && $name !== '') return $name;

    // fallback: one-time fetch if needed
    $albs = self::get_albums($catalog_id);
    if (!is_wp_error($albs)) {
        foreach (($albs['resources'] ?? []) as $a) {
            if (!empty($a['id']) && $a['id'] === $album_id) {
                $name = (string)($a['payload']['name'] ?? '');
                if ($name !== '') set_transient($key, $name, 12 * HOUR_IN_SECONDS);
                return $name;
            }
        }
    }
    return '';
}

public static function handle_attachment_deleted(int $post_id): void {
    if (get_post_type($post_id) !== 'attachment') return;

    $asset_id = (string) get_post_meta($post_id, '_lightsync_asset_id', true);
    if (!$asset_id) return;

    // Only unmark if the map STILL points to this attachment
    $mapped = 0;
    if (method_exists(__CLASS__, 'maybe_imported')) {
        $mapped = (int) self::maybe_imported($asset_id);
    }

    if ($mapped !== (int) $post_id) return;

    self::unmark($asset_id);
}

/**
 * Fetch assets from Lightroom WITHOUT importing to WordPress.
 * 
 *
 * Returns asset data array
 */
public static function fetch_assets(
    string $catalog_id,
    string $album_id,
    string $cursor = '',
    int $batch_size = 60,
    int $start_index = 0,
    $cap_remaining = null
): array {
    $started = microtime(true);
    $time_budget = 40.0;

    $batch_size = max(5, min($batch_size, 200));
    $start_index = max(0, $start_index);

    $target_total = is_null($cap_remaining) ? PHP_INT_MAX : (int)$cap_remaining;
    if ($target_total < 0) $target_total = 0;

    $current_cursor = $cursor !== '' ? $cursor : null;

    $assets = [];
    $skipped = 0;
    $fetched = 0;
    $hit_cap = false;

    $seen_assets = [];
    $cursor_history = [];
    $max_same_cursor = 2;

    $return_cursor = $current_cursor ?: '';
    $return_next_index = 0;

    $reason_counts = [
        'no_asset_id'      => 0,
        'not_image'        => 0,
        'deleted'          => 0,
        'no_rendition_url' => 0,
        'duplicate'        => 0,
        'other'            => 0,
    ];

    while ($fetched < $target_total && (microtime(true) - $started) < $time_budget) {

        // Cursor loop guard
        $cursor_hash = md5($current_cursor . $album_id);
        if (!isset($cursor_history[$cursor_hash])) {
            $cursor_history[$cursor_hash] = 0;
        }
        $cursor_history[$cursor_hash]++;

        if ($cursor_history[$cursor_hash] > $max_same_cursor) {
            Logger::debug("[LSP fetch_assets] Cursor loop detected, ending.");
            $return_cursor = '';
            $return_next_index = 0;
            break;
        }

        // Fetch page from Lightroom
        $data = self::get_album_assets($catalog_id, $album_id, $current_cursor, $batch_size);
        if (is_wp_error($data)) {
            return [
                'error'   => $data->get_error_message(),
                'assets'  => [],
                'cursor'  => $return_cursor,
                'next_index' => 0,
                'count'   => 0,
                'done_all' => false,
            ];
        }

        $resources = (array)($data['resources'] ?? []);
        $next_cursor = self::derive_next_cursor($data, (string)($current_cursor ?? ''));

        if (empty($resources)) {
            if ($next_cursor !== '' && $next_cursor !== $current_cursor) {
                $current_cursor = $next_cursor;
                $start_index = 0;
                $return_cursor = $current_cursor;
                continue;
            }
            $return_cursor = '';
            $return_next_index = 0;
            break;
        }

        if ($start_index < 0 || $start_index >= count($resources)) {
            $start_index = 0;
        }

        // Process resources
        for ($i = $start_index; $i < count($resources); $i++) {

            if ($fetched >= $target_total) {
                $hit_cap = true;
                break;
            }

            if ((microtime(true) - $started) >= ($time_budget - 3.0)) {
                $return_cursor = $current_cursor ?: '';
                $return_next_index = $i;
                break 2;
            }

            $res = $resources[$i];

            // Extract asset ID
            $asset_id = self::extract_asset_id_from_resource($res);
            if (!$asset_id) {
                $skipped++;
                $reason_counts['no_asset_id']++;
                continue;
            }

            // Dedupe
            if (isset($seen_assets[$asset_id])) {
                $skipped++;
                $reason_counts['duplicate']++;
                continue;
            }
            $seen_assets[$asset_id] = true;

            // Check if it's an importable image
            $asset = (array)($res['asset'] ?? []);
            $payload = (array)($asset['payload'] ?? ($res['payload'] ?? []));

            // Skip deleted/trashed
            $status = strtolower((string)($payload['status'] ?? ''));
            $state = strtolower((string)($payload['state'] ?? ''));
            if (in_array($status, ['deleted', 'trashed'], true) || 
                in_array($state, ['deleted', 'hidden', 'trashed'], true)) {
                $skipped++;
                $reason_counts['deleted']++;
                continue;
            }

            // Check mime type
            $mime = strtolower((string)($payload['mime'] ?? ($payload['contentType'] ?? '')));
            if ($mime !== '' && strpos($mime, 'image/') !== 0) {
                $skipped++;
                $reason_counts['not_image']++;
                continue;
            }

            // Build rendition URL
            $rendition_url = self::build_rendition_url_for_asset($catalog_id, $asset_id, $payload);
            if (!$rendition_url) {
                $skipped++;
                $reason_counts['no_rendition_url']++;
                continue;
            }

            // Extract filename
            $filename = self::extract_filename_from_payload($payload, $asset_id);

          $updated = (string)(
    $payload['develop']['timestamp'] ??
    $payload['userUpdated'] ??
    $payload['lastModified'] ??
    $res['updated'] ?? 
    $asset['updated'] ?? 
    $payload['updated'] ?? 
    ''
);

// TEMP DEBUG - log what fields exist in payload
Logger::debug('[LSP fetch_assets] payload keys: ' . implode(', ', array_keys($payload)));
if (isset($payload['develop'])) {
    Logger::debug('[LSP fetch_assets] develop keys: ' . implode(', ', array_keys($payload['develop'])));
}

            // Build asset object
            $assets[] = [
                'id'            => $asset_id,
                'asset_id'      => $asset_id,
                'rendition_url' => $rendition_url,
                'filename'      => $filename,
                'updated'       => $updated,
                'mime'          => $mime ?: 'image/jpeg',
                'payload'       => $payload, // Include full payload for metadata extraction
            ];

            $fetched++;
        }

        $start_index = 0;

        if ($hit_cap) {
            $return_cursor = $next_cursor ?: '';
            $return_next_index = 0;
            if ($return_cursor === $current_cursor || $return_cursor === '') {
                $return_cursor = '';
            }
            break;
        }

        if ($next_cursor !== '' && $next_cursor !== $current_cursor) {
            $current_cursor = $next_cursor;
            $return_cursor = $current_cursor;
            $return_next_index = 0;
            continue;
        }

        if ($next_cursor !== '' && $next_cursor === $current_cursor) {
            $return_cursor = '';
            $return_next_index = 0;
            break;
        }

        $return_cursor = '';
        $return_next_index = 0;
        break;
    }

    $done_all = ($return_cursor === '' && $return_next_index === 0 && !$hit_cap);

    $elapsed = (int)round((microtime(true) - $started) * 1000);

    return [
        'assets'       => $assets,
        'count'        => $fetched,
        'skipped'      => $skipped,
        'elapsed_ms'   => $elapsed,
        'cursor'       => $return_cursor,
        'next_cursor'  => $return_cursor,
        'next_index'   => $return_next_index,
        'done_all'     => $done_all,
        'hit_cap'      => $hit_cap,
        'reasons'      => $reason_counts,
    ];
}

/**
 * Build the best rendition URL for an asset.
 * Prefers 2048, falls back to 1280, then fullsize.
 */
private static function build_rendition_url_for_asset(string $catalog_id, string $asset_id, array $payload): string {
    // Check if renditions are available in payload
    $derived = (array)($payload['derived'] ?? []);
    
    // Preferred sizes in order
    $sizes = ['2048', '1280', 'fullsize'];
    
    foreach ($sizes as $size) {
        if (!empty($derived[$size]['origin'])) {
            // Has a derived rendition - use the endpoint
            return Endpoints::rendition($catalog_id, $asset_id, $size);
        }
    }
    
    // Fallback: try 2048 anyway (Lightroom may generate on demand)
    return Endpoints::rendition($catalog_id, $asset_id, '2048');
}

/**
 * Extract a usable filename from asset payload.
 */
private static function extract_filename_from_payload(array $payload, string $asset_id): string {
    $candidates = [
        $payload['importSource']['fileName'] ?? '',
        $payload['importSource']['originalFilename'] ?? '',
        $payload['fileName'] ?? '',
        $payload['filename'] ?? '',
        $payload['name'] ?? '',
    ];

    foreach ($candidates as $name) {
        $name = is_string($name) ? trim($name) : '';
        if ($name !== '') {
            return $name;
        }
    }

    return 'lr-' . $asset_id . '.jpg';
}




    public static function sync_album_slice(
    string $catalog_id,
    string $album_id,
    string $cursor = '',
    int $start_index = 0,
    int $page_size = 60,
    float $timeBudget = 40.0
): array {

    $batchSz = max(5, min(200, (int)$page_size));
    $limit   = 200;

    $start_index = max(0, (int)$start_index);

    $out = self::batch_import(
        $catalog_id,
        $album_id,
        $cursor,
        $batchSz,
        $limit,
        $start_index,
        $timeBudget
    );

    if (is_wp_error($out)) {
        $msg = $out->get_error_message();
        Logger::debug("[LSP] sync_album_slice fatal {$album_id}: {$msg}");
        throw new \Exception(esc_html($msg));
    }

    // Normalize continuation pointers
    $next_cursor = (string)($out['next_cursor'] ?? $out['cursor'] ?? '');
    $next_index  = (int)($out['next_index'] ?? 0);
    $done_all    = !empty($out['done_all']);

    return [
        // summary counts for the caller (broker/extension/admin UI)
        'imported'      => (array)($out['imported'] ?? []),
        'updated'       => (array)($out['updated'] ?? []),
        'meta_updated'  => (array)($out['meta_updated'] ?? []),
        'skipped'       => (int)($out['skipped'] ?? 0),
        'count'         => (int)($out['count'] ?? 0),
        'processed'     => (int)($out['processed'] ?? 0),
        'elapsed_ms'    => (int)($out['elapsed_ms'] ?? 0),
        'hit_cap'       => !empty($out['hit_cap']),
        'reasons'       => (array)($out['reasons'] ?? []),

        // ✅ continuation signals
        'cursor'        => $next_cursor,
        'next_cursor'   => $next_cursor,
        'next_index'    => $next_index,
        'done_all'      => $done_all,
    ];
}


    public static function sync_album(string $catalog_id, string $album_id): array
    {
        $cursor      = '';
        $pages       = 0;
        $importedN   = 0;
        $updatedN    = 0;
        $skippedN    = 0;
        $metaN       = 0;
        $lastMessage = '';

        $batchSize = method_exists(__CLASS__, 'max_batch') ? max(1, (int) self::max_batch()) : 200;
        $limit     = 100; // Adobe page hint; server may clamp

        while (true) {
            $out = self::batch_import($catalog_id, $album_id, $cursor, $batchSize, $limit);
            if (is_wp_error($out)) {
                $msg = $out->get_error_message();
                Logger::debug("[LSP] sync_album fatal {$album_id}: {$msg}");
                throw new \Exception(esc_html($msg));
            }

            $importedN += isset($out['imported']) ? count((array) $out['imported']) : 0;
            $updatedN  += isset($out['updated'])  ? count((array) $out['updated'])  : 0;
            $skippedN  += isset($out['skipped'])  ? (int) $out['skipped']           : 0;
            $metaN     += isset($out['meta_updated']) ? count((array) $out['meta_updated']) : 0;

            if (!empty($out['message'])) {
                $lastMessage = (string) $out['message'];
            }

            $pages++;
            $hasMore = isset($out['has_more']) ? (bool) $out['has_more'] : !empty($out['cursor']);
            $cursor  = (string) ($out['cursor'] ?? '');

            if (!$hasMore) break;
            if ($pages > 10000) {
                Logger::debug("[LSP] sync_album abort {$album_id}: excessive paging (cursor loop?)");
                break;
            }
        }

        $summary = [
            'album'            => $album_id,
            'pages'            => $pages,
            'imported'         => $importedN,
            'updated'          => $updatedN,
            'skipped'          => $skippedN,
            'meta_updated'     => $metaN,
            'message'          => $lastMessage ?: sprintf(
                'Album %s — imported %d, updated %d, skipped %d.',
                $album_id, $importedN, $updatedN, $skippedN
            ),
        ];

        Logger::debug('[LSP] sync_album summary: ' . wp_json_encode($summary));
        return $summary;
    }

    // --- Logger shim to avoid fatals if Util\Logger lacks info()/error() ---
    private static function log($level, $message, array $context = []) {
        if (class_exists('LightSyncPro\\Util\\Logger') && method_exists('LightSyncPro\\Util\\Logger', $level)) {
            \LightSyncPro\Util\Logger::$level($message, $context);
            return;
        }
        $prefix = '[LSP]['.strtoupper($level).'] ';
        error_log($prefix.$message.( $context ? ' '.json_encode($context) : '' ));
    }

   private static function push_activity(array $row): void {
    $items = (array) \LightSyncPro\Admin\Admin::get_opt('lightsync_activity', []);
    if (!is_array($items)) $items = [];

    $row = array_merge([
        'ts'      => time(),
        'type'    => 'sync',
        'source'  => 'auto',
        'status'  => 'info',
        'msg'     => '',
        'catalog' => '',
        'album'   => '',
        'meta'    => [],
    ], $row);

    // Back-compat: if caller used "message", map to "msg"
    if (empty($row['msg']) && !empty($row['message'])) {
        $row['msg'] = (string) $row['message'];
    }

    array_unshift($items, $row);
    $items = array_slice($items, 0, 60);

    \LightSyncPro\Admin\Admin::set_opt(['lightsync_activity' => $items]);
}


    private static function logInfo($message, array $context = [])  { self::log('info', $message, $context); }
    private static function logError($message, array $context = []) { self::log('error', $message, $context); }
    // --- end shim ---

    private static function next_cursor_from_link_header($linkHeader){
        if (!is_string($linkHeader) || $linkHeader === '') return '';
        $linkHeader = html_entity_decode($linkHeader, ENT_QUOTES);

        foreach (explode(',', $linkHeader) as $part){
            $part = trim($part);
            if (stripos($part, 'rel=') === false) continue;
            if (stripos($part, 'rel="next"') === false && stripos($part, 'rel=next') === false) continue;
            if (preg_match('~<([^>]+)>~', $part, $m)) {
                $href = $m[1];
                if ($href && preg_match('/[?&]cursor=([^&]+)/', $href, $mm)) {
                    return urldecode($mm[1]);
                }
            }
        }
        return '';
    }

    private static function time_exceeded(float $started, float $maxSeconds): bool {
    return (microtime(true) - $started) >= $maxSeconds;
}



   private static function queue_rendition_retry($catalog_id, $album_id, $asset_id) {
    $key = "lightsync_retry_{$catalog_id}_{$album_id}_{$asset_id}";
    if (get_transient($key)) return;

    $attempts_key = $key . '_n';
    $attempts = (int) get_transient($attempts_key);
    $attempts++;

    set_transient($attempts_key, $attempts, DAY_IN_SECONDS);

    $delay = min(3600, 60 * pow(2, min($attempts, 6))); // 1m → 64m cap
    set_transient($key, 1, 15 * MINUTE_IN_SECONDS);

    if (!wp_next_scheduled(self::CRON_HOOK_RETRY, [$catalog_id, $album_id, $asset_id])) {
        wp_schedule_single_event(time() + $delay, self::CRON_HOOK_RETRY, [$catalog_id, $album_id, $asset_id]);
    }
}


    public static function debug_asset($catalog_id, $asset_id) {
        $url = "https://lr.adobe.io/v2/catalogs/{$catalog_id}/assets/{$asset_id}";
        $resp = Client::get($url, 30);
        if (is_wp_error($resp)) {
            error_log('[LSP debug] asset check error: ' . $resp->get_error_message());
            return;
        }
        $body = wp_remote_retrieve_body($resp);
        error_log('[LSP debug] asset check body: ' . substr($body, 0, 400));
    }

    private static function http_get_strict($url, $expect='json', $extra_args=[]){
        $args = wp_parse_args($extra_args, [
            'timeout' => 30,
        ]);

        $headers = \LightSyncPro\OAuth\OAuth::headers();

        if ($expect === 'image') {
            $headers['Accept'] = 'image/*,image/jpeg,image/png;q=0.9,*/*;q=0.1';
        } else {
            if (empty($headers['Accept'])) {
                $headers['Accept'] = 'application/json';
            }
        }

        // Merge/override with any provided headers (preserve Cache-Control/Pragma)
        if (!empty($extra_args['headers']) && is_array($extra_args['headers'])) {
            $headers = array_merge($headers, $extra_args['headers']);
        }

        $args['headers'] = $headers;

        $resp = wp_remote_get($url, $args);
        $code = wp_remote_retrieve_response_code($resp);
        $ct   = wp_remote_retrieve_header($resp, 'content-type') ?: '';
        $body = wp_remote_retrieve_body($resp);

        if ($code === 401 || $code === 403) {
            $t = \LightSyncPro\OAuth\OAuth::ensure_token();
            if (is_wp_error($t)) {
                return $t;
            }

            $headers = \LightSyncPro\OAuth\OAuth::headers();
            if ($expect === 'image') {
                $headers['Accept'] = 'image/*,image/jpeg,image/png;q=0.9,*/*;q=0.1';
            } else {
                if (empty($headers['Accept'])) {
                    $headers['Accept'] = 'application/json';
                }
            }
            if (!empty($extra_args['headers']) && is_array($extra_args['headers'])) {
                $headers = array_merge($headers, $extra_args['headers']);
            }

            $args['headers'] = $headers;
            $resp = wp_remote_get($url, $args);
            $code = wp_remote_retrieve_response_code($resp);
            $ct   = wp_remote_retrieve_header($resp, 'content-type') ?: '';
            $body = wp_remote_retrieve_body($resp);
        }

        if ($code < 200 || $code >= 300){
            error_log(sprintf(
                '[LSP] bad response %d @ %s :: %s',
                $code,
                $url,
                substr(preg_replace('/\s+/', ' ', $body), 0, 200)
            ));
            return new \WP_Error(
                'bad_code',
                'HTTP ' . $code . ' @ ' . $url,
                ['code' => $code, 'body' => $body, 'ct' => $ct, 'url' => $url]
            );
        }

        if ($expect === 'image'){
            if (stripos($ct, 'image/') !== 0){
                $peek = substr(preg_replace('/\s+/', ' ', $body), 0, 160);
                return new \WP_Error('not_image', "CT {$ct} @ {$url} :: {$peek}");
            }

            if (strlen($body) < 64 || !self::is_probably_image($body)) {
                return new \WP_Error(
                    'empty_image',
                    "Empty/invalid image bytes @ {$url} (len=" . strlen($body) . ')'
                );
            }
        }

        return $resp;
    }

    /**
 * Return candidate asset IDs from the SAME album as this attachment.
 * This is used to pick the "Photoshop version" or newer asset to relink to.
 */
public static function ajax_relink_candidates(): void {
    if ( ! current_user_can('upload_files') ) {
        wp_send_json_error(['error' => 'forbidden'], 403);
    }
    
    $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
    if ( ! wp_verify_nonce($nonce, 'lightsync_relink_attachment') ) {
        wp_send_json_error(['error' => 'bad_nonce'], 403);
    }
    
    $att_id = isset($_POST['attachment_id']) ? (int) $_POST['attachment_id'] : 0;
    if ( ! $att_id || get_post_type($att_id) !== 'attachment' ) {
        wp_send_json_error(['error' => 'bad_attachment'], 400);
    }
    
    $catalog_id = (string) get_post_meta($att_id, '_lightsync_catalog_id', true);
    $album_id   = (string) get_post_meta($att_id, '_lightsync_album_id', true);
    if ( $catalog_id === '' || $album_id === '' ) {
        wp_send_json_error(['error' => 'missing_album_scope'], 400);
    }
    
    $current_asset = (string) get_post_meta($att_id, '_lightsync_asset_id', true);
    
    // Get all existing LR asset IDs in the media library for this album
    global $wpdb;
    $existing_map = [];
    $rows = $wpdb->get_results($wpdb->prepare(
        "SELECT pm.meta_value as asset_id, pm.post_id 
         FROM {$wpdb->postmeta} pm
         INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID AND p.post_status = 'inherit'
         INNER JOIN {$wpdb->postmeta} pm2 ON pm.post_id = pm2.post_id
         WHERE pm.meta_key = '_lightsync_asset_id'
         AND pm2.meta_key = '_lightsync_album_id'
         AND pm2.meta_value = %s",
        $album_id
    ), ARRAY_A);
    
    foreach ($rows as $row) {
        $existing_map[$row['asset_id']] = (int)$row['post_id'];
    }
    
    // Pull first ~200 album members
    $data = self::get_album_assets($catalog_id, $album_id, null, 200);
    if ( is_wp_error($data) ) {
        wp_send_json_error(['error' => 'adobe_error', 'detail' => $data->get_error_message()], 500);
    }
    
    $resources = (array) ($data['resources'] ?? []);
    $seen  = [];
    $items = [];
    
    foreach ($resources as $res) {
        $res = (array) $res;
        $aid = self::extract_asset_id_from_resource($res);
        if (!$aid) continue;
        if (isset($seen[$aid])) continue;
        $seen[$aid] = true;
        
        // Extract title and date
        $fields = self::extract_asset_label_fields($res);
        
        // Check if already imported
        $existing_att_id = $existing_map[$aid] ?? 0;
        $is_imported = ($existing_att_id > 0);
        $is_current = ($current_asset && $aid === $current_asset);
        
        $items[] = [
            'asset_id'      => $aid,
            'title'         => $fields['title'],
            'updated'       => $fields['updated_label'],
            'updated_raw'   => $fields['updated_raw'],
            'is_imported'   => $is_imported,
            'is_current'    => $is_current,
            'attachment_id' => $existing_att_id,
        ];
    }
    
    // Sort by updated date descending (newest first)
    usort($items, function($a, $b) {
        return strcmp($b['updated_raw'] ?? '', $a['updated_raw'] ?? '');
    });
    
    wp_send_json_success([
        'attachment_id' => $att_id,
        'catalog_id'    => $catalog_id,
        'album_id'      => $album_id,
        'current_asset' => $current_asset,
        'candidates'    => $items,
    ]);
}
/**
 * Switch the attachment to a NEW Lightroom asset ID,
 * and immediately overwrite the file/metadata in-place.
 */
public static function ajax_switch_source(): void {
    if ( ! current_user_can('upload_files') ) {
        wp_send_json_error(['error' => 'forbidden'], 403);
    }
    $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
    if ( ! wp_verify_nonce($nonce, 'lightsync_relink_attachment') ) {
        wp_send_json_error(['error' => 'bad_nonce'], 403);
    }
    $att_id    = isset($_POST['attachment_id']) ? (int) $_POST['attachment_id'] : 0;
    $new_asset = isset($_POST['new_asset_id']) ? sanitize_text_field(wp_unslash($_POST['new_asset_id'])) : '';
    if ( ! $att_id || get_post_type($att_id) !== 'attachment' ) {
        wp_send_json_error(['error' => 'bad_attachment'], 400);
    }
    if ( $new_asset === '' ) {
        wp_send_json_error(['error' => 'missing_new_asset_id'], 400);
    }
    $catalog_id = (string) get_post_meta($att_id, '_lightsync_catalog_id', true);
    $album_id   = (string) get_post_meta($att_id, '_lightsync_album_id', true);
    $album_name = (string) get_post_meta($att_id, '_lightsync_album_name', true);
    if ( $catalog_id === '' || $album_id === '' ) {
        wp_send_json_error(['error' => 'missing_album_scope'], 400);
    }
    // Optional safety: only allow switching within same album (highly recommended)
    if ( ! self::album_contains_asset_id($catalog_id, $album_id, $new_asset) ) {
        wp_send_json_error(['error' => 'asset_not_in_album'], 400);
    }
    
    // =============================================
    // SAFEGUARD: Unlink any OTHER attachments that have this asset ID
    // This prevents duplicate mappings - file stays, just not synced anymore
    // =============================================
    global $wpdb;
    
    $other_attachments = $wpdb->get_col($wpdb->prepare(
        "SELECT post_id FROM {$wpdb->postmeta} 
         WHERE meta_key = '_lightsync_asset_id' 
         AND meta_value = %s 
         AND post_id != %d",
        $new_asset,
        $att_id
    ));
    
    $unlinked_count = 0;
    foreach ($other_attachments as $other_att_id) {
        $other_att_id = (int) $other_att_id;
        
        // Store unlink history (for reference/debugging)
        $old_asset = get_post_meta($other_att_id, '_lightsync_asset_id', true);
        update_post_meta($other_att_id, '_lightsync_unlinked_asset_id', $old_asset);
        update_post_meta($other_att_id, '_lightsync_unlinked_at', current_time('mysql'));
        update_post_meta($other_att_id, '_lightsync_unlinked_reason', 'switched_to_attachment_' . $att_id);
        
        // Remove the Lightroom link (file stays in Media Library, just not synced)
        delete_post_meta($other_att_id, '_lightsync_asset_id');
        delete_post_meta($other_att_id, '_lightsync_catalog_id');
        delete_post_meta($other_att_id, '_lightsync_album_id');
        delete_post_meta($other_att_id, '_lightsync_checksum');
        
        $unlinked_count++;
    }
    
    // Also clear from internal asset map so the duplicate check passes
    if (method_exists(__CLASS__, 'unmark')) {
        self::unmark($new_asset);
    }
    
    // Do the in-place overwrite (your existing function)
    $result = self::switch_attachment_source_in_place($att_id, $catalog_id, $album_id, $album_name, $new_asset);
    if ( is_wp_error($result) ) {
        wp_send_json_error(['error' => 'switch_failed', 'detail' => $result->get_error_message()], 500);
    }
    
    // ✅ Generate AVIF primary + WebP backup (best-effort)
    $avif_primary = null;
    try {
        // Your existing compression logic here
        $avif_primary = null;
    } catch (\Throwable $e) {
        $avif_primary = null;
    }
    
    // Return enriched response for your JS
    if (!is_array($result)) $result = ['ok' => true];
    $result['new_asset_id']  = $new_asset;
    $result['avif_primary']  = $avif_primary;
    $result['unlinked_count'] = $unlinked_count; // How many other attachments were unlinked
    
    wp_send_json_success($result);
}

/**
 * Extract a best-effort title + updated date from an Adobe album "resource".
 * Handles multiple payload shapes seen across Lightroom responses.
 *
 * Returns:
 *  - title
 *  - updated_raw (string)
 *  - updated_label (string, display-ready)
 */
private static function extract_asset_label_fields(array $res): array {
    // Lightroom responses can be nested weirdly; try a few likely locations.
    $asset   = isset($res['asset']) && is_array($res['asset']) ? $res['asset'] : [];
    $payload = [];
    if (isset($asset['payload']) && is_array($asset['payload'])) {
        $payload = $asset['payload'];
    } elseif (isset($res['payload']) && is_array($res['payload'])) {
        $payload = $res['payload'];
    } elseif (isset($asset['metadata']) && is_array($asset['metadata'])) {
        $payload = $asset['metadata'];
    }
    
    // Title candidates (try many, because "title" is often empty)
    $title = '';
    $title_candidates = [
        $payload['title'] ?? '',
        $payload['name'] ?? '',
        $asset['title'] ?? '',
        $asset['name'] ?? '',
        $payload['fileName'] ?? '',
        $payload['filename'] ?? '',
        ($payload['importSource']['fileName'] ?? ''),
        ($payload['importSource']['originalFilename'] ?? ''),
        ($asset['importSource']['fileName'] ?? ''),
    ];
    foreach ($title_candidates as $t) {
        $t = is_string($t) ? trim($t) : '';
        if ($t !== '') { $title = $t; break; }
    }
    
    // Updated candidates - prioritize asset-level dates over membership dates
    // captureDate is usually unique per photo
    $updated_raw_candidates = [
        // Asset-level timestamps (most reliable for uniqueness)
        $asset['updated'] ?? '',
        $payload['userUpdated'] ?? '',
        $payload['develop']['timestamp'] ?? '',
        
        // Capture date - usually unique per photo
        $payload['captureDate'] ?? '',
        $asset['captureDate'] ?? '',
        $payload['capturedAt'] ?? '',
        
        // Import timestamp - often unique
        $payload['importTimestamp'] ?? '',
        $asset['importTimestamp'] ?? '',
        
        // Fallbacks
        $payload['updated'] ?? '',
        $payload['updateDate'] ?? '',
        $payload['modified'] ?? '',
        $payload['modifiedAt'] ?? '',
        $asset['created'] ?? '',
        
        // Album membership date (least useful - often same for batch adds)
        $res['updated'] ?? '',
        $res['created'] ?? '',
    ];
    
    $updated_raw = '';
    foreach ($updated_raw_candidates as $u) {
        $u = is_string($u) ? trim($u) : '';
        if ($u !== '') { $updated_raw = $u; break; }
    }
    
    return [
        'title'        => $title,
        'updated_raw'  => $updated_raw,
        'updated_label'=> self::format_lr_date_label($updated_raw),
    ];
}




/**
 * Format Lightroom timestamps into a nice admin label.
 * Accepts ISO 8601 strings (e.g. 2025-12-24T18:22:11.123Z) or unix-like strings.
 */
private static function format_lr_date_label(string $ts): string {
    $ts = trim((string)$ts);
    if ($ts === '') return '';

    // If it's numeric, treat as unix seconds
    if (ctype_digit($ts)) {
        $n = (int)$ts;
        if ($n > 0) {
            return wp_date('M j, Y g:ia', $n);
        }
        return '';
    }

    // ISO 8601 parsing
    $time = strtotime($ts);
    if (!$time) return '';

    return wp_date('M j, Y g:ia', $time);
}



/**
 * Verify a given asset_id is in the album (fast: first 200 only).
 * For huge albums you can expand this, but this is a solid guardrail.
 */
private static function album_contains_asset_id(string $catalog_id, string $album_id, string $asset_id): bool {
    $data = self::get_album_assets($catalog_id, $album_id, null, 200);
    if (is_wp_error($data)) return false;

    $resources = (array)($data['resources'] ?? []);
    foreach ($resources as $res) {
        $aid = self::extract_asset_id_from_resource((array)$res);
        if ($aid && $aid === $asset_id) return true;
    }
    return false;
}


/**
 * Core: keep same attachment ID, overwrite file + regenerate metadata,
 * then point mapping/meta to the new asset id.
 */
private static function switch_attachment_source_in_place(
    int $att_id,
    string $catalog_id,
    string $album_id,
    string $album_name,
    string $new_asset_id
) {


    $existing_att = self::maybe_imported($new_asset_id);
if ($existing_att && $existing_att !== $att_id) {
    return new \WP_Error(
        'asset_already_imported',
        sprintf(
            'That Lightroom asset is already synced as attachment #%d. Relinking would create a duplicate.',
            (int) $existing_att
        ),
        ['existing_attachment_id' => (int) $existing_att]
    );
}


    // 1) Get current settings
    $rend   = \LightSyncPro\Admin\Admin::get_opt('rendition', '2048'); // 'auto'|'2048'|'1280'|'fullsize'
    $maxDim = (int) \LightSyncPro\Admin\Admin::get_opt('max_dimension', 2048);
    if ($maxDim <= 0) $maxDim = 2048;

    // Compression prefs
    $enableAvif   = (int) \LightSyncPro\Admin\Admin::get_opt('avif_enable', 1);
    $avifQuality  = (int) \LightSyncPro\Admin\Admin::get_opt('avif_quality', 70);
    $webpQuality  = (int) \LightSyncPro\Admin\Admin::get_opt('webp_quality', 82);

    // 2) Fetch bytes from Lightroom for the new asset
    $bytes = self::get_rendition_bytes($catalog_id, $new_asset_id, $rend);
    if (is_wp_error($bytes)) return $bytes;

    if (!is_string($bytes) || strlen($bytes) < 128 || !self::is_probably_image($bytes)) {
        return new \WP_Error('bad_rendition', 'Rendition bytes invalid/empty for the selected Lightroom asset.');
    }

    // 3) Write to temp and process
    if (!function_exists('wp_tempnam')) require_once ABSPATH . 'wp-admin/includes/file.php';

    $tmp = wp_tempnam('lsp-switch');
    if (!$tmp || @file_put_contents($tmp, $bytes) === false) {
        return new \WP_Error('tmp_write_failed', 'Could not write temporary file for switching.');
    }

    $prepared = \LightSyncPro\Util\Image::prepareForMedia($tmp, $maxDim);
    if (!is_array($prepared) || empty($prepared['path']) || !file_exists($prepared['path']) || filesize($prepared['path']) < 128) {
        wp_delete_file($tmp);
        return new \WP_Error('prepare_failed', 'prepareForMedia failed during switch.');
    }
    if (is_file($tmp) && $tmp !== ($prepared['path'] ?? '')) wp_delete_file($tmp);

    // Helper: infer extension from mime (PHP 7.x compatible)
    $infer_ext = function(string $mime): string {
        $mime = strtolower(trim($mime));
        $map = [
            'image/png'  => '.png',
            'image/webp' => '.webp',
            'image/avif' => '.avif',
            'image/gif'  => '.gif',
        ];
        return isset($map[$mime]) ? $map[$mime] : '.jpg';
    };

    // 4) Move into uploads (base/original) + overwrite this attachment’s file path
    $baseMime = function_exists('wp_get_image_mime') ? (string) wp_get_image_mime($prepared['path']) : '';
    if (!$baseMime) $baseMime = 'image/jpeg';

    $suggestedName = 'lr-' . $new_asset_id . $infer_ext($baseMime);

    $moved = self::move_prepared_to_uploads($prepared['path'], $suggestedName);
if (is_wp_error($moved)) return $moved;

$finalPath = (string)($moved['path'] ?? '');
$finalMime = (string)($moved['mime'] ?? $baseMime ?? 'image/jpeg');

    if (!$finalPath || !file_exists($finalPath)) {
        return new \WP_Error('final_move_failed', 'Could not move the new rendition into uploads.');
    }

    // ----
    // 4b) Create WebP backup FIRST (best-effort)
    // ----
    $webpBackupPath = '';
    try {
        $pi = pathinfo($finalPath);
        $candidateWebp = $pi['dirname'] . '/' . $pi['filename'] . '.webp';

        $editor = wp_get_image_editor($finalPath);
        if (!is_wp_error($editor) && method_exists($editor, 'save')) {
            if (method_exists($editor, 'set_quality')) {
                $editor->set_quality($webpQuality);
            }
            $saved = $editor->save($candidateWebp, 'image/webp');
            if (!is_wp_error($saved) && !empty($saved['path']) && file_exists($saved['path']) && filesize($saved['path']) >= 128) {
                $webpBackupPath = (string) $saved['path'];
                update_post_meta($att_id, '_lightsync_backup_webp', $webpBackupPath);
            } else {
                delete_post_meta($att_id, '_lightsync_backup_webp');
            }
        } else {
            delete_post_meta($att_id, '_lightsync_backup_webp');
        }
    } catch (\Throwable $e) {
        delete_post_meta($att_id, '_lightsync_backup_webp');
        \LightSyncPro\Util\Logger::debug('[LSP] WebP backup exception: ' . $e->getMessage());
    }

    // ----
    // 4c) Then try AVIF and promote it to primary if it succeeds
    // ----
    $avifSucceeded = false;
    if ($enableAvif && method_exists(__CLASS__, 'editor_supports_avif') && self::editor_supports_avif()) {
        try {
            $pi = pathinfo($finalPath);
            $candidateAvif = $pi['dirname'] . '/' . $pi['filename'] . '.avif';

            $editor = wp_get_image_editor($finalPath);
            if (!is_wp_error($editor) && method_exists($editor, 'save')) {
                if (method_exists($editor, 'set_quality')) {
                    $editor->set_quality($avifQuality);
                }
                $saved = $editor->save($candidateAvif, 'image/avif');
                if (!is_wp_error($saved) && !empty($saved['path']) && file_exists($saved['path']) && filesize($saved['path']) >= 128) {
                    $finalPath = (string) $saved['path'];
                    $finalMime = 'image/avif';
                    $avifSucceeded = true;
                }
            }
        } catch (\Throwable $e) {
            \LightSyncPro\Util\Logger::debug('[LSP] AVIF convert exception: ' . $e->getMessage());
        }
    }

    update_post_meta($att_id, '_lightsync_primary_format', $avifSucceeded ? 'avif' : 'orig');
    update_post_meta($att_id, '_lightsync_primary_mime', $finalMime);
    update_post_meta($att_id, '_lightsync_primary_path', $finalPath);

    // 5) Overwrite attachment file reference + regenerate metadata
    if (!function_exists('wp_generate_attachment_metadata')) {
        require_once ABSPATH . 'wp-admin/includes/image.php';
    }

    // Overwrite in place: move processed file back to original location to preserve URL
    $old_path = get_attached_file($att_id);
    if ($old_path && $old_path !== $finalPath && file_exists($finalPath)) {
        $old_ext = strtolower(pathinfo($old_path, PATHINFO_EXTENSION));
        $new_ext = strtolower(pathinfo($finalPath, PATHINFO_EXTENSION));
        $target_path = $old_path;

        if ($old_ext !== $new_ext) {
            $target_path = preg_replace('/\.' . preg_quote($old_ext, '/') . '$/', '.' . $new_ext, $old_path);
        }

        // Delete old thumbnails
        $old_meta = wp_get_attachment_metadata($att_id);
        if (!empty($old_meta['sizes'])) {
            $old_dir = dirname($old_path);
            foreach ($old_meta['sizes'] as $sz) {
                $t = $old_dir . '/' . $sz['file'];
                if (file_exists($t)) @unlink($t);
            }
        }

        if (copy($finalPath, $target_path)) {
            @unlink($finalPath);
            // Also remove WebP backup if it was at the old temp location
            if (!empty($webpBackupPath) && file_exists($webpBackupPath) && dirname($webpBackupPath) !== dirname($target_path)) {
                $newWebpPath = dirname($target_path) . '/' . pathinfo($target_path, PATHINFO_FILENAME) . '.webp';
                if ($webpBackupPath !== $newWebpPath) {
                    @copy($webpBackupPath, $newWebpPath);
                    @unlink($webpBackupPath);
                    update_post_meta($att_id, '_lightsync_backup_webp', $newWebpPath);
                }
            }
            if ($target_path !== $old_path && file_exists($old_path)) {
                @unlink($old_path);
            }
            $finalPath = $target_path;
        }
    }

    update_attached_file($att_id, $finalPath);
    wp_update_post(['ID' => $att_id, 'post_mime_type' => $finalMime]);

    // Update primary path meta to match
    update_post_meta($att_id, '_lightsync_primary_path', $finalPath);

    $meta = wp_generate_attachment_metadata($att_id, $finalPath);
    if (!is_wp_error($meta)) {
        wp_update_attachment_metadata($att_id, $meta);
    }

    $old_asset_id = (string) get_post_meta($att_id, '_lightsync_asset_id', true);


      if ($old_asset_id && $old_asset_id !== $new_asset_id && method_exists(__CLASS__, 'unmark')) {
    self::unmark($old_asset_id);
}

    // 6) Update mapping + LSP meta
    update_post_meta($att_id, '_lightsync_asset_id', $new_asset_id);

    // Preserve existing updated/rev best-effort
    $doc = self::fetch_asset_doc_with_revision($catalog_id, $new_asset_id);
    $rev = (string)($doc['_lightsync_revision'] ?? '');

    // Stamp + set kind
    self::stamp_attachment_lightsync_meta(
        $att_id,
        $catalog_id,
        $album_id,
        $album_name,
        $new_asset_id,
        (string)($doc['updated'] ?? ''),
        $rev,
        'switch'
    );


    // Update internal asset map so future sync updates this attachment
    if (method_exists(__CLASS__, 'mark')) {
        self::mark($new_asset_id, $att_id, (string)($doc['updated'] ?? ''));
    }

    // Ensure album term is attached
    if (method_exists(__CLASS__, 'assign_album_term')) {
        self::assign_album_term($att_id, $catalog_id, $album_id, $album_name ?: $album_id);
    }

    return [
        'attachment_id' => $att_id,
        'new_asset_id'  => $new_asset_id,
        'kind'          => 'SWITCH',
        'label'         => self::lightsync_format_time((string)get_post_meta($att_id, '_lightsync_last_synced_at', true)),
    ];
}


    /** Find an existing attachment that already represents this Lightroom asset_id (legacy helper) */
    private static function find_attachment_by_asset_meta(string $asset_id): int {
     // phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_key, WordPress.DB.SlowDBQuery.slow_db_query_meta_value
        $q = new \WP_Query([
            'post_type'      => 'attachment',
            'posts_per_page' => 1,
            'post_status'    => 'inherit',
            'meta_key'       => '_lightsync_asset_id',
            'meta_value'     => $asset_id,
            'fields'         => 'ids',
        ]);
        // phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_key, WordPress.DB.SlowDBQuery.slow_db_query_meta_value

        if (!empty($q->posts[0])) {
            return (int) $q->posts[0];
        }
        return 0;
    }

    protected static function request_partner_renditions($catalog_id, $asset_id){
        $url  = Endpoints::generate_renditions($catalog_id, $asset_id);
        $body = [
            'sizes' => ['2048', '1280', 'fullsize'],
        ];

        $resp = Client::post($url, $body, 15);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            $code = is_wp_error($resp)
                ? $resp->get_error_message()
                : wp_remote_retrieve_response_code($resp);
            Logger::debug('[LSP] requested generateRenditions for ' . $asset_id . ' => ' . $code);
        }
    }

    protected static function ensure_renditions_for_asset( $catalog_id, $asset_id, $sizes = [ '2048', '1280' ] ) {
        $url = \LightSyncPro\Http\Endpoints::generate_renditions( $catalog_id, $asset_id );

        $payload = [
            'types' => array_values( array_unique( $sizes ) ),
        ];

        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log(
                '[LSP] generateRenditions request for asset ' . $asset_id .
                ' sizes=' . implode( ',', $payload['types'] )
            );
        }

        $resp = \LightSyncPro\Http\Client::post( $url, $payload, 30 );

        if ( is_wp_error( $resp ) ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log(
                    '[LSP] generateRenditions FAILED for asset ' . $asset_id . ': ' .
                    $resp->get_error_message()
                );
            }
            return $resp;
        }

        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            $code = wp_remote_retrieve_response_code( $resp );
            error_log(
                '[LSP] generateRenditions accepted for asset ' . $asset_id .
                ' (HTTP ' . $code . ')'
            );
        }

        return $resp;
    }

    /** Simple magic-byte sniff */
    private static function is_probably_image(string $bytes): bool {
        $sig8 = substr($bytes, 0, 8);
        return (
            strncmp($sig8, "\xFF\xD8\xFF", 3) === 0 ||               // JPEG
            $sig8 === "\x89PNG\r\n\x1A\n"        ||                  // PNG
            (substr($bytes, 0, 4) === "RIFF" && substr($bytes, 8, 4) === "WEBP") // WEBP
            // If Lightroom ever serves AVIF directly, extend here
            // || (strncmp(substr($bytes, 4, 4), "ftyp", 4) === 0 && strpos(substr($bytes, 8, 8), "avif") !== false)
        );
    }

    private static function pro(){ return false; }
    private static function max_photos(){ return apply_filters('lightsyncpro_max_photos', self::pro() ? 5000 : 100); }
    private static function max_batch(){ return 10000; }

    public static function register_taxonomy(){
        register_taxonomy('lightsync_album', 'attachment', [
            'label' => 'Lightroom Albums',
            'rewrite' => false,
            'public' => false,
            'show_ui' => true,
            'show_admin_column' => true,
            'hierarchical' => false,
            'update_count_callback' => '_update_post_term_count',
        ]);
    }

    public static function get_catalogs() {
        $t = OAuth::ensure_token();
        if (is_wp_error($t)) return $t;

        $url = Endpoints::catalogs();
        error_log('[LSP] get_catalogs URL: ' . $url);

        $resp = Client::get($url, 20);
        if (is_wp_error($resp)) {
            error_log('[LSP] get_catalogs WP_Error: ' . $resp->get_error_message());
            return $resp;
        }

        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);

        error_log('[LSP] get_catalogs status: ' . $code);
        error_log('[LSP] get_catalogs body: ' . substr($body, 0, 1000));

        $decoded = Adobe::decode($body);

        if (!isset($decoded['catalogs']) && isset($decoded['id']) && (($decoded['type'] ?? '') === 'catalog')) {
            $decoded = ['catalogs' => [$decoded]];
        }

        $count = (is_array($decoded) && isset($decoded['catalogs']) && is_array($decoded['catalogs']))
            ? count($decoded['catalogs'])
            : 0;
        error_log('[LSP] get_catalogs count: ' . $count);

        return $decoded;
    }

    public static function get_rendition_bytes( $catalog_id, $asset_id, $size = '2048' ) {
        $sizes = ($size === 'auto') ? [ '2048','1280','fullsize' ] : [ $size,'1280','fullsize' ];

        static $render_tried = [];
        $asset_key = $catalog_id . ':' . $asset_id;

        // Read the current revision and nudge generateRenditions so edits are baked
        $doc = self::fetch_asset_doc_with_revision($catalog_id, $asset_id);
        $rev = (string)($doc['_lightsync_revision'] ?? '');
        self::ensure_renditions_for_asset($catalog_id, $asset_id, $sizes);

        foreach ($sizes as $try_size) {
            $baseUrl = \LightSyncPro\Http\Endpoints::rendition( $catalog_id, $asset_id, $try_size );

            // Strong cache buster tied to Lightroom revision (falls back to time)
            $cb  = $rev ?: (string)time();
            $url = (strpos($baseUrl,'?') !== false) ? ($baseUrl.'&cb='.$cb) : ($baseUrl.'?cb='.$cb);

            // Request with no-cache semantics to avoid CDN staleness
            $resp = self::http_get_strict($url, 'image', [
                'timeout' => 30,
                'headers' => [
                    'Cache-Control' => 'no-cache, max-age=0',
                    'Pragma'        => 'no-cache',
                ],
            ]);

            if (is_wp_error($resp)) {
                $data = $resp->get_error_data();
                $code = (int) ( is_array($data) ? ($data['code'] ?? 0) : 0 );
                $body = is_array($data) && is_string($data['body'] ?? null) ? $data['body'] : '';
                $slug = $resp->get_error_code();

                $is_missing_rendition = (
                    ($code === 404) && (
                        ($body && strpos($body, 'ResourceNotFoundError') !== false) ||
                        $slug === 'lightsync_asset_missing'
                    )
                );

                if ($is_missing_rendition && empty($render_tried[$asset_key])) {
                    $render_tried[$asset_key] = true;

                    self::ensure_renditions_for_asset($catalog_id, $asset_id, $sizes);
                    sleep(2);

                    $resp = self::http_get_strict($url, 'image', [
                        'timeout' => 30,
                        'headers' => [
                            'Cache-Control' => 'no-cache, max-age=0',
                            'Pragma'        => 'no-cache',
                        ],
                    ]);

                    if (is_wp_error($resp)) {
                        return new \WP_Error('rendition_not_ready', 'Lightroom rendition is not ready yet after trigger/retry.', [
                            'asset_id'=>$asset_id,'catalog_id'=>$catalog_id,'size'=>$try_size
                        ]);
                    }
                } else {
                    continue; // try next size
                }
            }

            $body = wp_remote_retrieve_body($resp);
            if (!empty($body)) return $body;
        }

        return new \WP_Error('rendition_not_ready', 'Lightroom renditions are not available yet for this asset.', [
            'asset_id'=>$asset_id,'catalog_id'=>$catalog_id,'sizes'=>$sizes
        ]);
    }

  public static function get_albums($catalog_id, $force = false, $ttl = 300){
    $catalog_id = (string)$catalog_id;
    if ($catalog_id === '') {
        return new \WP_Error('missing_catalog', 'Missing catalog id');
    }

    $cache_key = 'lightsync_albums_' . md5($catalog_id);

    if (!$force) {
        $cached = get_transient($cache_key);
        if ($cached !== false && is_array($cached)) {
            return $cached;
        }
    }

    $t = OAuth::ensure_token();
    if (is_wp_error($t)) return $t;

    $url  = Endpoints::albums($catalog_id);
    $resp = Client::get($url, 30);

    if (is_wp_error($resp)) return $resp;

    $body    = wp_remote_retrieve_body($resp);
    $decoded = Adobe::decode($body);

    // Only cache if response looks valid
    if (is_array($decoded) && isset($decoded['resources']) && is_array($decoded['resources'])) {
        $ttl = (int)$ttl;
        if ($ttl <= 0) $ttl = 300;
        set_transient($cache_key, $decoded, $ttl);
    }

    return $decoded;
}



    public static function get_album_assets($catalog_id, $album_id, $cursor = '', $limit = 100) {

    $limit = max(1, min((int)$limit, 200));
    $cursor = is_string($cursor) ? trim($cursor) : '';

    // Base endpoint (full URL)
    $base = Endpoints::album_assets($catalog_id, $album_id);

    /**
     * If $cursor looks like a "next href" (full URL or path with querystring),
     * request it DIRECTLY instead of appending "&cursor=".
     *
     * This is the key fix for the "captured_after repeats forever" loop.
     */
    $url = '';

    // Full URL
    if ($cursor && preg_match('#^https?://#i', $cursor)) {
        $url = $cursor;

    // Relative "albums/{id}/assets?..." style (what you’re seeing)
    } elseif ($cursor && strpos($cursor, 'albums/') === 0 && strpos($cursor, '?') !== false) {
        // Build catalog base: https://lr.adobe.io/v2/catalogs/{catalogId}
        $albumsUrl   = Endpoints::albums($catalog_id); // .../v2/catalogs/{catalogId}/albums
        $catalogBase = preg_replace('#/albums$#', '', rtrim($albumsUrl, '/'));
        $url         = $catalogBase . '/' . ltrim($cursor, '/');

    // Generic relative path "/v2/..." or "v2/..." with query
    } elseif ($cursor && strpos($cursor, '?') !== false) {
        // Safest: if it already looks like an API path, attach to the scheme/host from $base
        $p = wp_parse_url($base);
        $origin = '';
        if (is_array($p) && !empty($p['scheme']) && !empty($p['host'])) {
            $origin = $p['scheme'] . '://' . $p['host'];
            if (!empty($p['port'])) $origin .= ':' . $p['port'];
        }
        $url = $origin ? ($origin . '/' . ltrim($cursor, '/')) : $base;

    } else {
        // Cursor token style (safe)
        $sep = (strpos($base, '?') === false) ? '?' : '&';
        $url = $base . $sep . 'limit=' . $limit;

        if ($cursor !== '') {
            $url .= '&cursor=' . rawurlencode($cursor);
        }
    }

    // Ensure limit exists if direct URL didn’t include it
    if (strpos($url, 'limit=') === false) {
        $url .= (strpos($url, '?') === false ? '?' : '&') . 'limit=' . $limit;
    }

    // Always embed asset if not included (keeps your downstream expectations stable)
    if (strpos($url, 'embed=asset') === false) {
        $url .= (strpos($url, '?') === false ? '?' : '&') . 'embed=asset';
    }

    // Do the request
    $resp = self::http_get_strict($url, 'json', ['timeout' => 20]);
    if (is_wp_error($resp)) return $resp;

    $body = wp_remote_retrieve_body($resp);
    $body = preg_replace('/^\s*while\s*\(\s*1\s*\)\s*\{\s*\}\s*/', '', (string)$body);
    $data = json_decode($body, true);

    if (!is_array($data)) {
        return new \WP_Error('lightsync_bad_json', 'Invalid JSON from Adobe.');
    }
    $data['_lightsync_catalog_id'] = $catalog_id;
    // Attach headers for your link-header fallback
    $headers = wp_remote_retrieve_headers($resp);
    $data['__headers'] = [
        'link' => isset($headers['link']) ? (string)$headers['link'] : '',
    ];

    return $data;
}


    private static function get_map(){ return Admin::get_opt('asset_map', []); }
    private static function set_map($m){ Admin::set_opt(['asset_map'=>$m]); }
    private static function maybe_imported($asset_id){ $map=self::get_map(); return !empty($map[$asset_id]['attachment_id']) ? intval($map[$asset_id]['attachment_id']) : 0; }
    private static function last_updated($asset_id){ $map=self::get_map(); return $map[$asset_id]['updated'] ?? ''; }
    private static function mark($asset_id,$att,$updated){ $m=self::get_map(); $m[$asset_id]=['attachment_id'=>$att,'updated'=>$updated]; self::set_map($m); }

private static function assign_album_term(int $attachment_id, string $catalog_id, string $album_id, string $album_name){
    if (!$attachment_id || !$album_id) return;

    // Catalog-scoped slug so different accounts can’t collide
    $ns   = substr(md5($catalog_id), 0, 8);
    $base = sanitize_title($album_name ?: $album_id);
    $slug = "lr-{$ns}-{$base}";

    $term = term_exists($slug, 'lightsync_album');
    if (!$term){
        $term = wp_insert_term(
            $album_name ?: ('Album '.$album_id),
            'lightsync_album',
            ['slug' => $slug]
        );
    }

    if (is_wp_error($term)) return;

    $term_id = (int) (is_array($term) ? ($term['term_id'] ?? 0) : $term);
    if (!$term_id) return;

    // Stamp ownership to the active catalog (no user IDs stored)
    update_term_meta($term_id, '_lightsync_catalog_id', $catalog_id);

    // Remove any album terms on this attachment that belong to *other* catalogs
    $attached = wp_get_object_terms($attachment_id, 'lightsync_album', ['fields' => 'ids']);
    $removed  = [];

    if (is_array($attached) && $attached) {
        foreach ($attached as $tid) {
            $tid = (int) $tid;
            if ($tid === $term_id) continue;

            $belongs = (string) get_term_meta($tid, '_lightsync_catalog_id', true);
            if ($belongs && $belongs !== $catalog_id) {
                wp_remove_object_terms($attachment_id, $tid, 'lightsync_album');
                $removed[] = $tid;
            }
        }
    }

    // Assign the correct (scoped) album term
    wp_set_object_terms($attachment_id, $term_id, 'lightsync_album', true);

    // Prevent stale cached term relationships/counts in admin screens
    clean_object_term_cache($attachment_id, 'attachment');

    // Force WP to refresh taxonomy counts for attachments (WP can be flaky here)
    $to_recount = array_values(array_unique(array_merge([$term_id], $removed)));
    if ($to_recount) {
        wp_update_term_count_now($to_recount, 'lightsync_album');
    }
}

    /** Move a prepared image file into uploads and return final path+mime. */
   private static function move_prepared_to_uploads(string $preparedPath, string $suggestedName) {
        if ( ! function_exists('wp_handle_sideload') ) {
            require_once ABSPATH.'wp-admin/includes/file.php';
        }

        $ext = strtolower(pathinfo($preparedPath, PATHINFO_EXTENSION));
        if (!in_array($ext, ['webp','jpg','jpeg','png','avif'], true)) {
            $ext = 'jpg';
        }

        $baseName = sanitize_file_name( pathinfo($suggestedName, PATHINFO_FILENAME) ) . '.' . $ext;

        $file_array = [
            'name'     => $baseName,
            'tmp_name' => $preparedPath,
        ];
        $overrides = [
            'test_form' => false,
            'mimes'     => [
                'webp' => 'image/webp',
                'jpg'  => 'image/jpeg',
                'jpeg' => 'image/jpeg',
                'png'  => 'image/png',
                'avif' => 'image/avif',
            ],
        ];
        $moved = wp_handle_sideload($file_array, $overrides);

        if (is_array($moved) && empty($moved['error']) && !empty($moved['file']) && !empty($moved['type'])) {
            if (is_file($preparedPath) && $preparedPath !== $moved['file']) wp_delete_file($preparedPath);
            return ['path' => $moved['file'], 'mime' => $moved['type']];
        }

        $uploads = wp_upload_dir();
        if (!empty($moved['error'])) {
            Logger::debug('[LightSyncPro] wp_handle_sideload error: ' . $moved['error']);
        }

        $destDir = rtrim($uploads['path'], '/');
        if (!is_dir($destDir)) {
            wp_mkdir_p($destDir);
        }

        $unique = wp_unique_filename($destDir, $baseName);
        $dest   = trailingslashit($destDir) . $unique;

        global $wp_filesystem;

if ( ! $wp_filesystem ) {
    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem(); // sets $wp_filesystem
}
if ( ! $wp_filesystem ) {
    return new \WP_Error('fs_unavailable', 'Filesystem API unavailable (WP_Filesystem not initialized).');
}

        if (!@copy($preparedPath, $dest)) {
            if (!$wp_filesystem->move($preparedPath, $dest)) {
                Logger::debug('[LightSyncPro] Failed to move prepared image into uploads: '.$preparedPath.' -> '.$dest);
                $mime = wp_check_filetype($preparedPath)['type'] ?: 'image/jpeg';
                return ['path' => $preparedPath, 'mime' => $mime];
            }
        }

        if (is_file($preparedPath) && $preparedPath !== $dest) wp_delete_file($preparedPath);

        $mime = wp_check_filetype($dest)['type'] ?: 'image/jpeg';
        return ['path' => $dest, 'mime' => $mime];
    }

    /**
     * Lookup an existing attachment by stored Lightroom asset_id.
     */
    private static function find_attachment_by_asset_id( string $asset_id ) : int {
        if ( ! $asset_id ) return 0;
       // phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_key, WordPress.DB.SlowDBQuery.slow_db_query_meta_value
        $q = new \WP_Query([
            'post_type'      => 'attachment',
            'posts_per_page' => 1,
            'post_status'    => 'inherit',
            'meta_key'       => '_lightsync_asset_id',
            'meta_value'     => $asset_id,
            'fields'         => 'ids',
        ]);

        if ( ! empty( $q->posts ) && isset( $q->posts[0] ) ) {
            return (int) $q->posts[0];
        }

        return 0;
    }

    /** Delete temp files safely if they exist and are not the final. */
    private static function cleanup_temps(array $paths, string $finalPath): void {
        foreach ($paths as $p) {
            if (!$p) continue;
            if ($p !== $finalPath && file_exists($p)) {
                wp_delete_file($p);
            }
        }
    }

    /** Map an absolute uploads path to its public URL. */
    private static function url_from_path(string $absPath): string {
        $uploads = wp_upload_dir();
        $basedir = trailingslashit($uploads['basedir']);
        $baseurl = trailingslashit($uploads['baseurl']);

        if (strpos($absPath, $basedir) === 0) {
            $rel = ltrim(str_replace($basedir, '', $absPath), '/\\');
            $rel = str_replace(DIRECTORY_SEPARATOR, '/', $rel);
            return $baseurl . $rel;
        }
        return trailingslashit(home_url()) . basename($absPath);
    }

    /** Build token map for patterns from LR/meta */
    private static function build_tokens(array $lr_meta, string $album_name, string $originalBasename, string $ext, int $sequence = 1) : array {
        $title   = trim((string)($lr_meta['title']   ?? ''));
        $caption = trim((string)($lr_meta['caption'] ?? ''));
        $keywords= (array)($lr_meta['keywords'] ?? []);
        $camera  = trim((string)($lr_meta['camera']  ?? ''));
        $iso     = trim((string)($lr_meta['iso']     ?? ''));

        $dateISO = '';
        if (!empty($lr_meta['capture_date'])) {
            $ts = is_numeric($lr_meta['capture_date']) ? (int)$lr_meta['capture_date'] : strtotime($lr_meta['capture_date']);
            $dateISO = gmdate('Y-m-d', $ts ?: time());
        } else {
            $dateISO = gmdate('Y-m-d');
        }

        return [
            '{album}'             => sanitize_title($album_name),
            '{title}'             => $title,
            '{caption}'           => $caption,
            '{date}'              => $dateISO,
            '{sequence}'          => (string)$sequence,
            '{camera}'            => $camera,
            '{iso}'               => $iso,
            '{keywords}'          => implode(', ', array_map('sanitize_text_field', $keywords)),
            '{original_filename}' => pathinfo($originalBasename, PATHINFO_FILENAME),
            '{ext}'               => ltrim(strtolower($ext), '.'),
        ];
    }

    /**
     * Fetch the full Lightroom asset document and enrich $lr_meta
     * with title, caption, keywords, capture_date, camera, iso, etc.
     */
private static function fetch_full_lr_meta(string $catalog_id, string $asset_id, array $seed = []) : array {
    static $cache = [];

    $ck = $catalog_id . ':' . $asset_id;
    if (isset($cache[$ck])) {
        // merge cached into seed (cached wins)
        return array_merge($seed, $cache[$ck]);
    }

        $meta = array_merge([
            'title'        => '',
            'caption'      => '',
            'keywords'     => [],
            'capture_date' => $seed['capture_date'] ?? '',
            'camera'       => $seed['camera']       ?? '',
            'iso'          => $seed['iso']          ?? '',
        ], $seed);

        $url  = "https://lr.adobe.io/v2/catalogs/{$catalog_id}/assets/{$asset_id}";
        $resp = self::http_get_strict($url, 'json', ['timeout' => 20]);

        if (is_wp_error($resp)) {
            Logger::debug(
                '[LightSyncPro] [LSP fetch_full_lr_meta] HTTP error for asset ' .
                $asset_id . ': ' . $resp->get_error_message()
            );

    return $meta;
        }

        $body = wp_remote_retrieve_body($resp);
        $body = preg_replace('/^\s*while\s*\(\s*1\s*\)\s*\{\s*\}\s*/', '', $body);

        $data = json_decode($body, true);
        if (!is_array($data)) {
            Logger::debug(
                '[LightSyncPro] [LSP fetch_full_lr_meta] JSON decode failed for asset ' .
                $asset_id . ' body peek=' . substr(preg_replace('/\s+/', ' ', (string) $body), 0, 160)
            );
            return $meta;
        }

        $found = [
            'title'        => null,
            'caption'      => null,
            'keywords'     => null,
            'capture_date' => null,
            'camera'       => null,
            'iso'          => null,
        ];

        $walker = function ($node) use (&$walker, &$found) {
            if (!is_array($node)) return;

            foreach ($node as $k => $v) {
                $lk = strtolower((string) $k);

                if (is_array($v)) {
                    if (
                        $found['keywords'] === null &&
                        (
                            $lk === 'keywords' ||
                            $lk === 'tags' ||
                            $lk === 'subject' ||
                            strpos($lk, 'keyword') !== false
                        )
                    ) {
                        $flat = [];
                        foreach ($v as $item) {
                            if (is_string($item)) {
                                $flat[] = $item;
                            } elseif (is_array($item)) {
                                if (isset($item['label']) && is_string($item['label'])) {
                                    $flat[] = $item['label'];
                                } elseif (isset($item['value']) && is_string($item['value'])) {
                                    $flat[] = $item['value'];
                                }
                            }
                        }
                        if ($flat) {
                            $found['keywords'] = $flat;
                        }
                    }

                    $walker($v);
                    continue;
                }

                if (!is_scalar($v)) continue;
                $vs = trim((string) $v);
                if ($vs === '') continue;

                if (
                    $found['title'] === null &&
                    ($lk === 'title' || strpos($lk, 'title') !== false)
                ) {
                    $found['title'] = $vs;
                    continue;
                }

                if (
                    $found['caption'] === null &&
                    (
                        $lk === 'caption' ||
                        $lk === 'description' ||
                        strpos($lk, 'caption') !== false ||
                        strpos($lk, 'description') !== false
                    )
                ) {
                    $found['caption'] = $vs;
                    continue;
                }

                if (
                    $found['capture_date'] === null &&
                    ($lk === 'capturedate' || $lk === 'capture_date')
                ) {
                    $found['capture_date'] = $vs;
                    continue;
                }

                if (
                    $found['camera'] === null &&
                    ($lk === 'camera' || strpos($lk, 'camera') !== false)
                ) {
                    $found['camera'] = $vs;
                    continue;
                }

                if ($found['iso'] === null && $lk === 'iso') {
                    $found['iso'] = $vs;
                    continue;
                }
            }
        };

        $walker($data);

        if ($found['title'] !== null)        $meta['title']        = $found['title'];
        if ($found['caption'] !== null)      $meta['caption']      = $found['caption'];
        if ($found['keywords'] !== null)     $meta['keywords']     = $found['keywords'];
        if ($found['capture_date'] !== null) $meta['capture_date'] = $found['capture_date'];
        if ($found['camera'] !== null)       $meta['camera']       = $found['camera'];
        if ($found['iso'] !== null)          $meta['iso']          = $found['iso'];

        Logger::debug(
            '[LightSyncPro] [LSP fetch_full_lr_meta] enriched meta for asset ' .
            $asset_id . ': ' . json_encode($meta)
        );
$cache[$ck] = $meta;
        return $meta;
    }

    /** Apply tokens; optionally sanitize for filename */
    private static function apply_pattern(string $pattern, array $tokens, bool $forFilename = false) : string {
        $out = $pattern;
        foreach ($tokens as $k=>$v) { $out = str_replace($k, (string)$v, $out); }
        $out = trim($out);
        if ($forFilename) {
            $out = remove_accents($out);
            $out = preg_replace('/[^A-Za-z0-9\\-_.]+/', '-', $out);
            $out = preg_replace('/-+/', '-', $out);
            $out = trim($out, '-._');
            if ($out === '') $out = 'photo';
        }
        return $out;
    }

    /** Physically rename a file in uploads; returns new path or original on failure */
    private static function rename_physical_file(string $absPath, string $newBasenameNoExt, string $ext) : string {
        $dir = dirname($absPath);
        if (!is_dir($dir)) return $absPath;

        $target = wp_unique_filename($dir, $newBasenameNoExt . '.' . ltrim($ext,'.'));
        $dst    = trailingslashit($dir) . $target;

        if ($absPath === $dst) return $absPath;

        global $wp_filesystem;

if ( ! $wp_filesystem ) {
    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem(); // sets $wp_filesystem
}
if ( ! $wp_filesystem ) {
    return new \WP_Error('fs_unavailable', 'Filesystem API unavailable (WP_Filesystem not initialized).');
}
        if ($wp_filesystem->move($absPath, $dst)) return $dst;

        if (@copy($absPath, $dst)) {
            wp_delete_file($absPath);
            return $dst;
        }
        return $absPath;
    }

    /** Set Title/Caption/Tags/ALT on an attachment based on LR/meta + admin opts */
    private static function set_attachment_text_meta(int $att_id, array $lr_meta, string $album_name, array $opts) : void {
        $title = '';
        switch ($opts['title_source'] ?? 'lr_title') {
            case 'filename':
                $p = pathinfo(get_attached_file($att_id));
                $title = $p ? ($p['filename'] ?? '') : '';
                break;
            case 'lr_title':
                $title = (string)($lr_meta['title'] ?? '');
                break;
            default: $title = '';
        }
        if ($title !== '') {
            wp_update_post(['ID'=>$att_id, 'post_title'=> wp_strip_all_tags($title)]);
        }

        if (($opts['caption_source'] ?? 'lr_caption') === 'lr_caption') {
            $cap = (string)($lr_meta['caption'] ?? '');
            if ($cap !== '') {
                wp_update_post(['ID'=>$att_id, 'post_excerpt'=> $cap]);
            }
        }

        if (($opts['tags_source'] ?? 'lr_keywords') === 'lr_keywords') {
            $tags = array_filter(array_map('trim', (array)($lr_meta['keywords'] ?? [])));
            if (!empty($tags)) {
                // If you'd like to preserve manual tags, make this true or a setting
                $preserve_manual = (bool) ($opts['preserve_manual_tags'] ?? false);
                wp_set_post_tags($att_id, $tags, $preserve_manual);
            }
        }

        $pattern = (string)($opts['alt_pattern'] ?? '{title} — {album}');
        $file = get_attached_file($att_id);
        $ext  = pathinfo($file, PATHINFO_EXTENSION);
        $orig = basename($file);
        $tokens = self::build_tokens($lr_meta, $album_name, $orig, $ext, 1);
        $alt = self::apply_pattern($pattern, $tokens, false);
        if ($alt !== '') {
            update_post_meta($att_id, '_wp_attachment_image_alt', wp_strip_all_tags($alt));
        }
    }

    // Extract the real asset id from an album-assets resource
    public static function extract_asset_id_from_resource(array $asset): string {
        if (!empty($asset['asset']['id'])) {
            return $asset['asset']['id'];
        }
        if (!empty($asset['payload']['asset']['id'])) {
            return $asset['payload']['asset']['id'];
        }
        if (!empty($asset['links']['/rels/asset']['href'])) {
            $href = $asset['links']['/rels/asset']['href'];
            if (preg_match('~assets/([^/?]+)~', $href, $m)) {
                return $m[1];
            }
        }
        if (!empty($asset['id'])) {
            return $asset['id'];
        }
        return '';
    }

    public static function import_or_update($catalog_id, $album_id, $album_name, $asset){
        $asset_id = self::extract_asset_id_from_resource($asset);
        if (!$asset_id) {
            Logger::debug('[LSP] asset missing real asset_id, skipping: ' . json_encode($asset));
            return ['skipped' => true, 'reason' => 'no_asset_id'];
        }

        $assetDoc   = self::fetch_asset_doc_with_revision($catalog_id, $asset_id);
        $currentRev = (string)($assetDoc['_lightsync_revision'] ?? '');

        // Prevent concurrent duplicate handling
        if (!self::acquire_asset_lock($asset_id)) {
            return ['skipped'=>true, 'reason'=>'locked'];
        }

        try {
            $updated  = $asset['updated'] ?? ($asset['payload']['imported'] ?? '');
            $filename = ($asset['payload']['derived'] ?? [])['fullsize']['origin'] ?? ('lr-'.$asset_id.'.jpg');

           $existing_id = self::maybe_imported($asset_id);

if ($existing_id) {
    // If the map points to an attachment that is now linked to a DIFFERENT asset,
    // treat it as not imported and clear the stale mapping.
    $mapped_asset = (string) get_post_meta($existing_id, '_lightsync_asset_id', true);

    if ($mapped_asset !== '' && $mapped_asset !== $asset_id) {
        Logger::debug(sprintf(
            '[LSP] asset_map mismatch: asset=%s mapped_att=%d but att_asset=%s (clearing map entry)',
            $asset_id,
            $existing_id,
            $mapped_asset
        ));

        if (method_exists(__CLASS__, 'unmark')) {
            self::unmark($asset_id);
        }

        $existing_id = 0;
        $known_rev = '';
        $known_updated = '';
    }
}

            $known_rev     = $existing_id ? (string) get_post_meta($existing_id, '_lightsync_rev', true) : '';
            $known_updated = self::last_updated($asset_id);

            // If the map says we have one, keep it even if the physical file is gone.
        // If mapped attachment exists but its file is gone, treat as not imported.
// This fixes: delete from Media Library → resync should re-import.
if ($existing_id) {
    $file = get_attached_file($existing_id);
    if (!$file || !file_exists($file)) {
        Logger::debug("[LSP] mapped attachment missing file; forcing re-import asset={$asset_id} att={$existing_id}");

        // Clear mapping so maybe_imported($asset_id) won't keep returning a dead attachment
        // NOTE: implement these helpers OR inline your real mapping storage.
        if (method_exists(__CLASS__, 'unmark')) {
            self::unmark($asset_id); // preferred if you have it
        } else {
            // Fallback: if your mark() uses an option map, clear it here.
            // Example only; replace with your real mapping key.
            $map = (array) get_option('lightsync_asset_map', []);
            if (isset($map[$asset_id]) && (int)$map[$asset_id] === (int)$existing_id) {
                unset($map[$asset_id]);
                update_option('lightsync_asset_map', $map, false);
            }
        }

        $existing_id   = 0;
        $known_rev     = '';
        $known_updated = '';
    }
}

$file_on_disk = $existing_id ? get_attached_file($existing_id) : '';
if ($existing_id && $file_on_disk && file_exists($file_on_disk)
    && $known_updated && $updated && $known_updated === $updated
    && $known_rev !== '' && $currentRev !== '' && $known_rev === $currentRev){

                // FAST PATH: Image binary unchanged (same rev + updated timestamp)
                // Do a quick metadata comparison using data already in memory (NO API call)
                
                $payload      = (array)($asset['payload'] ?? []);
                $assetPayload = (array)($asset['asset']['payload'] ?? []);
                
                // Extract metadata from already-loaded asset (no API call)
                $lr_title    = (string)($payload['name'] ?? $payload['title'] ?? $assetPayload['name'] ?? $assetPayload['title'] ?? '');
                $lr_caption  = (string)($payload['caption'] ?? $assetPayload['caption'] ?? '');
                $lr_keywords = (array)($payload['keywords'] ?? $assetPayload['keywords'] ?? []);
                
                // Get current WordPress metadata (fast local DB reads)
                $wp_title    = (string)get_post_field('post_title', $existing_id);
                $wp_caption  = (string)get_post_field('post_excerpt', $existing_id);
                $wp_keywords = (array)wp_get_post_tags($existing_id, ['fields' => 'names']);
                
                // Normalize for comparison
                $norm = function($arr) {
                    return array_values(array_unique(array_map(function($v) {
                        return trim(mb_strtolower((string)$v));
                    }, array_filter((array)$arr))));
                };
                
                $meta_changed = (
                    (trim($lr_title) !== '' && trim($lr_title) !== trim($wp_title)) ||
                    (trim($lr_caption) !== '' && trim($lr_caption) !== trim($wp_caption)) ||
                    (!empty($lr_keywords) && $norm($lr_keywords) !== $norm($wp_keywords))
                );
                
                if ($meta_changed) {
                    // Metadata changed - update it (still no image download needed)
                    Logger::debug("[LSP] asset {$asset_id} binary unchanged but metadata differs, updating meta only");
                    
                    $o = \LightSyncPro\Admin\Admin::get_opt();
                    $lr_meta = [
                        'title'    => $lr_title,
                        'caption'  => $lr_caption,
                        'keywords' => $lr_keywords,
                    ];
                    
                    self::set_attachment_text_meta(
                        $existing_id,
                        $lr_meta,
                        $album_name,
                        [
                            'alt_pattern'    => (string)($o['alt_pattern'] ?? '{title} — {album}'),
                            'title_source'   => (string)($o['title_source'] ?? 'lr_title'),
                            'caption_source' => (string)($o['caption_source'] ?? 'lr_caption'),
                            'tags_source'    => (string)($o['tags_source'] ?? 'lr_keywords'),
                        ]
                    );
                    
                    self::assign_album_term($existing_id, $catalog_id, $album_id, $album_name);
                    self::mark($asset_id, $existing_id, $updated);

                    // Stamp last_synced_at so album badge clears
                    update_post_meta($existing_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
                    update_post_meta($existing_id, '_lightsync_last_sync_kind', 'meta');

                    return [
                        'updated_meta' => true,
                        'id'           => $existing_id,
                        'reason'       => 'meta_refreshed',
                    ];
                }
                
                // Truly unchanged - fast skip
                Logger::debug("[LSP] asset {$asset_id} unchanged (rev+updated+meta match), fast-skipping");
                self::assign_album_term($existing_id, $catalog_id, $album_id, $album_name);

                // Still stamp last_synced_at so album badge clears after sync
                update_post_meta($existing_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
                update_post_meta($existing_id, '_lightsync_last_sync_kind', 'skip');

                return [
                    'skipped' => true,
                    'reason'  => 'no_change',
                    'id'      => $existing_id,
                ];
            }

            // Fetch rendition bytes from LR based on enum option
            $rend   = Admin::get_opt('rendition', '2048'); // 'auto'|'2048'|'1280'|'fullsize'
            $bytes  = self::get_rendition_bytes($catalog_id, $asset_id, $rend);

            if (is_wp_error($bytes)) {
                $code = $bytes->get_error_code();

                if ($code === 'rendition_not_ready') {

                    // De-dup queue entry per asset
                    $queue = (array) get_option('lightsync_pending_renditions', []);
                    $exists = false;
                    foreach ($queue as $row) {
                        if (!empty($row['asset']) && $row['asset'] === $asset_id) { $exists = true; break; }
                    }
                    if (!$exists) {
                        $queue[] = [
                            'catalog' => $catalog_id,
                            'asset'   => $asset_id,
                            'album'   => $album_id,
                            'ts'      => time(),
                        ];
                        // Cap queue size to avoid unbounded growth
                        if (count($queue) > 2000) {
                            $queue = array_slice($queue, -2000);
                        }
                        update_option('lightsync_pending_renditions', $queue, false);
                    }

                    Logger::debug('[LSP] rendition not ready (queued) '.$asset_id.': '.$bytes->get_error_message());
                    self::queue_rendition_retry($catalog_id, $album_id, $asset_id);

                    return ['skipped'=>true, 'reason'=>'no_rend'];
                }

                Logger::debug('[LSP] rendition hard error '.$asset_id.': '.$bytes->get_error_message());
                return ['skipped'=>true, 'reason'=>'other'];
            }

            if (strlen($bytes) < 128 || !self::is_probably_image($bytes)) {
                Logger::debug('[LSP] rendition too small/invalid '.$asset_id.' len='.strlen($bytes));
                return ['skipped'=>true, 'reason'=>'empty_rendition'];
            }

            // Track original size for storage stats
            $originalBytes = strlen($bytes);

            // Ensure wp_tempnam() exists (not loaded outside admin)
if (!function_exists('wp_tempnam')) {
    require_once ABSPATH . 'wp-admin/includes/file.php';
}


            $tmp = \wp_tempnam($filename);
            if (!$tmp || @file_put_contents($tmp, $bytes) === false) {
                Logger::debug('[LSP] temp write failed for asset '.$asset_id.' -> '.$tmp);
                return ['skipped'=>true, 'reason'=>'tmp_write_failed'];
            }
            if (@filesize($tmp) < 128) {
                wp_delete_file($tmp);
                Logger::debug('[LSP] tmp file too small after write '.$asset_id);
                return ['skipped'=>true, 'reason'=>'tmp_too_small'];
            }

            // Local resize dimension is a separate integer option
            $maxDim   = (int) Admin::get_opt('max_dimension', 2048);
            if ($maxDim <= 0) $maxDim = 2048;

            $prepared = \LightSyncPro\Util\Image::prepareForMedia($tmp, $maxDim);
            if (!is_array($prepared) || empty($prepared['path']) || !file_exists($prepared['path']) || filesize($prepared['path']) < 128) {
                wp_delete_file($tmp);
                Logger::debug('[LSP] prepareForMedia failed or tiny for asset '.$asset_id);
                return ['skipped'=>true, 'reason'=>'prepare_failed'];
            }

            if (is_file($tmp) && $tmp !== ($prepared['path'] ?? '')) wp_delete_file($tmp);

           $moved = self::move_prepared_to_uploads($prepared['path'], $filename);
if (is_wp_error($moved)) {
    Logger::debug('[LSP] move_prepared_to_uploads error: ' . $moved->get_error_message());
    return ['skipped'=>true, 'reason'=>'final_move_failed'];
}

$finalPath = (string)($moved['path'] ?? '');
$finalMime = (string)($moved['mime'] ?? 'image/jpeg');


            if (!file_exists($finalPath)) {
                return ['skipped'=>true, 'reason'=>'final_move_failed'];
            }

            $enableAvif  = (int) Admin::get_opt('avif_enable', 1);
            $avifQuality = (int) Admin::get_opt('avif_quality', 70);
            if ($enableAvif && self::editor_supports_avif()){
                $conv = \LightSyncPro\LightSync_Compress::convert_to_avif($finalPath, $avifQuality);
                if ($conv && !empty($conv['ok']) && file_exists($conv['dst']) && filesize($conv['dst']) >= 128) {
                    $finalPath = $conv['dst'];
                    $finalMime = 'image/avif';
                } elseif (!empty($conv) && empty($conv['ok'])) {
                    Logger::debug('[LSP] AVIF convert failed: '.json_encode($conv));
                }
            }

            // Track optimized size for storage stats
            $optimizedBytes = file_exists($finalPath) ? (int)filesize($finalPath) : 0;

            $o = \LightSyncPro\Admin\Admin::get_opt();
            $name_pattern   = (string)($o['name_pattern']  ?? '{album}-{title}-{date}');
            $alt_pattern    = (string)($o['alt_pattern']   ?? '{title} — {album}');
            $title_source   = (string)($o['title_source']  ?? 'lr_title');
            $caption_source = (string)($o['caption_source']?? 'lr_caption');
            $tags_source    = (string)($o['tags_source']   ?? 'lr_keywords');
            $keep_original  = !empty($o['keep_original_filename']);

            $payload = (array)($asset['payload'] ?? []);
            $lr_meta = [
                'title'        => (string)($payload['title']   ?? ''),
                'caption'      => (string)($payload['caption'] ?? ''),
                'keywords'     => (array)($payload['keywords'] ?? []),
                'capture_date' => (string)($payload['captureDate'] ?? ($payload['capture_date'] ?? '')),
                'camera'       => (string)($payload['exif']['camera'] ?? ($payload['camera'] ?? '')),
                'iso'          => (string)($payload['exif']['iso']    ?? ($payload['iso'] ?? '')),
            ];

            $lr_meta = self::fetch_full_lr_meta($catalog_id, $asset_id, $lr_meta);

            if (!$keep_original && file_exists($finalPath)) {
                $ext      = pathinfo($finalPath, PATHINFO_EXTENSION);
                $origBase = basename($finalPath);
                $tokens   = self::build_tokens($lr_meta, $album_name, $origBase, $ext, 1);
                $basename = self::apply_pattern($name_pattern, $tokens, true);

                $basename = preg_replace('/\\.' . preg_quote($ext,'/') . '$/i', '', $basename);

                $renamed = self::rename_physical_file($finalPath, $basename, $ext);
              if (is_wp_error($renamed)) {
   Logger::debug('[LSP] rename failed: ' . $renamed->get_error_message());
} elseif ($renamed && file_exists($renamed)) {
   $finalPath = $renamed;
}
            }

            if ($existing_id){
                if ( ! function_exists('wp_generate_attachment_metadata') ) {
                    require_once ABSPATH.'wp-admin/includes/image.php';
                }
                $dir = dirname($finalPath);
                if (!is_dir($dir)) wp_mkdir_p($dir);

                if (!file_exists($finalPath) || filesize($finalPath) < 128) {
                    Logger::debug('[LSP] final file invalid before UPDATE '.$asset_id.' path='.$finalPath);
                    return ['skipped'=>true, 'reason'=>'final_too_small'];
                }

                // Overwrite in place: preserve existing URL to keep Shopify/external refs valid
                $old_path = get_attached_file($existing_id);
                if ($old_path && $old_path !== $finalPath) {
                    $old_ext = strtolower(pathinfo($old_path, PATHINFO_EXTENSION));
                    $new_ext = strtolower(pathinfo($finalPath, PATHINFO_EXTENSION));
                    $target_path = $old_path;

                    // If extension changed (e.g. jpg → avif), adjust the target
                    if ($old_ext !== $new_ext) {
                        $target_path = preg_replace('/\.' . preg_quote($old_ext, '/') . '$/', '.' . $new_ext, $old_path);
                    }

                    // Delete old thumbnails BEFORE overwriting
                    $old_meta = wp_get_attachment_metadata($existing_id);
                    if (!empty($old_meta['sizes'])) {
                        $old_dir = dirname($old_path);
                        foreach ($old_meta['sizes'] as $sz) {
                            $t = $old_dir . '/' . $sz['file'];
                            if (file_exists($t)) @unlink($t);
                        }
                    }

                    // Copy processed file to original location
                    if (copy($finalPath, $target_path)) {
                        // Remove the temp file from move_prepared_to_uploads
                        @unlink($finalPath);

                        // If extension changed, remove old file
                        if ($target_path !== $old_path && file_exists($old_path)) {
                            @unlink($old_path);
                        }

                        $finalPath = $target_path;
                        Logger::debug("[LSP] Overwrite-in-place: {$target_path}");
                    }
                    // If copy fails, fall through with the new path (safe fallback)
                }

                update_attached_file($existing_id, $finalPath);
                $meta = wp_generate_attachment_metadata($existing_id, $finalPath);
                if (!is_wp_error($meta)) {
                    wp_update_attachment_metadata($existing_id, $meta);
                }

                $lr_meta = self::fetch_full_lr_meta($catalog_id, $asset_id, $lr_meta);

                self::set_attachment_text_meta(
                    $existing_id,
                    $lr_meta,
                    $album_name,
                    [
                        'alt_pattern'    => $alt_pattern,
                        'title_source'   => $title_source,
                        'caption_source' => $caption_source,
                        'tags_source'    => $tags_source,
                    ]
                );

                wp_update_post(['ID'=>$existing_id, 'post_mime_type'=>$finalMime]);

                update_post_meta($existing_id, '_lightsync_asset_id', $asset_id);

                self::assign_album_term($existing_id, $catalog_id, $album_id, $album_name);
                self::mark($asset_id, $existing_id, $updated);

                Logger::debug("[LSP] UPDATED #{$existing_id} {$finalPath}");
                if ($currentRev !== '') {
                    update_post_meta($existing_id, '_lightsync_rev', $currentRev);
                }

                self::stamp_attachment_lightsync_meta(
    $existing_id,
    $catalog_id,
    $album_id,
    $album_name,
    $asset_id,
    (string)$updated,
    (string)$currentRev,
    'update'
);


                return ['updated'=>true, 'id'=>$existing_id];
            }

            $finalUrl   = self::url_from_path($finalPath);
            $attachment = [
                'guid'           => $finalUrl,
                'post_mime_type' => $finalMime,
                'post_title'     => sanitize_text_field( pathinfo($filename, PATHINFO_FILENAME) ),
                'post_content'   => '',
                'post_status'    => 'inherit',
            ];
            $att_id = wp_insert_attachment($attachment, $finalPath);

            if (!file_exists($finalPath) || filesize($finalPath) < 128) {
                Logger::debug('[LightSyncPro] Final file missing or too small: '.$finalPath.' size='.(@filesize($finalPath) ?: 0));
                return ['skipped'=>true, 'reason'=>'final_too_small'];
            }
            if (is_wp_error($att_id)) {
                self::cleanup_temps([$finalPath], '___never_equals___');
                return ['skipped'=>true, 'reason'=>'attach_insert_fail'];
            }

            update_post_meta($att_id, '_lightsync_asset_id', $asset_id);

            // Store file size stats for analytics
            if (!empty($originalBytes) && $originalBytes > 0) {
                update_post_meta($att_id, '_lightsync_original_bytes', $originalBytes);
                update_post_meta($att_id, '_lightsync_optimized_bytes', $optimizedBytes);
                Admin::bump_storage_stats($originalBytes, $optimizedBytes, 'lightroom');
            }

            if ( ! function_exists('wp_generate_attachment_metadata') ) {
                require_once ABSPATH.'wp-admin/includes/image.php';
            }
            $meta = wp_generate_attachment_metadata($att_id, $finalPath);
            if (!is_wp_error($meta)) {
                wp_update_attachment_metadata($att_id, $meta);
            }

            self::set_attachment_text_meta(
                $att_id,
                $lr_meta,
                $album_name,
                [
                    'alt_pattern'    => $alt_pattern,
                    'title_source'   => $title_source,
                    'caption_source' => $caption_source,
                    'tags_source'    => $tags_source,
                ]
            );

           self::assign_album_term($att_id, $catalog_id, $album_id, $album_name);

            self::mark($asset_id, $att_id, $updated);

            Logger::debug("[LSP] IMPORTED #{$att_id} {$finalPath}");
            if ($currentRev !== '') {
                update_post_meta($att_id, '_lightsync_rev', $currentRev);
            }

            self::stamp_attachment_lightsync_meta(
    $att_id,
    $catalog_id,
    $album_id,
    $album_name,
    $asset_id,
    (string)$updated,
    (string)$currentRev,
    'import'
);

return ['imported'=>true, 'id'=>$att_id];


        } finally {
            self::release_asset_lock($asset_id);
        }
    }

  private static function editor_supports_avif(): bool {
    if (function_exists('wp_image_editor_supports')) {
        return wp_image_editor_supports(['mime_type' => 'image/avif']);
    }
    $mimes = (array) get_allowed_mime_types();
    return in_array('image/avif', $mimes, true) || isset($mimes['avif']);
}


    // (Optional) sweep pending list on each init tick with a soft cap
    public static function sweep_pending_renditions(){
        $q = (array) get_option('lightsync_pending_renditions', []);
        if (!$q) return;
        $now = time();
        $kept = [];
        foreach ($q as $row){
            if (!empty($row['ts']) && ($now - (int)$row['ts']) > 60){
                if (!wp_next_scheduled(self::CRON_HOOK_RETRY, [$row['catalog'], $row['album'], $row['asset']])){
                    wp_schedule_single_event(time()+5, self::CRON_HOOK_RETRY, [$row['catalog'], $row['album'], $row['asset']]);
                }
            } else {
                $kept[] = $row;
            }
        }
        // keep cap in place even after sweep
        if (count($kept) > 2000) {
            $kept = array_slice($kept, -2000);
        }
        update_option('lightsync_pending_renditions', $kept, false);
    }

    // --- Per-asset lock to avoid duplicate work across concurrent ticks ---
    private static function acquire_asset_lock(string $asset_id): bool {
        $k = 'lightsync_lock_' . md5($asset_id);
        if (get_transient($k)) return false;              // someone else is working on it
        set_transient($k, 1, 300);                        // 300s for heavy conversions
        return true;
    }
    private static function release_asset_lock(string $asset_id): void {
        $k = 'lightsync_lock_' . md5($asset_id);
        delete_transient($k);
    }

    private static function membership_is_image_importable(array $res, array &$why = []): bool {
        $why = [];

        $asset   = $res['asset']   ?? [];
        $payload = $asset['payload'] ?? ($res['payload'] ?? []);

        $status = strtolower((string)($payload['status'] ?? ''));
        if ($status === 'deleted' || $status === 'trashed') {
            $why['deleted'] = true;
            return false;
        }

        $mime = strtolower((string)($payload['mime'] ?? ''));
        if ($mime === '') {
            $mime = strtolower((string)($payload['contentType'] ?? ''));
        }
        if ($mime !== '' && strpos($mime, 'image/') !== 0) {
            $why['video_or_nonimage'] = true;
            return false;
        }

        $derived = (array)($payload['derived'] ?? []);
        $hasRenderable = false;
        foreach (['2048','1280','fullsize'] as $sz) {
            if (!empty($derived[$sz]['origin'])) { $hasRenderable = true; break; }
        }
        if (!$hasRenderable) {
            $why['pending_rendition'] = true;
        }

        return true;
    }

    // ---- REVISION UTILITIES (ADD) ----

    // Extract a "revision-ish" token from an asset doc (timestamp, counter, or strong tag)
    private static function extract_asset_revision(array $assetDoc): string {
        $cand = [];

        // Top-level
        if (!empty($assetDoc['revision'])) $cand[] = (string)$assetDoc['revision'];
        if (!empty($assetDoc['updated']))  $cand[] = (string)$assetDoc['updated'];

        // payload/common shapes
        $p = (array)($assetDoc['payload'] ?? []);
        foreach ([
            'revision','updated',
            'edits.revision','edits.updated',
            'develop.revision',
            'derived.2048.updated','derived.1280.updated','derived.fullsize.updated'
        ] as $path) {
            $ref = $p;
            foreach (explode('.', $path) as $seg) {
                if (!is_array($ref) || !array_key_exists($seg, $ref)) { $ref = null; break; }
                $ref = $ref[$seg];
            }
            if (is_scalar($ref) && (string)$ref !== '') $cand[] = (string)$ref;
        }

        $cand = array_values(array_filter(array_map('strval', $cand)));
        if (!$cand) return '';

        // Prefer ISO8601 timestamps
        $is_iso = static function($v){ return (bool)preg_match('/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}/', $v); };
        $iso = array_values(array_filter($cand, $is_iso));
        if ($iso) { sort($iso, SORT_STRING); return end($iso); }

        // Then numerics
        $nums = array_values(array_filter($cand, fn($v)=>preg_match('/^\d+$/', $v)));
        if ($nums) { sort($nums, SORT_NUMERIC); return (string)end($nums); }

        // Fallback: longest string
        usort($cand, fn($a,$b)=>strlen($a)<=>strlen($b));
        return (string)end($cand);
    }

    // Fetch a single asset doc and annotate with '_lightsync_revision'
    private static function fetch_asset_doc_with_revision(string $catalog_id, string $asset_id): array {
        $url  = "https://lr.adobe.io/v2/catalogs/{$catalog_id}/assets/{$asset_id}";
        $resp = self::http_get_strict($url, 'json', ['timeout'=>20]);
        if (is_wp_error($resp)) return [];

        $body = wp_remote_retrieve_body($resp);
        $body = preg_replace('/^\s*while\s*\(\s*1\s*\)\s*\{\s*\}\s*/', '', (string)$body);
        $doc  = json_decode($body, true);
        if (!is_array($doc)) return [];

        $doc['_lightsync_revision'] = self::extract_asset_revision($doc);
        return $doc;
    }
    // ---- /REVISION UTILITIES ----

    // --- add this helper so estimate can read true asset ids & dedupe like import ---
    private static function membership_asset_id(array $res): string {
        return self::extract_asset_id_from_resource($res);
    }

    // --- helper: loose importability check (no embed required) ---
    private static function membership_is_image_importable_loose(array $res, array &$why = []) : bool {
        $why = [];

        $asset    = (array)($res['asset'] ?? []);
        $payload  = (array)($asset['payload'] ?? ($res['payload'] ?? []));

        $status = strtolower((string)($payload['status'] ?? ''));
        if ($status === 'deleted' || $status === 'trashed') {
            $why['deleted'] = true;
            return false;
        }

        $mime = strtolower((string)($payload['mime'] ?? ($payload['contentType'] ?? '')));
        if ($mime !== '' && strpos($mime, 'image/') !== 0) {
            $why['video_or_nonimage'] = true;
            return false;
        }

        $derived = (array)($payload['derived'] ?? []);
        $hasRenderable = false;
        foreach (['2048','1280','fullsize'] as $sz) {
            if (!empty($derived[$sz]['origin'])) { $hasRenderable = true; break; }
        }
        if (!$hasRenderable) {
            $why['maybe_pending'] = true;
        }

        return true;
    }

    // --- helper: robust unique key (asset id preferred; fallback to membership id) ---
    private static function membership_unique_key(array $res) : string {
        $aid = self::extract_asset_id_from_resource($res);
        if ($aid) return 'a:' . $aid;

        if (!empty($res['id'])) return 'm:' . (string)$res['id'];

        return 'h:' . substr(md5(json_encode($res)), 0, 12);
    }

    public static function count_album_members(
        string $catalog_id,
        string $album_id,
        int $per_page = 200,
        float $time_limit_seconds = 25.0,
        int $max_pages = 500
    ) {
        $catalog_id = trim($catalog_id);
        $album_id   = trim($album_id);
        if ($catalog_id === '' || $album_id === '') {
            return new \WP_Error('bad_args', 'Missing catalog or album id.');
        }

        $t = \LightSyncPro\OAuth\OAuth::ensure_token();
        if (is_wp_error($t)) return $t;

        $per_page = max(5, min($per_page, 200));
        $started  = microtime(true);
        $pages    = 0;
        $count    = 0;
        $cursor   = null;

        $seen = [];

        while (true) {

            if ((microtime(true) - $started) > $time_limit_seconds) break;
            if ($pages >= $max_pages) break;

            $data = self::get_album_assets($catalog_id, $album_id, $cursor, $per_page);
            if (is_wp_error($data)) return $data;

            $resources = (array)($data['resources'] ?? []);
            if (!$resources && !$cursor) break;

            foreach ($resources as $res) {
                $aid = self::extract_asset_id_from_resource($res);
                if ($aid === '') continue;
                if (isset($seen[$aid])) continue;

                $mime = '';
                if (!empty($res['asset']['payload']['mime'])) {
                    $mime = strtolower((string)$res['asset']['payload']['mime']);
                } elseif (!empty($res['payload']['mime'])) {
                    $mime = strtolower((string)$res['payload']['mime']);
                }
                if ($mime && strpos($mime, 'image/') !== 0) {
                    continue;
                }

                $state = (string)($res['payload']['state'] ?? '');
                if ($state && in_array(strtolower($state), ['deleted','hidden'], true)) {
                    continue;
                }

                $seen[$aid] = true;
                $count++;
            }

            $pages++;
           $next = self::derive_next_cursor($data, (string)($cursor ?? ''));
            if (!$next) break;
            $cursor = $next;
        }

        return (int)$count;
    }

    /**
     * Try to resolve the canonical/master asset id for a membership row.
     * If the row is a version/virtual copy, Lightroom typically exposes a
     * /rels/master link → we count that master instead of the version.
     */
    private static function canonical_asset_id_from_membership(array $res): string {
        // 1) If there's an explicit /rels/master link, extract the id from it.
        $links = (array)($res['links'] ?? $res['_links'] ?? []);
        $master = $links['/rels/master'] ?? $links['master'] ?? null;
        if ($master) {
            $href = is_array($master) ? ($master['href'] ?? '') : (string)$master;
            if ($href && preg_match('~assets/([^/?#]+)~i', $href, $m)) {
                return $m[1];
            }
        }

        // 2) Fall back to the real asset id on this row.
        return self::extract_asset_id_from_resource($res);
    }

    /**
     * Very strict check for "countable right now":
     * - Must be image/*
     * - Not deleted/trashed/hidden
     * - Must have at least one derived origin (so we can actually import it)
     */
    private static function is_countable_image_now(array $res): bool {
        $asset   = (array)($res['asset'] ?? []);
        $payload = (array)($asset['payload'] ?? ($res['payload'] ?? []));

        // status/state filters
        $status = strtolower((string)($payload['status'] ?? ''));
        $state  = strtolower((string)($payload['state'] ?? ''));
        $visibility = strtolower((string)($payload['visibility'] ?? ''));
        if (in_array($status, ['deleted','trashed'], true)) return false;
        if (in_array($state,  ['deleted','hidden','trashed'], true)) return false;
        if ($visibility === 'hidden') return false;

        // strict image mime
        $mime = strtolower((string)($payload['mime'] ?? ($payload['contentType'] ?? '')));
        if (strpos($mime, 'image/') !== 0) return false;

        // Needs a concrete rendition origin (counts what's actually importable now)
        $derived = (array)($payload['derived'] ?? []);
        foreach (['2048','1280','fullsize'] as $sz) {
            if (!empty($derived[$sz]['origin'])) return true;
        }
        return false;
    }

    /**
     * Estimate images in an album by counting only unique, importable-now masters.
     * This should match what your importer will actually bring in on this pass.
     */
    public static function estimate_album($catalog_id, $album_id){
        $cursor = null;

        // Dedupe by canonical master id to avoid counting versions/virtual copies twice
        $seen_master = [];

        $count_now       = 0;
        $skipped_deleted = 0;
        $skipped_other   = 0;

        for ($i = 0; $i < 2000; $i++) {
            $data = self::get_album_assets($catalog_id, $album_id, $cursor, 200);
            if (is_wp_error($data)) return $data;

            $resources = (array)($data['resources'] ?? []);
            if (!$resources && !$cursor) break;

            foreach ($resources as $res) {
                // Resolve canonical/master id
                $master_id = self::canonical_asset_id_from_membership($res);
                if ($master_id === '') {
                    $skipped_other++;
                    continue;
                }
                if (isset($seen_master[$master_id])) {
                    // Already counted this photo (version/duplicate membership)
                    continue;
                }

                // Strict "importable now" filter
                if (!self::is_countable_image_now($res)) {
                    // Track deletes separately (useful for debug)
                    $asset   = (array)($res['asset'] ?? []);
                    $payload = (array)($asset['payload'] ?? ($res['payload'] ?? []));
                    $status  = strtolower((string)($payload['status'] ?? ''));
                    $state   = strtolower((string)($payload['state'] ?? ''));
                    if (in_array($status, ['deleted','trashed'], true) || in_array($state, ['deleted','hidden','trashed'], true)) {
                        $skipped_deleted++;
                    } else {
                        $skipped_other++;
                    }
                    continue;
                }

                $seen_master[$master_id] = true;
                $count_now++;
            }

           $cursor = self::derive_next_cursor($data, (string)($cursor ?? ''));

            if (!$cursor) break;
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log(sprintf(
                '[LSP estimate strict] album=%s unique_masters=%d importable_now=%d skipped_deleted=%d skipped_other=%d',
                $album_id,
                count($seen_master),
                $count_now,
                $skipped_deleted,
                $skipped_other
            ));
        }

        return (int)$count_now;
    }

    private static function get_albums_cached(string $catalog_id, int $ttl = 300) {
    $catalog_id = (string) $catalog_id;
    if ($catalog_id === '') return new \WP_Error('missing_catalog', 'Missing catalog id');

    $cache_key = 'lightsync_albums_' . md5($catalog_id);
    $albums = get_transient($cache_key);

    if ($albums !== false && is_array($albums)) {
        return $albums;
    }

    $albums = self::get_albums($catalog_id);
    if (!is_wp_error($albums)) {
        set_transient($cache_key, $albums, max(60, $ttl));
    }

    return $albums;
}


   public static function batch_import($catalog_id, $album_id, $cursor, $batchSz, $limit, $start_index = 0, $timeBudget = 40.0, $capRemaining = null) {

    // Memory optimization: clean up any accumulated garbage
    if (function_exists('gc_collect_cycles')) {
        gc_collect_cycles();
    }

    $maxBatch = self::max_batch();

    $batchSz      = (int)($batchSz ?: 0);
    $limit        = (int)($limit   ?: 0);
    $start_index  = max(0, (int)$start_index);

    // Per-page fetch size: interpret batchSz as "page size", not total cap.
    $perPageLimit = $batchSz ?: $limit ?: 200;
    $perPageLimit = max(5, min($perPageLimit, 60));

    // Total items allowed this call: do NOT tie to batchSz.
    $targetTotal = is_null($capRemaining) ? PHP_INT_MAX : (int)$capRemaining;
    if ($targetTotal < 0) $targetTotal = 0;
    $hitCap = false;

    error_log('[LSP batch_import] req batchSz='.$batchSz.' limit='.$limit.' maxBatch='.$maxBatch.' start_index='.$start_index);
    error_log('[LSP batch_import] perPageLimit='.$perPageLimit.' targetTotal='.$targetTotal);

    $started = microtime(true);

// Respect the parameter, but keep it safe for gateways (WP Engine often ~60s)
$timeBudget = (float)$timeBudget;
if ($timeBudget <= 0) $timeBudget = 40.0;
$timeBudget = max(15.0, min($timeBudget, 45.0)); // keep under ~60s end-to-end


    // Cursor for pagination
    $currentCursor = $cursor ? (string)$cursor : null;

    $done        = 0;
    $billableDone  = 0; // imported + updated only (counts toward plan usage/cap)
$processedDone = 0; // everything we touched (imported+updated+meta+skipped)
    $imported    = [];
    $updated     = [];
    $metaUpdated = [];
    $skipped     = 0;

    $reasonCounts = [
        'no_asset_id'        => 0,
        'filter'             => 0,
        'no_rend'            => 0,
        'empty_rendition'    => 0,
        'tmp_write_failed'   => 0,
        'tmp_too_small'      => 0,
        'prepare_failed'     => 0,
        'final_move_failed'  => 0,
        'final_too_small'    => 0,
        'attach_insert_fail' => 0,
        'meta_refreshed'     => 0,
        'duplicate'          => 0,
        'other'              => 0,
        'no_change'          => 0,
        'locked'             => 0,
    ];

    // Resolve album name once
  $album_name = self::get_cached_album_name($catalog_id, $album_id);
$albs = self::get_albums_cached($catalog_id, 300);
if (!is_wp_error($albs)){
    foreach (($albs['resources'] ?? []) as $a){
        if (!empty($a['id']) && (string)$a['id'] === (string)$album_id){
            $album_name = $a['payload']['name'] ?? '';
            break;
        }
    }
}

    $seen_assets = [];

    $fetchPage = function($cursorIn) use ($catalog_id,$album_id,$perPageLimit){
        $data = self::get_album_assets($catalog_id, $album_id, $cursorIn, $perPageLimit);
        if (is_wp_error($data)) return $data;

        return [
            'resources'  => (array)($data['resources'] ?? []),
            'nextCursor' => self::derive_next_cursor($data, (string)$cursorIn),
        ];
    };

    // These are what we return to JS so it can continue
    $returnCursor    = $currentCursor ?: '';
    $returnNextIndex = 0;
    $cursor_history   = [];
$max_same_cursor  = 2;


    while ($billableDone < $targetTotal && $timeBudget > (microtime(true) - $started)) {

        // ✅ CURSOR LOOP GUARD: Track how many times we've seen this cursor
        $cursor_hash = md5($currentCursor . $album_id);
        if (!isset($cursor_history[$cursor_hash])) {
            $cursor_history[$cursor_hash] = 0;
        }
        $cursor_history[$cursor_hash]++;
        
        if ($cursor_history[$cursor_hash] > $max_same_cursor) {
            Logger::debug("[LSP] Cursor loop: same cursor seen {$cursor_history[$cursor_hash]} times, completing sync");
            $returnCursor    = '';
            $returnNextIndex = 0;
            break; // Force exit - we're done
        }

        $page = $fetchPage($currentCursor);
        if (is_wp_error($page)) return $page;

        $resources  = (array)$page['resources'];
        $nextCursor = (string)($page['nextCursor'] ?? '');

        // ✅ EMPTY RESOURCES CHECK: Handle pagination edge cases
        if (empty($resources)) {
            if ($nextCursor !== '' && $nextCursor !== $currentCursor) {
                // Valid next cursor exists, advance to it
                $currentCursor = $nextCursor;
                $start_index   = 0;
                $returnCursor  = $currentCursor;
                continue;
            }
            // No resources and no new cursor = we're done
            $returnCursor    = '';
            $returnNextIndex = 0;
            break;
        }

        // Safety: if start_index is beyond this page, reset
        if ($start_index < 0 || $start_index >= count($resources)) {
            $start_index = 0;
        }

        // Process this page from start_index onward
        for ($i = $start_index; $i < count($resources); $i++) {

            // ✅ CAP CHECK: Stop if we've hit the plan limit
           if ($billableDone >= $targetTotal) {
                $hitCap = true; 
                break; 
            }

            // ✅ TIMEOUT CHECK: Exit cleanly if time budget exceeded
            if ($timeBudget <= (microtime(true) - $started)) {
                error_log("[LSP] batch_import timeout at {$done}/{$targetTotal}");
                // TIME BUDGET HIT mid-page → resume same cursor + index next time
                $returnCursor    = $currentCursor ?: '';
                $returnNextIndex = $i; // resume at this exact index
                break 2; // break out of both loops
            }

            $res = $resources[$i];

            $asset_id = self::extract_asset_id_from_resource($res);
            if (!$asset_id) {
                $skipped++; 
                $reasonCounts['no_asset_id']++;
                continue;
            }
            
            // Dedupe within this batch run
            if (isset($seen_assets[$asset_id])) {
                $skipped++; 
                $reasonCounts['duplicate']++;
                continue;
            }
            $seen_assets[$asset_id] = true;

$elapsedNow = microtime(true) - $started;

$buffer = 3.0;

if ($elapsedNow >= ($timeBudget - $buffer)) {
    error_log("[LSP] batch_import near time budget (elapsed={$elapsedNow}s, budget={$timeBudget}s). Pausing at i={$i}");
    $returnCursor    = $currentCursor ?: '';
    $returnNextIndex = $i; 
    break 2;
}


            $out = self::import_or_update($catalog_id, $album_id, $album_name, $res);

          if (!empty($out['imported'])) {
    $billableDone++;
    $processedDone++;
    $imported[] = $asset_id;
    continue;
}

if (!empty($out['updated'])) {
    $billableDone++;
    $processedDone++;
    $updated[] = $asset_id;
    continue;
}

if (!empty($out['updated_meta'])) {
    $metaUpdated[] = $asset_id;
    $reasonCounts['meta_refreshed']++;
    continue;
}

            // skipped
            $skipped++;
            $processedDone++;
            $reason = (string)($out['reason'] ?? 'other');
            if (!isset($reasonCounts[$reason])) {
                $reasonCounts['other']++;
            } else {
                $reasonCounts[$reason]++;
            }
        }

        // Finished the entire page → reset index and advance cursor
        $start_index = 0;

        // ✅ HIT CAP: Advance cursor even when capping so we can resume
        if ($hitCap) {
            $returnCursor    = $nextCursor ?: '';  
            $returnNextIndex = 0;
            
            // Safety: if cursor didn't advance, we're actually done
            if ($returnCursor === $currentCursor || $returnCursor === '') {
                Logger::debug("[LSP] Hit cap but cursor unchanged, completing sync");
                $returnCursor = '';
            }
            
            break;
        }

        // ✅ CURSOR ADVANCEMENT: Move to next page if available
        if ($nextCursor !== '' && $nextCursor !== $currentCursor) {
            $currentCursor   = $nextCursor;
            $returnCursor    = $currentCursor;
            $returnNextIndex = 0;
            continue;
        }

        // ✅ SAME CURSOR: If next cursor matches current, we're done
        if ($nextCursor !== '' && $nextCursor === $currentCursor) {
            Logger::debug("[LSP] Next cursor matches current cursor, completing sync");
            $returnCursor    = '';
            $returnNextIndex = 0;
            break;
        }

        // No next cursor → finished
        $returnCursor    = '';
        $returnNextIndex = 0;
        break;
    }

    $elapsed = (int) round((microtime(true) - $started) * 1000);
    $allUpdated = array_values(array_unique($updated));

    // ✅ DONE_ALL: Only true when cursor is clear AND no resume index AND didn't hit cap
    $doneAll = ($returnCursor === '' && (int)$returnNextIndex === 0 && !$hitCap);


    return [
        'imported'      => $imported,
        'updated'       => $allUpdated,
        'meta_updated'  => $metaUpdated,
        'skipped'       => $skipped,
        'count'      => $billableDone,
        'processed'  => $processedDone,    
        'elapsed_ms'    => $elapsed,

        // Continuation signals for JS
        'cursor'        => $returnCursor,
        'next_cursor'   => $returnCursor,
        'next_index'    => (int)$returnNextIndex,

        'done_all'      => $doneAll,
        'hit_cap'       => $hitCap,
        'max_photos'    => self::max_photos(),
        'is_pro'        => self::pro() ? 1 : 0,
        'reasons'       => $reasonCounts,
        'renditions_pending' => (int)($reasonCounts['no_rend'] ?? 0),
    ];
}

private static function lightsync_abs_next_href(array $data, string $href): string {
    $href = trim($href);
    if ($href === '') return '';

    // Already absolute
    if (preg_match('~^https?://~i', $href)) return $href;

    // Use Adobe-provided base if present
    $base = rtrim((string)($data['base'] ?? ''), '/');
    if ($base !== '') {
        return $base . '/' . ltrim($href, '/');
    }

    // Fallback: derive origin from the albums endpoint
    $cat = (string)($data['_lightsync_catalog_id'] ?? '');
    if ($cat !== '') {
        $albumsUrl = Endpoints::albums($cat); // https://lr.adobe.io/v2/catalogs/{cat}/albums
        $parts = wp_parse_url($albumsUrl);
        if (is_array($parts) && !empty($parts['scheme']) && !empty($parts['host'])) {
            $origin = $parts['scheme'] . '://' . $parts['host'];
            if (!empty($parts['port'])) $origin .= ':' . $parts['port'];
            return $origin . '/' . ltrim($href, '/');
        }
    }

    return $href;
}



 public static function derive_next_cursor(array $data, string $currentCursor = ''): string {

    // 1) Adobe canonical JSON pagination
    $href = '';

    if (!empty($data['links']['next']['href'])) {
        $href = (string)$data['links']['next']['href'];
    } elseif (!empty($data['links']['next']) && is_array($data['links']['next']) && !empty($data['links']['next']['href'])) {
        $href = (string)$data['links']['next']['href'];
    } elseif (!empty($data['next']['href'])) {
        $href = (string)$data['next']['href'];
    }

    if ($href !== '') {
        $next = self::lightsync_abs_next_href($data, $href);

        // ✅ HARD GUARD: never allow stalled pagination
        if ($currentCursor !== '' && $next === $currentCursor) {
            Logger::debug('[LSP] derive_next_cursor stalled (next == current). Ending pagination.', [
                'current' => $currentCursor,
                'next'    => $next,
            ]);
            return '';
        }
        return $next;
    }

    // 2) Link header fallback (your get_album_assets stores __headers['link'])
    if (!empty($data['__headers']['link'])) {
        $maybe = self::next_cursor_from_link_header($data['__headers']['link']);
        if ($maybe) {
            // If it’s only a token cursor, keep returning it
            // (get_album_assets can accept token cursors)
            if ($currentCursor !== '' && $maybe === $currentCursor) {
                Logger::debug('[LSP] link-header cursor stalled (token == current). Ending pagination.', [
                    'current' => $currentCursor,
                    'next'    => $maybe,
                ]);
                return '';
            }
            return (string)$maybe;
        }
    }

    return '';
}


  /**
 * Cron hook entrypoint: ONE SAFE SLICE PER RUN.
 *
 * Hook signature (3 args):
 *   do_action( self::CRON_HOOK_SYNC, $catalog_id, $album_id, $source );
 */
public static function cron_sync(?string $catalog_id = null, ?string $album_id = null, string $source = 'auto'): void {

$catalog_id = $catalog_id ?: '';
$album_id   = $album_id   ?: '';
  Logger::debug(sprintf(
    '[LSP cron_sync] START catalog=%s album=%s source=%s',
    (string)$catalog_id,
    (string)$album_id,
    (string)$source
));



    // 0) Resolve scope
    $catalog_id = $catalog_id ?: (string) self::get_catalog_id();
    $album_id   = $album_id   ?: (string) self::get_current_album();
    $source     = $source ?: 'auto';
$started_key = "lightsync_sync_started_{$catalog_id}_{$album_id}_{$source}";
if (!get_transient($started_key)) {
    set_transient($started_key, 1, 10 * MINUTE_IN_SECONDS);
    
    // Reset cumulative stats for this album sync
    $stats_key = "lightsync_sync_stats_{$catalog_id}_{$album_id}";
    update_option($stats_key, ['new' => 0, 'updated' => 0, 'skipped' => 0, 'failed' => 0], false);

    self::push_activity([
        'type'    => 'sync',
        'source'  => $source,
        'status'  => 'started',
        'catalog' => (string)$catalog_id,
        'album'   => (string)$album_id,
        'message' => ($source === 'extension') ? 'Extension sync started'
                    : (($source === 'manual') ? 'Manual sync started' : 'Auto-sync started'),
    ]);
}


    if (!$catalog_id || !$album_id) {
        Logger::debug('[LSP cron_sync] missing catalog_id or album_id');
        return;
    }

    // 1) Keep token fresh (cron safe)
    $tok = \LightSyncPro\OAuth\OAuth::maybe_refresh();
    if (is_wp_error($tok)) {
        self::logError('LSP cron_sync token refresh failed', ['err' => $tok->get_error_message()]);
        // retry soon
        self::schedule_sync_tick($catalog_id, $album_id, $source, 60);
        return;
    }

    // 2) Persisted pointers (resume across ticks)
    $cursor_opt_key = "lightsync_sync_cursor_{$catalog_id}_{$album_id}";
    $idx_opt_key    = "lightsync_sync_next_index_{$catalog_id}_{$album_id}";
    $last_run_key   = "lightsync_sync_last_run_{$catalog_id}_{$album_id}";

    $cursor     = get_option($cursor_opt_key, '');
    $cursor     = $cursor !== '' ? (string)$cursor : '';
    $start_index = (int) get_option($idx_opt_key, 0);
    $start_index = max(0, $start_index);

    // 3) Optional: album_name cache (avoid get_albums() every tick)
    $album_name = '';
    $name_key = 'lightsync_album_name_' . md5($catalog_id . '|' . $album_id);
    $album_name = get_transient($name_key);
    if ($album_name === false) {
        $album_name = '';
        $albs = self::get_albums($catalog_id);
        if (!is_wp_error($albs)) {
            foreach (($albs['resources'] ?? []) as $a) {
                if (!empty($a['id']) && (string)$a['id'] === (string)$album_id) {
                    $album_name = (string)($a['payload']['name'] ?? '');
                    break;
                }
            }
        }
        set_transient($name_key, $album_name, 5 * MINUTE_IN_SECONDS);
    }

    // 4) Cursor loop guard (cron-side)
    $loop_key = "lightsync_sync_cursor_loop_{$catalog_id}_{$album_id}";
    $loop_n_key = $loop_key . "_n";
    $last_cursor = (string) get_option($loop_key, '');

    if ($cursor !== '' && $cursor === $last_cursor) {
        $n = (int) get_option($loop_n_key, 0) + 1;
        update_option($loop_n_key, $n, false);

        if ($n >= 3) {
            Logger::debug("[LSP cron_sync] cursor loop detected; bailing complete album={$album_id}");
            delete_option($cursor_opt_key);
            delete_option($idx_opt_key);
            delete_option($loop_key);
            delete_option($loop_n_key);
            update_option($last_run_key, time(), false);
            return;
        }
    } else {
        update_option($loop_key, $cursor, false);
        update_option($loop_n_key, 0, false);
    }

    // 5) Choose safe tick params
    // - page_size: how many assets we ask Adobe for per fetch
    // - timeBudget: keep under gateway limits (45s is safe on WP Engine-ish stacks)
    $page_size  = (int) \LightSyncPro\Admin\Admin::get_opt('batch_size', 60);
    $page_size  = max(20, min(120, $page_size)); // conservative and stable
    $timeBudget = 40.0; // safe tick budget

    // 6) Run ONE slice via your proven engine
    $out = self::batch_import(
        (string)$catalog_id,
        (string)$album_id,
        $cursor,
        $page_size,   // batchSz = perPageLimit here
        200,          // limit is a fallback; batch_import clamps anyway
        $start_index,
        $timeBudget
    );

if (is_array($out)) {
    $usage_used = (int) count($out['imported'] ?? []) + (int) count($out['updated'] ?? []);
    if ($usage_used > 0) {
        // Admin is already imported at the top of this file
        \LightSyncPro\Admin\Admin::usage_consume($usage_used);
    }
    
    // Track cumulative stats across ticks for this album sync
    $stats_key = "lightsync_sync_stats_{$catalog_id}_{$album_id}";
    $stats = get_option($stats_key, ['new' => 0, 'updated' => 0, 'skipped' => 0, 'failed' => 0]);
    $stats['new']     += count($out['imported'] ?? []);
    $stats['updated'] += count($out['updated'] ?? []);
    $stats['skipped'] += (int)($out['skipped'] ?? 0);
    $stats['failed']  += (int)($out['failed'] ?? 0);
    update_option($stats_key, $stats, false);
}


    if (is_wp_error($out)) {
        self::logError('LSP cron_sync batch_import error', ['err' => $out->get_error_message()]);
        update_option($last_run_key, time(), false);
        self::schedule_sync_tick($catalog_id, $album_id, $source, 90);
        return;
    }

    // 7) Persist continuation pointers for next tick
    $next_cursor = (string)($out['next_cursor'] ?? $out['cursor'] ?? '');
    $next_index  = (int)($out['next_index'] ?? 0);
    $done_all    = !empty($out['done_all']);

    update_option($last_run_key, time(), false);

  if ($done_all || $next_cursor === '') {
    // Finished album
    delete_option($cursor_opt_key);
    delete_option($idx_opt_key);
    delete_option($loop_key);
    delete_option($loop_n_key);

    update_option('lightsyncpro_last_sync_complete', current_time('mysql'));

    // Get cumulative stats for this sync
    $stats_key = "lightsync_sync_stats_{$catalog_id}_{$album_id}";
    $stats = get_option($stats_key, ['new' => 0, 'updated' => 0, 'skipped' => 0, 'failed' => 0]);
    delete_option($stats_key);
    
    // Build completion message with stats
    $source_label = ($source === 'extension') ? 'Extension' : (($source === 'manual') ? 'Manual' : 'Auto');
    $stats_str = sprintf('(new: %d, updated: %d, skipped: %d, failed: %d)', 
        $stats['new'], $stats['updated'], $stats['skipped'], $stats['failed']);
    $completion_msg = "{$source_label} sync completed {$stats_str}";

    self::push_activity([
      'type'    => 'sync',
      'source'  => $source,
      'status'  => 'completed',
      'catalog' => (string)$catalog_id,
      'album'   => (string)$album_id,
      'message' => $completion_msg,
      'meta'    => $stats,
    ]);
delete_transient($started_key);
    return;
}


    // Not done: store resume pointers and schedule next tick soon
    update_option($cursor_opt_key, $next_cursor, false);
    update_option($idx_opt_key, $next_index, false);

    // quick continuation
    self::schedule_sync_tick($catalog_id, $album_id, $source, 20);
}

/**
 * Schedule ONE future tick for a specific album.
 * (Your version is good; keep it.)
 */
public static function schedule_sync_tick(string $catalog_id, string $album_id, string $source = 'auto', int $delay = 60): void {
    $args = [$catalog_id, $album_id, $source];
    if (!wp_next_scheduled(self::CRON_HOOK_SYNC, $args)) {
        wp_schedule_single_event(time() + max(5, (int)$delay), self::CRON_HOOK_SYNC, $args);
    }
}


    /* ========================================================================
 * LightSync Pro – Attachment Provenance + Media Library UI
 * (ADD-ONLY: does not alter existing logic)
 * ====================================================================== */

/**
 * Stamp attachment with LightSync provenance + timestamps
 */
private static function stamp_attachment_lightsync_meta(
    int $att_id,
    string $catalog_id,
    string $album_id,
    string $album_name,
    string $asset_id,
    string $source_updated = '',
    string $source_rev = '',
    string $kind = 'import' // import | update | meta
): void {
    if (!$att_id || !$asset_id) return;

    update_post_meta($att_id, '_lightsync_source', 'lightroom');
    update_post_meta($att_id, '_lightsync_asset_id', $asset_id);
    update_post_meta($att_id, '_lightsync_catalog_id', $catalog_id);
    update_post_meta($att_id, '_lightsync_album_id', $album_id);
    update_post_meta($att_id, '_lightsync_album_name', wp_strip_all_tags($album_name));

    // UTC timestamp so it's deterministic
    update_post_meta($att_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));

    if ($source_updated !== '') {
        update_post_meta($att_id, '_lightsync_source_updated', sanitize_text_field($source_updated));
    }

    if ($source_rev !== '') {
        update_post_meta($att_id, '_lightsync_rev', sanitize_text_field($source_rev));
    }

    update_post_meta($att_id, '_lightsync_last_sync_kind', sanitize_text_field($kind));
}

/**
 * Format UTC MySQL datetime into WP-local display
 */
private static function lightsync_format_time(?string $utc_mysql): string {
    if (!$utc_mysql) return '—';
    $ts = strtotime($utc_mysql . ' UTC');
    if (!$ts) return '—';
    return date_i18n('M j, Y g:ia', $ts);
}

private static function unmark(string $asset_id): void {
    $m = self::get_map();
    if (isset($m[$asset_id])) {
        unset($m[$asset_id]);
        self::set_map($m);
    }
}


public static function register_media_library_ui(): void {

    static $registered = false;
    if ($registered) return;
    $registered = true;

    // List view column
    add_filter('manage_upload_columns', function($cols){
        $cols['lightsync_synced'] = 'LightSync Pro';
        return $cols;
    });

    add_filter('manage_upload_sortable_columns', function($cols){
        $cols['lightsync_synced'] = 'lightsync_synced';
        return $cols;
    });

    // Add source filter dropdown
    add_action('restrict_manage_posts', function($post_type){
        if ($post_type !== 'attachment') return;
        
        $current = isset($_GET['lightsync_source']) ? sanitize_text_field($_GET['lightsync_source']) : '';
        ?>
        <select name="lightsync_source" style="float:none;">
            <option value="">All Sources</option>
            <option value="lightroom" <?php selected($current, 'lightroom'); ?>>Lightroom</option>
            <option value="canva" <?php selected($current, 'canva'); ?>>Canva</option>
            <option value="figma" <?php selected($current, 'figma'); ?>>Figma</option>
            <option value="dropbox" <?php selected($current, 'dropbox'); ?>>Dropbox</option>
            <option value="shutterstock" <?php selected($current, 'shutterstock'); ?>>Shutterstock</option>
            <option value="openrouter" <?php selected($current, 'openrouter'); ?>>AI Generate</option>
            <option value="any" <?php selected($current, 'any'); ?>>Any LightSync</option>
        </select>
        <?php
    });

    add_action('pre_get_posts', function($q){
        if (!is_admin() || !$q->is_main_query()) return;
        
        // Handle source filter
        if (!empty($_GET['lightsync_source'])) {
            $source = sanitize_text_field($_GET['lightsync_source']);
            $meta_query = $q->get('meta_query') ?: [];
            
            if ($source === 'lightroom') {
                // Lightroom: has asset_id but no canva, figma, dropbox, or shutterstock meta
                $meta_query[] = [
                    'relation' => 'AND',
                    ['key' => '_lightsync_asset_id', 'compare' => 'EXISTS'],
                    ['key' => '_lightsync_canva_design_id', 'compare' => 'NOT EXISTS'],
                    ['key' => '_lightsync_figma_file_key', 'compare' => 'NOT EXISTS'],
                    ['key' => '_lightsync_dropbox_file_id', 'compare' => 'NOT EXISTS'],
                    ['key' => '_lightsync_shutterstock_id', 'compare' => 'NOT EXISTS'],
                ];
            } elseif ($source === 'canva') {
                $meta_query[] = ['key' => '_lightsync_canva_design_id', 'compare' => 'EXISTS'];
            } elseif ($source === 'figma') {
                $meta_query[] = ['key' => '_lightsync_figma_file_key', 'compare' => 'EXISTS'];
            } elseif ($source === 'dropbox') {
                $meta_query[] = ['key' => '_lightsync_dropbox_file_id', 'compare' => 'EXISTS'];
            } elseif ($source === 'shutterstock') {
                $meta_query[] = ['key' => '_lightsync_shutterstock_id', 'compare' => 'EXISTS'];
            } elseif ($source === 'openrouter') {
                $meta_query[] = ['key' => '_lightsync_source', 'value' => 'openrouter'];
            } elseif ($source === 'any') {
                // Any LightSync: has asset_id OR dropbox_file_id OR shutterstock_id OR openrouter source
                $meta_query[] = [
                    'relation' => 'OR',
                    ['key' => '_lightsync_asset_id', 'compare' => 'EXISTS'],
                    ['key' => '_lightsync_dropbox_file_id', 'compare' => 'EXISTS'],
                    ['key' => '_lightsync_shutterstock_id', 'compare' => 'EXISTS'],
                    ['key' => '_lightsync_source', 'value' => 'openrouter'],
                ];
            }
            
            if (!empty($meta_query)) {
                $q->set('meta_query', $meta_query);
            }
        }
        
        if ($q->get('orderby') === 'lightsync_synced') {
            $q->set('meta_key', '_lightsync_last_synced_at');
            $q->set('orderby', 'meta_value');
            $q->set('meta_type', 'DATETIME');
        }
    });

    add_action('manage_media_custom_column', function($col, $post_id){
        if ($col !== 'lightsync_synced') return;

        $asset = (string) get_post_meta($post_id, '_lightsync_asset_id', true);
        $dropbox_file = (string) get_post_meta($post_id, '_lightsync_dropbox_file_id', true);
        $lsp_source = (string) get_post_meta($post_id, '_lightsync_source', true);
        
        // Check if this is a LightSync image (any source)
        if (!$asset && !$dropbox_file && !$lsp_source) { echo '—'; return; }

        $last  = (string) get_post_meta($post_id, '_lightsync_last_synced_at', true);
        $kind  = (string) get_post_meta($post_id, '_lightsync_last_sync_kind', true);

        $label = self::lightsync_format_time($last);
        $badge = strtoupper($kind ?: 'SYNC');
        $nonce = wp_create_nonce('lightsync_relink_attachment');
        
        // Detect source (Canva, Figma, Dropbox, Shutterstock, AI, or Lightroom)
        $canva_id = get_post_meta($post_id, '_lightsync_canva_design_id', true);
        $figma_file = get_post_meta($post_id, '_lightsync_figma_file_key', true);
        $dropbox_file = get_post_meta($post_id, '_lightsync_dropbox_file_id', true);
        $shutterstock_id = get_post_meta($post_id, '_lightsync_shutterstock_id', true);
        $source = 'lightroom';
        $source_label = 'Lightroom';
        $source_color = '#dc2626'; // Red
        if ($lsp_source === 'openrouter') {
            $source = 'openrouter';
            $source_label = 'AI Generate';
            $source_color = '#f97316'; // Orange (AI)
        } elseif ($lsp_source === 'shutterstock' || $shutterstock_id) {
            $source = 'shutterstock';
            $source_label = 'Shutterstock';
            $source_color = '#ee2d24'; // Shutterstock red
        } elseif ($canva_id) {
            $source = 'canva';
            $source_label = 'Canva';
            $source_color = '#7c3aed'; // Purple
        } elseif ($figma_file) {
            $source = 'figma';
            $source_label = 'Figma';
            $source_color = '#2563eb'; // Blue
        } elseif ($dropbox_file) {
            $source = 'dropbox';
            $source_label = 'Dropbox';
            $source_color = '#0891b2'; // Cyan
        }

        echo '<div style="display:flex;flex-direction:column;gap:6px;">';

        // Source badge with color
        echo '<span style="display:inline-flex;gap:8px;align-items:center;">';
        echo '<span style="font-size:11px;padding:2px 8px;background:'.esc_attr($source_color).';color:#fff;border-radius:999px;font-weight:500;">'.esc_html($source_label).'</span>';
        echo '<span style="font-size:11px;padding:2px 6px;border:1px solid rgba(0,0,0,.15);border-radius:999px;">'.esc_html($badge).'</span>';
        echo '</span>';
        
        echo '<span style="opacity:.8;font-size:12px;">'.esc_html($label).'</span>';

        echo '<span style="display:inline-flex;gap:10px;align-items:center;font-size:12px;">';
        // Only show Relink for Lightroom sources (others don't have album-based versioning)
        if ($source === 'lightroom') {
            echo '<a href="#" class="lsp-relink" data-att="'.esc_attr($post_id).'" data-asset="'.esc_attr($asset).'" data-nonce="'.esc_attr($nonce).'" data-source="'.esc_attr($source).'">Relink</a>';
            echo '<span style="opacity:.6">·</span>';
        }
        echo '<a href="#" class="lsp-unlink" data-att="'.esc_attr($post_id).'" data-nonce="'.esc_attr($nonce).'" style="color:#b32d2e;">Unlink</a>';
        echo '</span>';

        echo '</div>';
    }, 10, 2);

    // Attachment Details sidebar (media modal)
    add_filter('attachment_fields_to_edit', function($fields, $post){
        $asset = (string) get_post_meta($post->ID, '_lightsync_asset_id', true);
        $dropbox_file = (string) get_post_meta($post->ID, '_lightsync_dropbox_file_id', true);
        $shutterstock_id = (string) get_post_meta($post->ID, '_lightsync_shutterstock_id', true);
        $lsp_source = (string) get_post_meta($post->ID, '_lightsync_source', true);
        
        // Check if this is a LightSync image (any source)
        if (!$asset && !$dropbox_file && !$shutterstock_id && !$lsp_source) return $fields;

        $last  = (string) get_post_meta($post->ID, '_lightsync_last_synced_at', true);
        $album = (string) get_post_meta($post->ID, '_lightsync_album_name', true);
        $rev   = (string) get_post_meta($post->ID, '_lightsync_rev', true);
        
        // Detect source (AI, Canva, Figma, Dropbox, Shutterstock, or Lightroom)
        $canva_id = get_post_meta($post->ID, '_lightsync_canva_design_id', true);
        $figma_file = get_post_meta($post->ID, '_lightsync_figma_file_key', true);
        $figma_node = get_post_meta($post->ID, '_lightsync_figma_node_id', true);
        $dropbox_file = get_post_meta($post->ID, '_lightsync_dropbox_file_id', true);
        $dropbox_path = get_post_meta($post->ID, '_lightsync_dropbox_path', true);
        $shutterstock_license = get_post_meta($post->ID, '_lightsync_shutterstock_license_id', true);
        
        $source = 'lightroom';
        $source_label = 'Lightroom';
        if ($lsp_source === 'openrouter') {
            $source = 'openrouter';
            $source_label = 'AI Generate';
        } elseif ($lsp_source === 'shutterstock' || $shutterstock_id) {
            $source = 'shutterstock';
            $source_label = 'Shutterstock';
        } elseif ($canva_id) {
            $source = 'canva';
            $source_label = 'Canva';
        } elseif ($figma_file) {
            $source = 'figma';
            $source_label = 'Figma';
        } elseif ($dropbox_file) {
            $source = 'dropbox';
            $source_label = 'Dropbox';
        }

        $html  = '<div style="line-height:1.5">';
        $html .= '<strong>Source:</strong> ' . esc_html($source_label) . '<br/>';
        $html .= '<strong>Last synced:</strong> ' . esc_html(self::lightsync_format_time($last)) . '<br/>';

        if ($album) $html .= '<strong>Album:</strong> ' . esc_html($album) . '<br/>';
        if ($rev)   $html .= '<strong>Revision:</strong> ' . esc_html($rev) . '<br/>';
        if ($canva_id) $html .= '<strong>Design ID:</strong> <code style="font-size:12px;">' . esc_html($canva_id) . '</code><br/>';
        if ($figma_file) {
            $html .= '<strong>File Key:</strong> <code style="font-size:12px;">' . esc_html($figma_file) . '</code><br/>';
            if ($figma_node) {
                $html .= '<strong>Node ID:</strong> <code style="font-size:12px;">' . esc_html($figma_node) . '</code><br/>';
            }
        }
        if ($dropbox_file) {
            $html .= '<strong>File ID:</strong> <code style="font-size:12px;">' . esc_html($dropbox_file) . '</code><br/>';
            if ($dropbox_path) {
                $html .= '<strong>Path:</strong> <code style="font-size:12px;">' . esc_html($dropbox_path) . '</code><br/>';
            }
        }
        if ($shutterstock_id) {
            $html .= '<strong>Image ID:</strong> <code style="font-size:12px;">' . esc_html($shutterstock_id) . '</code><br/>';
            if ($shutterstock_license) {
                $html .= '<strong>License ID:</strong> <code style="font-size:12px;">' . esc_html($shutterstock_license) . '</code><br/>';
            }
        }
        // AI-specific metadata
        if ($source === 'openrouter') {
            $ai_prompt = get_post_meta($post->ID, '_lightsync_ai_prompt', true);
            $ai_model  = get_post_meta($post->ID, '_lightsync_ai_model', true);
            $ai_ver    = get_post_meta($post->ID, '_lightsync_ai_version', true);
            if ($ai_prompt) $html .= '<strong>Prompt:</strong> ' . esc_html(wp_trim_words($ai_prompt, 15)) . '<br/>';
            if ($ai_model)  $html .= '<strong>Model:</strong> <code style="font-size:12px;">' . esc_html($ai_model) . '</code><br/>';
            if ($ai_ver)    $html .= '<strong>Version:</strong> v' . esc_html($ai_ver) . '<br/>';
        }

        if ($asset) {
            $html .= '<strong>Asset ID:</strong> <code style="font-size:12px;">' . esc_html($asset) . '</code>';
        }
        $html .= '</div>';

        $nonce = wp_create_nonce('lightsync_relink_attachment');
        $html .= '<div style="margin-top:10px;display:flex;gap:8px;flex-wrap:wrap;">';
        // Only show Relink for Lightroom sources (others don't have album-based versioning)
        if ($source === 'lightroom') {
            $html .= '<button type="button" class="button button-small lsp-relink" data-att="'.esc_attr($post->ID).'" data-asset="'.esc_attr($asset).'" data-nonce="'.esc_attr($nonce).'" data-source="'.esc_attr($source).'">Relink</button>';
        }
        $html .= '<button type="button" class="button button-small lsp-unlink" data-att="'.esc_attr($post->ID).'" data-nonce="'.esc_attr($nonce).'" style="border-color:#b32d2e;color:#b32d2e;">Unlink</button>';
        $html .= '</div>';

        $fields['lightsync_info'] = [
            'label' => 'LightSync Pro',
            'input' => 'html',
            'html'  => $html,
        ];

        return $fields;
    }, 10, 2);
}


    public static function ajax_relink_attachment(): void {
    if ( ! current_user_can('upload_files') ) {
        wp_send_json_error(['error' => 'forbidden'], 403);
    }

    $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
    if ( ! wp_verify_nonce($nonce, 'lightsync_relink_attachment') ) {
        wp_send_json_error(['error' => 'bad_nonce'], 403);
    }

    $att_id  = isset($_POST['attachment_id']) ? (int) $_POST['attachment_id'] : 0;
    $asset_id = isset($_POST['asset_id']) ? sanitize_text_field(wp_unslash($_POST['asset_id'])) : '';

    if (!$att_id || get_post_type($att_id) !== 'attachment') {
        wp_send_json_error(['error' => 'bad_attachment'], 400);
    }

    // If asset_id not passed, try to use existing meta
    if ($asset_id === '') {
        $asset_id = (string) get_post_meta($att_id, '_lightsync_asset_id', true);
    }
    if ($asset_id === '') {
        wp_send_json_error(['error' => 'missing_asset_id'], 400);
    }

    // Pull what we can from existing attachment meta (no external calls)
    $catalog_id  = (string) get_post_meta($att_id, '_lightsync_catalog_id', true);
    $album_id    = (string) get_post_meta($att_id, '_lightsync_album_id', true);
    $album_name  = (string) get_post_meta($att_id, '_lightsync_album_name', true);
    $updated     = (string) get_post_meta($att_id, '_lightsync_source_updated', true);
    $rev         = (string) get_post_meta($att_id, '_lightsync_rev', true);

    // Re-stamp core meta so it’s consistent
    update_post_meta($att_id, '_lightsync_asset_id', $asset_id);
    update_post_meta($att_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
    update_post_meta($att_id, '_lightsync_last_sync_kind', 'relink');

    // Rebuild your internal map (this is the key piece)
    // mark($asset_id, $attachment_id, $updated)
    if (method_exists(__CLASS__, 'mark')) {
        self::mark($asset_id, $att_id, $updated);
    }

    // Re-assign album term if we have enough info
    if ($catalog_id && $album_id) {
        if (method_exists(__CLASS__, 'assign_album_term')) {
            self::assign_album_term($att_id, $catalog_id, $album_id, $album_name ?: $album_id);
        }
    }

    wp_send_json_success([
        'attachment_id' => $att_id,
        'asset_id'      => $asset_id,
        'label'         => self::lightsync_format_time((string) get_post_meta($att_id, '_lightsync_last_synced_at', true)),
        'kind'          => 'RELINK',
    ]);
}

public static function ajax_unlink_attachment(): void {
    if ( ! current_user_can('upload_files') ) {
        wp_send_json_error(['error' => 'forbidden'], 403);
    }

    $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
    if ( ! wp_verify_nonce($nonce, 'lightsync_relink_attachment') ) {
        wp_send_json_error(['error' => 'bad_nonce'], 403);
    }

    $att_id  = isset($_POST['attachment_id']) ? (int) $_POST['attachment_id'] : 0;
    if (!$att_id || get_post_type($att_id) !== 'attachment') {
        wp_send_json_error(['error' => 'bad_attachment'], 400);
    }

    $asset_id = (string) get_post_meta($att_id, '_lightsync_asset_id', true);

    // Remove map entry if possible (optional but recommended)
    if ($asset_id && method_exists(__CLASS__, 'unmark')) {
        self::unmark($asset_id);
    }

    // Remove LSP meta (keep it minimal)
    foreach ([
        '_lightsync_asset_id','_lightsync_catalog_id','_lightsync_album_id','_lightsync_album_name',
        '_lightsync_last_synced_at','_lightsync_source_updated','_lightsync_rev','_lightsync_last_sync_kind'
    ] as $k) {
        delete_post_meta($att_id, $k);
    }

    wp_send_json_success(['attachment_id'=>$att_id]);
}


// phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_error_log
}