<?php
namespace LightSyncPro\Shopify;

use LightSyncPro\Admin\Admin;
use LightSyncPro\OAuth\OAuth;


class Shopify {

    private const API_VERSION = '2025-10';

    public static function broker_base(): string {
        return 'https://lightsyncpro.com';
    }

    public static function pickup_install(string $state) {
        $url = self::broker_base() . '/wp-admin/admin-ajax.php?action=lsp_shopify_install_pickup&state=' . rawurlencode($state);
        $resp = wp_remote_get($url, ['timeout' => 20]);
        if (is_wp_error($resp)) return $resp;
        $body = wp_remote_retrieve_body($resp);
        $json = json_decode($body, true);
        if (empty($json['success']) || empty($json['data'])) {
            return new \WP_Error('shopify_pickup_failed', 'Shopify connection pickup failed.');
        }
        return $json['data'];
    }

    /* ==================== SHOPIFY FILES API ==================== */

    /**
     * Upload Lightroom assets directly to Shopify Files library.
     * No product ID required - images go to the general Files section.
     */
/**
 * Clear all Shopify file mappings for a shop (forces re-upload on next sync)
 */
public static function clear_files_map(string $shop): int {
    global $wpdb;
    $tbl = self::files_map_table();
    $shop = self::normalize_shop($shop);
    
    if (!$shop) return 0;
    
    $deleted = $wpdb->delete($tbl, ['shop_domain' => $shop], ['%s']);
    
    return (int)$deleted;
}

  public static function push_assets_to_files(array $assets, string $catalog_id, string $shop, array $ctx = []): array {
    $shop = self::normalize_shop($shop);

    if (!$shop) {
        return ['ok' => false, 'error' => 'Invalid shop domain'];
    }

    $token = self::get_token($shop);
    if (!$token) {
        return ['ok' => false, 'error' => 'Shop not connected (missing access token)'];
    }

    self::ensure_files_map_table();

    $uploaded = 0;
    $updated = 0;
    $skipped = 0;
    $failed = 0;
    $errors = [];

    $album_id = (string)($ctx['album_id'] ?? '');

    // Get user's pattern settings
    $settings = \LightSyncPro\Admin\Admin::get_opt();
    $alt_pattern = (string)($settings['alt_pattern'] ?? '{title} — {album}');
    $name_pattern = (string)($settings['name_pattern'] ?? '{album}-{title}-{date}');
    $keep_original = !empty($settings['keep_original_filename']);

    // Get album name for patterns
    $album_name = \LightSyncPro\Admin\Admin::get_album_name_cached($catalog_id, $album_id);

    foreach ($assets as $asset) {
        if (is_object($asset)) {
            $asset = (array)$asset;
        }

        $asset_id = (string)(
            $asset['id'] ??
            $asset['asset_id'] ??
            $asset['assetId'] ??
            ''
        );

        if ($asset_id === '') {
            $skipped++;
            $errors[] = ['reason' => 'missing_asset_id'];
            continue;
        }

        // Get rendition URL
        $image_url = $asset['rendition_url'] ?? '';
        if (!$image_url) {
            $skipped++;
            $errors[] = ['asset_id' => $asset_id, 'reason' => 'no_rendition_url'];
            continue;
        }

        // Get payload and metadata
        $payload = $asset['payload'] ?? [];

        // Get modify timestamp
        $modify_ts = (string)(
            $asset['updated'] ??
            $payload['develop']['timestamp'] ??
            $payload['importTimestamp'] ??
            $payload['captureDate'] ??
            ''
        );

        // Get raw metadata from Lightroom (check xmp for Adobe fields)
        $xmp = $payload['xmp'] ?? [];
        $import_source = $payload['importSource'] ?? [];
        $dc = $xmp['dc'] ?? [];
        $photoshop = $xmp['photoshop'] ?? [];

        // Build lr_meta array (same structure as WordPress sync uses)
        $lr_meta = [
            'title'    => (string)($dc['title'] ?? $photoshop['Headline'] ?? $payload['title'] ?? ''),
            'caption'  => (string)($dc['description'] ?? $photoshop['Caption-Abstract'] ?? $payload['caption'] ?? ''),
            'keywords' => (array)($dc['subject'] ?? $payload['keywords'] ?? []),
            'camera'   => (string)($xmp['tiff']['Model'] ?? $payload['cameraModel'] ?? ''),
            'iso'      => (string)($xmp['exif']['ISOSpeedRatings'] ?? ''),
            'date'     => (string)($payload['captureDate'] ?? $modify_ts ?? ''),
        ];

        $original_filename = (string)(
            $import_source['fileName'] ??
            $asset['filename'] ?? 
            ''
        );

        $filename_no_ext = pathinfo($original_filename, PATHINFO_FILENAME);
        $file_ext = pathinfo($original_filename, PATHINFO_EXTENSION) ?: 'jpg';

        // Determine title based on user setting (matches WP logic)
        $title_source = (string)($settings['title_source'] ?? 'lr_title');
        switch ($title_source) {
            case 'filename':
                $lr_title = $filename_no_ext;
                break;
            case 'lr_title':
                $lr_title = (string)($lr_meta['title'] ?? '');
                if ($lr_title === '') {
                    $lr_title = $filename_no_ext; // fallback
                }
                break;
            default:
                $lr_title = '';
        }

        // Determine caption based on user setting (matches WP logic)
        $caption_source = (string)($settings['caption_source'] ?? 'lr_caption');
        $lr_caption = '';
        if ($caption_source === 'lr_caption') {
            $lr_caption = (string)($lr_meta['caption'] ?? '');
        }

        // Build tokens (same as WP sync)
        $tokens = [
            'title'             => $lr_title,
            'caption'           => $lr_caption,
            'album'             => $album_name,
            'date'              => $lr_meta['date'] ? date('Y-m-d', strtotime($lr_meta['date'])) : '',
            'camera'            => $lr_meta['camera'],
            'iso'               => $lr_meta['iso'],
            'keywords'          => implode('-', array_slice($lr_meta['keywords'], 0, 3)),
            'sequence'          => str_pad((string)($uploaded + $updated + 1), 3, '0', STR_PAD_LEFT),
            'original_filename' => $filename_no_ext,
            'ext'               => $file_ext,
        ];

        // Include title + caption in checksum so metadata changes trigger update
       // Include settings in checksum so settings changes trigger update
$settings_hash = md5($alt_pattern . '|' . $name_pattern . '|' . $title_source . '|' . $caption_source . '|' . ($keep_original ? '1' : '0'));

// Checksum includes Lightroom data + LightSync Pro settings
$checksum = hash('sha256', $asset_id . '|' . $modify_ts . '|' . $lr_title . '|' . $lr_caption . '|' . $settings_hash);

        // Check if already uploaded
        $existing = self::get_files_map_row($shop, $asset_id);

        $is_update = false;

        if ($existing) {
            if (!empty($existing['checksum']) && hash_equals((string)$existing['checksum'], $checksum)) {
                $skipped++;
                continue;
            }
            $is_update = true;
        }

        // Fetch image bytes from Lightroom
        $bytes = self::fetch_lightroom_image($image_url);
        if (is_wp_error($bytes)) {
            $failed++;
            $errors[] = [
                'asset_id' => $asset_id,
                'reason' => 'fetch_failed',
                'error' => $bytes->get_error_message(),
            ];
            continue;
        }

        if (!is_string($bytes) || strlen($bytes) < 128) {
            $skipped++;
            $errors[] = [
                'asset_id' => $asset_id,
                'reason' => 'bad_bytes',
                'size' => is_string($bytes) ? strlen($bytes) : 0,
            ];
            continue;
        }

        // Get mime type
        $mime_type = $asset['mime'] ?? self::detect_mime_from_bytes($bytes) ?? 'image/jpeg';

      // Compression settings
$plan = \LightSyncPro\Admin\Admin::plan();
$is_paid = in_array($plan, ['pro', 'agency'], true);

$avif_enabled = $is_paid && !empty($settings['avif_enable']) && function_exists('imageavif');
$avif_quality = (int)($settings['avif_quality'] ?? 70);
$webp_quality = (int)($settings['webp_quality'] ?? 82);

// For updates, match the stored mime_type to preserve Shopify file ID
// This ensures fileUpdate succeeds (Shopify requires extension to match)
$target_mime = null;
if ($is_update && !empty($existing['mime_type'])) {
    $target_mime = $existing['mime_type'];
    \LightSyncPro\Util\Logger::debug('[LSP Shopify] Update: targeting stored mime_type=' . $target_mime);
}

$compressed = self::compress_image($bytes, $mime_type, [
    'avif_enabled' => $avif_enabled,
    'avif_quality' => $avif_quality,
    'webp_quality' => $webp_quality,
    'target_mime' => $target_mime, // Force specific format for updates
]);

if ($compressed && !is_wp_error($compressed)) {
    $bytes = $compressed['bytes'];
    $mime_type = $compressed['mime'];
    $file_ext = $compressed['ext'];
} else {
    // Keep original if compression fails
    $file_ext = pathinfo($original_filename, PATHINFO_EXTENSION) ?: 'jpg';
}

      // Build filename based on settings
if ($keep_original && $original_filename !== '') {
    // Keep original name but update extension if compressed
    $filename = pathinfo($original_filename, PATHINFO_FILENAME) . '.' . $file_ext;
} else {
    $filename = self::apply_pattern($name_pattern, $tokens);
    $filename = sanitize_file_name($filename);
    if (!pathinfo($filename, PATHINFO_EXTENSION)) {
        $filename .= '.' . $file_ext;
    }
}

        // Build alt text using pattern (same as WP sync)
        $alt = self::apply_pattern($alt_pattern, $tokens);

        // Fallback alt if pattern produces empty result
        if (trim($alt) === '') {
            $alt = $lr_title ?: $album_name ?: ('Image ' . substr($asset_id, 0, 8));
        }

        // Upload or Update
        if ($is_update && !empty($existing['shopify_file_id'])) {
            // TRUE REPLACE - keeps same Shopify file ID
            $result = self::update_shopify_file(
                $shop,
                $token,
                $existing['shopify_file_id'],
                $bytes,
                $filename,
                $mime_type,
                $alt
            );
        } else {
            // New upload
            $result = self::upload_file_to_shopify($shop, $token, $bytes, $filename, $mime_type, $alt);
        }

        if (!$result['ok']) {
            $failed++;
            $errors[] = [
                'asset_id' => $asset_id,
                'reason' => $is_update ? 'shopify_update_failed' : 'shopify_upload_failed',
                'error' => $result['error'] ?? 'Unknown error',
            ];
            continue;
        }

        // Store/update mapping (file_id stays the same for updates!)
        self::upsert_files_map_row(
            $shop,
            $asset_id,
            $catalog_id,
            $album_id,
            $result['file_id'] ?? $existing['shopify_file_id'] ?? '',
            $result['file_url'] ?? '',
            $checksum,
            $mime_type // Store format for future updates
        );

        if ($is_update) {
            $updated++;
        } else {
            $uploaded++;
        }
    }

    return [
        'ok' => true,
        'uploaded' => $uploaded,
        'updated' => $updated,
        'skipped' => $skipped,
        'failed' => $failed,
        'errors' => array_slice($errors, -5),
    ];
}

/**
 * Compress image to WebP or AVIF
 */
private static function compress_image(string $bytes, string $mime_type, array $opts = []): ?array {
    $avif_enabled = $opts['avif_enabled'] ?? false;
    $avif_quality = $opts['avif_quality'] ?? 70;
    $webp_quality = $opts['webp_quality'] ?? 82;
    $target_mime = $opts['target_mime'] ?? null; // Force specific output format

    // If target_mime matches input, no compression needed
    if ($target_mime && $target_mime === $mime_type) {
        return null;
    }
    
    // Skip if already WebP/AVIF (unless targeting different format)
    if (!$target_mime && in_array($mime_type, ['image/webp', 'image/avif'], true)) {
        return null;
    }

    // Skip non-image types
    if (strpos($mime_type, 'image/') !== 0) {
        return null;
    }

    // Create image resource from bytes
    $image = @imagecreatefromstring($bytes);
    if (!$image) {
        return null;
    }

    // Preserve transparency for PNG
    imagealphablending($image, true);
    imagesavealpha($image, true);

    ob_start();

    // If target_mime is set, output that specific format
    if ($target_mime === 'image/avif' && function_exists('imageavif')) {
        $success = @imageavif($image, null, $avif_quality);
        if ($success) {
            $output = ob_get_clean();
            imagedestroy($image);
            if (strlen($output) > 0) {
                return ['bytes' => $output, 'mime' => 'image/avif', 'ext' => 'avif'];
            }
        }
        ob_end_clean();
        imagedestroy($image);
        return null;
    }
    
    if ($target_mime === 'image/webp' && function_exists('imagewebp')) {
        $success = @imagewebp($image, null, $webp_quality);
        if ($success) {
            $output = ob_get_clean();
            imagedestroy($image);
            if (strlen($output) > 0) {
                return ['bytes' => $output, 'mime' => 'image/webp', 'ext' => 'webp'];
            }
        }
        ob_end_clean();
        imagedestroy($image);
        return null;
    }
    
    if ($target_mime === 'image/jpeg' || $target_mime === 'image/jpg') {
        $success = @imagejpeg($image, null, 90);
        if ($success) {
            $output = ob_get_clean();
            imagedestroy($image);
            if (strlen($output) > 0) {
                return ['bytes' => $output, 'mime' => 'image/jpeg', 'ext' => 'jpg'];
            }
        }
        ob_end_clean();
        imagedestroy($image);
        return null;
    }
    
    // If target_mime is set but not handled above, skip compression
    if ($target_mime) {
        ob_end_clean();
        imagedestroy($image);
        return null;
    }

    // Normal compression logic: Try AVIF first if enabled
    if ($avif_enabled && function_exists('imageavif')) {
        $success = @imageavif($image, null, $avif_quality);
        if ($success) {
            $output = ob_get_clean();
            imagedestroy($image);
            if (strlen($output) > 0) {
                return [
                    'bytes' => $output,
                    'mime'  => 'image/avif',
                    'ext'   => 'avif',
                ];
            }
        }
        ob_clean();
    }

    // Fall back to WebP
    if (function_exists('imagewebp')) {
        $success = @imagewebp($image, null, $webp_quality);
        if ($success) {
            $output = ob_get_clean();
            imagedestroy($image);
            if (strlen($output) > 0) {
                return [
                    'bytes' => $output,
                    'mime'  => 'image/webp',
                    'ext'   => 'webp',
                ];
            }
        }
    }

    ob_end_clean();
    imagedestroy($image);

    return null;
}

/**
 * Apply a pattern with token replacement
 */
private static function apply_pattern(string $pattern, array $tokens): string {
    $result = $pattern;

    foreach ($tokens as $key => $value) {
        $result = str_replace('{' . $key . '}', (string)$value, $result);
    }

    // Clean up empty tokens and extra separators
    $result = preg_replace('/\{[^}]+\}/', '', $result);   // Remove unused tokens
    $result = preg_replace('/[-_]{2,}/', '-', $result);   // Collapse multiple separators
    $result = preg_replace('/\s+/', ' ', $result);        // Collapse multiple spaces
    $result = trim($result, ' -_—');                      // Trim edges

    return $result;
}

/**
 * Update an existing Shopify file by replacing its content (keeps same ID)
 * Uses fileUpdate mutation with originalSource to replace content in-place
 * Falls back to delete + re-upload if in-place update fails (e.g., extension mismatch)
 */
private static function update_shopify_file(
    string $shop,
    string $token,
    string $file_id,
    string $bytes,
    string $filename,
    string $mime_type,
    string $alt = ''
): array {
    \LightSyncPro\Util\Logger::debug('[LSP Shopify Update] Starting in-place update for file_id=' . $file_id);
    
    // Step 1: Stage upload the new file
    $staged = self::create_staged_upload($shop, $token, $filename, $mime_type, strlen($bytes));
    if (!$staged['ok']) {
        \LightSyncPro\Util\Logger::debug('[LSP Shopify Update] Stage upload failed: ' . ($staged['error'] ?? 'unknown'));
        return $staged;
    }
    
    $staged_target = $staged['target'];
    \LightSyncPro\Util\Logger::debug('[LSP Shopify Update] Staged upload created');
    
    // Step 2: Upload bytes to staged URL
    $upload_result = self::upload_to_staged_target($staged_target, $bytes, $mime_type);
    if (!$upload_result['ok']) {
        \LightSyncPro\Util\Logger::debug('[LSP Shopify Update] Upload to staged target failed: ' . ($upload_result['error'] ?? 'unknown'));
        return $upload_result;
    }
    
    \LightSyncPro\Util\Logger::debug('[LSP Shopify Update] Uploaded to staged URL');
    
    // Step 3: Try fileUpdate mutation to replace content (keeps same ID)
    $query = <<<'GRAPHQL'
mutation fileUpdate($files: [FileUpdateInput!]!) {
    fileUpdate(files: $files) {
        files {
            id
            fileStatus
            alt
            ... on MediaImage {
                image {
                    url
                }
            }
            ... on GenericFile {
                url
            }
        }
        userErrors {
            field
            message
            code
        }
    }
}
GRAPHQL;

    $variables = [
        'files' => [[
            'id' => $file_id,
            'originalSource' => $staged_target['resourceUrl'],
            'alt' => $alt ?: null,
        ]],
    ];

    $response = self::graphql_request($shop, $token, $query, $variables);

    // Check if fileUpdate succeeded
    $update_success = false;
    if ($response['ok']) {
        $data = $response['data']['fileUpdate'] ?? [];
        $userErrors = $data['userErrors'] ?? [];
        
        if (empty($userErrors) && !empty($data['files'][0])) {
            $file = $data['files'][0];
            $file_url = $file['image']['url'] ?? $file['url'] ?? '';
            
            \LightSyncPro\Util\Logger::debug('[LSP Shopify Update] SUCCESS - file updated in-place, same id=' . $file_id);
            
            return [
                'ok' => true,
                'file_id' => $file_id, // Same ID preserved!
                'file_url' => $file_url,
                'updated_in_place' => true,
            ];
        }
        
        // Log the error for debugging
        if (!empty($userErrors)) {
            $error_msg = $userErrors[0]['message'] ?? 'Unknown error';
            $error_code = $userErrors[0]['code'] ?? '';
            \LightSyncPro\Util\Logger::debug('[LSP Shopify Update] fileUpdate failed: ' . $error_msg . ' (code: ' . $error_code . ')');
        }
    }
    
    // Step 4: Fallback - delete old file and create new one
    // This happens if extension changed (e.g., jpg→avif) or other fileUpdate issues
    \LightSyncPro\Util\Logger::debug('[LSP Shopify Update] In-place update failed, falling back to delete + re-upload');
    
    $delete_result = self::delete_shopify_file($shop, $token, $file_id);
    if ($delete_result['ok']) {
        \LightSyncPro\Util\Logger::debug('[LSP Shopify Update] Old file deleted');
    }
    
    // Create new file from the already-staged upload
    $file_result = self::create_file_from_staged(
        $shop,
        $token,
        $staged_target['resourceUrl'],
        $filename,
        $alt
    );
    
    if ($file_result['ok']) {
        \LightSyncPro\Util\Logger::debug('[LSP Shopify Update] FALLBACK SUCCESS - new file_id=' . ($file_result['file_id'] ?? ''));
        \LightSyncPro\Util\Logger::debug('[LSP Shopify Update] WARNING: File ID changed! Old=' . $file_id . ', New=' . ($file_result['file_id'] ?? ''));
        $file_result['id_changed'] = true;
        $file_result['old_file_id'] = $file_id;
    }
    
    return $file_result;
}

/**
 * Delete a file from Shopify
 */
private static function delete_shopify_file(
    string $shop,
    string $token,
    string $file_id
): array {
    $query = <<<'GRAPHQL'
mutation fileDelete($fileIds: [ID!]!) {
    fileDelete(fileIds: $fileIds) {
        deletedFileIds
        userErrors {
            field
            message
        }
    }
}
GRAPHQL;

    $variables = [
        'fileIds' => [$file_id],
    ];

    $response = self::graphql_request($shop, $token, $query, $variables);

    if (!$response['ok']) {
        return ['ok' => false, 'error' => 'GraphQL error: ' . ($response['error'] ?? 'unknown')];
    }

    $data = $response['data']['fileDelete'] ?? [];
    $userErrors = $data['userErrors'] ?? [];

    if (!empty($userErrors)) {
        return ['ok' => false, 'error' => $userErrors[0]['message'] ?? 'Unknown error'];
    }

    return ['ok' => true];
}


    /**
     * Upload a single file to Shopify Files using staged uploads.
     */
    private static function upload_file_to_shopify(
        string $shop,
        string $token,
        string $bytes,
        string $filename,
        string $mime_type,
        string $alt = ''
    ): array {

        // Step 1: Create staged upload target
        $staged = self::create_staged_upload($shop, $token, $filename, $mime_type, strlen($bytes));
        if (!$staged['ok']) {
            return $staged;
        }

        $staged_target = $staged['target'];

        // Step 2: Upload bytes to staged URL
        $upload_result = self::upload_to_staged_target($staged_target, $bytes, $mime_type);
        if (!$upload_result['ok']) {
            return $upload_result;
        }

        // Step 3: Create the file in Shopify
        $file_result = self::create_file_from_staged(
            $shop,
            $token,
            $staged_target['resourceUrl'],
            $filename,
            $alt
        );

        return $file_result;
    }

    /**
     * Public method to upload Canva design bytes directly to Shopify Files
     * 
     * @param string $bytes Raw image bytes
     * @param string $filename Filename with extension
     * @param string $mime_type MIME type (image/png, image/jpeg, etc.)
     * @param string $alt Alt text for the image
     * @param string $asset_id Unique asset ID for tracking
     * @param string $content_hash Optional hash of original content for change detection
     * @return array Result with ok, file_id, file_url, updated, skipped or error
     */
    public static function upload_canva_to_shopify(
        string $bytes,
        string $filename,
        string $mime_type,
        string $alt,
        string $asset_id,
        string $content_hash = ''
    ): array {
        // Get shop domain
        $shop = \LightSyncPro\Admin\Admin::get_opt('shopify_shop_domain');
        if (!$shop) {
            return ['ok' => false, 'error' => 'Shopify not connected'];
        }
        
        $shop = self::normalize_shop($shop);
        $token = self::get_token($shop);
        
        if (!$token) {
            return ['ok' => false, 'error' => 'Missing Shopify access token'];
        }
        
        if (empty($bytes) || strlen($bytes) < 100) {
            return ['ok' => false, 'error' => 'Invalid image bytes'];
        }
        
        // Ensure files map table exists
        self::ensure_files_map_table();
        
        // Generate checksum from actual image content only
        // The bytes are what matter - if the image content changed, bytes change
        // Don't include content_hash (md5 of URL) as Canva URLs change every export
        $checksum = hash('sha256', $bytes);
        
        // Check if already uploaded (for updates)
        $existing = self::get_files_map_row($shop, $asset_id);
        $is_update = false;
        $was_skipped = false;
        
        // Debug logging
        \LightSyncPro\Util\Logger::debug('[LSP Canva→Shopify] asset_id=' . $asset_id . ', existing=' . ($existing ? 'yes (file_id=' . ($existing['shopify_file_id'] ?? 'none') . ')' : 'no'));
        if ($existing) {
            \LightSyncPro\Util\Logger::debug('[LSP Canva→Shopify] stored_checksum=' . substr($existing['checksum'] ?? '', 0, 16) . '..., new_checksum=' . substr($checksum, 0, 16) . '...');
        }
        
        if ($existing && !empty($existing['shopify_file_id'])) {
            // Check if content has changed
            if (!empty($existing['checksum']) && hash_equals((string)$existing['checksum'], $checksum)) {
                // No changes - skip upload
                \LightSyncPro\Util\Logger::debug('[LSP Canva→Shopify] Skipping - checksums match');
                return [
                    'ok' => true,
                    'file_id' => $existing['shopify_file_id'],
                    'file_url' => $existing['shopify_file_url'] ?? '',
                    'updated' => false,
                    'skipped' => true,
                ];
            }
            
            // Content changed - update existing file
            \LightSyncPro\Util\Logger::debug('[LSP Canva→Shopify] Updating file_id=' . $existing['shopify_file_id']);
            $is_update = true;
            $result = self::update_shopify_file(
                $shop,
                $token,
                $existing['shopify_file_id'],
                $bytes,
                $filename,
                $mime_type,
                $alt
            );
            
            if (!$result['ok']) {
                \LightSyncPro\Util\Logger::debug('[LSP Canva→Shopify] Update failed: ' . ($result['error'] ?? 'unknown'));
            } else {
                \LightSyncPro\Util\Logger::debug('[LSP Canva→Shopify] Update succeeded');
            }
        } else {
            // New upload
            \LightSyncPro\Util\Logger::debug('[LSP Canva→Shopify] New upload');
            $result = self::upload_file_to_shopify($shop, $token, $bytes, $filename, $mime_type, $alt);
        }
        
        if (!$result['ok']) {
            return $result;
        }
        
        // Store/update mapping with new checksum
        self::upsert_files_map_row(
            $shop,
            $asset_id,
            'canva',  // catalog_id
            '',       // album_id (not applicable for Canva)
            $result['file_id'] ?? $existing['shopify_file_id'] ?? '',
            $result['file_url'] ?? '',
            $checksum,
            $mime_type // Store format for future updates
        );
        
        return [
            'ok' => true,
            'file_id' => $result['file_id'] ?? '',
            'file_url' => $result['file_url'] ?? '',
            'updated' => $is_update,
            'skipped' => false,
        ];
    }

    /**
     * Upload a file to Shopify Files (for Dropbox/generic uploads)
     * Includes duplicate detection and mapping tracking like Canva
     * 
     * @param string $bytes Raw file bytes
     * @param string $filename Filename with extension
     * @param string $source_id Unique ID from source (e.g. Dropbox file ID)
     * @param string $source Source name (e.g. 'dropbox')
     * @param string $alt Optional alt text
     * @return array|WP_Error Result with file_id, file_url, skipped, updated flags
     */
    public static function upload_file(string $bytes, string $filename, string $source_id = '', string $source = 'dropbox', string $alt = '') {
        // Get shop domain
        $shop = \LightSyncPro\Admin\Admin::get_opt('shopify_shop_domain');
        if (!$shop) {
            return new \WP_Error('not_connected', 'Shopify not connected');
        }
        
        $shop = self::normalize_shop($shop);
        $token = self::get_token($shop);
        
        if (!$token) {
            return new \WP_Error('no_token', 'Missing Shopify access token');
        }
        
        if (empty($bytes) || strlen($bytes) < 100) {
            return new \WP_Error('invalid_bytes', 'Invalid file bytes');
        }
        
        // Determine MIME type from filename
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $mime_map = [
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
            'gif' => 'image/gif',
            'webp' => 'image/webp',
            'avif' => 'image/avif',
            'svg' => 'image/svg+xml',
        ];
        $mime_type = $mime_map[$ext] ?? 'image/jpeg';
        
        \LightSyncPro\Util\Logger::debug('[LSP Shopify] upload_file: ' . $filename . ' (' . strlen($bytes) . ' bytes, ' . $mime_type . ', source_id=' . $source_id . ')');
        
        // If we have a source_id, check for existing mapping to prevent duplicates
        $is_update = false;
        $was_skipped = false;
        $existing = null;
        
        if (!empty($source_id)) {
            // Ensure files map table exists
            self::ensure_files_map_table();
            
            // Generate checksum from actual image content
            $checksum = hash('sha256', $bytes);
            
            // Check if already uploaded
            $existing = self::get_files_map_row($shop, $source_id);
            
            \LightSyncPro\Util\Logger::debug('[LSP Shopify] source_id=' . $source_id . ', existing=' . ($existing ? 'yes (file_id=' . ($existing['shopify_file_id'] ?? 'none') . ')' : 'no'));
            
            if ($existing && !empty($existing['shopify_file_id'])) {
                // Check if content has changed
                if (!empty($existing['checksum']) && hash_equals((string)$existing['checksum'], $checksum)) {
                    // Content unchanged - verify file still exists on Shopify before skipping
                    $file_still_exists = self::verify_shopify_file_exists($shop, $token, $existing['shopify_file_id']);
                    if ($file_still_exists) {
                        // File exists and content unchanged - safe to skip
                        \LightSyncPro\Util\Logger::debug('[LSP Shopify] Skipping - checksums match and file exists');
                        return [
                            'file_id' => $existing['shopify_file_id'],
                            'file_url' => $existing['shopify_file_url'] ?? '',
                            'updated' => false,
                            'skipped' => true,
                        ];
                    }
                    // File was deleted from Shopify - clear mapping and re-upload
                    \LightSyncPro\Util\Logger::debug('[LSP Shopify] File deleted from Shopify - re-uploading');
                    self::delete_files_map_row($shop, $source_id);
                    $existing = null;
                }
                
                if ($existing) {
                    // Content changed - update existing file (preserves Shopify file ID)
                    // update_shopify_file tries fileUpdate first, falls back to delete+reupload if needed
                    \LightSyncPro\Util\Logger::debug('[LSP Shopify] Updating file_id=' . $existing['shopify_file_id'] . ' (mime: ' . ($existing['mime_type'] ?? 'unknown') . ' → ' . $mime_type . ')');
                    $is_update = true;
                    $result = self::update_shopify_file(
                        $shop,
                        $token,
                        $existing['shopify_file_id'],
                        $bytes,
                        $filename,
                        $mime_type,
                        $alt
                    );
                } else {
                    // Mapping was cleared (file deleted from Shopify) - fresh upload
                    \LightSyncPro\Util\Logger::debug('[LSP Shopify] Fresh upload after mapping cleared');
                    $result = self::upload_file_to_shopify($shop, $token, $bytes, $filename, $mime_type, $alt);
                }
            } else {
                // New upload
                \LightSyncPro\Util\Logger::debug('[LSP Shopify] New upload');
                $result = self::upload_file_to_shopify($shop, $token, $bytes, $filename, $mime_type, $alt);
            }
        } else {
            // No source_id - just upload without tracking (legacy behavior)
            $result = self::upload_file_to_shopify($shop, $token, $bytes, $filename, $mime_type, $alt);
            $checksum = '';
        }
        
        if (!$result['ok']) {
            \LightSyncPro\Util\Logger::debug('[LSP Shopify] upload_file failed: ' . ($result['error'] ?? 'unknown'));
            return new \WP_Error('upload_failed', $result['error'] ?? 'Shopify upload failed');
        }
        
        // Store/update mapping if we have a source_id
        if (!empty($source_id)) {
            self::upsert_files_map_row(
                $shop,
                $source_id,
                $source,  // catalog_id field used for source type
                '',       // album_id (not applicable)
                $result['file_id'] ?? $existing['shopify_file_id'] ?? '',
                $result['file_url'] ?? '',
                $checksum,
                $mime_type // Store format for future updates
            );
        }
        
        \LightSyncPro\Util\Logger::debug('[LSP Shopify] upload_file success: file_id=' . ($result['file_id'] ?? ''));
        
        return [
            'file_id' => $result['file_id'] ?? '',
            'file_url' => $result['file_url'] ?? '',
            'updated' => $is_update,
            'skipped' => false,
        ];
    }

    /**
     * Step 1: Request a staged upload URL from Shopify.
     */
    private static function create_staged_upload(
        string $shop,
        string $token,
        string $filename,
        string $mime_type,
        int $filesize
    ): array {

        $query = <<<'GRAPHQL'
mutation stagedUploadsCreate($input: [StagedUploadInput!]!) {
  stagedUploadsCreate(input: $input) {
    stagedTargets {
      url
      resourceUrl
      parameters {
        name
        value
      }
    }
    userErrors {
      field
      message
    }
  }
}
GRAPHQL;

        $variables = [
            'input' => [
                [
                    'filename' => $filename,
                    'mimeType' => $mime_type,
                    'fileSize' => (string)$filesize,
                    'resource' => 'FILE',
                    'httpMethod' => 'POST',
                ],
            ],
        ];

        $response = self::graphql_request($shop, $token, $query, $variables);

        if (!$response['ok']) {
            return $response;
        }

        $data = $response['data']['stagedUploadsCreate'] ?? [];

        if (!empty($data['userErrors'])) {
            $error_msg = $data['userErrors'][0]['message'] ?? 'Staged upload failed';
            return ['ok' => false, 'error' => $error_msg];
        }

        $target = $data['stagedTargets'][0] ?? null;
        if (!$target || empty($target['url'])) {
            return ['ok' => false, 'error' => 'No staged target returned'];
        }

        return ['ok' => true, 'target' => $target];
    }

    /**
     * Step 2: Upload file bytes to the staged URL.
     */
    private static function upload_to_staged_target(array $target, string $bytes, string $mime_type): array {
        $url = $target['url'];
        $parameters = $target['parameters'] ?? [];

        // Build multipart form data
        $boundary = wp_generate_password(24, false);

        $body = '';

        // Add parameters first (required by Shopify's staged upload)
        foreach ($parameters as $param) {
            $body .= "--{$boundary}\r\n";
            $body .= "Content-Disposition: form-data; name=\"{$param['name']}\"\r\n\r\n";
            $body .= "{$param['value']}\r\n";
        }

        // Add file
        $body .= "--{$boundary}\r\n";
        $body .= "Content-Disposition: form-data; name=\"file\"; filename=\"upload\"\r\n";
        $body .= "Content-Type: {$mime_type}\r\n\r\n";
        $body .= $bytes . "\r\n";
        $body .= "--{$boundary}--\r\n";

        $response = wp_remote_post($url, [
            'timeout' => 120,
            'headers' => [
                'Content-Type' => "multipart/form-data; boundary={$boundary}",
            ],
            'body' => $body,
        ]);

        if (is_wp_error($response)) {
            return ['ok' => false, 'error' => $response->get_error_message()];
        }

        $code = (int)wp_remote_retrieve_response_code($response);

        // Shopify returns 201 or 204 on success
        if ($code >= 200 && $code < 300) {
            return ['ok' => true];
        }

        $body = wp_remote_retrieve_body($response);
        return ['ok' => false, 'error' => "Upload failed with HTTP {$code}: " . substr($body, 0, 300)];
    }

    /**
     * Step 3: Create the file record in Shopify from staged upload.
     */
    private static function create_file_from_staged(
        string $shop,
        string $token,
        string $resource_url,
        string $filename,
        string $alt = ''
    ): array {

        $query = <<<'GRAPHQL'
mutation fileCreate($files: [FileCreateInput!]!) {
  fileCreate(files: $files) {
    files {
      id
      alt
      createdAt
      ... on MediaImage {
        image {
          url
        }
      }
    }
    userErrors {
      field
      message
    }
  }
}
GRAPHQL;

        $file_input = [
            'originalSource' => $resource_url,
            'filename' => $filename,
            'contentType' => 'IMAGE',
        ];

        if ($alt !== '') {
            $file_input['alt'] = $alt;
        }

        $variables = [
            'files' => [$file_input],
        ];

        $response = self::graphql_request($shop, $token, $query, $variables);

        if (!$response['ok']) {
            return $response;
        }

        $data = $response['data']['fileCreate'] ?? [];

        if (!empty($data['userErrors'])) {
            $error_msg = $data['userErrors'][0]['message'] ?? 'File create failed';
            return ['ok' => false, 'error' => $error_msg];
        }

        $file = $data['files'][0] ?? null;
        if (!$file) {
            return ['ok' => false, 'error' => 'No file returned'];
        }

        return [
            'ok' => true,
            'file_id' => $file['id'] ?? '',
            'file_url' => $file['image']['url'] ?? '',
            'alt' => $file['alt'] ?? '',
        ];
    }

    /**
     * Make a GraphQL request to Shopify.
     */
    private static function graphql_request(string $shop, string $token, string $query, array $variables = []): array {
        $url = "https://{$shop}/admin/api/" . self::API_VERSION . "/graphql.json";

        $body = ['query' => $query];
        if (!empty($variables)) {
            $body['variables'] = $variables;
        }

        $response = wp_remote_post($url, [
            'timeout' => 60,
            'headers' => [
                'X-Shopify-Access-Token' => $token,
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($body),
        ]);

        if (is_wp_error($response)) {
            return ['ok' => false, 'error' => $response->get_error_message()];
        }

        $code = (int)wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $json = json_decode($body, true);

        if ($code !== 200) {
            return ['ok' => false, 'error' => "HTTP {$code}: " . substr($body, 0, 300)];
        }

        if (!empty($json['errors'])) {
            $error_msg = is_array($json['errors'])
                ? ($json['errors'][0]['message'] ?? wp_json_encode($json['errors']))
                : (string)$json['errors'];
            return ['ok' => false, 'error' => $error_msg];
        }

        return ['ok' => true, 'data' => $json['data'] ?? []];
    }

    /* ==================== LIGHTROOM IMAGE FETCHING ==================== */

   /**
 * Fetch image bytes from Lightroom
 */
/**
 * Fetch image bytes from Lightroom using existing OAuth infrastructure
 */
private static function fetch_lightroom_image(string $url) {
    $token_check = OAuth::ensure_token();
    if (is_wp_error($token_check)) {
        return new \WP_Error('no_adobe_token', 'Adobe OAuth token not available: ' . $token_check->get_error_message());
    }

    $headers = OAuth::headers();

    $response = wp_remote_get($url, [
        'timeout' => 60,
        'headers' => $headers,
    ]);

    if (is_wp_error($response)) {
        return $response;
    }

    $code = (int)wp_remote_retrieve_response_code($response);
    if ($code !== 200) {
        return new \WP_Error(
            'fetch_failed',
            sprintf('Lightroom returned %d: %s', $code, substr(wp_remote_retrieve_body($response), 0, 200))
        );
    }

    return wp_remote_retrieve_body($response);
}

    private static function detect_mime_from_bytes(string $bytes): string {
        $sig = substr($bytes, 0, 8);

        if (strncmp($sig, "\xFF\xD8\xFF", 3) === 0) {
            return 'image/jpeg';
        }
        if ($sig === "\x89PNG\r\n\x1A\n") {
            return 'image/png';
        }
        if (substr($bytes, 0, 4) === "RIFF" && substr($bytes, 8, 4) === "WEBP") {
            return 'image/webp';
        }

        return 'image/jpeg';
    }

    /* ==================== FILES MAPPING TABLE ==================== */

    private static function files_map_table(): string {
        global $wpdb;
        return $wpdb->prefix . 'lightsync_shopify_files_map';
    }

    public static function ensure_files_map_table(): void {
        global $wpdb;
        $tbl = self::files_map_table();

        $exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $tbl));
        if ($exists === $tbl) {
            // Check if mime_type column exists, add if not
            $col_exists = $wpdb->get_var("SHOW COLUMNS FROM {$tbl} LIKE 'mime_type'");
            if (!$col_exists) {
                $wpdb->query("ALTER TABLE {$tbl} ADD COLUMN mime_type VARCHAR(64) NULL AFTER checksum");
            }
            return;
        }

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        $charset = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$tbl} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            shop_domain VARCHAR(255) NOT NULL,
            lr_asset_id VARCHAR(64) NOT NULL,
            catalog_id VARCHAR(64) NULL,
            album_id VARCHAR(64) NULL,
            shopify_file_id VARCHAR(128) NULL,
            shopify_file_url TEXT NULL,
            checksum CHAR(64) NULL,
            mime_type VARCHAR(64) NULL,
            created_at DATETIME NULL,
            updated_at DATETIME NULL,
            PRIMARY KEY (id),
            UNIQUE KEY uniq_asset (shop_domain, lr_asset_id),
            KEY catalog_album (catalog_id, album_id)
        ) {$charset};";

        dbDelta($sql);
    }

    private static function get_files_map_row(string $shop, string $asset_id): ?array {
        global $wpdb;
        $tbl = self::files_map_table();

        $row = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$tbl} WHERE shop_domain=%s AND lr_asset_id=%s LIMIT 1",
                $shop, $asset_id
            ),
            ARRAY_A
        );

        return $row ?: null;
    }

    private static function upsert_files_map_row(
        string $shop,
        string $asset_id,
        string $catalog_id,
        string $album_id,
        string $shopify_file_id,
        string $shopify_file_url,
        string $checksum,
        string $mime_type = ''
    ): void {
        global $wpdb;
        $tbl = self::files_map_table();
        $now = gmdate('Y-m-d H:i:s');

        $data = [
            'shop_domain' => $shop,
            'lr_asset_id' => $asset_id,
            'catalog_id' => $catalog_id,
            'album_id' => $album_id,
            'shopify_file_id' => $shopify_file_id,
            'shopify_file_url' => $shopify_file_url,
            'checksum' => $checksum,
            'mime_type' => $mime_type,
            'updated_at' => $now,
        ];

        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$tbl} WHERE shop_domain=%s AND lr_asset_id=%s LIMIT 1",
            $shop, $asset_id
        ));

        if ($exists) {
            $wpdb->update($tbl, $data, ['id' => (int)$exists]);
        } else {
            $data['created_at'] = $now;
            $wpdb->insert($tbl, $data);
        }
    }

    /**
     * Delete a mapping row from the files map table
     */
    private static function delete_files_map_row(string $shop, string $asset_id): void {
        global $wpdb;
        $tbl = self::files_map_table();
        $wpdb->delete($tbl, [
            'shop_domain' => $shop,
            'lr_asset_id' => $asset_id,
        ]);
    }

    /**
     * Verify a file still exists on Shopify (lightweight check)
     */
    private static function verify_shopify_file_exists(string $shop, string $token, string $file_id): bool {
        $query = <<<'GRAPHQL'
query fileCheck($id: ID!) {
    node(id: $id) {
        ... on MediaImage { id fileStatus }
        ... on GenericFile { id }
    }
}
GRAPHQL;

        $response = self::graphql_request($shop, $token, $query, ['id' => $file_id]);
        
        if (!$response['ok'] || empty($response['data']['node'])) {
            return false;
        }
        
        return true;
    }

    /* ==================== HELPERS ==================== */

    private static function normalize_shop(string $shop): string {
        $shop = strtolower(trim($shop));
        $shop = preg_replace('#^https?://#', '', $shop);
        $shop = preg_replace('#/.*$#', '', $shop);
        return preg_match('/^[a-z0-9][a-z0-9\-]*\.myshopify\.com$/', $shop) ? $shop : '';
    }

    private static function build_alt(array $ctx, string $lr_asset_id): string {
        $catalog_id = (string)($ctx['catalog_id'] ?? '');
        $album_id = (string)($ctx['album_id'] ?? '');

        if ($catalog_id !== '' && $album_id !== '' && $lr_asset_id !== '') {
            return "LSP:{$catalog_id}:{$album_id}:{$lr_asset_id}";
        }
        return $lr_asset_id !== '' ? "LSP:{$lr_asset_id}" : '';
    }

  private static function get_token(string $shop): string {
    // First check local storage
    $o = get_option('lightsyncpro_settings', []);
    if (!is_array($o)) {
        $o = [];
    }
    
    // Check if shopify_access_token exists and is an array
    $tokens = $o['shopify_access_token'] ?? [];
    if (!is_array($tokens)) {
        $tokens = [];
    }
    
    if (isset($tokens[$shop])) {
        $local = trim((string)$tokens[$shop]);
        if ($local !== '') return $local;
    }

    // Fetch from broker
    $broker_token = self::get_broker_token();
    if (!$broker_token) return '';

    $response = wp_remote_post('https://lightsyncpro.com/wp-json/lsp-broker/v1/shopify/token', [
        'timeout' => 15,
        'headers' => [
            'Authorization' => 'Bearer ' . $broker_token,
            'Content-Type'  => 'application/json',
        ],
        'body' => wp_json_encode(['shop_domain' => $shop]),
    ]);

    if (is_wp_error($response)) return '';

    $body = json_decode(wp_remote_retrieve_body($response), true);
    if (empty($body['access_token'])) return '';

    // Cache locally - ensure shopify_access_token is an array
    if (!is_array($o['shopify_access_token'] ?? null)) {
        $o['shopify_access_token'] = [];
    }
    $o['shopify_access_token'][$shop] = $body['access_token'];
    update_option('lightsyncpro_settings', $o, false);

    return $body['access_token'];
}

private static function get_broker_token(): string {
    $o = get_option('lightsyncpro_settings', []);
    $enc = (string)($o['broker_token_enc'] ?? '');
    if ($enc === '') return '';

    return (string)\LightSyncPro\Util\Crypto::dec($enc);
}

    /* ==================== LEGACY METHODS (keep for backward compat) ==================== */



    public static function asset_ids_to_attachment_ids(array $asset_ids): array {
        $asset_ids = array_values(array_filter(array_map('strval', $asset_ids)));
        if (!$asset_ids) return [];

        global $wpdb;
        $meta_key = '_lightsync_lr_asset_id';

        $out = [];
        foreach (array_chunk($asset_ids, 200) as $chunk) {
            $placeholders = implode(',', array_fill(0, count($chunk), '%s'));
            $sql = "
                SELECT post_id
                FROM {$wpdb->postmeta}
                WHERE meta_key = %s
                  AND meta_value IN ($placeholders)
            ";
            $params = array_merge([$meta_key], $chunk);
            $ids = $wpdb->get_col($wpdb->prepare($sql, ...$params));
            foreach ($ids as $id) $out[(int)$id] = true;
        }
        return array_values(array_keys($out));
    }

    /**
     * Hub: Upload file to Shopify (new file)
     * 
     * @param string $shop Shopify shop domain
     * @param string $token Access token
     * @param string $bytes Raw image bytes
     * @param string $filename Filename with extension
     * @param string $mime_type MIME type
     * @param string $alt Alt text
     * @return array Result with ok, file_id, file_url or error
     */
    public static function hub_upload_file(
        string $shop,
        string $token,
        string $bytes,
        string $filename,
        string $mime_type,
        string $alt = ''
    ): array {
        return self::upload_file_to_shopify($shop, $token, $bytes, $filename, $mime_type, $alt);
    }

    /**
     * Hub: Update existing Shopify file (delete old + upload new)
     * 
     * @param string $shop Shopify shop domain
     * @param string $token Access token
     * @param string $existing_id Existing file ID to replace
     * @param string $bytes Raw image bytes
     * @param string $filename Filename with extension
     * @param string $mime_type MIME type
     * @param string $alt Alt text
     * @return array Result with ok, file_id, file_url or error
     */
    public static function hub_update_file(
        string $shop,
        string $token,
        string $existing_id,
        string $bytes,
        string $filename,
        string $mime_type,
        string $alt = '',
        string $stored_mime_type = ''
    ): array {
        // If we have a stored mime_type and it differs from current, convert to match
        if ($stored_mime_type && $stored_mime_type !== $mime_type) {
            \LightSyncPro\Util\Logger::debug("[LSP Shopify Hub] Converting {$mime_type} to stored format {$stored_mime_type}");
            
            $converted = self::convert_to_mime($bytes, $mime_type, $stored_mime_type);
            if ($converted) {
                $bytes = $converted['bytes'];
                $mime_type = $converted['mime'];
                // Update filename extension
                $ext_map = ['image/jpeg' => 'jpg', 'image/png' => 'png', 'image/webp' => 'webp', 'image/avif' => 'avif'];
                $new_ext = $ext_map[$mime_type] ?? pathinfo($filename, PATHINFO_EXTENSION);
                $filename = pathinfo($filename, PATHINFO_FILENAME) . '.' . $new_ext;
                \LightSyncPro\Util\Logger::debug("[LSP Shopify Hub] Converted to {$mime_type}, filename={$filename}");
            } else {
                \LightSyncPro\Util\Logger::debug("[LSP Shopify Hub] Conversion failed, proceeding with original format");
            }
        }
        
        // Use the same in-place update as regular syncs
        return self::update_shopify_file($shop, $token, $existing_id, $bytes, $filename, $mime_type, $alt);
    }
    
    /**
     * Convert image bytes to a specific mime type
     */
    private static function convert_to_mime(string $bytes, string $from_mime, string $to_mime): ?array {
        // Create image resource
        $image = @imagecreatefromstring($bytes);
        if (!$image) {
            return null;
        }
        
        imagealphablending($image, true);
        imagesavealpha($image, true);
        
        ob_start();
        $success = false;
        $ext = '';
        
        switch ($to_mime) {
            case 'image/jpeg':
            case 'image/jpg':
                $success = @imagejpeg($image, null, 90);
                $to_mime = 'image/jpeg';
                $ext = 'jpg';
                break;
            case 'image/png':
                $success = @imagepng($image, null, 9);
                $ext = 'png';
                break;
            case 'image/webp':
                if (function_exists('imagewebp')) {
                    $success = @imagewebp($image, null, 85);
                    $ext = 'webp';
                }
                break;
            case 'image/avif':
                if (function_exists('imageavif')) {
                    $success = @imageavif($image, null, 70);
                    $ext = 'avif';
                }
                break;
        }
        
        $output = ob_get_clean();
        imagedestroy($image);
        
        if ($success && strlen($output) > 0) {
            return ['bytes' => $output, 'mime' => $to_mime, 'ext' => $ext];
        }
        
        return null;
    }
}