<?php
namespace LightSyncPro\OAuth;

use LightSyncPro\Admin\Admin;
use LightSyncPro\Util\Crypto;
use LightSyncPro\Util\Logger;


/**
 * Shutterstock OAuth handler for LightSync Pro
 * Handles broker-based OAuth flow and token management for Shutterstock
 * 
 * Allows users to sync their previously licensed/purchased images to WordPress
 */
class ShutterstockOAuth {
    const BROKER_URL = 'https://lightsyncpro.com';
    const API_BASE = 'https://api.shutterstock.com/v2';

    public static function init() {
        // Handle ?lsp_shutterstock_connected=1 callback
        add_action('admin_init', [__CLASS__, 'handle_connection_callback']);
    }

    /**
     * Check if Shutterstock is connected
     */
    public static function is_connected() {
        $o = Admin::get_opt();
        return !empty($o['shutterstock_broker_token_enc']);
    }

    /**
     * Get Shutterstock auth URL (starts OAuth flow via broker)
     */
    public static function auth_url() {
        $state  = wp_create_nonce('lightsync_shutterstock_oauth_state');
        $site   = home_url();
        $return = admin_url('admin.php?page=lightsyncpro&tab=source');
        
        return self::BROKER_URL . '/wp-json/lsp-broker/v1/shutterstock/connect/start'
             . '?site='   . rawurlencode($site)
             . '&state='  . rawurlencode($state)
             . '&return=' . rawurlencode($return);
    }

    /**
     * Handle the connection callback (?lsp_shutterstock_connected=1)
     */
    public static function handle_connection_callback() {
        if (!isset($_GET['lsp_shutterstock_connected']) || $_GET['lsp_shutterstock_connected'] !== '1') {
            return;
        }

        // Verify state nonce
        $state = isset($_GET['state']) ? sanitize_text_field($_GET['state']) : '';
        if (!wp_verify_nonce($state, 'lightsync_shutterstock_oauth_state')) {
            Logger::debug('[LSP Shutterstock] Invalid state nonce on callback');
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error"><p>Shutterstock connection failed: Invalid security token. Please try again.</p></div>';
            });
            return;
        }

        // Pick up the broker token
        $site = home_url();
        $pickup_url = self::BROKER_URL . '/wp-json/lsp-broker/v1/shutterstock/install/pickup'
                    . '?site=' . rawurlencode($site)
                    . '&state=' . rawurlencode($state);

        $response = wp_remote_get($pickup_url, [
            'timeout' => 20,
            'headers' => ['Accept' => 'application/json'],
        ]);

        if (is_wp_error($response)) {
            Logger::debug('[LSP Shutterstock] Pickup request failed: ' . $response->get_error_message());
            add_action('admin_notices', function() use ($response) {
                echo '<div class="notice notice-error"><p>Shutterstock connection failed: ' . esc_html($response->get_error_message()) . '</p></div>';
            });
            return;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (empty($data['success']) || empty($data['data']['broker_token'])) {
            $error = $data['data']['error'] ?? 'Unknown error';
            Logger::debug('[LSP Shutterstock] Pickup failed: ' . $error);
            add_action('admin_notices', function() use ($error) {
                echo '<div class="notice notice-error"><p>Shutterstock connection failed: ' . esc_html($error) . '</p></div>';
            });
            return;
        }

        // Encrypt and store the broker token
        $broker_token = $data['data']['broker_token'];
        $encrypted = Crypto::enc($broker_token);

        Admin::set_opt([
            'shutterstock_broker_token_enc' => $encrypted,
            'shutterstock_connected_at'     => time(),
        ]);

        Logger::debug('[LSP Shutterstock] Successfully connected and stored broker token');

        add_action('admin_notices', function() {
            echo '<div class="notice notice-success"><p>✓ Successfully connected to Shutterstock!</p></div>';
        });
    }

    /**
     * Ensure access token is valid, refresh if needed
     */
    public static function ensure_token() {
        $o   = Admin::get_opt();
        $exp = (int)($o['shutterstock_token_expires'] ?? 0);
        $now = time();

        // Shutterstock non-expiring tokens have expires=0, always valid
        // For expiring tokens, check if we have >60s remaining
        if (!empty($o['shutterstock_access_token'])) {
            if ($exp === 0 || ($exp - $now) > 60) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    Logger::debug('[LSP Shutterstock] ensure_token: using cached token');
                }
                return true;
            }
        }

        // Need to refresh from broker
        $enc = $o['shutterstock_broker_token_enc'] ?? '';
        if (!$enc) {
            Logger::debug('[LSP Shutterstock] ensure_token: no broker token stored');
            return new \WP_Error('no_broker', 'Shutterstock not connected. Click Connect to Shutterstock.');
        }

        $broker = Crypto::dec($enc);
        if (!$broker) {
            Logger::debug('[LSP Shutterstock] ensure_token: failed to decrypt broker token');
            return new \WP_Error('bad_broker', 'Stored Shutterstock broker token not readable.');
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Shutterstock] ensure_token: refreshing from broker...');
        }

        $resp = wp_remote_post(
            self::BROKER_URL . '/wp-json/lsp-broker/v1/shutterstock/token/refresh',
            [
                'timeout' => 20,
                'headers' => [
                    'Authorization' => 'Bearer ' . $broker,
                    'Accept'        => 'application/json',
                ],
            ]
        );

        if (is_wp_error($resp)) {
            Logger::debug('[LSP Shutterstock] ensure_token: broker request failed - ' . $resp->get_error_message());
            return $resp;
        }

        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);
        $data = json_decode($body, true);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Shutterstock] ensure_token: broker response code=' . $code);
        }

        $ok      = is_array($data) && array_key_exists('success', $data) ? (bool)$data['success'] : true;
        $payload = (is_array($data) && array_key_exists('data', $data)) ? $data['data'] : $data;

        if (!$ok) {
            $msg = (is_array($payload) && !empty($payload['error'])) ? $payload['error'] : 'broker error';
            Logger::debug('[LSP Shutterstock] ensure_token: broker returned error - ' . $msg);
            
            if (!empty($payload['reconnect'])) {
                self::disconnect();
                return new \WP_Error('reconnect_required', 'Shutterstock authorization expired. Please reconnect.');
            }
            
            return new \WP_Error('no_access', 'Shutterstock broker refused: ' . $msg);
        }

        if (empty($payload['access_token'])) {
            Logger::debug('[LSP Shutterstock] ensure_token: no access_token in response, code=' . $code . ' body=' . $body);
            return new \WP_Error('no_access', 'Could not obtain Shutterstock access token from broker');
        }

        // Non-expiring tokens (v2/) have expires_in=0
        $expires_in = (int)($payload['expires_in'] ?? 0);
        $cushion    = 60;
        
        Admin::set_opt([
            'shutterstock_access_token'  => $payload['access_token'],
            'shutterstock_token_expires' => $expires_in > 0 ? ($now + max(300, $expires_in - $cushion)) : 0,
        ]);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Shutterstock] ensure_token: got new token, expires_in=' . $expires_in);
        }

        return true;
    }

    /**
     * Disconnect Shutterstock
     */
    public static function disconnect() {
        Admin::set_opt([
            'shutterstock_broker_token_enc' => '',
            'shutterstock_access_token'     => '',
            'shutterstock_token_expires'    => 0,
            'shutterstock_connected_at'     => 0,
        ]);
        Logger::debug('[LSP Shutterstock] Disconnected');
    }

    /**
     * Get auth headers for API requests
     */
    public static function headers() {
        $o = Admin::get_opt();
        return [
            'Authorization' => 'Bearer ' . ($o['shutterstock_access_token'] ?? ''),
            'User-Agent'    => 'LightSyncPro/' . LIGHTSYNC_PRO_VERSION,
        ];
    }

    /**
     * Get user's license history (purchased images)
     * 
     * @param int $page Page number (1-based)
     * @param int $per_page Items per page (max 100)
     * @return array|WP_Error
     */
    public static function get_licenses($page = 1, $per_page = 50) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $params = [
            'page'     => $page,
            'per_page' => min($per_page, 100),
            'image_view' => 'full', // Request full image data including preview URLs
        ];

        $response = wp_remote_get(
            self::API_BASE . '/images/licenses?' . http_build_query($params),
            [
                'timeout' => 30,
                'headers' => self::headers(),
            ]
        );

        if (is_wp_error($response)) {
            Logger::debug('[LSP Shutterstock] get_licenses error: ' . $response->get_error_message());
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Shutterstock] get_licenses failed: ' . $code . ' - ' . $body);
            return new \WP_Error('shutterstock_api_error', 'Shutterstock API error: ' . $code);
        }

        $data = json_decode($body, true);
        
        return [
            'licenses'    => $data['data'] ?? [],
            'total_count' => $data['total_count'] ?? 0,
            'page'        => $data['page'] ?? 1,
            'per_page'    => $data['per_page'] ?? $per_page,
        ];
    }

    /**
     * Redownload a previously licensed image
     * 
     * @param string $license_id The license ID
     * @return array|WP_Error Download URL and metadata
     */
    public static function redownload($license_id) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $response = wp_remote_post(
            self::API_BASE . '/images/licenses/' . $license_id . '/downloads',
            [
                'timeout' => 30,
                'headers' => array_merge(self::headers(), [
                    'Content-Type' => 'application/json',
                ]),
                'body' => '{}',
            ]
        );

        if (is_wp_error($response)) {
            Logger::debug('[LSP Shutterstock] redownload error: ' . $response->get_error_message());
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Shutterstock] redownload failed: ' . $code . ' - ' . $body);
            $data = json_decode($body, true);
            $msg = $data['message'] ?? ('HTTP ' . $code);
            return new \WP_Error('shutterstock_api_error', 'Redownload failed: ' . $msg);
        }

        return json_decode($body, true);
    }

    /**
     * Get image details by ID
     * 
     * @param string $image_id Shutterstock image ID
     * @return array|WP_Error
     */
    public static function get_image($image_id) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $response = wp_remote_get(
            self::API_BASE . '/images/' . $image_id,
            [
                'timeout' => 15,
                'headers' => self::headers(),
            ]
        );

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Shutterstock] get_image failed: ' . $code . ' - ' . $body);
            return new \WP_Error('shutterstock_api_error', 'Shutterstock API error: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Search for an image by ID to get its preview URL
     * Tries both creative and editorial endpoints
     * 
     * @param string $image_id
     * @return array|null Image data with preview URL
     */
    public static function search_image_by_id($image_id) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return null;
        }

        // Try creative images first
        $url = self::API_BASE . '/images/search?' . http_build_query([
            'query' => $image_id,
            'view' => 'full',
            'per_page' => 1,
        ]);

        $response = wp_remote_get($url, [
            'timeout' => 15,
            'headers' => self::headers(),
        ]);

        if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            foreach (($data['data'] ?? []) as $img) {
                if (($img['id'] ?? '') === $image_id) {
                    return $img;
                }
            }
        }
        
        // Try editorial images
        $url = self::API_BASE . '/editorial/images/search?' . http_build_query([
            'query' => $image_id,
            'per_page' => 1,
        ]);

        $response = wp_remote_get($url, [
            'timeout' => 15,
            'headers' => self::headers(),
        ]);

        if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            foreach (($data['data'] ?? []) as $img) {
                if (($img['id'] ?? '') === $image_id) {
                    return $img;
                }
            }
        }
        
        // Try direct image fetch (single image endpoint)
        $url = self::API_BASE . '/images/' . $image_id . '?view=full';
        
        $response = wp_remote_get($url, [
            'timeout' => 15,
            'headers' => self::headers(),
        ]);

        if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            if (!empty($data['id'])) {
                return $data;
            }
        }
        
        return null;
    }

    /**
     * Bulk fetch image details (for getting previews)
     * Uses GET /v2/images?id=xxx&id=yyy&view=full
     * 
     * @param array $image_ids Array of image IDs (max 100)
     * @return array|WP_Error
     */
    public static function get_images_bulk($image_ids) {
        // Debug: track entry
        update_option('lsp_shutterstock_bulk_debug', ['step' => 'entered', 'ids_count' => count($image_ids)]);
        
        if (empty($image_ids)) {
            update_option('lsp_shutterstock_bulk_debug', ['step' => 'empty_ids']);
            return [];
        }
        
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            update_option('lsp_shutterstock_bulk_debug', [
                'step' => 'token_error',
                'error' => $token_ok->get_error_message()
            ]);
            return $token_ok;
        }

        // Shutterstock API expects repeated id parameter: ?id=123&id=456&view=full
        $query_parts = ['view=full'];
        foreach ($image_ids as $id) {
            $query_parts[] = 'id=' . urlencode($id);
        }
        $query_string = implode('&', $query_parts);

        $url = self::API_BASE . '/images?' . $query_string;
        
        update_option('lsp_shutterstock_bulk_debug', [
            'step' => 'making_request',
            'url' => $url
        ]);

        $response = wp_remote_get(
            $url,
            [
                'timeout' => 30,
                'headers' => self::headers(),
            ]
        );

        if (is_wp_error($response)) {
            update_option('lsp_shutterstock_bulk_debug', [
                'step' => 'wp_error',
                'error' => $response->get_error_message()
            ]);
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        // Store raw response for debug
        update_option('lsp_shutterstock_bulk_debug', [
            'step' => 'got_response',
            'code' => $code,
            'body' => substr($body, 0, 2000),
        ]);

        if ($code !== 200) {
            return new \WP_Error('shutterstock_api_error', 'API error ' . $code . ': ' . substr($body, 0, 200));
        }

        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new \WP_Error('json_error', 'JSON parse error: ' . json_last_error_msg());
        }
        
        // Return as id => image map
        $result = [];
        foreach (($data['data'] ?? []) as $img) {
            if (!empty($img['id'])) {
                $result[$img['id']] = $img;
            }
        }
        
        return $result;
    }

    /**
     * Get user's collections
     * 
     * @return array|WP_Error
     */
    public static function get_collections() {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $response = wp_remote_get(
            self::API_BASE . '/images/collections',
            [
                'timeout' => 15,
                'headers' => self::headers(),
            ]
        );

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Shutterstock] get_collections failed: ' . $code . ' - ' . $body);
            return new \WP_Error('shutterstock_api_error', 'Shutterstock API error: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Get images in a collection
     * 
     * @param string $collection_id
     * @param int $page
     * @param int $per_page
     * @return array|WP_Error
     */
    public static function get_collection_items($collection_id, $page = 1, $per_page = 50) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $params = [
            'page'     => $page,
            'per_page' => min($per_page, 100),
        ];

        $response = wp_remote_get(
            self::API_BASE . '/images/collections/' . $collection_id . '/items?' . http_build_query($params),
            [
                'timeout' => 20,
                'headers' => self::headers(),
            ]
        );

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Shutterstock] get_collection_items failed: ' . $code . ' - ' . $body);
            return new \WP_Error('shutterstock_api_error', 'Shutterstock API error: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Download image content from a URL
     * 
     * @param string $url Download URL from redownload()
     * @return string|WP_Error Raw image bytes
     */
    public static function download($url) {
        Logger::debug('[LSP Shutterstock] download() starting for URL');

        $response = wp_remote_get($url, [
            'timeout' => 120,
        ]);

        if (is_wp_error($response)) {
            Logger::debug('[LSP Shutterstock] download() failed: ' . $response->get_error_message());
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);

        if ($code !== 200) {
            Logger::debug('[LSP Shutterstock] download() HTTP error: ' . $code);
            return new \WP_Error('download_error', 'Download failed: HTTP ' . $code);
        }

        $body = wp_remote_retrieve_body($response);
        Logger::debug('[LSP Shutterstock] download() success: ' . strlen($body) . ' bytes');
        
        return $body;
    }

    /**
     * Get thumbnail URL for an image (for preview in UI)
     * 
     * @param array $image Image data from get_licenses or get_image
     * @return string|null
     */
    public static function get_thumbnail_url($image) {
        $assets = $image['assets'] ?? [];
        
        // Prefer preview, then small_thumb, then large_thumb
        foreach (['preview', 'small_thumb', 'large_thumb'] as $size) {
            if (!empty($assets[$size]['url'])) {
                return $assets[$size]['url'];
            }
        }
        
        return null;
    }
}
