<?php
namespace LightSyncPro\OAuth;

use LightSyncPro\Admin\Admin;
use LightSyncPro\Util\Crypto;
use LightSyncPro\Util\Logger;


/**
 * Figma OAuth handler for LightSync Pro
 * Handles broker-based OAuth flow and token management for Figma
 * 
 * Figma API structure:
 * - Files contain Pages
 * - Pages contain Frames (artboards)
 * - We export Frames as images (PNG, JPG, SVG, PDF)
 */
class FigmaOAuth {
    const BROKER_URL = 'https://lightsyncpro.com';
    const API_BASE   = 'https://api.figma.com/v1';
    const CRON_FIGMA_TOKEN_GUARD = 'lightsync_figma_token_guard';
    
    // Rate limit tracking
    private static $rate_limit_until = null;

    public static function init() {
        // Handle ?lsp_figma_connected=1 callback
        add_action('admin_init', [__CLASS__, 'handle_connection_callback']);
        
        // Token refresh guard
        add_action('init', [__CLASS__, 'ensure_guard_scheduled']);
        add_action(self::CRON_FIGMA_TOKEN_GUARD, [__CLASS__, 'cron_token_guard']);
    }

    /**
     * Centralized API request with rate limit handling
     * 
     * @param string $endpoint API endpoint (without base URL)
     * @param array $args wp_remote_get args
     * @param int $max_retries Maximum retry attempts for rate limits
     * @return array|WP_Error Response data or error
     */
    public static function api_request($endpoint, $args = [], $max_retries = 3) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }
        
        // Check if we're still in a rate limit cooldown
        $cooldown_key = 'lsp_figma_rate_limit_until';
        $cooldown_until = get_transient($cooldown_key);
        if ($cooldown_until && time() < $cooldown_until) {
            $wait_seconds = $cooldown_until - time();
            
            // If wait time is crazy (>5 min), clear it - something went wrong
            if ($wait_seconds > 300) {
                Logger::debug('[LSP Figma] Clearing stuck rate limit cooldown: ' . $wait_seconds . 's');
                delete_transient($cooldown_key);
            } else {
                Logger::debug('[LSP Figma] In rate limit cooldown, ' . $wait_seconds . 's remaining');
                return new \WP_Error('figma_rate_limited', 
                    'Figma rate limit active. Please wait ' . $wait_seconds . ' seconds before retrying.',
                    ['retry_after' => $wait_seconds]
                );
            }
        }
        
        $url = strpos($endpoint, 'http') === 0 ? $endpoint : self::API_BASE . $endpoint;
        
        $defaults = [
            'timeout' => 30,
            'headers' => self::headers(),
        ];
        $args = wp_parse_args($args, $defaults);
        
        $attempts = 0;
        
        while ($attempts < $max_retries) {
            $attempts++;
            
            Logger::debug('[LSP Figma] API request attempt ' . $attempts . ': ' . $url);
            
            $response = wp_remote_get($url, $args);
            
            if (is_wp_error($response)) {
                Logger::debug('[LSP Figma] WP error: ' . $response->get_error_message());
                return $response;
            }
            
            $code = wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);
            $headers = wp_remote_retrieve_headers($response);
            
            // Handle rate limiting (429)
            if ($code === 429) {
                $retry_after = isset($headers['retry-after']) ? (int)$headers['retry-after'] : 60;
                $plan_tier = isset($headers['x-figma-plan-tier']) ? $headers['x-figma-plan-tier'] : 'unknown';
                $rate_type = isset($headers['x-figma-rate-limit-type']) ? $headers['x-figma-rate-limit-type'] : 'unknown';
                
                // Cap retry_after to max 5 minutes to prevent stuck states
                $retry_after = min($retry_after, 300);
                
                Logger::debug('[LSP Figma] Rate limited! Retry-After: ' . $retry_after . 's, Plan: ' . $plan_tier . ', Type: ' . $rate_type);
                
                // Store cooldown for other requests
                set_transient($cooldown_key, time() + $retry_after, $retry_after + 10);
                
                // If this is our last attempt, return error with details
                if ($attempts >= $max_retries) {
                    return new \WP_Error('figma_rate_limited', 
                        'Figma rate limit exceeded. Please wait ' . $retry_after . ' seconds. ' .
                        '(Plan: ' . $plan_tier . ', Limit type: ' . $rate_type . ')',
                        [
                            'retry_after' => $retry_after,
                            'plan_tier' => $plan_tier,
                            'rate_limit_type' => $rate_type,
                        ]
                    );
                }
                
                // Wait and retry (only if we have attempts left and retry is short)
                if ($retry_after <= 10) {
                    Logger::debug('[LSP Figma] Waiting ' . $retry_after . 's before retry...');
                    sleep($retry_after);
                    continue;
                } else {
                    // Don't wait more than 10s, just return the error
                    return new \WP_Error('figma_rate_limited', 
                        'Figma rate limit exceeded. Please wait ' . $retry_after . ' seconds.',
                        ['retry_after' => $retry_after]
                    );
                }
            }
            
            // Handle other errors
            if ($code !== 200) {
                $error_data = json_decode($body, true);
                $figma_error = $error_data['message'] ?? $error_data['err'] ?? 'Unknown error';
                
                if ($code === 403) {
                    return new \WP_Error('figma_api_error', 
                        'Figma API access denied (403). OAuth scopes may need updating. Try disconnecting and reconnecting Figma. Error: ' . $figma_error
                    );
                }
                if ($code === 404) {
                    return new \WP_Error('figma_api_error', 
                        'File not found (404). Check the file URL and your access permissions.'
                    );
                }
                
                return new \WP_Error('figma_api_error', 'Figma API error ' . $code . ': ' . $figma_error);
            }
            
            // Success!
            return json_decode($body, true);
        }
        
        return new \WP_Error('figma_api_error', 'Max retries exceeded');
    }

    /**
     * Check if currently rate limited
     */
    public static function is_rate_limited() {
        $cooldown_until = get_transient('lsp_figma_rate_limit_until');
        return $cooldown_until && time() < $cooldown_until;
    }

    /**
     * Get seconds until rate limit expires (capped at 5 min)
     */
    public static function rate_limit_remaining() {
        $cooldown_until = get_transient('lsp_figma_rate_limit_until');
        if (!$cooldown_until) return 0;
        $remaining = max(0, $cooldown_until - time());
        // Cap at 5 minutes - if higher, something is wrong
        return min($remaining, 300);
    }
    
    /**
     * Clear rate limit cooldown (for debugging/reset)
     */
    public static function clear_rate_limit() {
        delete_transient('lsp_figma_rate_limit_until');
        Logger::debug('[LSP Figma] Rate limit cooldown cleared');
    }

    /**
     * Check if Figma is connected
     */
    public static function is_connected() {
        $o = Admin::get_opt();
        return !empty($o['figma_broker_token_enc']);
    }

    /**
     * Get Figma auth URL (starts OAuth flow via broker)
     */
    public static function auth_url() {
        $state  = wp_create_nonce('lightsync_figma_oauth_state');
        $site   = home_url();
        $return = admin_url('admin.php?page=lightsyncpro&tab=source');
        
        return self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_figma_connect_start'
             . '&site='   . rawurlencode($site)
             . '&state='  . rawurlencode($state)
             . '&return=' . rawurlencode($return);
    }

    /**
     * Handle the connection callback (?lsp_figma_connected=1)
     * Pick up the broker token from the broker
     */
    public static function handle_connection_callback() {
        if (!isset($_GET['lsp_figma_connected']) || $_GET['lsp_figma_connected'] !== '1') {
            return;
        }

        Logger::debug('[LSP Figma] Connection callback triggered');

        // Get state from callback
        $state = isset($_GET['state']) ? sanitize_text_field($_GET['state']) : '';

        // Pick up the broker token
        $pickup_url = self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_figma_install_pickup'
                    . '&state=' . rawurlencode($state);

        Logger::debug('[LSP Figma] Calling pickup URL: ' . $pickup_url);

        $response = wp_remote_get($pickup_url, [
            'timeout' => 20,
            'headers' => ['Accept' => 'application/json'],
        ]);

        if (is_wp_error($response)) {
            Logger::debug('[LSP Figma] Pickup request failed: ' . $response->get_error_message());
            add_action('admin_notices', function() use ($response) {
                echo '<div class="notice notice-error"><p>Figma connection failed: ' . esc_html($response->get_error_message()) . '</p></div>';
            });
            return;
        }

        $body = wp_remote_retrieve_body($response);
        Logger::debug('[LSP Figma] Pickup response: ' . "\n" . $body);
        $data = json_decode($body, true);

        if (empty($data['success']) || empty($data['data']['broker_token'])) {
            $error = $data['data']['error'] ?? 'not ready or expired';
            Logger::debug('[LSP Figma] Pickup failed: ' . $error);
            add_action('admin_notices', function() use ($error) {
                echo '<div class="notice notice-error"><p>Figma connection failed: ' . esc_html($error) . '</p></div>';
            });
            return;
        }

        // Encrypt and store the broker token
        $broker_token = $data['data']['broker_token'];
        $encrypted = Crypto::enc($broker_token);

        Admin::set_opt([
            'figma_broker_token_enc' => $encrypted,
            'figma_connected_at'     => time(),
            'figma_access_token'     => '',  // Clear any stale token
            'figma_token_expires'    => 0,
        ]);

        Logger::debug('[LSP Figma] Successfully connected and stored broker token');

        // Show success notice
        add_action('admin_notices', function() {
            echo '<div class="notice notice-success"><p>✓ Successfully connected to Figma!</p></div>';
        });
    }

    /**
     * Ensure access token is valid, refresh if needed
     */
    public static function ensure_token() {
        $o   = Admin::get_opt();
        $exp = (int)($o['figma_token_expires'] ?? 0);
        $now = time();

        Logger::debug('[LSP Figma] ensure_token: exp=' . $exp . ' now=' . $now . ' diff=' . ($exp - $now));

        // If we have a valid token with >60s remaining, we're good
        if (!empty($o['figma_access_token']) && ($exp - $now) > 60) {
            Logger::debug('[LSP Figma] ensure_token: using cached token');
            return true;
        }

        Logger::debug('[LSP Figma] ensure_token: need to refresh');

        // Need to refresh from broker
        $enc = $o['figma_broker_token_enc'] ?? '';
        if (!$enc) {
            return new \WP_Error('no_broker', 'Figma not connected. Click Connect to Figma.');
        }

        $broker = Crypto::dec($enc);
        if (!$broker) {
            return new \WP_Error('bad_broker', 'Stored Figma broker token not readable.');
        }

        $resp = wp_remote_post(
            self::BROKER_URL . '/wp-admin/admin-ajax.php?action=lsp_figma_token_refresh',
            [
                'timeout' => 20,
                'headers' => [
                    'Authorization' => 'Bearer ' . $broker,
                    'Accept'        => 'application/json',
                ],
            ]
        );

        if (is_wp_error($resp)) {
            return $resp;
        }

        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);
        $data = json_decode($body, true);

        Logger::debug('[LSP Figma] ensure_token: broker response code=' . $code);

        $ok      = is_array($data) && array_key_exists('success', $data) ? (bool)$data['success'] : true;
        $payload = (is_array($data) && array_key_exists('data', $data)) ? $data['data'] : $data;

        if (!$ok) {
            $msg = (is_array($payload) && !empty($payload['error'])) ? $payload['error'] : 'broker error';
            return new \WP_Error('no_access', 'Figma broker refused: ' . $msg);
        }

        if (empty($payload['access_token'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                Logger::debug('[LSP Figma] broker refresh code=' . $code . ' body=' . $body);
            }
            return new \WP_Error('no_access', 'Could not obtain Figma access token from broker');
        }

        $new_token = $payload['access_token'];
        Logger::debug('[LSP Figma] ensure_token: got new token len=' . strlen($new_token));

        // Store with cushion for early refresh
        $expires_in = (int)($payload['expires_in'] ?? 3600);
        $cushion    = 30;
        Admin::set_opt([
            'figma_access_token'  => $new_token,
            'figma_token_expires' => $now + max(300, $expires_in - $cushion),
        ]);

        return true;
    }

    /**
     * Proactively refresh when <5 minutes remain
     */
    public static function maybe_refresh() {
        $o   = Admin::get_opt();
        $exp = (int)($o['figma_token_expires'] ?? 0);
        if (!$exp) return self::ensure_token();
        if (($exp - time()) < 300) {
            return self::ensure_token();
        }
        return true;
    }

    /**
     * Disconnect Figma
     */
    public static function disconnect() {
        Admin::set_opt([
            'figma_access_token'      => '',
            'figma_token_expires'     => 0,
            'figma_broker_token_enc'  => '',
            'figma_connected_at'      => 0,
            'figma_selected_files'    => [],
        ]);

        // Clear cached data
        global $wpdb;
        $like = $wpdb->esc_like('_transient_lightsync_figma_') . '%';
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options}
                 WHERE option_name LIKE %s
                    OR option_name LIKE %s",
                $like,
                str_replace('_transient_', '_transient_timeout_', $like)
            )
        );

        Logger::debug('[LSP Figma] Disconnected – tokens cleared + cache purged');
    }

    /**
     * Get headers for Figma API calls
     */
    public static function headers() {
        $o = Admin::get_opt();
        return [
            'Authorization' => 'Bearer ' . ($o['figma_access_token'] ?? ''),
            'Accept'        => 'application/json',
        ];
    }

    /**
     * Schedule token guard cron
     */
    public static function ensure_guard_scheduled() {
        if (!self::is_connected()) return;
        
        if (!wp_next_scheduled(self::CRON_FIGMA_TOKEN_GUARD)) {
            wp_schedule_event(time() + 120, 'hourly', self::CRON_FIGMA_TOKEN_GUARD);
        }
    }

    /**
     * Cron handler for token refresh
     */
    public static function cron_token_guard() {
        $ok = self::maybe_refresh();
        if (is_wp_error($ok) && defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('[LSP Figma] token_guard failed: ' . $ok->get_error_message());
        }
    }

    /* ----------------------------------------------------------------
     * Figma API Methods
     * ---------------------------------------------------------------- */

    /**
     * Get current user info
     */
    public static function get_me() {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $o = Admin::get_opt();
        $token = $o['figma_access_token'] ?? '';
        Logger::debug('[LSP Figma] get_me token length: ' . strlen($token));

        $response = wp_remote_get(self::API_BASE . '/me', [
            'timeout' => 30,
            'headers' => self::headers(),
        ]);

        if (is_wp_error($response)) {
            Logger::debug('[LSP Figma] get_me WP error: ' . $response->get_error_message());
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        Logger::debug('[LSP Figma] get_me response: ' . $code . ' - ' . substr($body, 0, 500));

        if ($code !== 200) {
            $error_data = json_decode($body, true);
            $figma_error = $error_data['message'] ?? $error_data['err'] ?? 'Unknown error';
            
            if ($code === 403) {
                return new \WP_Error('figma_api_error', 
                    'Figma API access denied (403). The OAuth token may be missing required scopes. ' .
                    'Try disconnecting and reconnecting Figma. Figma says: ' . $figma_error
                );
            }
            
            return new \WP_Error('figma_api_error', 'Figma API error ' . $code . ': ' . $figma_error);
        }

        return json_decode($body, true);
    }

    /**
     * Get user's recent files (via team projects)
     * Note: Figma doesn't have a direct "list all files" endpoint
     * We'll use the files the user has access to via projects
     */
    public static function get_team_projects($team_id) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $response = wp_remote_get(self::API_BASE . '/teams/' . $team_id . '/projects', [
            'timeout' => 30,
            'headers' => self::headers(),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Figma] get_team_projects failed: ' . $code . ' - ' . $body);
            return new \WP_Error('figma_api_error', 'Figma API error: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Get files in a project
     */
    public static function get_project_files($project_id) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $response = wp_remote_get(self::API_BASE . '/projects/' . $project_id . '/files', [
            'timeout' => 30,
            'headers' => self::headers(),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Figma] get_project_files failed: ' . $code . ' - ' . $body);
            return new \WP_Error('figma_api_error', 'Figma API error: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Get a specific file's structure (pages, frames, components)
     */
    public static function get_file($file_key, $depth = 2) {
        $endpoint = '/files/' . $file_key . '?depth=' . (int)$depth;
        Logger::debug('[LSP Figma] get_file requesting depth ' . $depth);
        return self::api_request($endpoint, ['timeout' => 60]);
    }

    /**
     * Get file metadata only (faster than full file)
     */
    public static function get_file_meta($file_key) {
        $endpoint = '/files/' . $file_key . '/meta';
        return self::api_request($endpoint, ['timeout' => 30]);
    }

    /**
     * Export frames/nodes as images
     * 
     * @param string $file_key The Figma file key
     * @param array $node_ids Array of node IDs to export
     * @param string $format png, jpg, svg, or pdf
     * @param float $scale Export scale (1 = 1x, 2 = 2x, etc.)
     * @return array|WP_Error Map of node_id => image_url
     */
    public static function export_images($file_key, $node_ids, $format = 'png', $scale = 1) {
        // Validate format
        $valid_formats = ['png', 'jpg', 'svg', 'pdf'];
        if (!in_array($format, $valid_formats, true)) {
            $format = 'png';
        }

        // Build endpoint with node IDs
        $ids = implode(',', array_map('sanitize_text_field', (array)$node_ids));
        $endpoint = '/images/' . $file_key . '?' . http_build_query([
            'ids'    => $ids,
            'format' => $format,
            'scale'  => min(4, max(0.01, (float)$scale)), // Figma limits: 0.01 to 4
        ]);

        $data = self::api_request($endpoint, ['timeout' => 120]); // Image generation can be slow
        
        if (is_wp_error($data)) {
            return $data;
        }

        if (!empty($data['err'])) {
            return new \WP_Error('figma_export_error', 'Figma export error: ' . $data['err']);
        }

        // Return the images map (node_id => url)
        // Note: URLs expire after 30 days
        return $data['images'] ?? [];
    }

    /**
     * Get image fills (uploaded images in a file)
     */
    public static function get_image_fills($file_key) {
        $token_ok = self::ensure_token();
        if (is_wp_error($token_ok)) {
            return $token_ok;
        }

        $response = wp_remote_get(self::API_BASE . '/files/' . $file_key . '/images', [
            'timeout' => 30,
            'headers' => self::headers(),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            Logger::debug('[LSP Figma] get_image_fills failed: ' . $code . ' - ' . $body);
            return new \WP_Error('figma_api_error', 'Figma API error: ' . $code);
        }

        return json_decode($body, true);
    }

    /**
     * Extract frames from a file for selection UI
     * Returns simplified list: id, name, page_name, type
     * 
     * @param string $file_key The Figma file key
     * @param bool $include_nested Whether to include nested elements within frames
     * @return array|WP_Error
     */
    public static function get_file_frames($file_key, $include_nested = false) {
        // Use depth 3 for nested elements to get more hierarchy
        $depth = $include_nested ? 3 : 2;
        $file = self::get_file($file_key, $depth);
        if (is_wp_error($file)) {
            return $file;
        }

        $frames = [];
        $document = $file['document'] ?? [];
        
        // Exportable node types
        $top_level_types = ['FRAME', 'COMPONENT', 'COMPONENT_SET'];
        $nested_types = ['FRAME', 'COMPONENT', 'COMPONENT_SET', 'INSTANCE', 'GROUP', 'VECTOR', 'TEXT', 'RECTANGLE', 'ELLIPSE', 'POLYGON', 'STAR', 'LINE', 'BOOLEAN_OPERATION'];
        
        // Iterate through pages (children of document)
        foreach (($document['children'] ?? []) as $page) {
            if ($page['type'] !== 'CANVAS') continue;
            
            $page_name = $page['name'] ?? 'Untitled Page';
            
            // Get frames within this page
            foreach (($page['children'] ?? []) as $node) {
                // Only include top-level FRAME, COMPONENT, and COMPONENT_SET types
                if (!in_array($node['type'], $top_level_types, true)) {
                    continue;
                }
                
                $frames[] = [
                    'id'        => $node['id'],
                    'name'      => $node['name'] ?? 'Untitled',
                    'page_name' => $page_name,
                    'page_id'   => $page['id'],
                    'type'      => $node['type'],
                    'parent'    => null,
                ];
                
                // If nested elements requested, traverse children
                if ($include_nested && !empty($node['children'])) {
                    self::collect_nested_nodes($node['children'], $frames, $page_name, $page['id'], $node['name'], $nested_types);
                }
            }
        }

        return [
            'file_name'     => $file['name'] ?? 'Untitled',
            'last_modified' => $file['lastModified'] ?? null,
            'thumbnail'     => $file['thumbnailUrl'] ?? null,
            'frames'        => $frames,
        ];
    }

    /**
     * Recursively collect nested nodes for export
     */
    private static function collect_nested_nodes($children, &$frames, $page_name, $page_id, $parent_name, $allowed_types, $depth = 0) {
        // Limit depth to prevent excessive nesting
        if ($depth > 3) return;
        
        foreach ($children as $node) {
            if (!in_array($node['type'], $allowed_types, true)) {
                continue;
            }
            
            // Skip very small or unnamed nodes that are likely not useful
            $name = $node['name'] ?? '';
            if (empty($name) || $name === 'Rectangle' || $name === 'Ellipse') {
                // Check if it has a meaningful name
                if (strlen($name) < 2) continue;
            }
            
            $frames[] = [
                'id'        => $node['id'],
                'name'      => $name ?: 'Untitled',
                'page_name' => $page_name,
                'page_id'   => $page_id,
                'type'      => $node['type'],
                'parent'    => $parent_name,
            ];
            
            // Recurse into children
            if (!empty($node['children'])) {
                self::collect_nested_nodes($node['children'], $frames, $page_name, $page_id, $name, $allowed_types, $depth + 1);
            }
        }
    }

    /**
     * Parse file key from Figma URL
     * Supports: figma.com/file/KEY/..., figma.com/design/KEY/...
     * Note: Buzz files not yet supported by Figma API
     */
    public static function parse_file_key($url_or_key) {
        // Already a key (no slashes, reasonable length)
        if (!str_contains($url_or_key, '/') && strlen($url_or_key) > 10 && strlen($url_or_key) < 50) {
            return $url_or_key;
        }

        // Parse from URL - supports design and file
        if (preg_match('#figma\.com/(file|design)/([a-zA-Z0-9]+)#', $url_or_key, $m)) {
            return $m[2];
        }

        return null;
    }
}
