<?php
/**
 * LightSync Pro - Centralized Mapping System
 * 
 * Single source of truth for all sync mappings (WordPress).
 * ALL code must use this class for mapping lookups - never query directly.
 * 
 * This prevents bugs like:
 * - Wrong table names
 * - Mismatched meta keys
 * - Duplicate uploads from missed mappings
 * - Migration issues between free/pro versions
 *
 * @package LightSyncPro
 */

namespace LightSyncPro\Mapping;

use LightSyncPro\Admin\Admin;
use LightSyncPro\Util\Logger;

if (!defined('ABSPATH')) exit;

class Mapping {

    /* ========================================================================
     * CONSTANTS - Single Source of Truth
     * ======================================================================== */

    /**
     */

    /**
     * WordPress meta keys for source ID mapping (Pro version)
     * Key = source type, Value = meta key
     */
    const META_KEYS = [
        'lightroom'  => '_lightsync_asset_id',
        'figma'      => '_lightsync_figma_node_id',
        'canva'      => '_lightsync_canva_design_id',
        'dropbox'    => '_lightsync_dropbox_file_id',
        'openrouter' => '_lightsync_ai_asset_id',
    ];

    /**
     * Legacy meta keys from free versions (for migration)
     */
    const LEGACY_META_KEYS = [
        'lightroom' => '_lightsync_lr_asset_id',    // Some places used this
        'figma'     => '_lightsync_asset_id',       // Free Figma used generic key
        'canva'     => '_lightsync_asset_id',       // Free Canva used generic key
    ];

    /**
     * Additional meta keys stored per source
     */
    const SOURCE_META = [
        'lightroom' => [
            'catalog_id'     => '_lightsync_catalog_id',
            'album_id'       => '_lightsync_album_id',
            'album_name'     => '_lightsync_album_name',
            'rev'            => '_lightsync_rev',
            'source_updated' => '_lightsync_source_updated',
        ],
        'figma' => [
            'file_key'     => '_lightsync_figma_file_key',
            'file_version' => '_lightsync_figma_file_version',
        ],
        'canva' => [
            'rev'        => '_lightsync_rev',
            'page'       => '_lightsync_canva_page',
            'total_pages'=> '_lightsync_canva_total_pages',
        ],
        'dropbox' => [
            'path'     => '_lightsync_dropbox_path',
            'checksum' => '_lightsync_dropbox_checksum',
        ],
        'openrouter' => [
            'prompt'       => '_lightsync_ai_prompt',
            'model'        => '_lightsync_ai_model',
            'aspect_ratio' => '_lightsync_ai_aspect_ratio',
            'is_free'      => '_lightsync_ai_is_free',
            'version'      => '_lightsync_ai_version',
        ],
    ];

    /**
     * Common meta keys used across all sources
     */
    const COMMON_META = [
        'source'         => '_lightsync_source',
        'last_synced_at' => '_lightsync_last_synced_at',
        'last_sync_kind' => '_lightsync_last_sync_kind',
        'original_bytes' => '_lightsync_original_bytes',
        'optimized_bytes'=> '_lightsync_optimized_bytes',
    ];


    /* ========================================================================
     * WORDPRESS MAPPING - Lookup Methods
     * ======================================================================== */

    /**
     * Get WordPress attachment ID for a source item
     * Automatically checks both Pro and legacy (free) meta keys
     * 
     * @param string $source   Source type: lightroom, figma, canva, dropbox
     * @param string $source_id The source-specific ID
     * @return int|null Attachment ID or null if not found
     */
    public static function get_wp_attachment(string $source, string $source_id): ?int {
        global $wpdb;
        
        $source = strtolower(trim($source));
        $source_id = trim($source_id);
        
        if (empty($source_id)) {
            return null;
        }

        // 1. Check Pro meta key first
        $pro_key = self::META_KEYS[$source] ?? null;
        if ($pro_key) {
            $id = self::find_attachment_by_meta($pro_key, $source_id);
            if ($id) {
                return $id;
            }
        }

        // 2. Check legacy (free version) meta key
        $legacy_key = self::LEGACY_META_KEYS[$source] ?? null;
        if ($legacy_key && $legacy_key !== $pro_key) {
            $id = self::find_attachment_by_meta($legacy_key, $source_id);
            if ($id) {
                // Auto-migrate: Add Pro key for faster future lookups
                if ($pro_key) {
                    update_post_meta($id, $pro_key, $source_id);
                    Logger::debug("[LSP Mapping] Auto-migrated {$source} mapping for attachment #{$id}");
                }
                return $id;
            }
        }

        // 3. For Figma, also check by asset_key (file_key:node_id format)
        if ($source === 'figma' && strpos($source_id, ':') !== false) {
            // source_id might be the node_id, try finding by combined key
            $id = self::find_attachment_by_meta('_lightsync_asset_id', $source_id);
            if ($id) {
                return $id;
            }
        }

        return null;
    }

    /**
     * Check if an item is synced to WordPress
     */
    public static function is_synced_to_wp(string $source, string $source_id): bool {
        return self::get_wp_attachment($source, $source_id) !== null;
    }

    /**
     * Find attachment by meta key/value
     */
    private static function find_attachment_by_meta(string $meta_key, string $meta_value): ?int {
        global $wpdb;

        $id = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} 
             WHERE meta_key = %s AND meta_value = %s 
             LIMIT 1",
            $meta_key,
            $meta_value
        ));

        return $id ? (int) $id : null;
    }


    /* ========================================================================
     * SYNC TARGET METHODS
     * ======================================================================== */

            /* ========================================================================
     * COMBINED METHODS - Multi-destination Checks
     * ======================================================================== */

    /**
     * Get all destinations where an item is synced
     * 
     * @param string $source    Source type
     * @param string $source_id Source-specific ID
     * @return array Array of destinations: ['wp'] etc.r []
     */
    public static function get_sync_targets(string $source, string $source_id): array {
        $targets = [];

        if (self::is_synced_to_wp($source, $source_id)) {
            $targets[] = 'wp';
        }

        return $targets;
    }

    /**
     * Determine sync target string for auto-sync
     * Returns 'wp' or null if not synced
     * 
     * @param string $source        Source type
     * @param string $source_id     Source-specific ID
     * @param int|null $attachment_id Optional: Known WP attachment ID (optimization)
     * @return string|null
     */
    public static function determine_sync_target(string $source, string $source_id, ?int $attachment_id = null): ?string {
        $has_wp = $attachment_id !== null || self::is_synced_to_wp($source, $source_id);

        return $has_wp ? 'wp' : null;
    }

    /**
     * Check if an item is synced to ANY destination
     */
    public static function is_synced(string $source, string $source_id): bool {
        return self::is_synced_to_wp($source, $source_id);
    }


    /* ========================================================================
     * SAVE METHODS - Centralized Mapping Storage
     * ======================================================================== */

    /**
     * Save WordPress mapping for a synced item
     * 
     * @param string $source        Source type
     * @param string $source_id     Source-specific ID
     * @param int    $attachment_id WordPress attachment ID
     * @param array  $extra_meta    Additional meta to save (source-specific)
     */
    public static function save_wp_mapping(
        string $source, 
        string $source_id, 
        int $attachment_id, 
        array $extra_meta = []
    ): void {
        $source = strtolower(trim($source));
        
        // Save source ID with correct meta key
        $meta_key = self::META_KEYS[$source] ?? '_lightsync_asset_id';
        update_post_meta($attachment_id, $meta_key, $source_id);

        // Save source type
        update_post_meta($attachment_id, self::COMMON_META['source'], $source);

        // Save last synced timestamp
        update_post_meta($attachment_id, self::COMMON_META['last_synced_at'], gmdate('Y-m-d H:i:s'));

        // Save any extra meta
        foreach ($extra_meta as $key => $value) {
            // Check if it's a known source-specific key
            $source_keys = self::SOURCE_META[$source] ?? [];
            $meta_key = $source_keys[$key] ?? $key;
            
            // If key doesn't start with _, assume it needs the prefix
            if (strpos($meta_key, '_') !== 0) {
                $meta_key = '_lightsync_' . $meta_key;
            }
            
            update_post_meta($attachment_id, $meta_key, $value);
        }

        Logger::debug("[LSP Mapping] Saved WP mapping: {$source}/{$source_id} -> attachment #{$attachment_id}");
    }

        /* ========================================================================
     * MIGRATION METHODS - Free Version Support
     * ======================================================================== */

    /**
     * Check if there are legacy mappings that need migration
     */
    public static function needs_migration(): bool {
        global $wpdb;

        // Check for legacy Lightroom key
        $legacy_lr = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} 
             WHERE meta_key = '_lightsync_lr_asset_id'"
        );

        // Check for generic asset_id without source marker
        $generic = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} pm
             WHERE pm.meta_key = '_lightsync_asset_id'
             AND NOT EXISTS (
                 SELECT 1 FROM {$wpdb->postmeta} pm2 
                 WHERE pm2.post_id = pm.post_id 
                 AND pm2.meta_key = '_lightsync_source'
             )"
        );

        return ($legacy_lr > 0) || ($generic > 0);
    }

    /**
     * Migrate all legacy mappings to Pro format
     * 
     * @return array Migration stats
     */
    public static function migrate_all(): array {
        global $wpdb;

        $stats = [
            'lightroom' => 0,
            'figma'     => 0,
            'canva'     => 0,
            'total'     => 0,
            'errors'    => 0,
        ];

        // 1. Migrate _lightsync_lr_asset_id -> _lightsync_asset_id
        $legacy_lr = $wpdb->get_results(
            "SELECT post_id, meta_value FROM {$wpdb->postmeta} 
             WHERE meta_key = '_lightsync_lr_asset_id'"
        );

        foreach ($legacy_lr as $row) {
            $existing = get_post_meta($row->post_id, '_lightsync_asset_id', true);
            if (empty($existing)) {
                update_post_meta($row->post_id, '_lightsync_asset_id', $row->meta_value);
                update_post_meta($row->post_id, '_lightsync_source', 'lightroom');
                $stats['lightroom']++;
                $stats['total']++;
            }
        }

        // 2. Identify Figma items by _lightsync_figma_file_key presence
        $figma_items = $wpdb->get_results(
            "SELECT pm.post_id FROM {$wpdb->postmeta} pm
             INNER JOIN {$wpdb->postmeta} pm2 ON pm.post_id = pm2.post_id
             WHERE pm.meta_key = '_lightsync_asset_id'
             AND pm2.meta_key = '_lightsync_figma_file_key'
             AND NOT EXISTS (
                 SELECT 1 FROM {$wpdb->postmeta} pm3 
                 WHERE pm3.post_id = pm.post_id 
                 AND pm3.meta_key = '_lightsync_source'
             )"
        );

        foreach ($figma_items as $row) {
            update_post_meta($row->post_id, '_lightsync_source', 'figma');
            $stats['figma']++;
            $stats['total']++;
        }

        // 3. Identify Canva items by _lightsync_canva_design_id presence
        $canva_items = $wpdb->get_results(
            "SELECT pm.post_id FROM {$wpdb->postmeta} pm
             INNER JOIN {$wpdb->postmeta} pm2 ON pm.post_id = pm2.post_id
             WHERE pm.meta_key = '_lightsync_asset_id'
             AND pm2.meta_key = '_lightsync_canva_design_id'
             AND NOT EXISTS (
                 SELECT 1 FROM {$wpdb->postmeta} pm3 
                 WHERE pm3.post_id = pm.post_id 
                 AND pm3.meta_key = '_lightsync_source'
             )"
        );

        foreach ($canva_items as $row) {
            update_post_meta($row->post_id, '_lightsync_source', 'canva');
            $stats['canva']++;
            $stats['total']++;
        }

        // 4. Mark remaining items as Lightroom (default for old free version)
        $remaining = $wpdb->get_results(
            "SELECT pm.post_id FROM {$wpdb->postmeta} pm
             WHERE pm.meta_key = '_lightsync_asset_id'
             AND NOT EXISTS (
                 SELECT 1 FROM {$wpdb->postmeta} pm2 
                 WHERE pm2.post_id = pm.post_id 
                 AND pm2.meta_key = '_lightsync_source'
             )"
        );

        foreach ($remaining as $row) {
            update_post_meta($row->post_id, '_lightsync_source', 'lightroom');
            $stats['lightroom']++;
            $stats['total']++;
        }

        Logger::debug("[LSP Mapping] Migration complete", $stats);

        return $stats;
    }

    /**
     * Get migration status for display
     */
    public static function get_migration_status(): array {
        global $wpdb;

        // Count items by source
        $by_source = $wpdb->get_results(
            "SELECT meta_value as source, COUNT(*) as count 
             FROM {$wpdb->postmeta} 
             WHERE meta_key = '_lightsync_source'
             GROUP BY meta_value"
        );

        $sources = [];
        foreach ($by_source as $row) {
            $sources[$row->source] = (int) $row->count;
        }

        // Count legacy items
        $legacy = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} pm
             WHERE pm.meta_key = '_lightsync_asset_id'
             AND NOT EXISTS (
                 SELECT 1 FROM {$wpdb->postmeta} pm2 
                 WHERE pm2.post_id = pm.post_id 
                 AND pm2.meta_key = '_lightsync_source'
             )"
        );

        return [
            'sources'      => $sources,
            'legacy_count' => (int) $legacy,
            'needs_migration' => $legacy > 0,
        ];
    }


    /* ========================================================================
     * UTILITY METHODS
     * ======================================================================== */

                /**
     * Get the meta key for a source type
     */
    public static function get_meta_key(string $source): string {
        return self::META_KEYS[strtolower($source)] ?? '_lightsync_asset_id';
    }

    /**
     * Get checksum meta key for a source
     */
    public static function get_checksum_key(string $source): string {
        $source = strtolower($source);
        
        if ($source === 'dropbox') {
            return '_lightsync_dropbox_checksum';
        }
        
        return '_lightsync_rev';  // Default for Lightroom, Canva, Figma
    }

    /**
     * Delete all mappings for an attachment (when deleted from WP)
     */
    public static function delete_wp_mappings(int $attachment_id): void {
        // Get source info before deleting
        $source = get_post_meta($attachment_id, '_lightsync_source', true);
        $source_id = null;

        if ($source) {
            $meta_key = self::META_KEYS[$source] ?? '_lightsync_asset_id';
            $source_id = get_post_meta($attachment_id, $meta_key, true);
        }

        // Delete all LightSync meta
        global $wpdb;
        $wpdb->query($wpdb->prepare(
            "DELETE FROM {$wpdb->postmeta} 
             WHERE post_id = %d 
             AND meta_key LIKE '_lightsync_%'",
            $attachment_id
        ));

        Logger::debug("[LSP Mapping] Deleted WP mappings for attachment #{$attachment_id}");
    }

    /**
     * Get all synced items for a source type
     * 
     * @param string $source Source type
     * @return array Array of [attachment_id => source_id]
     */
    public static function get_all_wp_mappings(string $source): array {
        global $wpdb;
        
        $meta_key = self::META_KEYS[strtolower($source)] ?? '_lightsync_asset_id';

        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE meta_key = %s",
            $meta_key
        ));

        $mappings = [];
        foreach ($results as $row) {
            $mappings[(int) $row->post_id] = $row->meta_value;
        }

        return $mappings;
    }
}
