<?php
/**
 * Weekly Digest Email & Stats Tracking
 * 
 * Tracks sync activity by source and sends weekly summary emails
 */

namespace LightSyncPro\Admin;

if (!defined('ABSPATH')) exit;

class WeeklyDigest {

    const STATS_OPT = 'lightsync_weekly_stats';
    const CRON_HOOK = 'lightsync_weekly_digest_cron';

    /**
     * Initialize weekly digest
     */
    public static function init(): void {
        // Register cron schedule
        add_filter('cron_schedules', [__CLASS__, 'add_weekly_schedule']);
        
        // Schedule weekly cron if not scheduled
        if (!wp_next_scheduled(self::CRON_HOOK)) {
            // Schedule for Monday 9am local time
            $next_monday = strtotime('next monday 9:00am');
            wp_schedule_event($next_monday, 'weekly', self::CRON_HOOK);
        }
        
        // Hook for sending digest
        add_action(self::CRON_HOOK, [__CLASS__, 'send_weekly_digest']);
        
        // AJAX for test email
        add_action('wp_ajax_lsp_send_test_digest', [__CLASS__, 'ajax_send_test_digest']);
    }

    /**
     * Add weekly schedule to cron
     */
    public static function add_weekly_schedule(array $schedules): array {
        $schedules['weekly'] = [
            'interval' => WEEK_IN_SECONDS,
            'display'  => __('Once Weekly', 'lightsyncpro'),
        ];
        return $schedules;
    }

    /**
     * Track a sync event
     * 
     * @param string $source lightroom|figma|canva|dropbox
     * @param int $count Number of images synced
     * @param int $original_bytes Original file size
     * @param int $optimized_bytes Optimized file size
     * @param bool $is_autosync Whether this was an autosync
     */
    public static function track_sync(
        string $source, 
        int $count = 1, 
        int $original_bytes = 0, 
        int $optimized_bytes = 0,
        bool $is_autosync = false
    ): void {
        $stats = self::get_current_week_stats();
        $source = strtolower($source);
        
        // Ensure source exists
        if (!isset($stats['sources'][$source])) {
            $stats['sources'][$source] = [
                'synced' => 0,
                'bytes_original' => 0,
                'bytes_optimized' => 0,
            ];
        }
        
        // Update source stats
        $stats['sources'][$source]['synced'] += $count;
        $stats['sources'][$source]['bytes_original'] += $original_bytes;
        $stats['sources'][$source]['bytes_optimized'] += $optimized_bytes;
        
        // Update totals
        $stats['total_synced'] += $count;
        $stats['total_bytes_original'] += $original_bytes;
        $stats['total_bytes_optimized'] += $optimized_bytes;
        
        // Track autosyncs
        if ($is_autosync) {
            $stats['autosync_runs'] = ($stats['autosync_runs'] ?? 0) + 1;
        }
        
        self::save_stats($stats);
    }

    /**
     * Track a sync error
     */
    public static function track_error(string $source, string $message = ''): void {
        $stats = self::get_current_week_stats();
        $stats['errors'] = ($stats['errors'] ?? 0) + 1;
        
        // Store last few errors for debugging
        if (!isset($stats['error_log'])) {
            $stats['error_log'] = [];
        }
        $stats['error_log'][] = [
            'source' => $source,
            'message' => substr($message, 0, 200),
            'time' => current_time('mysql'),
        ];
        // Keep only last 10 errors
        $stats['error_log'] = array_slice($stats['error_log'], -10);
        
        self::save_stats($stats);
    }

    /**
     * Track Hub distribution
     */
    public static function track_hub_distribution(string $site_url, int $images, bool $success): void {
        $stats = self::get_current_week_stats();
        
        if (!isset($stats['hub'])) {
            $stats['hub'] = [
                'distributions' => 0,
                'total_images' => 0,
                'sites' => [],
                'failures' => 0,
            ];
        }
        
        $stats['hub']['distributions']++;
        $stats['hub']['total_images'] += $images;
        
        // Track per-site
        $site_key = sanitize_title($site_url);
        if (!isset($stats['hub']['sites'][$site_key])) {
            $stats['hub']['sites'][$site_key] = [
                'url' => $site_url,
                'images' => 0,
                'syncs' => 0,
                'failures' => 0,
            ];
        }
        $stats['hub']['sites'][$site_key]['images'] += $images;
        $stats['hub']['sites'][$site_key]['syncs']++;
        
        if (!$success) {
            $stats['hub']['failures']++;
            $stats['hub']['sites'][$site_key]['failures']++;
        }
        
        self::save_stats($stats);
    }

    /**
     * Track Hub result from lsp_hub_sync_asset response
     * 
     * @param array $hub_result Result from lsp_hub_sync_asset
     * @param string $source Source of the sync (canva, figma, dropbox, lightroom)
     */
    public static function track_hub_result(array $hub_result, string $source = ''): void {
        $synced = (int)($hub_result['synced'] ?? 0);
        $failed = (int)($hub_result['failed'] ?? 0);
        $sites_synced = $hub_result['sites_synced'] ?? [];
        
        if ($synced <= 0 && $failed <= 0) {
            return;
        }
        
        $stats = self::get_current_week_stats();
        
        if (!isset($stats['hub'])) {
            $stats['hub'] = [
                'distributions' => 0,
                'total_images' => 0,
                'sites' => [],
                'failures' => 0,
            ];
        }
        
        $stats['hub']['distributions']++;
        $stats['hub']['total_images'] += $synced;
        $stats['hub']['failures'] += $failed;
        
        // Track per-site if we have site details
        if (!empty($sites_synced) && is_array($sites_synced)) {
            foreach ($sites_synced as $site_url) {
                $site_key = sanitize_title($site_url);
                if (!isset($stats['hub']['sites'][$site_key])) {
                    $stats['hub']['sites'][$site_key] = [
                        'url' => $site_url,
                        'images' => 0,
                        'syncs' => 0,
                        'failures' => 0,
                    ];
                }
                $stats['hub']['sites'][$site_key]['images']++;
                $stats['hub']['sites'][$site_key]['syncs']++;
            }
        }
        
        self::save_stats($stats);
    }

    /**
     * Get current week's stats
     */
    public static function get_current_week_stats(): array {
        $stats = get_option(self::STATS_OPT, []);
        
        // Check if we need to start a new week
        $current_week_start = self::get_week_start();
        
        if (empty($stats) || ($stats['week_start'] ?? '') !== $current_week_start) {
            // Archive old stats if they exist
            if (!empty($stats) && isset($stats['week_start'])) {
                self::archive_week($stats);
            }
            
            // Start fresh week
            $stats = self::create_empty_week($current_week_start);
        }
        
        return $stats;
    }

    /**
     * Get week start date (Monday)
     */
    private static function get_week_start(): string {
        return date('Y-m-d', strtotime('monday this week'));
    }

    /**
     * Create empty week stats structure
     */
    private static function create_empty_week(string $week_start): array {
        return [
            'week_start' => $week_start,
            'week_end' => date('Y-m-d', strtotime($week_start . ' +6 days')),
            'sources' => [
                'lightroom' => ['synced' => 0, 'bytes_original' => 0, 'bytes_optimized' => 0],
                'figma' => ['synced' => 0, 'bytes_original' => 0, 'bytes_optimized' => 0],
                'canva' => ['synced' => 0, 'bytes_original' => 0, 'bytes_optimized' => 0],
                'dropbox' => ['synced' => 0, 'bytes_original' => 0, 'bytes_optimized' => 0],
            ],
            'total_synced' => 0,
            'total_bytes_original' => 0,
            'total_bytes_optimized' => 0,
            'autosync_runs' => 0,
            'errors' => 0,
            'hub' => [
                'distributions' => 0,
                'total_images' => 0,
                'sites' => [],
                'failures' => 0,
            ],
        ];
    }

    /**
     * Archive a completed week
     */
    private static function archive_week(array $stats): void {
        $archive = get_option('lightsync_weekly_archive', []);
        $archive[] = $stats;
        // Keep last 12 weeks
        $archive = array_slice($archive, -12);
        update_option('lightsync_weekly_archive', $archive, false);
    }

    /**
     * Save current stats
     */
    private static function save_stats(array $stats): void {
        update_option(self::STATS_OPT, $stats, false);
    }

    /**
     * Calculate estimated time saved (in minutes)
     * 
     * Based on: ~2 min per image manual workflow
     */
    public static function calculate_time_saved(int $images_synced, int $sites_updated = 1): int {
        // Base: 2 min per image for export, rename, optimize, upload, metadata
        $base_time_per_image = 2;
        
        // Multiply by sites for Hub distributions
        $total_operations = $images_synced * max(1, $sites_updated);
        
        return $total_operations * $base_time_per_image;
    }

    /**
     * Format minutes into human readable
     */
    public static function format_time_saved(int $minutes): string {
        if ($minutes < 60) {
            return $minutes . ' min';
        }
        
        $hours = floor($minutes / 60);
        $mins = $minutes % 60;
        
        if ($hours < 24) {
            return $mins > 0 ? "{$hours}h {$mins}m" : "{$hours} hours";
        }
        
        $days = floor($hours / 24);
        $remaining_hours = $hours % 24;
        
        return $remaining_hours > 0 ? "{$days}d {$remaining_hours}h" : "{$days} days";
    }

    /**
     * Format bytes
     */
    public static function format_bytes(int $bytes): string {
        if ($bytes <= 0) return '0 B';
        
        $units = ['B', 'KB', 'MB', 'GB'];
        $pow = floor(log($bytes, 1024));
        $pow = min($pow, count($units) - 1);
        
        return round($bytes / pow(1024, $pow), 1) . ' ' . $units[$pow];
    }

    /**
     * Send weekly digest email
     */
    public static function send_weekly_digest(): void {
        $settings = Admin::get_opt();
        
        // Check if digest is enabled
        if (empty($settings['weekly_digest_enabled'])) {
            return;
        }
        
        // Get last week's stats (the one we just completed)
        $archive = get_option('lightsync_weekly_archive', []);
        if (empty($archive)) {
            return;
        }
        
        $stats = end($archive);
        if (empty($stats) || $stats['total_synced'] === 0) {
            return; // Don't send empty digest
        }
        
        // Get recipient
        $recipient = $settings['weekly_digest_email'] ?? get_option('admin_email');
        if (empty($recipient)) {
            return;
        }
        
        // Get brand config
        $brand = function_exists('lsp_get_brand') ? lsp_get_brand() : ['name' => 'LightSync Pro', 'is_enterprise' => false];
        
        // Build and send email
        $subject = self::build_subject($stats, $brand);
        $html_body = self::build_email_body($stats, $brand);
        $plain_body = self::build_plain_text_body($stats, $brand);
        
        // Better headers to avoid spam
        $site_domain = parse_url(home_url(), PHP_URL_HOST);
        $boundary = md5(time());
        
        $headers = [
            'MIME-Version: 1.0',
            'Content-Type: multipart/alternative; boundary="' . $boundary . '"',
            'From: ' . $brand['name'] . ' <wordpress@' . $site_domain . '>',
            'Reply-To: ' . get_option('admin_email'),
            'X-Mailer: ' . $brand['name'],
            'List-Unsubscribe: <' . admin_url('admin.php?page=lightsyncpro') . '>',
        ];
        
        // Build multipart message (plain + HTML)
        $message = "--{$boundary}\r\n";
        $message .= "Content-Type: text/plain; charset=UTF-8\r\n";
        $message .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
        $message .= $plain_body . "\r\n\r\n";
        $message .= "--{$boundary}\r\n";
        $message .= "Content-Type: text/html; charset=UTF-8\r\n";
        $message .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
        $message .= $html_body . "\r\n\r\n";
        $message .= "--{$boundary}--";
        
        wp_mail($recipient, $subject, $message, $headers);
        
        // Also send Hub email if applicable
        if (!empty($stats['hub']['distributions']) && !empty($settings['hub_digest_enabled'])) {
            self::send_hub_digest($stats, $settings);
        }
    }

    /**
     * Build email subject (no emoji - triggers spam filters)
     */
    private static function build_subject(array $stats, array $brand = null): string {
        $brand = $brand ?? ['name' => 'LightSync Pro'];
        $total = $stats['total_synced'] ?? 0;
        $time_saved = self::calculate_time_saved($total, count($stats['hub']['sites'] ?? []));
        $time_str = self::format_time_saved($time_saved);
        
        return $brand['name'] . " Weekly Summary: {$total} images synced";
    }

    /**
     * Build plain text email body (for multipart)
     */
    private static function build_plain_text_body(array $stats, array $brand = null): string {
        $brand = $brand ?? ['name' => 'LightSync Pro'];
        $site_name = get_bloginfo('name');
        $week_start = date('M j', strtotime($stats['week_start']));
        $week_end = date('M j, Y', strtotime($stats['week_end']));
        
        $total_synced = $stats['total_synced'] ?? 0;
        $bytes_original = $stats['total_bytes_original'] ?? 0;
        $bytes_optimized = $stats['total_bytes_optimized'] ?? 0;
        $savings_percent = $bytes_original > 0 ? round((($bytes_original - $bytes_optimized) / $bytes_original) * 100) : 0;
        
        $hub_sites = count($stats['hub']['sites'] ?? []);
        $time_saved = self::calculate_time_saved($total_synced, $hub_sites ?: 1);
        $time_str = self::format_time_saved($time_saved);
        
        $errors = $stats['errors'] ?? 0;
        $status = $errors > 0 ? "{$errors} error(s) occurred" : "All syncs successful";
        
        $text = strtoupper($brand['name']) . " - WEEKLY SUMMARY\n";
        $text .= "{$week_start} - {$week_end}\n";
        $text .= str_repeat("-", 40) . "\n\n";
        
        $text .= "OVERVIEW\n";
        $text .= "Images Synced: {$total_synced}\n";
        $text .= "Time Saved: {$time_str}\n";
        $text .= "Status: {$status}\n\n";
        
        $text .= "That's {$time_str} you didn't spend exporting, renaming,\n";
        $text .= "optimizing, uploading, and adding metadata";
        if ($hub_sites > 1) {
            $text .= " - across {$hub_sites} sites";
        }
        $text .= ".\n\n";
        
        $text .= "BY SOURCE\n";
        foreach ($stats['sources'] ?? [] as $source => $data) {
            if ($data['synced'] > 0) {
                $text .= "- " . ucfirst($source) . ": {$data['synced']}\n";
            }
        }
        $text .= "\n";
        
        if ($bytes_original > 0) {
            $text .= "STORAGE\n";
            $text .= self::format_bytes($bytes_original) . " -> " . self::format_bytes($bytes_optimized);
            $text .= " ({$savings_percent}% smaller)\n\n";
        }
        
        if ($hub_sites > 0) {
            $hub_total = $stats['hub']['total_images'] ?? 0;
            $text .= "HUB DISTRIBUTION\n";
            $text .= "{$hub_total} images distributed to {$hub_sites} sites\n\n";
        }
        
        $text .= str_repeat("-", 40) . "\n";
        $text .= $brand['name'] . " on {$site_name}\n";
        $text .= "Manage preferences: " . admin_url('admin.php?page=lightsyncpro') . "\n";
        
        return $text;
    }

    /**
     * Build email body HTML
     */
    private static function build_email_body(array $stats, $brand_or_is_hub = false): string {
        // Handle both old (bool $is_hub) and new (array $brand) signatures
        if (is_array($brand_or_is_hub)) {
            $brand = $brand_or_is_hub;
            $is_hub = false;
        } else {
            $brand = ['name' => 'LightSync Pro', 'is_enterprise' => false];
            $is_hub = (bool) $brand_or_is_hub;
        }
        
        $site_name = get_bloginfo('name');
        $week_start = date('M j', strtotime($stats['week_start']));
        $week_end = date('M j, Y', strtotime($stats['week_end']));
        
        $total_synced = $stats['total_synced'] ?? 0;
        $bytes_original = $stats['total_bytes_original'] ?? 0;
        $bytes_optimized = $stats['total_bytes_optimized'] ?? 0;
        $bytes_saved = max(0, $bytes_original - $bytes_optimized);
        $savings_percent = $bytes_original > 0 ? round(($bytes_saved / $bytes_original) * 100) : 0;
        
        $hub_sites = count($stats['hub']['sites'] ?? []);
        $time_saved = self::calculate_time_saved($total_synced, $hub_sites ?: 1);
        $time_str = self::format_time_saved($time_saved);
        
        $errors = $stats['errors'] ?? 0;
        $autosyncs = $stats['autosync_runs'] ?? 0;
        
        // Source breakdown with SVG icons
        $sources_html = '';
        foreach ($stats['sources'] ?? [] as $source => $data) {
            if ($data['synced'] > 0) {
                $icon_svg = self::get_source_svg($source);
                $sources_html .= "
                <tr>
                    <td style='padding:12px 16px;border-bottom:1px solid #f1f5f9;vertical-align:middle;'>
                        <table cellpadding='0' cellspacing='0' border='0'><tr>
                            <td style='width:24px;vertical-align:middle;'>{$icon_svg}</td>
                            <td style='padding-left:10px;font-size:14px;color:#334155;'>" . ucfirst($source) . "</td>
                        </tr></table>
                    </td>
                    <td style='padding:12px 16px;border-bottom:1px solid #f1f5f9;text-align:right;font-weight:600;font-size:14px;color:#1e293b;'>{$data['synced']}</td>
                </tr>";
            }
        }
        
        if (empty($sources_html)) {
            $sources_html = "<tr><td colspan='2' style='padding:16px;color:#64748b;text-align:center;font-size:14px;'>No syncs this week</td></tr>";
        }

        // Status
        $status_bg = $errors > 0 ? '#fef2f2' : '#f0fdf4';
        $status_border = $errors > 0 ? '#fecaca' : '#bbf7d0';
        $status_color = $errors > 0 ? '#dc2626' : '#16a34a';
        $status_text = $errors > 0 ? "{$errors} sync error(s) this week" : 'All syncs completed successfully';
        $status_icon = $errors > 0 ? self::get_svg_icon('alert') : self::get_svg_icon('check');

        // Brand-aware logo URL (external SVG links work in email, just not inline SVG)
        $logo_file = $brand['is_enterprise'] ? 'syncific-logo-white.svg' : 'lsp-logo-white.svg';
        $logo_url = plugins_url('assets/' . $logo_file, dirname(dirname(__FILE__)));

        $html = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <title>' . esc_html($brand['name']) . ' Weekly Summary</title>
</head>
<body style="margin:0;padding:0;background-color:#f8fafc;font-family:system-ui,-apple-system,BlinkMacSystemFont,Segoe UI,Roboto,sans-serif;">
    
    <!-- Wrapper Table -->
    <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color:#f8fafc;">
        <tr>
            <td align="center" style="padding:32px 16px;">
                
                <!-- Email Container -->
                <table width="100%" cellpadding="0" cellspacing="0" border="0" style="max-width:560px;background-color:#ffffff;border-radius:12px;overflow:hidden;box-shadow:0 1px 3px rgba(0,0,0,0.1);">
                    
                    <!-- Header -->
                    <tr>
                        <td style="background:#ff5757;background:linear-gradient(135deg,#ff5757,#2563eb);padding:28px 32px;text-align:center;">
                            <img src="' . esc_url($logo_url) . '" alt="' . esc_attr($brand['name']) . '" width="180" height="45" style="display:block;margin:0 auto;max-width:100%;height:auto;">
                            <p style="margin:16px 0 0 0;color:rgba(255,255,255,0.9);font-size:13px;">Weekly Summary</p>
                        </td>
                    </tr>
                    
                    <!-- Date Range -->
                    <tr>
                        <td style="padding:20px 32px 0 32px;text-align:center;">
                            <p style="margin:0;color:#64748b;font-size:13px;">' . $week_start . ' — ' . $week_end . '</p>
                        </td>
                    </tr>
                    
                    <!-- Main Stats -->
                    <tr>
                        <td style="padding:24px 32px;">
                            <table width="100%" cellpadding="0" cellspacing="0" border="0">
                                <tr>
                                    <td width="50%" style="text-align:center;padding:16px 8px;border-right:1px solid #e2e8f0;">
                                        <div style="font-size:32px;font-weight:700;color:#6366f1;line-height:1;">' . $total_synced . '</div>
                                        <div style="font-size:12px;color:#64748b;margin-top:6px;text-transform:uppercase;letter-spacing:0.5px;">Images Synced</div>
                                    </td>
                                    <td width="50%" style="text-align:center;padding:16px 8px;">
                                        <div style="font-size:32px;font-weight:700;color:#16a34a;line-height:1;">' . $time_str . '</div>
                                        <div style="font-size:12px;color:#64748b;margin-top:6px;text-transform:uppercase;letter-spacing:0.5px;">Time Saved</div>
                                    </td>
                                </tr>
                            </table>
                        </td>
                    </tr>
                    
                    <!-- Time Saved Context -->
                    <tr>
                        <td style="padding:0 32px 24px 32px;">
                            <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color:#f0fdf4;border:1px solid #bbf7d0;border-radius:8px;">
                                <tr>
                                    <td style="padding:14px 16px;">
                                        <p style="margin:0 0 8px 0;font-size:13px;font-weight:600;color:#166534;">That\'s ' . $time_str . ' you didn\'t spend:</p>
                                        <p style="margin:0;font-size:12px;color:#15803d;line-height:1.5;">Exporting files, renaming, optimizing for web, uploading to WordPress, and adding metadata — for each image' . ($hub_sites > 1 ? ', across ' . $hub_sites . ' sites' : '') . '.</p>
                                    </td>
                                </tr>
                            </table>
                        </td>
                    </tr>
                    
                    <!-- Status Banner -->
                    <tr>
                        <td style="padding:0 32px 24px 32px;">
                            <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color:' . $status_bg . ';border:1px solid ' . $status_border . ';border-radius:8px;">
                                <tr>
                                    <td style="padding:12px 16px;">
                                        <table cellpadding="0" cellspacing="0" border="0"><tr>
                                            <td style="width:20px;vertical-align:middle;">' . $status_icon . '</td>
                                            <td style="padding-left:10px;font-size:13px;color:' . $status_color . ';font-weight:500;">' . $status_text . '</td>
                                        </tr></table>
                                    </td>
                                </tr>
                            </table>
                        </td>
                    </tr>
                    
                    <!-- Source Breakdown -->
                    <tr>
                        <td style="padding:0 32px 24px 32px;">
                            <p style="margin:0 0 12px 0;font-size:11px;font-weight:600;color:#64748b;text-transform:uppercase;letter-spacing:0.5px;">By Source</p>
                            <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color:#f8fafc;border-radius:8px;overflow:hidden;">
                                ' . $sources_html . '
                            </table>
                        </td>
                    </tr>';
        
        // Storage stats (if any optimization occurred)
        if ($bytes_original > 0) {
            $html .= '
                    <!-- Storage Optimization -->
                    <tr>
                        <td style="padding:0 32px 24px 32px;">
                            <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color:#f8fafc;border-radius:8px;">
                                <tr>
                                    <td style="padding:16px;">
                                        <table cellpadding="0" cellspacing="0" border="0"><tr>
                                            <td style="width:20px;vertical-align:middle;">' . self::get_svg_icon('storage') . '</td>
                                            <td style="padding-left:10px;font-size:11px;font-weight:600;color:#64748b;text-transform:uppercase;letter-spacing:0.5px;">Storage Optimized</td>
                                        </tr></table>
                                        <p style="margin:12px 0 0 0;font-size:14px;color:#334155;">
                                            ' . self::format_bytes($bytes_original) . ' <span style="color:#94a3b8;">→</span> ' . self::format_bytes($bytes_optimized) . '
                                            <span style="display:inline-block;background-color:#16a34a;color:#ffffff;padding:2px 8px;border-radius:10px;font-size:11px;font-weight:600;margin-left:8px;">' . $savings_percent . '% smaller</span>
                                        </p>
                                    </td>
                                </tr>
                            </table>
                        </td>
                    </tr>';
        }
        
        // Hub section if applicable
        if ($hub_sites > 0) {
            $hub_distributions = $stats['hub']['distributions'] ?? 0;
            $hub_total_images = $stats['hub']['total_images'] ?? 0;
            
            $html .= '
                    <!-- Hub Distribution -->
                    <tr>
                        <td style="padding:0 32px 24px 32px;">
                            <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color:#eff6ff;border:1px solid #bfdbfe;border-radius:8px;">
                                <tr>
                                    <td style="padding:16px;">
                                        <table cellpadding="0" cellspacing="0" border="0"><tr>
                                            <td style="width:20px;vertical-align:middle;">' . self::get_svg_icon('hub') . '</td>
                                            <td style="padding-left:10px;font-size:11px;font-weight:600;color:#1e40af;text-transform:uppercase;letter-spacing:0.5px;">Hub Distribution</td>
                                        </tr></table>
                                        <table width="100%" cellpadding="0" cellspacing="0" border="0" style="margin-top:12px;">
                                            <tr>
                                                <td width="33%" style="text-align:center;">
                                                    <div style="font-size:24px;font-weight:700;color:#1e40af;">' . $hub_sites . '</div>
                                                    <div style="font-size:11px;color:#64748b;">Sites</div>
                                                </td>
                                                <td width="33%" style="text-align:center;">
                                                    <div style="font-size:24px;font-weight:700;color:#1e40af;">' . $hub_distributions . '</div>
                                                    <div style="font-size:11px;color:#64748b;">Syncs</div>
                                                </td>
                                                <td width="33%" style="text-align:center;">
                                                    <div style="font-size:24px;font-weight:700;color:#1e40af;">' . $hub_total_images . '</div>
                                                    <div style="font-size:11px;color:#64748b;">Delivered</div>
                                                </td>
                                            </tr>
                                        </table>
                                    </td>
                                </tr>
                            </table>
                        </td>
                    </tr>';
        }
        
        // Autosync info
        if ($autosyncs > 0) {
            $html .= '
                    <tr>
                        <td style="padding:0 32px 24px 32px;text-align:center;">
                            <p style="margin:0;padding:12px;background-color:#f8fafc;border-radius:8px;font-size:13px;color:#64748b;">
                                ' . self::get_svg_icon('auto') . ' ' . $autosyncs . ' automatic sync' . ($autosyncs > 1 ? 's' : '') . ' ran this week
                            </p>
                        </td>
                    </tr>';
        }

        $html .= '
                    <!-- Footer -->
                    <tr>
                        <td style="padding:24px 32px;background-color:#f8fafc;border-top:1px solid #e2e8f0;text-align:center;">
                            <p style="margin:0 0 8px 0;color:#64748b;font-size:12px;">
                                Powered by <strong style="color:#334155;">' . esc_html($brand['name']) . '</strong> on ' . esc_html($site_name) . '
                            </p>
                            <p style="margin:0;font-size:11px;">
                                <a href="' . esc_url(admin_url('admin.php?page=lightsyncpro')) . '" style="color:#6366f1;text-decoration:none;">Manage email preferences</a>
                            </p>
                        </td>
                    </tr>
                    
                </table>
                
            </td>
        </tr>
    </table>
    
</body>
</html>';

        return $html;
    }

    /**
     * Get SVG icon for source (email-compatible colored dot using table)
     */
    private static function get_source_svg(string $source): string {
        $colors = [
            'lightroom' => '#6366f1',
            'figma' => '#a855f7',
            'canva' => '#06b6d4',
            'dropbox' => '#3b82f6',
        ];
        $color = $colors[$source] ?? '#64748b';
        
        // Use table-based circle for maximum email client compatibility
        return '<table cellpadding="0" cellspacing="0" border="0" style="display:inline-table;"><tr><td style="width:20px;height:20px;border-radius:10px;background-color:' . $color . ';font-size:1px;line-height:1px;">&nbsp;</td></tr></table>';
    }

    /**
     * Get generic SVG icon (email-compatible)
     */
    private static function get_svg_icon(string $name): string {
        $icons = [
            'check' => '<table cellpadding="0" cellspacing="0" border="0" style="display:inline-table;"><tr><td style="width:18px;height:18px;border-radius:9px;background-color:#16a34a;color:#ffffff;font-size:12px;font-weight:bold;text-align:center;line-height:18px;font-family:Arial,sans-serif;">✓</td></tr></table>',
            'alert' => '<table cellpadding="0" cellspacing="0" border="0" style="display:inline-table;"><tr><td style="width:18px;height:18px;border-radius:9px;background-color:#dc2626;color:#ffffff;font-size:12px;font-weight:bold;text-align:center;line-height:18px;font-family:Arial,sans-serif;">!</td></tr></table>',
            'storage' => '<table cellpadding="0" cellspacing="0" border="0" style="display:inline-table;"><tr><td style="width:18px;height:18px;border-radius:4px;background-color:#e2e8f0;color:#64748b;font-size:11px;text-align:center;line-height:18px;">◉</td></tr></table>',
            'hub' => '<table cellpadding="0" cellspacing="0" border="0" style="display:inline-table;"><tr><td style="width:18px;height:18px;border-radius:9px;background-color:#1e40af;color:#ffffff;font-size:11px;text-align:center;line-height:18px;">⚡</td></tr></table>',
            'auto' => '<span style="color:#64748b;font-size:12px;">⚡</span>',
        ];
        return $icons[$name] ?? '';
    }

    /**
     * Get source icon (emoji fallback for plain text)
     */
    private static function get_source_icon(string $source): string {
        $icons = [
            'lightroom' => 'Lr',
            'figma' => 'Fg',
            'canva' => 'Cv',
            'dropbox' => 'Db',
        ];
        return $icons[$source] ?? '--';
    }

    /**
     * Send Hub-specific digest (enterprise)
     */
    private static function send_hub_digest(array $stats, array $settings): void {
        $recipient = $settings['hub_digest_email'] ?? $settings['weekly_digest_email'] ?? get_option('admin_email');
        
        $hub = $stats['hub'] ?? [];
        $sites_count = count($hub['sites'] ?? []);
        $total_images = $hub['total_images'] ?? 0;
        
        $subject = "Syncific Report: {$total_images} images to {$sites_count} sites";
        $body = self::build_hub_email_body($stats);
        
        $site_domain = parse_url(home_url(), PHP_URL_HOST);
        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            'From: Syncific <wordpress@' . $site_domain . '>',
            'Reply-To: ' . get_option('admin_email'),
            'List-Unsubscribe: <' . admin_url('admin.php?page=lightsyncpro') . '>',
        ];
        
        wp_mail($recipient, $subject, $body, $headers);
    }

    /**
     * Build Hub-specific email body
     */
    private static function build_hub_email_body(array $stats): string {
        $site_name = get_bloginfo('name');
        $week_start = date('M j', strtotime($stats['week_start']));
        $week_end = date('M j, Y', strtotime($stats['week_end']));
        
        $hub = $stats['hub'] ?? [];
        $sites = $hub['sites'] ?? [];
        $total_distributions = $hub['distributions'] ?? 0;
        $total_images = $hub['total_images'] ?? 0;
        $failures = $hub['failures'] ?? 0;
        
        // Time saved calculation for Hub
        $time_saved = self::calculate_time_saved($stats['total_synced'] ?? 0, count($sites));
        $time_str = self::format_time_saved($time_saved);
        
        // Sort sites by images synced
        uasort($sites, function($a, $b) {
            return ($b['images'] ?? 0) - ($a['images'] ?? 0);
        });
        
        // Build sites table
        $sites_html = '';
        foreach ($sites as $site_data) {
            $url = esc_html($site_data['url'] ?? 'Unknown');
            $images = $site_data['images'] ?? 0;
            $syncs = $site_data['syncs'] ?? 0;
            $site_failures = $site_data['failures'] ?? 0;
            
            $status_bg = $site_failures > 0 ? '#fef2f2' : '#f0fdf4';
            $status_color = $site_failures > 0 ? '#dc2626' : '#16a34a';
            $status_text = $site_failures > 0 ? $site_failures . ' failed' : 'OK';
            
            $sites_html .= '
                <tr>
                    <td style="padding:12px 16px;border-bottom:1px solid #f1f5f9;">
                        <div style="font-weight:600;color:#334155;font-size:14px;">' . $url . '</div>
                        <div style="font-size:12px;color:#94a3b8;">' . $syncs . ' sync(s)</div>
                    </td>
                    <td style="padding:12px 16px;border-bottom:1px solid #f1f5f9;text-align:center;font-weight:600;font-size:14px;color:#1e293b;">' . $images . '</td>
                    <td style="padding:12px 16px;border-bottom:1px solid #f1f5f9;text-align:right;">
                        <span style="background:' . $status_bg . ';color:' . $status_color . ';padding:3px 10px;border-radius:10px;font-size:11px;font-weight:500;">' . $status_text . '</span>
                    </td>
                </tr>';
        }

        $status_bg = $failures > 0 ? '#fef2f2' : '#f0fdf4';
        $status_border = $failures > 0 ? '#fecaca' : '#bbf7d0';
        $status_color = $failures > 0 ? '#dc2626' : '#16a34a';
        $status_text = $failures > 0 ? "{$failures} distribution(s) failed" : 'All distributions successful';
        $status_icon = $failures > 0 ? self::get_svg_icon('alert') : self::get_svg_icon('check');

        // Hub is Enterprise-only, so always use Syncific branding
        $logo_url = plugins_url('assets/syncific-logo-white.svg', dirname(dirname(__FILE__)));

        $html = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Syncific Distribution Report</title>
</head>
<body style="margin:0;padding:0;background-color:#f8fafc;font-family:system-ui,-apple-system,BlinkMacSystemFont,Segoe UI,Roboto,sans-serif;">
    
    <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color:#f8fafc;">
        <tr>
            <td align="center" style="padding:32px 16px;">
                
                <table width="100%" cellpadding="0" cellspacing="0" border="0" style="max-width:600px;background-color:#ffffff;border-radius:12px;overflow:hidden;box-shadow:0 1px 3px rgba(0,0,0,0.1);">
                    
                    <!-- Header -->
                    <tr>
                        <td style="background:#ff5757;background:linear-gradient(135deg,#ff5757,#2563eb);padding:28px 32px;text-align:center;">
                            <img src="' . esc_url($logo_url) . '" alt="Syncific" width="150" height="45" style="display:block;margin:0 auto;max-width:100%;height:auto;">
                            <p style="margin:16px 0 0 0;color:rgba(255,255,255,0.9);font-size:13px;">Distribution Report</p>
                        </td>
                    </tr>
                    
                    <!-- Date -->
                    <tr>
                        <td style="padding:20px 32px 0 32px;text-align:center;">
                            <p style="margin:0;color:#64748b;font-size:13px;">' . $week_start . ' — ' . $week_end . '</p>
                        </td>
                    </tr>
                    
                    <!-- Main Stats -->
                    <tr>
                        <td style="padding:24px 32px;">
                            <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color:#eff6ff;border-radius:8px;">
                                <tr>
                                    <td width="33%" style="text-align:center;padding:20px 8px;">
                                        <div style="font-size:28px;font-weight:700;color:#1e40af;line-height:1;">' . count($sites) . '</div>
                                        <div style="font-size:11px;color:#64748b;margin-top:6px;text-transform:uppercase;">Sites</div>
                                    </td>
                                    <td width="33%" style="text-align:center;padding:20px 8px;border-left:1px solid #bfdbfe;border-right:1px solid #bfdbfe;">
                                        <div style="font-size:28px;font-weight:700;color:#1e40af;line-height:1;">' . $total_images . '</div>
                                        <div style="font-size:11px;color:#64748b;margin-top:6px;text-transform:uppercase;">Delivered</div>
                                    </td>
                                    <td width="33%" style="text-align:center;padding:20px 8px;">
                                        <div style="font-size:28px;font-weight:700;color:#16a34a;line-height:1;">' . $time_str . '</div>
                                        <div style="font-size:11px;color:#64748b;margin-top:6px;text-transform:uppercase;">Saved</div>
                                    </td>
                                </tr>
                            </table>
                        </td>
                    </tr>
                    
                    <!-- Time Saved Context -->
                    <tr>
                        <td style="padding:0 32px 24px 32px;">
                            <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color:#f0fdf4;border:1px solid #bbf7d0;border-radius:8px;">
                                <tr>
                                    <td style="padding:14px 16px;">
                                        <p style="margin:0 0 8px 0;font-size:13px;font-weight:600;color:#166534;">That\'s ' . $time_str . ' you didn\'t spend:</p>
                                        <p style="margin:0;font-size:12px;color:#15803d;line-height:1.5;">Without Syncific, you\'d manually export, optimize, upload, and add metadata for each image — repeated across all ' . count($sites) . ' sites. Syncific did it automatically.</p>
                                    </td>
                                </tr>
                            </table>
                        </td>
                    </tr>
                    
                    <!-- Status -->
                    <tr>
                        <td style="padding:0 32px 24px 32px;">
                            <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color:' . $status_bg . ';border:1px solid ' . $status_border . ';border-radius:8px;">
                                <tr>
                                    <td style="padding:12px 16px;">
                                        <table cellpadding="0" cellspacing="0" border="0"><tr>
                                            <td style="width:20px;vertical-align:middle;">' . $status_icon . '</td>
                                            <td style="padding-left:10px;font-size:13px;color:' . $status_color . ';font-weight:500;">' . $status_text . '</td>
                                        </tr></table>
                                    </td>
                                </tr>
                            </table>
                        </td>
                    </tr>
                    
                    <!-- Sites Table -->
                    <tr>
                        <td style="padding:0 32px 24px 32px;">
                            <p style="margin:0 0 12px 0;font-size:11px;font-weight:600;color:#64748b;text-transform:uppercase;letter-spacing:0.5px;">Site Breakdown</p>
                            <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color:#f8fafc;border-radius:8px;overflow:hidden;">
                                <tr style="background-color:#f1f5f9;">
                                    <td style="padding:10px 16px;font-size:11px;font-weight:600;color:#64748b;text-transform:uppercase;">Site</td>
                                    <td style="padding:10px 16px;font-size:11px;font-weight:600;color:#64748b;text-transform:uppercase;text-align:center;">Images</td>
                                    <td style="padding:10px 16px;font-size:11px;font-weight:600;color:#64748b;text-transform:uppercase;text-align:right;">Status</td>
                                </tr>
                                ' . $sites_html . '
                            </table>
                        </td>
                    </tr>
                    
                    <!-- Footer -->
                    <tr>
                        <td style="padding:24px 32px;background-color:#f8fafc;border-top:1px solid #e2e8f0;text-align:center;">
                            <p style="margin:0 0 8px 0;color:#64748b;font-size:12px;">
                                <strong style="color:#334155;">Syncific</strong> on ' . esc_html($site_name) . '
                            </p>
                            <p style="margin:0;font-size:11px;">
                                <a href="' . esc_url(admin_url('admin.php?page=lightsyncpro')) . '" style="color:#0369a1;text-decoration:none;">Manage preferences</a>
                            </p>
                        </td>
                    </tr>
                    
                </table>
                
            </td>
        </tr>
    </table>
    
</body>
</html>';

        return $html;
    }

    /**
     * AJAX: Send test digest
     */
    public static function ajax_send_test_digest(): void {
        check_ajax_referer('lightsyncpro_ajax_nonce', '_wpnonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized']);
        }
        
        // Get brand config
        $brand = function_exists('lsp_get_brand') ? lsp_get_brand() : ['name' => 'LightSync Pro', 'is_enterprise' => false];
        
        $recipient = sanitize_email($_POST['email'] ?? get_option('admin_email'));
        
        if (empty($recipient)) {
            $recipient = get_option('admin_email');
        }
        
        // Always use demo data for test emails
        $stats = [
            'week_start' => date('Y-m-d', strtotime('monday this week')),
            'week_end' => date('Y-m-d', strtotime('sunday this week')),
            'sources' => [
                'lightroom' => ['synced' => 32, 'bytes_original' => 85000000, 'bytes_optimized' => 23000000],
                'figma' => ['synced' => 15, 'bytes_original' => 25000000, 'bytes_optimized' => 7000000],
                'canva' => ['synced' => 8, 'bytes_original' => 17000000, 'bytes_optimized' => 4000000],
                'dropbox' => ['synced' => 0, 'bytes_original' => 0, 'bytes_optimized' => 0],
            ],
            'total_synced' => 55,
            'total_bytes_original' => 127000000,
            'total_bytes_optimized' => 34000000,
            'autosync_runs' => 7,
            'errors' => 0,
            'hub' => [
                'distributions' => 0,
                'total_images' => 0,
                'sites' => [],
                'failures' => 0,
            ],
        ];
        
        $subject = '[Test] ' . self::build_subject($stats, $brand);
        $html_body = self::build_email_body($stats, $brand);
        
        // Use simple HTML email (more compatible)
        $site_domain = parse_url(home_url(), PHP_URL_HOST);
        
        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $brand['name'] . ' <wordpress@' . $site_domain . '>',
        ];
        
        $sent = wp_mail($recipient, $subject, $html_body, $headers);
        
        if ($sent) {
            wp_send_json_success(['message' => 'Test email sent to ' . $recipient]);
        } else {
            wp_send_json_error(['error' => 'Failed to send email. Check your WordPress mail configuration.']);
        }
    }
}
