<?php

namespace LightSyncPro\Admin;

use LightSyncPro\OAuth\OAuth;
use LightSyncPro\Sync\Sync;
use LightSyncPro\Shopify\Shopify;
use LightSyncPro\Util\Crypto;
use LightSyncPro\Util\Logger;


class Admin {
    /** @var Admin|null Singleton instance */
    private static $instance = null;
    
    const OPT      = 'lightsyncpro_settings';
    const USAGE_OPT = 'lightsync_usage_data'; 
    const MENU     = 'lightsyncpro';
    const AJAX_NS  = 'lightsyncpro_ajax';
    const CRON     = 'lightsyncpro_cron_event';
    const CRON_ALBUM = 'lightsyncpro_sync_album';
    const CRON_RETRY_HOOK = 'lightsyncpro_retry_renditions';
    const CRON_CLEANUP = 'lightsync_cleanup_orphaned_crons';
    const TIMEOUT_CRON_TICK = 45.0;
    const TIMEOUT_BATCH = 55.0;
    const TIMEOUT_API_SHORT = 20;
    const TIMEOUT_API_LONG = 45;

    /**
     * Get singleton instance (creates one if needed for background processing)
     * @return Admin
     */
    public static function get_instance(): Admin {
        if (self::$instance === null) {
            // Create minimal instance for background processing
            self::$instance = new self();
        }
        return self::$instance;
    }

    public static function init(){
        $self = new self();
        self::$instance = $self;

        // One-time migration of usage data to separate option
        if (!get_option('lightsync_usage_migrated')) {
            $old = self::get_opt();
            if (!empty($old['usage_month_count']) || !empty($old['usage_total_count'])) {
                update_option(self::USAGE_OPT, [
                    'month_key'   => $old['usage_month_key'] ?? gmdate('Y-m'),
                    'month_count' => (int)($old['usage_month_count'] ?? 0),
                    'total_count' => (int)($old['usage_total_count'] ?? 0),
                ], false);
            }
            update_option('lightsync_usage_migrated', 1, false);
        }
        add_action('init',                     [$self,'ensure_rendition_cron_on_boot']);
        add_action('admin_menu',               [$self,'menu']);
        add_action('admin_init',               [$self,'register']);
        add_action('admin_init',               [$self,'handle_oauth_callback'], 5); // Early priority
        add_action('admin_enqueue_scripts',    [$self,'assets']);
        add_action('wp_ajax_lightsyncpro_disconnect', [$self, 'ajax_disconnect']);

        // AJAX
        add_action('wp_ajax_'.self::AJAX_NS.'_list',           [$self,'ajax_list']);
        add_action('wp_ajax_'.self::AJAX_NS.'_save_selection', [$self,'ajax_save_selection']);
        add_action('wp_ajax_'.self::AJAX_NS.'_estimate',       [$self,'ajax_estimate']);
        add_action('wp_ajax_'.self::AJAX_NS.'_batch',          [$self,'ajax_batch']);
        add_action('wp_ajax_'.self::AJAX_NS.'_save_broker',    [__CLASS__,'ajax_save_broker']);
        add_action('wp_ajax_'.self::AJAX_NS.'_diag',           [$self,'ajax_diag']);
        add_action('wp_ajax_'.self::AJAX_NS.'_save_options',   [$self,'ajax_save_options']);
        add_action('wp_ajax_'.self::AJAX_NS.'_save_digest',    [__CLASS__,'ajax_save_digest']);
        add_action('wp_ajax_lsp_send_test_digest',             ['\\LightSyncPro\\Admin\\WeeklyDigest', 'ajax_send_test_digest']);
        add_action('wp_ajax_'.self::AJAX_NS.'_background_status', [$self, 'ajax_background_status']);
        add_action('wp_ajax_'.self::AJAX_NS.'_cancel_background', [$self, 'ajax_cancel_background']);
        add_action('wp_ajax_'.self::AJAX_NS.'_get_album_assets', [$self, 'ajax_get_album_assets']);
        
        
        add_action('wp_ajax_lightsync_debug_sync', [$self, 'ajax_debug_sync']);
        
        
        
        
        
        

        // Canva AJAX handlers
        add_action('wp_ajax_lsp_canva_get_designs',   [$self, 'ajax_canva_get_designs']);
        add_action('wp_ajax_lsp_canva_disconnect',    [$self, 'ajax_canva_disconnect']);
        add_action('wp_ajax_lsp_canva_sync_designs',  [$self, 'ajax_canva_sync_designs']);
        add_action('wp_ajax_lsp_canva_sync_single',   [$self, 'ajax_canva_sync_single']);
        add_action('wp_ajax_lsp_canva_get_synced',    [$self, 'ajax_canva_get_synced']);
        add_action('wp_ajax_lsp_canva_save_target',   [$self, 'ajax_canva_save_target']);
        add_action('wp_ajax_lsp_canva_background_sync', [$self, 'ajax_canva_background_sync']);
        add_action('wp_ajax_lsp_canva_background_status', [$self, 'ajax_canva_background_status']);

        // Figma AJAX handlers
        add_action('wp_ajax_lsp_figma_get_files',     [$self, 'ajax_figma_get_files']);
        add_action('wp_ajax_lsp_figma_get_frames',    [$self, 'ajax_figma_get_frames']);
        add_action('wp_ajax_lsp_figma_disconnect',    [$self, 'ajax_figma_disconnect']);
        add_action('wp_ajax_lsp_figma_sync_frames',   [$self, 'ajax_figma_sync_frames']);
        add_action('wp_ajax_lsp_figma_save_target',   [$self, 'ajax_figma_save_target']);
        add_action('wp_ajax_lsp_figma_add_file',      [$self, 'ajax_figma_add_file']);
        add_action('wp_ajax_lsp_figma_remove_file',   [$self, 'ajax_figma_remove_file']);
        add_action('wp_ajax_lsp_figma_refresh_files', [$self, 'ajax_figma_refresh_files']);
        add_action('wp_ajax_lsp_figma_clear_rate_limit', [$self, 'ajax_figma_clear_rate_limit']);
        add_action('wp_ajax_lsp_figma_queue_sync', [$self, 'ajax_figma_queue_sync']);
        add_action('wp_ajax_lsp_figma_background_status', [$self, 'ajax_figma_background_status']);
        add_action('wp_ajax_lsp_figma_check_updates', [$self, 'ajax_figma_check_updates']);
        // Figma background processing cron
        add_action('lightsync_process_figma_queue', [$self, 'process_figma_background_queue']);
        // Canva background processing cron
        add_action('lightsync_process_canva_queue', [$self, 'process_canva_background_queue']);
        // Dropbox background processing cron
        add_action('lightsync_process_dropbox_queue', [$self, 'process_dropbox_background_queue']);
        // Figma browsing handlers
        add_action('wp_ajax_lsp_figma_get_teams',     [$self, 'ajax_figma_get_teams']);
        add_action('wp_ajax_lsp_figma_get_projects',  [$self, 'ajax_figma_get_projects']);
        add_action('wp_ajax_lsp_figma_browse_files',  [$self, 'ajax_figma_browse_files']);
        add_action('wp_ajax_lsp_figma_add_file_by_key', [$self, 'ajax_figma_add_file_by_key']);

        // Dropbox AJAX handlers
        add_action('wp_ajax_lsp_dropbox_disconnect', [$self, 'ajax_dropbox_disconnect']);

        // OpenRouter AI AJAX handlers
        add_action('wp_ajax_lsp_openrouter_disconnect', [$self, 'ajax_openrouter_disconnect']);

        // Shutterstock AJAX handlers
        add_action('wp_ajax_lsp_shutterstock_disconnect', [$self, 'ajax_shutterstock_disconnect']);
        add_action('wp_ajax_lsp_shutterstock_get_licenses', [$self, 'ajax_shutterstock_get_licenses']);
        add_action('wp_ajax_lsp_shutterstock_sync', [$self, 'ajax_shutterstock_sync']);
        add_action('wp_ajax_lsp_shutterstock_save_target', [$self, 'ajax_shutterstock_save_target']);
        add_action('wp_ajax_lsp_shutterstock_background_sync', [$self, 'ajax_shutterstock_background_sync']);
        add_action('wp_ajax_lsp_shutterstock_background_status', [$self, 'ajax_shutterstock_background_status']);

        // OpenRouter AJAX handlers

        // Shopify AJAX handlers
        add_action('wp_ajax_lightsync_shopify_status',        [$self, 'ajax_shopify_status']);
        add_action('wp_ajax_lightsync_shopify_save_settings', [$self, 'ajax_shopify_save_settings']);
        add_action('wp_ajax_lightsync_shopify_disconnect',    [$self, 'ajax_shopify_disconnect']);
        add_action('wp_ajax_lightsync_shopify_connect_start', [$self, 'ajax_shopify_connect_start']);
        add_action('wp_ajax_lightsync_shopify_reset_sync',    [$self, 'ajax_shopify_reset_sync']);
        add_action('wp_ajax_lsp_dropbox_list_folder', [$self, 'ajax_dropbox_list_folder']);
        add_action('wp_ajax_lsp_dropbox_get_synced', [$self, 'ajax_dropbox_get_synced']);
        add_action('wp_ajax_lsp_dropbox_save_target', [$self, 'ajax_dropbox_save_target']);
        add_action('wp_ajax_lsp_dropbox_get_thumbnail', [$self, 'ajax_dropbox_get_thumbnail']);
        add_action('wp_ajax_lsp_dropbox_sync_files', [$self, 'ajax_dropbox_sync_files']);
        add_action('wp_ajax_lsp_dropbox_sync_single', [$self, 'ajax_dropbox_sync_single']);
        add_action('wp_ajax_lsp_dropbox_background_status', [$self, 'ajax_dropbox_background_status']);
        add_action('wp_ajax_lsp_dropbox_process_next', [$self, 'ajax_dropbox_process_next']);

        // Auto-sync AJAX handlers


        add_action('wp_ajax_lsp_dropbox_process_queue', [$self, 'ajax_dropbox_process_queue']);
        add_action('wp_ajax_lsp_get_album_cover', [$self, 'ajax_get_album_cover']);

        // AI Insights AJAX handlers


        // AI Performance tracking (public endpoint for frontend)


        add_action('wp_ajax_lsp_dismiss_celebration', [$self, 'ajax_dismiss_celebration']);
        add_action('wp_ajax_lsp_toggle_helpers', [$self, 'ajax_toggle_helpers']);
        add_action('wp_ajax_lsp_tour_complete', [$self, 'ajax_tour_complete']);

        // AI auto-optimization cron

        // Media Library columns for tracking stats

        add_action('admin_init', function() {
            if (get_option('lightsync_background_sync_queue')) {
                if (!defined('DOING_CRON') && !wp_doing_ajax()) {
                    spawn_cron();
                }
            }
        });

        // REST API endpoint for Hub to replace media files in-place
        add_action('rest_api_init', function() {
            register_rest_route('lightsync/v1', '/replace-media/(?P<id>\d+)', [
                'methods' => 'POST',
                'callback' => [Admin::class, 'rest_replace_media'],
                'permission_callback' => function($request) {
                    return current_user_can('upload_files');
                },
                'args' => [
                    'id' => [
                        'required' => true,
                        'type' => 'integer',
                    ],
                ],
            ]);
        });
        add_action('wp_ajax_lightsyncpro_ajax_usage_get', [$self,'ajax_usage_get']);
        add_action('lsp_canva_background_sync_run', [__CLASS__, 'canva_background_sync_run'], 10, 1);

        register_activation_hook(LIGHTSYNC_PRO_FILE,   [$self,'activate']);
        register_deactivation_hook(LIGHTSYNC_PRO_FILE, [$self,'deactivate']);
        add_action(self::CRON_RETRY_HOOK, [__CLASS__, 'cron_retry_renditions']);

        // Frontend performance tracking script
    }

    /* ========== LICENSE MODAL CHECK ========== */

    /**
     * Check if the blocking license modal should be shown.
     * Only blocks if this is the PAID plugin AND license is not active.
     */
    private function should_show_license_modal(): bool {
        return false;
    }

    /* ========== USAGE TRACKING (SEPARATE OPTION - SURVIVES DISCONNECTS) ========== */

    private static function get_usage_data(): array {
        $data = get_option(self::USAGE_OPT, []);
        if (!is_array($data)) $data = [];
        
        $curKey = gmdate('Y-m');
        $oldKey = (string)($data['month_key'] ?? '');
        
        if ($oldKey !== $curKey) {
            $data['month_key'] = $curKey;
            $data['month_count'] = 0;
            update_option(self::USAGE_OPT, $data, false);
        }
        
        return $data;
    }

    private static function set_usage_data(array $data): void {
        update_option(self::USAGE_OPT, $data, false);
    }

    public static function usage_remaining_month(): int {
        $usage = self::usage_get();
        $caps  = self::usage_caps();
        if ((int)$caps['month'] === 0) return PHP_INT_MAX;
        return max(0, (int)$caps['month'] - (int)$usage['month_count']);
    }

    public static function usage_consume(int $n): array {
        $n = max(0, (int)$n);
        $chk = self::usage_can_consume($n);
        if (!$chk['ok']) return ['ok'=>false] + $chk;

        $after = self::usage_bump($n);
        return ['ok'=>true, 'after'=>$after] + $chk;
    }

    public static function usage_caps(): array {
        $plan = self::plan_matrix()[ self::plan() ] ?? self::plan_matrix()['free'];
        return [
            'month'  => (int)($plan['photos_month'] ?? 0),
            'albums' => (int)($plan['max_albums'] ?? 1),
        ];
    }

    public static function usage_get(): array {
        $data = self::get_usage_data();
        
        return [
            'month_key'   => (string)($data['month_key'] ?? gmdate('Y-m')),
            'month_count' => (int)($data['month_count'] ?? 0),
            'total_count' => (int)($data['total_count'] ?? 0),
            'plan'        => self::plan(),
        ];
    }

    private static function usage_can_consume(int $count): array {
        $caps = self::usage_caps();
        $cap  = (int) ($caps['month'] ?? 0);

        $u    = self::usage_get();
        $used = (int) ($u['month_count'] ?? 0);

        if ($count <= 0) {
            return ['ok' => true, 'remaining' => max(0, $cap - $used), 'cap' => $cap, 'used' => $used];
        }

        if ($cap <= 0) {
            return ['ok' => true, 'remaining' => PHP_INT_MAX, 'cap' => 0, 'used' => $used];
        }

        $remaining = $cap - $used;
        return [
            'ok'        => ($count <= $remaining),
            'remaining' => max(0, $remaining),
            'cap'       => $cap,
            'used'      => $used,
        ];
    }

    public static function usage_bump(int $n): array {
        if ($n <= 0) return self::usage_get();

        $data = self::get_usage_data();

        $month = (int)($data['month_count'] ?? 0);
        $total = (int)($data['total_count'] ?? 0);

        $data['month_key'] = gmdate('Y-m');
        $data['month_count'] = $month + $n;
        $data['total_count'] = $total + $n;

        self::set_usage_data($data);

        return self::usage_get();
    }

    public static function usage_bump_photos(int $delta): void {
        if ($delta <= 0) return;
        self::usage_bump($delta);
    }

    /* ========== STORAGE STATS TRACKING ========== */

    const STORAGE_STATS_OPT = 'lightsync_storage_stats';

    public static function get_storage_stats(): array {
        $data = get_option(self::STORAGE_STATS_OPT, []);
        if (!is_array($data)) $data = [];
        
        return [
            'original_bytes'  => (int)($data['original_bytes'] ?? 0),
            'optimized_bytes' => (int)($data['optimized_bytes'] ?? 0),
            'images_tracked'  => (int)($data['images_tracked'] ?? 0),
        ];
    }

    public static function bump_storage_stats(int $original, int $optimized, string $source = ''): void {
        if ($original <= 0) return;
        
        $data = get_option(self::STORAGE_STATS_OPT, []);
        if (!is_array($data)) $data = [];
        
        $data['original_bytes']  = (int)($data['original_bytes'] ?? 0) + $original;
        $data['optimized_bytes'] = (int)($data['optimized_bytes'] ?? 0) + $optimized;
        $data['images_tracked']  = (int)($data['images_tracked'] ?? 0) + 1;
        
        update_option(self::STORAGE_STATS_OPT, $data, false);
        
        // Track for weekly digest if source provided
        if ($source && class_exists('\\LightSyncPro\\Admin\\WeeklyDigest')) {
            WeeklyDigest::track_sync($source, 1, $original, $optimized);
        }
    }

    public static function format_bytes(int $bytes, int $precision = 1): string {
        if ($bytes <= 0) return '0 B';
        
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $pow = floor(log($bytes, 1024));
        $pow = min($pow, count($units) - 1);
        
        return round($bytes / pow(1024, $pow), $precision) . ' ' . $units[$pow];
    }

    /* ========== END USAGE TRACKING ========== */

    private function render_storage_stats(): void {
        $usage = self::usage_get();
        $storage = self::get_storage_stats();
        
        $totalPhotos = (int)($usage['total_count'] ?? 0);
        $monthPhotos = (int)($usage['month_count'] ?? 0);
        $originalBytes = (int)($storage['original_bytes'] ?? 0);
        $optimizedBytes = (int)($storage['optimized_bytes'] ?? 0);
        
        $savedBytes = max(0, $originalBytes - $optimizedBytes);
        $savingsPercent = ($originalBytes > 0) ? round(($savedBytes / $originalBytes) * 100) : 0;
        
        // Format - always WebP in free version
        $format = 'WebP';
        $formatClass = 'lsp-badge-wp';
        
        echo '<div class="lsp-stats-card" style="margin-top:14px;">';
        echo '<section class="section" style="border-radius:18px;">';
        echo '<div class="section-head"><h3 class="lsp-card-title">📊 Optimization Stats</h3></div>';
        echo '<div class="panel"><div class="lsp-card-body">';
        
        echo '<div class="lsp-stats-grid" style="display:grid;grid-template-columns:repeat(auto-fit,minmax(140px,1fr));gap:16px;">';
        
        // Total Photos
        echo '<div class="lsp-stat-item" style="text-align:center;padding:12px;background:rgba(0,0,0,0.03);border-radius:12px;">';
        echo '<div style="font-size:28px;font-weight:800;color:var(--lsp-primary);">' . number_format($totalPhotos) . '</div>';
        echo '<div style="font-size:12px;color:#6b7280;margin-top:4px;">Images Synced</div>';
        echo '</div>';
        
        // This Month
        echo '<div class="lsp-stat-item" style="text-align:center;padding:12px;background:rgba(0,0,0,0.03);border-radius:12px;">';
        echo '<div style="font-size:28px;font-weight:800;color:var(--lsp-accent);">' . number_format($monthPhotos) . '</div>';
        echo '<div style="font-size:12px;color:#6b7280;margin-top:4px;">This Month</div>';
        echo '</div>';
        
        // Format
        echo '<div class="lsp-stat-item" style="text-align:center;padding:12px;background:rgba(0,0,0,0.03);border-radius:12px;">';
        echo '<div style="font-size:28px;font-weight:800;"><span class="badge ' . esc_attr($formatClass) . '" style="font-size:16px;">' . esc_html($format) . '</span></div>';
        echo '<div style="font-size:12px;color:#6b7280;margin-top:4px;">Output Format</div>';
        echo '</div>';
        
        // Savings
        echo '<div class="lsp-stat-item" style="text-align:center;padding:12px;background:rgba(0,0,0,0.03);border-radius:12px;">';
        if ($originalBytes > 0) {
            echo '<div style="font-size:28px;font-weight:800;color:#22c55e;">' . esc_html($savingsPercent) . '%</div>';
            echo '<div style="font-size:12px;color:#6b7280;margin-top:4px;">Space Saved</div>';
        } else {
            echo '<div style="font-size:28px;font-weight:800;color:#9ca3af;">—</div>';
            echo '<div style="font-size:12px;color:#6b7280;margin-top:4px;">Space Saved</div>';
        }
        echo '</div>';
        
        echo '</div>'; // end stats-grid
        
        // Storage breakdown (only show if we have data)
        if ($originalBytes > 0) {
            echo '<div style="margin-top:16px;padding-top:16px;border-top:1px solid #e5e7eb;display:flex;justify-content:space-between;flex-wrap:wrap;gap:12px;">';
            echo '<div style="font-size:13px;color:#6b7280;">';
            echo '<strong>Original:</strong> ' . esc_html(self::format_bytes($originalBytes));
            echo ' → <strong>Optimized:</strong> ' . esc_html(self::format_bytes($optimizedBytes));
            echo '</div>';
            echo '<div style="font-size:13px;color:#22c55e;font-weight:600;">';
            echo '💾 ' . esc_html(self::format_bytes($savedBytes)) . ' saved';
            echo '</div>';
            echo '</div>';
        }
        
        echo '</div></div>'; // end card-body, panel
        echo '</section></div>'; // end section, stats-card
    }

    private function render_recent_activity(): void {
        $items = get_option('lightsync_recent_activity', []);

        if (!is_array($items) || empty($items)) {
            $items = (array) self::get_opt('lightsync_activity', []);
        }

        // Always render the section container so the Activity nav link works
        echo '<div class="lsp-activity-card" id="lsp-activity" style="margin-top:14px;">';
        echo '<section class="section" style="border-radius:18px;">';
        echo '<div class="section-head"><h3 class="lsp-card-title">Recent Activity</h3></div>';
        echo '<div class="twocol"><div class="panel"><div class="lsp-card-body">';

        if (!is_array($items) || empty($items)) {
            // Show empty state
            echo '<div style="text-align:center;padding:30px 20px;color:#64748b;">';
            echo '<span class="dashicons dashicons-clock" style="font-size:32px;width:32px;height:32px;margin-bottom:12px;opacity:0.5;"></span>';
            echo '<p style="margin:0;font-size:14px;">No sync activity yet</p>';
            echo '<p style="margin:8px 0 0;font-size:13px;color:#94a3b8;">Activity will appear here after your first sync.</p>';
            echo '</div>';
            
            // Still show digest settings in empty state
            $this->render_digest_settings();
            
            echo '</div></div>';
            $brand = lsp_get_brand();
            echo '<aside class="help"><h3>Recent Activity</h3><p>A quick snapshot of what ' . esc_html( $brand['name'] ) . ' has been doing on your site.</p><p><strong>Weekly Digest:</strong> Get a summary email every Monday with sync stats.</p><p><a href="' . esc_url( $brand['docs_url'] ) . '" target="_blank">Learn more →</a></p></aside>';
            echo '</div></section></div>';
            return;
        }

        // Get stats for KPI strip
        $usage = self::usage_get();
        $storage = self::get_storage_stats();
        $totalPhotos = (int)($usage['total_count'] ?? 0);
        $originalBytes = (int)($storage['original_bytes'] ?? 0);
        $optimizedBytes = (int)($storage['optimized_bytes'] ?? 0);
        $savedBytes = max(0, $originalBytes - $optimizedBytes);
        $savingsPercent = ($originalBytes > 0) ? round(($savedBytes / $originalBytes) * 100) : 0;
        
        // Format - always WebP in free version
        $format = 'WebP';
        echo '<div class="lsp-kpis" style="grid-template-columns:repeat(4,1fr);margin-bottom:14px;">';
        
        echo '<div class="lsp-kpi">';
        echo '<span class="label">Synced</span>';
        echo '<span class="value">' . number_format($totalPhotos) . '</span>';
        echo '</div>';
        
        echo '<div class="lsp-kpi">';
        echo '<span class="label">Format</span>';
        echo '<span class="value" style="font-size:12px;">' . esc_html($format) . '</span>';
        echo '</div>';
        
        echo '<div class="lsp-kpi">';
        echo '<span class="label">Saved</span>';
        echo '<span class="value" style="color:#22c55e;">' . ($originalBytes > 0 ? $savingsPercent . '%' : '—') . '</span>';
        echo '</div>';
        
        echo '<div class="lsp-kpi">';
        echo '<span class="label">Storage</span>';
        echo '<span class="value" style="font-size:11px;color:#22c55e;">' . ($savedBytes > 0 ? self::format_bytes($savedBytes) : '—') . '</span>';
        echo '</div>';
        
        echo '</div>'; // end lsp-kpis
        
        echo '<ul class="lsp-activity">';

        foreach ($items as $it) {
            if (!is_array($it)) continue;

            $ts     = (int)($it['ts'] ?? 0);
            $status = (string)($it['status'] ?? 'info');
            $type   = (string)($it['type'] ?? 'sync');
            $source = (string)($it['source'] ?? '');
            $msg    = (string)($it['message'] ?? ($it['msg'] ?? ''));

            if ($msg === '') continue;

            $ago = $ts ? human_time_diff($ts, time()) . ' ago' : '';

            echo '<li class="lsp-activity-item type-'.esc_attr($status).'">';
            echo '<span class="msg">'.esc_html($msg).'</span>';

            if ($source !== '') {
                echo ' <span class="badge">'.esc_html(strtoupper($source)).'</span>';
            }

            if ($ago) echo '<span class="ago">'.esc_html($ago).'</span>';
            echo '</li>';
        }

        echo '</ul>';
        
        // Weekly Digest settings inline
        $this->render_digest_settings();
        
        echo '</div></div>';
        $brand = lsp_get_brand();
        echo '<aside class="help"><h3>Recent Activity</h3><p>A quick snapshot of what ' . esc_html( $brand['name'] ) . ' has been doing on your site.</p><p><strong>Weekly Digest:</strong> Get a summary email every Monday with sync stats, storage saved, and time saved.</p><p><a href="' . esc_url( $brand['docs_url'] ) . '" target="_blank">Learn more →</a></p></aside>';
        echo '</div></section></div>';
    }
    
    /**
     * Render digest settings (used in activity panel)
     */
    private function render_digest_settings(): void {
        $digest_enabled = (int) self::get_opt('weekly_digest_enabled', 0);
        $digest_email = self::get_opt('weekly_digest_email', get_option('admin_email'));
        
        echo '<div style="margin-top:20px;padding-top:20px;border-top:1px solid #e2e8f0;">';
        echo '<div style="display:flex;align-items:center;justify-content:space-between;margin-bottom:12px;">';
        echo '<h4 style="margin:0;font-size:13px;font-weight:600;color:#374151;">Weekly Digest</h4>';
        echo '<label class="lsp-toggle" style="margin:0;">';
        echo '<input type="checkbox" id="lsp-digest-enable" value="1" ' . checked($digest_enabled, 1, false) . '>';
        echo '<span class="track"><span class="thumb"></span></span>';
        echo '</label>';
        echo '</div>';
        
        echo '<div id="lsp-digest-options" style="' . ($digest_enabled ? '' : 'display:none;') . '">';
        echo '<p style="margin:0 0 12px 0;font-size:12px;color:#64748b;">Get a summary of your sync activity every Monday.</p>';
        
        echo '<div style="margin-bottom:12px;">';
        echo '<label style="font-size:12px;color:#64748b;display:block;margin-bottom:4px;">Email Address</label>';
        echo '<input type="email" id="lsp-digest-email" value="' . esc_attr($digest_email) . '" placeholder="' . esc_attr(get_option('admin_email')) . '" style="width:100%;padding:8px 12px;border:1px solid #e2e8f0;border-radius:6px;font-size:13px;">';
        echo '</div>';
        
        echo '<div style="display:flex;gap:8px;align-items:center;margin-top:12px;">';
        echo '<button type="button" id="lsp-digest-test" class="btn ghost" style="padding:6px 14px;font-size:12px;">Send Test</button>';
        echo '<span id="lsp-digest-status" style="font-size:12px;color:#64748b;"></span>';
        echo '</div>';
        echo '</div>'; // end digest-options
        echo '</div>'; // end digest wrapper
        
        // Add JavaScript for digest settings
        ?>
        <script>
        (function(){
            var debounceTimer;
            function saveDigestSettings() {
                clearTimeout(debounceTimer);
                debounceTimer = setTimeout(function(){
                    var data = {
                        action: 'lightsyncpro_ajax_save_digest',
                        _ajax_nonce: '<?php echo esc_js(wp_create_nonce(self::AJAX_NS . '_nonce')); ?>',
                        weekly_digest_enabled: document.getElementById('lsp-digest-enable').checked ? 1 : 0,
                        weekly_digest_email: document.getElementById('lsp-digest-email').value
                    };
                    
                    jQuery.post(ajaxurl, data, function(resp){
                        if (resp.success) {
                            var status = document.getElementById('lsp-digest-status');
                            if (status) {
                                status.textContent = 'Saved';
                                status.style.color = '#22c55e';
                                setTimeout(function(){ status.textContent = ''; }, 2000);
                            }
                        }
                    });
                }, 300);
            }
            
            // Toggle handler
            var digestEnable = document.getElementById('lsp-digest-enable');
            var digestOptions = document.getElementById('lsp-digest-options');
            
            if (digestEnable) {
                digestEnable.addEventListener('change', function(){
                    if (digestOptions) {
                        digestOptions.style.display = this.checked ? '' : 'none';
                    }
                    saveDigestSettings();
                });
            }
            
            // Email input handler
            var digestEmail = document.getElementById('lsp-digest-email');
            if (digestEmail) {
                digestEmail.addEventListener('change', saveDigestSettings);
                digestEmail.addEventListener('blur', saveDigestSettings);
            }
            
            // Test button handler
            var testBtn = document.getElementById('lsp-digest-test');
            if (testBtn) {
                testBtn.addEventListener('click', function(){
                    var btn = this;
                    var status = document.getElementById('lsp-digest-status');
                    btn.disabled = true;
                    btn.textContent = 'Sending...';
                    if (status) status.textContent = '';
                    
                    jQuery.post(ajaxurl, {
                        action: 'lsp_send_test_digest',
                        _wpnonce: '<?php echo esc_js(wp_create_nonce('lightsyncpro_ajax_nonce')); ?>',
                        email: document.getElementById('lsp-digest-email').value
                    }, function(resp){
                        btn.disabled = false;
                        btn.textContent = 'Send Test';
                        if (status) {
                            if (resp.success) {
                                status.textContent = resp.data.message || 'Test email sent!';
                                status.style.color = '#22c55e';
                            } else {
                                status.textContent = resp.data.error || 'Failed to send';
                                status.style.color = '#ef4444';
                            }
                        }
                    }).fail(function(){
                        btn.disabled = false;
                        btn.textContent = 'Send Test';
                        if (status) {
                            status.textContent = 'Request failed';
                            status.style.color = '#ef4444';
                        }
                    });
                });
            }
        })();
        </script>
        <?php
    }

    public function ajax_debug_sync() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_ajax_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('forbidden', 403);
        }
        
        $queue = get_option('lightsync_background_sync_queue', []);
        $catalog = $queue['catalog'] ?? '';
        $albums = $queue['albums'] ?? [];
        $current_idx = (int)($queue['current_index'] ?? 0);
        $current_album = $albums[$current_idx] ?? '';
        
        $lock_key = "lightsync_tick_lock_{$catalog}_{$current_album}";
        $cursor_opt = "lightsync_sync_cursor_{$catalog}_{$current_album}";
        $idx_opt = "lightsync_sync_next_index_{$catalog}_{$current_album}";
        
        $crons = _get_cron_array();
        $tick_events = [];
        if (is_array($crons)) {
            foreach ($crons as $timestamp => $hooks) {
                if (isset($hooks['lightsyncpro_sync_tick'])) {
                    foreach ($hooks['lightsyncpro_sync_tick'] as $hash => $event) {
                        $tick_events[] = [
                            'timestamp' => $timestamp,
                            'in_seconds' => $timestamp - time(),
                            'args' => $event['args'],
                        ];
                    }
                }
            }
        }
        
        wp_send_json_success([
            'queue' => $queue,
            'catalog' => $catalog,
            'current_album' => $current_album,
            'lock_key' => $lock_key,
            'lock_value' => get_transient($lock_key),
            'cursor' => get_option($cursor_opt, '(not set)'),
            'next_index' => get_option($idx_opt, '(not set)'),
            'scheduled_ticks' => $tick_events,
            'time_now' => time(),
            'cron_disabled' => defined('DISABLE_WP_CRON') && DISABLE_WP_CRON,
            'usage_data' => self::usage_get(),
        ]);
    }

    public function ajax_cancel_background() {
        if (!check_ajax_referer(self::AJAX_NS.'_nonce', '_ajax_nonce', false)) {
            wp_send_json_error(['error' => 'bad_nonce'], 403);
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'forbidden'], 403);
        }

        $queue = get_option('lightsync_background_sync_queue', []);
        
        if (empty($queue)) {
            wp_send_json_success([
                'cancelled' => false,
                'message' => 'No background sync was running',
            ]);
            return;
        }

        $albums = (array)($queue['albums'] ?? []);
        $current = (int)($queue['current_index'] ?? 0);
        $catalog = (string)($queue['catalog'] ?? '');

        delete_option('lightsync_background_sync_queue');

        foreach ($albums as $album_id) {
            delete_option("lightsync_sync_cursor_{$catalog}_{$album_id}");
            delete_option("lightsync_sync_next_index_{$catalog}_{$album_id}");
            delete_transient("lightsync_tick_lock_{$catalog}_{$album_id}");
        }

        $crons = _get_cron_array();
        if (is_array($crons)) {
            foreach ($crons as $timestamp => $hooks) {
                if (isset($hooks['lightsyncpro_sync_tick'])) {
                    foreach ($hooks['lightsyncpro_sync_tick'] as $hash => $event) {
                        $args = $event['args'] ?? [];
                        if (isset($args[0]) && $args[0] === $catalog) {
                            wp_unschedule_event($timestamp, 'lightsyncpro_sync_tick', $args);
                        }
                    }
                }
            }
        }

        self::add_activity(
            sprintf('Lightroom → WordPress Sync Cancelled (Background): %d of %d album(s) completed', $current, count($albums)),
            'warning',
            'manual-background'
        );

        wp_send_json_success([
            'cancelled' => true,
            'albums_completed' => $current,
            'albums_total' => count($albums),
            'message' => sprintf('Background sync cancelled. %d of %d albums were completed.', $current, count($albums)),
        ]);
    }

    public function ajax_background_status() {
        if (!check_ajax_referer(self::AJAX_NS.'_nonce', '_ajax_nonce', false)) {
            wp_send_json_error(['error' => 'bad_nonce'], 403);
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'forbidden'], 403);
        }

        $queue = get_option('lightsync_background_sync_queue', []);
        
        if (!empty($queue)) {
            $catalog = $queue['catalog'] ?? '';
            $albums = $queue['albums'] ?? [];
            $current_idx = (int)($queue['current_index'] ?? 0);
            $current_album = $albums[$current_idx] ?? '';
            
            if ($catalog && $current_album) {
                $lock_key = "lightsync_tick_lock_{$catalog}_{$current_album}";
                
                if (!get_transient($lock_key)) {
                    try {
                        \LightSyncPro\Util\Logger::debug('[LSP ajax_background_status] Running tick for ' . $current_album);
                        self::cron_sync_tick($catalog, $current_album, 'manual-background');
                    } catch (\Throwable $e) {
                        \LightSyncPro\Util\Logger::debug('[LSP ajax_background_status] tick error: ' . $e->getMessage());
                        delete_transient($lock_key);
                    }
                } else {
                    \LightSyncPro\Util\Logger::debug('[LSP ajax_background_status] tick locked, skipping');
                }
            }
            
            $queue = get_option('lightsync_background_sync_queue', []);
        }
        
        if (empty($queue)) {
            wp_send_json_success([
                'running' => false,
                'message' => 'No background sync in progress',
            ]);
            return;
        }

        $albums = (array)($queue['albums'] ?? []);
        $current = (int)($queue['current_index'] ?? 0);
        $started = (int)($queue['started_at'] ?? time());
        $catalog = (string)($queue['catalog'] ?? '');
        $total = count($albums);

        $current_album_id = $albums[$current] ?? '';
        $current_album_name = $current_album_id 
            ? self::get_album_name_cached($catalog, $current_album_id) 
            : '';

        $pct = 0;
        if ($total > 0) {
            $base_pct = ($current / $total) * 100;
            $in_progress_bump = (1 / $total) * 50;
            $pct = round($base_pct + $in_progress_bump);
            // Cap at 98% so user knows it's still working, 100% only when truly done
            $pct = max(5, min(98, $pct));
        }

        wp_send_json_success([
            'running' => true,
            'total_albums' => $total,
            'current_album_index' => $current,
            'current_album_name' => $current_album_name,
            'elapsed' => human_time_diff($started, time()),
            'started_at' => $started,
            'progress_pct' => $pct,
        ]);
    }

    /**
     * Get asset IDs from Lightroom albums
     * Used by Hub distribution to get actual asset IDs instead of album IDs
     */
    public function ajax_get_album_assets() {
        if (!check_ajax_referer('lightsyncpro_nonce', 'nonce', false)) {
            wp_send_json_error(['error' => 'Invalid nonce'], 403);
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Permission denied'], 403);
        }
        
        $catalog_id = sanitize_text_field($_POST['catalog_id'] ?? '');
        $album_ids = array_map('sanitize_text_field', (array) ($_POST['album_ids'] ?? []));
        
        if (empty($catalog_id) || empty($album_ids)) {
            wp_send_json_error(['error' => 'Missing catalog_id or album_ids']);
        }
        
        $all_asset_ids = [];
        
        foreach ($album_ids as $album_id) {
            $album_id = sanitize_text_field($album_id);
            if (empty($album_id)) continue;
            
            // Get assets from this album (up to 200)
            $data = Sync::get_album_assets($catalog_id, $album_id, null, 200);
            
            if (!empty($data['resources'])) {
                foreach ($data['resources'] as $resource) {
                    $asset_id = $resource['asset']['id'] ?? null;
                    if ($asset_id && !in_array($asset_id, $all_asset_ids)) {
                        $all_asset_ids[] = $asset_id;
                    }
                }
            }
        }
        
        wp_send_json_success([
            'asset_ids' => $all_asset_ids,
            'count' => count($all_asset_ids),
        ]);
    }

    /**
     * Save selected Hub sites for distribution
     */
    public function ajax_save_hub_sites() {
        if (!check_ajax_referer('lightsyncpro_nonce', 'nonce', false)) {
            wp_send_json_error(['error' => 'Invalid nonce'], 403);
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Permission denied'], 403);
        }
        
        $site_ids = isset($_POST['site_ids']) ? array_map('intval', (array) $_POST['site_ids']) : [];
        
        self::set_opt(['hub_selected_sites' => $site_ids]);
        
        wp_send_json_success(['saved' => count($site_ids)]);
    }


    private static function get_broker_token(): string {
        $o = self::get_opt();
        $enc = (string)($o['broker_token_enc'] ?? '');
        if ($enc === '') return '';
        
        return (string)\LightSyncPro\Util\Crypto::dec($enc);
    }

    /* ==================== SHOPIFY AJAX HANDLERS ==================== */

    public function ajax_shopify_status() {
        if (!check_ajax_referer(self::AJAX_NS . '_nonce', '_ajax_nonce', false)) {
            wp_send_json_error(['error' => 'bad_nonce'], 403);
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'forbidden'], 403);
        }
        
        $o = self::get_opt();
        $shop = (string)($o['shopify_shop_domain'] ?? '');
        $token = self::get_shopify_token($shop);
        
        $connected = ($shop !== '' && $token !== '');
        
        wp_send_json_success([
            'connected'   => $connected,
            'shop_domain' => $connected ? $shop : '',
        ]);
    }

    public function ajax_shopify_reset_sync() {
        if (!check_ajax_referer(self::AJAX_NS . '_nonce', '_ajax_nonce', false)) {
            wp_send_json_error(['error' => 'bad_nonce'], 403);
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'forbidden'], 403);
        }

        $o = self::get_opt();
        $shop = (string)($o['shopify_shop_domain'] ?? '');

        if (!$shop) {
            wp_send_json_error(['error' => 'No Shopify store connected']);
        }

        $cleared = Shopify::clear_files_map($shop);

        self::add_activity(
            sprintf('Shopify sync reset: cleared %d file mappings', $cleared),
            'info',
            'manual'
        );

        wp_send_json_success([
            'cleared' => $cleared,
            'message' => sprintf('Cleared %d file mappings. Next sync will re-upload all images.', $cleared),
        ]);
    }

    private static function get_shopify_token(string $shop): string {
        if ($shop === '') return '';
        
        $o = self::get_opt();
        
        if (isset($o['shopify_access_token'])) {
            if (is_array($o['shopify_access_token']) && isset($o['shopify_access_token'][$shop])) {
                $cached = trim((string)$o['shopify_access_token'][$shop]);
                if ($cached !== '') return $cached;
            }
            if (is_string($o['shopify_access_token']) && trim($o['shopify_access_token']) !== '') {
                return trim($o['shopify_access_token']);
            }
        }
        
        $broker_token = self::get_broker_token();
        if (!$broker_token) return '';
        
        $response = wp_remote_post('https://lightsyncpro.com/wp-json/lsp-broker/v1/shopify/token', [
            'timeout' => 15,
            'headers' => [
                'Authorization' => 'Bearer ' . $broker_token,
                'Content-Type'  => 'application/json',
            ],
            'body' => wp_json_encode(['shop_domain' => $shop]),
        ]);
        
        if (is_wp_error($response)) return '';
        
        $code = (int)wp_remote_retrieve_response_code($response);
        if ($code !== 200) return '';
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (empty($body['access_token'])) return '';
        
        $tokens = is_array($o['shopify_access_token'] ?? null) ? $o['shopify_access_token'] : [];
        $tokens[$shop] = $body['access_token'];
        self::set_opt(['shopify_access_token' => $tokens]);
        
        return $body['access_token'];
    }

    public function ajax_shopify_connect_start() {
        if (!current_user_can('manage_options')) {
            wp_die('Forbidden', 403);
        }

        $site   = isset($_GET['site']) ? esc_url_raw($_GET['site']) : '';
        $state  = isset($_GET['state']) ? sanitize_text_field($_GET['state']) : '';
        $return = isset($_GET['return']) ? esc_url_raw($_GET['return']) : '';

        if (!$site || !$state || !$return) {
            wp_die('Missing parameters', 400);
        }

        $broker_url = add_query_arg([
            'site'   => $site,
            'state'  => $state,
            'return' => $return,
        ], 'https://lightsyncpro.com/wp-json/lsp-broker/v1/shopify/connect');

        wp_redirect($broker_url);
        exit;
    }

    public function ajax_shopify_save_settings() {
        if (!check_ajax_referer(self::AJAX_NS . '_nonce', '_ajax_nonce', false)) {
            wp_send_json_error(['error' => 'bad_nonce'], 403);
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'forbidden'], 403);
        }

        $sync_target = isset($_POST['sync_target']) 
            ? sanitize_text_field(wp_unslash($_POST['sync_target'])) 
            : 'wp';
        
        if (!in_array($sync_target, ['wp', 'shopify', 'both'], true)) {
            $sync_target = 'wp';
        }

        self::set_opt([
            'sync_target' => $sync_target,
        ]);

        wp_send_json_success(['saved' => true]);
    }

    public function ajax_shopify_disconnect() {
        if (!check_ajax_referer(self::AJAX_NS . '_nonce', '_ajax_nonce', false)) {
            wp_send_json_error(['error' => 'bad_nonce'], 403);
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'forbidden'], 403);
        }

        // Track the old shop domain before clearing
        $old_shop = (string)(self::get_opt('shopify_shop_domain') ?? '');

        self::set_opt([
            'shopify_connected'    => 0,
            'shopify_shop_domain'  => '',
            'shopify_shop_id'      => '',
            'shopify_access_token' => '',
            'sync_target'          => 'wp',
        ]);

        if ($old_shop !== '') {
            self::add_activity(
                sprintf('Disconnected from Shopify store: %s', $old_shop),
                'info',
                'shopify'
            );
        }

        wp_send_json_success(['disconnected' => true]);
    }

    /* ==================== CANVA AJAX HANDLERS ==================== */

    /**
     * Get Canva designs list
     */
    public function ajax_canva_get_designs() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\CanvaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Canva not connected']);
        }

        // Extend timeout for pagination
        @set_time_limit(120);

        // Get ALL designs with pagination (up to 10 pages = ~250 designs)
        $result = \LightSyncPro\OAuth\CanvaOAuth::get_all_designs(10);

        if (is_wp_error($result)) {
            wp_send_json_error(['error' => $result->get_error_message()]);
        }

        wp_send_json_success($result);
    }

    /**
     * Disconnect Canva
     */
    public function ajax_canva_disconnect() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        \LightSyncPro\OAuth\CanvaOAuth::disconnect();
        wp_send_json_success(['disconnected' => true]);
    }

    /**
     * Get list of already-synced Canva design IDs
     */
    public function ajax_canva_get_synced() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        global $wpdb;
        
        // Get all Canva design IDs with their sync timestamps
        // Use _lightsync_last_synced_at meta (updated even on skipped syncs)
        // Fall back to post_modified_gmt if meta doesn't exist
        $results = $wpdb->get_results(
            "SELECT 
                pm_id.meta_value as design_id, 
                pm_id.post_id as attachment_id,
                COALESCE(pm_sync.meta_value, p.post_modified_gmt) as synced_at
             FROM {$wpdb->postmeta} pm_id
             JOIN {$wpdb->posts} p ON pm_id.post_id = p.ID
             LEFT JOIN {$wpdb->postmeta} pm_sync ON pm_id.post_id = pm_sync.post_id 
                AND pm_sync.meta_key = '_lightsync_last_synced_at'
             WHERE pm_id.meta_key = '_lightsync_canva_design_id'
             GROUP BY pm_id.meta_value"
        );

        // Get Shopify mappings
        $shopify_table = $wpdb->prefix . 'lightsync_shopify_files_map';
        $shop = self::get_opt('shopify_shop_domain', '');
        
        $shopify_synced = [];
        if ($shop && $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $shopify_table)) === $shopify_table) {
            $shopify_results = $wpdb->get_results($wpdb->prepare(
                "SELECT lr_asset_id FROM {$shopify_table} WHERE shop_domain = %s AND shopify_file_id IS NOT NULL",
                $shop
            ));
            foreach ($shopify_results as $row) {
                $shopify_synced[$row->lr_asset_id] = true;
            }
        }

        // Build array with design_id => {timestamp, destinations}
        $synced = [];
        foreach ($results as $row) {
            $has_wp = true;
            $has_shopify = isset($shopify_synced[$row->design_id]);
            
            $dest = 'wp';
            if ($has_wp && $has_shopify) {
                $dest = 'both';
            } elseif ($has_shopify) {
                $dest = 'shopify';
            }
            
            $synced[$row->design_id] = [
                'time' => strtotime($row->synced_at . ' UTC'),
                'dest' => $dest,
            ];
        }
        
        // Also check for Shopify-only syncs (designs synced to Shopify but not WordPress)
        foreach ($shopify_synced as $asset_id => $v) {
            if (!isset($synced[$asset_id])) {
                $synced[$asset_id] = [
                    'time' => time(),
                    'dest' => 'shopify',
                ];
            }
        }

        wp_send_json_success($synced);
    }

    /**
     * Save Canva sync target preference
     */
    public function ajax_canva_save_target() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $target = isset($_POST['target']) ? sanitize_text_field($_POST['target']) : 'wp';
        
        if (!in_array($target, ['wp', 'shopify', 'both'], true)) {
            $target = 'wp';
        }

        self::set_opt(['canva_sync_target' => $target]);
        wp_send_json_success(['target' => $target]);
    }

    /**
     * Sync selected Canva designs to WordPress
     */
    public function ajax_canva_sync_designs() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        // Sync gate check
        $gate = lsp_gate_check();
        if ( is_wp_error( $gate ) ) {
            wp_send_json_error([
                'error' => 'sync_disabled',
                'message' => $gate->get_error_message(),
            ], 403);
        }

        // Extend timeout for long-running exports
        @set_time_limit(300);
        
        $design_ids = isset($_POST['design_ids']) ? array_map('sanitize_text_field', (array)$_POST['design_ids']) : [];

        if (empty($design_ids)) {
            wp_send_json_error(['error' => 'No designs selected']);
        }

        // Check Canva connection
        if (!\LightSyncPro\OAuth\CanvaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Canva not connected. Please reconnect.']);
        }

        // Get sync target
        $sync_target = self::get_opt('canva_sync_target') ?: 'wp';

        $synced = 0;
        $errors = [];
        $total_pages = 0;

        try {
            foreach ($design_ids as $design_id) {
                $result = $this->sync_canva_design($design_id, $sync_target);
                if (is_wp_error($result)) {
                    $errors[] = $design_id . ': ' . $result->get_error_message();
                } else {
                    $synced++;
                    $total_pages += (int)($result['pages'] ?? 1);
                }
            }
        } catch (\Exception $e) {
            wp_send_json_error(['error' => 'Exception: ' . $e->getMessage()]);
        } catch (\Error $e) {
            wp_send_json_error(['error' => 'Error: ' . $e->getMessage()]);
        }

        // Log activity
        if ($synced > 0) {
            $page_text = $total_pages > $synced ? " ({$total_pages} pages)" : '';
            $dest_label = 'WordPress';
            self::add_activity(
                "Canva → {$dest_label} Sync Complete (Manual): {$synced} design(s){$page_text}",
                'success',
                'canva'
            );
            
            // Bump usage stats
            self::usage_bump($total_pages);
            
            // ✅ Update last sync timestamp
            self::set_opt([
                'lightsync_last_sync_ts'     => time(),
                'lightsync_last_sync_source' => 'canva',
                'lightsync_last_sync_status' => 'complete',
            ]);
        }

        if ($synced > 0) {
            wp_send_json_success([
                'synced' => $synced,
                'pages'  => $total_pages,
                'errors' => $errors,
                'lightsync_last_sync_ts' => (int) self::get_opt('lightsync_last_sync_ts', 0),
            ]);
        } else {
            wp_send_json_error([
                'error' => 'Failed to sync designs',
                'details' => $errors,
            ]);
        }
    }

    /**
     * Sync a SINGLE Canva design (for progress updates)
     */
    public function ajax_canva_sync_single() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        // Sync gate check
        $gate = lsp_gate_check();
        if ( is_wp_error( $gate ) ) {
            wp_send_json_error([
                'error' => 'sync_disabled',
                'message' => $gate->get_error_message(),
            ], 403);
        }

        // Extend timeout for export polling
        @set_time_limit(120);

        $design_id = isset($_POST['design_id']) ? sanitize_text_field($_POST['design_id']) : '';

        if (empty($design_id)) {
            wp_send_json_error(['error' => 'No design specified']);
        }

        // Check Canva connection
        if (!\LightSyncPro\OAuth\CanvaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Canva not connected']);
        }

        // Get sync target - prefer POST value over saved option
        $sync_target = isset($_POST['sync_target']) ? sanitize_text_field($_POST['sync_target']) : (self::get_opt('canva_sync_target') ?: 'wp');

        // Sync this single design
        $result = $this->sync_canva_design($design_id, $sync_target);

        if (is_wp_error($result)) {
            wp_send_json_error(['error' => $result->get_error_message()]);
        }

        // Get counts from result
        $pages = (int)($result['pages'] ?? 1);
        $wp_count = (int)($result['wp_count'] ?? 0);
        $wp_updated = (int)($result['wp_updated'] ?? 0);
        $unchanged = !empty($result['unchanged']);
        $is_resync = !empty($result['is_resync']);
        
        // Build activity message based on what was synced
        $dest_parts = [];
        if ($wp_count > 0) $dest_parts[] = 'Media Library';
        $dest_text = implode(' & ', $dest_parts) ?: 'Media Library';
        
        $page_text = $pages > 1 ? " ({$pages} pages)" : '';
        
        // Build activity message with proper action text
        if ($unchanged) {
            $action_text = 'Unchanged';
            $action_text = 'Updated';
        } else {
            $action_text = 'Synced';
        }
        $dest_label = $dest_text === 'Media Library' ? 'WordPress' : $dest_text;
        self::add_activity(
            "Canva → {$dest_label} Sync Complete (Manual): {$action_text} 1 design{$page_text}",
            'success',
            'canva'
        );
        
        // Bump usage
        self::usage_bump($pages);
        
        // ✅ Update last sync timestamp
        self::set_opt([
            'lightsync_last_sync_ts'     => time(),
            'lightsync_last_sync_source' => 'canva',
            'lightsync_last_sync_status' => 'complete',
        ]);

        wp_send_json_success([
            'synced'          => 1,
            'pages'           => $pages,
            'updated'         => !empty($result['updated']),
            'unchanged'       => $unchanged,
            'is_resync'       => $is_resync,
            'wp_count'        => $wp_count,
            'wp_updated'      => $wp_updated,
            'wp_skipped'      => (int)($result['wp_skipped'] ?? 0),
            'sync_target'     => $sync_target,
            'lightsync_last_sync_ts' => (int) self::get_opt('lightsync_last_sync_ts', 0),
        ]);
    }

    /**
     * Sync a single Canva design to WordPress
     * Handles multi-page designs, compression, versioning
     */
    /**
     * Sync Canva image bytes to Shopify Files
     */
    private function sync_canva_to_shopify_bytes($bytes, $filename, $alt_text, $asset_id, $content_hash = '') {
        if (!class_exists('\LightSyncPro\Shopify\Shopify')) {
            return new \WP_Error('shopify_not_available', 'Shopify integration not available');
        }

        if (!$bytes || strlen($bytes) < 100) {
            return new \WP_Error('no_bytes', 'No file data provided');
        }

        // Detect mime type from extension
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $mime_map = [
            'png'  => 'image/png',
            'jpg'  => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'webp' => 'image/webp',
            'avif' => 'image/avif',
            'gif'  => 'image/gif',
        ];
        $mime_type = $mime_map[$ext] ?? 'image/png';

        // Use the direct upload method with content hash for change detection
        $result = \LightSyncPro\Shopify\Shopify::upload_canva_to_shopify(
            $bytes,
            $filename,
            $mime_type,
            $alt_text,
            $asset_id,
            $content_hash
        );

        if (!empty($result['ok'])) {
            return $result;
        }

        return new \WP_Error('shopify_upload_failed', $result['error'] ?? 'Shopify upload failed');
    }

    private function sync_canva_design($design_id, $sync_target = 'wp') {
        try {
            // Get design info first for naming
            $design_info = $this->get_canva_design_info($design_id);
            $design_name = $design_info['title'] ?? 'Canva Design';
            
            // Check if already synced (versioning)
            $existing = $this->find_canva_attachment($design_id);
            $is_update = !empty($existing);

            // Start export
            $export = \LightSyncPro\OAuth\CanvaOAuth::export_design($design_id, 'png');
            if (is_wp_error($export)) {
                return $export;
            }

            if (empty($export['job']['id'])) {
                return new \WP_Error('export_failed', 'Export job not created');
            }

            $job_id = $export['job']['id'];

            // Poll for completion (max 60 seconds)
            $max_attempts = 30;
            $attempt = 0;
            $download_urls = [];

            while ($attempt < $max_attempts) {
                $attempt++;
                sleep(2);

                $status = \LightSyncPro\OAuth\CanvaOAuth::get_export($job_id);
                if (is_wp_error($status)) {
                    return $status;
                }

                if (!empty($status['job']['status']) && $status['job']['status'] === 'success') {
                    if (!empty($status['job']['urls']) && is_array($status['job']['urls'])) {
                        $download_urls = $status['job']['urls'];
                        break;
                    }
                } elseif (!empty($status['job']['status']) && $status['job']['status'] === 'failed') {
                    return new \WP_Error('export_failed', 'Canva export failed');
                }
            }

            if (empty($download_urls)) {
                return new \WP_Error('export_timeout', 'Export timed out');
            }

            // Process each page
            $imported_ids = [];
            $wp_count = 0;
            $wp_updated = 0;
            $wp_skipped = 0;
            $wp_failed = 0;
            $page_count = count($download_urls);

        foreach ($download_urls as $page_index => $url) {
            $page_num = $page_index + 1;
            
            // Build filename
            $page_suffix = $page_count > 1 ? " - Page {$page_num}" : '';
            $filename = sanitize_file_name($design_name . $page_suffix);
            $asset_id = $design_id . ($page_count > 1 ? "-p{$page_num}" : '');

            // Check for existing attachment (for updates)
            $existing_att = $this->find_canva_attachment($asset_id);

            // Always download to check for changes (can't know without the bytes)
            $tmp_file = download_url($url, 60);
            if (is_wp_error($tmp_file)) {
                continue; // Skip this page but continue with others
            }

            // Track original size for storage stats
            $original_size = @filesize($tmp_file) ?: 0;

            // Apply WebP compression
            $compressed = $this->compress_canva_image($tmp_file, $filename);
            if ($compressed && isset($compressed['path']) && $compressed['path'] !== $tmp_file) {
                @unlink($tmp_file);
                $tmp_file = $compressed['path'];
                $filename = $compressed['filename'];
            }

            // Track optimized size
            $optimized_size = @filesize($tmp_file) ?: 0;

            // Determine file extension
            $ext = pathinfo($filename, PATHINFO_EXTENSION);
            if (!$ext) {
                $ext = $compressed ? pathinfo($compressed['filename'], PATHINFO_EXTENSION) : 'png';
                $filename .= '.' . $ext;
            }

            // Read bytes for checksum
            $file_bytes = @file_get_contents($tmp_file);
            if (!$file_bytes) {
                @unlink($tmp_file);
                continue;
            }
            
            // Compute content hash from actual bytes
            $content_hash = hash('sha256', $file_bytes);
            
            // Check if WordPress needs update
            $wp_needs_update = true;
            if ($existing_att) {
                $stored_rev = get_post_meta($existing_att, '_lightsync_rev', true);
                if ($stored_rev && $stored_rev === $content_hash) {
                    // Content unchanged - skip WordPress upload
                    $wp_needs_update = false;
                }
            }

            $attachment_id = null;

            // Sync to WordPress if target is 'wp' or 'both' AND content changed
            if (($sync_target === 'wp' || $sync_target === 'both') && $wp_needs_update) {
                $was_wp_update = false;
                
                // Ensure image functions are loaded
                if (!function_exists('wp_generate_attachment_metadata')) {
                    require_once ABSPATH . 'wp-admin/includes/image.php';
                }
                if (!function_exists('media_handle_sideload')) {
                    require_once ABSPATH . 'wp-admin/includes/file.php';
                    require_once ABSPATH . 'wp-admin/includes/media.php';
                }
                
                if ($existing_att) {
                    // Update existing attachment
                    $attachment_id = $this->update_canva_attachment($existing_att, $tmp_file, $filename);
                    $was_wp_update = true;
                    $tmp_file = null; // File was moved by update function
                } else {
                    // Create new attachment
                    $file_array = [
                        'name'     => $filename,
                        'tmp_name' => $tmp_file,
                    ];
                    $attachment_id = media_handle_sideload($file_array, 0, $design_name . $page_suffix);
                    $tmp_file = null; // File was moved by media_handle_sideload
                    
                    // Ensure thumbnails are generated
                    if (!is_wp_error($attachment_id)) {
                        $file_path = get_attached_file($attachment_id);
                        if ($file_path && file_exists($file_path)) {
                            $metadata = wp_generate_attachment_metadata($attachment_id, $file_path);
                            if (!empty($metadata)) {
                                wp_update_attachment_metadata($attachment_id, $metadata);
                            }
                        }
                    }
                }

                if (!is_wp_error($attachment_id)) {
                    // Add tracking meta
                    update_post_meta($attachment_id, '_lightsync_source', 'canva');
                    update_post_meta($attachment_id, '_lightsync_asset_id', $asset_id);
                    update_post_meta($attachment_id, '_lightsync_canva_design_id', $design_id);
                    update_post_meta($attachment_id, '_lightsync_synced_at', gmdate('Y-m-d H:i:s'));
                    update_post_meta($attachment_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
                    update_post_meta($attachment_id, '_lightsync_rev', $content_hash);

                    // Track storage savings
                    if ($original_size > 0) {
                        update_post_meta($attachment_id, '_lightsync_original_bytes', $original_size);
                        update_post_meta($attachment_id, '_lightsync_optimized_bytes', $optimized_size);
                        self::bump_storage_stats($original_size, $optimized_size, 'canva');
                    }

                    if ($page_count > 1) {
                        update_post_meta($attachment_id, '_lightsync_canva_page', $page_num);
                        update_post_meta($attachment_id, '_lightsync_canva_total_pages', $page_count);
                    }

                    $imported_ids[] = $attachment_id;
                    $wp_count++;
                    if ($was_wp_update) {
                        $wp_updated++;
                    }
                } else {
                    // WordPress upload failed - track and log
                    $wp_failed++;
                    \LightSyncPro\Util\Logger::debug('[LSP Canva] WordPress upload failed: ' . $attachment_id->get_error_message());
                    self::add_activity(
                        sprintf('Canva → WordPress: Failed "%s" - %s', $design_name . $page_suffix, $attachment_id->get_error_message()),
                        'error',
                        'canva'
                    );
                }
            } elseif (($sync_target === 'wp' || $sync_target === 'both') && !$wp_needs_update && $existing_att) {
                // WordPress skipped (unchanged) - still track the ID and update sync time
                $imported_ids[] = $existing_att;
                $wp_skipped++;
                
                // Update sync timestamp so "Updated" indicator clears
                update_post_meta($existing_att, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
            }


            // Sync to Shopify if target is 'shopify' or 'both'
            if (($sync_target === 'shopify' || $sync_target === 'both') && self::get_opt('shopify_connected') && $file_bytes) {
                \LightSyncPro\Util\Logger::debug('[LSP Canva→Shopify] sync_target=' . $sync_target . ', bytes_len=' . strlen($file_bytes) . ', content_hash=' . substr($content_hash, 0, 16) . '...');

                $shopify_result = $this->sync_canva_to_shopify_bytes($file_bytes, $filename, $design_name . $page_suffix, $asset_id, $content_hash);

                \LightSyncPro\Util\Logger::debug('[LSP Canva→Shopify] result=' . wp_json_encode($shopify_result));

                if (!is_wp_error($shopify_result) && !empty($shopify_result['ok'])) {
                    if (!empty($shopify_result['skipped'])) {
                        // Shopify skipped (unchanged)
                        $shopify_skipped = ($shopify_skipped ?? 0) + 1;
                    } else {
                        $shopify_count = ($shopify_count ?? 0) + 1;
                        if (!empty($shopify_result['updated'])) {
                            $shopify_updated_count = ($shopify_updated_count ?? 0) + 1;
                        }
                    }
                    // If only syncing to Shopify, track the ID
                    if ($sync_target === 'shopify') {
                        $imported_ids[] = 'shopify:' . $asset_id;
                    }
                } elseif (is_wp_error($shopify_result)) {
                    $shopify_failed = ($shopify_failed ?? 0) + 1;
                }
            }

            // Sync to Hub if target is 'hub'
            if ($sync_target === 'hub' && function_exists('lsp_hub_sync_asset') && $file_bytes) {
                \LightSyncPro\Util\Logger::debug('[LSP Canva→Hub] Starting hub sync, bytes_len=' . strlen($file_bytes) . ', filename=' . $filename);
                
                // Get selected Hub sites from options
                $hub_site_ids = (array) self::get_opt('hub_selected_sites', []);
                
                // Determine content type from actual filename (may be .avif, .webp, or .png)
                $content_type = wp_check_filetype($filename)['type'] ?? 'image/png';
                
                $hub_result = lsp_hub_sync_asset([
                    'image_data' => $file_bytes,
                    'filename' => $filename,
                    'content_type' => $content_type,
                    'title' => $design_name . $page_suffix,
                    'alt_text' => '',
                    'caption' => '',
                ], $hub_site_ids, 'canva', $asset_id);
                
                \LightSyncPro\Util\Logger::debug('[LSP Canva→Hub] result=' . wp_json_encode($hub_result));
                
                if (!empty($hub_result['success'])) {
                    $imported_ids[] = 'hub:' . $asset_id;
                    $hub_synced_count = $hub_result['synced'] ?? 0;
                    $hub_skipped_count = $hub_result['skipped'] ?? 0;
                    
                    // Track Hub distribution for weekly digest
                    if (class_exists('\\LightSyncPro\\Admin\\WeeklyDigest')) {
                        WeeklyDigest::track_hub_result($hub_result, 'canva');
                    }
                    
                    if ($hub_synced_count > 0) {
                        self::add_activity(
                            sprintf('Canva → Hub: Synced "%s" to %d site(s)', $design_name . $page_suffix, $hub_synced_count),
                            'success',
                            'Canva'
                        );
                    } elseif ($hub_skipped_count > 0) {
                        self::add_activity(
                            sprintf('Canva → Hub: Skipped "%s" (unchanged on %d site(s))', $design_name . $page_suffix, $hub_skipped_count),
                            'info',
                            'Canva'
                        );
                    }
                } elseif (!empty($hub_result['error'])) {
                    self::add_activity(
                        sprintf('Canva → Hub: Failed "%s" - %s', $design_name . $page_suffix, $hub_result['error']),
                        'error',
                        'Canva'
                    );
                } elseif (($hub_result['failed'] ?? 0) > 0) {
                    self::add_activity(
                        sprintf('Canva → Hub: Failed "%s" on %d site(s)', $design_name . $page_suffix, $hub_result['failed']),
                        'error',
                        'Canva'
                    );
                }
            }

            // Clean up temp file if it still exists
            if ($tmp_file && file_exists($tmp_file)) {
                @unlink($tmp_file);
            }
        }

        // Consider success if anything was processed (including skipped unchanged files)
        if (empty($imported_ids)) {
            return new \WP_Error('import_failed', 'No pages could be imported');
        }

        return [
            'attachment_ids'   => $imported_ids,
            'pages'            => count($imported_ids),
            'wp_count'         => $wp_count,
            'wp_updated'       => $wp_updated,
            'wp_skipped'       => $wp_skipped,
            'wp_failed'        => $wp_failed,
            'sync_target'      => $sync_target,
            'is_resync'        => $is_update,  // Was this a re-sync of existing design?
        ];
        } catch (\Exception $e) {
            return new \WP_Error('sync_exception', 'Exception: ' . $e->getMessage());
        } catch (\Error $e) {
            return new \WP_Error('sync_error', 'Error: ' . $e->getMessage());
        }
    }

    /**
     * Get Canva design info by ID
     */
    private function get_canva_design_info($design_id) {
        $designs = \LightSyncPro\OAuth\CanvaOAuth::get_all_designs(10);
        if (is_wp_error($designs) || empty($designs['items'])) {
            return ['title' => 'Canva Design'];
        }

        foreach ($designs['items'] as $d) {
            if ($d['id'] === $design_id) {
                return $d;
            }
        }

        return ['title' => 'Canva Design'];
    }

    /**
     * Find existing attachment by Canva asset ID (only active attachments)
     */
    private function find_canva_attachment($asset_id) {
        global $wpdb;
        
        $att_id = $wpdb->get_var($wpdb->prepare(
            "SELECT pm.post_id FROM {$wpdb->postmeta} pm
             INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID AND p.post_status = 'inherit'
             WHERE pm.meta_key = '_lightsync_asset_id' 
             AND pm.meta_value = %s 
             LIMIT 1",
            $asset_id
        ));

        return $att_id ? (int)$att_id : 0;
    }

    /**
     * Update existing Canva attachment with new file
     */
    private function update_canva_attachment($attachment_id, $tmp_file, $filename) {
        // Ensure image functions are loaded
        if (!function_exists('wp_generate_attachment_metadata')) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }

        // Get current file path — overwrite in place to preserve URL & Shopify mapping
        $current_path = get_attached_file($attachment_id);
        if (!$current_path) {
            return new \WP_Error('no_attached_file', 'Cannot find current attachment file path');
        }

        $current_ext = strtolower(pathinfo($current_path, PATHINFO_EXTENSION));
        $new_ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $target_path = $current_path;

        // If extension changed (e.g. PNG → WebP), adjust path but keep same directory/base
        if ($current_ext !== $new_ext) {
            $target_path = preg_replace('/\.' . preg_quote($current_ext, '/') . '$/', '.' . $new_ext, $current_path);
        }

        // Delete old thumbnails BEFORE overwriting
        $old_meta = wp_get_attachment_metadata($attachment_id);
        if (!empty($old_meta['sizes'])) {
            $old_dir = dirname($current_path);
            foreach ($old_meta['sizes'] as $size) {
                $thumb = $old_dir . '/' . $size['file'];
                if (file_exists($thumb)) {
                    @unlink($thumb);
                }
            }
        }

        // Write new file to same path (overwrite in place)
        if (!copy($tmp_file, $target_path)) {
            return new \WP_Error('copy_failed', 'Could not overwrite attachment file');
        }

        // Clean up temp file
        if (file_exists($tmp_file) && $tmp_file !== $target_path) {
            @unlink($tmp_file);
        }

        // If extension changed, remove old file and update path
        if ($target_path !== $current_path) {
            if (file_exists($current_path)) {
                @unlink($current_path);
            }
            update_attached_file($attachment_id, $target_path);
        }

        // Update MIME type
        $mime_type = wp_check_filetype($target_path);
        if (!empty($mime_type['type'])) {
            wp_update_post([
                'ID' => $attachment_id,
                'post_mime_type' => $mime_type['type'],
            ]);
        }

        // Regenerate thumbnails
        $metadata = wp_generate_attachment_metadata($attachment_id, $target_path);
        wp_update_attachment_metadata($attachment_id, $metadata);

        return $attachment_id;
    }

    /**
     * Compress Canva image to WebP
     */
    private function compress_canva_image($file_path, $filename) {
        if (!file_exists($file_path)) {
            return false;
        }

        if (!preg_match('/\.[^.]+$/', $filename)) {
            $filename .= '.png';
        }

        try {
            $quality = 82;
            $webp_path = preg_replace('/\.[^.]+$/', '.webp', $file_path);
            $image = wp_get_image_editor($file_path);
            if (!is_wp_error($image)) {
                $image->set_quality($quality);
                $result = $image->save($webp_path, 'image/webp');
                if (!is_wp_error($result) && !empty($result['path'])) {
                    $new_filename = preg_replace('/\.[^.]+$/', '.webp', $filename);
                    return [
                        'path' => $result['path'],
                        'filename' => $new_filename,
                    ];
                }
            }
        } catch (\Exception $e) {
            \LightSyncPro\Util\Logger::debug('[LSP Canva] WebP compression failed: ' . $e->getMessage());
        } catch (\Error $e) {
            \LightSyncPro\Util\Logger::debug('[LSP Canva] WebP compression error: ' . $e->getMessage());
        }

        return false;
    }

    /**
     * Start Canva background sync
     */
    public function ajax_canva_background_sync() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $design_ids = isset($_POST['design_ids']) ? array_map('sanitize_text_field', (array)$_POST['design_ids']) : [];
        $sync_target = isset($_POST['sync_target']) ? sanitize_text_field($_POST['sync_target']) : 'wp';

        if (empty($design_ids)) {
            wp_send_json_error(['error' => 'No designs selected']);
        }

        // Store sync request in option (more reliable than transient for background)
        $sync_data = [
            'design_ids'     => $design_ids,
            'sync_target'    => $sync_target,
            'started_at'     => time(),
            'running'        => true,
            'synced'         => 0,
            'total_pages'    => 0,
            'errors'         => [],
            'completed_ids'  => [],
            'current_index'  => 0,
        ];
        update_option('lsp_canva_background_sync', $sync_data, false);

        // Send success response immediately
        wp_send_json_success(['started' => true, 'count' => count($design_ids)]);
    }

    /**
     * Check Canva background sync status and process next item
     */
    public function ajax_canva_background_status() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $sync_data = get_option('lsp_canva_background_sync');

        if (!$sync_data || empty($sync_data['running'])) {
            wp_send_json_success(['running' => false]);
        }

        // Process next design in queue (piggyback on status check)
        $this->process_canva_background_tick($sync_data);

        // Re-fetch updated data
        $sync_data = get_option('lsp_canva_background_sync');

        wp_send_json_success([
            'running'       => !empty($sync_data['running']),
            'synced'        => (int)($sync_data['synced'] ?? 0),
            'total'         => count($sync_data['design_ids'] ?? []),
            'current'       => (int)($sync_data['current_index'] ?? 0),
            'errors'        => $sync_data['errors'] ?? [],
            'design_ids'    => $sync_data['completed_ids'] ?? [],
            'total_pages'   => (int)($sync_data['total_pages'] ?? 0),
            'wp_count'      => (int)($sync_data['wp_count'] ?? 0),
            'sync_target'   => $sync_data['sync_target'] ?? 'wp',
            'lightsync_last_sync_ts' => (int) self::get_opt('lightsync_last_sync_ts', 0),
        ]);
    }

    /**
     * Process one design in background queue
     */
    private function process_canva_background_tick(&$sync_data) {
        $design_ids = $sync_data['design_ids'] ?? [];
        $current_index = (int)($sync_data['current_index'] ?? 0);

        // Initialize counts if not present
        if (!isset($sync_data['wp_count'])) $sync_data['wp_count'] = 0;

        // Already done? (shouldn't reach here due to running=false check, but safety)
        if ($current_index >= count($design_ids)) {
            $sync_data['running'] = false;
            update_option('lsp_canva_background_sync', $sync_data, false);
            return;
        }

        // Check for lock to prevent parallel processing
        $lock_key = 'lsp_canva_bg_lock';
        if (get_transient($lock_key)) {
            return; // Another request is processing
        }
        set_transient($lock_key, true, 120); // 2 minute lock

        try {
            // Extend timeout for this request
            @set_time_limit(120);

            $design_id = $design_ids[$current_index];
            // Use sync_target from the background sync request, fall back to saved option
            $sync_target = $sync_data['sync_target'] ?? (self::get_opt('canva_sync_target') ?: 'wp');

            // Sync this design
            $result = $this->sync_canva_design($design_id, $sync_target);

            if (is_wp_error($result)) {
                $sync_data['errors'][] = $design_id . ': ' . $result->get_error_message();
            } else {
                $sync_data['synced']++;
                $sync_data['completed_ids'][] = $design_id;
                $pages = (int)($result['pages'] ?? 1);
                $sync_data['total_pages'] += $pages;
                
                // Track WP counts
                $sync_data['wp_count'] += (int)($result['wp_count'] ?? 0);
                
                // Bump usage
                self::usage_bump($pages);
            }

            $sync_data['current_index'] = $current_index + 1;

            // Check if done - log activity HERE, not on next tick
            if ($sync_data['current_index'] >= count($design_ids)) {
                $sync_data['running'] = false;
                
                // Log completion activity NOW (before next poll sees running=false)
                if ($sync_data['synced'] > 0) {
                    $dest_parts = [];
                    if ($sync_data['wp_count'] > 0) $dest_parts[] = 'WordPress';
                    $dest_label = implode(' & ', $dest_parts) ?: 'WordPress';
                    
                    $page_text = $sync_data['total_pages'] > $sync_data['synced'] ? " ({$sync_data['total_pages']} pages)" : '';
                    self::add_activity(
                        "Canva → {$dest_label} Sync Complete (Background): {$sync_data['synced']} design(s){$page_text}",
                        'success',
                        'canva'
                    );
                    
                    // Update last sync timestamp
                    self::set_opt([
                        'lightsync_last_sync_ts'     => time(),
                        'lightsync_last_sync_source' => 'canva',
                        'lightsync_last_sync_status' => 'complete',
                    ]);
                }
            }

            update_option('lsp_canva_background_sync', $sync_data, false);
        } catch (\Exception $e) {
            $sync_data['errors'][] = $design_ids[$current_index] . ': Exception - ' . $e->getMessage();
            $sync_data['current_index'] = $current_index + 1;
            if ($sync_data['current_index'] >= count($design_ids)) {
                $sync_data['running'] = false;
            }
            update_option('lsp_canva_background_sync', $sync_data, false);
        } catch (\Error $e) {
            $sync_data['errors'][] = $design_ids[$current_index] . ': Error - ' . $e->getMessage();
            $sync_data['current_index'] = $current_index + 1;
            if ($sync_data['current_index'] >= count($design_ids)) {
                $sync_data['running'] = false;
            }
            update_option('lsp_canva_background_sync', $sync_data, false);
        }
        
        // Always release lock
        delete_transient($lock_key);
    }

    /**
     * Run background Canva sync (called by cron)
     */
    /**
     * Legacy cron handler - no longer used, kept for compatibility
     * Background sync now uses polling-based tick processing
     */
    public static function canva_background_sync_run($design_ids) {
        // No-op - background sync is now processed via status polling or cron
        return;
    }

    /**
     * Process Canva background sync queue (cron-based for autosync)
     */
    public function process_canva_background_queue() {
        \LightSyncPro\Util\Logger::debug('[LSP Canva Background] === QUEUE PROCESSOR STARTED ===');
        
        $sync_data = get_option('lsp_canva_background_sync');
        
        if (!$sync_data || empty($sync_data['running'])) {
            \LightSyncPro\Util\Logger::debug('[LSP Canva Background] No active sync, nothing to process');
            return;
        }
        
        $design_ids = $sync_data['design_ids'] ?? [];
        $current_index = (int) ($sync_data['current_index'] ?? 0);
        
        if ($current_index >= count($design_ids)) {
            // All done
            \LightSyncPro\Util\Logger::debug('[LSP Canva Background] All designs processed');
            $sync_data['running'] = false;
            update_option('lsp_canva_background_sync', $sync_data, false);
            
            // Add completion activity
            $synced = $sync_data['synced'] ?? 0;
            if ($synced > 0) {
                $sync_target = $sync_data['sync_target'] ?? 'wp';
                $dest_label = 'WordPress';
                
                self::add_activity(
                    "Canva → {$dest_label} Sync Complete (Background): {$synced} design(s)",
                    'success',
                    'canva'
                );
                
                self::set_opt([
                    'lightsync_last_sync_ts'     => time(),
                    'lightsync_last_sync_source' => 'canva',
                    'lightsync_last_sync_status' => 'complete',
                ]);
            }
            return;
        }
        
        $design_id = $design_ids[$current_index];
        $sync_target = $sync_data['sync_target'] ?? 'wp';
        
        \LightSyncPro\Util\Logger::debug('[LSP Canva Background] Processing design: ' . $design_id . ' (target: ' . $sync_target . ')');
        
        // Process this design using existing tick function
        $this->process_canva_background_tick($sync_data);
        
        // Re-fetch and check if more to process
        $sync_data = get_option('lsp_canva_background_sync');
        
        if (!empty($sync_data['running'])) {
            // Schedule next item
            wp_schedule_single_event(time() + 3, 'lightsync_process_canva_queue');
            \LightSyncPro\Util\Logger::debug('[LSP Canva Background] Scheduled next item in 3 seconds');
        }
    }

    /* ==================== END CANVA AJAX HANDLERS ==================== */

    /* ==================== FIGMA AJAX HANDLERS ==================== */

    /**
     * Get stored Figma files and their info
     */
    public function ajax_figma_get_files() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\FigmaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Figma not connected']);
        }

        $files = (array) self::get_opt('figma_files', []);
        $refresh_after = 4 * HOUR_IN_SECONDS; // Refresh thumbnails older than 4 hours
        $now = time();
        $updated = false;

        foreach ($files as $file_key => &$file) {
            $last_refresh = $file['thumbnail_refreshed'] ?? ($file['added_at'] ?? 0);
            
            // Refresh if thumbnail is stale or missing
            if (($now - $last_refresh) > $refresh_after || empty($file['thumbnail'])) {
                $file_meta = \LightSyncPro\OAuth\FigmaOAuth::get_file_meta($file_key);
                if (!is_wp_error($file_meta)) {
                    $file_data = $file_meta['file'] ?? $file_meta;
                    $file['name'] = $file_data['name'] ?? $file['name'];
                    $file['thumbnail'] = $file_data['thumbnail_url'] ?? $file_data['thumbnailUrl'] ?? '';
                    $file['thumbnail_refreshed'] = $now;
                    $updated = true;
                }
                usleep(100000); // 100ms rate limit protection
            }
        }

        if ($updated) {
            self::set_opt(['figma_files' => $files]);
        }

        wp_send_json_success([
            'files' => array_values($files),
        ]);
    }

    /**
     * Add a Figma file by URL/key
     */
    public function ajax_figma_add_file() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\FigmaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Figma not connected']);
        }

        $url_or_key = isset($_POST['file_url']) ? sanitize_text_field($_POST['file_url']) : '';
        if (!$url_or_key) {
            wp_send_json_error(['error' => 'No file URL provided']);
        }

        $file_key = \LightSyncPro\OAuth\FigmaOAuth::parse_file_key($url_or_key);
        if (!$file_key) {
            wp_send_json_error(['error' => 'Invalid Figma file URL']);
        }

        // Get file metadata
        $file_meta = \LightSyncPro\OAuth\FigmaOAuth::get_file_meta($file_key);
        if (is_wp_error($file_meta)) {
            wp_send_json_error(['error' => $file_meta->get_error_message()]);
        }

        // Figma meta endpoint wraps response in 'file' object
        $file_data = $file_meta['file'] ?? $file_meta;

        // Store file info
        $files = (array) self::get_opt('figma_files', []);
        
        // Check if already added
        if (isset($files[$file_key])) {
            wp_send_json_error(['error' => 'This file is already added']);
        }

        // Figma meta uses snake_case (thumbnail_url, last_modified)
        $files[$file_key] = [
            'key'        => $file_key,
            'name'       => $file_data['name'] ?? 'Untitled',
            'thumbnail'  => $file_data['thumbnail_url'] ?? $file_data['thumbnailUrl'] ?? '',
            'lastModified' => $file_data['last_modified'] ?? $file_data['lastModified'] ?? null,
            'added_at'   => time(),
        ];

        self::set_opt(['figma_files' => $files]);

        wp_send_json_success([
            'file' => $files[$file_key],
            'files' => array_values($files),
        ]);
    }

    /**
     * Remove a Figma file from LightSync Pro
     */
    public function ajax_figma_remove_file() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $file_key = isset($_POST['file_key']) ? sanitize_text_field($_POST['file_key']) : '';
        if (!$file_key) {
            wp_send_json_error(['error' => 'No file key provided']);
        }

        $files = (array) self::get_opt('figma_files', []);
        
        if (!isset($files[$file_key])) {
            wp_send_json_error(['error' => 'File not found']);
        }

        // Remove the file
        unset($files[$file_key]);
        self::set_opt(['figma_files' => $files]);
        
        // Also remove selected frames for this file
        delete_option('lightsync_figma_selected_frames_' . $file_key);

        wp_send_json_success([
            'removed' => $file_key,
            'files' => array_values($files),
        ]);
    }

    /**
     * Refresh metadata for all Figma files (fix Untitled issues)
     */
    public function ajax_figma_refresh_files() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\FigmaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Figma not connected']);
        }

        $files = (array) self::get_opt('figma_files', []);
        $updated = 0;
        $errors = [];

        foreach ($files as $file_key => &$file) {
            // Skip if already has a real name
            if (!empty($file['name']) && $file['name'] !== 'Untitled') {
                continue;
            }

            $file_meta = \LightSyncPro\OAuth\FigmaOAuth::get_file_meta($file_key);
            if (!is_wp_error($file_meta)) {
                // Figma meta endpoint wraps response in 'file' object and uses snake_case
                $file_data = $file_meta['file'] ?? $file_meta;
                $file['name'] = $file_data['name'] ?? $file['name'];
                $file['thumbnail'] = $file_data['thumbnail_url'] ?? $file_data['thumbnailUrl'] ?? $file['thumbnail'];
                $file['lastModified'] = $file_data['last_modified'] ?? $file_data['lastModified'] ?? $file['lastModified'];
                $updated++;
            } else {
                $errors[] = $file_key . ': ' . $file_meta->get_error_message();
            }

            // Rate limit protection
            usleep(200000); // 200ms between requests
        }

        self::set_opt(['figma_files' => $files]);

        wp_send_json_success([
            'updated' => $updated,
            'errors' => $errors,
            'files' => array_values($files),
        ]);
    }

    /**
     * Clear Figma rate limit cooldown
     */
    public function ajax_figma_clear_rate_limit() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        \LightSyncPro\OAuth\FigmaOAuth::clear_rate_limit();
        
        // Also clear any stale frame caches
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient%lsp_figma%'");

        wp_send_json_success(['cleared' => true]);
    }

    /**
     * Queue Figma frames for background sync
     */
    public function ajax_figma_queue_sync() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $file_key = isset($_POST['file_key']) ? sanitize_text_field($_POST['file_key']) : '';
        $frame_ids = isset($_POST['frame_ids']) ? array_map('sanitize_text_field', (array)$_POST['frame_ids']) : [];
        $format = isset($_POST['format']) ? sanitize_text_field($_POST['format']) : 'png';
        $scale = isset($_POST['scale']) ? floatval($_POST['scale']) : 2;
        $sync_target = isset($_POST['sync_target']) ? sanitize_text_field($_POST['sync_target']) : 'wp';

        if (empty($file_key) || empty($frame_ids)) {
            wp_send_json_error(['error' => 'Missing file key or frame IDs']);
        }

        // Get existing queue or create new
        $queue = get_option('lightsync_figma_sync_queue', []);
        
        // Add to queue
        foreach ($frame_ids as $frame_id) {
            $queue[] = [
                'file_key'    => $file_key,
                'frame_id'    => $frame_id,
                'format'      => $format,
                'scale'       => $scale,
                'sync_target' => $sync_target,
                'queued_at'   => time(),
            ];
        }

        update_option('lightsync_figma_sync_queue', $queue, false);
        
        // Track progress info - don't reset total if already running
        $progress = get_option('lightsync_figma_sync_progress', []);
        $was_running = $progress['running'] ?? false;
        
        if (!$was_running) {
            // Fresh start - reset counters
            $progress['total'] = count($queue);
            $progress['synced'] = 0;
            $progress['started_at'] = time();
        } else {
            // Already running - update total to current queue size
            $progress['total'] = count($queue);
        }
        
        $progress['running'] = true;
        $progress['sync_target'] = $sync_target;
        update_option('lightsync_figma_sync_progress', $progress, false);

        // Schedule background processing if not already scheduled
        if (!wp_next_scheduled('lightsync_process_figma_queue')) {
            wp_schedule_single_event(time() + 5, 'lightsync_process_figma_queue');
        }

        wp_send_json_success([
            'queued' => count($frame_ids),
            'total_queue' => count($queue),
        ]);
    }

    /**
     * Get Figma background sync status for polling
     */
    public function ajax_figma_background_status() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $progress = get_option('lightsync_figma_sync_progress', []);
        $queue = get_option('lightsync_figma_sync_queue', []);

        // Calculate progress
        $total = $progress['total'] ?? 0;
        $synced = $progress['synced'] ?? 0;
        $running = !empty($queue) || ($progress['running'] ?? false);
        
        // If queue is empty, mark as not running
        if (empty($queue) && $running) {
            $progress['running'] = false;
            update_option('lightsync_figma_sync_progress', $progress, false);
            $running = false;
        }

        wp_send_json_success([
            'total'       => $total,
            'synced'      => $synced,
            'remaining'   => count($queue),
            'running'     => $running,
            'sync_target' => $progress['sync_target'] ?? 'wp',
        ]);
    }

    /**
     * Check for updates in synced Figma files
     */
    public function ajax_figma_check_updates() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\FigmaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Figma not connected']);
        }

        $files = (array) self::get_opt('figma_files', []);
        
        if (empty($files)) {
            wp_send_json_success(['updates' => [], 'message' => 'No files to check']);
        }

        $updates = [];
        $files_with_updates = 0;
        $frames_need_update = 0;
        $debug_info = [];

        foreach ($files as $file_key => $file_info) {
            // Get current file metadata from Figma (lightweight call)
            $meta = \LightSyncPro\OAuth\FigmaOAuth::get_file_meta($file_key);
            
            if (is_wp_error($meta)) {
                $debug_info[$file_key] = ['error' => $meta->get_error_message()];
                continue;
            }

            $current_modified = $meta['file']['lastModified'] ?? $meta['lastModified'] ?? null;
            if (!$current_modified) {
                $debug_info[$file_key] = ['error' => 'No lastModified in response', 'meta_keys' => array_keys($meta)];
                continue;
            }

            // Get synced frames for this file and compare
            $synced_info = $this->get_figma_synced_frames_info($file_key, $current_modified);
            
            $file_debug = [
                'figma_last_modified' => $current_modified,
                'synced_frames' => count($synced_info),
                'frame_versions' => [],
            ];
            
            $file_updates = 0;
            foreach ($synced_info as $node_id => $info) {
                $file_debug['frame_versions'][$node_id] = [
                    'stored_version' => $info['file_version'] ?? 'NOT SET',
                    'needs_update' => $info['needs_update'],
                ];
                if (!empty($info['needs_update'])) {
                    $file_updates++;
                    $frames_need_update++;
                }
            }
            
            $debug_info[$file_key] = $file_debug;

            if ($file_updates > 0) {
                $files_with_updates++;
                $updates[$file_key] = [
                    'name' => $file_info['name'] ?? $file_key,
                    'frames_need_update' => $file_updates,
                    'last_modified' => $current_modified,
                ];
            }
            
            // Update cached file info with new lastModified
            // Clear cache so next load shows updated status
            delete_transient('lsp_figma_frames_' . $file_key . '_nested');
            delete_transient('lsp_figma_frames_' . $file_key . '_top');
        }

        $message = $frames_need_update > 0 
            ? "{$frames_need_update} frame(s) in {$files_with_updates} file(s) have updates available"
            : 'All synced frames are up to date';

        wp_send_json_success([
            'updates' => $updates,
            'files_with_updates' => $files_with_updates,
            'frames_need_update' => $frames_need_update,
            'message' => $message,
            'debug' => $debug_info,
        ]);
    }

    /**
     * Process Figma background sync queue
     */
    public function process_figma_background_queue() {
        $queue = get_option('lightsync_figma_sync_queue', []);
        
        if (empty($queue)) {
            return;
        }

        \LightSyncPro\Util\Logger::debug('[LSP Figma] Processing background queue: ' . count($queue) . ' items');

        // Process up to 5 items at a time to avoid timeouts
        $batch = array_splice($queue, 0, 5);
        update_option('lightsync_figma_sync_queue', $queue, false);

        // Group by file_key for efficient export
        $by_file = [];
        foreach ($batch as $item) {
            $file_key = $item['file_key'];
            if (!isset($by_file[$file_key])) {
                $by_file[$file_key] = [
                    'frame_ids' => [],
                    'format'    => $item['format'],
                    'scale'     => $item['scale'],
                    'sync_target' => $item['sync_target'],
                ];
            }
            $by_file[$file_key]['frame_ids'][] = $item['frame_id'];
        }

        $total_synced = 0;
        $files_synced = [];
        $synced_by_type = []; // Track element types across all files

        // Process each file
        foreach ($by_file as $file_key => $data) {
            $format = $data['format'];
            $scale = $data['scale'];
            $sync_target = $data['sync_target'];
            $frame_ids = $data['frame_ids'];

            // Get file info for lastModified
            $file_info = \LightSyncPro\OAuth\FigmaOAuth::get_file_meta($file_key);
            $last_modified = !is_wp_error($file_info) ? ($file_info['file']['lastModified'] ?? $file_info['lastModified'] ?? null) : null;

            // Export images from Figma
            $figma_format = in_array($format, ['webp', 'avif'], true) ? 'png' : $format;
            $images = \LightSyncPro\OAuth\FigmaOAuth::export_images($file_key, $frame_ids, $figma_format, $scale);

            if (is_wp_error($images)) {
                \LightSyncPro\Util\Logger::debug('[LSP Figma] Background export failed: ' . $images->get_error_message());
                continue;
            }

            // Get frame names and types
            $frames_result = \LightSyncPro\OAuth\FigmaOAuth::get_file_frames($file_key, true);
            $frames_list = is_wp_error($frames_result) ? [] : ($frames_result['frames'] ?? []);
            $frame_names = [];
            $frame_types = [];
            foreach ($frames_list as $f) {
                $frame_names[$f['id']] = $f['name'];
                $frame_types[$f['id']] = $f['type'] ?? 'FRAME';
            }

            // Import each frame
            $file_synced = 0;
            foreach ($images as $node_id => $image_url) {
                $frame_name = $frame_names[$node_id] ?? 'Figma Frame';
                $result = $this->import_figma_frame($file_key, $node_id, $image_url, $frame_name, $format, $sync_target, $last_modified);
                if (!is_wp_error($result)) {
                    $file_synced++;
                    $total_synced++;
                    
                    // Track by type
                    $type = $frame_types[$node_id] ?? 'FRAME';
                    $synced_by_type[$type] = ($synced_by_type[$type] ?? 0) + 1;
                    
                    // Update progress
                    $progress = get_option('lightsync_figma_sync_progress', []);
                    $progress['synced'] = ($progress['synced'] ?? 0) + 1;
                    update_option('lightsync_figma_sync_progress', $progress, false);
                }
            }
            
            if ($file_synced > 0) {
                $files_synced[] = $file_key;
                // Clear frame cache for this file
                // Cache key format: lsp_figma_frames_{file_key}_{nested|top}
                delete_transient('lsp_figma_frames_' . $file_key . '_nested');
                delete_transient('lsp_figma_frames_' . $file_key . '_top');
            }
        }

        // Update last sync header and add activity if we synced anything
        if ($total_synced > 0) {
            self::usage_consume($total_synced);
            
            $dest_label = 'WordPress';
            
            // Build type breakdown string
            $type_labels = [
                'FRAME' => 'frame',
                'COMPONENT' => 'component',
                'COMPONENT_SET' => 'component set',
                'GROUP' => 'group',
                'RECTANGLE' => 'rectangle',
                'ELLIPSE' => 'ellipse',
                'TEXT' => 'text',
                'VECTOR' => 'vector',
                'INSTANCE' => 'instance',
            ];
            
            $type_parts = [];
            foreach ($synced_by_type as $type => $count) {
                $label = $type_labels[$type] ?? strtolower($type);
                $type_parts[] = $count . ' ' . $label . ($count > 1 ? 's' : '');
            }
            $type_breakdown = !empty($type_parts) ? ' (' . implode(', ', $type_parts) . ')' : '';
            
            self::add_activity(
                "Figma → {$dest_label} Sync Complete (Background): {$total_synced} element(s){$type_breakdown}",
                'success',
                'figma'
            );
            
            self::set_opt([
                'lightsync_last_sync_ts'     => time(),
                'lightsync_last_sync_source' => 'figma',
                'lightsync_last_sync_status' => 'complete',
            ]);
        }

        // If there are more items, schedule another run
        if (!empty($queue)) {
            wp_schedule_single_event(time() + 5, 'lightsync_process_figma_queue');
        } else {
            // Mark as complete
            $progress = get_option('lightsync_figma_sync_progress', []);
            $progress['running'] = false;
            update_option('lightsync_figma_sync_progress', $progress, false);
        }

        \LightSyncPro\Util\Logger::debug('[LSP Figma] Background batch complete. Synced: ' . $total_synced . ', Remaining: ' . count($queue));
    }

    /**
     * Process Dropbox background sync queue
     */
    public function process_dropbox_background_queue() {
        \LightSyncPro\Util\Logger::debug('[LSP Dropbox Background] === QUEUE PROCESSOR STARTED ===');
        
        $queue = get_option('lightsync_dropbox_sync_queue', []);
        
        if (empty($queue)) {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox Background] Queue is empty, nothing to process');
            return;
        }

        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Processing background queue: ' . count($queue) . ' items');

        // Process one at a time for Dropbox (downloads can be large)
        $item = array_shift($queue);
        update_option('lightsync_dropbox_sync_queue', $queue, false);

        $file_id = $item['file_id'];
        $file_path = $item['file_path'];
        $file_name = $item['file_name'];
        $sync_target = $item['sync_target'] ?? 'wp';

        \LightSyncPro\Util\Logger::debug('[LSP Dropbox Background] Syncing file: ' . $file_name . ' (target: ' . $sync_target . ', path: ' . $file_path . ')');

        $result = $this->sync_dropbox_file($file_id, $file_path, $file_name, $sync_target);

        if (!is_wp_error($result)) {
            // Update progress
            $progress = get_option('lightsync_dropbox_sync_progress', []);
            $progress['synced'] = ($progress['synced'] ?? 0) + 1;
            
            // Track synced file IDs
            if (!isset($progress['synced_ids'])) {
                $progress['synced_ids'] = [];
            }
            $progress['synced_ids'][] = $file_id;
            
            update_option('lightsync_dropbox_sync_progress', $progress, false);

            self::usage_consume(1);

            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Background synced: ' . $file_name);
        } else {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Background sync failed: ' . $result->get_error_message());
        }

        // If there are more items, schedule another run
        if (!empty($queue)) {
            wp_schedule_single_event(time() + 2, 'lightsync_process_dropbox_queue');
        } else {
            // Mark as complete
            $progress = get_option('lightsync_dropbox_sync_progress', []);
            $progress['running'] = false;
            update_option('lightsync_dropbox_sync_progress', $progress, false);

            // Add activity log
            $synced = $progress['synced'] ?? 0;
            if ($synced > 0) {
                $dest_label = 'WordPress';
                
                self::add_activity(
                    "Dropbox → {$dest_label} Sync Complete (Background): {$synced} file(s)",
                    'success',
                    'dropbox'
                );
                
                self::set_opt([
                    'lightsync_last_sync_ts'     => time(),
                    'lightsync_last_sync_source' => 'dropbox',
                    'lightsync_last_sync_status' => 'complete',
                ]);
            }
        }

        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Background batch complete. Remaining: ' . count($queue));
    }

    /**
     * Get frames from a Figma file
     */
    public function ajax_figma_get_frames() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\FigmaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Figma not connected']);
        }

        $file_key = isset($_POST['file_key']) ? sanitize_text_field($_POST['file_key']) : '';
        $with_thumbnails = !empty($_POST['with_thumbnails']);
        $include_nested = !empty($_POST['include_nested']);
        $force_refresh = !empty($_POST['force_refresh']);
        
        if (!$file_key) {
            wp_send_json_error(['error' => 'No file key provided']);
        }

        // Check cache first (30 minute cache to respect Figma rate limits)
        $cache_key = 'lsp_figma_frames_' . $file_key . '_' . ($include_nested ? 'nested' : 'top');
        $cached = get_transient($cache_key);
        
        // Also check if we're rate limited
        $rate_limit_remaining = \LightSyncPro\OAuth\FigmaOAuth::rate_limit_remaining();
        
        if ($cached && !$force_refresh) {
            \LightSyncPro\Util\Logger::debug('[LSP Figma] Using cached frames for ' . $file_key);
            
            // Get previously selected frames for this file
            $selected = (array) self::get_opt('figma_selected_frames_' . $file_key, []);
            
            // Get synced frames status (always fresh, not cached)
            $synced_info = $this->get_figma_synced_frames_info($file_key, $cached['last_modified'] ?? null);
            
            wp_send_json_success([
                'file_name'      => $cached['file_name'],
                'thumbnail'      => $cached['thumbnail'],
                'last_modified'  => $cached['last_modified'] ?? null,
                'frames'         => $cached['frames'],
                'selected'       => $selected,
                'synced'         => $synced_info,
                'include_nested' => $include_nested,
                'cached'         => true,
                'rate_limit_remaining' => $rate_limit_remaining,
            ]);
        }

        $result = \LightSyncPro\OAuth\FigmaOAuth::get_file_frames($file_key, $include_nested);
        if (is_wp_error($result)) {
            $error_msg = $result->get_error_message();
            $error_data = $result->get_error_data();
            
            // Check for rate limit
            if (strpos($error_msg, '429') !== false || strpos(strtolower($error_msg), 'rate limit') !== false) {
                $retry_after = is_array($error_data) && isset($error_data['retry_after']) ? $error_data['retry_after'] : 60;
                
                // If we have cached data, use it even if expired
                $stale_cache = get_option('lsp_figma_frames_stale_' . $file_key);
                if ($stale_cache) {
                    $selected = (array) self::get_opt('figma_selected_frames_' . $file_key, []);
                    $synced_info = $this->get_figma_synced_frames_info($file_key, $stale_cache['last_modified'] ?? null);
                    wp_send_json_success([
                        'file_name'      => $stale_cache['file_name'],
                        'thumbnail'      => $stale_cache['thumbnail'],
                        'last_modified'  => $stale_cache['last_modified'] ?? null,
                        'frames'         => $stale_cache['frames'],
                        'selected'       => $selected,
                        'synced'         => $synced_info,
                        'include_nested' => $include_nested,
                        'cached'         => true,
                        'rate_limited'   => true,
                        'retry_after'    => $retry_after,
                    ]);
                }
                
                wp_send_json_error([
                    'error' => 'Figma rate limit reached. Please wait ' . $retry_after . ' seconds and try again.',
                    'retry_after' => $retry_after,
                ]);
            }
            
            wp_send_json_error(['error' => $error_msg]);
        }

        // Get previously selected frames for this file
        $selected = (array) self::get_opt('figma_selected_frames_' . $file_key, []);

        $frames = $result['frames'];
        
        // Cache the result (30 minutes for transient, also save stale copy)
        $cache_data = [
            'file_name'     => $result['file_name'],
            'thumbnail'     => $result['thumbnail'],
            'last_modified' => $result['last_modified'] ?? null,
            'frames'        => $frames,
        ];
        set_transient($cache_key, $cache_data, 30 * MINUTE_IN_SECONDS);
        update_option('lsp_figma_frames_stale_' . $file_key, $cache_data, false);
        
        // Optionally fetch thumbnails for each frame (expensive - skip by default)
        if ($with_thumbnails && !empty($frames)) {
            $frame_ids = array_column($frames, 'id');
            
            // Limit thumbnails to first 20 to avoid rate limits
            $thumbnail_ids = array_slice($frame_ids, 0, 20);
            
            // Fetch at small scale for thumbnails (0.5x = fast, reasonable quality)
            $thumbnails = \LightSyncPro\OAuth\FigmaOAuth::export_images($file_key, $thumbnail_ids, 'png', 0.5);
            
            if (!is_wp_error($thumbnails) && is_array($thumbnails)) {
                // Add thumbnail URL to each frame
                foreach ($frames as &$frame) {
                    $frame['thumbnail'] = $thumbnails[$frame['id']] ?? null;
                }
                unset($frame);
            }
        }

        // Get synced frames status to show "needs update" badges
        $synced_info = $this->get_figma_synced_frames_info($file_key, $result['last_modified'] ?? null);

        wp_send_json_success([
            'file_name'      => $result['file_name'],
            'thumbnail'      => $result['thumbnail'],
            'last_modified'  => $result['last_modified'] ?? null,
            'frames'         => $frames,
            'selected'       => $selected,
            'synced'         => $synced_info,
            'include_nested' => $include_nested,
            'cached'         => false,
        ]);
    }

    /**
     * Get info about synced Figma frames for a file
     */
    private function get_figma_synced_frames_info($file_key, $current_file_modified = null) {
        global $wpdb;
        
        // Get all attachments synced from this Figma file
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT p.ID, pm_node.meta_value as node_id, pm_ver.meta_value as file_version, pm_sync.meta_value as synced_at
                 FROM {$wpdb->posts} p
                 INNER JOIN {$wpdb->postmeta} pm_file ON p.ID = pm_file.post_id AND pm_file.meta_key = '_lightsync_figma_file_key'
                 INNER JOIN {$wpdb->postmeta} pm_node ON p.ID = pm_node.post_id AND pm_node.meta_key = '_lightsync_figma_node_id'
                 LEFT JOIN {$wpdb->postmeta} pm_ver ON p.ID = pm_ver.post_id AND pm_ver.meta_key = '_lightsync_figma_file_version'
                 LEFT JOIN {$wpdb->postmeta} pm_sync ON p.ID = pm_sync.post_id AND pm_sync.meta_key = '_lightsync_last_synced_at'
                 WHERE p.post_type = 'attachment'
                 AND p.post_status = 'inherit'
                 AND pm_file.meta_value = %s",
                $file_key
            )
        );

        $synced = [];
        
        // Get Shopify mappings for destination detection
        // Figma stores in Shopify table with format: figma-{file_key}-{node_id}
        $shopify_table = $wpdb->prefix . 'lightsync_shopify_files_map';
        $shop = self::get_opt('shopify_shop_domain', '');
        
        $shopify_synced = [];
        if ($shop && $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $shopify_table)) === $shopify_table) {
            // Get all Figma mappings for this shop (they start with 'figma-')
            $shopify_results = $wpdb->get_results($wpdb->prepare(
                "SELECT lr_asset_id FROM {$shopify_table} WHERE shop_domain = %s AND lr_asset_id LIKE %s AND shopify_file_id IS NOT NULL",
                $shop,
                'figma-' . $file_key . '-%'
            ));
            foreach ($shopify_results as $row) {
                // Extract node_id from asset_key format: figma-{file_key}-{node_id}
                $prefix = 'figma-' . $file_key . '-';
                if (strpos($row->lr_asset_id, $prefix) === 0) {
                    $node_id = substr($row->lr_asset_id, strlen($prefix));
                    $shopify_synced[$node_id] = true;
                }
            }
        }
        
        foreach ($results as $row) {
            $needs_update = false;
            
            // Compare file versions to determine if update needed
            if ($current_file_modified) {
                if (!$row->file_version) {
                    $needs_update = true;
                } else {
                    $synced_version = strtotime($row->file_version);
                    $current_version = strtotime($current_file_modified);
                    
                    if ($current_version && $synced_version && $current_version > $synced_version) {
                        $needs_update = true;
                    }
                }
            }
            
            // Determine destination
            $has_wp = true;
            $has_shopify = isset($shopify_synced[$row->node_id]);
            
            $dest = 'wp';
            if ($has_wp && $has_shopify) {
                $dest = 'both';
            } elseif ($has_shopify) {
                $dest = 'shopify';
            }
            
            $synced[$row->node_id] = [
                'attachment_id' => (int) $row->ID,
                'synced_at'     => $row->synced_at,
                'file_version'  => $row->file_version,
                'needs_update'  => $needs_update,
                'dest'          => $dest,
            ];
        }
        
        // Also check for Shopify-only syncs (frames synced to Shopify but not WordPress)
        foreach ($shopify_synced as $node_id => $v) {
            if (!isset($synced[$node_id])) {
                $synced[$node_id] = [
                    'attachment_id' => 0,
                    'synced_at'     => null,
                    'file_version'  => null,
                    'needs_update'  => false,
                    'dest'          => 'shopify',
                ];
            }
        }

        return $synced;
    }

    /**
     * Disconnect Figma
     */
    public function ajax_figma_disconnect() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        \LightSyncPro\OAuth\FigmaOAuth::disconnect();
        
        // Also clear stored files
        self::set_opt([
            'figma_files' => [],
        ]);

        wp_send_json_success(['disconnected' => true]);
    }

    /**
     * Disconnect Dropbox
     */
    public function ajax_dropbox_disconnect() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        \LightSyncPro\OAuth\DropboxOAuth::disconnect();
        
        // Clear any stored folder selections
        self::set_opt([
            'dropbox_selected_folders' => [],
        ]);

        wp_send_json_success(['disconnected' => true]);
    }

    /**
     * Disconnect OpenRouter AI
     */
    public function ajax_openrouter_disconnect() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        \LightSyncPro\OAuth\OpenRouterOAuth::disconnect();

        wp_send_json_success(['disconnected' => true]);
    }

    /**
     * Disconnect Shutterstock
     */
    public function ajax_shutterstock_disconnect() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        \LightSyncPro\OAuth\ShutterstockOAuth::disconnect();

        wp_send_json_success(['disconnected' => true]);
    }

    /**
     * Get Shutterstock licensed images
     */
    public function ajax_shutterstock_get_licenses() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('upload_files')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\ShutterstockOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Shutterstock not connected. Please reconnect.']);
        }

        $page = isset($_POST['page']) ? (int)$_POST['page'] : 1;
        $per_page = isset($_POST['per_page']) ? min((int)$_POST['per_page'], 50) : 20;

        $result = \LightSyncPro\OAuth\ShutterstockOAuth::get_licenses($page, $per_page);

        if (is_wp_error($result)) {
            wp_send_json_error(['error' => $result->get_error_message()]);
        }

        // Get already synced image data with timestamps and destinations
        global $wpdb;
        $synced_rows = $wpdb->get_results($wpdb->prepare(
            "SELECT pm1.meta_value as image_id, pm2.post_id, pm3.meta_value as synced_at
             FROM {$wpdb->postmeta} pm1
             LEFT JOIN {$wpdb->postmeta} pm2 ON pm1.post_id = pm2.post_id AND pm2.meta_key = '_lightsync_source'
             LEFT JOIN {$wpdb->postmeta} pm3 ON pm1.post_id = pm3.post_id AND pm3.meta_key = '_lightsync_synced_at'
             WHERE pm1.meta_key = %s",
            '_lightsync_shutterstock_id'
        ), ARRAY_A);
        
        $synced_map = [];
        foreach ($synced_rows as $row) {
            $synced_at = !empty($row['synced_at']) ? strtotime($row['synced_at']) : time();
            $synced_map[$row['image_id']] = [
                'time' => $synced_at,
                'dest' => 'wp',
            ];
        }
        
        // Check Shopify sync status
        $shopify_table = $wpdb->prefix . 'lightsync_shopify_files_map';
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$shopify_table}'") === $shopify_table;
        
        if ($table_exists) {
            // Get all Shutterstock images synced to Shopify
            $shopify_rows = $wpdb->get_results(
                "SELECT lr_asset_id, updated_at FROM {$shopify_table} WHERE lr_asset_id LIKE 'shutterstock_%'",
                ARRAY_A
            );
            
            foreach ($shopify_rows as $row) {
                // Extract image_id from 'shutterstock_123456'
                $image_id = str_replace('shutterstock_', '', $row['lr_asset_id']);
                $shopify_time = !empty($row['updated_at']) ? strtotime($row['updated_at']) : time();
                
                if (isset($synced_map[$image_id])) {
                    // Already synced to WP, add Shopify
                    $synced_map[$image_id]['dest'] = 'both';
                    // Use most recent time
                    if ($shopify_time > $synced_map[$image_id]['time']) {
                        $synced_map[$image_id]['time'] = $shopify_time;
                    }
                } else {
                    // Only synced to Shopify
                    $synced_map[$image_id] = [
                        'time' => $shopify_time,
                        'dest' => 'shopify',
                    ];
                }
            }
        }

        // Collect all image IDs for bulk fetch
        $all_image_ids = [];
        $licenses_by_id = [];
        
        // Log first license to see structure
        if (!empty($result['licenses'])) {
            \LightSyncPro\Util\Logger::debug('[LSP Shutterstock] First license structure: ' . substr(json_encode($result['licenses'][0]), 0, 1000));
        }
        
        foreach ($result['licenses'] as $license) {
            $image = $license['image'] ?? [];
            $image_id = $image['id'] ?? '';
            if ($image_id) {
                $all_image_ids[] = $image_id;
                $licenses_by_id[$image_id] = $license;
            }
        }

        // Note: Search API doesn't find licensed images (returns 0)
        // CDN fallback will be used for thumbnails
        $image_details = [];

        // Format licenses for frontend
        $images = [];
        foreach ($result['licenses'] as $license) {
            $image = $license['image'] ?? [];
            $image_id = $image['id'] ?? '';
            $license_id = $license['id'] ?? '';
            
            // Use search-fetched details if available
            if (isset($image_details[$image_id])) {
                $full_image = $image_details[$image_id];
            } else {
                $full_image = $image;
            }
            
            // Get thumbnail URL from assets
            $thumb_url = '';
            $thumb_source = 'none';
            $assets = $full_image['assets'] ?? [];
            
            // Check assets object for preview URLs (in order of preference)
            foreach (['small_thumb', 'large_thumb', 'preview', 'huge_thumb', 'preview_1000', 'preview_1500'] as $size) {
                if (!empty($assets[$size]['url'])) {
                    $thumb_url = $assets[$size]['url'];
                    $thumb_source = 'assets.' . $size;
                    break;
                }
            }
            
            // Also check preview directly on full_image
            if (empty($thumb_url) && !empty($full_image['preview']['url'])) {
                $thumb_url = $full_image['preview']['url'];
                $thumb_source = 'full_image.preview';
            }
            
            // Fallback: Use Shutterstock's public "z" CDN pattern
            // This works for most images without needing contributor ID
            if (empty($thumb_url) && $image_id) {
                $thumb_url = 'https://image.shutterstock.com/z/stock-photo-' . $image_id . '.jpg';
                $thumb_source = 'cdn_z';
            }

            $images[] = [
                'id' => $image_id,
                'license_id' => $license_id,
                'description' => $full_image['description'] ?? $image['description'] ?? '',
                'thumbnail' => $thumb_url,
                'thumb_source' => $thumb_source,
                'width' => $full_image['original_width'] ?? $image['original_width'] ?? 0,
                'height' => $full_image['original_height'] ?? $image['original_height'] ?? 0,
                'format' => $license['format'] ?? ['size' => 'unknown'],
                'download_date' => $license['download_time'] ?? '',
            ];
        }

        wp_send_json_success([
            'images' => $images,
            'synced' => $synced_map,
            'total_count' => $result['total_count'],
            'page' => $result['page'],
            'per_page' => $result['per_page'],
            'has_more' => ($result['page'] * $result['per_page']) < $result['total_count'],
        ]);
    }

    /**
     * Sync Shutterstock image to WordPress
     */
    public function ajax_shutterstock_sync() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('upload_files')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\ShutterstockOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Shutterstock not connected. Please reconnect.']);
        }

        $license_id = isset($_POST['license_id']) ? sanitize_text_field($_POST['license_id']) : '';
        $image_id = isset($_POST['image_id']) ? sanitize_text_field($_POST['image_id']) : '';
        $description = isset($_POST['description']) ? sanitize_text_field($_POST['description']) : '';

        if (!$license_id || !$image_id) {
            wp_send_json_error(['error' => 'Missing license_id or image_id']);
        }

        // Get sync target
        $o = self::get_opt();
        $sync_target = $o['shutterstock_sync_target'] ?? 'wp';

        // Check if already synced to WordPress
        global $wpdb;
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = %s AND meta_value = %s LIMIT 1",
            '_lightsync_shutterstock_id', $image_id
        ));
        
        // If syncing only to WP and already exists, return early
        if ($existing && $sync_target === 'wp') {
            wp_send_json_success([
                'attachment_id' => (int)$existing,
                'already_synced' => true,
            ]);
            return;
        }

        // Redownload the image
        $download = \LightSyncPro\OAuth\ShutterstockOAuth::redownload($license_id);
        if (is_wp_error($download)) {
            wp_send_json_error(['error' => 'Redownload failed: ' . $download->get_error_message()]);
        }

        $download_url = $download['url'] ?? '';
        if (!$download_url) {
            wp_send_json_error(['error' => 'No download URL returned']);
        }

        // Download the actual image bytes
        $image_data = \LightSyncPro\OAuth\ShutterstockOAuth::download($download_url);
        if (is_wp_error($image_data)) {
            wp_send_json_error(['error' => 'Download failed: ' . $image_data->get_error_message()]);
        }

        // Determine filename
        $filename = 'shutterstock-' . $image_id . '.jpg';
        if (!empty($description)) {
            $slug = sanitize_title(substr($description, 0, 50));
            $filename = $slug . '-' . $image_id . '.jpg';
        }

        // Always convert to WebP for optimization (like Figma/Dropbox)
        $final_bytes = $image_data;
        $final_filename = $filename;
        $shopify_bytes = null; // Separate bytes for Shopify (may be resized)
        
        // Save temp file for conversion
        $temp_file = wp_tempnam($filename);
        file_put_contents($temp_file, $image_data);
        
        $webp_path = preg_replace('/\.[^.]+$/', '.webp', $temp_file);
        $image = wp_get_image_editor($temp_file);
        if (!is_wp_error($image)) {
            $size = $image->get_size();
            $megapixels = ($size['width'] * $size['height']) / 1000000;
            
            // For Shopify: resize if > 25MP (Shopify limit)
            if (in_array($sync_target, ['shopify', 'both']) && $megapixels > 25) {
                // Calculate new dimensions to fit under 25MP while maintaining aspect ratio
                $scale = sqrt(24 / $megapixels); // Target 24MP to be safe
                $new_width = (int)($size['width'] * $scale);
                $new_height = (int)($size['height'] * $scale);
                
                \LightSyncPro\Util\Logger::debug('[LSP Shutterstock] Resizing for Shopify: ' . $size['width'] . 'x' . $size['height'] . ' (' . round($megapixels, 1) . 'MP) → ' . $new_width . 'x' . $new_height);
                
                // Create resized version for Shopify
                $shopify_image = wp_get_image_editor($temp_file);
                if (!is_wp_error($shopify_image)) {
                    $shopify_image->resize($new_width, $new_height, false);
                    $shopify_image->set_quality(82);
                    $shopify_webp_path = preg_replace('/\.[^.]+$/', '-shopify.webp', $temp_file);
                    $shopify_result_file = $shopify_image->save($shopify_webp_path, 'image/webp');
                    if (!is_wp_error($shopify_result_file) && !empty($shopify_result_file['path']) && file_exists($shopify_result_file['path'])) {
                        $shopify_bytes = file_get_contents($shopify_result_file['path']);
                        @unlink($shopify_result_file['path']);
                    }
                }
            }
            
            // Convert to WebP for WordPress (full resolution)
            $image->set_quality(82);
            $result = $image->save($webp_path, 'image/webp');
            if (!is_wp_error($result) && !empty($result['path']) && file_exists($result['path']) && filesize($result['path']) > 0) {
                $final_bytes = file_get_contents($result['path']);
                $final_filename = pathinfo($filename, PATHINFO_FILENAME) . '.webp';
                \LightSyncPro\Util\Logger::debug('[LSP Shutterstock] WebP conversion: ' . filesize($temp_file) . ' → ' . filesize($result['path']) . ' bytes');
                @unlink($result['path']);
            }
        }
        @unlink($temp_file);
        
        // Use resized version for Shopify if available, otherwise use full resolution
        if (!$shopify_bytes) {
            $shopify_bytes = $final_bytes;
        }

        $attachment_id = null;
        $shopify_result = null;
        $alt_text = $description ?: ('Shutterstock ' . $image_id);

        // Sync to WordPress if target includes WP
        if (in_array($sync_target, ['wp', 'both'])) {
            if (!$existing) {
                $upload = wp_upload_bits($final_filename, null, $final_bytes);
                if (!empty($upload['error'])) {
                    wp_send_json_error(['error' => 'Upload failed: ' . $upload['error']]);
                }

                // Create attachment
                $filetype = wp_check_filetype($upload['file']);
                $attachment = [
                    'post_mime_type' => $filetype['type'],
                    'post_title' => $alt_text,
                    'post_content' => '',
                    'post_status' => 'inherit',
                ];

                $attachment_id = wp_insert_attachment($attachment, $upload['file']);
                if (is_wp_error($attachment_id)) {
                    wp_send_json_error(['error' => 'Attachment creation failed']);
                }

                // Generate metadata
                require_once(ABSPATH . 'wp-admin/includes/image.php');
                $attach_data = wp_generate_attachment_metadata($attachment_id, $upload['file']);
                wp_update_attachment_metadata($attachment_id, $attach_data);

                // Set alt text
                update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt_text);

                // Store Shutterstock metadata
                update_post_meta($attachment_id, '_lightsync_source', 'shutterstock');
                update_post_meta($attachment_id, '_lightsync_shutterstock_id', $image_id);
                update_post_meta($attachment_id, '_lightsync_shutterstock_license_id', $license_id);
                update_post_meta($attachment_id, '_lightsync_synced_at', gmdate('c'));
                update_post_meta($attachment_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
                update_post_meta($attachment_id, '_lightsync_last_sync_kind', 'SYNC');
            } else {
                $attachment_id = (int)$existing;
            }
        }

        // Sync to Shopify if target includes Shopify
        if (in_array($sync_target, ['shopify', 'both']) && class_exists('\LightSyncPro\Shopify\Shopify')) {
            $source_id = 'shutterstock_' . $image_id;
            $shopify_result = \LightSyncPro\Shopify\Shopify::upload_file(
                $shopify_bytes,
                $final_filename,
                $source_id,
                'shutterstock',
                $alt_text
            );
            
            if (is_wp_error($shopify_result)) {
                // Log but don't fail if Shopify fails
                \LightSyncPro\Util\Logger::debug('[LSP Shutterstock] Shopify upload failed: ' . $shopify_result->get_error_message());
            }
        }

        // Log activity
        $dest_label = $sync_target === 'both' ? 'WordPress + Shopify' : ($sync_target === 'shopify' ? 'Shopify' : 'WordPress');
        self::add_activity(
            'Shutterstock → ' . $dest_label . ': synced image ' . $image_id,
            'success',
            'shutterstock'
        );

        wp_send_json_success([
            'attachment_id' => $attachment_id,
            'url' => $attachment_id ? wp_get_attachment_url($attachment_id) : null,
            'shopify' => is_wp_error($shopify_result) ? null : $shopify_result,
            'destination' => $sync_target,
        ]);
    }

    /**
     * Save Shutterstock sync target preference
     */
    public function ajax_shutterstock_save_target() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $target = isset($_POST['target']) ? sanitize_text_field($_POST['target']) : 'wp';
        if (!in_array($target, ['wp', 'shopify', 'both'])) {
            $target = 'wp';
        }

        self::set_opt(['shutterstock_sync_target' => $target]);
        wp_send_json_success(['target' => $target]);
    }

    /**
     * Start Shutterstock background sync (processes inline for reliability)
     */
    public function ajax_shutterstock_background_sync() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('upload_files')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\ShutterstockOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Shutterstock not connected.']);
        }

        $image_ids = isset($_POST['image_ids']) ? array_map('sanitize_text_field', (array)$_POST['image_ids']) : [];
        $sync_target = isset($_POST['sync_target']) ? sanitize_text_field($_POST['sync_target']) : 'wp';

        if (empty($image_ids)) {
            wp_send_json_error(['error' => 'No images selected']);
        }

        // Get license data for the selected images
        $result = \LightSyncPro\OAuth\ShutterstockOAuth::get_licenses(1, 100);
        if (is_wp_error($result)) {
            wp_send_json_error(['error' => $result->get_error_message()]);
        }

        // Build lookup of image_id => license data
        $license_map = [];
        foreach ($result['licenses'] as $license) {
            $img_id = $license['image']['id'] ?? '';
            if ($img_id) {
                $license_map[$img_id] = $license;
            }
        }

        // Store progress in transient for polling
        $total = count($image_ids);
        set_transient('lsp_shutterstock_bg_total', $total, HOUR_IN_SECONDS);
        set_transient('lsp_shutterstock_bg_done', 0, HOUR_IN_SECONDS);
        set_transient('lsp_shutterstock_bg_running', true, HOUR_IN_SECONDS);

        // Process each image
        $done = 0;
        $errors = [];

        foreach ($image_ids as $image_id) {
            if (!isset($license_map[$image_id])) {
                $errors[] = $image_id;
                continue;
            }

            $license = $license_map[$image_id];
            $license_id = $license['id'] ?? '';
            $description = $license['image']['description'] ?? '';

            // Call the sync method
            $_POST['license_id'] = $license_id;
            $_POST['image_id'] = $image_id;
            $_POST['description'] = $description;

            // Capture sync result
            ob_start();
            $this->ajax_shutterstock_sync_internal($sync_target);
            ob_end_clean();

            $done++;
            set_transient('lsp_shutterstock_bg_done', $done, HOUR_IN_SECONDS);
        }

        // Mark complete
        delete_transient('lsp_shutterstock_bg_running');

        wp_send_json_success([
            'done' => $done,
            'errors' => count($errors),
            'total' => $total,
        ]);
    }

    /**
     * Internal sync method for background processing
     */
    private function ajax_shutterstock_sync_internal($sync_target) {
        $license_id = isset($_POST['license_id']) ? sanitize_text_field($_POST['license_id']) : '';
        $image_id = isset($_POST['image_id']) ? sanitize_text_field($_POST['image_id']) : '';
        $description = isset($_POST['description']) ? sanitize_text_field($_POST['description']) : '';

        if (!$license_id || !$image_id) return false;

        // Check if already synced
        global $wpdb;
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = %s AND meta_value = %s LIMIT 1",
            '_lightsync_shutterstock_id', $image_id
        ));
        if ($existing && $sync_target === 'wp') return true;

        // Redownload the image
        $download = \LightSyncPro\OAuth\ShutterstockOAuth::redownload($license_id);
        if (is_wp_error($download)) return false;

        $download_url = $download['url'] ?? '';
        if (!$download_url) return false;

        $image_data = \LightSyncPro\OAuth\ShutterstockOAuth::download($download_url);
        if (is_wp_error($image_data)) return false;

        // Filename
        $filename = 'shutterstock-' . $image_id . '.jpg';
        if (!empty($description)) {
            $slug = sanitize_title(substr($description, 0, 50));
            $filename = $slug . '-' . $image_id . '.jpg';
        }

        // Always convert to WebP for optimization
        $final_bytes = $image_data;
        $final_filename = $filename;
        $shopify_bytes = null; // Separate bytes for Shopify (may be resized)
        
        $temp_file = wp_tempnam($filename);
        file_put_contents($temp_file, $image_data);
        
        $webp_path = preg_replace('/\.[^.]+$/', '.webp', $temp_file);
        $image = wp_get_image_editor($temp_file);
        if (!is_wp_error($image)) {
            $size = $image->get_size();
            $megapixels = ($size['width'] * $size['height']) / 1000000;
            
            // For Shopify: resize if > 25MP (Shopify limit)
            if (in_array($sync_target, ['shopify', 'both']) && $megapixels > 25) {
                $scale = sqrt(24 / $megapixels); // Target 24MP to be safe
                $new_width = (int)($size['width'] * $scale);
                $new_height = (int)($size['height'] * $scale);
                
                $shopify_image = wp_get_image_editor($temp_file);
                if (!is_wp_error($shopify_image)) {
                    $shopify_image->resize($new_width, $new_height, false);
                    $shopify_image->set_quality(82);
                    $shopify_webp_path = preg_replace('/\.[^.]+$/', '-shopify.webp', $temp_file);
                    $shopify_result_file = $shopify_image->save($shopify_webp_path, 'image/webp');
                    if (!is_wp_error($shopify_result_file) && !empty($shopify_result_file['path']) && file_exists($shopify_result_file['path'])) {
                        $shopify_bytes = file_get_contents($shopify_result_file['path']);
                        @unlink($shopify_result_file['path']);
                    }
                }
            }
            
            $image->set_quality(82);
            $result = $image->save($webp_path, 'image/webp');
            if (!is_wp_error($result) && !empty($result['path']) && file_exists($result['path']) && filesize($result['path']) > 0) {
                $final_bytes = file_get_contents($result['path']);
                $final_filename = pathinfo($filename, PATHINFO_FILENAME) . '.webp';
                @unlink($result['path']);
            }
        }
        @unlink($temp_file);
        
        if (!$shopify_bytes) {
            $shopify_bytes = $final_bytes;
        }

        $alt_text = $description ?: ('Shutterstock ' . $image_id);

        // Sync to WordPress
        if (in_array($sync_target, ['wp', 'both']) && !$existing) {
            $upload = wp_upload_bits($final_filename, null, $final_bytes);
            if (empty($upload['error'])) {
                $filetype = wp_check_filetype($upload['file']);
                $attachment_id = wp_insert_attachment([
                    'post_mime_type' => $filetype['type'],
                    'post_title' => $alt_text,
                    'post_content' => '',
                    'post_status' => 'inherit',
                ], $upload['file']);

                if (!is_wp_error($attachment_id)) {
                    require_once(ABSPATH . 'wp-admin/includes/image.php');
                    $attach_data = wp_generate_attachment_metadata($attachment_id, $upload['file']);
                    wp_update_attachment_metadata($attachment_id, $attach_data);
                    update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt_text);
                    update_post_meta($attachment_id, '_lightsync_source', 'shutterstock');
                    update_post_meta($attachment_id, '_lightsync_shutterstock_id', $image_id);
                    update_post_meta($attachment_id, '_lightsync_shutterstock_license_id', $license_id);
                    update_post_meta($attachment_id, '_lightsync_synced_at', gmdate('c'));
                    update_post_meta($attachment_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
                    update_post_meta($attachment_id, '_lightsync_last_sync_kind', 'SYNC');
                }
            }
        }

        // Sync to Shopify
        if (in_array($sync_target, ['shopify', 'both']) && class_exists('\LightSyncPro\Shopify\Shopify')) {
            \LightSyncPro\Shopify\Shopify::upload_file(
                $shopify_bytes,
                $final_filename,
                'shutterstock_' . $image_id,
                'shutterstock',
                $alt_text
            );
        }

        return true;
    }

    /**
     * Get Shutterstock background sync status
     */
    public function ajax_shutterstock_background_status() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        $running = get_transient('lsp_shutterstock_bg_running');
        $total = (int)get_transient('lsp_shutterstock_bg_total');
        $done = (int)get_transient('lsp_shutterstock_bg_done');

        wp_send_json_success([
            'running' => (bool)$running,
            'total' => $total,
            'done' => $done,
        ]);
    }

    /**
     * List Dropbox folder contents
     */
    public function ajax_dropbox_list_folder() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\DropboxOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Dropbox not connected. Please reconnect.']);
        }

        // Get path from request (empty string = root)
        $path = isset($_POST['path']) ? sanitize_text_field($_POST['path']) : '';
        
        // Extend timeout for large folders
        @set_time_limit(120);

        // Debug logging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[LSP Dropbox] list_folder AJAX called, path: "' . $path . '"');
        }

        $result = \LightSyncPro\OAuth\DropboxOAuth::list_folder_all($path);

        if (is_wp_error($result)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('[LSP Dropbox] list_folder_all error: ' . $result->get_error_message());
            }
            wp_send_json_error(['error' => $result->get_error_message()]);
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[LSP Dropbox] list_folder_all returned ' . count($result['entries'] ?? []) . ' entries');
        }

        // Separate folders and images
        $entries = $result['entries'] ?? [];
        $folders = [];
        $images = [];
        
        // Only show web-compatible formats (RAW files require server-side conversion which most hosts don't support)
        $image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp', 'tiff', 'tif'];
        
        foreach ($entries as $entry) {
            if (($entry['.tag'] ?? '') === 'folder') {
                $folders[] = [
                    'id'   => $entry['id'] ?? '',
                    'name' => $entry['name'] ?? '',
                    'path' => $entry['path_lower'] ?? $entry['path_display'] ?? '',
                ];
            } elseif (($entry['.tag'] ?? '') === 'file') {
                $name = strtolower($entry['name'] ?? '');
                $ext = pathinfo($name, PATHINFO_EXTENSION);
                
                if (in_array($ext, $image_extensions, true)) {
                    $images[] = [
                        'id'            => $entry['id'] ?? '',
                        'name'          => $entry['name'] ?? '',
                        'path'          => $entry['path_lower'] ?? $entry['path_display'] ?? '',
                        'size'          => $entry['size'] ?? 0,
                        // Use server_modified (when Dropbox received the file) for update detection
                        'modified'      => $entry['server_modified'] ?? $entry['client_modified'] ?? '',
                        'content_hash'  => $entry['content_hash'] ?? '',
                    ];
                }
            }
        }
        
        // Sort folders and images alphabetically
        usort($folders, fn($a, $b) => strcasecmp($a['name'], $b['name']));
        usort($images, fn($a, $b) => strcasecmp($a['name'], $b['name']));

        wp_send_json_success([
            'path'    => $path,
            'folders' => $folders,
            'images'  => $images,
        ]);
    }

    /**
     * Get list of already-synced Dropbox file IDs
     */
    public function ajax_dropbox_get_synced() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        global $wpdb;
        
        // Get all Dropbox file IDs with their sync timestamps
        $results = $wpdb->get_results(
            "SELECT 
                pm_id.meta_value as file_id, 
                pm_id.post_id as attachment_id,
                COALESCE(pm_sync.meta_value, p.post_modified_gmt) as synced_at
             FROM {$wpdb->postmeta} pm_id
             JOIN {$wpdb->posts} p ON pm_id.post_id = p.ID
             LEFT JOIN {$wpdb->postmeta} pm_sync ON pm_id.post_id = pm_sync.post_id 
                AND pm_sync.meta_key = '_lightsync_last_synced_at'
             WHERE pm_id.meta_key = '_lightsync_dropbox_file_id'
             GROUP BY pm_id.meta_value"
        );

        // Get Shopify mappings
        $shopify_table = $wpdb->prefix . 'lightsync_shopify_files_map';
        $shop = self::get_opt('shopify_shop_domain', '');
        
        $shopify_synced = [];
        if ($shop && $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $shopify_table)) === $shopify_table) {
            $shopify_results = $wpdb->get_results($wpdb->prepare(
                "SELECT lr_asset_id FROM {$shopify_table} WHERE shop_domain = %s AND shopify_file_id IS NOT NULL",
                $shop
            ));
            foreach ($shopify_results as $row) {
                $shopify_synced[$row->lr_asset_id] = true;
            }
        }

        // Build array with file_id => {time, dest}
        $synced = [];
        foreach ($results as $row) {
            $has_wp = true;
            $has_shopify = isset($shopify_synced[$row->file_id]);
            
            $dest = 'wp';
            if ($has_wp && $has_shopify) {
                $dest = 'both';
            } elseif ($has_shopify) {
                $dest = 'shopify';
            }
            
            $synced[$row->file_id] = [
                'time' => strtotime($row->synced_at . ' UTC'),
                'dest' => $dest,
            ];
        }
        
        // Also check for Shopify-only syncs (files synced to Shopify but not WordPress)
        foreach ($shopify_synced as $asset_id => $v) {
            if (!isset($synced[$asset_id])) {
                $synced[$asset_id] = [
                    'time' => time(),
                    'dest' => 'shopify',
                ];
            }
        }

        wp_send_json_success($synced);
    }

    /**
     * Save Dropbox sync target preference
     */
    public function ajax_dropbox_save_target() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $target = isset($_POST['target']) ? sanitize_text_field($_POST['target']) : 'wp';
        if (!in_array($target, ['wp', 'shopify', 'both'], true)) {
            $target = 'wp';
        }

        self::set_opt(['dropbox_sync_target' => $target]);
        wp_send_json_success(['target' => $target]);
    }

    /**
     * Get Dropbox thumbnail
     */
    public function ajax_dropbox_get_thumbnail() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\DropboxOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Dropbox not connected']);
        }

        $path = isset($_POST['path']) ? sanitize_text_field($_POST['path']) : '';
        if (!$path) {
            wp_send_json_error(['error' => 'Missing path']);
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[LSP Dropbox] get_thumbnail AJAX for: ' . $path);
        }

        // Use caching to avoid repeated API calls
        $cache_key = 'lsp_dropbox_thumb_' . md5($path);
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('[LSP Dropbox] get_thumbnail returning cached');
            }
            wp_send_json_success(['thumbnail' => $cached]);
        }

        $result = \LightSyncPro\OAuth\DropboxOAuth::get_thumbnail($path);

        if (is_wp_error($result)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('[LSP Dropbox] get_thumbnail error: ' . $result->get_error_message());
            }
            wp_send_json_error(['error' => $result->get_error_message()]);
        }

        // Cache for 1 hour
        set_transient($cache_key, $result, HOUR_IN_SECONDS);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[LSP Dropbox] get_thumbnail success, size: ' . strlen($result));
        }

        wp_send_json_success(['thumbnail' => $result]);
    }

    /**
     * Queue Dropbox files for background sync
     */
    public function ajax_dropbox_sync_files() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        // Sync gate check
        $gate = lsp_gate_check();
        if ( is_wp_error( $gate ) ) {
            wp_send_json_error([
                'error' => 'sync_disabled',
                'message' => $gate->get_error_message(),
            ], 403);
        }

        if (!\LightSyncPro\OAuth\DropboxOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Dropbox not connected']);
        }

        $file_ids = isset($_POST['file_ids']) ? array_map('sanitize_text_field', (array)$_POST['file_ids']) : [];
        $file_paths = isset($_POST['file_paths']) ? array_map('sanitize_text_field', (array)$_POST['file_paths']) : [];
        $file_names = isset($_POST['file_names']) ? array_map('sanitize_text_field', (array)$_POST['file_names']) : [];
        $sync_target = isset($_POST['sync_target']) ? sanitize_text_field($_POST['sync_target']) : 'wp';

        if (empty($file_ids)) {
            wp_send_json_error(['error' => 'No files selected']);
        }

        // Get existing queue or create new
        $queue = get_option('lightsync_dropbox_sync_queue', []);
        
        // Add to queue
        foreach ($file_ids as $i => $file_id) {
            $queue[] = [
                'file_id'     => $file_id,
                'file_path'   => $file_paths[$i] ?? '',
                'file_name'   => $file_names[$i] ?? '',
                'sync_target' => $sync_target,
                'queued_at'   => time(),
            ];
        }

        update_option('lightsync_dropbox_sync_queue', $queue, false);
        
        // Track progress info
        $progress = get_option('lightsync_dropbox_sync_progress', []);
        $progress['total'] = count($queue);
        $progress['synced'] = 0;
        $progress['running'] = true;
        $progress['sync_target'] = $sync_target;
        $progress['started_at'] = time();
        update_option('lightsync_dropbox_sync_progress', $progress, false);

        // Schedule background processing
        if (!wp_next_scheduled('lightsync_process_dropbox_queue')) {
            wp_schedule_single_event(time() + 5, 'lightsync_process_dropbox_queue');
        }

        wp_send_json_success([
            'queued' => count($file_ids),
            'total_queue' => count($queue),
        ]);
    }

    /**
     * Sync single Dropbox file (foreground mode)
     */
    public function ajax_dropbox_sync_single() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        // Sync gate check
        $gate = lsp_gate_check();
        if ( is_wp_error( $gate ) ) {
            wp_send_json_error([
                'error' => 'sync_disabled',
                'message' => $gate->get_error_message(),
            ], 403);
        }

        if (!\LightSyncPro\OAuth\DropboxOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Dropbox not connected']);
        }

        $file_id = isset($_POST['file_id']) ? sanitize_text_field($_POST['file_id']) : '';
        $file_path = isset($_POST['file_path']) ? sanitize_text_field($_POST['file_path']) : '';
        $file_name = isset($_POST['file_name']) ? sanitize_text_field($_POST['file_name']) : '';
        $sync_target = isset($_POST['sync_target']) ? sanitize_text_field($_POST['sync_target']) : 'wp';

        if (!$file_id || !$file_path) {
            wp_send_json_error(['error' => 'Missing file info']);
        }

        // Extend timeout
        @set_time_limit(120);

        $result = $this->sync_dropbox_file($file_id, $file_path, $file_name, $sync_target);

        if (is_wp_error($result)) {
            wp_send_json_error(['error' => $result->get_error_message()]);
        }

        wp_send_json_success($result);
    }

    /**
     * Get Dropbox background sync status for polling
     * LIGHTWEIGHT - just returns status, no processing
     */
    public function ajax_dropbox_background_status() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $progress = get_option('lightsync_dropbox_sync_progress', []);
        $queue = get_option('lightsync_dropbox_sync_queue', []);

        $total = $progress['total'] ?? 0;
        $synced = $progress['synced'] ?? 0;
        $running = !empty($queue) || ($progress['running'] ?? false);
        
        // If queue is empty, mark as not running
        if (empty($queue) && $running) {
            $progress['running'] = false;
            update_option('lightsync_dropbox_sync_progress', $progress, false);
            $running = false;
        }

        // Get recently synced file IDs for UI update
        $synced_ids = $progress['synced_ids'] ?? [];

        wp_send_json_success([
            'total'       => $total,
            'synced'      => $synced,
            'remaining'   => count($queue),
            'running'     => $running,
            'sync_target' => $progress['sync_target'] ?? 'wp',
            'file_ids'    => $synced_ids,
            'needs_process' => !empty($queue), // Signal JS to call process endpoint
        ]);
    }
    
    /**
     * Process next Dropbox file in queue (separate from status to avoid timeouts)
     */
    public function ajax_dropbox_process_next() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $queue = get_option('lightsync_dropbox_sync_queue', []);
        
        if (empty($queue)) {
            wp_send_json_success(['processed' => false, 'reason' => 'queue_empty']);
            return;
        }

        // Check for lock to prevent parallel processing
        $lock_key = 'lsp_dropbox_bg_lock';
        if (get_transient($lock_key)) {
            wp_send_json_success(['processed' => false, 'reason' => 'locked']);
            return;
        }
        set_transient($lock_key, true, 180); // 3 minute lock

        try {
            // Extend timeout
            @set_time_limit(180);
            @ini_set('max_execution_time', '180');

            // Process one item
            $item = array_shift($queue);
            update_option('lightsync_dropbox_sync_queue', $queue, false);

            $file_id = $item['file_id'];
            $file_path = $item['file_path'];
            $file_name = $item['file_name'];
            $sync_target = $item['sync_target'] ?? 'wp';
            $retry_count = $item['retry_count'] ?? 0;

            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Processing: ' . $file_name . ' (' . $file_path . ') retry=' . $retry_count);

            $result = $this->sync_dropbox_file($file_id, $file_path, $file_name, $sync_target);

            if (!is_wp_error($result)) {
                // Update progress
                $progress = get_option('lightsync_dropbox_sync_progress', []);
                $progress['synced'] = ($progress['synced'] ?? 0) + 1;
                
                // Track synced file IDs
                if (!isset($progress['synced_ids'])) {
                    $progress['synced_ids'] = [];
                }
                $progress['synced_ids'][] = $file_id;
                
                update_option('lightsync_dropbox_sync_progress', $progress, false);
                self::usage_consume(1);

                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Processed successfully: ' . $file_name);
                
                wp_send_json_success([
                    'processed' => true,
                    'file_name' => $file_name,
                    'file_id' => $file_id,
                    'remaining' => count($queue),
                ]);
            } else {
                $error_msg = $result->get_error_message();
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Process failed: ' . $error_msg);
                
                // Retry logic: if timeout or connection error, retry up to 2 times
                $is_retryable = (
                    strpos($error_msg, 'timeout') !== false ||
                    strpos($error_msg, 'timed out') !== false ||
                    strpos($error_msg, 'cURL') !== false ||
                    strpos($error_msg, 'connection') !== false
                );
                
                if ($is_retryable && $retry_count < 2) {
                    // Put back in queue for retry
                    $item['retry_count'] = $retry_count + 1;
                    $queue[] = $item; // Add to end of queue
                    update_option('lightsync_dropbox_sync_queue', $queue, false);
                    
                    \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Queued for retry (' . ($retry_count + 1) . '/2): ' . $file_name);
                    
                    wp_send_json_success([
                        'processed' => false,
                        'retry' => true,
                        'retry_count' => $retry_count + 1,
                        'error' => $error_msg,
                        'file_name' => $file_name,
                        'remaining' => count($queue),
                    ]);
                } else {
                    // Permanent failure - track it
                    $progress = get_option('lightsync_dropbox_sync_progress', []);
                    if (!isset($progress['failed'])) {
                        $progress['failed'] = [];
                    }
                    $progress['failed'][] = [
                        'file_name' => $file_name,
                        'file_id' => $file_id,
                        'error' => $error_msg,
                        'retries' => $retry_count,
                    ];
                    update_option('lightsync_dropbox_sync_progress', $progress, false);
                    
                    wp_send_json_success([
                        'processed' => false,
                        'error' => $error_msg,
                        'file_name' => $file_name,
                        'remaining' => count($queue),
                    ]);
                }
            }
        } finally {
            delete_transient($lock_key);
        }
    }

    /**
     * Process one Dropbox file in background queue (called during status poll)
     */
    private function process_dropbox_background_tick() {
        $queue = get_option('lightsync_dropbox_sync_queue', []);
        
        if (empty($queue)) {
            return;
        }

        // Check for lock to prevent parallel processing
        $lock_key = 'lsp_dropbox_bg_lock';
        if (get_transient($lock_key)) {
            return; // Another request is processing
        }
        set_transient($lock_key, true, 120); // 2 minute lock

        try {
            // Extend timeout for this request
            @set_time_limit(120);

            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Processing background tick: ' . count($queue) . ' items remaining');

            // Process one item
            $item = array_shift($queue);
            update_option('lightsync_dropbox_sync_queue', $queue, false);

            $file_id = $item['file_id'];
            $file_path = $item['file_path'];
            $file_name = $item['file_name'];
            $sync_target = $item['sync_target'] ?? 'wp';

            $result = $this->sync_dropbox_file($file_id, $file_path, $file_name, $sync_target);

            if (!is_wp_error($result)) {
                // Update progress
                $progress = get_option('lightsync_dropbox_sync_progress', []);
                $progress['synced'] = ($progress['synced'] ?? 0) + 1;
                
                // Track synced file IDs
                if (!isset($progress['synced_ids'])) {
                    $progress['synced_ids'] = [];
                }
                $progress['synced_ids'][] = $file_id;
                
                update_option('lightsync_dropbox_sync_progress', $progress, false);

                self::usage_consume(1);

                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Background tick synced: ' . $file_name);
            } else {
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Background tick failed: ' . $result->get_error_message());
            }

            // If queue is now empty, mark complete and log activity
            if (empty($queue)) {
                $progress = get_option('lightsync_dropbox_sync_progress', []);
                $progress['running'] = false;
                update_option('lightsync_dropbox_sync_progress', $progress, false);

                $synced = $progress['synced'] ?? 0;
                if ($synced > 0) {
                    $dest_label = 'WordPress';
                    
                    self::add_activity(
                        "Dropbox → {$dest_label} Sync Complete (Background): {$synced} file(s)",
                        'success',
                        'dropbox'
                    );
                    
                    self::set_opt([
                        'lightsync_last_sync_ts'     => time(),
                        'lightsync_last_sync_source' => 'dropbox',
                        'lightsync_last_sync_status' => 'complete',
                    ]);
                }
            }
        } finally {
            delete_transient($lock_key);
        }
    }

    /**
     * Sync a single Dropbox file to WordPress
     */
    private function sync_dropbox_file($file_id, $file_path, $file_name, $sync_target = 'wp') {
        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Starting sync for: ' . $file_name . ' (path: ' . $file_path . ', target: ' . $sync_target . ')');
        
        $used_rendition = false;
        $image_data = null;
        
        // Check file size first - use rendition for files > 5MB to avoid server timeout
        $metadata = \LightSyncPro\OAuth\DropboxOAuth::get_metadata($file_path);
        $file_size = 0;
        if (!is_wp_error($metadata) && isset($metadata['size'])) {
            $file_size = (int) $metadata['size'];
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] File size: ' . $file_size . ' bytes (' . round($file_size / 1024 / 1024, 2) . ' MB)');
        }
        
        // For files > 5MB, skip straight to rendition (avoids server gateway timeout)
        $size_threshold = 5 * 1024 * 1024; // 5MB
        if ($file_size > $size_threshold) {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Large file detected (' . round($file_size / 1024 / 1024, 2) . ' MB), using 2048px rendition...');
            $image_data = \LightSyncPro\OAuth\DropboxOAuth::get_large_rendition($file_path);
            
            if (!is_wp_error($image_data) && !empty($image_data)) {
                $used_rendition = true;
            } else {
                $error = is_wp_error($image_data) ? $image_data->get_error_message() : 'empty response';
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Rendition failed for large file: ' . $error);
                return new \WP_Error('large_file_failed', 'File too large (' . round($file_size / 1024 / 1024, 1) . 'MB) and rendition failed: ' . $error);
            }
        } else {
            // Normal size file - try direct download with shorter timeout
            $image_data = \LightSyncPro\OAuth\DropboxOAuth::download($file_path);
        
        if (is_wp_error($image_data)) {
            $error_msg = $image_data->get_error_message();
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Direct download failed: ' . $error_msg . ' - trying temporary link...');
            
            // Check if it's a timeout error
            $is_timeout = (
                strpos($error_msg, 'timeout') !== false ||
                strpos($error_msg, 'timed out') !== false ||
                strpos($error_msg, 'cURL error 28') !== false
            );
            
            // Fallback: Get temporary download link
            $link_result = \LightSyncPro\OAuth\DropboxOAuth::get_temporary_link($file_path);
            
            if (is_wp_error($link_result)) {
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Failed to get temp link: ' . $link_result->get_error_message());
                
                // If we can't even get a temp link, try rendition as last resort
                if ($is_timeout) {
                    \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Timeout detected, trying 2048px rendition fallback...');
                    $image_data = \LightSyncPro\OAuth\DropboxOAuth::get_large_rendition($file_path);
                    if (!is_wp_error($image_data)) {
                        $used_rendition = true;
                    }
                }
                
                if (is_wp_error($image_data) || empty($image_data)) {
                    return $link_result;
                }
            } else {
                $download_url = $link_result['link'] ?? '';
                if (!$download_url) {
                    return new \WP_Error('no_link', 'Could not get download link');
                }
                
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Got temp link, downloading...');

                // Download the file with extended timeout
                $response = wp_remote_get($download_url, [
                    'timeout' => 120,
                    'sslverify' => true,
                ]);
                
                if (is_wp_error($response)) {
                    $dl_error = $response->get_error_message();
                    \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Download via temp link failed: ' . $dl_error);
                    
                    // Check if timeout - fall back to rendition
                    $is_dl_timeout = (
                        strpos($dl_error, 'timeout') !== false ||
                        strpos($dl_error, 'timed out') !== false ||
                        strpos($dl_error, 'cURL error 28') !== false
                    );
                    
                    if ($is_dl_timeout) {
                        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Download timeout, trying 2048px rendition fallback...');
                        $image_data = \LightSyncPro\OAuth\DropboxOAuth::get_large_rendition($file_path);
                        if (!is_wp_error($image_data)) {
                            $used_rendition = true;
                        } else {
                            return new \WP_Error('download_failed', 'Download timed out and rendition failed: ' . $image_data->get_error_message());
                        }
                    } else {
                        return new \WP_Error('download_failed', 'Download failed: ' . $dl_error);
                    }
                } else {
                    $http_code = wp_remote_retrieve_response_code($response);
                    if ($http_code !== 200) {
                        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Download returned HTTP ' . $http_code);
                        return new \WP_Error('download_http_error', 'Download failed with HTTP ' . $http_code);
                    }

                    $image_data = wp_remote_retrieve_body($response);
                }
            }
        }
        } // End else: normal size file download
        
        if (empty($image_data)) {
            return new \WP_Error('empty_download', 'Downloaded file is empty');
        }

        // If we used rendition, update filename to .jpg
        if ($used_rendition) {
            $base_name = pathinfo($file_name, PATHINFO_FILENAME);
            $file_name = $base_name . '.jpg';
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Used 2048px rendition for large file, saved as: ' . $file_name . ' (' . strlen($image_data) . ' bytes)');
        } else {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Downloaded file: ' . $file_name . ' (' . strlen($image_data) . ' bytes)');
        }

        // Get file extension
        $ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        
        // Handle RAW files - these MUST be converted (WordPress can't handle RAW)
        $raw_formats = ['nef', 'cr2', 'cr3', 'arw', 'dng', 'orf', 'rw2', 'pef', 'raf'];
        $is_raw = in_array($ext, $raw_formats, true);
        
        // Create temp directory
        $upload_dir = wp_upload_dir();
        $temp_dir = $upload_dir['basedir'] . '/lightsync-temp';
        if (!file_exists($temp_dir)) {
            wp_mkdir_p($temp_dir);
        }
        
        $base_name = sanitize_file_name(pathinfo($file_name, PATHINFO_FILENAME));
        $temp_file = $temp_dir . '/' . wp_unique_filename($temp_dir, $base_name . '.' . $ext);
        
        if (file_put_contents($temp_file, $image_data) === false) {
            return new \WP_Error('write_failed', 'Could not write temp file');
        }

        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Temp file written: ' . $temp_file . ' (' . filesize($temp_file) . ' bytes)');

        // If RAW, convert to JPEG first (required - WordPress can't handle RAW)
        if ($is_raw) {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Processing RAW file (' . strtoupper($ext) . ')...');
            
            $jpeg_file = $this->convert_raw_file_to_jpeg($temp_file, $ext);
            if ($jpeg_file && file_exists($jpeg_file)) {
                @unlink($temp_file); // Remove original RAW
                $temp_file = $jpeg_file;
                $file_name = $base_name . '.jpg';
                $ext = 'jpg';
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Converted RAW to JPEG: ' . $jpeg_file . ' (' . filesize($jpeg_file) . ' bytes)');
            } else {
                @unlink($temp_file);
                $format_name = strtoupper($ext);
                return new \WP_Error(
                    'raw_convert_failed', 
                    'RAW files (' . $format_name . ') require server-side conversion. Your server does not support this format. Please export to JPEG/PNG from Lightroom or your photo editor before uploading to Dropbox.'
                );
            }
        }

        // Apply WebP compression
        $final_file = $temp_file;
        $final_name = sanitize_file_name($file_name);
        
        // Try compression - but don't fail if it's not available
        $compressed = $this->compress_dropbox_image($temp_file, $final_name);
        if ($compressed && isset($compressed['path']) && file_exists($compressed['path'])) {
            // Compression succeeded
            if ($compressed['path'] !== $temp_file) {
                @unlink($temp_file); // Remove uncompressed version
            }
            $final_file = $compressed['path'];
            $final_name = $compressed['filename'];
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Compressed to: ' . $final_name . ' (' . filesize($final_file) . ' bytes)');
        } else {
            // Compression not available or failed - use JPEG/original
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Compression skipped, using: ' . $final_name);
        }

        $results = [
            'file_id'   => $file_id,
            'file_name' => $final_name,
            'wp_id'     => null,

        ];

        // Save compressed bytes for Shopify BEFORE WP upload (media_handle_sideload moves the file)
        $shopify_bytes = null;
        if ($sync_target === 'shopify' || $sync_target === 'both') {
            $shopify_bytes = file_exists($final_file) ? @file_get_contents($final_file) : $image_data;
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Saved ' . strlen($shopify_bytes) . ' bytes for Shopify (from ' . (file_exists($final_file) ? 'compressed file' : 'original data') . ')');
        }

        // Sync to WordPress
        if ($sync_target === 'wp' || $sync_target === 'both') {
            // Check if file already exists in WordPress by Dropbox file ID
            global $wpdb;
            $existing_wp_id = $wpdb->get_var($wpdb->prepare(
                "SELECT pm.post_id FROM {$wpdb->postmeta} pm
                 INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID AND p.post_status = 'inherit'
                 WHERE pm.meta_key = '_lightsync_dropbox_file_id' AND pm.meta_value = %s LIMIT 1",
                $file_id
            ));
            
            // Generate checksum from downloaded content for change detection
            $new_checksum = hash('sha256', $image_data);
            
            if ($existing_wp_id) {
                // Check if content has actually changed
                $stored_checksum = get_post_meta($existing_wp_id, '_lightsync_dropbox_checksum', true);
                
                if (!empty($stored_checksum) && hash_equals($stored_checksum, $new_checksum)) {
                    // No changes - skip upload
                    \LightSyncPro\Util\Logger::debug('[LSP Dropbox] File unchanged (checksum match) - skipping attachment #' . $existing_wp_id);
                    update_post_meta($existing_wp_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
                    update_post_meta($existing_wp_id, '_lightsync_last_sync_kind', 'SKIP');
                    $results['wp_id'] = (int)$existing_wp_id;
                    $results['wp_skipped'] = true;
                } else {
                    // Content changed - replace the file
                    \LightSyncPro\Util\Logger::debug('[LSP Dropbox] File changed - updating attachment #' . $existing_wp_id);
                    
                    // Get the current attached file path — overwrite in place to preserve URL
                    $attached_file = get_attached_file($existing_wp_id);
                    
                    if ($attached_file) {
                        $attach_dir = dirname($attached_file);
                        $current_ext = strtolower(pathinfo($attached_file, PATHINFO_EXTENSION));
                        $new_ext = strtolower(pathinfo($final_name, PATHINFO_EXTENSION));
                        
                        // Same path unless extension changed
                        $new_file_path = $attached_file;
                        if ($current_ext !== $new_ext) {
                            $new_file_path = preg_replace('/\.' . preg_quote($current_ext, '/') . '$/', '.' . $new_ext, $attached_file);
                        }
                        
                        // Delete old thumbnails BEFORE overwriting
                        $old_meta = wp_get_attachment_metadata($existing_wp_id);
                        if (!empty($old_meta['sizes'])) {
                            foreach ($old_meta['sizes'] as $size_info) {
                                $thumb_path = $attach_dir . '/' . $size_info['file'];
                                if (file_exists($thumb_path)) {
                                    @unlink($thumb_path);
                                }
                            }
                        }
                        
                        // Copy new file to same path (overwrite in place)
                        if (copy($final_file, $new_file_path)) {
                            // If extension changed, remove old file and update path
                            if ($new_file_path !== $attached_file) {
                                if (file_exists($attached_file)) {
                                    @unlink($attached_file);
                                }
                                update_attached_file($existing_wp_id, $new_file_path);
                            }
                            
                            // Update post mime type if extension changed
                            $new_mime = wp_check_filetype($new_file_path)['type'];
                            if ($new_mime) {
                                wp_update_post([
                                    'ID' => $existing_wp_id,
                                    'post_mime_type' => $new_mime,
                                ]);
                            }
                            
                            // Regenerate thumbnails
                            require_once ABSPATH . 'wp-admin/includes/image.php';
                            $attach_data = wp_generate_attachment_metadata($existing_wp_id, $new_file_path);
                            wp_update_attachment_metadata($existing_wp_id, $attach_data);
                            
                            // Update LightSync metadata
                            update_post_meta($existing_wp_id, '_lightsync_dropbox_checksum', $new_checksum);
                            update_post_meta($existing_wp_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
                            update_post_meta($existing_wp_id, '_lightsync_last_sync_kind', 'UPDATE');
                            
                            $results['wp_id'] = (int)$existing_wp_id;
                            $results['wp_updated'] = true;
                            
                            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Successfully updated attachment #' . $existing_wp_id . ' in place: ' . $new_file_path);
                        } else {
                            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Failed to copy new file to uploads');
                            update_post_meta($existing_wp_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
                            $results['wp_id'] = (int)$existing_wp_id;
                            $results['wp_error'] = 'Failed to write updated file';
                        }
                    } else {
                        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Could not find original file path for attachment #' . $existing_wp_id);
                        update_post_meta($existing_wp_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
                        $results['wp_id'] = (int)$existing_wp_id;
                        $results['wp_error'] = 'Original file not found';
                    }
                }
            } else {
                // Copy file since media_handle_sideload may move it
                $upload_file = $temp_dir . '/' . wp_unique_filename($temp_dir, $final_name);
                if (!copy($final_file, $upload_file)) {
                    $upload_file = $final_file; // Use original if copy fails
                }
                
                $wp_result = $this->upload_file_to_wordpress($upload_file, $final_name, [
                    '_lightsync_dropbox_file_id'  => $file_id,
                    '_lightsync_dropbox_path'     => $file_path,
                    '_lightsync_dropbox_checksum' => $new_checksum,
                    '_lightsync_source'           => 'dropbox',
                    '_lightsync_last_synced_at'   => gmdate('Y-m-d H:i:s'),
                    '_lightsync_last_sync_kind'   => 'SYNC',
                ]);
                
                // Clean up copy if it exists and wasn't moved
                if ($upload_file !== $final_file && file_exists($upload_file)) {
                    @unlink($upload_file);
                }
                
                if (!is_wp_error($wp_result)) {
                    $results['wp_id'] = $wp_result;
                    \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Uploaded to WordPress: attachment #' . $wp_result);
                } else {
                    \LightSyncPro\Util\Logger::debug('[LSP Dropbox] WordPress upload failed: ' . $wp_result->get_error_message());
                    // Log activity BEFORE returning error so user sees what happened
                    self::add_activity(
                        sprintf('Dropbox → WordPress: Failed "%s" - %s', $file_name, $wp_result->get_error_message()),
                        'error',
                        'dropbox'
                    );
                    @unlink($final_file);
                    return $wp_result;
                }
            }
        }


        // Sync to Hub
        if ($sync_target === 'hub') {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox→Hub] Starting Hub sync check...');
            
            if (!function_exists('lsp_hub_sync_asset')) {
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox→Hub] ERROR: lsp_hub_sync_asset function not found! Is Hub plugin active?');
                self::add_activity(
                    sprintf('Dropbox → Hub: Failed "%s" - Hub plugin not active or loaded', $file_name),
                    'error',
                    'Dropbox'
                );
                @unlink($final_file);
                return ['error' => 'Hub plugin not active'];
            }
            
            if (!file_exists($final_file)) {
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox→Hub] ERROR: final_file does not exist: ' . $final_file);
                self::add_activity(
                    sprintf('Dropbox → Hub: Failed "%s" - Temp file not found', $file_name),
                    'error',
                    'Dropbox'
                );
                return ['error' => 'Temp file not found'];
            }
            
            $file_data = file_get_contents($final_file);
            if (!$file_data || strlen($file_data) < 100) {
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox→Hub] ERROR: File data empty or too small: ' . strlen($file_data ?? '') . ' bytes');
                self::add_activity(
                    sprintf('Dropbox → Hub: Failed "%s" - File data empty', $file_name),
                    'error',
                    'Dropbox'
                );
                @unlink($final_file);
                return ['error' => 'File data empty'];
            }
            
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox→Hub] Syncing ' . $final_name . ', bytes=' . strlen($file_data));
            
            $hub_site_ids = (array) self::get_opt('hub_selected_sites', []);
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox→Hub] hub_selected_sites=' . json_encode($hub_site_ids));
            
            if (empty($hub_site_ids)) {
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox→Hub] WARNING: No Hub sites selected! Will sync to all active sites.');
            }
            
            $hub_result = lsp_hub_sync_asset([
                'image_data' => $file_data,
                'filename' => $final_name,
                'content_type' => wp_check_filetype($final_name)['type'] ?? 'image/jpeg',
                'title' => pathinfo($final_name, PATHINFO_FILENAME),
                'alt_text' => '',
                'caption' => '',
            ], $hub_site_ids, 'dropbox', $file_id);
            
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox→Hub] result=' . wp_json_encode($hub_result));
            
            if (!empty($hub_result['success'])) {
                $results['hub_synced'] = $hub_result['synced'] ?? 0;
                $results['hub_skipped'] = $hub_result['skipped'] ?? 0;
                $hub_synced_count = $hub_result['synced'] ?? 0;
                $hub_skipped_count = $hub_result['skipped'] ?? 0;
                
                // Track Hub distribution for weekly digest
                if (class_exists('\\LightSyncPro\\Admin\\WeeklyDigest')) {
                    WeeklyDigest::track_hub_result($hub_result, 'dropbox');
                }
                
                if ($hub_synced_count > 0) {
                    self::add_activity(
                        sprintf('Dropbox → Hub: Synced "%s" to %d site(s)', $final_name, $hub_synced_count),
                        'success',
                        'Dropbox'
                    );
                } elseif ($hub_skipped_count > 0) {
                    self::add_activity(
                        sprintf('Dropbox → Hub: Skipped "%s" (unchanged on %d site(s))', $final_name, $hub_skipped_count),
                        'info',
                        'Dropbox'
                    );
                }
            } elseif (!empty($hub_result['error'])) {
                self::add_activity(
                    sprintf('Dropbox → Hub: Failed "%s" - %s', $final_name, $hub_result['error']),
                    'error',
                    'Dropbox'
                );
            } elseif (($hub_result['failed'] ?? 0) > 0) {
                self::add_activity(
                    sprintf('Dropbox → Hub: Failed "%s" on %d site(s)', $final_name, $hub_result['failed']),
                    'error',
                    'Dropbox'
                );
            } else {
                // No success, no error, no failed - unexpected state
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox→Hub] Unexpected result state: ' . wp_json_encode($hub_result));
                self::add_activity(
                    sprintf('Dropbox → Hub: Unknown result for "%s"', $final_name),
                    'warning',
                    'Dropbox'
                );
            }
            
            // Cleanup and return for Hub sync
            if (file_exists($final_file)) {
                @unlink($final_file);
            }
            
            // Track usage for Hub syncs
            if (!empty($hub_result['synced']) && $hub_result['synced'] > 0) {
                self::usage_consume($hub_result['synced']);
            }
            
            // Update last sync timestamp
            self::set_opt([
                'lightsync_last_sync'        => time(),
                'lightsync_last_sync_source' => 'dropbox',
            ]);
            
            return $results;
        }

        // Cleanup temp file
        if (file_exists($final_file)) {
            @unlink($final_file);
        }

        // Sync to Shopify if target includes Shopify
        if (($sync_target === 'shopify' || $sync_target === 'both') && class_exists('\LightSyncPro\Shopify\Shopify')) {
            if (!empty($shopify_bytes) && strlen($shopify_bytes) >= 100) {
                $is_webp = (substr($shopify_bytes, 0, 4) === 'RIFF' && substr($shopify_bytes, 8, 4) === 'WEBP');
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox→Shopify] Syncing ' . $final_name . ', bytes=' . strlen($shopify_bytes) . ', format=' . ($is_webp ? 'WebP' : 'original'));
                $shopify_result = \LightSyncPro\Shopify\Shopify::upload_file($shopify_bytes, $final_name, $file_id, 'dropbox');

                if (!is_wp_error($shopify_result)) {
                    if (!empty($shopify_result['skipped'])) {
                        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Shopify upload skipped - already exists');
                        $results['shopify_skipped'] = true;
                    } elseif (!empty($shopify_result['updated'])) {
                        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Shopify file updated: ' . ($shopify_result['file_id'] ?? ''));
                        $results['shopify_id'] = $shopify_result['file_id'] ?? '';
                        $results['shopify_updated'] = true;
                    } else {
                        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Uploaded to Shopify: ' . ($shopify_result['file_id'] ?? ''));
                        $results['shopify_id'] = $shopify_result['file_id'] ?? '';
                    }
                } else {
                    \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Shopify upload failed: ' . $shopify_result->get_error_message());
                    $results['shopify_error'] = $shopify_result->get_error_message();
                    self::add_activity(
                        sprintf('Dropbox → Shopify: Failed "%s" - %s', $file_name, $shopify_result->get_error_message()),
                        'error',
                        'dropbox'
                    );
                }
            }
        }

        // Track usage for newly synced files (not skipped)
        $was_new_sync = false;
        if (!empty($results['wp_id']) && empty($results['wp_skipped'])) {
            $was_new_sync = true;
            $was_new_sync = true;
        }
        
        if ($was_new_sync) {
            self::usage_consume(1);
            
            // Track storage stats for weekly digest
            // $image_data is original, $final_file was optimized
            $original_size = strlen($image_data ?? '');
            $optimized_size = isset($compressed['path']) && file_exists($compressed['path']) 
                ? filesize($compressed['path']) 
                : $original_size;
            if ($original_size > 0) {
                self::bump_storage_stats($original_size, $optimized_size, 'dropbox');
            }
        }

        // Add activity log entry
        $dest_text = 'WordPress';
        $activity_msg = 'Synced "' . $file_name . '" from Dropbox to ' . $dest_text;
        if (!empty($results['wp_skipped'])) {
            $activity_msg = 'Skipped "' . $file_name . '" from Dropbox (already synced)';
            $activity_msg = 'Synced "' . $file_name . '" from Dropbox (partial update)';
        }
        self::add_activity($activity_msg, $was_new_sync ? 'success' : 'info', 'dropbox');
        
        // Update last sync timestamp and source
        self::set_opt([
            'lightsync_last_sync'        => time(),
            'lightsync_last_sync_source' => 'dropbox',
        ]);

        return $results;
    }

    /**
     * Convert RAW file to JPEG using ImageMagick
     */
    private function convert_raw_file_to_jpeg($raw_file, $format) {
        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Attempting RAW conversion for format: ' . strtoupper($format));
        
        // Try PHP ImageMagick extension first (most reliable)
        if (extension_loaded('imagick')) {
            try {
                $imagick = new \Imagick();
                
                // Check if format is supported
                $formats = array_map('strtolower', $imagick->queryFormats());
                $format_map = [
                    'nef' => 'nef',
                    'cr2' => 'cr2',
                    'cr3' => 'cr3',
                    'arw' => 'arw',
                    'dng' => 'dng',
                    'orf' => 'orf',
                    'rw2' => 'rw2',
                    'pef' => 'pef',
                    'raf' => 'raf',
                ];
                
                $check_format = $format_map[$format] ?? $format;
                if (!in_array($check_format, $formats, true)) {
                    \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Imagick does not support format: ' . $format . '. Available formats: ' . implode(', ', array_slice($formats, 0, 20)) . '...');
                    // Don't return yet - try command line as fallback
                } else {
                    $imagick->readImage($raw_file);
                    $imagick->setImageFormat('jpeg');
                    $imagick->setImageCompressionQuality(92);
                    $imagick->autoOrient();
                    
                    $jpeg_file = preg_replace('/\.[^.]+$/', '.jpg', $raw_file);
                    $imagick->writeImage($jpeg_file);
                    $imagick->destroy();
                    
                    if (file_exists($jpeg_file) && filesize($jpeg_file) > 0) {
                        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Imagick conversion successful');
                        return $jpeg_file;
                    }
                }
            } catch (\Exception $e) {
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Imagick conversion failed: ' . $e->getMessage());
            }
        }
        
        // Try command line ImageMagick as fallback
        $convert_path = null;
        $paths = ['/usr/bin/convert', '/usr/local/bin/convert'];
        
        foreach ($paths as $path) {
            if (file_exists($path) && is_executable($path)) {
                $convert_path = $path;
                break;
            }
        }
        
        // Also try `which convert`
        if (!$convert_path) {
            $which_output = shell_exec('which convert 2>/dev/null');
            if ($which_output) {
                $convert_path = trim($which_output);
            }
        }

        if ($convert_path) {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Trying command line convert: ' . $convert_path);
            
            $jpeg_file = preg_replace('/\.[^.]+$/', '.jpg', $raw_file);
            $cmd = sprintf(
                '%s %s -auto-orient -quality 92 %s 2>&1',
                escapeshellcmd($convert_path),
                escapeshellarg($raw_file),
                escapeshellarg($jpeg_file)
            );

            $output = [];
            $return = 0;
            exec($cmd, $output, $return);

            if ($return === 0 && file_exists($jpeg_file) && filesize($jpeg_file) > 0) {
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Command line convert successful');
                return $jpeg_file;
            } else {
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] convert command failed (code ' . $return . '): ' . implode("\n", $output));
            }
        } else {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] No command line convert found');
        }
        
        // Check for dcraw as last resort (handles many RAW formats)
        $dcraw_path = shell_exec('which dcraw 2>/dev/null');
        if ($dcraw_path) {
            $dcraw_path = trim($dcraw_path);
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Trying dcraw: ' . $dcraw_path);
            
            // dcraw outputs PPM, which we then need to convert to JPEG
            $ppm_file = preg_replace('/\.[^.]+$/', '.ppm', $raw_file);
            $jpeg_file = preg_replace('/\.[^.]+$/', '.jpg', $raw_file);
            
            // Extract to PPM with -c (stdout) and -w (camera white balance)
            $cmd = sprintf('%s -c -w %s > %s 2>&1', escapeshellcmd($dcraw_path), escapeshellarg($raw_file), escapeshellarg($ppm_file));
            exec($cmd, $output, $return);
            
            if (file_exists($ppm_file) && filesize($ppm_file) > 0) {
                // Convert PPM to JPEG using GD
                $gd_img = @imagecreatefrompnm($ppm_file) ?: @imagecreatefromstring(file_get_contents($ppm_file));
                if ($gd_img) {
                    imagejpeg($gd_img, $jpeg_file, 92);
                    imagedestroy($gd_img);
                    @unlink($ppm_file);
                    
                    if (file_exists($jpeg_file) && filesize($jpeg_file) > 0) {
                        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] dcraw + GD conversion successful');
                        return $jpeg_file;
                    }
                }
                @unlink($ppm_file);
            }
        }
        
        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] All RAW conversion methods failed for format: ' . strtoupper($format));
        return false;
    }

    /**
     * Compress image to WebP
     * Returns false if compression fails - caller should use original file
     */
    private function compress_dropbox_image($file_path, $filename) {
        if (!file_exists($file_path)) {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Compression skipped - file not found: ' . $file_path);
            return false;
        }

        if (!preg_match('/\.[^.]+$/', $filename)) {
            $filename .= '.jpg';
        }

        // Skip if already WebP
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        if ($ext === 'webp') {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Already WebP, skipping compression: ' . $filename);
            return false;
        }

        try {
            $quality = 82;
            $original_size = filesize($file_path);
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Attempting WebP compression on ' . $filename . ' (' . $original_size . ' bytes, ext=' . $ext . ')');
            $webp_path = preg_replace('/\.[^.]+$/', '.webp', $file_path);
            $image = wp_get_image_editor($file_path);
            if (!is_wp_error($image)) {
                $image->set_quality($quality);
                $result = $image->save($webp_path, 'image/webp');
                if (!is_wp_error($result) && !empty($result['path']) && file_exists($result['path']) && filesize($result['path']) > 0) {
                    $new_size = filesize($result['path']);
                    $new_filename = preg_replace('/\.[^.]+$/', '.webp', $filename);
                    \LightSyncPro\Util\Logger::debug('[LSP Dropbox] WebP compression successful: ' . $original_size . ' → ' . $new_size . ' bytes (' . round((1 - $new_size / max($original_size, 1)) * 100) . '% savings)');
                    return [
                        'path' => $result['path'],
                        'filename' => $new_filename,
                    ];
                } else {
                    $error_msg = is_wp_error($result) ? $result->get_error_message() : 'save returned empty path';
                    \LightSyncPro\Util\Logger::debug('[LSP Dropbox] WebP save failed: ' . $error_msg . ' (webp_path=' . $webp_path . ')');
                }
            } else {
                \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Could not create image editor: ' . $image->get_error_message() . ' (file=' . $file_path . ', size=' . $original_size . ')');
            }
        } catch (\Exception $e) {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Compression exception: ' . $e->getMessage());
        } catch (\Error $e) {
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox] Compression error: ' . $e->getMessage());
        }

        \LightSyncPro\Util\Logger::debug('[LSP Dropbox] WebP compression failed, will use original format');
        return false;
    }

    /**
     * Compress image bytes to WebP
     * 
     * @param string $image_data Raw image bytes
     * @param string $filename Original filename
     * @return array ['data' => compressed bytes, 'filename' => new filename, 'content_type' => mime type] or original if fails
     */
    public static function compress_image_bytes($image_data, $filename) {
        // Create temp file for compression
        $upload_dir = wp_upload_dir();
        $temp_dir = $upload_dir['basedir'] . '/lightsync-temp';
        if (!file_exists($temp_dir)) {
            wp_mkdir_p($temp_dir);
        }
        
        $temp_file = $temp_dir . '/' . wp_unique_filename($temp_dir, $filename);
        if (file_put_contents($temp_file, $image_data) === false) {
            return [
                'data' => $image_data,
                'filename' => $filename,
                'content_type' => wp_check_filetype($filename)['type'] ?? 'image/jpeg',
            ];
        }
        
        $quality = 82;
        $result = null;
        
        try {
            $webp_path = preg_replace('/\.[^.]+$/', '.webp', $temp_file);
            $image = wp_get_image_editor($temp_file);
            if (!is_wp_error($image)) {
                $image->set_quality($quality);
                $saved = $image->save($webp_path, 'image/webp');
                if (!is_wp_error($saved) && !empty($saved['path']) && file_exists($saved['path'])) {
                    $result = [
                        'data' => file_get_contents($saved['path']),
                        'filename' => preg_replace('/\.[^.]+$/', '.webp', $filename),
                        'content_type' => 'image/webp',
                    ];
                    @unlink($saved['path']);
                }
            }
        } catch (\Throwable $e) {
            error_log('[LSP] compress_image_bytes exception: ' . $e->getMessage());
        }
        
        // Cleanup
        @unlink($temp_file);
        
        if ($result && !empty($result['data'])) {
            return $result;
        }
        
        return [
            'data' => $image_data,
            'filename' => $filename,
            'content_type' => wp_check_filetype($filename)['type'] ?? 'image/jpeg',
        ];
    }

    /**
     * Upload file to WordPress media library
     */
    private function upload_file_to_wordpress($file_path, $filename, $meta = []) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        if (!file_exists($file_path)) {
            return new \WP_Error('file_not_found', 'File not found: ' . $file_path);
        }

        // Get mime type
        $mime_type = wp_check_filetype($filename);
        
        $file_array = [
            'name'     => $filename,
            'type'     => $mime_type['type'] ?: 'image/jpeg',
            'tmp_name' => $file_path,
            'error'    => 0,
            'size'     => filesize($file_path),
        ];

        // Upload to media library
        $attachment_id = media_handle_sideload($file_array, 0);

        if (is_wp_error($attachment_id)) {
            return $attachment_id;
        }

        // Add custom meta
        foreach ($meta as $key => $value) {
            update_post_meta($attachment_id, $key, $value);
        }

        return $attachment_id;
    }

    /**
     * Upload image data to WordPress media library
     */
    private function upload_to_wordpress_media($image_data, $filename, $meta = []) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        // Create temp file
        $upload_dir = wp_upload_dir();
        $temp_file = $upload_dir['basedir'] . '/' . wp_unique_filename($upload_dir['basedir'], $filename);
        
        if (file_put_contents($temp_file, $image_data) === false) {
            return new \WP_Error('write_failed', 'Could not write temp file');
        }

        // Get mime type
        $mime_type = wp_check_filetype($filename);
        
        $file_array = [
            'name'     => $filename,
            'type'     => $mime_type['type'] ?: 'image/jpeg',
            'tmp_name' => $temp_file,
            'error'    => 0,
            'size'     => filesize($temp_file),
        ];

        // Upload to media library
        $attachment_id = media_handle_sideload($file_array, 0);

        // Clean up temp file if it still exists
        if (file_exists($temp_file)) {
            @unlink($temp_file);
        }

        if (is_wp_error($attachment_id)) {
            return $attachment_id;
        }

        // Add custom meta
        foreach ($meta as $key => $value) {
            update_post_meta($attachment_id, $key, $value);
        }

        return $attachment_id;
    }

    /**
     * Convert RAW image to JPEG using ImageMagick
     */
    private function convert_raw_to_jpeg($raw_data, $format) {
        // Check if ImageMagick is available
        if (!extension_loaded('imagick')) {
            // Return original data - WordPress will handle as-is
            return new \WP_Error('no_imagick', 'ImageMagick not available for RAW conversion');
        }

        try {
            $imagick = new \Imagick();
            $imagick->readImageBlob($raw_data);
            $imagick->setImageFormat('jpeg');
            $imagick->setImageCompressionQuality(92);
            
            // Auto-orient based on EXIF
            $imagick->autoOrient();
            
            $jpeg_data = $imagick->getImageBlob();
            $imagick->destroy();
            
            return $jpeg_data;
        } catch (\Exception $e) {
            return new \WP_Error('convert_failed', 'RAW conversion failed: ' . $e->getMessage());
        }
    }

    /**
     * Save Figma sync target preference
     */
    public function ajax_figma_save_target() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $target = isset($_POST['target']) ? sanitize_text_field($_POST['target']) : 'wp';
        if (!in_array($target, ['wp', 'shopify', 'both'], true)) {
            $target = 'wp';
        }

        self::set_opt(['figma_sync_target' => $target]);

        wp_send_json_success(['target' => $target]);
    }

    /**
     * Sync selected Figma frames to WordPress
     */
    public function ajax_figma_sync_frames() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        // Sync gate check
        $gate = lsp_gate_check();
        if ( is_wp_error( $gate ) ) {
            wp_send_json_error([
                'error' => 'sync_disabled',
                'message' => $gate->get_error_message(),
            ], 403);
        }

        if (!\LightSyncPro\OAuth\FigmaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Figma not connected. Please reconnect.']);
        }

        $file_key = isset($_POST['file_key']) ? sanitize_text_field($_POST['file_key']) : '';
        $frame_ids = isset($_POST['frame_ids']) ? array_map('sanitize_text_field', (array)$_POST['frame_ids']) : [];
        $format = isset($_POST['format']) ? sanitize_text_field($_POST['format']) : 'png';
        $scale = isset($_POST['scale']) ? (float)$_POST['scale'] : 2;
        // Accept sync_target from POST, fallback to saved option
        $sync_target = isset($_POST['sync_target']) ? sanitize_text_field($_POST['sync_target']) : (self::get_opt('figma_sync_target') ?: 'wp');
        
        \LightSyncPro\Util\Logger::debug('[LSP Figma] ajax_figma_sync_frames: sync_target=' . $sync_target . ', saved option=' . (self::get_opt('figma_sync_target') ?: 'not set'));

        if (!$file_key || empty($frame_ids)) {
            wp_send_json_error(['error' => 'No file or frames selected']);
        }

        // Save selection for this file
        self::set_opt(['figma_selected_frames_' . $file_key => $frame_ids]);

        $synced = 0;
        $errors = [];

        // Get file name for activity log
        $files = (array) self::get_opt('figma_files', []);
        $file_name = $files[$file_key]['name'] ?? 'Figma File';

        // For WebP/AVIF, export as PNG then convert locally
        $export_format = in_array($format, ['webp', 'avif'], true) ? 'png' : $format;

        // Export all frames at once (Figma API supports batch export)
        $export_result = \LightSyncPro\OAuth\FigmaOAuth::export_images($file_key, $frame_ids, $export_format, $scale);
        
        if (is_wp_error($export_result)) {
            wp_send_json_error(['error' => $export_result->get_error_message()]);
        }

        // Get frame names and file info for better filenames and version tracking
        $frame_info = \LightSyncPro\OAuth\FigmaOAuth::get_file_frames($file_key, true);
        $frame_names = [];
        $frame_types = [];
        $last_modified = null;
        if (!is_wp_error($frame_info)) {
            foreach ($frame_info['frames'] as $f) {
                $frame_names[$f['id']] = $f['name'];
                $frame_types[$f['id']] = $f['type'] ?? 'FRAME';
            }
            $last_modified = $frame_info['last_modified'] ?? null;
        }

        // Track synced element types
        $synced_by_type = [];
        
        // Download and import each exported frame (pass original user format for conversion)
        foreach ($export_result as $node_id => $image_url) {
            if (!$image_url) {
                $errors[] = "Export failed for frame: " . ($frame_names[$node_id] ?? $node_id);
                continue;
            }

            $result = $this->import_figma_frame($file_key, $node_id, $image_url, $frame_names[$node_id] ?? $node_id, $format, $sync_target, $last_modified);
            
            if (is_wp_error($result)) {
                $errors[] = $result->get_error_message();
            } else {
                $synced++;
                // Track by type
                $type = $frame_types[$node_id] ?? 'FRAME';
                $synced_by_type[$type] = ($synced_by_type[$type] ?? 0) + 1;
            }
        }

        $dest_label = 'WordPress';

        if ($synced > 0) {
            self::usage_consume($synced);

            // Build type breakdown string
            $type_labels = [
                'FRAME' => 'frame',
                'COMPONENT' => 'component',
                'COMPONENT_SET' => 'component set',
                'GROUP' => 'group',
                'RECTANGLE' => 'rectangle',
                'ELLIPSE' => 'ellipse',
                'TEXT' => 'text',
                'VECTOR' => 'vector',
                'INSTANCE' => 'instance',
            ];
            
            $type_parts = [];
            foreach ($synced_by_type as $type => $count) {
                $label = $type_labels[$type] ?? strtolower($type);
                $type_parts[] = $count . ' ' . $label . ($count > 1 ? 's' : '');
            }
            $type_breakdown = !empty($type_parts) ? ' (' . implode(', ', $type_parts) . ')' : '';

            self::add_activity(
                "Figma → {$dest_label} Sync Complete: {$synced} element(s){$type_breakdown} from \"{$file_name}\"",
                'success',
                'figma'
            );
            
            // Update last sync header (like Canva/Lightroom)
            self::set_opt([
                'lightsync_last_sync_ts'     => time(),
                'lightsync_last_sync_source' => 'figma',
                'lightsync_last_sync_status' => 'complete',
            ]);
            
            // Clear frame cache so synced status shows immediately
            // Cache key format: lsp_figma_frames_{file_key}_{nested|top}
            delete_transient('lsp_figma_frames_' . $file_key . '_nested');
            delete_transient('lsp_figma_frames_' . $file_key . '_top');
        }

        wp_send_json_success([
            'synced'  => $synced,
            'errors'  => $errors,
            'message' => "Synced {$synced} frame(s) to {$dest_label}",
            'last_sync_ts' => time(),
        ]);
    }

    /**
     * Import a single Figma frame to WordPress
     */
    private function import_figma_frame($file_key, $node_id, $image_url, $frame_name, $format, $sync_target, $file_last_modified = null) {
        \LightSyncPro\Util\Logger::debug('[LSP import_figma_frame] Starting: frame=' . $frame_name . ', sync_target=' . $sync_target);
        
        try {
            // For WebP/AVIF, we export as PNG from Figma then convert
            $figma_format = in_array($format, ['webp', 'avif'], true) ? 'png' : $format;
            $target_format = $format; // User's desired format
            
            // Generate filename from frame name
            $filename = sanitize_file_name($frame_name) . '.' . $figma_format;
            
            // Check if already synced
            $asset_key = 'figma-' . $file_key . '-' . $node_id;
            $existing = $this->find_figma_attachment($asset_key);

            // Download the image
            $tmp_file = download_url($image_url, 120); // Figma URLs can be slow
            if (is_wp_error($tmp_file)) {
                \LightSyncPro\Util\Logger::debug('[LSP Figma] Download failed: ' . $tmp_file->get_error_message());
                return $tmp_file;
            }

            // Track original size for storage stats
            $original_size = filesize($tmp_file);

            // Convert to WebP/AVIF if requested
            if (in_array($target_format, ['webp', 'avif'], true)) {
                $converted = $this->convert_figma_image($tmp_file, $filename, $target_format);
                if ($converted && file_exists($converted)) {
                    @unlink($tmp_file);
                    $tmp_file = $converted;
                    $filename = basename($converted);
                } else {
                    \LightSyncPro\Util\Logger::debug('[LSP Figma] Conversion to ' . $target_format . ' failed, using PNG');
                }
            }
            // Apply global compression settings for PNG/JPG if enabled
            elseif (in_array($figma_format, ['png', 'jpg'], true)) {
                $compressed = $this->compress_figma_image($tmp_file, $filename);
                if ($compressed && file_exists($compressed)) {
                    @unlink($tmp_file);
                    $tmp_file = $compressed;
                    $filename = basename($compressed);
                }
            }

            // Track optimized size for storage stats
            $optimized_size = filesize($tmp_file);

            // Save file bytes for Shopify BEFORE WP upload (media_handle_sideload moves the file)
            $shopify_bytes = null;
            if ($sync_target === 'shopify' || $sync_target === 'both') {
                $shopify_bytes = @file_get_contents($tmp_file);
            }

            $attachment_id = null;
            if ($sync_target === 'wp' || $sync_target === 'both') {
                if ($existing) {
                    // Update existing attachment
                    $attachment_id = $this->update_figma_attachment($existing, $tmp_file, $filename);
                    
                    if (is_wp_error($attachment_id)) {
                        \LightSyncPro\Util\Logger::debug('[LSP Figma] WordPress update failed: ' . $attachment_id->get_error_message());
                        self::add_activity(
                            sprintf('Figma → WordPress: Failed to update "%s" - %s', $frame_name, $attachment_id->get_error_message()),
                            'error',
                            'figma'
                        );
                        @unlink($tmp_file);
                        return $attachment_id;
                    }
                } else {
                    // Create new attachment
                    $file_array = [
                        'name'     => $filename,
                        'tmp_name' => $tmp_file,
                    ];

                    $attachment_id = media_handle_sideload($file_array, 0);

                    if (is_wp_error($attachment_id)) {
                        \LightSyncPro\Util\Logger::debug('[LSP Figma] WordPress upload failed: ' . $attachment_id->get_error_message());
                        self::add_activity(
                            sprintf('Figma → WordPress: Failed "%s" - %s', $frame_name, $attachment_id->get_error_message()),
                            'error',
                            'figma'
                        );
                        @unlink($tmp_file);
                        return $attachment_id;
                    }

                    // Set metadata
                    update_post_meta($attachment_id, '_lightsync_source', 'figma');
                    update_post_meta($attachment_id, '_lightsync_figma_file_key', $file_key);
                    update_post_meta($attachment_id, '_lightsync_figma_node_id', $node_id);
                    update_post_meta($attachment_id, '_lightsync_asset_id', $asset_key);
                    update_post_meta($attachment_id, '_lightsync_last_sync_kind', 'import');

                    // Set title from frame name
                    wp_update_post([
                        'ID'         => $attachment_id,
                        'post_title' => $frame_name,
                    ]);
                }
                
                // Update sync timestamp and file version for existing or new
                if ($attachment_id && !is_wp_error($attachment_id)) {
                    update_post_meta($attachment_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
                    if ($file_last_modified) {
                        update_post_meta($attachment_id, '_lightsync_figma_file_version', $file_last_modified);
                    }
                    
                    // Track storage savings
                    if ($original_size > 0) {
                        update_post_meta($attachment_id, '_lightsync_original_bytes', $original_size);
                        update_post_meta($attachment_id, '_lightsync_optimized_bytes', $optimized_size);
                        self::bump_storage_stats($original_size, $optimized_size, 'figma');
                    }
                    
                    // Log success
                    self::add_activity(
                        sprintf('Figma → WordPress: %s "%s"', $existing ? 'Updated' : 'Synced', $frame_name),
                        'success',
                        'figma'
                    );
                }
            }


            // Handle Hub sync if needed
            if ($sync_target === 'hub' && function_exists('lsp_hub_sync_asset')) {
                $file_for_hub = $tmp_file;
                if (file_exists($file_for_hub)) {
                    $file_bytes = @file_get_contents($file_for_hub);
                    if ($file_bytes) {
                        \LightSyncPro\Util\Logger::debug('[LSP Figma→Hub] Syncing ' . $frame_name . ', bytes=' . strlen($file_bytes));
                        
                        $hub_site_ids = (array) self::get_opt('hub_selected_sites', []);
                        
                        $hub_result = lsp_hub_sync_asset([
                            'image_data' => $file_bytes,
                            'filename' => $filename,
                            'content_type' => wp_check_filetype($filename)['type'] ?? 'image/png',
                            'title' => $frame_name,
                            'alt_text' => '',
                            'caption' => '',
                        ], $hub_site_ids, 'figma', $asset_key);
                        
                        \LightSyncPro\Util\Logger::debug('[LSP Figma→Hub] result=' . wp_json_encode($hub_result));
                        
                        if (!empty($hub_result['success'])) {
                            $hub_synced_count = $hub_result['synced'] ?? 0;
                            $hub_skipped_count = $hub_result['skipped'] ?? 0;
                            
                            // Track Hub distribution for weekly digest
                            if (class_exists('\\LightSyncPro\\Admin\\WeeklyDigest')) {
                                WeeklyDigest::track_hub_result($hub_result, 'figma');
                            }
                            
                            if ($hub_synced_count > 0) {
                                self::add_activity(
                                    sprintf('Figma → Hub: Synced "%s" to %d site(s)', $frame_name, $hub_synced_count),
                                    'success',
                                    'Figma'
                                );
                            } elseif ($hub_skipped_count > 0) {
                                self::add_activity(
                                    sprintf('Figma → Hub: Skipped "%s" (unchanged on %d site(s))', $frame_name, $hub_skipped_count),
                                    'info',
                                    'Figma'
                                );
                            }
                        } elseif (!empty($hub_result['error'])) {
                            self::add_activity(
                                sprintf('Figma → Hub: Failed "%s" - %s', $frame_name, $hub_result['error']),
                                'error',
                                'Figma'
                            );
                        } elseif (($hub_result['failed'] ?? 0) > 0) {
                            self::add_activity(
                                sprintf('Figma → Hub: Failed "%s" on %d site(s)', $frame_name, $hub_result['failed']),
                                'error',
                                'Figma'
                            );
                        }
                    }
                }
            }

            // Sync to Shopify if target is 'shopify' or 'both'
            if (($sync_target === 'shopify' || $sync_target === 'both') && class_exists('\LightSyncPro\Shopify\Shopify')) {
                if (!empty($shopify_bytes) && strlen($shopify_bytes) >= 100) {
                    \LightSyncPro\Util\Logger::debug('[LSP Figma→Shopify] Syncing ' . $frame_name . ' (' . strlen($shopify_bytes) . ' bytes, asset_key=' . $asset_key . ')');
                    $shopify_result = \LightSyncPro\Shopify\Shopify::upload_file($shopify_bytes, $filename, $asset_key, 'figma', $frame_name);

                    if (!is_wp_error($shopify_result)) {
                        \LightSyncPro\Util\Logger::debug('[LSP Figma→Shopify] Success: file_id=' . ($shopify_result['file_id'] ?? 'none'));
                        self::add_activity(
                            sprintf('Figma → Shopify: Synced "%s"', $frame_name),
                            'success',
                            'figma'
                        );
                    } else {
                        \LightSyncPro\Util\Logger::debug('[LSP Figma→Shopify] Error: ' . $shopify_result->get_error_message());
                        self::add_activity(
                            sprintf('Figma → Shopify: Failed "%s" - %s', $frame_name, $shopify_result->get_error_message()),
                            'error',
                            'figma'
                        );
                    }
                }
            }

            // Cleanup temp files
            if (file_exists($tmp_file)) {
                @unlink($tmp_file);
            }


            return ['attachment_id' => $attachment_id];

        } catch (\Throwable $e) {
            return new \WP_Error('sync_error', $e->getMessage());
        }
    }

    /**
     * Find existing attachment by Figma asset key (only active attachments)
     */
    private function find_figma_attachment($asset_key) {
        global $wpdb;

        $attachment_id = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT pm.post_id FROM {$wpdb->postmeta} pm
                 INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID AND p.post_status = 'inherit'
                 WHERE pm.meta_key = '_lightsync_asset_id' AND pm.meta_value = %s LIMIT 1",
                $asset_key
            )
        );

        return $attachment_id ? (int) $attachment_id : null;
    }

    /**
     * Update existing Figma attachment with new file
     */
    private function update_figma_attachment($attachment_id, $tmp_file, $filename) {
        if (!function_exists('wp_generate_attachment_metadata')) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }

        // Get current file path — overwrite in place to preserve URL & Shopify mapping
        $current_path = get_attached_file($attachment_id);
        if (!$current_path) {
            return new \WP_Error('no_attached_file', 'Cannot find current attachment file path');
        }

        $current_ext = strtolower(pathinfo($current_path, PATHINFO_EXTENSION));
        $new_ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $target_path = $current_path;

        // If extension changed, adjust path but keep same directory/base
        if ($current_ext !== $new_ext) {
            $target_path = preg_replace('/\.' . preg_quote($current_ext, '/') . '$/', '.' . $new_ext, $current_path);
        }

        // Delete old thumbnails BEFORE overwriting
        $old_meta = wp_get_attachment_metadata($attachment_id);
        if (!empty($old_meta['sizes'])) {
            $old_dir = dirname($current_path);
            foreach ($old_meta['sizes'] as $size) {
                $thumb = $old_dir . '/' . $size['file'];
                if (file_exists($thumb)) {
                    @unlink($thumb);
                }
            }
        }

        // Write new file to same path (overwrite in place)
        if (!copy($tmp_file, $target_path)) {
            return new \WP_Error('copy_failed', 'Failed to overwrite attachment file');
        }

        // If extension changed, remove old file and update path
        if ($target_path !== $current_path) {
            if (file_exists($current_path)) {
                @unlink($current_path);
            }
            update_attached_file($attachment_id, $target_path);
        }

        // Update MIME type
        $filetype = wp_check_filetype($target_path);
        if (!empty($filetype['type'])) {
            wp_update_post([
                'ID'             => $attachment_id,
                'post_mime_type' => $filetype['type'],
            ]);
        }

        // Regenerate thumbnails
        wp_update_attachment_metadata($attachment_id, wp_generate_attachment_metadata($attachment_id, $target_path));

        update_post_meta($attachment_id, '_lightsync_last_synced_at', gmdate('Y-m-d H:i:s'));
        update_post_meta($attachment_id, '_lightsync_last_sync_kind', 'update');

        return $attachment_id;
    }

    /**
     * Compress Figma image to WebP
     */
    private function compress_figma_image($file_path, $filename) {
        if (!file_exists($file_path)) {
            return null;
        }

        try {
            $quality = 82;
            $webp_path = preg_replace('/\.[^.]+$/', '.webp', $file_path);
            $image = wp_get_image_editor($file_path);
            if (!is_wp_error($image)) {
                $image->set_quality($quality);
                $result = $image->save($webp_path, 'image/webp');
                if (!is_wp_error($result) && !empty($result['path']) && file_exists($result['path']) && filesize($result['path']) > 0) {
                    return $result['path'];
                }
            }
        } catch (\Throwable $e) {
            \LightSyncPro\Util\Logger::debug('[LSP Figma] WebP compression failed: ' . $e->getMessage());
        }

        return null;
    }

    /**
     * Convert Figma image to WebP format
     */
    private function convert_figma_image($file_path, $filename, $target_format) {
        try {
            $base = pathinfo($filename, PATHINFO_FILENAME);
            $new_filename = $base . '.webp';
            
            $upload_dir = wp_upload_dir();
            $output_path = $upload_dir['path'] . '/' . wp_unique_filename($upload_dir['path'], $new_filename);
            
            $quality = 82;
            
            $image = wp_get_image_editor($file_path);
            if (!is_wp_error($image)) {
                $image->set_quality($quality);
                $result = $image->save($output_path, 'image/webp');
                if (!is_wp_error($result) && !empty($result['path']) && file_exists($result['path'])) {
                    \LightSyncPro\Util\Logger::debug('[LSP Figma] Converted to WebP: ' . $result['path']);
                    return $result['path'];
                }
            } else {
                \LightSyncPro\Util\Logger::debug('[LSP Figma] Image editor error: ' . $image->get_error_message());
            }
        } catch (\Throwable $e) {
            \LightSyncPro\Util\Logger::debug('[LSP Figma] Conversion to WebP failed: ' . $e->getMessage());
        }

        return null;
    }

    /**
     * Get user's Figma teams for browsing
     */
    public function ajax_figma_get_teams() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\FigmaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Figma not connected. Please reconnect.']);
        }

        // Clear cache if requested
        if (!empty($_POST['clear_cache'])) {
            delete_transient('lightsync_figma_teams');
        }

        // Cache teams for 5 minutes
        $cache_key = 'lightsync_figma_teams';
        $cached = get_transient($cache_key);
        if ($cached !== false && !empty($_POST['use_cache'])) {
            wp_send_json_success(['teams' => $cached, 'cached' => true]);
        }

        // Clear any stale cache for fresh fetch
        delete_transient($cache_key);

        $me = \LightSyncPro\OAuth\FigmaOAuth::get_me();
        if (is_wp_error($me)) {
            $error_msg = $me->get_error_message();
            \LightSyncPro\Util\Logger::debug('[LSP Figma] get_teams error: ' . $error_msg);
            
            // Check for common issues
            if (strpos($error_msg, '403') !== false) {
                wp_send_json_error([
                    'error' => 'Access denied (403). The Figma OAuth app may need the "files:read" scope enabled. Please check your Figma app settings at figma.com/developers/apps',
                    'debug' => $error_msg
                ]);
            }
            
            wp_send_json_error(['error' => $error_msg]);
        }

        \LightSyncPro\Util\Logger::debug('[LSP Figma] get_me response: ' . print_r($me, true));

        // Extract team memberships
        $teams = [];
        foreach (($me['teams'] ?? []) as $team) {
            $teams[] = [
                'id'   => $team['id'] ?? '',
                'name' => $team['name'] ?? 'Unknown Team',
            ];
        }

        // If no teams found, provide helpful message
        if (empty($teams)) {
            wp_send_json_success([
                'teams' => [],
                'user'  => [
                    'id'    => $me['id'] ?? '',
                    'email' => $me['email'] ?? '',
                    'handle' => $me['handle'] ?? '',
                ],
                'message' => 'No teams found. If you have a personal Figma account, use the "Paste URL" tab to add files directly.'
            ]);
        }

        set_transient($cache_key, $teams, 5 * MINUTE_IN_SECONDS);

        wp_send_json_success([
            'teams' => $teams,
            'user'  => [
                'id'    => $me['id'] ?? '',
                'email' => $me['email'] ?? '',
                'handle' => $me['handle'] ?? '',
            ],
        ]);
    }

    /**
     * Get projects in a Figma team
     */
    public function ajax_figma_get_projects() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\FigmaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Figma not connected']);
        }

        $team_id = isset($_POST['team_id']) ? sanitize_text_field($_POST['team_id']) : '';
        if (!$team_id) {
            wp_send_json_error(['error' => 'No team ID provided']);
        }

        // Cache projects per team for 2 minutes
        $cache_key = 'lightsync_figma_projects_' . $team_id;
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            wp_send_json_success(['projects' => $cached]);
        }

        $result = \LightSyncPro\OAuth\FigmaOAuth::get_team_projects($team_id);
        if (is_wp_error($result)) {
            wp_send_json_error(['error' => $result->get_error_message()]);
        }

        $projects = [];
        foreach (($result['projects'] ?? []) as $project) {
            $projects[] = [
                'id'   => $project['id'] ?? '',
                'name' => $project['name'] ?? 'Untitled Project',
            ];
        }

        set_transient($cache_key, $projects, 2 * MINUTE_IN_SECONDS);

        wp_send_json_success(['projects' => $projects]);
    }

    /**
     * Get files in a Figma project (for browsing)
     */
    public function ajax_figma_browse_files() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\FigmaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Figma not connected']);
        }

        $project_id = isset($_POST['project_id']) ? sanitize_text_field($_POST['project_id']) : '';
        if (!$project_id) {
            wp_send_json_error(['error' => 'No project ID provided']);
        }

        // Cache files per project for 2 minutes
        $cache_key = 'lightsync_figma_files_' . $project_id;
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            wp_send_json_success(['files' => $cached]);
        }

        $result = \LightSyncPro\OAuth\FigmaOAuth::get_project_files($project_id);
        if (is_wp_error($result)) {
            wp_send_json_error(['error' => $result->get_error_message()]);
        }

        $files = [];
        foreach (($result['files'] ?? []) as $file) {
            $files[] = [
                'key'          => $file['key'] ?? '',
                'name'         => $file['name'] ?? 'Untitled',
                'thumbnail_url' => $file['thumbnail_url'] ?? '',
                'last_modified' => $file['last_modified'] ?? null,
            ];
        }

        set_transient($cache_key, $files, 2 * MINUTE_IN_SECONDS);

        wp_send_json_success(['files' => $files]);
    }

    /**
     * Add a Figma file by key (from browser selection)
     */
    public function ajax_figma_add_file_by_key() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!\LightSyncPro\OAuth\FigmaOAuth::is_connected()) {
            wp_send_json_error(['error' => 'Figma not connected']);
        }

        $file_key = isset($_POST['file_key']) ? sanitize_text_field($_POST['file_key']) : '';
        $file_name = isset($_POST['file_name']) ? sanitize_text_field($_POST['file_name']) : '';
        $thumbnail = isset($_POST['thumbnail']) ? esc_url_raw($_POST['thumbnail']) : '';

        if (!$file_key) {
            wp_send_json_error(['error' => 'No file key provided']);
        }

        // Store file info
        $files = (array) self::get_opt('figma_files', []);
        
        // Check if already added
        if (isset($files[$file_key])) {
            wp_send_json_success([
                'file' => $files[$file_key],
                'files' => array_values($files),
                'already_added' => true,
            ]);
        }

        // Get fresh metadata if name not provided
        if (!$file_name) {
            $file_meta = \LightSyncPro\OAuth\FigmaOAuth::get_file_meta($file_key);
            if (!is_wp_error($file_meta)) {
                // Figma meta endpoint wraps response in 'file' object and uses snake_case
                $file_data = $file_meta['file'] ?? $file_meta;
                $file_name = $file_data['name'] ?? 'Untitled';
                $thumbnail = $file_data['thumbnail_url'] ?? $file_data['thumbnailUrl'] ?? $thumbnail;
            }
        }

        $files[$file_key] = [
            'key'        => $file_key,
            'name'       => $file_name ?: 'Untitled',
            'thumbnail'  => $thumbnail,
            'added_at'   => time(),
        ];

        self::set_opt(['figma_files' => $files]);

        wp_send_json_success([
            'file' => $files[$file_key],
            'files' => array_values($files),
        ]);
    }

    /* ==================== END FIGMA AJAX HANDLERS ==================== */

    /* ==================== AUTO-SYNC HANDLERS ==================== */
    
    /**
     * Manually process the Dropbox queue (for debugging/forcing processing)
     */
    public function ajax_dropbox_process_queue() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $queue = get_option('lightsync_dropbox_sync_queue', []);
        
        if (empty($queue)) {
            wp_send_json_success([
                'processed' => 0,
                'message' => 'Queue is empty',
            ]);
            return;
        }
        
        // Extend timeout
        @set_time_limit(180);
        @ini_set('max_execution_time', '180');
        
        $processed = 0;
        $errors = [];
        $max_process = min(5, count($queue)); // Process up to 5 files
        
        for ($i = 0; $i < $max_process; $i++) {
            $current_queue = get_option('lightsync_dropbox_sync_queue', []);
            if (empty($current_queue)) {
                break;
            }
            
            $item = array_shift($current_queue);
            update_option('lightsync_dropbox_sync_queue', $current_queue, false);
            
            $file_id = $item['file_id'] ?? '';
            $file_path = $item['file_path'] ?? '';
            $file_name = $item['file_name'] ?? '';
            $sync_target = $item['sync_target'] ?? 'wp';
            
            \LightSyncPro\Util\Logger::debug('[LSP Dropbox Manual Process] Processing: ' . $file_name);
            
            $result = $this->sync_dropbox_file($file_id, $file_path, $file_name, $sync_target);
            
            if (!is_wp_error($result)) {
                $processed++;
                
                // Update progress
                $progress = get_option('lightsync_dropbox_sync_progress', []);
                $progress['synced'] = ($progress['synced'] ?? 0) + 1;
                if (!isset($progress['synced_ids'])) {
                    $progress['synced_ids'] = [];
                }
                $progress['synced_ids'][] = $file_id;
                update_option('lightsync_dropbox_sync_progress', $progress, false);
                
                self::usage_consume(1);
            } else {
                $errors[] = [
                    'file' => $file_name,
                    'error' => $result->get_error_message(),
                ];
            }
        }
        
        $remaining = get_option('lightsync_dropbox_sync_queue', []);
        
        wp_send_json_success([
            'processed' => $processed,
            'errors' => $errors,
            'remaining' => count($remaining),
            'message' => "Processed {$processed} file(s), " . count($remaining) . " remaining",
        ]);
    }

    /**
     * Save Dropbox watched folders
     */
    public function ajax_save_watched_folders() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        if (!self::has_cap('autosync')) {
            wp_send_json_error(['error' => 'Feature not available']);
        }

        $folders = isset($_POST['folders']) ? json_decode(stripslashes($_POST['folders']), true) : [];
        
        if (!is_array($folders)) {
            $folders = [];
        }

        // Handle both old format (array of paths) and new format (array of {path, destination})
        $normalized_folders = [];
        foreach ($folders as $folder) {
            if (is_string($folder)) {
                // Old format: just a path string - convert to new format
                $normalized_folders[] = [
                    'path' => sanitize_text_field($folder),
                    'destination' => 'wp', // Default to WordPress
                ];
            } elseif (is_array($folder) && isset($folder['path'])) {
                // New format: {path, destination}
                $normalized_folders[] = [
                    'path' => sanitize_text_field($folder['path']),
                    'destination' => in_array($folder['destination'] ?? 'wp', ['wp', 'hub'], true) 
                        ? $folder['destination'] 
                        : 'wp',
                ];
            }
        }

        // Remove duplicates by path
        $seen_paths = [];
        $unique_folders = [];
        foreach ($normalized_folders as $folder) {
            if (!in_array($folder['path'], $seen_paths, true)) {
                $seen_paths[] = $folder['path'];
                $unique_folders[] = $folder;
            }
        }

        self::set_opt(['dropbox_synced_folders' => $unique_folders]);

        wp_send_json_success([
            'message' => 'Watched folders saved',
            'folders' => $unique_folders,
        ]);
    }

    /**
     * Get album schedules with album names
     */
    public function ajax_get_album_schedules() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $schedules = (array) self::get_opt('album_schedule', []);
        $album_ids = (array) self::get_opt('album_ids', []);

        wp_send_json_success([
            'schedules' => $schedules,
            'album_ids' => $album_ids,
        ]);
    }

    /**
     * Save individual album schedule
     */
    public function ajax_save_album_schedule() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $album_id = isset($_POST['album_id']) ? sanitize_text_field($_POST['album_id']) : '';
        $schedule = isset($_POST['schedule']) ? sanitize_text_field($_POST['schedule']) : 'off';

        if (empty($album_id)) {
            wp_send_json_error(['error' => 'No album ID provided']);
        }

        $valid_schedules = ['off', '15m', 'hourly', 'twicedaily', 'daily'];
        if (!in_array($schedule, $valid_schedules, true)) {
            $schedule = 'off';
        }

        $schedules = (array) self::get_opt('album_schedule', []);
        
        if ($schedule === 'off') {
            unset($schedules[$album_id]);
        } else {
            $schedules[$album_id] = $schedule;
        }

        self::set_opt(['album_schedule' => $schedules]);

        // Reschedule cron if needed
        $this->reschedule_album_cron();

        wp_send_json_success([
            'saved' => true,
            'album_id' => $album_id,
            'schedule' => $schedule,
        ]);
    }

    /**
     * Bulk save album schedules
     */
    public function ajax_save_album_schedules_bulk() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $album_ids = isset($_POST['album_ids']) ? json_decode(stripslashes($_POST['album_ids']), true) : [];
        $schedule = isset($_POST['schedule']) ? sanitize_text_field($_POST['schedule']) : 'off';

        if (!is_array($album_ids) || empty($album_ids)) {
            wp_send_json_error(['error' => 'No album IDs provided']);
        }

        $valid_schedules = ['off', '15m', 'hourly', 'twicedaily', 'daily'];
        if (!in_array($schedule, $valid_schedules, true)) {
            $schedule = 'off';
        }

        $schedules = (array) self::get_opt('album_schedule', []);

        foreach ($album_ids as $album_id) {
            $album_id = sanitize_text_field($album_id);
            if ($schedule === 'off') {
                unset($schedules[$album_id]);
            } else {
                $schedules[$album_id] = $schedule;
            }
        }

        self::set_opt(['album_schedule' => $schedules]);

        // Reschedule cron if needed
        $this->reschedule_album_cron();

        wp_send_json_success([
            'saved' => true,
            'count' => count($album_ids),
            'schedule' => $schedule,
        ]);
    }

    /**
     * Save album destinations
     */
    public function ajax_save_album_destinations() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $album_id = isset($_POST['album_id']) ? sanitize_text_field($_POST['album_id']) : '';
        // Don't use sanitize_text_field on JSON - it corrupts it
        $destinations_json = isset($_POST['destinations']) ? wp_unslash($_POST['destinations']) : '[]';
        
        Logger::debug("[LSP Dest] Raw JSON received: {$destinations_json}");
        
        if (empty($album_id)) {
            wp_send_json_error(['error' => 'Missing album ID']);
        }

        $destinations = json_decode($destinations_json, true);
        Logger::debug("[LSP Dest] Decoded destinations: " . print_r($destinations, true));
        
        if (!is_array($destinations)) {
            $destinations = ['wordpress'];
        }

        // Validate destinations
        $valid_dests = ['wordpress'];
        $destinations = array_filter($destinations, function($d) use ($valid_dests) {
            return in_array($d, $valid_dests);
        });

        // Ensure at least one destination
        if (empty($destinations)) {
            $destinations = ['wordpress'];
        }
        
        $destinations = array_values($destinations);
        Logger::debug("[LSP Dest] Final destinations to save: " . json_encode($destinations));

        // Clear any object cache first
        wp_cache_delete(self::OPT, 'options');
        
        // Get ALL current options directly from database
        global $wpdb;
        $raw = $wpdb->get_var($wpdb->prepare(
            "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s LIMIT 1",
            self::OPT
        ));
        
        $all_opts = maybe_unserialize($raw);
        if (!is_array($all_opts)) {
            $all_opts = json_decode($raw, true);
        }
        if (!is_array($all_opts)) {
            $all_opts = [];
        }
        
        // Initialize album_destinations if not exists
        if (!isset($all_opts['album_destinations']) || !is_array($all_opts['album_destinations'])) {
            $all_opts['album_destinations'] = [];
        }
        
        Logger::debug("[LSP Dest] Current destinations before save: " . json_encode($all_opts['album_destinations']));
        
        // Update this album's destinations
        $all_opts['album_destinations'][$album_id] = $destinations;
        
        // Save directly - bypass update_option to avoid any filters
        $serialized = maybe_serialize($all_opts);
        $result = $wpdb->update(
            $wpdb->options,
            ['option_value' => $serialized],
            ['option_name' => self::OPT],
            ['%s'],
            ['%s']
        );
        
        // Clear cache after save
        wp_cache_delete(self::OPT, 'options');
        
        Logger::debug("[LSP Dest] DB update result: " . var_export($result, true));
        
        // Verify it saved by reading directly from DB again
        $verify_raw = $wpdb->get_var($wpdb->prepare(
            "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s LIMIT 1",
            self::OPT
        ));
        $verify = maybe_unserialize($verify_raw);
        Logger::debug("[LSP Dest] Verified after save: " . json_encode($verify['album_destinations'] ?? []));

        wp_send_json_success([
            'saved' => true,
            'album_id' => $album_id,
            'destinations' => $destinations,
            'all_destinations' => $all_opts['album_destinations'],
            'db_result' => $result,
            'verified' => $verify['album_destinations'] ?? [],
        ]);
    }

    /**
     * Reschedule album cron based on current schedules
     */
    private function reschedule_album_cron(): void {
        $schedules = (array) self::get_opt('album_schedule', []);
        
        // Clear existing
        wp_clear_scheduled_hook(self::CRON);
        
        // Find fastest schedule
        $priority = ['hourly' => 1, 'twicedaily' => 2, 'daily' => 3];
        $fastest = null;
        
        foreach ($schedules as $album_id => $sched) {
            if ($sched !== 'off' && isset($priority[$sched])) {
                if ($fastest === null || $priority[$sched] < $priority[$fastest]) {
                    $fastest = $sched;
                }
            }
        }
        
        // Schedule if any active
        if ($fastest) {
            wp_schedule_event(time() + 60, $fastest, self::CRON);
        }
    }

    /**
     * Get album cover thumbnail URL (with caching)
     * Falls back to first asset in album if no cover is set
     */
    public function ajax_get_album_cover() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $catalog_id = isset($_POST['catalog_id']) ? sanitize_text_field($_POST['catalog_id']) : '';
        $album_id = isset($_POST['album_id']) ? sanitize_text_field($_POST['album_id']) : '';
        $cover_id = isset($_POST['cover_id']) ? sanitize_text_field($_POST['cover_id']) : '';

        if (empty($catalog_id) || empty($album_id)) {
            wp_send_json_error(['error' => 'Missing parameters']);
        }

        // Check cache first (keyed by album_id since cover might change)
        $cache_key = 'lsp_cover_' . md5($catalog_id . '_' . $album_id);
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            wp_send_json_success(['url' => $cached]);
        }

        // Ensure we have a valid token
        $t = \LightSyncPro\OAuth\OAuth::ensure_token();
        if (is_wp_error($t)) {
            wp_send_json_error(['error' => 'Auth failed']);
        }

        // If no cover_id provided, fetch first asset from album (lightweight call)
        if (empty($cover_id)) {
            $url = \LightSyncPro\Http\Endpoints::album_assets($catalog_id, $album_id, null, 1);
            $resp = \LightSyncPro\Http\Client::get($url, 10);
            
            if (!is_wp_error($resp)) {
                $body = wp_remote_retrieve_body($resp);
                $data = \LightSyncPro\Util\Adobe::decode($body);
                
                if (!empty($data['resources'][0]['asset']['id'])) {
                    $cover_id = $data['resources'][0]['asset']['id'];
                }
            }
        }

        if (empty($cover_id)) {
            wp_send_json_error(['error' => 'No assets in album']);
        }

        // Build rendition URL for thumbnail size
        $url = \LightSyncPro\Http\Endpoints::rendition($catalog_id, $cover_id, '640');

        $resp = \LightSyncPro\Http\Client::get($url, 15);
        
        if (is_wp_error($resp)) {
            wp_send_json_error(['error' => 'Failed to fetch cover']);
        }

        $body = wp_remote_retrieve_body($resp);
        $code = wp_remote_retrieve_response_code($resp);

        if ($code !== 200 || empty($body)) {
            wp_send_json_error(['error' => 'No image data']);
        }

        // Build data URL
        $mime = wp_remote_retrieve_header($resp, 'content-type') ?: 'image/jpeg';
        $data_url = 'data:' . $mime . ';base64,' . base64_encode($body);
        
        // Cache for 1 week
        set_transient($cache_key, $data_url, WEEK_IN_SECONDS);

        wp_send_json_success(['url' => $data_url]);
    }

    /* ==================== END AUTO-SYNC HANDLERS ==================== */

    /* ==================== AI INSIGHTS HANDLERS ==================== */

    /**
     * Analyze images with AI
     * Supports both Lightroom assets and Canva designs
     */
    public function ajax_ai_analyze() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $o = self::get_opt();
        $provider = $o['ai_provider'] ?? '';
        
        if (empty($provider)) {
            wp_send_json_error(['error' => 'No AI provider configured. Go to Settings → AI Insights.']);
        }

        $api_key = '';
        if ($provider === 'anthropic') {
            $api_key = $o['ai_anthropic_key'] ?? '';
        } elseif ($provider === 'openai') {
            $api_key = $o['ai_openai_key'] ?? '';
        }

        if (empty($api_key)) {
            wp_send_json_error(['error' => 'API key not configured for ' . ucfirst($provider)]);
        }

        // Get images to analyze (either asset_ids for Lightroom or design_ids for Canva)
        $source = isset($_POST['source']) ? sanitize_text_field($_POST['source']) : 'lightroom';
        $image_urls = isset($_POST['image_urls']) ? array_map('esc_url_raw', (array)$_POST['image_urls']) : [];
        $image_ids = isset($_POST['image_ids']) ? array_map('sanitize_text_field', (array)$_POST['image_ids']) : [];

        if (empty($image_urls)) {
            wp_send_json_error(['error' => 'No images provided for analysis']);
        }

        // Limit to 20 images per request
        $image_urls = array_slice($image_urls, 0, 20);
        $image_ids = array_slice($image_ids, 0, 20);

        @set_time_limit(120);

        // Build analysis prompt
        $prompt = $this->build_ai_analysis_prompt(count($image_urls));

        // Call AI API
        $result = $this->call_ai_vision_api($provider, $api_key, $image_urls, $prompt);

        if (is_wp_error($result)) {
            wp_send_json_error(['error' => $result->get_error_message()]);
        }

        // Parse and structure response
        $analysis = $this->parse_ai_analysis_response($result, $image_ids, $image_urls);

        wp_send_json_success([
            'analysis' => $analysis,
            'provider' => $provider,
            'count' => count($image_urls),
        ]);
    }

    /**
     * Build the AI analysis prompt
     */
    private function build_ai_analysis_prompt($image_count) {
        return "Analyze these {$image_count} images for a website. For each image, provide:

1. **quality_score** (1-10): Technical quality (sharpness, exposure, composition)
2. **hero_score** (1-10): Suitability as a hero/banner image (visual impact, composition)
3. **suggested_use**: One of: hero, gallery, product, background, thumbnail
4. **alt_text**: SEO-optimized alt text (max 125 chars, descriptive, no 'image of')
5. **tags**: 3-5 relevant tags for categorization
6. **duplicate_group**: If images are near-duplicates, assign same group number (null if unique)

Respond in JSON format:
{
  \"images\": [
    {
      \"index\": 0,
      \"quality_score\": 8,
      \"hero_score\": 9,
      \"suggested_use\": \"hero\",
      \"alt_text\": \"Golden sunset over mountain lake with reflection\",
      \"tags\": [\"landscape\", \"sunset\", \"mountains\", \"lake\"],
      \"duplicate_group\": null
    }
  ],
  \"hero_pick\": 0,
  \"summary\": \"Brief overall assessment\"
}";
    }

    /**
     * Call AI Vision API (Anthropic or OpenAI)
     */
    private function call_ai_vision_api($provider, $api_key, $image_urls, $prompt) {
        if ($provider === 'anthropic') {
            return $this->call_anthropic_vision($api_key, $image_urls, $prompt);
        } elseif ($provider === 'openai') {
            return $this->call_openai_vision($api_key, $image_urls, $prompt);
        }
        return new \WP_Error('invalid_provider', 'Unknown AI provider');
    }

    /**
     * Call Anthropic Claude Vision API
     */
    private function call_anthropic_vision($api_key, $image_urls, $prompt) {
        // Build content array with images
        $content = [];
        
        foreach ($image_urls as $url) {
            $content[] = [
                'type' => 'image',
                'source' => [
                    'type' => 'url',
                    'url' => $url,
                ],
            ];
        }
        
        $content[] = [
            'type' => 'text',
            'text' => $prompt,
        ];

        $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'timeout' => 120,
            'headers' => [
                'Content-Type' => 'application/json',
                'x-api-key' => $api_key,
                'anthropic-version' => '2024-10-22',
            ],
            'body' => wp_json_encode([
                'model' => 'claude-sonnet-4-20250514',
                'max_tokens' => 4096,
                'messages' => [
                    ['role' => 'user', 'content' => $content],
                ],
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            $error = json_decode($body, true);
            $msg = $error['error']['message'] ?? "API error: {$code}";
            return new \WP_Error('anthropic_error', $msg);
        }

        $data = json_decode($body, true);
        return $data['content'][0]['text'] ?? '';
    }

    /**
     * Call OpenAI GPT-4 Vision API
     */
    private function call_openai_vision($api_key, $image_urls, $prompt) {
        // Build content array with images
        $content = [];
        
        foreach ($image_urls as $url) {
            $content[] = [
                'type' => 'image_url',
                'image_url' => ['url' => $url],
            ];
        }
        
        $content[] = [
            'type' => 'text',
            'text' => $prompt,
        ];

        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', [
            'timeout' => 120,
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key,
            ],
            'body' => wp_json_encode([
                'model' => 'gpt-4o',
                'max_tokens' => 4096,
                'messages' => [
                    ['role' => 'user', 'content' => $content],
                ],
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            $error = json_decode($body, true);
            $msg = $error['error']['message'] ?? "API error: {$code}";
            return new \WP_Error('openai_error', $msg);
        }

        $data = json_decode($body, true);
        return $data['choices'][0]['message']['content'] ?? '';
    }

    /**
     * Parse AI response into structured data
     */
    private function parse_ai_analysis_response($response, $image_ids, $image_urls) {
        // Extract JSON from response (may have markdown code blocks)
        $json_str = $response;
        if (preg_match('/```json?\s*(.*?)\s*```/s', $response, $matches)) {
            $json_str = $matches[1];
        }

        $data = json_decode($json_str, true);
        
        if (!$data || !isset($data['images'])) {
            // Fallback: return raw response
            return [
                'raw' => $response,
                'images' => [],
                'hero_pick' => null,
                'parse_error' => true,
            ];
        }

        // Merge image IDs and URLs into response
        foreach ($data['images'] as $i => &$img) {
            $img['id'] = $image_ids[$i] ?? null;
            $img['url'] = $image_urls[$i] ?? null;
        }

        return $data;
    }

    /**
     * Apply AI recommendations (save alt text, reorder, etc.)
     */
    public function ajax_ai_apply() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $action_type = isset($_POST['action_type']) ? sanitize_text_field($_POST['action_type']) : '';
        $images = isset($_POST['images']) ? $_POST['images'] : [];

        if (empty($images)) {
            wp_send_json_error(['error' => 'No images provided']);
        }

        $updated = 0;

        foreach ($images as $img) {
            $attachment_id = isset($img['attachment_id']) ? intval($img['attachment_id']) : 0;
            
            if (!$attachment_id) continue;

            // Apply alt text
            if ($action_type === 'alt_text' || $action_type === 'all') {
                if (!empty($img['alt_text'])) {
                    update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($img['alt_text']));
                    $updated++;
                }
            }

            // Apply tags
            if ($action_type === 'tags' || $action_type === 'all') {
                if (!empty($img['tags']) && is_array($img['tags'])) {
                    wp_set_object_terms($attachment_id, array_map('sanitize_text_field', $img['tags']), 'post_tag', true);
                }
            }

            // Store AI scores for performance tracking
            if (!empty($img['quality_score'])) {
                update_post_meta($attachment_id, '_lightsync_ai_quality', intval($img['quality_score']));
            }
            if (!empty($img['hero_score'])) {
                update_post_meta($attachment_id, '_lightsync_ai_hero', intval($img['hero_score']));
            }
            if (!empty($img['suggested_use'])) {
                update_post_meta($attachment_id, '_lightsync_ai_use', sanitize_text_field($img['suggested_use']));
            }
        }

        self::add_activity("AI recommendations applied to {$updated} image(s)", 'success', 'ai');

        wp_send_json_success(['updated' => $updated]);
    }

    /**
     * Track image impression (view)
     */
    public function ajax_track_impression() {
        $attachment_id = isset($_POST['attachment_id']) ? intval($_POST['attachment_id']) : 0;
        
        if (!$attachment_id) {
            wp_send_json_error(['error' => 'Invalid attachment']);
        }

        // Rate limiting: max 100 impressions per IP per minute
        $ip_hash = md5($_SERVER['REMOTE_ADDR'] ?? 'unknown');
        $rate_key = 'lsp_rate_imp_' . $ip_hash;
        $rate_count = (int) get_transient($rate_key);
        
        if ($rate_count >= 100) {
            wp_send_json_error(['error' => 'Rate limit exceeded'], 429);
        }
        
        set_transient($rate_key, $rate_count + 1, 60); // 1 minute window

        // Get context data
        $context = $this->get_tracking_context($_POST);

        // Store contextual performance data
        $this->record_performance_event($attachment_id, 'impression', $context);

        // Also increment legacy counter for backward compatibility
        $impressions = (int) get_post_meta($attachment_id, '_lightsync_impressions', true);
        update_post_meta($attachment_id, '_lightsync_impressions', $impressions + 1);

        wp_send_json_success(['impressions' => $impressions + 1]);
    }

    /**
     * Track image click with context
     */
    public function ajax_track_click() {
        $attachment_id = isset($_POST['attachment_id']) ? intval($_POST['attachment_id']) : 0;
        
        if (!$attachment_id) {
            wp_send_json_error(['error' => 'Invalid attachment']);
        }

        // Rate limiting: max 50 clicks per IP per minute
        $ip_hash = md5($_SERVER['REMOTE_ADDR'] ?? 'unknown');
        $rate_key = 'lsp_rate_click_' . $ip_hash;
        $rate_count = (int) get_transient($rate_key);
        
        if ($rate_count >= 50) {
            wp_send_json_error(['error' => 'Rate limit exceeded'], 429);
        }
        
        set_transient($rate_key, $rate_count + 1, 60); // 1 minute window

        // Get context data
        $context = $this->get_tracking_context($_POST);

        // Store contextual performance data
        $this->record_performance_event($attachment_id, 'click', $context);

        // Also increment legacy counter for backward compatibility
        $clicks = (int) get_post_meta($attachment_id, '_lightsync_clicks', true);
        update_post_meta($attachment_id, '_lightsync_clicks', $clicks + 1);

        // Update last click timestamp
        update_post_meta($attachment_id, '_lightsync_last_click', time());

        wp_send_json_success(['clicks' => $clicks + 1]);
    }

    /**
     * Extract tracking context from request
     */
    private function get_tracking_context($post_data) {
        return [
            'page_id'      => isset($post_data['page_id']) ? intval($post_data['page_id']) : 0,
            'page_type'    => isset($post_data['page_type']) ? sanitize_key($post_data['page_type']) : 'unknown',
            'page_title'   => isset($post_data['page_title']) ? sanitize_text_field($post_data['page_title']) : '',
            'position'     => isset($post_data['position']) ? sanitize_key($post_data['position']) : 'unknown',
            'position_idx' => isset($post_data['position_idx']) ? intval($post_data['position_idx']) : 0,
            'device'       => isset($post_data['device']) ? sanitize_key($post_data['device']) : 'unknown',
            'viewport_w'   => isset($post_data['viewport_w']) ? intval($post_data['viewport_w']) : 0,
            'is_linked'    => isset($post_data['is_linked']) ? (bool) $post_data['is_linked'] : false,
            'link_target'  => isset($post_data['link_target']) ? esc_url_raw($post_data['link_target']) : '',
        ];
    }

    /**
     * Record a performance event with context
     */
    private function record_performance_event($attachment_id, $event_type, $context) {
        // Get existing performance data
        $performance = get_post_meta($attachment_id, '_lightsync_performance', true);
        if (!is_array($performance)) {
            $performance = [];
        }

        // Create context key (page + position)
        $context_key = $context['page_id'] . '_' . $context['position'];
        
        // Initialize context if not exists
        if (!isset($performance[$context_key])) {
            $performance[$context_key] = [
                'page_id'      => $context['page_id'],
                'page_type'    => $context['page_type'],
                'page_title'   => $context['page_title'],
                'position'     => $context['position'],
                'impressions'  => 0,
                'clicks'       => 0,
                'first_seen'   => time(),
                'last_seen'    => time(),
                'devices'      => ['desktop' => 0, 'mobile' => 0, 'tablet' => 0],
            ];
        }

        // Update counts
        if ($event_type === 'impression') {
            $performance[$context_key]['impressions']++;
        } elseif ($event_type === 'click') {
            $performance[$context_key]['clicks']++;
        }

        // Update device breakdown
        $device = $context['device'] ?: 'desktop';
        if (isset($performance[$context_key]['devices'][$device])) {
            $performance[$context_key]['devices'][$device]++;
        }

        // Update timestamps
        $performance[$context_key]['last_seen'] = time();

        // Store is_linked if this is a click
        if ($event_type === 'click' && $context['is_linked']) {
            $performance[$context_key]['is_linked'] = true;
        }

        // Save updated performance data
        update_post_meta($attachment_id, '_lightsync_performance', $performance);

        // Also update global page performance option for quick lookups
        $this->update_page_performance_cache($context['page_id'], $attachment_id, $event_type);
    }

    /**
     * Update page-level performance cache for quick lookups
     */
    private function update_page_performance_cache($page_id, $attachment_id, $event_type) {
        if (!$page_id) return;

        $cache_key = 'lsp_page_performance_' . $page_id;
        $page_perf = get_transient($cache_key);
        
        if (!is_array($page_perf)) {
            $page_perf = ['images' => [], 'updated' => time()];
        }

        if (!isset($page_perf['images'][$attachment_id])) {
            $page_perf['images'][$attachment_id] = ['impressions' => 0, 'clicks' => 0];
        }

        if ($event_type === 'impression') {
            $page_perf['images'][$attachment_id]['impressions']++;
        } else {
            $page_perf['images'][$attachment_id]['clicks']++;
        }

        $page_perf['updated'] = time();

        // Cache for 30 days
        set_transient($cache_key, $page_perf, 30 * DAY_IN_SECONDS);
    }

    /**
     * Performance-based gallery optimization (cron job)
     * Analyzes click performance and reorders galleries - NO AI API REQUIRED
     */
    public function ai_optimize_galleries() {
        $o = self::get_opt();
        
        // Check if performance optimization is enabled
        if (empty($o['ai_performance_optimize'])) {
            return;
        }

        // No API key needed - this is pure math based on CTR

        global $wpdb;

        // Get images with performance data
        $results = $wpdb->get_results(
            "SELECT p.ID, p.post_title,
                    pm_imp.meta_value as impressions,
                    pm_click.meta_value as clicks,
                    pm_album.meta_value as album_id
             FROM {$wpdb->posts} p
             LEFT JOIN {$wpdb->postmeta} pm_imp ON p.ID = pm_imp.post_id AND pm_imp.meta_key = '_lightsync_impressions'
             LEFT JOIN {$wpdb->postmeta} pm_click ON p.ID = pm_click.post_id AND pm_click.meta_key = '_lightsync_clicks'
             LEFT JOIN {$wpdb->postmeta} pm_album ON p.ID = pm_album.post_id AND pm_album.meta_key = '_lightsync_album_id'
             WHERE p.post_type = 'attachment'
             AND pm_imp.meta_value > 0
             ORDER BY pm_album.meta_value, (CAST(pm_click.meta_value AS UNSIGNED) / CAST(pm_imp.meta_value AS UNSIGNED)) DESC"
        );

        if (empty($results)) {
            return;
        }

        // Group by album
        $albums = [];
        foreach ($results as $row) {
            $album_id = $row->album_id ?: 'ungrouped';
            if (!isset($albums[$album_id])) {
                $albums[$album_id] = [];
            }
            
            $impressions = (int) $row->impressions;
            $clicks = (int) $row->clicks;
            $ctr = $impressions > 0 ? ($clicks / $impressions) * 100 : 0;
            
            $albums[$album_id][] = [
                'id' => $row->ID,
                'title' => $row->post_title,
                'impressions' => $impressions,
                'clicks' => $clicks,
                'ctr' => round($ctr, 2),
            ];
        }

        // For each album, identify optimization opportunities
        foreach ($albums as $album_id => $images) {
            if (count($images) < 2) continue;

            // Find best and worst performers
            usort($images, function($a, $b) {
                return $b['ctr'] <=> $a['ctr'];
            });

            $best = $images[0];
            $worst = end($images);

            // If significant CTR difference, log recommendation
            if ($best['ctr'] > 0 && $worst['ctr'] < $best['ctr'] * 0.3) {
                // Store optimization recommendation
                $recommendation = sprintf(
                    'Album %s: Consider promoting "%s" (%.1f%% CTR) over "%s" (%.1f%% CTR)',
                    $album_id,
                    $best['title'],
                    $best['ctr'],
                    $worst['title'],
                    $worst['ctr']
                );

                // Store recommendations for admin review
                $recommendations = get_option('lsp_performance_recommendations', []);
                $recommendations[] = [
                    'ts' => time(),
                    'album_id' => $album_id,
                    'message' => $recommendation,
                    'best_id' => $best['id'],
                    'worst_id' => $worst['id'],
                ];
                $recommendations = array_slice($recommendations, -20); // Keep last 20
                update_option('lsp_performance_recommendations', $recommendations, false);

                // Update menu order to promote best performer
                wp_update_post([
                    'ID' => $best['id'],
                    'menu_order' => 0,
                ]);

                self::add_activity($recommendation, 'info', 'performance');
            }
        }
    }

    /**
     * Get performance insights for display
     * Returns winners, swap candidates, and actionable suggestions with contextual analysis
     */
    public static function get_performance_insights() {
        global $wpdb;

        // Minimum impressions to be statistically meaningful
        $min_impressions = 30;

        // Get ALL images with performance data (both legacy and contextual)
        // Include images with _lightsync_source OR _lightsync_catalog_id (for older Lightroom imports)
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT p.ID, p.post_title,
                    COALESCE(pm_imp.meta_value, 0) as impressions,
                    COALESCE(pm_click.meta_value, 0) as clicks,
                    pm_album.meta_value as album_id,
                    pm_album_name.meta_value as album_name,
                    pm_perf.meta_value as performance_data
             FROM {$wpdb->posts} p
             LEFT JOIN {$wpdb->postmeta} pm_src ON p.ID = pm_src.post_id AND pm_src.meta_key = '_lightsync_source'
             LEFT JOIN {$wpdb->postmeta} pm_cat ON p.ID = pm_cat.post_id AND pm_cat.meta_key = '_lightsync_catalog_id'
             LEFT JOIN {$wpdb->postmeta} pm_imp ON p.ID = pm_imp.post_id AND pm_imp.meta_key = '_lightsync_impressions'
             LEFT JOIN {$wpdb->postmeta} pm_click ON p.ID = pm_click.post_id AND pm_click.meta_key = '_lightsync_clicks'
             LEFT JOIN {$wpdb->postmeta} pm_album ON p.ID = pm_album.post_id AND pm_album.meta_key = '_lightsync_album_id'
             LEFT JOIN {$wpdb->postmeta} pm_album_name ON p.ID = pm_album_name.post_id AND pm_album_name.meta_key = '_lightsync_album_name'
             LEFT JOIN {$wpdb->postmeta} pm_perf ON p.ID = pm_perf.post_id AND pm_perf.meta_key = '_lightsync_performance'
             WHERE p.post_type = 'attachment'
             AND (pm_src.meta_value IS NOT NULL OR pm_cat.meta_value IS NOT NULL)
             AND CAST(COALESCE(pm_imp.meta_value, 0) AS UNSIGNED) >= %d
             ORDER BY CAST(COALESCE(pm_imp.meta_value, 0) AS UNSIGNED) DESC
             LIMIT 100",
            $min_impressions
        ));

        $insights = [
            'has_data' => false,
            'total_impressions' => 0,
            'total_clicks' => 0,
            'total_images' => 0,
            'winners' => [],
            'swap_candidates' => [],
            'suggestions' => [],
            'page_insights' => [],
            'position_insights' => [],
            'device_insights' => [],
        ];

        if (empty($results)) {
            // Check if we have any data at all (below threshold)
            $any_data = $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = '_lightsync_impressions' AND CAST(meta_value AS UNSIGNED) > 0"
            );
            if ($any_data > 0) {
                $insights['suggestions'][] = [
                    'type' => 'info',
                    'icon' => '📊',
                    'text' => "Collecting data... Need at least {$min_impressions} views per image for meaningful insights.",
                ];
            }
            return $insights;
        }

        $insights['has_data'] = true;
        $insights['total_images'] = count($results);

        // Aggregation containers
        $all_images = [];
        $total_impressions = 0;
        $total_clicks = 0;
        $albums = [];
        $pages = [];           // page_id => [images => [...], title, type]
        $positions = [];       // position => [impressions, clicks]
        $devices = ['desktop' => ['imp' => 0, 'click' => 0], 'mobile' => ['imp' => 0, 'click' => 0], 'tablet' => ['imp' => 0, 'click' => 0]];
        $image_contexts = [];  // image_id => [contexts...]

        foreach ($results as $row) {
            $impressions = (int) $row->impressions;
            $clicks = (int) $row->clicks;
            
            $total_impressions += $impressions;
            $total_clicks += $clicks;

            // Parse contextual performance data
            $perf_data = maybe_unserialize($row->performance_data);
            $contexts = [];
            
            if (is_array($perf_data) && !empty($perf_data)) {
                foreach ($perf_data as $ctx_key => $ctx) {
                    $contexts[] = $ctx;
                    
                    // Aggregate by page
                    $page_id = $ctx['page_id'] ?? 0;
                    if ($page_id) {
                        if (!isset($pages[$page_id])) {
                            $pages[$page_id] = [
                                'title' => $ctx['page_title'] ?? 'Unknown',
                                'type' => $ctx['page_type'] ?? 'page',
                                'images' => [],
                                'total_imp' => 0,
                                'total_click' => 0,
                            ];
                        }
                        $pages[$page_id]['images'][$row->ID] = [
                            'title' => $row->post_title,
                            'impressions' => $ctx['impressions'] ?? 0,
                            'clicks' => $ctx['clicks'] ?? 0,
                            'position' => $ctx['position'] ?? 'unknown',
                        ];
                        $pages[$page_id]['total_imp'] += ($ctx['impressions'] ?? 0);
                        $pages[$page_id]['total_click'] += ($ctx['clicks'] ?? 0);
                    }
                    
                    // Aggregate by position
                    $pos = $ctx['position'] ?? 'unknown';
                    if (!isset($positions[$pos])) {
                        $positions[$pos] = ['impressions' => 0, 'clicks' => 0];
                    }
                    $positions[$pos]['impressions'] += ($ctx['impressions'] ?? 0);
                    $positions[$pos]['clicks'] += ($ctx['clicks'] ?? 0);
                    
                    // Aggregate by device
                    if (isset($ctx['devices']) && is_array($ctx['devices'])) {
                        foreach ($ctx['devices'] as $device => $count) {
                            if (isset($devices[$device])) {
                                $devices[$device]['imp'] += $count;
                            }
                        }
                    }
                }
            }

            $image_contexts[$row->ID] = $contexts;

            // Engagement score
            $engagement_score = ($clicks * 10) + ($impressions * 0.1);
            $ctr = $impressions > 0 ? ($clicks / $impressions) : 0;

            $thumb = wp_get_attachment_image_src($row->ID, 'thumbnail');
            
            $image_data = [
                'id' => $row->ID,
                'title' => $row->post_title,
                'thumb' => $thumb ? $thumb[0] : '',
                'impressions' => $impressions,
                'clicks' => $clicks,
                'ctr' => $ctr,
                'engagement_score' => $engagement_score,
                'album_id' => $row->album_id,
                'album_name' => $row->album_name ?: 'Ungrouped',
                'contexts' => $contexts,
            ];

            $all_images[] = $image_data;

            // Group by album
            $album_key = $row->album_id ?: 'ungrouped';
            if (!isset($albums[$album_key])) {
                $albums[$album_key] = ['name' => $row->album_name ?: 'Ungrouped', 'images' => []];
            }
            $albums[$album_key]['images'][] = $image_data;
        }

        $insights['total_impressions'] = $total_impressions;
        $insights['total_clicks'] = $total_clicks;

        $avg_ctr = $total_impressions > 0 ? ($total_clicks / $total_impressions) : 0;

        // Sort by engagement score
        usort($all_images, function($a, $b) {
            return $b['engagement_score'] <=> $a['engagement_score'];
        });

        // Top 5 winners
        $insights['winners'] = array_slice($all_images, 0, 5);

        // Swap candidates
        $swap_candidates = [];
        foreach ($all_images as $img) {
            if ($img['impressions'] >= $min_impressions && $img['ctr'] < ($avg_ctr * 0.5)) {
                $swap_candidates[] = $img;
            }
        }
        usort($swap_candidates, function($a, $b) { return $b['impressions'] <=> $a['impressions']; });
        $insights['swap_candidates'] = array_slice($swap_candidates, 0, 3);

        // ========== GENERATE CONTEXTUAL SUGGESTIONS ==========
        $suggestions = [];

        // 1. Best overall performer
        if (count($all_images) >= 2) {
            $top = $all_images[0];
            $top_ctr_pct = round($top['ctr'] * 100, 1);
            $suggestions[] = [
                'type' => 'success',
                'icon' => '🏆',
                'text' => "\"{$top['title']}\" is your top performer ({$top_ctr_pct}% engagement). Feature it more prominently.",
            ];
        }

        // 2. Page-specific insights: Find same image performing differently on different pages
        foreach ($image_contexts as $img_id => $contexts) {
            if (count($contexts) >= 2) {
                $best_ctx = null;
                $worst_ctx = null;
                
                foreach ($contexts as $ctx) {
                    $ctx_imp = $ctx['impressions'] ?? 0;
                    $ctx_click = $ctx['clicks'] ?? 0;
                    if ($ctx_imp < 20) continue; // Need enough data
                    
                    $ctx_ctr = $ctx_imp > 0 ? ($ctx_click / $ctx_imp) : 0;
                    $ctx['_ctr'] = $ctx_ctr;
                    
                    if (!$best_ctx || $ctx_ctr > $best_ctx['_ctr']) {
                        $best_ctx = $ctx;
                    }
                    if (!$worst_ctx || $ctx_ctr < $worst_ctx['_ctr']) {
                        $worst_ctx = $ctx;
                    }
                }
                
                // If significant difference between best and worst context
                if ($best_ctx && $worst_ctx && $best_ctx['_ctr'] > $worst_ctx['_ctr'] * 2 && $best_ctx['_ctr'] > 0.01) {
                    $img_title = '';
                    foreach ($all_images as $img) {
                        if ($img['id'] == $img_id) { $img_title = $img['title']; break; }
                    }
                    
                    $best_pct = round($best_ctx['_ctr'] * 100, 1);
                    $worst_pct = round($worst_ctx['_ctr'] * 100, 1);
                    $best_page = $best_ctx['page_title'] ?: 'a page';
                    $worst_page = $worst_ctx['page_title'] ?: 'another page';
                    
                    $suggestions[] = [
                        'type' => 'info',
                        'icon' => '🎯',
                        'text' => "\"{$img_title}\" gets {$best_pct}% CTR on {$best_page} but only {$worst_pct}% on {$worst_page}. Use it where it performs best.",
                    ];
                    break; // Only one of these suggestions
                }
            }
        }

        // 3. Position insights: Which positions convert best
        if (!empty($positions)) {
            $position_ctrs = [];
            foreach ($positions as $pos => $data) {
                if ($data['impressions'] >= 50) {
                    $position_ctrs[$pos] = [
                        'ctr' => $data['impressions'] > 0 ? ($data['clicks'] / $data['impressions']) : 0,
                        'impressions' => $data['impressions'],
                        'clicks' => $data['clicks'],
                    ];
                }
            }
            
            if (count($position_ctrs) >= 2) {
                uasort($position_ctrs, function($a, $b) { return $b['ctr'] <=> $a['ctr']; });
                $best_pos = key($position_ctrs);
                $best_pos_data = reset($position_ctrs);
                $worst_pos = array_key_last($position_ctrs);
                $worst_pos_data = end($position_ctrs);
                
                if ($best_pos_data['ctr'] > $worst_pos_data['ctr'] * 1.5) {
                    $best_pct = round($best_pos_data['ctr'] * 100, 1);
                    $worst_pct = round($worst_pos_data['ctr'] * 100, 1);
                    
                    $pos_labels = [
                        'hero' => 'Hero/Banner',
                        'above_fold' => 'Above the fold',
                        'gallery' => 'Gallery',
                        'product' => 'Product',
                        'sidebar' => 'Sidebar',
                        'content' => 'Content area',
                        'header' => 'Header',
                        'footer' => 'Footer',
                    ];
                    
                    $best_label = $pos_labels[$best_pos] ?? ucfirst($best_pos);
                    $worst_label = $pos_labels[$worst_pos] ?? ucfirst($worst_pos);
                    
                    $suggestions[] = [
                        'type' => 'info',
                        'icon' => '📍',
                        'text' => "{$best_label} images convert at {$best_pct}% vs {$worst_pct}% in {$worst_label}. Prioritize {$best_label} placement.",
                    ];
                }
            }
            
            $insights['position_insights'] = $position_ctrs;
        }

        // 4. Device insights
        $mobile_imp = $devices['mobile']['imp'];
        $desktop_imp = $devices['desktop']['imp'];
        if ($mobile_imp >= 50 && $desktop_imp >= 50) {
            // We tracked device on impressions, estimate clicks proportionally
            $mobile_ratio = $mobile_imp / ($mobile_imp + $desktop_imp);
            
            if ($mobile_ratio > 0.6) {
                $suggestions[] = [
                    'type' => 'info',
                    'icon' => '📱',
                    'text' => round($mobile_ratio * 100) . "% of views come from mobile. Ensure images look great on small screens.",
                ];
            } elseif ($mobile_ratio < 0.3) {
                $suggestions[] = [
                    'type' => 'info',
                    'icon' => '🖥️',
                    'text' => round((1 - $mobile_ratio) * 100) . "% of views come from desktop. Optimize for larger displays.",
                ];
            }
            
            $insights['device_insights'] = $devices;
        }

        // 5. Page performance comparison
        if (count($pages) >= 2) {
            $page_ctrs = [];
            foreach ($pages as $page_id => $pdata) {
                if ($pdata['total_imp'] >= 30) {
                    $page_ctrs[$page_id] = [
                        'title' => $pdata['title'],
                        'type' => $pdata['type'],
                        'ctr' => $pdata['total_imp'] > 0 ? ($pdata['total_click'] / $pdata['total_imp']) : 0,
                        'impressions' => $pdata['total_imp'],
                    ];
                }
            }
            
            if (count($page_ctrs) >= 2) {
                uasort($page_ctrs, function($a, $b) { return $b['ctr'] <=> $a['ctr']; });
                $best_page = reset($page_ctrs);
                $worst_page = end($page_ctrs);
                
                if ($best_page['ctr'] > $worst_page['ctr'] * 2 && $best_page['ctr'] > 0.01) {
                    $best_pct = round($best_page['ctr'] * 100, 1);
                    $worst_pct = round($worst_page['ctr'] * 100, 1);
                    
                    $suggestions[] = [
                        'type' => 'warning',
                        'icon' => '📄',
                        'text' => "Images on \"{$best_page['title']}\" ({$best_pct}% CTR) outperform \"{$worst_page['title']}\" ({$worst_pct}%). Review underperforming page.",
                    ];
                }
            }
            
            $insights['page_insights'] = array_slice($page_ctrs, 0, 5, true);
        }

        // 6. Swap recommendation
        if (!empty($swap_candidates)) {
            $worst = $swap_candidates[0];
            $worst_ctr_pct = round($worst['ctr'] * 100, 1);
            $avg_ctr_pct = round($avg_ctr * 100, 1);
            
            $better_alt = null;
            if ($worst['album_id'] && isset($albums[$worst['album_id']])) {
                foreach ($albums[$worst['album_id']]['images'] as $alt) {
                    if ($alt['id'] !== $worst['id'] && $alt['ctr'] > $worst['ctr'] * 1.5) {
                        $better_alt = $alt;
                        break;
                    }
                }
            }

            if ($better_alt) {
                $better_ctr_pct = round($better_alt['ctr'] * 100, 1);
                $suggestions[] = [
                    'type' => 'warning',
                    'icon' => '🔄',
                    'text' => "Swap \"{$worst['title']}\" ({$worst_ctr_pct}%) with \"{$better_alt['title']}\" ({$better_ctr_pct}%) — same album, better results.",
                ];
            } else {
                $suggestions[] = [
                    'type' => 'warning',
                    'icon' => '📉',
                    'text' => "\"{$worst['title']}\" gets {$worst['impressions']} views but only {$worst_ctr_pct}% engagement. Consider replacing it.",
                ];
            }
        }

        // 7. Non-linked images
        $non_clickable_count = 0;
        foreach ($all_images as $img) {
            if ($img['clicks'] === 0 && $img['impressions'] >= $min_impressions * 2) {
                $non_clickable_count++;
            }
        }
        if ($non_clickable_count >= 3) {
            $suggestions[] = [
                'type' => 'info',
                'icon' => '🔗',
                'text' => "{$non_clickable_count} images get views but zero clicks. Add links to drive engagement.",
            ];
        }

        // Limit and prioritize suggestions
        $insights['suggestions'] = array_slice($suggestions, 0, 5);

        return $insights;
    }

    /**
     * AJAX handler for performance insights
     */
    public function ajax_get_performance_stats() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $insights = self::get_performance_insights();
        wp_send_json_success($insights);
    }

    /**
     * AJAX handler to clear all tracking data
     */
    public function ajax_clear_tracking_data() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        global $wpdb;

        // Clear all tracking meta
        $meta_keys = [
            '_lightsync_impressions',
            '_lightsync_clicks',
            '_lightsync_last_click',
            '_lightsync_performance',
        ];

        $deleted = 0;
        foreach ($meta_keys as $key) {
            $deleted += $wpdb->query($wpdb->prepare(
                "DELETE FROM {$wpdb->postmeta} WHERE meta_key = %s",
                $key
            ));
        }

        // Clear page performance transients
        $wpdb->query(
            "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_lsp_page_performance_%' OR option_name LIKE '_transient_timeout_lsp_page_performance_%'"
        );

        // Clear recommendations
        delete_option('lsp_performance_recommendations');

        // Clear A/B tests
        delete_option('lsp_ab_tests');

        self::add_activity("Cleared all tracking data ({$deleted} records)", 'info', 'performance');

        wp_send_json_success(['deleted' => $deleted, 'message' => 'All tracking data cleared']);
    }

    /**
     * AJAX handler to export insights as CSV
     */
    public function ajax_export_insights_csv() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        global $wpdb;

        // Get all images with tracking data
        // Include images with _lightsync_source OR _lightsync_catalog_id (for older Lightroom imports)
        $results = $wpdb->get_results(
            "SELECT p.ID, p.post_title, p.post_date,
                    COALESCE(pm_imp.meta_value, 0) as impressions,
                    COALESCE(pm_click.meta_value, 0) as clicks,
                    pm_album.meta_value as album_id,
                    pm_album_name.meta_value as album_name,
                    pm_perf.meta_value as performance_data,
                    COALESCE(pm_source.meta_value, CASE WHEN pm_cat.meta_value IS NOT NULL THEN 'lightroom' ELSE NULL END) as source
             FROM {$wpdb->posts} p
             LEFT JOIN {$wpdb->postmeta} pm_source ON p.ID = pm_source.post_id AND pm_source.meta_key = '_lightsync_source'
             LEFT JOIN {$wpdb->postmeta} pm_cat ON p.ID = pm_cat.post_id AND pm_cat.meta_key = '_lightsync_catalog_id'
             LEFT JOIN {$wpdb->postmeta} pm_imp ON p.ID = pm_imp.post_id AND pm_imp.meta_key = '_lightsync_impressions'
             LEFT JOIN {$wpdb->postmeta} pm_click ON p.ID = pm_click.post_id AND pm_click.meta_key = '_lightsync_clicks'
             LEFT JOIN {$wpdb->postmeta} pm_album ON p.ID = pm_album.post_id AND pm_album.meta_key = '_lightsync_album_id'
             LEFT JOIN {$wpdb->postmeta} pm_album_name ON p.ID = pm_album_name.post_id AND pm_album_name.meta_key = '_lightsync_album_name'
             LEFT JOIN {$wpdb->postmeta} pm_perf ON p.ID = pm_perf.post_id AND pm_perf.meta_key = '_lightsync_performance'
             WHERE p.post_type = 'attachment'
             AND (pm_source.meta_value IS NOT NULL OR pm_cat.meta_value IS NOT NULL)
             ORDER BY CAST(COALESCE(pm_imp.meta_value, 0) AS UNSIGNED) DESC"
        );

        // Build CSV data
        $csv_rows = [];
        $csv_rows[] = ['ID', 'Title', 'Source', 'Album', 'Impressions', 'Clicks', 'CTR %', 'Date Added', 'Top Position', 'Top Page', 'Desktop %', 'Mobile %'];

        foreach ($results as $row) {
            $impressions = (int) $row->impressions;
            $clicks = (int) $row->clicks;
            $ctr = $impressions > 0 ? round(($clicks / $impressions) * 100, 2) : 0;

            // Parse performance data for context insights
            $perf_data = maybe_unserialize($row->performance_data);
            $top_position = '';
            $top_page = '';
            $desktop_pct = 0;
            $mobile_pct = 0;

            if (is_array($perf_data) && !empty($perf_data)) {
                // Find top performing context
                $best_ctr = 0;
                $total_desktop = 0;
                $total_mobile = 0;
                $total_device = 0;

                foreach ($perf_data as $ctx) {
                    $ctx_imp = $ctx['impressions'] ?? 0;
                    $ctx_click = $ctx['clicks'] ?? 0;
                    $ctx_ctr = $ctx_imp > 0 ? ($ctx_click / $ctx_imp) : 0;

                    if ($ctx_ctr > $best_ctr && $ctx_imp >= 10) {
                        $best_ctr = $ctx_ctr;
                        $top_position = $ctx['position'] ?? '';
                        $top_page = $ctx['page_title'] ?? '';
                    }

                    if (isset($ctx['devices'])) {
                        $total_desktop += ($ctx['devices']['desktop'] ?? 0);
                        $total_mobile += ($ctx['devices']['mobile'] ?? 0) + ($ctx['devices']['tablet'] ?? 0);
                    }
                }

                $total_device = $total_desktop + $total_mobile;
                if ($total_device > 0) {
                    $desktop_pct = round(($total_desktop / $total_device) * 100, 1);
                    $mobile_pct = round(($total_mobile / $total_device) * 100, 1);
                }
            }

            $csv_rows[] = [
                $row->ID,
                $row->post_title,
                $row->source ?: 'Unknown',
                $row->album_name ?: 'Ungrouped',
                $impressions,
                $clicks,
                $ctr,
                $row->post_date,
                $top_position,
                $top_page,
                $desktop_pct,
                $mobile_pct,
            ];
        }

        // Generate CSV content
        $csv_content = '';
        foreach ($csv_rows as $row) {
            $csv_content .= '"' . implode('","', array_map(function($v) {
                return str_replace('"', '""', $v);
            }, $row)) . "\"\n";
        }

        wp_send_json_success([
            'csv' => $csv_content,
            'filename' => 'lightsync-insights-' . date('Y-m-d') . '.csv',
            'count' => count($results),
        ]);
    }

    /**
     * AJAX handler to create an A/B test
     */
    public function ajax_ab_test_create() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $name = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
        $image_a = isset($_POST['image_a']) ? intval($_POST['image_a']) : 0;
        $image_b = isset($_POST['image_b']) ? intval($_POST['image_b']) : 0;
        $position = isset($_POST['position']) ? sanitize_key($_POST['position']) : 'any';

        if (empty($name) || !$image_a || !$image_b) {
            wp_send_json_error(['error' => 'Missing required fields']);
        }

        if ($image_a === $image_b) {
            wp_send_json_error(['error' => 'Images must be different']);
        }

        $tests = get_option('lsp_ab_tests', []);
        
        $test_id = 'ab_' . uniqid();
        $tests[$test_id] = [
            'id' => $test_id,
            'name' => $name,
            'image_a' => $image_a,
            'image_b' => $image_b,
            'position' => $position,
            'created' => time(),
            'status' => 'active',
            // Snapshot baseline stats at creation
            'baseline_a' => [
                'impressions' => (int) get_post_meta($image_a, '_lightsync_impressions', true),
                'clicks' => (int) get_post_meta($image_a, '_lightsync_clicks', true),
            ],
            'baseline_b' => [
                'impressions' => (int) get_post_meta($image_b, '_lightsync_impressions', true),
                'clicks' => (int) get_post_meta($image_b, '_lightsync_clicks', true),
            ],
        ];

        update_option('lsp_ab_tests', $tests, false);

        self::add_activity("Created A/B test: {$name}", 'info', 'performance');

        wp_send_json_success(['test' => $tests[$test_id]]);
    }

    /**
     * AJAX handler to list A/B tests with current results
     */
    public function ajax_ab_test_list() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $tests = get_option('lsp_ab_tests', []);
        $results = [];

        foreach ($tests as $test_id => $test) {
            // Get current stats
            $current_a = [
                'impressions' => (int) get_post_meta($test['image_a'], '_lightsync_impressions', true),
                'clicks' => (int) get_post_meta($test['image_a'], '_lightsync_clicks', true),
            ];
            $current_b = [
                'impressions' => (int) get_post_meta($test['image_b'], '_lightsync_impressions', true),
                'clicks' => (int) get_post_meta($test['image_b'], '_lightsync_clicks', true),
            ];

            // Calculate delta since test started
            $delta_a = [
                'impressions' => $current_a['impressions'] - ($test['baseline_a']['impressions'] ?? 0),
                'clicks' => $current_a['clicks'] - ($test['baseline_a']['clicks'] ?? 0),
            ];
            $delta_b = [
                'impressions' => $current_b['impressions'] - ($test['baseline_b']['impressions'] ?? 0),
                'clicks' => $current_b['clicks'] - ($test['baseline_b']['clicks'] ?? 0),
            ];

            // Calculate CTR for the test period
            $ctr_a = $delta_a['impressions'] > 0 ? ($delta_a['clicks'] / $delta_a['impressions']) * 100 : 0;
            $ctr_b = $delta_b['impressions'] > 0 ? ($delta_b['clicks'] / $delta_b['impressions']) * 100 : 0;

            // Determine winner (need at least 30 impressions each for significance)
            $winner = null;
            $confidence = 'low';
            $min_impressions = min($delta_a['impressions'], $delta_b['impressions']);
            
            if ($min_impressions >= 100) {
                $confidence = 'high';
                $winner = $ctr_a > $ctr_b ? 'a' : ($ctr_b > $ctr_a ? 'b' : 'tie');
            } elseif ($min_impressions >= 30) {
                $confidence = 'medium';
                $winner = $ctr_a > $ctr_b ? 'a' : ($ctr_b > $ctr_a ? 'b' : 'tie');
            }

            // Get image details
            $thumb_a = wp_get_attachment_image_src($test['image_a'], 'thumbnail');
            $thumb_b = wp_get_attachment_image_src($test['image_b'], 'thumbnail');

            $results[] = [
                'id' => $test_id,
                'name' => $test['name'],
                'position' => $test['position'],
                'created' => $test['created'],
                'status' => $test['status'],
                'image_a' => [
                    'id' => $test['image_a'],
                    'title' => get_the_title($test['image_a']),
                    'thumb' => $thumb_a ? $thumb_a[0] : '',
                    'impressions' => $delta_a['impressions'],
                    'clicks' => $delta_a['clicks'],
                    'ctr' => round($ctr_a, 2),
                ],
                'image_b' => [
                    'id' => $test['image_b'],
                    'title' => get_the_title($test['image_b']),
                    'thumb' => $thumb_b ? $thumb_b[0] : '',
                    'impressions' => $delta_b['impressions'],
                    'clicks' => $delta_b['clicks'],
                    'ctr' => round($ctr_b, 2),
                ],
                'winner' => $winner,
                'confidence' => $confidence,
                'lift' => $ctr_b > 0 && $ctr_a > 0 ? round((max($ctr_a, $ctr_b) / min($ctr_a, $ctr_b) - 1) * 100, 1) : 0,
            ];
        }

        // Sort by created date descending
        usort($results, function($a, $b) { return $b['created'] <=> $a['created']; });

        wp_send_json_success(['tests' => $results]);
    }

    /**
     * AJAX handler to delete an A/B test
     */
    public function ajax_ab_test_delete() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $test_id = isset($_POST['test_id']) ? sanitize_text_field($_POST['test_id']) : '';

        if (empty($test_id)) {
            wp_send_json_error(['error' => 'Missing test ID']);
        }

        $tests = get_option('lsp_ab_tests', []);
        
        if (!isset($tests[$test_id])) {
            wp_send_json_error(['error' => 'Test not found']);
        }

        $test_name = $tests[$test_id]['name'];
        unset($tests[$test_id]);
        update_option('lsp_ab_tests', $tests, false);

        self::add_activity("Deleted A/B test: {$test_name}", 'info', 'performance');

        wp_send_json_success(['deleted' => $test_id]);
    }

    /**
     * AJAX handler to dismiss first-sync celebration
     */
    public function ajax_dismiss_celebration() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $source = isset($_POST['source']) ? sanitize_text_field($_POST['source']) : '';

        if ($source === 'lightroom') {
            update_option('lsp_celebrated_lightroom', 1);
        } elseif ($source === 'canva') {
            update_option('lsp_celebrated_canva', 1);
        } elseif ($source === 'dropbox') {
            update_option('lsp_celebrated_dropbox', 1);
        } elseif ($source === 'figma') {
            update_option('lsp_celebrated_figma', 1);
        }

        wp_send_json_success(['dismissed' => $source]);
    }

    /**
     * Toggle helper sidebars visibility
     */
    public function ajax_toggle_helpers() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        $hidden = isset($_POST['hidden']) ? (bool) $_POST['hidden'] : false;
        update_user_meta(get_current_user_id(), 'lsp_helpers_hidden', $hidden ? 1 : 0);

        wp_send_json_success(['hidden' => $hidden]);
    }

    /**
     * REST API endpoint to replace media file in-place (for Hub autosync)
     * Keeps the same attachment ID while replacing the file
     */
    public static function rest_replace_media(\WP_REST_Request $request) {
        $attachment_id = (int) $request->get_param('id');
        
        if (!$attachment_id) {
            return new \WP_Error('invalid_id', 'Invalid attachment ID', ['status' => 400]);
        }
        
        // Verify attachment exists
        $attachment = get_post($attachment_id);
        if (!$attachment || $attachment->post_type !== 'attachment') {
            return new \WP_Error('not_found', 'Attachment not found', ['status' => 404]);
        }
        
        // Get uploaded file
        $files = $request->get_file_params();
        if (empty($files['file'])) {
            return new \WP_Error('no_file', 'No file uploaded', ['status' => 400]);
        }
        
        $file = $files['file'];
        
        // Validate file
        if ($file['error'] !== UPLOAD_ERR_OK) {
            return new \WP_Error('upload_error', 'File upload error: ' . $file['error'], ['status' => 400]);
        }
        
        // Get current file path
        $current_file = get_attached_file($attachment_id);
        if (!$current_file) {
            return new \WP_Error('no_file_path', 'Could not determine attachment file path', ['status' => 500]);
        }
        
        // Get upload directory
        $upload_dir = wp_upload_dir();
        $current_dir = dirname($current_file);
        
        // Ensure we have image functions
        if (!function_exists('wp_generate_attachment_metadata')) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }
        
        // Delete old file and thumbnails
        $metadata = wp_get_attachment_metadata($attachment_id);
        if (!empty($metadata['sizes'])) {
            foreach ($metadata['sizes'] as $size => $size_info) {
                $size_file = $current_dir . '/' . $size_info['file'];
                if (file_exists($size_file)) {
                    @unlink($size_file);
                }
            }
        }
        if (file_exists($current_file)) {
            @unlink($current_file);
        }
        
        // Move new file to same location
        $new_filename = wp_unique_filename($current_dir, $file['name']);
        $new_file = $current_dir . '/' . $new_filename;
        
        if (!move_uploaded_file($file['tmp_name'], $new_file)) {
            return new \WP_Error('move_failed', 'Failed to move uploaded file', ['status' => 500]);
        }
        
        // Update attachment file path
        update_attached_file($attachment_id, $new_file);
        
        // Update post with new file info
        $filetype = wp_check_filetype($new_filename, null);
        wp_update_post([
            'ID' => $attachment_id,
            'post_mime_type' => $filetype['type'],
        ]);
        
        // Regenerate metadata and thumbnails
        $new_metadata = wp_generate_attachment_metadata($attachment_id, $new_file);
        wp_update_attachment_metadata($attachment_id, $new_metadata);
        
        // Get new URL
        $new_url = wp_get_attachment_url($attachment_id);
        
        Logger::debug("[LSP] Replaced attachment $attachment_id: $new_file");
        
        return rest_ensure_response([
            'success' => true,
            'id' => $attachment_id,
            'url' => $new_url,
            'file' => $new_filename,
        ]);
    }

    /**
     * Mark guided tour as completed
     */
    public function ajax_tour_complete() {
        check_ajax_referer(self::AJAX_NS . '_nonce', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Unauthorized'], 403);
        }

        update_user_meta(get_current_user_id(), 'lsp_tour_completed', time());

        wp_send_json_success(['completed' => true]);
    }

    /**
     * Add tracking stats column to Media Library
     */
    public function add_media_tracking_column($columns) {
        $o = self::get_opt();
        if (!empty($o['ai_performance_optimize'])) {
            // Custom chart icon matching LightSync design
            $columns['lsp_tracking'] = '<span title="LightSync Performance" style="display:inline-flex;align-items:center;"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#6366f1" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M3 3v18h18"/><path d="M18 9l-5 5-4-4-3 3"/></svg></span>';
        }
        return $columns;
    }

    /**
     * Render tracking stats in Media Library column
     */
    public function render_media_tracking_column($column_name, $attachment_id) {
        if ($column_name !== 'lsp_tracking') {
            return;
        }

        // SVG icons matching LightSync design language
        $icon_eye = '<svg width="13" height="13" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="display:inline;vertical-align:middle;margin-right:3px;"><path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/><circle cx="12" cy="12" r="3"/></svg>';
        $icon_click = '<svg width="13" height="13" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="display:inline;vertical-align:middle;margin-right:3px;"><path d="M4 4l7.07 17 2.51-7.39L21 11.07z"/><path d="M15 15l6 6"/></svg>';
        $icon_percent = '<svg width="13" height="13" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="display:inline;vertical-align:middle;margin-right:3px;"><line x1="19" y1="5" x2="5" y2="19"/><circle cx="6.5" cy="6.5" r="2.5"/><circle cx="17.5" cy="17.5" r="2.5"/></svg>';

        // Check if this is a LightSync image (any source)
        $source = get_post_meta($attachment_id, '_lightsync_source', true);
        
        // Fallback: check for Lightroom-specific meta (for images synced before _lightsync_source was added)
        if (empty($source)) {
            $catalog_id = get_post_meta($attachment_id, '_lightsync_catalog_id', true);
            if (!empty($catalog_id)) {
                $source = 'lightroom';
                // Backfill the source meta for future checks
                update_post_meta($attachment_id, '_lightsync_source', 'lightroom');
            }
        }
        
        if (empty($source)) {
            echo '<span style="color:#94a3b8;">—</span>';
            return;
        }

        $impressions = (int) get_post_meta($attachment_id, '_lightsync_impressions', true);
        $clicks = (int) get_post_meta($attachment_id, '_lightsync_clicks', true);
        $ctr = $impressions > 0 ? round(($clicks / $impressions) * 100, 1) : 0;

        if ($impressions === 0) {
            echo '<span style="color:#94a3b8;font-size:11px;">No data</span>';
            return;
        }

        // Color code based on CTR
        $ctr_color = '#64748b';
        if ($ctr >= 5) $ctr_color = '#22c55e';
        elseif ($ctr >= 2) $ctr_color = '#eab308';
        elseif ($ctr < 1 && $impressions >= 50) $ctr_color = '#ef4444';

        echo '<div style="font-size:11px;line-height:1.6;color:#64748b;">';
        echo '<span title="Impressions" style="display:flex;align-items:center;">' . $icon_eye . number_format($impressions) . '</span>';
        echo '<span title="Clicks" style="display:flex;align-items:center;">' . $icon_click . number_format($clicks) . '</span>';
        echo '<span title="Click-through rate" style="display:flex;align-items:center;color:' . $ctr_color . ';font-weight:600;">' . $icon_percent . $ctr . '%</span>';
        echo '</div>';
    }

    /**
     * Cleanup old tracking data (cron job)
     * Removes contextual data older than retention period
     */
    public function cleanup_old_tracking_data() {
        $o = self::get_opt();
        $retention_days = (int) ($o['tracking_retention_days'] ?? 90);
        
        if ($retention_days <= 0) {
            return; // Retention disabled
        }

        $cutoff = time() - ($retention_days * DAY_IN_SECONDS);
        
        global $wpdb;
        
        // Get all images with performance data
        $results = $wpdb->get_results(
            "SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE meta_key = '_lightsync_performance'"
        );

        $cleaned = 0;
        foreach ($results as $row) {
            $perf_data = maybe_unserialize($row->meta_value);
            if (!is_array($perf_data)) continue;

            $modified = false;
            foreach ($perf_data as $key => $ctx) {
                if (isset($ctx['last_seen']) && $ctx['last_seen'] < $cutoff) {
                    unset($perf_data[$key]);
                    $modified = true;
                    $cleaned++;
                }
            }

            if ($modified) {
                if (empty($perf_data)) {
                    delete_post_meta($row->post_id, '_lightsync_performance');
                } else {
                    update_post_meta($row->post_id, '_lightsync_performance', $perf_data);
                }
            }
        }

        if ($cleaned > 0) {
            self::add_activity("Cleaned up {$cleaned} old tracking contexts (>{$retention_days} days)", 'info', 'performance');
        }
    }

    /**
     * Get tracking data storage size estimate
     */
    public static function get_tracking_storage_size() {
        global $wpdb;

        $meta_keys = [
            '_lightsync_impressions',
            '_lightsync_clicks',
            '_lightsync_last_click',
            '_lightsync_performance',
        ];

        $total_size = 0;
        $total_rows = 0;

        foreach ($meta_keys as $key) {
            $result = $wpdb->get_row($wpdb->prepare(
                "SELECT COUNT(*) as cnt, SUM(LENGTH(meta_value)) as size 
                 FROM {$wpdb->postmeta} WHERE meta_key = %s",
                $key
            ));
            if ($result) {
                $total_rows += (int) $result->cnt;
                $total_size += (int) $result->size;
            }
        }

        return [
            'rows' => $total_rows,
            'bytes' => $total_size,
            'formatted' => size_format($total_size),
        ];
    }

    /* ==================== END AI INSIGHTS HANDLERS ==================== */

    /**
     * Output frontend tracking script if AI performance tracking is enabled
     */
    public function maybe_output_tracking_script() {
        // Only output if performance optimization is enabled
        $o = self::get_opt();
        if (empty($o['ai_performance_optimize'])) {
            return;
        }

        // Check if we're on a page with LightSync images (check for the source meta)
        global $wpdb;
        $has_lsp_images = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = '_lightsync_source' LIMIT 1"
        );

        if (!$has_lsp_images) {
            return;
        }

        $ajax_url = admin_url('admin-ajax.php');
        
        // Get current page context for the script
        $page_id = get_queried_object_id();
        $page_type = 'page';
        $page_title = '';
        
        if (is_singular('post')) {
            $page_type = 'post';
            $page_title = get_the_title();
        } elseif (is_singular('product')) {
            $page_type = 'product';
            $page_title = get_the_title();
        } elseif (is_singular('page')) {
            $page_type = 'page';
            $page_title = get_the_title();
        } elseif (is_front_page()) {
            $page_type = 'home';
            $page_title = 'Homepage';
        } elseif (is_archive()) {
            $page_type = 'archive';
            $page_title = get_the_archive_title();
        } elseif (is_search()) {
            $page_type = 'search';
            $page_title = 'Search Results';
        }
        ?>
        <script>
        (function() {
            'use strict';
            
            // Page context
            var pageContext = {
                page_id: <?php echo (int) $page_id; ?>,
                page_type: <?php echo wp_json_encode($page_type); ?>,
                page_title: <?php echo wp_json_encode($page_title); ?>
            };
            
            var tracked = new Set();
            var ajaxUrl = <?php echo wp_json_encode($ajax_url); ?>;
            
            // Detect device type
            function getDeviceType() {
                var w = window.innerWidth;
                if (w < 768) return 'mobile';
                if (w < 1024) return 'tablet';
                return 'desktop';
            }
            
            // Detect image position/context
            function getImagePosition(img) {
                var rect = img.getBoundingClientRect();
                var viewH = window.innerHeight;
                var scrollY = window.scrollY || window.pageYOffset;
                var imgTop = rect.top + scrollY;
                
                // Check if it's above the fold on initial load
                var isAboveFold = imgTop < viewH;
                
                // Check common position indicators
                var parent = img.parentElement;
                var grandparent = parent ? parent.parentElement : null;
                var classes = (img.className || '') + ' ' + (parent ? parent.className || '' : '') + ' ' + (grandparent ? grandparent.className || '' : '');
                classes = classes.toLowerCase();
                
                // Detect hero/banner
                if (classes.match(/hero|banner|featured|cover|jumbotron|masthead/)) {
                    return 'hero';
                }
                
                // Detect gallery
                if (classes.match(/gallery|grid|masonry|lightbox|carousel|slider|swiper/)) {
                    return 'gallery';
                }
                
                // Detect product image
                if (classes.match(/product|woocommerce|shop/)) {
                    return 'product';
                }
                
                // Detect sidebar
                if (img.closest('aside, .sidebar, .widget, [role="complementary"]')) {
                    return 'sidebar';
                }
                
                // Detect header
                if (img.closest('header, .header, .site-header, [role="banner"]')) {
                    return 'header';
                }
                
                // Detect footer
                if (img.closest('footer, .footer, .site-footer, [role="contentinfo"]')) {
                    return 'footer';
                }
                
                // Position based on fold
                if (isAboveFold) {
                    return 'above_fold';
                }
                
                return 'content';
            }
            
            // Get position index (nth image on page)
            function getPositionIndex(img) {
                var allImages = document.querySelectorAll('img[class*="wp-image-"]');
                for (var i = 0; i < allImages.length; i++) {
                    if (allImages[i] === img) return i + 1;
                }
                return 0;
            }
            
            // Check if image is linked
            function getLinkInfo(img) {
                var link = img.closest('a');
                if (!link) return { is_linked: false, link_target: '' };
                return {
                    is_linked: true,
                    link_target: link.href || ''
                };
            }
            
            // Build tracking payload
            function buildPayload(attachmentId, img, extra) {
                var position = getImagePosition(img);
                var linkInfo = getLinkInfo(img);
                
                var payload = 'action=' + extra.action +
                    '&attachment_id=' + attachmentId +
                    '&page_id=' + pageContext.page_id +
                    '&page_type=' + encodeURIComponent(pageContext.page_type) +
                    '&page_title=' + encodeURIComponent(pageContext.page_title) +
                    '&position=' + position +
                    '&position_idx=' + getPositionIndex(img) +
                    '&device=' + getDeviceType() +
                    '&viewport_w=' + window.innerWidth +
                    '&is_linked=' + (linkInfo.is_linked ? '1' : '0') +
                    '&link_target=' + encodeURIComponent(linkInfo.link_target);
                
                return payload;
            }
            
            function trackImpression(attachmentId, img) {
                var key = attachmentId + '_' + pageContext.page_id;
                if (tracked.has(key)) return;
                tracked.add(key);
                
                var xhr = new XMLHttpRequest();
                xhr.open('POST', ajaxUrl, true);
                xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                xhr.send(buildPayload(attachmentId, img, { action: 'lsp_track_impression' }));
            }
            
            function trackClick(attachmentId, img) {
                var xhr = new XMLHttpRequest();
                xhr.open('POST', ajaxUrl, true);
                xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                xhr.send(buildPayload(attachmentId, img, { action: 'lsp_track_click' }));
            }
            
            // Find LightSync images and set up tracking
            function initTracking() {
                var images = document.querySelectorAll('img[class*="wp-image-"]');
                
                images.forEach(function(img) {
                    // Extract attachment ID from class
                    var match = img.className.match(/wp-image-(\d+)/);
                    if (!match) return;
                    
                    var attachmentId = match[1];
                    
                    // Intersection Observer for impressions
                    if ('IntersectionObserver' in window) {
                        var observer = new IntersectionObserver(function(entries) {
                            entries.forEach(function(entry) {
                                if (entry.isIntersecting) {
                                    trackImpression(attachmentId, entry.target);
                                    observer.unobserve(entry.target);
                                }
                            });
                        }, { threshold: 0.5 });
                        observer.observe(img);
                    } else {
                        // Fallback: track on load
                        trackImpression(attachmentId, img);
                    }
                    
                    // Click tracking (prevent double-counting when image is inside a link)
                    img.addEventListener('click', function(e) {
                        trackClick(attachmentId, img);
                        // If inside a link, we've already tracked - don't let link handler track again
                        e.lspTracked = true;
                    });
                    
                    // Also track parent link clicks (only if not already tracked by image click)
                    var parentLink = img.closest('a');
                    if (parentLink) {
                        parentLink.addEventListener('click', function(e) {
                            if (!e.lspTracked) {
                                trackClick(attachmentId, img);
                            }
                        });
                    }
                });
            }
            
            // Run on DOM ready
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', initTracking);
            } else {
                initTracking();
            }
        })();
        </script>
        <?php
    }

    private function any_album_cron_scheduled(): bool {
        $albums = (array) self::get_opt('album_ids',[]);
        foreach ($albums as $aid) {
            if (wp_next_scheduled(self::CRON_ALBUM, [(string)$aid])) return true;
        }
        return false;
    }

    private function lightsync_count_scheduled_for_hook( $hook ) : int {
        if ( ! function_exists( '_get_cron_array' ) ) {
            return 0;
        }
        $count = 0;
        $crons = _get_cron_array();
        if ( empty( $crons ) || ! is_array( $crons ) ) {
            return 0;
        }
        foreach ( $crons as $timestamp => $events ) {
            if ( empty( $events ) || ! is_array( $events ) ) {
                continue;
            }
            foreach ( $events as $h => $data ) {
                if ( $h === $hook && is_array( $data ) ) {
                    $count += count( $data );
                }
            }
        }
        return $count;
    }

    public static function add_activity(string $message, string $status = 'info', string $source = ''): void {
        $items = get_option('lightsync_recent_activity', []);
        if (!is_array($items)) $items = [];

        array_unshift($items, [
            'ts'      => time(),
            'message' => $message,
            'status'  => $status,
            'type'    => 'sync',
            'source'  => $source,
        ]);

        $items = array_slice($items, 0, 15);
        update_option('lightsync_recent_activity', $items, false);

        $legacy = (array) self::get_opt('lightsync_activity', []);
        array_unshift($legacy, [
            'ts'     => time(),
            'msg'    => $message,
            'type'   => $status,
            'status' => $status,
            'source' => $source,
        ]);
        $legacy = array_slice($legacy, 0, 15);
        self::set_opt(['lightsync_activity' => $legacy]);
    }

    private function lightsync_count_events( string $hook, array $args_filter = [] ): int {
        if ( ! function_exists('_get_cron_array') ) return 0;
        $crons = _get_cron_array();
        if ( empty($crons) || !is_array($crons) ) return 0;

        $count = 0;
        foreach ($crons as $timestamp => $events) {
            if (empty($events[$hook]) || !is_array($events[$hook])) continue;

            foreach ($events[$hook] as $hash => $event) {
                if (!$args_filter) { $count++; continue; }
                $matches = true;
                foreach ($args_filter as $k => $v) {
                    if (!isset($event['args'][0][$k]) || $event['args'][0][$k] !== $v) {
                        $matches = false; break;
                    }
                }
                if ($matches) $count++;
            }
        }
        return $count;
    }

    private function lightsync_clear_hook( string $hook ): void {
        if ( ! function_exists('_get_cron_array') ) return;
        $crons = _get_cron_array();
        if ( empty($crons) || !is_array($crons) ) return;

        foreach ($crons as $timestamp => $events) {
            if (empty($events[$hook])) continue;
            foreach ($events[$hook] as $hash => $event) {
                wp_unschedule_event($timestamp, $hook, $event['args'] ?? []);
            }
        }
    }

    private static function pending_count(): int {
        return count( (array) get_option('lightsync_pending_renditions', []) );
    }

    private static function queue_rendition_retry($catalog_id, $album_id, $asset_id) {
        $queue = (array) get_option('lightsync_pending_renditions', []);

        foreach ($queue as $q) {
            if ((string)$q['catalog'] === (string)$catalog_id
             && (string)$q['album']   === (string)$album_id
             && (string)$q['asset']   === (string)$asset_id) {
                return;
            }
        }

        $queue[] = [
            'catalog'  => (string) $catalog_id,
            'album'    => (string) $album_id,
            'asset'    => (string) $asset_id,
            'ts'       => time(),
            'attempts' => 0,
            'next_at'  => time() + 300,
        ];
        
        if (count($queue) > 2000) {
            $queue = array_slice($queue, -2000);
        }
        update_option('lightsync_pending_renditions', $queue, false);
    }

    private static function album_name_cache_key(string $catalog_id): string {
        return 'lightsync_album_names_' . preg_replace('/[^a-zA-Z0-9_\-]/', '_', $catalog_id);
    }

    public static function get_album_name_cached(string $catalog_id, string $album_id): string {
        $map = get_option(self::album_name_cache_key($catalog_id), []);
        if (is_array($map) && isset($map[$album_id]) && $map[$album_id] !== '') {
            return (string) $map[$album_id];
        }
        return $album_id;
    }

    private static function set_album_name_cache(string $catalog_id, array $map): void {
        $map = array_filter($map, static function($v){
            return is_string($v) && $v !== '';
        });

        update_option(self::album_name_cache_key($catalog_id), $map, false);
    }

    public static function cron_sync_tick(string $catalog_id, string $album_id, string $source = 'extension'): void {
    if (!$catalog_id || !$album_id) {
        \LightSyncPro\Util\Logger::debug('[LSP cron_sync_tick] bail: missing catalog or album');
        return;
    }

    // ✅ Lock to prevent overlapping ticks
    $lock_key = "lightsync_tick_lock_{$catalog_id}_{$album_id}";
    if (get_transient($lock_key)) {
        \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] bail: lock present ({$album_id})");
        return;
    }
    set_transient($lock_key, time(), 5 * MINUTE_IN_SECONDS);

    try {
        // ✅ Read saved cursor/index from options
        $cursor_opt = "lightsync_sync_cursor_{$catalog_id}_{$album_id}";
        $idx_opt    = "lightsync_sync_next_index_{$catalog_id}_{$album_id}";
        
        $cursor      = (string) get_option($cursor_opt, '');
        $start_index = (int) get_option($idx_opt, 0);

        \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] resuming album={$album_id} cursor_len=" . strlen($cursor) . " start_index={$start_index}");

        // Run one batch slice
        $batchSz = (int) self::get_opt('batch_size', 100);
        $limit   = 200;
        $time_budget = 50.0;

        // Import to WordPress
        $out = \LightSyncPro\Sync\Sync::batch_import(
                $catalog_id,
                $album_id,
                $cursor,
                $batchSz,
                $limit,
                $start_index,
                $time_budget
            );

        if (is_wp_error($out)) {
            \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] error: " . $out->get_error_message());
            delete_transient($lock_key);
            return;
        }

        // ✅ Update usage
        $imported = count((array)($out['imported'] ?? []));
        if ($imported > 0) {
            self::usage_consume($imported);
        }

        // ✅ Extract continuation pointers
        $next_cursor = (string)($out['next_cursor'] ?? $out['cursor'] ?? '');
        $next_index  = (int)($out['next_index'] ?? 0);
        
        // ✅ Album is done when cursor is empty AND index is 0
        $album_done = ($next_cursor === '' && $next_index === 0);

        \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] album_done={$album_done} next_cursor_len=" . strlen($next_cursor) . " next_index={$next_index} imported={$imported}");

        // ✅ If album not done, save pointers and schedule next tick
        if (!$album_done) {
            update_option($cursor_opt, $next_cursor, false);
            update_option($idx_opt, $next_index, false);

            // Schedule next tick in 3 seconds
            $args = [$catalog_id, $album_id, $source];
            if (!wp_next_scheduled('lightsyncpro_sync_tick', $args)) {
                wp_schedule_single_event(time() + 3, 'lightsyncpro_sync_tick', $args);
            }
        } else {
            // ✅ Cleanup: remove saved pointers when complete
            delete_option($cursor_opt);
            delete_option($idx_opt);

            // =============================================
            // BACKGROUND QUEUE - Process next album if this was a background sync
            // =============================================
            if ($source === 'manual-background') {
                \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] Background sync - checking queue");
                
                $queue = get_option('lightsync_background_sync_queue', []);
                
                if (!empty($queue) && is_array($queue['albums'] ?? null)) {
                    $albums = (array) $queue['albums'];
                    $current_idx = (int) ($queue['current_index'] ?? 0);
                    $next_idx = $current_idx + 1;
                    
                    \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] Queue found: current={$current_idx}, total=" . count($albums) . ", next={$next_idx}");
                    
                    if ($next_idx < count($albums)) {
                        // More albums to process - schedule next one
                        $next_album = $albums[$next_idx];
                        
                        $queue['current_index'] = $next_idx;
                        update_option('lightsync_background_sync_queue', $queue, false);
                        
                        \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] Scheduling next album: {$next_album}");
                        
                        \LightSyncPro\Sync\Sync::schedule_sync_tick(
                            (string) $catalog_id,
                            (string) $next_album,
                            'manual-background',
                            5
                        );
                        
                        \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] Background queue: starting album {$next_idx} of " . count($albums));
                    } else {
                        // All albums complete - clear the queue
                        \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] All albums complete, clearing queue");
                        
                        delete_option('lightsync_background_sync_queue');
                        
                        $elapsed = human_time_diff((int)($queue['started_at'] ?? time()), time());
                        
                        self::add_activity(
                            sprintf('Lightroom → WordPress Sync Complete (Background): %d album(s) in %s', count($albums), $elapsed),
                            'success',
                            'manual-background'
                        );
                        
                        \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] Background queue complete: " . count($albums) . " albums");
                    }
                } else {
                    \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] No background queue found");
                }
            }

            $album_name = self::get_album_name_cached((string)$catalog_id, (string)$album_id);
            
            // Map source to human-readable sync type
            $sync_type_map = [
                'extension' => 'Extension',
                'manual-background' => 'Background',
                'auto' => 'Auto',
                'manual' => 'Manual',
            ];
            $sync_type = $sync_type_map[$source] ?? ucfirst($source);
            $dest_label = 'WordPress';

            self::add_activity(
                sprintf('Lightroom → %s Sync Complete (%s): "%s"', $dest_label, $sync_type, $album_name),
                'success',
                (string)$source
            );


            // ✅ Update last sync timestamp
            self::set_opt([
                'lightsync_last_sync_ts'     => time(),
                'lightsync_last_sync_source' => $source,
            ]);

            \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] complete for album {$album_id}");
        }

    } catch (\Throwable $e) {
        \LightSyncPro\Util\Logger::debug("[LSP cron_sync_tick] exception: " . $e->getMessage());
    } finally {
        delete_transient($lock_key);
    }
}

    public static function cron_retry_renditions() {
        $queue = (array) get_option('lightsync_pending_renditions', []);
        if (empty($queue)) return;

        $updated_queue = [];
        $processed_this_run = 0;
        $max_per_run = 25;

        foreach ($queue as $item) {
            if ($processed_this_run >= $max_per_run) {
                $updated_queue[] = $item;
                continue;
            }

            $catalog_id = (string) ($item['catalog'] ?? '');
            $album_id   = (string) ($item['album']   ?? '');
            $asset_id   = (string) ($item['asset']   ?? '');
            $attempts   = (int)    ($item['attempts'] ?? 0);
            $next_at    = (int)    ($item['next_at']  ?? 0);

            if ($next_at > time()) {
                $updated_queue[] = $item;
                continue;
            }

            if (!$catalog_id || !$album_id || !$asset_id) {
                continue;
            }

            $resource = [
                'id'      => $asset_id,
                'asset'   => ['id' => $asset_id],
                'payload' => [],
            ];

            try {
                $album_name = '';
                $albs = \LightSyncPro\Sync\Sync::get_albums($catalog_id);
                if (!is_wp_error($albs)) {
                    foreach (($albs['resources'] ?? []) as $a) {
                        if (!empty($a['id']) && $a['id'] === $album_id) {
                            $album_name = $a['payload']['name'] ?? '';
                            break;
                        }
                    }
                }

                $out = \LightSyncPro\Sync\Sync::import_or_update($catalog_id, $album_id, $album_name, $resource);

                if (!empty($out['imported']) || !empty($out['updated']) || !empty($out['updated_meta'])) {
                    $processed_this_run++;
                    continue;
                }

                $attempts++;
                $backoff = min(3600, max(120, 60 * pow(2, min($attempts, 6))));
                $item['attempts'] = $attempts;
                $item['next_at']  = time() + $backoff;

                if ($attempts >= 10) {
                    Logger::debug('[LSP] dropping rendition retry after 10 attempts for asset '.$asset_id);
                    continue;
                }

                $updated_queue[] = $item;
                $processed_this_run++;

            } catch (\Throwable $e) {
                $attempts++;
                $item['attempts'] = $attempts;
                $item['next_at']  = time() + 600;
                $updated_queue[]  = $item;
                Logger::debug('[LSP] cron_retry_renditions error: '.$e->getMessage());
            }
        }

        update_option('lightsync_pending_renditions', $updated_queue, false);
    }

    public function ajax_disconnect() {
        $ok = check_ajax_referer(self::AJAX_NS . '_nonce', '_ajax_nonce', false);
        if ( ! $ok ) {
            $ok = check_ajax_referer('lightsyncpro_nonce', '_ajax_nonce', false);
        }
        if ( ! $ok ) {
            wp_send_json_error(['message'=>'bad_nonce'], 403);
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message'=>'forbidden'], 403);
        }

        // Usage is now in separate option - no need to preserve/restore
        \LightSyncPro\OAuth\OAuth::disconnect();

        wp_send_json_success([
            'message' => 'Disconnected from Adobe. You can now reconnect.',
        ]);
    }

    public static function ajax_save_broker() {
        if ( ! check_ajax_referer( self::AJAX_NS . '_nonce', '_ajax_nonce', false ) ) {
            wp_send_json_error( [ 'message' => 'bad_nonce' ], 403 );
        }

        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error( [ 'message' => 'forbidden' ], 403 );
        }

        $broker   = isset($_POST['broker_token']) ? sanitize_text_field( wp_unslash($_POST['broker_token']) ) : '';
        $clientId = isset($_POST['client_id'])    ? sanitize_text_field( wp_unslash($_POST['client_id']) )    : '';

        if ( empty($broker) ) {
            wp_send_json_error( [ 'message' => 'Missing broker token' ], 400 );
        }

        \LightSyncPro\Admin\Admin::set_opt( [
            'broker_token_enc' => \LightSyncPro\Util\Crypto::enc( $broker ),
            'client_id'        => $clientId,
        ] );

        delete_transient('lightsync_license_checked');
        $admin = new self();
        $admin->maybe_validate_license();
        
        if (method_exists($admin, 'activate_license_if_possible')) {
            $admin->activate_license_if_possible();
        }

        if ( defined('WP_DEBUG') && WP_DEBUG ) {
            Logger::debug('[LSP] ajax_save_broker – stored broker_token_enc + client_id=' . $clientId);
        }

        wp_send_json_success( [ 'message' => 'Adobe connection saved.', 'saved' => true ] );
    }

    public function ajax_usage_get() {
        if ( ! check_ajax_referer(self::AJAX_NS.'_nonce','_ajax_nonce', false) ) {
            wp_send_json_error(['error'=>'bad_nonce'], 403);
        }
        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error(['error'=>'forbidden'], 403);
        }

        $u    = self::usage_get();
        $caps = self::usage_caps();
        $cap  = (int) ($caps['month'] ?? 0);
        $used = (int) ($u['month_count'] ?? 0);

        $pct   = ($cap > 0) ? min(100, (int) round(($used / $cap) * 100)) : 0;
        $label = ($cap > 0) ? ($used . ' / ' . $cap) : (string) $used;

        wp_send_json_success([
            'used'      => $used,
            'cap'       => $cap,
            'pct'       => $pct,
            'label'     => $label,
            'remaining' => ($cap > 0) ? max(0, $cap - $used) : 0,
        ]);
    }

    public static function get_opt($k = null, $d = null){
        $o = get_option(self::OPT, []);
        if (!is_array($o)) {
            $try = json_decode((string)$o, true);
            if (!is_array($try)) $try = maybe_unserialize($o);
            $o = is_array($try) ? $try : [];
        }
        if ($k === null) return $o;
        return array_key_exists($k, $o) ? $o[$k] : $d;
    }
    
    public static function set_opt($arr){
        $cur = self::get_opt();
        $arr = (array) $arr;
        update_option(self::OPT, array_merge($cur, $arr), false);
    }

    /**
     * Get album destinations for JS, with debug logging
     */
    private static function get_album_destinations_for_js(): array {
        // Bypass any object cache
        wp_cache_delete(self::OPT, 'options');
        
        // Get fresh from database
        global $wpdb;
        $raw = $wpdb->get_var($wpdb->prepare(
            "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s LIMIT 1",
            self::OPT
        ));
        
        $all_opts = maybe_unserialize($raw);
        if (!is_array($all_opts)) {
            $all_opts = json_decode($raw, true);
        }
        if (!is_array($all_opts)) {
            $all_opts = [];
        }
        
        $destinations = (array) ($all_opts['album_destinations'] ?? []);
        
        return $destinations;
    }

    /**
     * Get sync status for each album (image count, last sync time, and destinations)
     */
    private static function get_album_sync_status(): array {
        global $wpdb;
        
        // Get WordPress sync counts per album
        $wp_results = $wpdb->get_results("
            SELECT 
                pm_album.meta_value as album_id,
                COUNT(DISTINCT p.ID) as image_count,
                MAX(pm_sync.meta_value) as last_sync
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm_album ON p.ID = pm_album.post_id 
                AND pm_album.meta_key = '_lightsync_album_id'
            LEFT JOIN {$wpdb->postmeta} pm_sync ON p.ID = pm_sync.post_id 
                AND pm_sync.meta_key = '_lightsync_last_synced_at'
            WHERE p.post_type = 'attachment'
                AND p.post_status = 'inherit'
            GROUP BY pm_album.meta_value
        ");
        
        $status = [];
        foreach ($wp_results as $row) {
            // _lightsync_last_synced_at is stored as gmdate('Y-m-d H:i:s') — UTC without marker
            // Must append ' UTC' so strtotime() interprets correctly regardless of server timezone
            $last_sync_ts = $row->last_sync ? strtotime($row->last_sync . ' UTC') : null;
            $status[$row->album_id] = [
                'count' => (int) $row->image_count,
                'last_sync' => $last_sync_ts,
                'wp' => true,
                'shopify' => false,
            ];
        }
        
        // Check Shopify syncs
        $shopify_table = $wpdb->prefix . 'lightsync_shopify_files_map';
        $shopify_table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $shopify_table));
        
        if ($shopify_table_exists) {
            $shop_domain = self::get_opt('shopify_shop_domain');
            if ($shop_domain) {
                $shopify_results = $wpdb->get_results($wpdb->prepare("
                    SELECT 
                        album_id,
                        COUNT(*) as shopify_count,
                        MAX(updated_at) as last_shopify_sync
                    FROM {$shopify_table}
                    WHERE shop_domain = %s
                        AND album_id IS NOT NULL
                        AND album_id != ''
                        AND shopify_file_id IS NOT NULL
                    GROUP BY album_id
                ", $shop_domain));
                
                foreach ($shopify_results as $row) {
                    if (!isset($status[$row->album_id])) {
                        $status[$row->album_id] = [
                            'count' => (int) $row->shopify_count,
                            'last_sync' => $row->last_shopify_sync ? strtotime($row->last_shopify_sync) : null,
                            'wp' => false,
                            'shopify' => true,
                        ];
                    } else {
                        $status[$row->album_id]['shopify'] = true;
                        $status[$row->album_id]['count'] = max($status[$row->album_id]['count'], (int) $row->shopify_count);
                        $shopify_ts = $row->last_shopify_sync ? strtotime($row->last_shopify_sync) : null;
                        if ($shopify_ts && (!$status[$row->album_id]['last_sync'] || $shopify_ts > $status[$row->album_id]['last_sync'])) {
                            $status[$row->album_id]['last_sync'] = $shopify_ts;
                        }
                    }
                }
            }
        }
        
        return $status;
    }

    /**
     * Check if a source has been synced to Hub
     * @param string $source_type 'lightroom', 'canva', 'figma', 'dropbox'
     * @param string $source_id Album ID, design ID, file key, or file ID
     */
    public static function plan(): string {
        return 'free';
    }
    
    public static function is_pro(): bool {
        return false;
    }

    private static function plan_matrix(): array {
        return [
            'free' => [
                'photos_month' => 0,
                'max_albums'   => 0,
            ],
        ];
    }
    
    public static function has_cap(string $cap): bool {
        // Free version capabilities
        $free_caps = [
            'naming'       => true,
            'alt'          => true,
            'multi_albums' => true,
        ];
        return (bool) ($free_caps[$cap] ?? false);
    }

    public static function delete_all_plugin_data(): void {
        global $wpdb;

        delete_option( self::OPT );
        delete_option( self::USAGE_OPT );
        delete_option( 'lightsync_usage_migrated' );
        delete_option( '_lightsync_cron_cadence' );
        delete_transient( 'lightsync_license_checked' );
        delete_option('lightsync_pending_renditions');

        if ( function_exists( 'wp_clear_scheduled_hook' ) ) {
            wp_clear_scheduled_hook( self::CRON );
            $self = new self();
            $self->unschedule_all_album_events();
            wp_clear_scheduled_hook( self::CRON_RETRY_HOOK );
        }

        $wpdb->query(
            "DELETE FROM {$wpdb->options}
             WHERE option_name LIKE '_transient_lightsync_%'
                OR option_name LIKE '_transient_timeout_lightsync_%'"
        );

        try {
            $broker_endpoint = 'https://lightsyncpro.com/wp-json/lsp-broker/v1/disconnect';
            $body = wp_json_encode([
                'site'   => home_url(),
                'reason' => 'plugin_deactivate_wipe',
            ]);

            $resp = wp_remote_post($broker_endpoint, [
                'timeout' => 10,
                'headers' => [ 'Content-Type' => 'application/json' ],
                'body'    => $body,
            ]);

            if ( is_wp_error( $resp ) ) {
                Logger::debug('[LSP uninstall] broker disconnect error: '.$resp->get_error_message());
            }
        } catch ( \Throwable $e ) {
            Logger::debug('[LSP uninstall] exception: '.$e->getMessage());
        }
    }

    public function add_cron_interval($schedules){
        $schedules['wplr_15mins']     = ['interval'=> 900,   'display'=>__('Every 15 minutes', 'lightsyncpro')];
        $schedules['wplr_hourly']     = ['interval'=> 3600,  'display'=>__('Every hour', 'lightsyncpro')];
        $schedules['wplr_twicedaily'] = ['interval'=> 43200, 'display'=>__('Twice daily', 'lightsyncpro')];
        $schedules['wplr_daily']      = ['interval'=> 86400, 'display'=>__('Daily', 'lightsyncpro')];

        if (!isset($schedules['lightsync_every_2_minutes'])) {
            $schedules['lightsync_every_2_minutes'] = [
                'interval' => 120,
                'display'  => __('LightSync Pro: Every 2 Minutes', 'lightsyncpro'),
            ];
        }
        return $schedules;
    }

    protected function freq_to_schedule_slug($freq) {
        switch ($freq) {
            case '15m':
            case '15min':
            case '15mins':
                return 'wplr_15mins';
            case 'hourly':
                return 'wplr_hourly';
            case 'twicedaily':
            case 'twice-daily':
                return 'wplr_twicedaily';
            case 'daily':
                return 'wplr_daily';
            default:
                return 'wplr_hourly';
        }
    }

    protected function unschedule_all_album_events(): void {
        $crons = _get_cron_array();
        if (!is_array($crons)) return;

        foreach ($crons as $timestamp => $hooks) {
            if (!isset($hooks[self::CRON_ALBUM])) continue;
            foreach ($hooks[self::CRON_ALBUM] as $sig => $event) {
                $args = $event['args'] ?? [];
                wp_unschedule_event($timestamp, self::CRON_ALBUM, $args);
            }
        }
    }

    public function unschedule_album_events_not_in(array $keep_album_ids): void {
        $keep = array_flip(array_map('strval', $keep_album_ids));

        $crons = _get_cron_array();
        if (empty($crons)) return;

        foreach ($crons as $timestamp => $hooks) {
            if (empty($hooks[self::CRON_ALBUM])) continue;

            foreach ($hooks[self::CRON_ALBUM] as $sig => $event) {
                $args = $event['args'] ?? [];
                $aid  = isset($args[0]) ? (string) $args[0] : '';

                if ($aid === '' || !isset($keep[$aid])) {
                    wp_unschedule_event((int)$timestamp, self::CRON_ALBUM, $args);
                }
            }
        }
    }

    public function cleanup_orphaned_crons(): void {
        $crons = _get_cron_array();
        if (empty($crons)) return;
        
        $current_catalog = self::get_opt('catalog_id', '');
        $current_albums = (array) self::get_opt('album_ids', []);
        
        foreach ($crons as $timestamp => $hooks) {
            if (empty($hooks[self::CRON_ALBUM])) continue;
            
            foreach ($hooks[self::CRON_ALBUM] as $sig => $event) {
                $args = $event['args'] ?? [];
                $aid = isset($args[0]) ? (string) $args[0] : '';
                
                if ($aid && !in_array($aid, $current_albums, true)) {
                    wp_unschedule_event((int)$timestamp, self::CRON_ALBUM, $args);
                    Logger::debug("[LSP] Cleaned orphaned cron for album {$aid}");
                }
            }
        }
    }

    public function album_cron_handler(string $album_id): void {
        if ( ! self::has_cap('autosync') ) {
            Logger::debug("[LSP] album cron bail: autosync not allowed (album {$album_id})");
            return;
        }

        // Check if Lightroom auto-sync is globally enabled
        if ( ! (bool) self::get_opt('lightroom_autosync_updates', true) ) {
            Logger::debug("[LSP] album cron bail: lightroom autosync disabled (album {$album_id})");
            return;
        }

        $allowed = array_slice((array) self::get_opt('album_ids', []), 0, 1);
        if (!in_array($album_id, array_map('strval',$allowed), true) && ! self::has_cap('multi_albums')) {
            Logger::debug("[LSP] album cron bail: free plan, album not allowed ({$album_id})");
            return;
        }

        $lock_key = "lightsync_cron_lock_{$album_id}";
        $lock_duration = 45 * MINUTE_IN_SECONDS;
        
        if ( get_transient($lock_key) ) {
            Logger::debug("[LSP] album cron bail: lock present ({$album_id})");
            return;
        }
        
        set_transient($lock_key, time(), $lock_duration);

        try {
            $cat = self::get_opt('catalog_id','');
            $lock_refresh_time = time();
            
            add_action('lightsyncpro_sync_progress', function() use ($lock_key, $lock_duration, &$lock_refresh_time) {
                if (time() - $lock_refresh_time > 300) {
                    set_transient($lock_key, time(), $lock_duration);
                    $lock_refresh_time = time();
                }
            });
            
            // Get per-album destination settings
            $all_destinations = (array) self::get_opt('album_destinations', []);
            $album_dests = $all_destinations[$album_id] ?? ['wordpress']; // Default to WordPress only
            $album_name = self::get_album_name_cached((string)$cat, (string)$album_id);
            
            $should_sync_wp = in_array('wordpress', $album_dests, true);

            
            Logger::debug("[LSP album_cron] Album {$album_id} ({$album_name}) destinations: " . implode(', ', $album_dests));
            
            // Sync to WordPress if enabled
            if ($should_sync_wp) {
                \LightSyncPro\Sync\Sync::sync_album($cat, $album_id);
                
                self::add_activity(
                    sprintf('Lightroom → WordPress Sync Complete (Auto): "%s"', $album_name),
                    'success',
                    'auto'
                );
            }

            self::set_opt([
                'lightsync_last_cron_run'  => time(),
                'lightsync_last_sync_ts'   => time(),
                'lightsync_last_sync_type' => 'auto',
            ]);

            Logger::debug("[LSP] album cron ok ({$album_id})");
        } catch (\Throwable $e) {
            Logger::debug("[LSP] album cron error ({$album_id}): " . $e->getMessage());
        } finally {
            delete_transient($lock_key);
        }
    }

    private function normalize_cron_key($val){
        $allowed = ['wplr_15mins','wplr_hourly','wplr_twicedaily','wplr_daily'];
        return in_array($val, $allowed, true) ? $val : 'wplr_hourly';
    }

    public function ensure_cron_on_boot(){
        $cat    = self::get_opt('catalog_id','');
        $albums = (array) self::get_opt('album_ids', []);
        if ( $cat && !empty($albums) && ! $this->any_album_cron_scheduled() ) {
            $this->reschedule_album_crons();
        }

        if ($this->any_album_cron_scheduled()) {
            wp_clear_scheduled_hook(self::CRON);
            update_option('_lightsync_cron_cadence', 'off', false);
            return;
        }

        $desired = $this->normalize_cron_key( self::get_opt('cron_interval', 'wplr_hourly') );
        $next    = wp_next_scheduled(self::CRON);

        if (!$next){
            wp_schedule_event(time()+60, $desired, self::CRON);
        } else {
            $current = get_option('_lightsync_cron_cadence');
            if ($current !== $desired){
                wp_clear_scheduled_hook(self::CRON);
                wp_schedule_event(time()+60, $desired, self::CRON);
            }
        }
        update_option('_lightsync_cron_cadence', $desired, false);
    }

    public function ensure_rendition_cron_on_boot(){
        if ( ! wp_next_scheduled( self::CRON_RETRY_HOOK ) ) {
            wp_schedule_event( time() + 60, 'lightsync_every_2_minutes', self::CRON_RETRY_HOOK );
        }
    }

    public function activate(){
        self::verify_mapping_integrity();
    }
    
    /**
     * Verify and repair asset mapping integrity
     * 
     * Cleans up orphaned entries where:
     * - Asset map points to deleted attachments
     * 
     * This runs on activation to catch any issues from:
     * - Manual database changes
     * - Attachments deleted while plugin was inactive
     * - Migration issues
     */
    private static function verify_mapping_integrity(): void {
        global $wpdb;
        
        $cleaned_asset_map = 0;

        
        // 1. Clean orphaned entries in lightsync_asset_map (Lightroom)
        $map = get_option('lightsync_asset_map', []);
        if (!empty($map) && is_array($map)) {
            foreach ($map as $asset_id => $entry) {
                $att_id = is_array($entry) ? (int)($entry['attachment_id'] ?? 0) : (int)$entry;
                
                // Check if attachment still exists
                if ($att_id && !get_post($att_id)) {
                    unset($map[$asset_id]);
                    $cleaned_asset_map++;
                }
            }
            
            if ($cleaned_asset_map > 0) {
                update_option('lightsync_asset_map', $map, false);
            }
        }
        
        // 3. Verify post meta mappings match asset_map (detect drift)
        // Get all attachments with _lightsync_asset_id
        $meta_assets = $wpdb->get_results(
            "SELECT post_id, meta_value as asset_id 
             FROM {$wpdb->postmeta} 
             WHERE meta_key = '_lightsync_asset_id' 
             AND meta_value != ''",
            ARRAY_A
        );
        
        $repaired_map = 0;
        foreach ($meta_assets as $row) {
            $post_id = (int)$row['post_id'];
            $asset_id = (string)$row['asset_id'];
            
            // Skip if attachment doesn't exist
            if (!get_post($post_id)) {
                continue;
            }
            
            // Check if asset_map has this entry
            if (!isset($map[$asset_id])) {
                // Post meta exists but asset_map doesn't - repair it
                $map[$asset_id] = $post_id;
                $repaired_map++;
            }
        }
        
        if ($repaired_map > 0) {
            update_option('lightsync_asset_map', $map, false);
        }
        
        // Log results
        $total_changes = $cleaned_asset_map + $repaired_map;
        if ($total_changes > 0 && defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug("[LSP] Mapping integrity check: cleaned {$cleaned_asset_map} orphaned, repaired {$repaired_map} missing from asset_map");
        }
    }
    
    public function deactivate() {
        $opts = self::get_opt();
        if ( ! empty( $opts['wipe_on_deactivate'] ) ) {
            self::delete_all_plugin_data();
        }
        wp_clear_scheduled_hook( self::CRON_RETRY_HOOK );
    }

    public static function cron_sync($force = false, $catalog_id = null, $album_id = null) {
        $force = (bool) $force;

        // Check if Lightroom auto-sync is globally enabled
        if ( ! (bool) \LightSyncPro\Admin\Admin::get_opt('lightroom_autosync_updates', true) ) {
            Logger::debug('[LSP] cron_sync bail: lightroom autosync disabled');
            return;
        }

        if ( (new self())->any_album_cron_scheduled() ) {
            Logger::debug('[LSP] admin cron_sync bail: per-album cron is active');
            return;
        }

        $cat    = \LightSyncPro\Admin\Admin::get_opt('catalog_id','');
        $albums = (array) \LightSyncPro\Admin\Admin::get_opt('album_ids', []);
        if (!$cat || empty($albums)) {
            Logger::debug('[LSP] admin cron bail: missing catalog or albums');
            return;
        }

        $sched = (array) \LightSyncPro\Admin\Admin::get_opt('album_schedule', []);
        $last  = (array) \LightSyncPro\Admin\Admin::get_opt('album_last_run', []);
        $now   = time();

        if (!\LightSyncPro\Admin\Admin::has_cap('multi_albums')) {
            $albums = array_slice($albums, 0, 1);
        }

        foreach ($albums as $album_id) {
            $album_id = (string) $album_id;

            $freq = $sched[$album_id] ?? 'off';
            $interval = ($freq==='15m') ? 900
                      : (($freq==='hourly') ? HOUR_IN_SECONDS
                      : (($freq==='twicedaily') ? 12 * HOUR_IN_SECONDS
                      : (($freq==='daily') ? DAY_IN_SECONDS : 0)));

            if ($interval === 0 && !$force) {
                Logger::debug("[LSP] admin cron skip {$album_id}: freq={$freq}");
                continue;
            }

            $lastRun = (int)($last[$album_id] ?? 0);
            $due     = $force || ($lastRun===0) || (($now - $lastRun) >= $interval);

            if (!$due) {
                Logger::debug("[LSP] admin cron not-due {$album_id}: last={$lastRun} interval={$interval}");
                continue;
            }

            $last[$album_id] = $now;
            \LightSyncPro\Admin\Admin::set_opt(['album_last_run' => $last]);

            try {
                \LightSyncPro\Sync\Sync::schedule_sync_tick((string)$cat, (string)$album_id, 'auto', 5);
                Logger::debug("[LSP] admin cron scheduled tick: {$album_id}");
            } catch (\Throwable $e) {
                Logger::debug("[LSP] admin cron schedule error {$album_id}: ".$e->getMessage());
            }
        }
    }

    public function register(){
        register_setting(self::OPT, self::OPT, [
            'sanitize_callback' => [__CLASS__, 'sanitize_options_merge']
        ]);
    }

    public static function sanitize_options_merge($incoming){
        if (!is_array($incoming)) $incoming = [];

        $existing = get_option(self::OPT, []);
        if (!is_array($existing)) $existing = [];

        $out = array_merge($existing, $incoming);
        if (!is_array($existing)) $existing = [];

        $out = array_merge($existing, $incoming);

        // Handle checkboxes - only update if a related field from the same form section is present
        // This prevents one form from resetting checkboxes in another form
        
        // AVIF form - only process if _avif_form marker is present
        if (!empty($incoming['_avif_form'])) {
            $out['avif_enable'] = !empty($incoming['avif_enable']) ? 1 : 0;
            // Sanitize avif_quality as integer 0-100
            if (isset($incoming['avif_quality'])) {
                $out['avif_quality'] = max(0, min(100, (int) $incoming['avif_quality']));
            }
        }
        
        // Main Lightroom form checkboxes - process if sync_target or catalog_id is present
        if (array_key_exists('sync_target', $incoming) || array_key_exists('catalog_id', $incoming)) {
            $out['keep_original_filename'] = !empty($incoming['keep_original_filename']) ? 1 : 0;
        }

        $must_preserve = ['broker_token_enc','license_ok','plan','license_caps','expires_at','album_destinations','album_schedule'];

        foreach ($must_preserve as $k) {
            if (!array_key_exists($k, $incoming) && array_key_exists($k, $existing)) {
                $out[$k] = $existing[$k];
            }
        }

        $out['avif_enable'] = 0; // AVIF not available in free version
        // Naming and alt text always available in free version

        if ( ! self::has_cap('multi_albums') ) {
            $albums = array_values((array)($out['album_ids'] ?? []));
            $out['album_ids'] = array_slice($albums, 0, 1);
        }

        return $out;
    }

    public function menu(){
        $brand = lsp_get_brand();
        
        // Use brand-appropriate icon
        $svg_path = $brand['is_enterprise'] 
            ? LIGHTSYNC_PRO_DIR . 'assets/syncific-icon.svg'
            : LIGHTSYNC_PRO_DIR . 'assets/lsp-icon.svg';
        
        $icon = 'dashicons-admin-generic';
        if ( file_exists($svg_path) ) {
            $svg  = preg_replace('/\s+/', ' ', file_get_contents($svg_path));
            $icon = 'data:image/svg+xml;base64,' . base64_encode($svg);
        }
        add_menu_page( $brand['name'], $brand['name'], 'manage_options', self::MENU, [ $this, 'render' ], $icon, 59);
    }

    public function assets($hook){
        $is_plugin_page = (strpos($hook, 'toplevel_page_' . self::MENU) !== false);
        $is_media_page  = ($hook === 'upload.php');

        if ($is_media_page) {
            $base = plugin_dir_url(dirname(__FILE__, 2));

            wp_enqueue_script(
                'lsp-media-relink',
                $base . 'assets/media-relink.js',
                ['jquery'],
                defined('LIGHTSYNC_VER') ? LIGHTSYNC_VER : '1.0.0',
                true
            );

            return;
        }

        if (!$is_plugin_page) return;

        add_action('admin_head', function(){
            echo '<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>';
        });

        wp_enqueue_style(
            'lightsyncpro-fonts',
            'https://fonts.googleapis.com/css2?family=Montserrat:wght@400;500;600;700&display=swap',
            array(),
            LIGHTSYNC_PRO_VERSION
        );

        // Cache-busting: use filemtime so browser always gets fresh assets after plugin update
        $css_ver = LIGHTSYNC_PRO_VERSION . '.' . @filemtime(LIGHTSYNC_PRO_DIR . 'assets/admin.css');
        $css_inline_ver = LIGHTSYNC_PRO_VERSION . '.' . @filemtime(LIGHTSYNC_PRO_DIR . 'assets/admin-inline.css');
        $js_ver = LIGHTSYNC_PRO_VERSION . '.' . @filemtime(LIGHTSYNC_PRO_DIR . 'assets/admin.js');
        $js_inline_ver = LIGHTSYNC_PRO_VERSION . '.' . @filemtime(LIGHTSYNC_PRO_DIR . 'assets/admin-inline.js');
        $js_sync_ver = LIGHTSYNC_PRO_VERSION . '.' . @filemtime(LIGHTSYNC_PRO_DIR . 'assets/admin-sync.js');

        wp_enqueue_style(
            'lightsyncpro-admin',
            LIGHTSYNC_PRO_URL.'assets/admin.css',
            ['lightsyncpro-fonts'],
            $css_ver
        );

        wp_enqueue_style(
            'lightsyncpro-admin-inline',
            LIGHTSYNC_PRO_URL.'assets/admin-inline.css',
            ['lightsyncpro-admin'],
            $css_inline_ver
        );

        wp_enqueue_script(
            'lightsyncpro-admin',
            LIGHTSYNC_PRO_URL.'assets/admin.js',
            ['jquery'],
            $js_ver,
            true
        );

        wp_enqueue_script(
            'lightsyncpro-admin-inline',
            LIGHTSYNC_PRO_URL.'assets/admin-inline.js',
            ['lightsyncpro-admin'],
            $js_inline_ver,
            true
        );

        wp_enqueue_script(
            'lightsyncpro-admin-sync',
            LIGHTSYNC_PRO_URL.'assets/admin-sync.js',
            ['lightsyncpro-admin-inline'],
            $js_sync_ver,
            true
        );

        // Enable WordPress media library modal for A/B testing image picker
        wp_enqueue_media();

        $o = self::get_opt();
        $user = wp_get_current_user();
        
        // Get license key from the License class (stored in lightsyncpro_license option)
        $license_key = '';
        // Fallback to settings for backwards compatibility
        if (!$license_key) {
            $license_key = (string)($o['license_key'] ?? '');
        }
        
        wp_localize_script('lightsyncpro-admin','LIGHTSYNCPRO',[
            'ajaxurl'       => admin_url('admin-ajax.php'),
            'adminurl'      => admin_url(),
            'nonce'         => wp_create_nonce(self::AJAX_NS.'_nonce'),
            'pluginUrl'     => plugins_url('', LIGHTSYNC_PRO_FILE),
            'broker_base'   => 'https://lightsyncpro.com',
            'broker_pickup' => 'https://lightsyncpro.com/wp-admin/admin-ajax.php?action=lsp_broker_install_pickup',
            'shopify_start' => 'https://lightsyncpro.com/wp-admin/admin-ajax.php?action=lsp_shopify_connect_start',
            'shopify_pickup'=> 'https://lightsyncpro.com/wp-admin/admin-ajax.php?action=lsp_shopify_install_pickup',
            'shopify_shop'  => (string)($o['shopify_shop_domain'] ?? ''),
            'shopify_connected' => !empty($o['shopify_shop_domain']) && (
                (!empty($o['shopify_access_token']) && is_string($o['shopify_access_token'])) ||
                (is_array($o['shopify_access_token'] ?? null) && !empty($o['shopify_access_token']))
            ),
            'openrouter_connected' => \LightSyncPro\OAuth\OpenRouterOAuth::is_connected(),
            'sync_target'   => (string)($o['sync_target'] ?? 'wp'),
            'license_key'   => '',
            'admin_email'   => $user ? (string) $user->user_email : '',
            'plan'          => 'free',
            'licensed'      => 0,
            'upsell'        => 'https://lightsyncpro.com/pricing',
            'saved'         => [
                'catalog'      => (string) self::get_opt('catalog_id',''),
                'albums'       => (array)  self::get_opt('album_ids',[]),
                'schedules'    => (array)  self::get_opt('album_schedule',[]),
                'destinations' => self::get_album_destinations_for_js(),
                'sync_status'  => self::get_album_sync_status(),
                'php_avif'     => (int)    self::get_opt('avif_enable', 1),
                'batch'        => (int)    self::get_opt('batch_size',200),
                'rend'         => (string) self::get_opt('rendition','2048'),
                'filter'       => (string) self::get_opt('name_filter',''),
            ],
            'celebrated_lightroom' => (int) get_option('lsp_celebrated_lightroom', 0),
            'celebrated_canva'     => (int) get_option('lsp_celebrated_canva', 0),
            'celebrated_dropbox'   => (int) get_option('lsp_celebrated_dropbox', 0),
            'celebrated_figma'     => (int) get_option('lsp_celebrated_figma', 0),
        ]);
    }

    /**
     * Handle OAuth callbacks early in admin_init before headers are sent
     */
    public function handle_oauth_callback() {
        // Only run on our admin page
        if (!isset($_GET['page']) || $_GET['page'] !== self::MENU) {
            return;
        }
        
        if (!current_user_can('manage_options')) {
            return;
        }

        // Handle Lightroom OAuth callback
        if ( isset( $_GET['lsp_connected'] ) && '1' === sanitize_text_field( wp_unslash( $_GET['lsp_connected'] ) ) && ! empty( $_GET['state'] ) ) {
            $state = sanitize_text_field( wp_unslash( $_GET['state'] ) );
            
            $process_key = 'lightsync_oauth_processed_' . md5($state);
            if ( get_transient($process_key) ) {
                \LightSyncPro\Util\Logger::debug('[LSP OAuth] Already processed, redirecting');
                wp_safe_redirect( admin_url('admin.php?page=' . self::MENU) );
                exit;
            }
            
            set_transient($process_key, 1, 300);
            
            \LightSyncPro\Util\Logger::debug('[LSP OAuth] Processing state: ' . $state);
            
            $pickup_url = 'https://lightsyncpro.com/wp-admin/admin-ajax.php?action=lsp_broker_install_pickup&state=' . rawurlencode($state);
            $resp = wp_remote_get($pickup_url, ['timeout'=>15]);
            
            if (is_wp_error($resp)) {
                \LightSyncPro\Util\Logger::debug('[LSP OAuth] Pickup failed: ' . $resp->get_error_message());
            } else {
                $body = wp_remote_retrieve_body($resp);
                $json = json_decode($body, true);
                \LightSyncPro\Util\Logger::debug('[LSP OAuth] Pickup response: ' . substr($body, 0, 200));
                
                if (!empty($json['success']) && !empty($json['data']['broker_token'])) {
                    $enc = \LightSyncPro\Util\Crypto::enc($json['data']['broker_token']);
                    \LightSyncPro\Util\Logger::debug('[LSP OAuth] Encrypted token: ' . ($enc ? 'yes' : 'no'));
                    
                    if ($enc) {
                        $settings = [
                            'broker_token_enc' => $enc,
                        ];

                        if (!empty($json['data']['client_id'])) {
                            $settings['client_id'] = sanitize_text_field($json['data']['client_id']);
                        }

                        self::set_opt($settings);
                        \LightSyncPro\Util\Logger::debug('[LSP OAuth] Saved broker_token_enc');
                        
                        // Verify it was saved
                        $verify = self::get_opt('broker_token_enc');
                        \LightSyncPro\Util\Logger::debug('[LSP OAuth] Verified saved: ' . ($verify ? 'yes' : 'no'));
                        
                        $this->activate_license_if_possible();
                    }
                } else {
                    \LightSyncPro\Util\Logger::debug('[LSP OAuth] No broker token in response');
                }
            }
            
            $redirect = admin_url('admin.php?page=' . self::MENU);
            \LightSyncPro\Util\Logger::debug('[LSP OAuth] Redirecting to: ' . $redirect);
            wp_safe_redirect($redirect);
            exit;
        }

        // Handle Shopify OAuth callback
        if ( isset( $_GET['lsp_shopify_connected'] ) && '1' === sanitize_text_field( wp_unslash( $_GET['lsp_shopify_connected'] ) ) && ! empty( $_GET['state'] ) ) {
            $state = sanitize_text_field( wp_unslash( $_GET['state'] ) );

            $process_key = 'lightsync_shopify_oauth_processed_' . md5($state);
            if ( get_transient($process_key) ) {
                wp_safe_redirect( admin_url('admin.php?page=' . self::MENU) );
                exit;
            }
            set_transient($process_key, 1, 300);

            $data = Shopify::pickup_install($state);
            if ( ! is_wp_error($data) ) {
                $new_shop = sanitize_text_field((string)($data['shop_domain'] ?? ''));
                $old_shop = (string)(self::get_opt('shopify_shop_domain') ?? '');

                // Store change tracking: log the switch but preserve old mappings
                // Mappings are scoped by shop_domain so old store's data stays intact for reconnect
                if ($old_shop !== '' && $new_shop !== '' && $old_shop !== $new_shop) {
                    self::add_activity(
                        sprintf('Shopify store changed from %s to %s — old store mappings preserved', $old_shop, $new_shop),
                        'info',
                        'shopify'
                    );
                }

                $settings = [
                    'shopify_connected'   => 1,
                    'shopify_shop_domain' => $new_shop,
                    'shopify_shop_id'     => sanitize_text_field((string)($data['shop_id'] ?? '')),
                ];
                self::set_opt($settings);
            }

            $redirect = admin_url('admin.php?page=' . self::MENU);
            wp_safe_redirect($redirect);
            exit;
        }

    }

    public function render(){
        if(!current_user_can('manage_options')) return;

        $o    = self::get_opt();
        $plan = strtolower( $o['plan'] ?? ($o['license_plan'] ?? 'free') );
        $brand = lsp_get_brand();
        
        // Show Enterprise when Hub is enabled
        $hub_enabled = function_exists('lsp_hub_enabled') && lsp_hub_enabled();
        $display_plan = $hub_enabled ? 'enterprise' : $plan;

        // Naming & alt always available in free version
        
        // Usage from separate option
        $caps   = self::usage_caps();
        $usage  = self::usage_get();
        $used   = (int) $usage['month_count'];
        $cap    = (int) $caps['month']; 
        $photos_label = $cap ? sprintf('%s/%s', number_format_i18n($used), number_format_i18n($cap))
                             : sprintf('%s/∞', number_format_i18n($used));
        $photos_pct   = $cap ? max(0, min(100, (int) round(($used / $cap) * 100))) : 0;

        $albums = (array) self::get_opt('album_ids', []);
        $nexts  = [];
        foreach ($albums as $aid) {
            $ts = wp_next_scheduled(self::CRON_ALBUM, [(string)$aid]);
            if ($ts) {
               $nexts[] = wp_date(get_option('date_format').' '.get_option('time_format'), $ts).' (Album '.$aid.')';
            }
        }

        $active    = !empty($nexts);
        // Auto-sync badge removed in free version
        
        $keep_original = !empty($o['keep_original_filename']);
        ?>
     <div class="wrap lightsyncpro">

      <style>
      .wrap.lightsyncpro .lsp-layout{display:grid;grid-template-columns:260px minmax(0,1fr);gap:24px;margin-top:12px}
      .wrap.lightsyncpro .lsp-side{position:sticky;top:32px;align-self:start;height:max-content}
      .wrap.lightsyncpro .lsp-main{min-width:0}
      .wrap.lightsyncpro .lsp-nav{list-style:none;margin:0;padding:0;display:grid;gap:2px}
      .wrap.lightsyncpro .lsp-nav a{display:flex;align-items:center;gap:8px;padding:10px 12px;border-radius:8px;background:transparent;border:none;text-decoration:none;color:var(--lsp-subtext);font-weight:500;transition:all 0.15s ease;position:relative}
      .wrap.lightsyncpro .lsp-nav a:hover{color:var(--lsp-text);background:rgba(0,0,0,0.03)}
      .wrap.lightsyncpro .lsp-nav a.active{color:var(--lsp-text);font-weight:600;background:transparent}
      .wrap.lightsyncpro .lsp-nav a.active::after{content:'';position:absolute;bottom:6px;left:12px;right:12px;height:2px;background:linear-gradient(135deg, var(--lsp-primary), var(--lsp-primary-600));border-radius:2px}
      .wrap.lightsyncpro .lsp-nav a.lsp-kb-link{color:#0369a1;margin-top:12px;padding-top:14px;border-top:1px solid var(--lsp-border)}
      .wrap.lightsyncpro .lsp-nav a.lsp-kb-link:hover{background:rgba(14,165,233,0.08)}
      .wrap.lightsyncpro .lsp-nav a.lsp-kb-link .dashicons{font-size:16px;width:16px;height:16px;margin-right:4px}
      .wrap.lightsyncpro .lsp-grid-cards{display:grid;grid-template-columns:repeat(auto-fit,minmax(520px,1fr));gap:18px}
      .wrap.lightsyncpro .lsp-card{border-radius:18px;}
      .wrap.lightsyncpro .lsp-card-header{padding:12px 14px; background: none}
      .wrap.lightsyncpro .lsp-card-body{padding:14px}
      .wrap.lightsyncpro .form-table th,.wrap.lightsyncpro .form-table td{padding:8px}
      .wrap.lightsyncpro .sub-card{background:var(--lsp-card);padding:0;margin:0}
      .wrap.lightsyncpro .lsp-grid .sub-card label{display:inline-block;margin-bottom:6px}
      .wrap.lightsyncpro #lsp-albums{margin-top:6px;}
      .wrap.lightsyncpro .lsp-kpis{display:flex;gap:12px;margin-top:8px;flex-wrap:wrap}
      .wrap.lightsyncpro .lsp-kpi{background:rgba(0,0,0,0.03);border:none;border-radius:10px;padding:8px 12px;display:flex;gap:8px}
      .wrap.lightsyncpro .lsp-kpi .label{color:var(--lsp-subtext);font-weight:600}
      .wrap.lightsyncpro .lsp-kpi .value{font-weight:700}
      @media (max-width:1024px){
        .wrap.lightsyncpro .lsp-layout{grid-template-columns:1fr}
        .wrap.lightsyncpro .lsp-side{display:none}
        .wrap.lightsyncpro .lsp-grid-cards{grid-template-columns:1fr}
        .wrap.lightsyncpro main{padding-bottom:calc(80px + env(safe-area-inset-bottom, 0px))}
      }
      /* Mobile Bottom Nav */
      .lsp-mobile-nav{display:none;position:fixed;bottom:0;left:0;right:0;background:rgba(255,255,255,0.95);backdrop-filter:blur(12px);border-top:1px solid rgba(0,0,0,0.06);padding:8px 12px calc(8px + env(safe-area-inset-bottom, 0px));z-index:9999;justify-content:space-around;align-items:center}
      @media (max-width:1024px){.lsp-mobile-nav{display:flex}}
      .lsp-mobile-nav-item{display:flex;flex-direction:column;align-items:center;gap:4px;padding:6px 12px;background:none;border:none;color:var(--lsp-subtext);font-size:10px;font-weight:600;text-decoration:none;cursor:pointer;border-radius:8px;transition:all 0.15s ease;position:relative}
      .lsp-mobile-nav-item svg{transition:transform 0.15s ease}
      .lsp-mobile-nav-item:hover,.lsp-mobile-nav-item.active{color:var(--lsp-primary)}
      .lsp-mobile-nav-item.active{background:rgba(255,87,87,0.08)}
      .lsp-mobile-nav-item.active svg{transform:scale(1.1)}
      .lsp-mobile-more-menu{display:none;position:absolute;bottom:100%;right:8px;margin-bottom:8px;background:#fff;border-radius:12px;box-shadow:0 8px 30px rgba(0,0,0,0.15);padding:8px 0;min-width:180px}
      .lsp-mobile-more-menu.open{display:block;animation:lsp-fade-up 0.15s ease}
      @keyframes lsp-fade-up{from{opacity:0;transform:translateY(8px)}to{opacity:1;transform:translateY(0)}}
      .lsp-mobile-more-menu a{display:flex;align-items:center;gap:10px;padding:10px 16px;color:var(--lsp-text);text-decoration:none;font-size:13px;font-weight:500;transition:background 0.1s}
      .lsp-mobile-more-menu a:hover{background:rgba(0,0,0,0.04)}
      .lsp-mobile-more-menu a svg{color:var(--lsp-subtext)}
      html{scroll-behavior:smooth}
      :root{
      --lsp-primary:#ff5757;
      --lsp-primary-600:#2563eb;
      --lsp-bg:#F4F7FF;
      --lsp-card:#ffffff;
      --lsp-text:#0f172a;
      --lsp-subtext:#64748b;
      --lsp-border:#e5e7eb;
      --lsp-ring: linear-gradient(135deg, var(--lsp-primary), var(--lsp-primary-600));
      --shadow: 0 10px 24px rgba(15,23,42,.08);
      --shadow-soft: 0 6px 18px rgba(15,23,42,.06);
      --radius:18px;
      --lsp-ring-grad: linear-gradient(135deg, var(--lsp-primary), var(--lsp-primary-600));
    }
    *{box-sizing:border-box}

/* Helper sidebars toggle */
.wrap.lightsyncpro.lsp-helpers-hidden aside.help {
  display: none !important;
}
.wrap.lightsyncpro.lsp-helpers-hidden .twocol {
  grid-template-columns: 1fr !important;
}

/* Guided Tour Styles */
.lsp-tour-overlay {
  position: fixed;
  inset: 0;
  background: rgba(15,23,42,0.6);
  backdrop-filter: blur(4px);
  z-index: 100000;
  pointer-events: none;
}
.lsp-tour-highlight {
  position: relative;
  z-index: 100001;
  box-shadow: 0 0 0 4px rgba(37,99,235,0.5), 0 0 0 9999px rgba(15,23,42,0.6);
  border-radius: 12px;
  pointer-events: auto;
}
.lsp-tour-popup {
  position: fixed;
  z-index: 100002;
  background: #fff;
  border-radius: 16px;
  box-shadow: 0 25px 60px rgba(0,0,0,0.25);
  padding: 20px 24px;
  max-width: 400px;
  width: 90%;
  left: 50%;
  top: 50%;
  transform: translate(-50%, -50%);
  animation: lspTourFadeIn 0.2s ease;
}
@keyframes lspTourFadeIn {
  from { opacity: 0; transform: translateY(8px); }
  to { opacity: 1; transform: translateY(0); }
}
.lsp-tour-popup h4 {
  margin: 0 0 8px;
  font-size: 16px;
  font-weight: 700;
  color: #1e293b;
  display: flex;
  align-items: center;
  gap: 8px;
}
.lsp-tour-popup h4 .step-badge {
  background: linear-gradient(135deg, #ff5757, #2563eb);
  color: #fff;
  font-size: 11px;
  padding: 2px 8px;
  border-radius: 10px;
  font-weight: 600;
}
.lsp-tour-popup p {
  margin: 0 0 16px;
  font-size: 14px;
  color: #64748b;
  line-height: 1.5;
}
.lsp-tour-actions {
  display: flex;
  gap: 8px;
  justify-content: flex-end;
}
.lsp-tour-actions button {
  padding: 8px 16px;
  border-radius: 8px;
  font-size: 13px;
  font-weight: 600;
  cursor: pointer;
  transition: all 0.15s;
}
.lsp-tour-actions .lsp-tour-skip {
  background: rgba(0,0,0,0.04);
  border: none;
  color: #64748b;
}
.lsp-tour-actions .lsp-tour-skip:hover {
  background: rgba(0,0,0,0.08);
}
.lsp-tour-actions .lsp-tour-prev {
  background: #f1f5f9;
  border: none;
  color: #475569;
}
.lsp-tour-actions .lsp-tour-prev:hover {
  background: #e2e8f0;
}
.lsp-tour-actions .lsp-tour-next {
  background: linear-gradient(135deg, #ff5757, #2563eb);
  border: none;
  color: #fff;
}
.lsp-tour-actions .lsp-tour-next:hover {
  opacity: 0.9;
}
.lsp-tour-progress {
  display: flex;
  gap: 4px;
  margin-bottom: 12px;
}
.lsp-tour-progress span {
  width: 24px;
  height: 3px;
  background: #e2e8f0;
  border-radius: 2px;
  transition: background 0.2s;
}
.lsp-tour-progress span.active {
  background: linear-gradient(135deg, #ff5757, #2563eb);
}
.lsp-tour-progress span.done {
  background: #10b981;
}

.lsp-modal { position: fixed; inset: 0; z-index: 10050 !important; display:none; }
.lsp-modal[aria-hidden="false"] { display:block; }
.lsp-modal-backdrop {
  position:absolute; inset:0;
  background: rgba(15,23,42,.55);
  backdrop-filter: blur(6px);
}
.lsp-modal-card{
  position:absolute;
  left:50%; top:18%;
  transform: translateX(-50%);
  width: min(560px, calc(100% - 28px));
  background:#fff;
  border: none;
  border-radius: 18px;
  box-shadow: 0 30px 80px rgba(15,23,42,.35);
  padding: 16px;
}
.lsp-modal-head{ display:flex; gap:10px; align-items:center; margin-bottom:10px; }
.lsp-modal-head h3{ margin:0; font-size:15px; font-weight:900; }
.lsp-modal-icon{
  width:34px; height:34px; border-radius: 12px;
  display:grid; place-items:center;
  background: rgba(245,158,11,.14);
  border: none;
  color:#92400e;
  font-weight:900;
}
.lsp-modal-body{ color: var(--lsp-subtext,#64748b); font-size:13px; line-height:1.5; }
.lsp-modal-actions{ display:flex; gap:10px; justify-content:flex-end; margin-top:14px; }

      </style>

      <script>
      document.addEventListener('DOMContentLoaded', function(){
        const links = Array.from(document.querySelectorAll('.lsp-side .lsp-nav a'));
        if (!links.length) return;
        const targets = links.map(a => { try { return document.querySelector(a.getAttribute('href')); } catch(e){ return null; } }).filter(Boolean);
        function onScroll(){
          const fromTop = window.scrollY + 120;
          let active = null;
          for (const t of targets){ if (t.offsetTop <= fromTop) active = t.id; }
          links.forEach(a => a.classList.toggle('active', a.getAttribute('href') === '#' + active));
        }
        window.addEventListener('scroll', onScroll, {passive:true});
        onScroll();
        links.forEach(a => a.addEventListener('click', () => {
          links.forEach(x => x.classList.remove('active'));
          a.classList.add('active');
        }));
      });
      </script>


      <header class="hero" id="top">
        <div class="hero-inner">
        <div class="logo">
             <h1 style="margin:.2em 0"><?php echo $brand['is_enterprise'] ? 'Media distribution infrastructure for your network' : 'Connect once, sync anytime → WordPress + Shopify'; ?></h1>
        </div>
          <div class="kpis">
            <div class="kpi"><div class="label">Status</div><div class="value"><?php echo !empty($o['broker_token_enc']) ? 'Connected' : 'Offline'; ?></div></div>

            <div class="kpi" id="lsp-kpi-albums">
              <div class="label" id="lsp-kpi-albums-label">Albums</div>
              <div class="value" id="lsp-kpi-albums-value"><?php echo count((array)($o['album_ids'] ?? [])); ?></div>
            </div>
           <?php
$last_ts = (int) self::get_opt('lightsync_last_sync_ts', 0);
if ($last_ts > 0) {
   $ago = human_time_diff($last_ts, time());
    echo '<div class="kpi"><div class="label">Last sync</div><div class="value" id="lsp-last-sync">'
       . esc_html($ago . ' ago')
       . '</div></div>';
} else {
    echo '<div class="kpi"><div class="label">Last sync</div><div class="value" id="lsp-last-sync">Never</div></div>';
}
?>
          </div>
        </div>
      </header>

      <div class="commandbar" role="region" aria-label="Command Bar">
      <div class="commandbar-left">
      <?php if ( $brand['is_enterprise'] ): ?>
      <a href="<?php echo esc_url( admin_url('admin.php?page=syncific-hub') ); ?>" class="syncific-badge" title="Syncific Hub - Enterprise Distribution">
        <img src="<?php echo esc_url( $brand['syncific_color'] ); ?>" alt="Syncific" height="20" />
      </a>
      <?php endif; ?>
      <div class="pillrow" style="align-items:center;">
        <?php if ( ! $brand['is_enterprise'] ): ?>
        <img src="<?php echo esc_url( $brand['logo_dark'] ); ?>" alt="LightSync Pro" style="height:22px;margin-right:8px;" />
        <?php endif; ?>
        <span class="pill" id="lsp-photos-pill"><strong><?php echo esc_html( $photos_label ); ?></strong> images</span>
      </div>
      </div>
      <div class="pillrow">      
        <button class="btn ghost small" id="lsp-estimate" type="button">
          Estimate <span id="lsp-estimate-out" class="status" style="display:none;"></span> 
        </button>
        <?php if (!empty($o['ai_provider'])): ?>
        <?php endif; ?>
        <button class="btn primary" id="lsp-start" type="button">
          Sync Now
        </button>
      </div>
    </div>

      <div class="lsp-layout">
        <aside class="lsp-side">
          <ul class="lsp-nav" id="nav">
            <li><a href="#lsp-sources">Select Source</a></li>
            <li id="lsp-nav-pick"><a href="#lsp-pick"><?php echo !empty($o['broker_token_enc']) ? 'Select Albums' : 'Lightroom'; ?></a></li>
            <?php if (!empty($o['broker_token_enc'])): ?>
            <li><a href="#lsp-naming">Naming &amp; Alt</a></li>
            <?php endif; ?>

            <li><a href="#lsp-destinations">Sync Destinations</a></li>
            <li><a href="#lsp-activity">Activity</a></li>

            <li><a href="<?php echo esc_url($brand['docs_url']); ?>" target="_blank" class="lsp-kb-link">
              <span class="dashicons dashicons-book-alt"></span> Knowledgebase
            </a></li>
            <li>  <?php if ($plan === 'free'): ?>
            <p><a class="button button-primary" href="https://lightsyncpro.com/pricing" target="_blank">🎉 Upgrade to Pro</a></p>
          <?php elseif ($plan === 'pro'): ?>
            <p><a class="button button-primary" href="https://billing.stripe.com/p/login/6oUfZa6ha9qsdIw2Nucs800" target="_blank">🎉 Upgrade to Agency</a></p>
          <?php endif; ?>
          <?php // No upgrade button for agency or enterprise ?>
          </li>
          </ul>
        </aside>

        <!-- Mobile Bottom Nav -->
        <nav class="lsp-mobile-nav" id="lsp-mobile-nav">
          <a href="#lsp-sources" class="lsp-mobile-nav-item active" data-section="sources">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <path d="M13 2L3 14h9l-1 8 10-12h-9l1-8z"/>
            </svg>
            <span>Source</span>
          </a>
          <a href="#lsp-pick" class="lsp-mobile-nav-item" data-section="pick" id="lsp-mobile-nav-pick">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <path d="M22 19a2 2 0 01-2 2H4a2 2 0 01-2-2V5a2 2 0 012-2h5l2 3h9a2 2 0 012 2z"/>
            </svg>
            <span class="lsp-mobile-nav-pick-label">Albums</span>
          </a>
          <a href="#lsp-activity" class="lsp-mobile-nav-item" data-section="settings">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <path d="M22 12h-4l-3 9L9 3l-3 9H2"/>
            </svg>
            <span>Activity</span>
          </a>
          <button type="button" class="lsp-mobile-nav-item lsp-mobile-nav-more" id="lsp-mobile-more-btn">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <circle cx="12" cy="12" r="1"/><circle cx="12" cy="5" r="1"/><circle cx="12" cy="19" r="1"/>
            </svg>
            <span>More</span>
          </button>
          <!-- More menu popup -->
          <div class="lsp-mobile-more-menu" id="lsp-mobile-more-menu">
            <a href="#lsp-activity">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="22 12 18 12 15 21 9 3 6 12 2 12"/></svg>
              Activity
            </a>
            <a href="https://lightsyncpro.com/docs/" target="_blank">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M4 19.5A2.5 2.5 0 016.5 17H20"/><path d="M6.5 2H20v20H6.5A2.5 2.5 0 014 19.5v-15A2.5 2.5 0 016.5 2z"/></svg>
              Knowledgebase
            </a>
          </div>
        </nav>

        <main>
            <div id="lsp-toast" class="lsp-toast" aria-live="polite" aria-atomic="true"></div>

<div id="lsp-modal" class="lsp-modal" aria-hidden="true" role="dialog" aria-modal="true">
  <div class="lsp-modal-backdrop" data-lsp-close></div>
  <div class="lsp-modal-card" role="document" aria-labelledby="lsp-modal-title">
    <div class="lsp-modal-head">
      <div class="lsp-modal-icon" id="lsp-modal-icon"></div>
      <h3 id="lsp-modal-title">Notice</h3>
    </div>
    <div class="lsp-modal-body" id="lsp-modal-body"></div>
    <div class="lsp-modal-actions">
      <button type="button" class="btn ghost" data-lsp-close>Close</button>
      <a href="#lsp-pick" class="btn primary" id="lsp-modal-primary" style="display:none;">Select Albums</a>
    </div>
  </div>
</div>


          <div>
            
            <!-- Helper Toggle & Tour Button -->
            <?php $helpers_hidden = get_user_meta(get_current_user_id(), 'lsp_helpers_hidden', true); ?>
            <style>
              #lsp-toolbar button:hover { color: var(--lsp-text) !important; }
              #lsp-toolbar button:hover svg { opacity: 1 !important; }
            </style>
            <div id="lsp-toolbar" style="display:flex;justify-content:flex-end;gap:16px;margin-bottom:12px;font-size:13px;">
              <button type="button" id="lsp-tour-btn" style="background:none;border:none;color:var(--lsp-subtext);cursor:pointer;display:inline-flex;align-items:center;gap:5px;padding:0;transition:color 0.15s;">
                <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="opacity:0.7;transition:opacity 0.15s;">
                  <circle cx="12" cy="12" r="10"/>
                  <path d="M9.09 9a3 3 0 015.83 1c0 2-3 3-3 3"/>
                  <path d="M12 17h.01"/>
                </svg>
                Take a Tour
              </button>
              <button type="button" id="lsp-helpers-toggle" style="background:none;border:none;color:var(--lsp-subtext);cursor:pointer;display:inline-flex;align-items:center;gap:5px;padding:0;transition:color 0.15s;" data-hidden="<?php echo $helpers_hidden ? '1' : '0'; ?>">
                <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="opacity:0.7;transition:opacity 0.15s;">
                  <circle cx="12" cy="12" r="10"/>
                  <path d="M12 16v-4"/>
                  <path d="M12 8h.01"/>
                </svg>
                <span class="lsp-helpers-label"><?php echo $helpers_hidden ? 'Show Helpers' : 'Hide Helpers'; ?></span>
              </button>
            </div>

            <form method="post" action="options.php" class="lsp-card-group">
              <?php settings_fields(self::OPT); ?>

            <!-- ====== SOURCE TABS ====== -->
            <section id="lsp-sources" class="section">
              <div class="section-head">
                <h3 class="lsp-card-title">Select Source</h3>
              </div>
              <div class="lsp-source-tabs">
                <?php 
                $lightroom_connected = \LightSyncPro\OAuth\OAuth::is_connected();
                $canva_connected = \LightSyncPro\OAuth\CanvaOAuth::is_connected();
                $figma_connected = \LightSyncPro\OAuth\FigmaOAuth::is_connected();
                $dropbox_connected = \LightSyncPro\OAuth\DropboxOAuth::is_connected();
                $openrouter_connected = \LightSyncPro\OAuth\OpenRouterOAuth::is_connected();
                $shutterstock_connected = \LightSyncPro\OAuth\ShutterstockOAuth::is_connected();
                $ai_connected = $openrouter_connected;

                // Build source list with connection status for dynamic ordering
                $sources = [
                    'lightroom' => [
                        'name' => 'Lightroom',
                        'connected' => $lightroom_connected,
                        'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="18" height="18" rx="2"/><circle cx="8.5" cy="8.5" r="1.5"/><path d="M21 15l-5-5L5 21"/></svg>',
                    ],
                    'canva' => [
                        'name' => 'Canva',
                        'connected' => $canva_connected,
                        'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 2L2 7l10 5 10-5-10-5z"/><path d="M2 17l10 5 10-5"/><path d="M2 12l10 5 10-5"/></svg>',
                    ],
                    'figma' => [
                        'name' => 'Figma',
                        'connected' => $figma_connected,
                        'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M5 5.5A3.5 3.5 0 0 1 8.5 2H12v7H8.5A3.5 3.5 0 0 1 5 5.5z"/><path d="M12 2h3.5a3.5 3.5 0 1 1 0 7H12V2z"/><path d="M12 12.5a3.5 3.5 0 1 1 7 0 3.5 3.5 0 1 1-7 0z"/><path d="M5 19.5A3.5 3.5 0 0 1 8.5 16H12v3.5a3.5 3.5 0 1 1-7 0z"/><path d="M5 12.5A3.5 3.5 0 0 1 8.5 9H12v7H8.5A3.5 3.5 0 0 1 5 12.5z"/></svg>',
                    ],
                    'dropbox' => [
                        'name' => 'Dropbox',
                        'connected' => $dropbox_connected,
                        'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 2L6 6l6 4-6 4 6 4 6-4-6-4 6-4-6-4z"/><path d="M6 14l6 4 6-4"/></svg>',
                    ],
                    'shutterstock' => [
                        'name' => 'Shutterstock',
                        'connected' => $shutterstock_connected,
                        'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="18" height="18" rx="2"/><path d="M9 9h6v6H9z"/><path d="M9 3v6"/><path d="M15 15v6"/></svg>',
                    ],
                    'ai' => [
                        'name' => 'AI Generate',
                        'connected' => $ai_connected,
                        'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 2a4 4 0 0 1 4 4v1a1 1 0 0 0 1 1h1a4 4 0 0 1 0 8h-1a1 1 0 0 0-1 1v1a4 4 0 0 1-8 0v-1a1 1 0 0 0-1-1H6a4 4 0 0 1 0-8h1a1 1 0 0 0 1-1V6a4 4 0 0 1 4-4z"/><circle cx="12" cy="12" r="2"/></svg>',
                    ],
                ];

                // Sort: connected first, then disconnected (preserve relative order within each group)
                $connected_sources = array_filter($sources, fn($s) => $s['connected']);
                $disconnected_sources = array_filter($sources, fn($s) => !$s['connected']);
                $sorted_sources = $connected_sources + $disconnected_sources;

                // Default active source: URL param > first connected > first in list
                $active_source = isset($_GET['source']) ? sanitize_text_field($_GET['source']) : '';
                if (!$active_source || !isset($sorted_sources[$active_source])) {
                    // Pick first connected, or first overall
                    $active_source = !empty($connected_sources) ? array_key_first($connected_sources) : array_key_first($sorted_sources);
                }

                // Render sorted tabs
                foreach ($sorted_sources as $source_key => $source) :
                ?>
                <button type="button" class="lsp-source-tab <?php echo $active_source === $source_key ? 'active' : ''; ?>" data-source="<?php echo esc_attr($source_key); ?>">
                  <span class="lsp-source-icon">
                    <?php echo $source['icon']; ?>
                  </span>
                  <span class="lsp-source-name"><?php echo esc_html($source['name']); ?></span>
                  <span class="lsp-source-status <?php echo $source['connected'] ? 'connected' : ''; ?>">
                    <?php echo $source['connected'] ? '✓ Connected' : 'Not connected'; ?>
                  </span>
                </button>
                <?php endforeach; ?>
              </div>
            </section>

            <!-- ====== LIGHTROOM CONTENT ====== -->
            <div id="lsp-lightroom-content" class="lsp-source-content <?php echo $active_source === 'lightroom' ? 'active' : ''; ?>">

            <?php if (!empty($o['broker_token_enc'])): ?>
            <!-- ====== CONNECTED: Show Album Picker ====== -->
            <section id="lsp-pick" class="section">
              <div class="section-head">
                <h3 class="lsp-card-title" id="lsp-pick-title">Select Albums</h3> 
                <span class="badge muted" id="lsp-pick-count">Albums: <?php echo count((array)($o['album_ids'] ?? [])); ?></span>
              </div>
              
              <div class="twocol">
                <div class="panel">
                  <div class="lsp-card-body">
              
                    <!-- Connected Banner -->
                    <div style="display:flex;align-items:center;gap:12px;margin-bottom:20px;padding:16px;background:rgba(239,68,68,0.08);border-radius:12px;">
                      <div style="width:40px;height:40px;background:#dc2626;border-radius:10px;display:flex;align-items:center;justify-content:center;">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2">
                          <path d="M20 6L9 17l-5-5"/>
                        </svg>
                      </div>
                      <div style="flex:1;">
                        <div style="font-weight:600;color:#991b1b;">Connected to Lightroom</div>
                        <div style="font-size:13px;color:#dc2626;">Ready to sync albums</div>
                      </div>
                    </div>
               
                    <div id="lsp-auth-banner" class="notice notice-warning" style="display:none;margin-bottom:16px;">
                      <p class="lsp-auth-message"></p>
                      <p>
                        <a href="<?php echo esc_url($this->auth_url()); ?>" class="btn primary">
                          Reconnect to Adobe
                        </a>
                      </p>
                    </div>

                    <div id="lsp-progress" class="lsp-progress is-circle" style="display:none;">
                      <div class="lsp-ring" role="progressbar" aria-valuemin="0" aria-valuemax="100" aria-valuenow="0"></div>
                      <div class="lsp-center">
                        <div class="lsp-percent">0%</div>
                        <div class="lsp-sub">You can close this window but keep this tab open while the sync runs.</div>
                        <div class="lsp-indeterminate">
                          <img src="<?php echo esc_url( LIGHTSYNC_PRO_URL . 'assets/lsp-cloud-sync.svg' ); ?>" alt="Syncing…" width="50" height="50" decoding="async" />
                        </div>
                      </div>
                      <div id="lsp-log" class="lsp-log"></div>
                    </div>

                    <!-- Album Selection -->
                    <div style="margin-bottom:20px;">
                      <div class="sub-card cat-hidden" style="max-width:320px;margin-bottom:16px;">
                        <label>Catalog</label>
                        <select id="lsp-catalog"></select>
                      </div>
                      
                      <label style="display:block;font-weight:600;color:#374151;margin-bottom:12px;">Albums</label>
                      
                      <!-- Hidden select to store values (for form submission compatibility) -->
                      <select id="lsp-albums" multiple style="display:none;"></select>
                      
                      <!-- Album Cards Grid -->
                      <div id="lsp-albums-grid" style="display:grid;grid-template-columns:repeat(auto-fill, minmax(160px, 1fr));gap:12px;min-height:100px;">
                        <div style="grid-column:1/-1;padding:32px;text-align:center;color:#6b7280;font-size:13px;">
                          Loading albums...
                        </div>
                      </div>
                    </div>

                    <!-- Sync Destination -->
                    <hr style="margin:20px 0;opacity:.25">
                    <label style="display:block;margin:0 0 10px;"><strong>Sync Destination</strong></label>
                    <?php 
                    $sync_target = (string)($o['sync_target'] ?? 'wp'); 
                    $shopify_connected = (bool)self::get_opt('shopify_connected');
                    ?>
                    <div class="lsp-dest-cards" style="display:grid;grid-template-columns:repeat(auto-fit, minmax(100px, 1fr));gap:12px;margin-bottom:16px;max-width:560px;">
                      <label class="lsp-dest-card <?php echo $sync_target === 'wp' ? 'selected' : ''; ?>">
                        <input type="radio" name="<?php echo esc_attr(self::OPT); ?>[sync_target]" value="wp" <?php checked($sync_target,'wp'); ?> style="display:none;" />
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                          <rect x="3" y="3" width="7" height="7" rx="1"/>
                          <rect x="14" y="3" width="7" height="7" rx="1"/>
                          <rect x="3" y="14" width="7" height="7" rx="1"/>
                          <rect x="14" y="14" width="7" height="7" rx="1"/>
                        </svg>
                        <span class="lsp-dest-name">WordPress</span>
                        <span class="lsp-dest-sub">Media Library</span>
                        <span class="lsp-dest-check">
                          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                        </span>
                      </label>
                      
                      <?php if ($shopify_connected): ?>
                      <label class="lsp-dest-card <?php echo $sync_target === 'shopify' ? 'selected' : ''; ?>">
                        <input type="radio" name="<?php echo esc_attr(self::OPT); ?>[sync_target]" value="shopify" <?php checked($sync_target,'shopify'); ?> style="display:none;" />
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                          <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                          <line x1="3" y1="6" x2="21" y2="6"/>
                          <path d="M16 10a4 4 0 01-8 0"/>
                        </svg>
                        <span class="lsp-dest-name">Shopify</span>
                        <span class="lsp-dest-sub">Files</span>
                        <span class="lsp-dest-check">
                          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                        </span>
                      </label>
                      
                      <label class="lsp-dest-card <?php echo $sync_target === 'both' ? 'selected' : ''; ?>">
                        <input type="radio" name="<?php echo esc_attr(self::OPT); ?>[sync_target]" value="both" <?php checked($sync_target,'both'); ?> style="display:none;" />
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                          <circle cx="6" cy="6" r="3"/>
                          <circle cx="18" cy="6" r="3"/>
                          <circle cx="6" cy="18" r="3"/>
                          <circle cx="18" cy="18" r="3"/>
                          <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                        </svg>
                        <span class="lsp-dest-name">Both</span>
                        <span class="lsp-dest-sub">WP + Shopify</span>
                        <span class="lsp-dest-check">
                          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                        </span>
                      </label>
                      <?php else: ?>
                      <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:pointer;" id="lsp-dest-shopify-connect">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                          <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                          <line x1="3" y1="6" x2="21" y2="6"/>
                          <path d="M16 10a4 4 0 01-8 0"/>
                        </svg>
                        <span class="lsp-dest-name">Shopify</span>
                        <span class="lsp-dest-sub">Click to connect</span>
                      </label>
                      
                      <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:not-allowed;">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                          <circle cx="6" cy="6" r="3"/>
                          <circle cx="18" cy="6" r="3"/>
                          <circle cx="6" cy="18" r="3"/>
                          <circle cx="18" cy="18" r="3"/>
                          <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                        </svg>
                        <span class="lsp-dest-name">Both</span>
                        <span class="lsp-dest-sub">Connect Shopify first</span>
                      </label>
                      <?php endif; ?>
                      
                    </div>


                    <p style="margin-top:12px;">
                      <button class="btn ghost" id="lsp-refresh">Refresh Lists</button>
                    </p>

                  </div><!-- end lsp-card-body -->
                </div><!-- end panel -->
                <aside class="help">
                  <h3>Lightroom Albums</h3>
                  
                  
                  <p>Select the Lightroom albums you want <?php echo esc_html( $brand['name'] ); ?> to watch. Only selected albums are included in estimates, syncs, and automation.</p>
                  
                  <p style="margin-top:12px;"><strong>Update Detection:</strong></p>
                  <ul style="margin:6px 0 12px 16px;padding:0;font-size:13px;color:#64748b;">
                    <li style="margin-bottom:4px;">Tracks Adobe's asset revision timestamps</li>
                    <li style="margin-bottom:4px;">Detects edits, crops, and metadata changes</li>

                  </ul>
                  
                  <p><strong>Tip:</strong> Some images may still be rendering in Lightroom. Re-sync once they're ready in Lightroom.</p>
                  <p><a href="<?php echo esc_url( $brand['docs_url'] ); ?>how-to-sync-lightroom-albums/" target="_blank">Album selection guide →</a></p>
                </aside>
              </div><!-- end twocol -->
            </section>

<?php else: ?>
            <!-- ====== NOT CONNECTED: Show Connect Prompt ====== -->
            <section id="lsp-pick" class="section">
              <div class="section-head">
                <h3 class="lsp-card-title" id="lsp-pick-title">Lightroom</h3> 
              </div>
              
              <div class="twocol">
                <div class="panel">
                  <div class="lsp-card-body">
                    <div class="lsp-lightroom-connect-prompt" style="text-align:center;padding:40px 20px;">
                      <h4 style="margin:0 0 8px;font-size:18px;">Connect to Lightroom</h4>
                      <p style="color:#64748b;margin:0 0 20px;">Sync your Adobe Lightroom photos directly to WordPress</p>
                      <a href="<?php echo esc_url($this->auth_url()); ?>" class="lsp-btn primary">
                        Connect with Adobe
                      </a>
                      <p style="font-size:11px;color:#94a3b8;margin:12px 0 0;">
                        Secure OAuth connection via Adobe
                      </p>
                    </div>
                  </div>
                </div>
                <aside class="help">
                  <h3>Connection & License</h3>
                  <p>
                    This connects your site to Adobe Lightroom through <?php echo esc_html( $brand['name'] ); ?>'s secure broker. Your photos stay in Lightroom — <?php echo esc_html( $brand['name'] ); ?> only syncs optimized copies to WordPress. If your connection ever expires, you can safely reconnect without affecting existing images.
                  </p>
                  
                  <p><a href="<?php echo esc_url( $brand['docs_url'] ); ?>how-to-connect-to-adobe-lightroom/" target="_blank">Connection guide →</a></p>
                </aside>
              </div>
            </section>
<?php endif; ?><!-- end if connected -->

<?php if (!empty($o['broker_token_enc'])): ?>

<?php
$mode_label = $keep_original ? 'Original' : 'Custom';
$mode_class = $keep_original ? 'is-original' : 'is-custom';
$name_pattern   = (string)($o['name_pattern']   ?? '{album}-{title}-{date}');
$alt_pattern    = (string)($o['alt_pattern']    ?? '{title} — {album}');
$title_source   = (string)($o['title_source']   ?? 'lr_title');
$caption_source = (string)($o['caption_source'] ?? 'lr_caption');
$tags_source    = (string)($o['tags_source']    ?? 'lr_keywords');
?>

              <section id="lsp-naming" class="section">
                <div class="section-head">
                  <h3 class="lsp-card-title">Naming & Alt Text</h3>
                  <span class="badge pro">Pro</span>
<span id="lsp-name-mode-badge" class="badge <?php echo esc_attr($mode_class); ?>">
  <?php echo esc_html($mode_label); ?>
</span>
                </div>
                <div class="twocol">
            <div class="panel">
                <div class="lsp-card-body" style="position:relative;">
                  <div>
                      <table class="form-table">
                        <tr>
                          <th scope="row">File name pattern</th>
                          <td>
                            <input type="text" class="regular-text code"
                                   name="<?php echo esc_html(self::OPT); ?>[name_pattern]"
                                   value="<?php echo esc_attr($name_pattern); ?>">
                            <p class="description" style="margin-top:6px">
                              Tokens: <code>{album}</code> <code>{title}</code> <code>{caption}</code> <code>{date}</code> <code>{sequence}</code> <code>{original_filename}</code> <code>{ext}</code>
                            </p>
                            <p style="margin:8px 0;">
                              <label class="lsp-toggle">
                                <input type="checkbox" id="lsp-keep-original"
                                       name="<?php echo esc_html(self::OPT); ?>[keep_original_filename]"
                                       value="1"
                                       <?php checked($keep_original); ?>>
                                <span class="track"><span class="thumb"></span></span>
                                <span>Keep original filename on disk (don't rename physically)</span>
                              </label>
                            </p>
                          </td>
                        </tr>
                        <tr>
                          <th scope="row">ALT text pattern</th>
                          <td>
                            <input type="text" class="regular-text code"
                                   name="<?php echo esc_html(self::OPT); ?>[alt_pattern]"
                                   value="<?php echo esc_attr($alt_pattern); ?>">
                            <p class="description" style="margin-top:6px">Example: <code>{title} — {album}</code></p>
                          </td>
                        </tr>
                        <tr>
                          <th scope="row">Content sources</th>
                          <td>
                            <div style="display:flex;gap:16px;align-items:center;flex-wrap:wrap">
                              <label>Title:
                                <select name="<?php echo esc_html(self::OPT); ?>[title_source]">
                                  <option value="lr_title"  <?php selected($title_source==='lr_title');  ?>>Lightroom Title</option>
                                  <option value="filename"  <?php selected($title_source==='filename');  ?>>Filename (no ext)</option>
                                  <option value="none"      <?php selected($title_source==='none');      ?>>(Leave empty)</option>
                                </select>
                              </label>
                              <label>Caption:
                                <select name="<?php echo esc_html(self::OPT); ?>[caption_source]">
                                  <option value="lr_caption" <?php selected($caption_source==='lr_caption'); ?>>Lightroom Caption</option>
                                  <option value="none"       <?php selected($caption_source==='none');       ?>>(Leave empty)</option>
                                </select>
                              </label>
                              <label>Tags:
                                <select name="<?php echo esc_html(self::OPT); ?>[tags_source]">
                                  <option value="lr_keywords" <?php selected($tags_source==='lr_keywords'); ?>>Lightroom Keywords → WP Tags</option>
                                  <option value="none"        <?php selected($tags_source==='none');        ?>>(Don't import)</option>
                                </select>
                              </label>
                            </div>
                          </td>
                        </tr>
                      </table>
                  </div>
                </div>
            </div>
  <aside class="help">
              <h3>File Names & Output</h3>
              <p>
              Control how images are named and stored in WordPress. Existing images are matched intelligently, so updates replace files instead of creating duplicates. If you're unsure, the default settings are optimized for most sites.
              </p>
              <p><strong>Tip:</strong> Changing naming rules affects future imports, not existing images.</p>
              <p><a href="https://lightsyncpro.com/docs/seo-friendly-file-names/" target="_blank">SEO naming patterns →</a></p>
            </aside>
        </div>
              </section>
            <?php endif; ?><!-- end if connected for naming section -->
            </form>
            </div><!-- END lsp-lightroom-content -->

            <!-- ====== CANVA CONTENT ====== -->
            <div id="lsp-canva-content" class="lsp-source-content <?php echo ($active_source === 'canva') ? 'active' : ''; ?>">
              <section id="lsp-canva-pick" class="section">
                <div class="section-head">
                  <h3 class="lsp-card-title">Canva Designs</h3>
                  <span class="badge muted" id="lsp-canva-count">Designs: 0</span>
                </div>
                <div class="twocol">
                  <div class="panel">
                    <div class="lsp-card-body">
                      <?php 
                      // Check if Canva integration is enabled via remote config
                      $canva_status = lsp_get_source_status('canva');
                      $canva_disabled = !$canva_status['enabled'];
                      $canva_message = $canva_status['message'] ?: 'Coming Soon';
                      ?>
                      <?php if ($canva_disabled): ?>
                        <div class="lsp-canva-connect-prompt" style="text-align:center;padding:40px 20px;">
                          <div style="width:60px;height:60px;margin:0 auto 16px;background:#f1f5f9;border-radius:16px;display:flex;align-items:center;justify-content:center;">
                            <svg width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="#94a3b8" stroke-width="2">
                              <path d="M12 2L2 7l10 5 10-5-10-5z"/>
                              <path d="M2 17l10 5 10-5"/>
                              <path d="M2 12l10 5 10-5"/>
                            </svg>
                          </div>
                          <h4 style="margin:0 0 8px;font-size:18px;color:#64748b;"><?php echo esc_html($canva_message); ?></h4>
                          <p style="color:#94a3b8;margin:0 0 20px;max-width:320px;margin-left:auto;margin-right:auto;">
                            We're finalizing our Canva API integration. This feature will be available shortly.
                          </p>
                          <button type="button" class="lsp-btn" disabled style="background:#e2e8f0;color:#94a3b8;cursor:not-allowed;">
                            Connect with Canva
                          </button>
                          <p style="margin-top:16px;font-size:12px;color:#cbd5e1;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display:inline;vertical-align:middle;margin-right:4px;">
                              <circle cx="12" cy="12" r="10"/>
                              <path d="M12 6v6l4 2"/>
                            </svg>
                            Check back soon for updates
                          </p>
                        </div>
                      <?php elseif (!\LightSyncPro\OAuth\CanvaOAuth::is_connected()): ?>
                        <div class="lsp-canva-connect-prompt" style="text-align:center;padding:40px 20px;">
                          <h4 style="margin:0 0 8px;font-size:18px;">Connect to Canva</h4>
                          <p style="color:#64748b;margin:0 0 20px;">Sync your Canva designs directly to WordPress</p>
                          <a href="<?php echo esc_url(\LightSyncPro\OAuth\CanvaOAuth::auth_url()); ?>" class="lsp-btn primary">
                            Connect with Canva
                          </a>
                          <p style="margin-top:16px;font-size:12px;color:#94a3b8;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display:inline;vertical-align:middle;margin-right:4px;">
                              <circle cx="12" cy="12" r="10"/>
                              <path d="M8 12l2 2 4-4"/>
                            </svg>
                            Powered by Canva
                          </p>
                        </div>
                      <?php else: ?>
                        <div class="lsp-canva-connected">
                          <div style="display:flex;align-items:center;gap:12px;margin-bottom:20px;padding:16px;background:rgba(124,58,237,0.08);border-radius:12px;">
                            <div style="width:40px;height:40px;background:#7c3aed;border-radius:10px;display:flex;align-items:center;justify-content:center;">
                              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2">
                                <path d="M20 6L9 17l-5-5"/>
                              </svg>
                            </div>
                            <div style="flex:1;">
                              <div style="font-weight:600;color:#5b21b6;">Connected to Canva</div>
                              <div style="font-size:13px;color:#7c3aed;">Powered by Canva — Ready to sync designs</div>
                            </div>
                          </div>

                          <!-- Search/Filter -->
                          <div class="lsp-canva-search" style="margin-bottom:16px;">
                            <div style="display:flex;gap:12px;align-items:center;">
                              <div style="flex:3;position:relative;">
                                <input type="text" id="lsp-canva-search" placeholder="Search designs..." 
                                       style="width:100%;padding:10px 12px 10px 36px;border:none;border-radius:8px;font-size:14px;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#94a3b8" stroke-width="2" 
                                     style="position:absolute;left:12px;top:50%;transform:translateY(-50%);">
                                  <circle cx="11" cy="11" r="8"/>
                                  <path d="M21 21l-4.35-4.35"/>
                                </svg>
                              </div>
                              <select id="lsp-canva-sort" style="flex:1;min-width:130px;padding:10px 12px;border:none;border-radius:8px;font-size:14px;">
                                <option value="newest">Newest First</option>
                                <option value="oldest">Oldest First</option>
                                <option value="name">Name A-Z</option>
                              </select>
                            </div>
                          </div>

                          <div class="lsp-canva-designs-grid" id="lsp-canva-designs">
                            <div style="text-align:center;padding:40px;color:#64748b;">
                              <div class="spinner is-active" style="float:none;margin:0 auto 12px;"></div>
                              <p>Loading your Canva designs...</p>
                            </div>
                          </div>

                          <div style="margin-top:20px;display:flex;gap:12px;flex-wrap:wrap;align-items:center;">
                            <button type="button" class="btn ghost" id="lsp-canva-refresh">
                              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right:6px;">
                                <path d="M21 2v6h-6"/>
                                <path d="M3 12a9 9 0 0 1 15-6.7L21 8"/>
                                <path d="M3 22v-6h6"/>
                                <path d="M21 12a9 9 0 0 1-15 6.7L3 16"/>
                              </svg>
                              Refresh Designs
                            </button>
                            <span id="lsp-canva-selection-count" style="color:#64748b;font-size:13px;"></span>
                          </div>

                          <hr style="margin:20px 0;border:0;border-top:1px solid #e2e8f0;">

                          <!-- Sync Destination -->
                          <label style="display:block;margin:0 0 10px;font-weight:600;">Sync Destination</label>
                          <?php 
                          $canva_sync_target = (string)(self::get_opt('canva_sync_target') ?: 'wp'); 
                          $shopify_connected = (bool)self::get_opt('shopify_connected');
                          ?>
                          <div class="lsp-dest-cards" style="display:grid;grid-template-columns:repeat(auto-fit, minmax(100px, 1fr));gap:12px;max-width:560px;">
                            <label class="lsp-dest-card <?php echo $canva_sync_target === 'wp' ? 'selected' : ''; ?>">
                              <input type="radio" name="lsp_canva_sync_target" value="wp" <?php checked($canva_sync_target,'wp'); ?> style="display:none;" />
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <rect x="3" y="3" width="7" height="7" rx="1"/>
                                <rect x="14" y="3" width="7" height="7" rx="1"/>
                                <rect x="3" y="14" width="7" height="7" rx="1"/>
                                <rect x="14" y="14" width="7" height="7" rx="1"/>
                              </svg>
                              <span class="lsp-dest-name">WordPress</span>
                              <span class="lsp-dest-sub">Media Library</span>
                              <span class="lsp-dest-check">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                              </span>
                            </label>
                            
                            <?php if ($shopify_connected): ?>
                            <label class="lsp-dest-card <?php echo $canva_sync_target === 'shopify' ? 'selected' : ''; ?>">
                              <input type="radio" name="lsp_canva_sync_target" value="shopify" <?php checked($canva_sync_target,'shopify'); ?> style="display:none;" />
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                                <line x1="3" y1="6" x2="21" y2="6"/>
                                <path d="M16 10a4 4 0 01-8 0"/>
                              </svg>
                              <span class="lsp-dest-name">Shopify</span>
                              <span class="lsp-dest-sub">Files</span>
                              <span class="lsp-dest-check">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                              </span>
                            </label>
                            
                            <label class="lsp-dest-card <?php echo $canva_sync_target === 'both' ? 'selected' : ''; ?>">
                              <input type="radio" name="lsp_canva_sync_target" value="both" <?php checked($canva_sync_target,'both'); ?> style="display:none;" />
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <circle cx="6" cy="6" r="3"/>
                                <circle cx="18" cy="6" r="3"/>
                                <circle cx="6" cy="18" r="3"/>
                                <circle cx="18" cy="18" r="3"/>
                                <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                              </svg>
                              <span class="lsp-dest-name">Both</span>
                              <span class="lsp-dest-sub">WP + Shopify</span>
                              <span class="lsp-dest-check">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                              </span>
                            </label>
                            <?php else: ?>
                            <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:not-allowed;">
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                                <line x1="3" y1="6" x2="21" y2="6"/>
                                <path d="M16 10a4 4 0 01-8 0"/>
                              </svg>
                              <span class="lsp-dest-name">Shopify</span>
                              <span class="lsp-dest-sub">Not connected</span>
                            </label>
                            
                            <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:not-allowed;">
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <circle cx="6" cy="6" r="3"/>
                                <circle cx="18" cy="6" r="3"/>
                                <circle cx="6" cy="18" r="3"/>
                                <circle cx="18" cy="18" r="3"/>
                                <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                              </svg>
                              <span class="lsp-dest-name">Both</span>
                              <span class="lsp-dest-sub">Connect Shopify</span>
                            </label>
                            <?php endif; ?>
                            
                          </div>


                        </div>
                      <?php endif; ?>
                    </div>
                  </div>
                  <aside class="help">
                    <h3>Canva Integration</h3>
                    
                    
                    <p>
                      Sync your finished Canva designs directly to WordPress. Select the designs you want to import, and <?php echo esc_html( $brand['name'] ); ?> will export them as high-quality images.
                    </p>
                    
                    <p style="margin-top:12px;"><strong>Update Detection:</strong></p>
                    <ul style="margin:6px 0 12px 16px;padding:0;font-size:13px;color:#64748b;">
                      <li style="margin-bottom:4px;">Content hash comparison detects design changes</li>
                      <li style="margin-bottom:4px;">Orange <span style="color:#f59e0b;">⟳</span> badge = updated since last sync</li>
                      <li>Re-sync to push latest version</li>
                    </ul>
                    
                    <p><strong>Tip:</strong> Designs are exported as PNG from Canva, then converted to WebP for optimal web performance. Multi-page designs will create multiple images.</p>
                    <p><a href="<?php echo esc_url( $brand['docs_url'] ); ?>canva-integration/" target="_blank">Canva guide →</a></p>
                  </aside>
                </div>
              </section>
            </div><!-- END lsp-canva-content -->

            <!-- ====== FIGMA CONTENT ====== -->
            <div id="lsp-figma-content" class="lsp-source-content <?php echo ($active_source === 'figma') ? 'active' : ''; ?>">
              <section id="lsp-figma-pick" class="section">
                <div class="section-head">
                  <h3 class="lsp-card-title">Figma Frames</h3>
                  <span class="badge muted" id="lsp-figma-count">Frames: 0</span>
                </div>
                <div class="twocol">
                  <div class="panel">
                    <div class="lsp-card-body">
                      <?php 
                      // Check if Figma integration is enabled via remote config
                      $figma_status = lsp_get_source_status('figma');
                      $figma_disabled = !$figma_status['enabled'];
                      $figma_message = $figma_status['message'] ?: 'Coming Soon';
                      ?>
                      <?php if ($figma_disabled): ?>
                        <div class="lsp-figma-connect-prompt" style="text-align:center;padding:40px 20px;">
                          <div style="width:60px;height:60px;margin:0 auto 16px;background:#f1f5f9;border-radius:16px;display:flex;align-items:center;justify-content:center;">
                            <svg width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="#94a3b8" stroke-width="2">
                              <path d="M5 5.5A3.5 3.5 0 0 1 8.5 2H12v7H8.5A3.5 3.5 0 0 1 5 5.5z"/>
                              <path d="M12 2h3.5a3.5 3.5 0 1 1 0 7H12V2z"/>
                              <path d="M12 12.5a3.5 3.5 0 1 1 7 0 3.5 3.5 0 1 1-7 0z"/>
                              <path d="M5 19.5A3.5 3.5 0 0 1 8.5 16H12v3.5a3.5 3.5 0 1 1-7 0z"/>
                              <path d="M5 12.5A3.5 3.5 0 0 1 8.5 9H12v7H8.5A3.5 3.5 0 0 1 5 12.5z"/>
                            </svg>
                          </div>
                          <h4 style="margin:0 0 8px;font-size:18px;color:#64748b;"><?php echo esc_html($figma_message); ?></h4>
                          <p style="color:#94a3b8;margin:0 0 20px;max-width:320px;margin-left:auto;margin-right:auto;">
                            We're finalizing our Figma API integration. This feature will be available shortly.
                          </p>
                          <button type="button" class="lsp-btn" disabled style="background:#e2e8f0;color:#94a3b8;cursor:not-allowed;">
                            Connect with Figma
                          </button>
                          <p style="margin-top:16px;font-size:12px;color:#cbd5e1;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display:inline;vertical-align:middle;margin-right:4px;">
                              <circle cx="12" cy="12" r="10"/>
                              <path d="M12 6v6l4 2"/>
                            </svg>
                            Check back soon for updates
                          </p>
                        </div>
                      <?php elseif (!\LightSyncPro\OAuth\FigmaOAuth::is_connected()): ?>
                        <div class="lsp-figma-connect-prompt" style="text-align:center;padding:40px 20px;">
                          <h4 style="margin:0 0 8px;font-size:18px;">Connect to Figma</h4>
                          <p style="color:#64748b;margin:0 0 20px;">Sync your Figma frames directly to WordPress</p>
                          <a href="<?php echo esc_url(\LightSyncPro\OAuth\FigmaOAuth::auth_url()); ?>" class="lsp-btn primary">
                            Connect with Figma
                          </a>
                          <p style="margin-top:16px;font-size:12px;color:#94a3b8;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display:inline;vertical-align:middle;margin-right:4px;">
                              <circle cx="12" cy="12" r="10"/>
                              <path d="M8 12l2 2 4-4"/>
                            </svg>
                            Secure OAuth connection via Figma
                          </p>
                        </div>
                      <?php else: ?>
                        <div class="lsp-figma-connected">
                          <!-- Connected Banner with Last Sync Info -->
                          <div style="display:flex;align-items:center;gap:12px;margin-bottom:20px;padding:16px;background:rgba(37,99,235,0.08);border-radius:12px;">
                            <div style="width:40px;height:40px;background:#2563eb;border-radius:10px;display:flex;align-items:center;justify-content:center;">
                              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2">
                                <path d="M20 6L9 17l-5-5"/>
                              </svg>
                            </div>
                            <div style="flex:1;">
                              <div style="font-weight:600;color:#1e40af;">Connected to Figma</div>
                              <div style="font-size:13px;color:#2563eb;" id="lsp-figma-last-sync">Ready to sync frames</div>
                            </div>
                          </div>

                          <!-- Add Figma File -->
                          <div class="lsp-figma-add-file" style="margin-bottom:20px;">
                            
                            <!-- Your Files Grid -->
                            <div id="lsp-figma-your-files" style="margin-bottom:16px;">
                              <div style="display:flex;align-items:center;justify-content:space-between;margin-bottom:12px;">
                                <label style="font-size:13px;font-weight:600;color:#374151;">Your Files</label>
                                <button type="button" id="lsp-figma-show-add-url" style="display:inline-flex;align-items:center;gap:4px;padding:6px 12px;background:#2563eb;color:#fff;border:none;border-radius:6px;font-size:12px;font-weight:500;cursor:pointer;">
                                  <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M12 5v14M5 12h14"/>
                                  </svg>
                                  Add File
                                </button>
                              </div>
                              
                              <!-- Empty state -->
                              <div id="lsp-figma-files-empty" style="display:none;padding:32px;text-align:center;background:rgba(0,0,0,0.03);border:2px dashed #e2e8f0;border-radius:12px;">
                                <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="#94a3b8" stroke-width="1.5" style="margin:0 auto 12px;">
                                  <path d="M5 5.5A3.5 3.5 0 0 1 8.5 2H12v7H8.5A3.5 3.5 0 0 1 5 5.5z"/>
                                  <path d="M12 2h3.5a3.5 3.5 0 1 1 0 7H12V2z"/>
                                  <path d="M12 12.5a3.5 3.5 0 1 1 7 0 3.5 3.5 0 1 1-7 0z"/>
                                  <path d="M5 19.5A3.5 3.5 0 0 1 8.5 16H12v3.5a3.5 3.5 0 1 1-7 0z"/>
                                  <path d="M5 12.5A3.5 3.5 0 0 1 8.5 9H12v7H8.5A3.5 3.5 0 0 1 5 12.5z"/>
                                </svg>
                                <p style="margin:0 0 4px;font-weight:600;color:#374151;">No files yet</p>
                                <p style="margin:0;font-size:13px;color:#64748b;">Add a Figma file URL to get started</p>
                              </div>
                              
                              <!-- Files grid -->
                              <div id="lsp-figma-files-grid" style="display:grid;grid-template-columns:repeat(auto-fill, minmax(140px, 1fr));gap:12px;"></div>
                            </div>
                            
                            <!-- Add URL panel (hidden by default) -->
                            <div id="lsp-figma-add-url-panel" style="display:none;padding:16px;background:rgba(0,0,0,0.03);border-radius:10px;margin-bottom:16px;">
                              <div style="display:flex;align-items:center;justify-content:space-between;margin-bottom:12px;">
                                <label style="font-size:13px;font-weight:600;color:#374151;">Add Figma File</label>
                                <button type="button" id="lsp-figma-hide-add-url" style="background:none;border:none;cursor:pointer;padding:4px;color:#64748b;">
                                  <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M18 6L6 18M6 6l12 12"/></svg>
                                </button>
                              </div>
                              <div style="display:flex;gap:8px;">
                                <input type="text" id="lsp-figma-file-url" placeholder="Paste Figma file URL (e.g. figma.com/design/ABC123/My-Design)" 
                                       style="flex:1;padding:10px 12px;border:none;border-radius:8px;font-size:14px;">
                                <button type="button" class="btn primary" id="lsp-figma-add-file-btn">
                                  <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right:4px;">
                                    <path d="M12 5v14M5 12h14"/>
                                  </svg>
                                  Add
                                </button>
                              </div>
                              <p class="description" style="margin-top:8px;font-size:12px;color:#64748b;">
                                Copy the URL from your browser when viewing a Figma file
                              </p>
                            </div>
                          </div>

                          <!-- Search/Filter (like Canva) -->
                          <div class="lsp-figma-search" style="margin-bottom:16px;">
                            <div style="display:flex;gap:10px;align-items:center;flex-wrap:wrap;">
                              <div style="flex:1;min-width:200px;position:relative;">
                                <input type="text" id="lsp-figma-search" placeholder="Search frames..." 
                                       style="width:100%;padding:10px 12px 10px 36px;border:none;border-radius:8px;font-size:14px;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#94a3b8" stroke-width="2" 
                                     style="position:absolute;left:12px;top:50%;transform:translateY(-50%);">
                                  <circle cx="11" cy="11" r="8"/>
                                  <path d="M21 21l-4.35-4.35"/>
                                </svg>
                              </div>
                              <select id="lsp-figma-file-filter" style="flex-shrink:0;padding:10px 12px;border:none;border-radius:8px;font-size:14px;">
                                <option value="all">All Files</option>
                              </select>
                              <select id="lsp-figma-type-filter" style="flex-shrink:0;padding:10px 12px;border:none;border-radius:8px;font-size:14px;display:none;">
                                <option value="all">All Types</option>
                                <option value="FRAME">Frames</option>
                                <option value="COMPONENT">Components</option>
                                <option value="INSTANCE">Instances</option>
                                <option value="GROUP">Groups</option>
                                <option value="VECTOR">Vectors</option>
                                <option value="TEXT">Text</option>
                                <option value="RECTANGLE">Rectangles</option>
                                <option value="ELLIPSE">Ellipses</option>
                              </select>
                              <label style="flex-shrink:0;display:flex;align-items:center;gap:6px;cursor:pointer;white-space:nowrap;padding:8px 12px;background:rgba(0,0,0,0.03);border:none;border-radius:8px;" title="Show icons, buttons, shapes inside frames">
                                <input type="checkbox" id="lsp-figma-show-nested" style="width:14px;height:14px;accent-color:#2563eb;">
                                <span style="font-size:13px;">Nested</span>
                              </label>
                            </div>
                          </div>

                          <!-- Frames Grid (like Canva designs grid) -->
                          <div class="lsp-figma-frames-grid" id="lsp-figma-frames" style="display:grid;grid-template-columns:repeat(auto-fill,minmax(140px,1fr));gap:12px;max-height:450px;overflow-y:auto;padding:4px;">
                            <div style="text-align:center;padding:40px;color:#64748b;grid-column:1/-1;">
                              <p>No frames yet. Add a Figma file URL above to get started.</p>
                            </div>
                          </div>

                          <div style="margin-top:20px;display:flex;gap:12px;flex-wrap:wrap;align-items:center;">
                            <button type="button" class="btn ghost" id="lsp-figma-refresh">
                              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right:6px;">
                                <path d="M21 2v6h-6"/>
                                <path d="M3 12a9 9 0 0 1 15-6.7L21 8"/>
                                <path d="M3 22v-6h6"/>
                                <path d="M21 12a9 9 0 0 1-15 6.7L3 16"/>
                              </svg>
                              Refresh
                            </button>
                            <button type="button" class="btn ghost" id="lsp-figma-check-updates" title="Check if any synced frames have been updated in Figma">
                              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right:6px;">
                                <path d="M12 22c5.523 0 10-4.477 10-10S17.523 2 12 2 2 6.477 2 12s4.477 10 10 10z"/>
                                <path d="M12 6v6l4 2"/>
                              </svg>
                              Check for Updates
                            </button>
                            <button type="button" class="btn ghost btn-sm" id="lsp-figma-select-all">Select All</button>
                            <button type="button" class="btn ghost btn-sm" id="lsp-figma-select-none">Select None</button>
                            <span id="lsp-figma-selection-count" style="color:#64748b;font-size:13px;margin-left:auto;"></span>
                          </div>

                          <hr style="margin:20px 0;border:0;border-top:1px solid #e2e8f0;">

                          <!-- Export Settings -->
                          <div class="lsp-figma-export-settings" style="margin-bottom:20px;">
                            <label style="display:block;margin:0 0 10px;font-weight:600;">Export Settings</label>
                            <div style="display:grid;grid-template-columns:1fr 1fr;gap:12px;">
                              <div>
                                <label style="display:block;font-size:13px;color:#64748b;margin-bottom:4px;">Format</label>
                                <select id="lsp-figma-export-format" style="width:100%;padding:8px 12px;border:1px solid #e2e8f0;border-radius:6px;background:#f8fafc;">
                                  <option value="webp" selected>WebP (modern, smaller)</option>
                                </select>
                              </div>
                              <div>
                                <label style="display:block;font-size:13px;color:#64748b;margin-bottom:4px;">Scale</label>
                                <select id="lsp-figma-export-scale" style="width:100%;padding:8px 12px;border:1px solid #e2e8f0;border-radius:6px;background:#f8fafc;">
                                  <option value="2" selected>2x (retina)</option>
                                </select>
                              </div>
                            </div>
                          </div>

                          <hr style="margin:20px 0;border:0;border-top:1px solid #e2e8f0;">

                          <!-- Sync Destination -->
                          <label style="display:block;margin:0 0 10px;font-weight:600;">Sync Destination</label>
                          <?php 
                          $figma_sync_target = (string)(self::get_opt('figma_sync_target') ?: 'wp'); 
                          $shopify_connected = (bool)self::get_opt('shopify_connected');
                          ?>
                          <div class="lsp-dest-cards" style="display:grid;grid-template-columns:repeat(auto-fit, minmax(100px, 1fr));gap:12px;max-width:560px;">
                            <label class="lsp-dest-card <?php echo $figma_sync_target === 'wp' ? 'selected' : ''; ?>">
                              <input type="radio" name="lsp_figma_sync_target" value="wp" <?php checked($figma_sync_target,'wp'); ?> style="display:none;" />
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <rect x="3" y="3" width="7" height="7" rx="1"/>
                                <rect x="14" y="3" width="7" height="7" rx="1"/>
                                <rect x="3" y="14" width="7" height="7" rx="1"/>
                                <rect x="14" y="14" width="7" height="7" rx="1"/>
                              </svg>
                              <span class="lsp-dest-name">WordPress</span>
                              <span class="lsp-dest-sub">Media Library</span>
                              <span class="lsp-dest-check">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                              </span>
                            </label>
                            
                            <?php if ($shopify_connected): ?>
                            <label class="lsp-dest-card <?php echo $figma_sync_target === 'shopify' ? 'selected' : ''; ?>">
                              <input type="radio" name="lsp_figma_sync_target" value="shopify" <?php checked($figma_sync_target,'shopify'); ?> style="display:none;" />
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                                <line x1="3" y1="6" x2="21" y2="6"/>
                                <path d="M16 10a4 4 0 01-8 0"/>
                              </svg>
                              <span class="lsp-dest-name">Shopify</span>
                              <span class="lsp-dest-sub">Files</span>
                              <span class="lsp-dest-check">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                              </span>
                            </label>
                            
                            <label class="lsp-dest-card <?php echo $figma_sync_target === 'both' ? 'selected' : ''; ?>">
                              <input type="radio" name="lsp_figma_sync_target" value="both" <?php checked($figma_sync_target,'both'); ?> style="display:none;" />
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <circle cx="6" cy="6" r="3"/>
                                <circle cx="18" cy="6" r="3"/>
                                <circle cx="6" cy="18" r="3"/>
                                <circle cx="18" cy="18" r="3"/>
                                <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                              </svg>
                              <span class="lsp-dest-name">Both</span>
                              <span class="lsp-dest-sub">WP + Shopify</span>
                              <span class="lsp-dest-check">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                              </span>
                            </label>
                            <?php else: ?>
                            <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:not-allowed;">
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                                <line x1="3" y1="6" x2="21" y2="6"/>
                                <path d="M16 10a4 4 0 01-8 0"/>
                              </svg>
                              <span class="lsp-dest-name">Shopify</span>
                              <span class="lsp-dest-sub">Not connected</span>
                            </label>
                            
                            <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:not-allowed;">
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <circle cx="6" cy="6" r="3"/>
                                <circle cx="18" cy="6" r="3"/>
                                <circle cx="6" cy="18" r="3"/>
                                <circle cx="18" cy="18" r="3"/>
                                <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                              </svg>
                              <span class="lsp-dest-name">Both</span>
                              <span class="lsp-dest-sub">Connect Shopify</span>
                            </label>
                            <?php endif; ?>
                            
                          </div>

                          <!-- Progress Section (floating card will be used instead) -->
                          <div id="lsp-figma-progress" style="display:none;margin-top:20px;padding:16px;background:#f1f5f9;border-radius:8px;">
                            <div style="display:flex;justify-content:space-between;margin-bottom:8px;">
                              <span id="lsp-figma-progress-text">Syncing...</span>
                              <span id="lsp-figma-progress-percent">0%</span>
                            </div>
                            <div style="height:8px;background:#e2e8f0;border-radius:4px;overflow:hidden;">
                              <div id="lsp-figma-progress-bar" style="height:100%;background:#2563eb;width:0%;transition:width 0.3s;"></div>
                            </div>
                          </div>
                        </div>
                      <?php endif; ?>
                    </div>
                  </div>
                  <aside class="help">
                    <h3>Figma Integration</h3>
                    
                    
                    <p>
                      Sync frames and components from your Figma files directly to WordPress. Select elements from the grid and click <strong>Sync Now</strong> to import.
                    </p>
                    
                    <p style="margin-top:12px;"><strong>Update Detection:</strong></p>
                    <ul style="margin:6px 0 12px 16px;padding:0;font-size:13px;color:#64748b;">
                      <li style="margin-bottom:4px;">Tracks Figma file's <code>lastModified</code> timestamp</li>
                      <li style="margin-bottom:4px;">Orange <span style="color:#f59e0b;">⟳</span> badge = file updated since last sync</li>
                      <li>Re-sync to get the latest design changes</li>
                    </ul>
                    
                    <p><strong>Tip:</strong> Use 2x scale for retina-ready images. Exports are automatically converted to WebP for optimal web performance.</p>
                    <p style="margin-top:12px;"><a href="https://lightsyncpro.com/docs/figma-integration/" target="_blank">Figma guide →</a></p>
                  </aside>
                </div>
              </section>
            </div><!-- END lsp-figma-content -->

            <!-- ====== DROPBOX CONTENT ====== -->
            <div id="lsp-dropbox-content" class="lsp-source-content <?php echo ($active_source === 'dropbox') ? 'active' : ''; ?>">
              <section id="lsp-dropbox-pick" class="section">
                <div class="section-head">
                  <h3 class="lsp-card-title">Dropbox Files</h3>
                  <span class="badge muted" id="lsp-dropbox-count">Files: 0</span>
                </div>
                <div class="twocol">
                  <div class="panel">
                    <div class="lsp-card-body">
                      <?php 
                      // Check if Dropbox integration is enabled via remote config
                      $dropbox_status = lsp_get_source_status('dropbox');
                      $dropbox_disabled = !$dropbox_status['enabled'];
                      $dropbox_message = $dropbox_status['message'] ?: 'Coming Soon';
                      ?>
                      <?php if ($dropbox_disabled): ?>
                        <div class="lsp-dropbox-connect-prompt" style="text-align:center;padding:40px 20px;">
                          <div style="width:60px;height:60px;margin:0 auto 16px;background:#f1f5f9;border-radius:16px;display:flex;align-items:center;justify-content:center;">
                            <svg width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="#94a3b8" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                              <path d="M12 2L6 6l6 4-6 4 6 4 6-4-6-4 6-4-6-4z"/>
                              <path d="M6 14l6 4 6-4"/>
                            </svg>
                          </div>
                          <h4 style="margin:0 0 8px;font-size:18px;color:#64748b;"><?php echo esc_html($dropbox_message); ?></h4>
                          <p style="color:#94a3b8;margin:0 0 20px;max-width:320px;margin-left:auto;margin-right:auto;">
                            We're finalizing our Dropbox API integration. This feature will be available shortly.
                          </p>
                          <button type="button" class="lsp-btn" disabled style="background:#e2e8f0;color:#94a3b8;cursor:not-allowed;">
                            Connect with Dropbox
                          </button>
                          <p style="margin-top:16px;font-size:12px;color:#cbd5e1;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display:inline;vertical-align:middle;margin-right:4px;">
                              <circle cx="12" cy="12" r="10"/>
                              <path d="M12 6v6l4 2"/>
                            </svg>
                            Check back soon for updates
                          </p>
                        </div>
                      <?php elseif (!\LightSyncPro\OAuth\DropboxOAuth::is_connected()): ?>
                        <!-- Not Connected: Show Connect Prompt -->
                        <div class="lsp-dropbox-connect-prompt" style="text-align:center;padding:40px 20px;">
                          <h4 style="margin:0 0 8px;font-size:18px;">Connect to Dropbox</h4>
                          <p style="color:#64748b;margin:0 0 20px;">Sync your Dropbox images directly to WordPress</p>
                          <a href="<?php echo esc_url(\LightSyncPro\OAuth\DropboxOAuth::auth_url()); ?>" class="lsp-btn primary">
                            Connect with Dropbox
                          </a>
                          <p style="margin-top:16px;font-size:12px;color:#94a3b8;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display:inline;vertical-align:middle;margin-right:4px;">
                              <circle cx="12" cy="12" r="10"/>
                              <path d="M8 12l2 2 4-4"/>
                            </svg>
                            Secure OAuth connection via Dropbox
                          </p>
                        </div>
                      <?php else: ?>
                        <!-- Connected: Show Folder Browser -->
                        <div class="lsp-dropbox-connected">
                          <!-- Connected Banner with Last Sync Info -->
                          <div style="display:flex;align-items:center;gap:12px;margin-bottom:20px;padding:16px;background:rgba(8,145,178,0.08);border-radius:12px;">
                            <div style="width:40px;height:40px;background:#0891b2;border-radius:10px;display:flex;align-items:center;justify-content:center;">
                              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2">
                                <path d="M20 6L9 17l-5-5"/>
                              </svg>
                            </div>
                            <div style="flex:1;">
                              <div style="font-weight:600;color:#155e75;">Connected to Dropbox</div>
                              <div style="font-size:13px;color:#0891b2;" id="lsp-dropbox-last-sync">Ready to sync files</div>
                            </div>
                          </div>

                          <!-- Breadcrumb Navigation -->
                          <div id="lsp-dropbox-breadcrumb" style="display:flex;align-items:center;gap:6px;margin-bottom:16px;padding:8px 12px;background:rgba(0,0,0,0.03);border-radius:8px;font-size:13px;flex-wrap:wrap;">
                            <span class="lsp-dropbox-crumb" data-path="" style="cursor:pointer;color:#0891b2;font-weight:500;">
                              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="vertical-align:middle;margin-right:4px;">
                                <path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/>
                                <polyline points="9 22 9 12 15 12 15 22"/>
                              </svg>
                              Dropbox
                            </span>
                          </div>

                          <!-- Search/Filter -->
                          <div class="lsp-dropbox-search" style="margin-bottom:16px;">
                            <div style="display:flex;gap:10px;align-items:center;flex-wrap:wrap;">
                              <div style="flex:1;min-width:200px;position:relative;">
                                <input type="text" id="lsp-dropbox-search" placeholder="Search in current folder..." 
                                       style="width:100%;padding:10px 12px 10px 36px;border:none;border-radius:8px;font-size:14px;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#94a3b8" stroke-width="2" 
                                     style="position:absolute;left:12px;top:50%;transform:translateY(-50%);">
                                  <circle cx="11" cy="11" r="8"/>
                                  <path d="M21 21l-4.35-4.35"/>
                                </svg>
                              </div>
                              <select id="lsp-dropbox-sort" style="flex-shrink:0;min-width:130px;padding:10px 12px;border:none;border-radius:8px;font-size:14px;">
                                <option value="name">Name A-Z</option>
                                <option value="newest">Newest First</option>
                                <option value="oldest">Oldest First</option>
                              </select>
                            </div>
                          </div>

                          <!-- Folder & Files Browser -->
                          <div id="lsp-dropbox-browser-content" style="display:grid;grid-template-columns:repeat(auto-fill,minmax(140px,1fr));gap:12px;max-height:450px;overflow-y:auto;padding:4px;">
                            <div style="text-align:center;padding:40px;color:#64748b;grid-column:1/-1;">
                              <div class="spinner is-active" style="float:none;margin:0 auto 12px;"></div>
                              <p>Loading your Dropbox...</p>
                            </div>
                          </div>

                          <div style="margin-top:20px;display:flex;gap:12px;flex-wrap:wrap;align-items:center;">
                            <button type="button" class="btn ghost" id="lsp-dropbox-refresh">
                              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right:6px;">
                                <path d="M21 2v6h-6"/>
                                <path d="M3 12a9 9 0 0 1 15-6.7L21 8"/>
                                <path d="M3 22v-6h6"/>
                                <path d="M21 12a9 9 0 0 1-15 6.7L3 16"/>
                              </svg>
                              Refresh
                            </button>
                            <button type="button" class="btn ghost btn-sm" id="lsp-dropbox-select-all">Select All Images</button>
                            <button type="button" class="btn ghost btn-sm" id="lsp-dropbox-select-none">Select None</button>
                            <span id="lsp-dropbox-selection-count" style="color:#64748b;font-size:13px;margin-left:auto;"></span>
                          </div>

                          <hr style="margin:20px 0;border:0;border-top:1px solid #e2e8f0;">

                          <!-- Sync Destination -->
                          <label style="display:block;margin:0 0 10px;font-weight:600;">Sync Destination</label>
                          <?php 
                          $dropbox_sync_target = (string)(self::get_opt('dropbox_sync_target') ?: 'wp'); 
                          $shopify_connected = (bool)self::get_opt('shopify_connected');
                          ?>
                          <div class="lsp-dest-cards" style="display:grid;grid-template-columns:repeat(auto-fit, minmax(100px, 1fr));gap:12px;max-width:560px;">
                            <label class="lsp-dest-card <?php echo $dropbox_sync_target === 'wp' ? 'selected' : ''; ?>">
                              <input type="radio" name="lsp_dropbox_sync_target" value="wp" <?php checked($dropbox_sync_target,'wp'); ?> style="display:none;" />
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <rect x="3" y="3" width="7" height="7" rx="1"/>
                                <rect x="14" y="3" width="7" height="7" rx="1"/>
                                <rect x="3" y="14" width="7" height="7" rx="1"/>
                                <rect x="14" y="14" width="7" height="7" rx="1"/>
                              </svg>
                              <span class="lsp-dest-name">WordPress</span>
                              <span class="lsp-dest-sub">Media Library</span>
                              <span class="lsp-dest-check">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                              </span>
                            </label>
                            
                            <?php if ($shopify_connected): ?>
                            <label class="lsp-dest-card <?php echo $dropbox_sync_target === 'shopify' ? 'selected' : ''; ?>">
                              <input type="radio" name="lsp_dropbox_sync_target" value="shopify" <?php checked($dropbox_sync_target,'shopify'); ?> style="display:none;" />
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                                <line x1="3" y1="6" x2="21" y2="6"/>
                                <path d="M16 10a4 4 0 01-8 0"/>
                              </svg>
                              <span class="lsp-dest-name">Shopify</span>
                              <span class="lsp-dest-sub">Files</span>
                              <span class="lsp-dest-check">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                              </span>
                            </label>
                            
                            <label class="lsp-dest-card <?php echo $dropbox_sync_target === 'both' ? 'selected' : ''; ?>">
                              <input type="radio" name="lsp_dropbox_sync_target" value="both" <?php checked($dropbox_sync_target,'both'); ?> style="display:none;" />
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <circle cx="6" cy="6" r="3"/>
                                <circle cx="18" cy="6" r="3"/>
                                <circle cx="6" cy="18" r="3"/>
                                <circle cx="18" cy="18" r="3"/>
                                <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                              </svg>
                              <span class="lsp-dest-name">Both</span>
                              <span class="lsp-dest-sub">WP + Shopify</span>
                              <span class="lsp-dest-check">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                              </span>
                            </label>
                            <?php else: ?>
                            <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:not-allowed;">
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                                <line x1="3" y1="6" x2="21" y2="6"/>
                                <path d="M16 10a4 4 0 01-8 0"/>
                              </svg>
                              <span class="lsp-dest-name">Shopify</span>
                              <span class="lsp-dest-sub">Not connected</span>
                            </label>
                            
                            <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:not-allowed;">
                              <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <circle cx="6" cy="6" r="3"/>
                                <circle cx="18" cy="6" r="3"/>
                                <circle cx="6" cy="18" r="3"/>
                                <circle cx="18" cy="18" r="3"/>
                                <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                              </svg>
                              <span class="lsp-dest-name">Both</span>
                              <span class="lsp-dest-sub">Connect Shopify</span>
                            </label>
                            <?php endif; ?>
                            
                          </div>

                          
                          <!-- Folder Picker Modal - placed outside autosync div for proper stacking -->

                          <!-- Progress Section -->
                          <div id="lsp-dropbox-progress" style="display:none;margin-top:20px;padding:16px;background:#f1f5f9;border-radius:8px;">
                            <div style="display:flex;justify-content:space-between;margin-bottom:8px;">
                              <span id="lsp-dropbox-progress-text">Syncing...</span>
                              <span id="lsp-dropbox-progress-percent">0%</span>
                            </div>
                            <div style="height:8px;background:#e2e8f0;border-radius:4px;overflow:hidden;">
                              <div id="lsp-dropbox-progress-bar" style="height:100%;background:#0891b2;width:0%;transition:width 0.3s;"></div>
                            </div>
                          </div>
                        </div>
                      <?php endif; ?>
                    </div>
                  </div>
                  <aside class="help">
                    <h3>Dropbox Integration</h3>
                    
                    
                    <p>
                      Sync images from your Dropbox folders directly to WordPress. Select folders to watch, and <?php echo esc_html( $brand['name'] ); ?> will import images automatically.
                    </p>
                    
                    <p style="margin-top:12px;"><strong>Update Detection:</strong></p>
                    <ul style="margin:6px 0 12px 16px;padding:0;font-size:13px;color:#64748b;">
                      <li style="margin-bottom:4px;">Compares Dropbox <code>server_modified</code> timestamp</li>
                      <li>Orange <span style="color:#f59e0b;">⟳</span> badge = file changed in Dropbox</li>
                    </ul>
                    
                    <p><strong>Supported formats:</strong> JPG, PNG, GIF, WebP, TIFF, BMP</p>
                    <p><strong>Tip:</strong> Images are automatically optimized and converted to WebP for optimal web performance.</p>
                    <p><strong>RAW files:</strong> NEF, CR2, ARW, etc. require server-side conversion that most hosts don't support. Use the Lightroom tab for RAW photos — Adobe handles the conversion automatically.</p>
                    <p style="margin-top:12px;"><a href="<?php echo esc_url( $brand['docs_url'] ); ?>dropbox-integration/" target="_blank">Dropbox guide →</a></p>
                  </aside>
                </div>
              </section>
            </div><!-- END lsp-dropbox-content -->

            <!-- ====== SHUTTERSTOCK CONTENT ====== -->
            <div id="lsp-shutterstock-content" class="lsp-source-content <?php echo ($active_source === 'shutterstock') ? 'active' : ''; ?>">
              <section class="section">
                <div class="section-head">
                  <h3 class="lsp-card-title">Shutterstock Licensed Images</h3>
                </div>
                <div class="twocol">
                  <div class="panel" style="flex:2">
                    <div class="lsp-card-body">
                      <?php if (!\LightSyncPro\OAuth\ShutterstockOAuth::is_connected()): ?>
                        <!-- Not Connected: Show Connect Prompt -->
                        <div class="lsp-shutterstock-connect-prompt" style="text-align:center;padding:40px 20px;">
                          <div style="width:60px;height:60px;margin:0 auto 16px;background:linear-gradient(135deg, #ee2d24 0%, #ff6b35 100%);border-radius:16px;display:flex;align-items:center;justify-content:center;">
                            <svg width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2">
                              <rect x="3" y="3" width="18" height="18" rx="2"/>
                              <path d="M9 9h6v6H9z"/>
                              <path d="M9 3v6"/>
                              <path d="M15 15v6"/>
                            </svg>
                          </div>
                          <h4 style="margin:0 0 8px;font-size:18px;">Connect to Shutterstock</h4>
                          <p style="color:#64748b;margin:0 0 20px;max-width:400px;margin-left:auto;margin-right:auto;">
                            Sync images you've already licensed with your Shutterstock subscription directly to WordPress
                          </p>
                          <a href="<?php echo esc_url(\LightSyncPro\OAuth\ShutterstockOAuth::auth_url()); ?>" class="lsp-btn primary">
                            Connect with Shutterstock
                          </a>
                          <p style="margin-top:16px;font-size:12px;color:#94a3b8;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display:inline;vertical-align:middle;margin-right:4px;">
                              <circle cx="12" cy="12" r="10"/>
                              <path d="M8 12l2 2 4-4"/>
                            </svg>
                            Access your previously purchased images
                          </p>
                        </div>
                      <?php else: ?>
                        <!-- Connected: Show License Browser -->
                        <div class="lsp-shutterstock-connected">
                          <div style="display:flex;align-items:center;gap:12px;margin-bottom:20px;padding:16px;background:rgba(238,45,36,0.08);border-radius:12px;">
                            <div style="width:40px;height:40px;background:#ee2d24;border-radius:10px;display:flex;align-items:center;justify-content:center;">
                              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2">
                                <path d="M20 6L9 17l-5-5"/>
                              </svg>
                            </div>
                            <div style="flex:1;">
                              <div style="font-weight:600;color:#b91c1c;">Connected to Shutterstock</div>
                              <div style="font-size:13px;color:#ee2d24;" id="lsp-shutterstock-status">Ready to sync licensed images</div>
                            </div>
                          </div>

                          <!-- License Browser Grid -->
                          <div id="lsp-shutterstock-browser">
                            <!-- Search and Filter Bar -->
                            <div id="lsp-shutterstock-filters" style="display:none;margin-bottom:16px;">
                              <div style="display:flex;gap:10px;flex-wrap:wrap;align-items:center;">
                                <div style="flex:1;min-width:200px;position:relative;">
                                  <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#94a3b8" stroke-width="2" style="position:absolute;left:12px;top:50%;transform:translateY(-50%);pointer-events:none;">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="M21 21l-4.35-4.35"/>
                                  </svg>
                                  <input type="text" id="lsp-shutterstock-search" placeholder="Search descriptions..." style="width:100%;padding:10px 12px 10px 38px;border:1px solid #e2e8f0;border-radius:8px;font-size:13px;transition:border-color 0.15s;">
                                </div>
                                <select id="lsp-shutterstock-filter-status" style="padding:10px 32px 10px 12px;border:1px solid #e2e8f0;border-radius:8px;font-size:13px;background:#fff url('data:image/svg+xml,<svg xmlns=\"http://www.w3.org/2000/svg\" width=\"12\" height=\"12\" viewBox=\"0 0 24 24\" fill=\"none\" stroke=\"%2394a3b8\" stroke-width=\"2\"><path d=\"M6 9l6 6 6-6\"/></svg>') no-repeat right 12px center;appearance:none;cursor:pointer;">
                                  <option value="all">All Images</option>
                                  <option value="unsynced">Not Synced</option>
                                  <option value="synced">Synced</option>
                                  <option value="wp">Synced to WordPress</option>
                                  <option value="shopify">Synced to Shopify</option>
                                </select>
                                <select id="lsp-shutterstock-sort" style="padding:10px 32px 10px 12px;border:1px solid #e2e8f0;border-radius:8px;font-size:13px;background:#fff url('data:image/svg+xml,<svg xmlns=\"http://www.w3.org/2000/svg\" width=\"12\" height=\"12\" viewBox=\"0 0 24 24\" fill=\"none\" stroke=\"%2394a3b8\" stroke-width=\"2\"><path d=\"M6 9l6 6 6-6\"/></svg>') no-repeat right 12px center;appearance:none;cursor:pointer;">
                                  <option value="newest">Newest First</option>
                                  <option value="oldest">Oldest First</option>
                                  <option value="az">A → Z</option>
                                  <option value="za">Z → A</option>
                                </select>
                              </div>
                              <div id="lsp-shutterstock-filter-info" style="margin-top:10px;font-size:12px;color:#64748b;display:none;">
                                Showing <span id="lsp-shutterstock-filtered-count">0</span> of <span id="lsp-shutterstock-total-display">0</span> images
                              </div>
                            </div>
                            
                            <div id="lsp-shutterstock-loading" style="text-align:center;padding:40px 20px;">
                              <div class="lsp-spinner" style="margin:0 auto 16px;"></div>
                              <p style="color:#64748b;">Loading your licensed images...</p>
                            </div>
                            <div id="lsp-shutterstock-grid" style="display:none;"></div>
                            <div id="lsp-shutterstock-empty" style="display:none;text-align:center;padding:40px 20px;background:#f8fafc;border-radius:12px;border:2px dashed #e2e8f0;">
                              <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="#94a3b8" stroke-width="1.5" style="margin-bottom:16px;">
                                <rect x="3" y="3" width="18" height="18" rx="2"/>
                                <circle cx="8.5" cy="8.5" r="1.5"/>
                                <path d="M21 15l-5-5L5 21"/>
                              </svg>
                              <h4 style="margin:0 0 8px;color:#64748b;">No Licensed Images Found</h4>
                              <p style="color:#94a3b8;margin:0;max-width:320px;margin-left:auto;margin-right:auto;">
                                Purchase images on Shutterstock and they'll appear here for syncing
                              </p>
                            </div>
                            <div id="lsp-shutterstock-no-results" style="display:none;text-align:center;padding:40px 20px;background:#f8fafc;border-radius:12px;border:2px dashed #e2e8f0;">
                              <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="#94a3b8" stroke-width="1.5" style="margin-bottom:16px;">
                                <circle cx="11" cy="11" r="8"/>
                                <path d="M21 21l-4.35-4.35"/>
                                <path d="M8 8l6 6M14 8l-6 6"/>
                              </svg>
                              <h4 style="margin:0 0 8px;color:#64748b;">No Matching Images</h4>
                              <p style="color:#94a3b8;margin:0;">
                                Try adjusting your search or filters
                              </p>
                              <button type="button" class="btn ghost btn-sm" id="lsp-shutterstock-clear-filters" style="margin-top:12px;">Clear Filters</button>
                            </div>
                          </div>

                          <!-- Selection Controls (below grid, like Dropbox) -->
                          <div id="lsp-shutterstock-controls" style="display:none;margin-top:20px;gap:12px;flex-wrap:wrap;align-items:center;">
                            <button type="button" class="btn ghost" id="lsp-shutterstock-refresh">
                              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right:6px;">
                                <path d="M21 2v6h-6"/>
                                <path d="M3 12a9 9 0 0 1 15-6.7L21 8"/>
                                <path d="M3 22v-6h6"/>
                                <path d="M21 12a9 9 0 0 1-15 6.7L3 16"/>
                              </svg>
                              Refresh
                            </button>
                            <button type="button" class="btn ghost btn-sm" id="lsp-shutterstock-select-all">Select All Images</button>
                            <button type="button" class="btn ghost btn-sm" id="lsp-shutterstock-select-none">Select None</button>
                            <span id="lsp-shutterstock-selection-count" style="color:#64748b;font-size:13px;margin-left:auto;"></span>
                          </div>

                          <!-- Pagination -->
                          <div id="lsp-shutterstock-pagination" style="display:none;margin-top:16px;text-align:center;">
                            <button type="button" class="btn ghost" id="lsp-shutterstock-load-more">Load More</button>
                          </div>

                          <hr style="margin:20px 0;border:0;border-top:1px solid #e2e8f0;">

                          <!-- Sync Destination (below grid) -->
                          <div id="lsp-shutterstock-dest-section" style="display:none;">
                            <label style="display:block;margin:0 0 10px;font-weight:600;">Sync Destination</label>
                            <?php 
                            $shutterstock_sync_target = (string)(self::get_opt('shutterstock_sync_target') ?: 'wp'); 
                            $shopify_connected = (bool)self::get_opt('shopify_connected');
                            ?>
                            <div class="lsp-dest-cards" style="display:grid;grid-template-columns:repeat(auto-fit, minmax(100px, 1fr));gap:12px;max-width:560px;">
                              <label class="lsp-dest-card <?php echo $shutterstock_sync_target === 'wp' ? 'selected' : ''; ?>">
                                <input type="radio" name="lsp_shutterstock_sync_target" value="wp" <?php checked($shutterstock_sync_target,'wp'); ?> style="display:none;" />
                                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                  <rect x="3" y="3" width="7" height="7" rx="1"/>
                                  <rect x="14" y="3" width="7" height="7" rx="1"/>
                                  <rect x="3" y="14" width="7" height="7" rx="1"/>
                                  <rect x="14" y="14" width="7" height="7" rx="1"/>
                                </svg>
                                <span class="lsp-dest-name">WordPress</span>
                                <span class="lsp-dest-sub">Media Library</span>
                                <span class="lsp-dest-check">
                                  <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                                </span>
                              </label>
                              
                              <?php if ($shopify_connected): ?>
                              <label class="lsp-dest-card <?php echo $shutterstock_sync_target === 'shopify' ? 'selected' : ''; ?>">
                                <input type="radio" name="lsp_shutterstock_sync_target" value="shopify" <?php checked($shutterstock_sync_target,'shopify'); ?> style="display:none;" />
                                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                  <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                                  <line x1="3" y1="6" x2="21" y2="6"/>
                                  <path d="M16 10a4 4 0 01-8 0"/>
                                </svg>
                                <span class="lsp-dest-name">Shopify</span>
                                <span class="lsp-dest-sub">Files</span>
                                <span class="lsp-dest-check">
                                  <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                                </span>
                              </label>
                              
                              <label class="lsp-dest-card <?php echo $shutterstock_sync_target === 'both' ? 'selected' : ''; ?>">
                                <input type="radio" name="lsp_shutterstock_sync_target" value="both" <?php checked($shutterstock_sync_target,'both'); ?> style="display:none;" />
                                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                  <circle cx="6" cy="6" r="3"/>
                                  <circle cx="18" cy="6" r="3"/>
                                  <circle cx="6" cy="18" r="3"/>
                                  <circle cx="18" cy="18" r="3"/>
                                  <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                                </svg>
                                <span class="lsp-dest-name">Both</span>
                                <span class="lsp-dest-sub">WP + Shopify</span>
                                <span class="lsp-dest-check">
                                  <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                                </span>
                              </label>
                              <?php else: ?>
                              <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:not-allowed;">
                                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                  <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                                  <line x1="3" y1="6" x2="21" y2="6"/>
                                  <path d="M16 10a4 4 0 01-8 0"/>
                                </svg>
                                <span class="lsp-dest-name">Shopify</span>
                                <span class="lsp-dest-sub">Not connected</span>
                              </label>
                              
                              <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:not-allowed;">
                                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                  <circle cx="6" cy="6" r="3"/>
                                  <circle cx="18" cy="6" r="3"/>
                                  <circle cx="6" cy="18" r="3"/>
                                  <circle cx="18" cy="18" r="3"/>
                                  <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                                </svg>
                                <span class="lsp-dest-name">Both</span>
                                <span class="lsp-dest-sub">Connect Shopify</span>
                              </label>
                              <?php endif; ?>
                            </div>
                            <span style="display:block;font-size:12px;color:var(--lsp-subtext, #64748b);margin-top:12px;">Use <strong>Sync Now</strong> to sync to destinations</span>
                          </div>

                          <!-- Progress Section -->
                          <div id="lsp-shutterstock-progress" style="display:none;margin-top:20px;padding:16px;background:#f1f5f9;border-radius:8px;">
                            <div style="display:flex;justify-content:space-between;margin-bottom:8px;">
                              <span id="lsp-shutterstock-progress-text">Syncing...</span>
                              <span id="lsp-shutterstock-progress-percent">0%</span>
                            </div>
                            <div style="height:8px;background:#e2e8f0;border-radius:4px;overflow:hidden;">
                              <div id="lsp-shutterstock-progress-bar" style="height:100%;background:#ee2d24;width:0%;transition:width 0.3s;"></div>
                            </div>
                          </div>
                        </div>
                      <?php endif; ?>
                    </div>
                  </div>
                  <aside class="help">
                    <h3>Shutterstock Integration</h3>
                    
                    <p>
                      Sync images you've already purchased with your Shutterstock subscription directly to WordPress or Shopify.
                    </p>
                    
                    <p style="margin-top:12px;"><strong>How it works:</strong></p>
                    <ul style="margin:6px 0 12px 16px;padding:0;font-size:13px;color:#64748b;">
                      <li style="margin-bottom:4px;">Connect your Shutterstock account via OAuth</li>
                      <li style="margin-bottom:4px;">Browse your license history</li>
                      <li>Select images and sync with one click</li>
                    </ul>
                    
                    <p><strong>Your licenses:</strong> Only images you've licensed appear here. LightSync Pro doesn't purchase images on your behalf.</p>
                    <p><strong>Supported formats:</strong> JPG (Shutterstock's standard delivery format)</p>
                    <p><strong>Tip:</strong> Images are automatically converted to WebP for optimal web performance.</p>
                    <p><strong>Synced badges:</strong> Green <span style="color:#10b981;">✓</span> badge = already imported to WordPress</p>
                  </aside>
                </div>
              </section>
            </div><!-- END lsp-shutterstock-content -->

            <!-- ====== AI GENERATE CONTENT ====== -->
            <div id="lsp-ai-content" class="lsp-source-content <?php echo ($active_source === 'ai') ? 'active' : ''; ?>">
              <section class="section">
                <div class="section-head">
                  <h3 class="lsp-card-title">AI Image Generation</h3>
                </div>
                <div class="twocol">
                  <div class="panel" style="flex:2">
                    <div class="lsp-card-body">
                      <?php if ( $openrouter_connected ) : ?>

                        <!-- Generate Form -->
                        <div id="lsp-ai-generate-form">
                          <div class="lsp-field-group" style="margin-bottom:16px">
                            <label for="lsp-ai-prompt" class="lsp-label">Describe what you want to create</label>
                            <textarea id="lsp-ai-prompt" class="lsp-textarea" rows="3" placeholder="A modern minimalist product photo of a ceramic coffee mug on a marble countertop, soft natural lighting, professional studio"></textarea>
                          </div>

                          <div style="display:flex;gap:12px;margin-bottom:16px;flex-wrap:wrap">
                            <div class="lsp-field-group" style="flex:1;min-width:200px">
                              <label for="lsp-ai-model" class="lsp-label">Model</label>
                              <select id="lsp-ai-model" class="lsp-select">
                                <option value="">Loading models...</option>
                              </select>
                            </div>
                            <div class="lsp-field-group" style="min-width:130px">
                              <label for="lsp-ai-aspect" class="lsp-label">Aspect Ratio</label>
                              <select id="lsp-ai-aspect" class="lsp-select">
                                <option value="1:1">1:1 Square</option>
                                <option value="16:9" selected>16:9 Wide</option>
                                <option value="9:16">9:16 Tall</option>
                                <option value="4:3">4:3 Standard</option>
                                <option value="3:4">3:4 Portrait</option>
                                <option value="3:2">3:2 Photo</option>
                              </select>
                            </div>
                          </div>

                          <div style="display:flex;gap:8px;align-items:center;margin-bottom:20px">
                            <button type="button" id="lsp-ai-generate-btn" class="btn primary" style="min-width:160px">
                              ✨ Generate Preview
                            </button>
                            <span id="lsp-ai-cost-label" style="display:none;font-size:11px;font-weight:700;padding:3px 8px;border-radius:6px;letter-spacing:.3px"></span>
                            <span id="lsp-ai-generating" style="display:none;color:#666">
                              <span class="spinner is-active" style="float:none;margin:0 4px 0 0"></span>
                              Generating...
                            </span>
                          </div>
                        </div>

                        <!-- Preview Area -->
                        <div id="lsp-ai-preview" style="display:none">
                          <div style="display:flex;gap:16px;flex-wrap:wrap;margin-bottom:16px">
                            <div id="lsp-ai-preview-image" style="flex:1;min-width:300px;border-radius:8px;overflow:hidden;background:#f0f0f0;min-height:200px;display:flex;align-items:center;justify-content:center">
                              <!-- Generated image renders here -->
                            </div>
                          </div>
                          <div style="display:flex;gap:8px;margin-bottom:16px">
                            <button type="button" id="lsp-ai-commit-btn" class="btn primary">
                              ✓ Use This — Save to Media Library
                            </button>
                            <button type="button" id="lsp-ai-regenerate-btn" class="btn ghost">
                              ↻ Regenerate
                            </button>
                            <button type="button" id="lsp-ai-discard-btn" class="btn ghost">
                              ✕ Discard
                            </button>
                          </div>
                        </div>

                        <!-- Commit Result -->
                        <div id="lsp-ai-committed" style="display:none">
                          <div class="lsp-notice lsp-notice-success" style="margin-bottom:16px">
                            <strong>✓ Saved to Media Library</strong>
                            <span id="lsp-ai-committed-details"></span>
                          </div>
                          <div style="display:flex;gap:8px">
                            <button type="button" id="lsp-ai-new-btn" class="btn primary">
                              + Generate Another
                            </button>
                            <a id="lsp-ai-edit-link" href="#" class="btn ghost" target="_blank">
                              View in Media Library →
                            </a>
                          </div>
                        </div>

                      <?php else : ?>
                        <!-- Not connected — show connect prompt -->
                        <div style="text-align:center;padding:40px 20px">
                          <div style="margin-bottom:16px">
                            <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="#dc2626" stroke-width="1.5" style="opacity:0.85">
                              <path d="M12 2a4 4 0 0 1 4 4v1a1 1 0 0 0 1 1h1a4 4 0 0 1 0 8h-1a1 1 0 0 0-1 1v1a4 4 0 0 1-8 0v-1a1 1 0 0 0-1-1H6a4 4 0 0 1 0-8h1a1 1 0 0 0 1-1V6a4 4 0 0 1 4-4z"/>
                              <circle cx="12" cy="12" r="2"/>
                            </svg>
                          </div>
                          <h3 style="margin:0 0 8px">AI Image Generation</h3>
                          <p style="color:#666;margin:0 0 24px;max-width:480px;margin-left:auto;margin-right:auto">
                            Generate product photos, hero images, blog graphics, and more with AI.
                            Connect to OpenRouter to get started.
                          </p>

                          <div style="text-align:center;margin-bottom:20px">
                            <a href="<?php echo esc_url( \LightSyncPro\OAuth\OpenRouterOAuth::auth_url() ); ?>" class="lsp-btn primary">
                              Connect OpenRouter
                            </a>
                            <p style="margin-top:8px;font-size:12px;color:#999;max-width:280px;margin-left:auto;margin-right:auto">
                              400+ AI models. Pay-per-use from $0.01/image — no subscriptions.
                              <a href="https://openrouter.ai" target="_blank">Learn more →</a>
                            </p>
                          </div>
                        </div>
                      <?php endif; ?>

                        <!-- AI Generated Assets Browser — ALWAYS shown (images are WordPress assets) -->
                        <div id="lsp-ai-browser" style="margin-top:24px;border-top:1px solid rgba(0,0,0,.06);padding-top:20px">
                          <div style="display:flex;align-items:center;gap:10px;margin-bottom:14px;flex-wrap:wrap">
                            <h4 style="margin:0;font-size:15px;font-weight:800;letter-spacing:.2px">AI Generated Images</h4>
                            <button type="button" class="btn ghost btn-sm" id="lsp-ai-select-all">Select All</button>
                            <button type="button" class="btn ghost btn-sm" id="lsp-ai-select-none">Select None</button>
                            <span id="lsp-ai-selection-count" style="color:var(--lsp-subtext, #64748b);font-size:12px;font-weight:600;margin-left:auto"></span>
                          </div>
                          <div id="lsp-ai-batch-bar" style="display:none;margin-bottom:14px;padding:10px 16px;background:rgba(255,87,87,.06);border:1px solid rgba(255,87,87,.15);border-radius:12px;align-items:center;gap:10px;flex-wrap:wrap">
                            <span id="lsp-ai-batch-count" style="font-size:13px;font-weight:700;color:var(--lsp-primary, #ff5757)"></span>
                            <span style="font-size:12px;color:var(--lsp-subtext, #64748b);margin-left:auto">Use <strong>Sync Now</strong> to sync to destinations</span>
                          </div>
                          <div id="lsp-ai-grid" class="lsp-grid" style="display:grid;grid-template-columns:repeat(auto-fill,minmax(150px,1fr));gap:10px">
                            <!-- Populated by JS -->
                          </div>
                          <div id="lsp-ai-grid-empty" style="display:none;text-align:center;padding:24px;color:var(--lsp-subtext, #666)">
                            No AI-generated images yet.<?php if ( !$openrouter_connected ) : ?> Connect OpenRouter above to start creating.<?php else : ?> Create your first one above!<?php endif; ?>
                          </div>
                        </div>

                        <!-- AI Sync Destination (below grid, matching Figma/Dropbox pattern) -->
                        <hr style="margin:20px 0;border:0;border-top:1px solid #e2e8f0;">
                        <label style="display:block;margin:0 0 10px;font-weight:600;">Sync Destination</label>
                        <?php $shopify_connected = (bool)self::get_opt('shopify_connected'); ?>
                        <div class="lsp-dest-cards" style="display:grid;grid-template-columns:repeat(auto-fit, minmax(100px, 1fr));gap:12px;max-width:560px;">
                          <label class="lsp-dest-card selected" data-ai-dest="wp">
                            <input type="radio" name="lsp_ai_dest" value="wp" checked style="display:none;" />
                            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                              <rect x="3" y="3" width="7" height="7" rx="1"/>
                              <rect x="14" y="3" width="7" height="7" rx="1"/>
                              <rect x="3" y="14" width="7" height="7" rx="1"/>
                              <rect x="14" y="14" width="7" height="7" rx="1"/>
                            </svg>
                            <span class="lsp-dest-name">WordPress</span>
                            <span class="lsp-dest-sub">Media Library</span>
                            <span class="lsp-dest-check">
                              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                            </span>
                          </label>
                          
                          <?php if ($shopify_connected): ?>
                          <label class="lsp-dest-card" data-ai-dest="shopify">
                            <input type="radio" name="lsp_ai_dest" value="shopify" style="display:none;" />
                            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                              <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                              <line x1="3" y1="6" x2="21" y2="6"/>
                              <path d="M16 10a4 4 0 01-8 0"/>
                            </svg>
                            <span class="lsp-dest-name">Shopify</span>
                            <span class="lsp-dest-sub">Files</span>
                            <span class="lsp-dest-check">
                              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                            </span>
                          </label>
                          
                          <label class="lsp-dest-card" data-ai-dest="both">
                            <input type="radio" name="lsp_ai_dest" value="both" style="display:none;" />
                            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                              <circle cx="6" cy="6" r="3"/>
                              <circle cx="18" cy="6" r="3"/>
                              <circle cx="6" cy="18" r="3"/>
                              <circle cx="18" cy="18" r="3"/>
                              <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                            </svg>
                            <span class="lsp-dest-name">Both</span>
                            <span class="lsp-dest-sub">WP + Shopify</span>
                            <span class="lsp-dest-check">
                              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M20 6L9 17l-5-5"/></svg>
                            </span>
                          </label>
                          <?php else: ?>
                          <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:pointer;" onclick="document.querySelector('#lsp-destinations').scrollIntoView({behavior:'smooth'})">
                            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                              <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                              <line x1="3" y1="6" x2="21" y2="6"/>
                              <path d="M16 10a4 4 0 01-8 0"/>
                            </svg>
                            <span class="lsp-dest-name">Shopify</span>
                            <span class="lsp-dest-sub">Click to connect</span>
                          </label>
                          
                          <label class="lsp-dest-card disabled" style="opacity:0.6;cursor:not-allowed;">
                            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                              <circle cx="6" cy="6" r="3"/>
                              <circle cx="18" cy="6" r="3"/>
                              <circle cx="6" cy="18" r="3"/>
                              <circle cx="18" cy="18" r="3"/>
                              <path d="M6 9v6M18 9v6M9 6h6M9 18h6"/>
                            </svg>
                            <span class="lsp-dest-name">Both</span>
                            <span class="lsp-dest-sub">Connect Shopify first</span>
                          </label>
                          <?php endif; ?>
                        </div><!-- .lsp-dest-cards -->
                    </div>
                  </div>

                  <!-- AI Sidebar -->
                  <aside class="help">
                    <h3>AI Image Generation</h3>

                    <p>
                      Generate product photos, hero images, blog graphics, and more with AI — then sync them to WordPress and Shopify.
                    </p>

                    <?php if ( $openrouter_connected ) : ?>
                      <p style="margin-top:12px;"><strong>How it works:</strong></p>
                      <ol style="margin:6px 0 12px 16px;padding:0;font-size:13px;color:#64748b;">
                        <li style="margin-bottom:4px;">Type a description of the image you need</li>
                        <li style="margin-bottom:4px;">Pick a model and click Generate Preview</li>
                        <li style="margin-bottom:4px;">Click Use This to save to your media library</li>
                        <li>Image is optimized (WebP) automatically</li>
                      </ol>

                      <p style="margin-top:12px;"><strong>Version History</strong></p>
                      <p style="font-size:13px;color:#64748b;">
                        Every time you regenerate an image, the previous version is backed up automatically. Click the expand icon on any image to view all past versions and restore any one with a single click. Your current version is also preserved before restoring, so you can always switch back.
                      </p>
                    <?php else : ?>
                      <p style="margin-top:12px;"><strong>Why connect?</strong></p>
                      <ul style="margin:6px 0 12px 16px;padding:0;font-size:13px;color:#64748b;">
                        <li style="margin-bottom:4px;"><strong>400+ models</strong> — Gemini, DALL-E, Flux, and more</li>
                        <li style="margin-bottom:4px;"><strong>Pay-per-use</strong> from $0.01/image — no subscriptions</li>
                        <li style="margin-bottom:4px;">Credentials encrypted &amp; secure</li>
                        <li>Sync AI images to Shopify too</li>
                      </ul>
                    <?php endif; ?>

                    <p><strong>Tip:</strong> Models from <a href="https://openrouter.ai/credits" target="_blank">OpenRouter</a> are pay-per-use — no subscriptions, credits never expire.</p>
                    <p><a href="<?php echo esc_url( $brand['docs_url'] ); ?>ai-image-generation/" target="_blank">AI generation guide →</a></p>
                  </aside>
                </div>
              </section>
            </div><!-- END lsp-ai-content -->

            <!-- ====== SYNC DESTINATIONS ====== -->
            <section id="lsp-destinations" class="section">
              <div class="section-head">
                <h3 class="lsp-card-title">Sync Destinations</h3>
              </div>
              <div class="twocol">
                <div class="panel">
                  <div class="lsp-card-body">
                    <?php $shopify_connected = (bool)self::get_opt('shopify_connected'); ?>
                    
                    <!-- WordPress - Always available -->
                    <div style="display:flex;align-items:center;gap:12px;padding:14px;background:rgba(34,197,94,0.08);border-radius:10px;margin-bottom:12px;">
                      <div style="width:40px;height:40px;background:#22c55e;border-radius:8px;display:flex;align-items:center;justify-content:center;flex-shrink:0;">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2">
                          <rect x="3" y="3" width="7" height="7" rx="1"/>
                          <rect x="14" y="3" width="7" height="7" rx="1"/>
                          <rect x="3" y="14" width="7" height="7" rx="1"/>
                          <rect x="14" y="14" width="7" height="7" rx="1"/>
                        </svg>
                      </div>
                      <div style="flex:1;">
                        <div style="font-weight:600;color:#166534;">WordPress Media Library</div>
                        <div style="font-size:13px;color:#15803d;">✓ Always available</div>
                      </div>
                    </div>

                    <!-- Shopify Connection -->
                    <div id="lsp-shopify-global-box">
                      <?php if ($shopify_connected): ?>
                      <div id="lsp-shopify-global-connected" style="display:flex;align-items:center;gap:12px;padding:14px;background:rgba(168,85,247,0.08);border-radius:10px;">
                        <div style="width:40px;height:40px;background:#a855f7;border-radius:8px;display:flex;align-items:center;justify-content:center;flex-shrink:0;">
                          <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2">
                            <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                            <line x1="3" y1="6" x2="21" y2="6"/>
                            <path d="M16 10a4 4 0 01-8 0"/>
                          </svg>
                        </div>
                        <div style="flex:1;">
                          <div style="font-weight:600;color:#7c3aed;">Shopify Files</div>
                          <div style="font-size:13px;color:#9333ea;">✓ Connected — <?php echo esc_html(self::get_opt('shopify_shop_domain') ?: 'your store'); ?></div>
                        </div>
                        <button type="button" id="lsp-shopify-global-disconnect" class="btn ghost btn-sm" style="color:#dc2626;">Disconnect</button>
                      </div>
                      <?php else: ?>
                      <div id="lsp-shopify-global-disconnected" style="display:flex;align-items:center;gap:12px;padding:14px;background:rgba(0,0,0,0.03);border-radius:10px;">
                        <div style="width:40px;height:40px;background:#e5e7eb;border-radius:8px;display:flex;align-items:center;justify-content:center;flex-shrink:0;">
                          <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#9ca3af" stroke-width="2">
                            <path d="M6 2L3 6v14a2 2 0 002 2h14a2 2 0 002-2V6l-3-4z"/>
                            <line x1="3" y1="6" x2="21" y2="6"/>
                            <path d="M16 10a4 4 0 01-8 0"/>
                          </svg>
                        </div>
                        <div style="flex:1;">
                          <div style="font-weight:600;color:#374151;">Shopify Files</div>
                          <div style="font-size:13px;color:#6b7280;">Sync photos to your Shopify store</div>
                        </div>
                        <button type="button" id="lsp-shopify-global-connect" class="btn primary btn-sm">Connect Shopify</button>
                      </div>
                      <?php endif; ?>
                    </div>

                    <p style="margin-top:12px;font-size:12px;color:#64748b;">
                      Choose where to sync in each source tab. Connect Shopify to unlock Shopify and Both destination options.
                    </p>
                  </div>
                </div>
                <aside class="help">
                  <h3>Sync Destinations</h3>
                  <p>
                    <?php echo esc_html( $brand['name'] ); ?> can sync your photos to WordPress Media Library, Shopify Files, or both at once.
                  </p>
                  <p><strong>WordPress:</strong> Always available. Photos sync to your Media Library with full metadata.</p>
                  <p><strong>Shopify:</strong> Connect your store to sync photos directly to Shopify Files for use in products and themes.</p>
                  <p><a href="<?php echo esc_url( $brand['docs_url'] ); ?>shopify-integration/" target="_blank">Shopify integration guide →</a></p>
                </aside>
              </div>
            </section>

             <?php $this->render_recent_activity(); ?>

              <section id="lsp-danger-zone" class="section">
                <div class="twocol">
                  <div class="panel">
                 <div class="lsp-card-body">
              <p style="margin-top:12px;">
  <label class="lsp-toggle">
    <input type="checkbox"
           name="<?php echo esc_html(self::OPT); ?>[wipe_on_deactivate]"
           value="1"
           <?php checked( ! empty( $o['wipe_on_deactivate'] ) ); ?>>
    <span class="track"><span class="thumb"></span></span>
    <span>Delete all <?php echo esc_html( $brand['name'] ); ?> data when the plugin is deactivated</span>
  </label>
  <p class="description">
  When enabled, deactivating <?php echo esc_html( $brand['name'] ); ?> will remove its settings, catalog/album selections,
  usage counters, and cron schedules from the WordPress database. Use with caution.
</p>
</p>
<hr/>
<p style="display:flex;gap:12px;flex-wrap:wrap;">
    <button type="button" class="btn ghost" id="lsp-disconnect-adobe">
        Disconnect from Adobe
    </button>
    <?php if (\LightSyncPro\OAuth\CanvaOAuth::is_connected()): ?>
    <button type="button" class="btn ghost" id="lsp-canva-disconnect" style="color:#dc2626;border-color:#fecaca;">
        Disconnect from Canva
    </button>
    <?php endif; ?>
    <?php if (\LightSyncPro\OAuth\FigmaOAuth::is_connected()): ?>
    <button type="button" class="btn ghost" id="lsp-figma-disconnect" style="color:#dc2626;border-color:#fecaca;">
        Disconnect from Figma
    </button>
    <?php endif; ?>
    <?php if (\LightSyncPro\OAuth\DropboxOAuth::is_connected()): ?>
    <button type="button" class="btn ghost" id="lsp-dropbox-disconnect" style="color:#dc2626;border-color:#fecaca;">
        Disconnect from Dropbox
    </button>
    <?php endif; ?>
    <?php if (\LightSyncPro\OAuth\ShutterstockOAuth::is_connected()): ?>
    <button type="button" class="btn ghost" id="lsp-shutterstock-disconnect" style="color:#dc2626;border-color:#fecaca;">
        Disconnect from Shutterstock
    </button>
    <?php endif; ?>
    <?php if (\LightSyncPro\OAuth\OpenRouterOAuth::is_connected()): ?>
    <button type="button" class="btn ghost" id="lsp-ai-disconnect-btn" style="color:#dc2626;border-color:#fecaca;">
        Disconnect from OpenRouter
    </button>
    <?php endif; ?>
</p>
<small>Use this to reset your connections if you see token errors.</small>
</div>
</div>
 <aside class="help">
              <h3>Disconnect Sources</h3>
              <p>
               Disconnecting stops <?php echo esc_html( $brand['name'] ); ?> from syncing with Adobe Lightroom, Canva, Figma, Dropbox, Shutterstock, or OpenRouter AI. Existing images on your site are not removed or changed. You can reconnect at any time to resume syncing.
              </p>
              <p><a href="<?php echo esc_url( $brand['docs_url'] ); ?>how-to-connect-to-adobe-lightroom/" target="_blank">Reconnection guide →</a></p>
            </aside>
</div>
</section>

          </div>
        </main>
      </div>
<p class="description-pending">
 U.S. Patent Pending
</p>
    </div>
        <?php
    }

    private function redirect_uri(){ return home_url('/wp-json/lightsyncpro/v1/oauth-callback'); }
    private function auth_url(){ return OAuth::auth_url( $this->redirect_uri() ); }

    private static function license_ok(){ return true; }

    public function admin_notice_unlicensed(){
    }

    public function maybe_validate_license(){
    }

    private function activate_license_if_possible(){
    }

    public function ajax_save_options(){
        if ( ! check_ajax_referer(self::AJAX_NS.'_nonce','_ajax_nonce', false) ) {
            wp_send_json_error(['error'=>'bad_nonce'], 403);
        }
        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error(['error'=>'forbidden'], 403);
        }

        ob_start();

        $incoming_raw = isset( $_POST[ self::OPT ] ) ? array_map( 'sanitize_text_field', wp_unslash( (array) $_POST[ self::OPT ] ) ) : [];
        $sanitized    = self::sanitize_options_merge($incoming_raw);

        update_option(self::OPT, $sanitized, false);

        $junk = ob_get_clean();
        if ($junk) {
            Logger::debug('[LSP ajax_save_options] stray output: ' . trim($junk));
        }

        wp_send_json_success(['saved'=>true]);
    }

    /**
     * AJAX handler for saving Weekly Digest settings
     */
    public static function ajax_save_digest(){
        if ( ! check_ajax_referer(self::AJAX_NS.'_nonce','_ajax_nonce', false) ) {
            wp_send_json_error(['error'=>'bad_nonce'], 403);
        }
        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error(['error'=>'forbidden'], 403);
        }

        $digest_enabled = isset($_POST['weekly_digest_enabled']) ? (int) $_POST['weekly_digest_enabled'] : 0;
        $digest_email = isset($_POST['weekly_digest_email']) ? sanitize_email($_POST['weekly_digest_email']) : '';
        $hub_digest_enabled = isset($_POST['hub_digest_enabled']) ? (int) $_POST['hub_digest_enabled'] : 0;
        $hub_digest_email = isset($_POST['hub_digest_email']) ? sanitize_email($_POST['hub_digest_email']) : '';

        // Use admin email if no email provided
        if (empty($digest_email)) {
            $digest_email = get_option('admin_email');
        }

        self::set_opt([
            'weekly_digest_enabled' => $digest_enabled,
            'weekly_digest_email'   => $digest_email,
            'hub_digest_enabled'    => $hub_digest_enabled,
            'hub_digest_email'      => $hub_digest_email,
        ]);

        wp_send_json_success([
            'saved' => true,
            'weekly_digest_enabled' => $digest_enabled,
            'weekly_digest_email'   => $digest_email,
            'hub_digest_enabled'    => $hub_digest_enabled,
            'hub_digest_email'      => $hub_digest_email,
        ]);
    }

    /**
     * AJAX handler for saving AI settings
     */
    public static function ajax_save_ai(){
        if ( ! check_ajax_referer(self::AJAX_NS.'_nonce','_ajax_nonce', false) ) {
            wp_send_json_error(['error'=>'bad_nonce'], 403);
        }
        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error(['error'=>'forbidden'], 403);
        }

        $performance = isset($_POST['ai_performance_optimize']) ? (int) $_POST['ai_performance_optimize'] : 0;
        $provider = isset($_POST['ai_provider']) ? sanitize_text_field($_POST['ai_provider']) : '';
        $anthropic_key = isset($_POST['ai_anthropic_key']) ? sanitize_text_field($_POST['ai_anthropic_key']) : '';
        $openai_key = isset($_POST['ai_openai_key']) ? sanitize_text_field($_POST['ai_openai_key']) : '';
        $auto_alt = isset($_POST['ai_auto_alt']) ? (int) $_POST['ai_auto_alt'] : 0;
        $retention_days = isset($_POST['tracking_retention_days']) ? (int) $_POST['tracking_retention_days'] : null;

        // Only allow valid providers
        if ($provider && !in_array($provider, ['anthropic', 'openai'], true)) {
            $provider = '';
        }

        $settings = [
            'ai_performance_optimize' => $performance,
            'ai_provider' => $provider,
            'ai_anthropic_key' => $anthropic_key,
            'ai_openai_key' => $openai_key,
            'ai_auto_alt' => $auto_alt,
        ];

        // Only update retention if explicitly provided
        if ($retention_days !== null) {
            $settings['tracking_retention_days'] = $retention_days;
        }

        self::set_opt($settings);

        wp_send_json_success([
            'saved' => true,
            'performance' => $performance,
            'provider' => $provider,
            'auto_alt' => $auto_alt,
            'retention_days' => $retention_days,
        ]);
    }

    public function ajax_diag(){
        check_ajax_referer(self::AJAX_NS.'_nonce','_ajax_nonce');
        if(!current_user_can('manage_options')) wp_send_json_error('forbidden',403);

        $t = \LightSyncPro\OAuth\OAuth::ensure_token();
        if (is_wp_error($t)) wp_send_json_error($t->get_error_message());

        $u = \LightSyncPro\Http\Client::get('https://lr.adobe.io/v2/user', 20);
        $user = is_wp_error($u) ? ['error'=>$u->get_error_message()] : \LightSyncPro\Util\Adobe::decode(wp_remote_retrieve_body($u));

        $c = \LightSyncPro\Http\Client::get('https://lr.adobe.io/v2/catalog', 20);
        $cat = is_wp_error($c) ? ['error'=>$c->get_error_message(),'body'=>wp_remote_retrieve_body($c)]
                               : \LightSyncPro\Util\Adobe::decode(wp_remote_retrieve_body($c));

        wp_send_json_success([
            'headers_sent' => \LightSyncPro\OAuth\OAuth::headers(),
            'user'         => $user,
            'catalogs_raw' => is_wp_error($c) ? wp_remote_retrieve_body($c) : null,
            'catalogs'     => $cat,
        ]);
    }

    public function ajax_list() {
        check_ajax_referer(self::AJAX_NS.'_nonce','_ajax_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('forbidden', 403);
        }

        Logger::debug('[LSP] Catalog URL: ' . \LightSyncPro\Http\Endpoints::catalogs());

        $cats = Sync::get_catalogs();
        if (is_wp_error($cats)) {
            $code = $cats->get_error_code();
            $msg  = $cats->get_error_message();

            $needsReconnect = in_array($code, [
                'oauth_failed',
                'invalid_token',
                'expired_token',
                'unauthorized',
            ], true);

            if ($needsReconnect) {
                if (method_exists('\LightSyncPro\OAuth\OAuth', 'disconnect')) {
                    \LightSyncPro\OAuth\OAuth::disconnect();
                }

                wp_send_json_error([
                    'message'       => $msg ?: 'Your Adobe connection has expired. Please reconnect.',
                    'needReconnect' => true,
                    'helpUrl'       => 'https://lightsyncpro.com/docs/how-to-connect-to-adobe-lightroom/',
                ]);
            }

            wp_send_json_error([
                'message' => $msg ?: 'Failed to load Lightroom catalogs.',
                'helpUrl' => 'https://lightsyncpro.com/docs/not-connected-error/',
            ]);
        }

        $selCat    = (string) self::get_opt('catalog_id', '');
        $selAlbums = (array)  self::get_opt('album_ids', []);

        $catalogList = (array) ($cats['catalogs'] ?? []);

        $allCatalogIds = array_map(
            static fn($c) => (string) ($c['id'] ?? ''),
            $catalogList
        );

        if (
            count($catalogList) === 1
            || $selCat === ''
            || !in_array($selCat, $allCatalogIds, true)
        ) {
            if (!empty($catalogList[0]['id'])) {
                $selCat = (string) $catalogList[0]['id'];
                self::set_opt(['catalog_id' => $selCat]);
            }
        }

        $out = [
            'catalogs' => [],
            'selected' => [
                'catalog' => $selCat,
                'albums'  => $selAlbums,
            ],
        ];

        foreach ($catalogList as $c) {
            $cid = (string) ($c['id'] ?? '');

            $albums = Sync::get_albums($cid);

            $alist = [];
            $name_map = [];

            if (!is_wp_error($albums)) {
                $resources = (array) ($albums['resources'] ?? []);
                
                foreach ($resources as $a) {
                    $aid  = (string) ($a['id'] ?? '');
                    $name = (string) ($a['payload']['name'] ?? '');
                    
                    // Album's last updated timestamp from Lightroom
                    $album_updated = (string) ($a['updated'] ?? '');
                    
                    // Cover asset ID can be in different locations depending on API version
                    $cover_id = '';
                    if (!empty($a['payload']['cover']['id'])) {
                        $cover_id = (string) $a['payload']['cover']['id'];
                    } elseif (!empty($a['payload']['cover']['asset']['id'])) {
                        $cover_id = (string) $a['payload']['cover']['asset']['id'];
                    } elseif (!empty($a['payload']['coverAsset'])) {
                        $cover_id = (string) $a['payload']['coverAsset'];
                    }
                    
                    // Check if cover is cached (by album_id now, since we fallback to first asset)
                    $cover_url = '';
                    if ($aid !== '') {
                        $cache_key = 'lsp_cover_' . md5($cid . '_' . $aid);
                        $cached = get_transient($cache_key);
                        if ($cached !== false) {
                            $cover_url = $cached;
                        }
                    }

                    $alist[] = [
                        'id'        => $aid,
                        'name'      => ($name !== '' ? $name : '(no name)'),
                        'cover_id'  => $cover_id,
                        'cover_url' => $cover_url,
                        'updated'   => $album_updated ? strtotime($album_updated) : null,
                    ];

                    if ($aid !== '' && $name !== '') {
                        $name_map[$aid] = $name;
                    }
                }
            }

            if ($cid !== '' && !empty($name_map)) {
                self::set_album_name_cache($cid, $name_map);
            }

            $catName = trim((string) ($c['name'] ?? ''));
            if ($catName === '' || $catName === '(no name)') {
                $catName = 'Lightroom Catalog';
            }

            $out['catalogs'][] = [
                'id'     => $cid,
                'name'   => $catName,
                'albums' => $alist,
            ];

            if ($cid === $selCat && empty($selAlbums) && count($alist) === 1 && !empty($alist[0]['id'])) {
                $selAlbums = [ (string) $alist[0]['id'] ];
                $out['selected']['albums'] = $selAlbums;
                self::set_opt(['album_ids' => $selAlbums]);
            }
        }

        $out['selected']['catalog'] = $selCat;

        wp_send_json_success($out);
    }

    public function ajax_save_selection() {
        if ( ! check_ajax_referer(self::AJAX_NS . '_nonce', '_ajax_nonce', false) ) {
            wp_send_json_error(['error' => 'bad_nonce'], 403);
        }
        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error(['error' => 'forbidden'], 403);
        }

        ob_start();

        $catalog = isset($_POST['catalog'])
            ? sanitize_text_field( wp_unslash($_POST['catalog']) )
            : self::get_opt('catalog_id', '');

        $albums = [];
        if ( isset($_POST['albums']) ) {
            $raw = (array) wp_unslash($_POST['albums']);
            foreach ($raw as $v) {
                $v = sanitize_text_field($v);
                if ($v !== '') $albums[] = (string) $v;
            }
        }

        $schedules = [];
        if ( isset($_POST['schedules']) ) {
            $raw = (array) wp_unslash($_POST['schedules']);
            foreach ($raw as $k => $v) {
                $aid = sanitize_text_field((string) $k);
                $val = sanitize_text_field((string) $v);
                if ($aid !== '') {
                    $schedules[$aid] = $val;
                }
            }
        } else {
            $raw = (array) self::get_opt('album_schedule', []);
            foreach ($raw as $k => $v) {
                $aid = sanitize_text_field((string) $k);
                $val = sanitize_text_field((string) $v);
                if ($aid !== '') $schedules[$aid] = $val;
            }
        }

        $plan   = self::plan();
        $matrix = self::plan_matrix();
        $maxAlbums = (int) ($matrix[$plan]['max_albums'] ?? 1);

        if (count($albums) > $maxAlbums) {
            $albums = array_slice($albums, 0, $maxAlbums);
        }

        if ( empty($matrix[$plan]['autosync']) || ! self::has_cap('autosync') ) {
            foreach ($schedules as $aid => $v) {
                $schedules[$aid] = 'off';
            }
        }

        self::set_opt([
            'catalog_id'     => $catalog,
            'album_ids'      => $albums,
            'album_schedule' => $schedules,
        ]);

        $this->reschedule_album_crons();

        $junk = ob_get_clean();
        if ($junk) {
            Logger::debug('[LSP ajax_save_selection] stray output: ' . trim($junk));
        }

        wp_send_json_success([
            'catalog' => $catalog,
            'albums'  => $albums,
            'saved'   => true,
        ]);
    }

    public function reschedule_album_crons(): void {
        if ( ! self::has_cap('autosync') ) {
            $this->unschedule_all_album_events();
            return;
        }
        $cat      = self::get_opt('catalog_id','');
        $albums   = (array) self::get_opt('album_ids', []);
        $schedule = (array) self::get_opt('album_schedule', []);

        $albums = array_values(array_filter(array_map('strval', $albums)));

        $this->unschedule_album_events_not_in($albums);

        if (!$cat || empty($albums)) {
            return;
        }

        $wp_schedules = wp_get_schedules();

        foreach ($albums as $album_id) {
            $freq = $schedule[$album_id] ?? 'off';
            if ($freq === 'off') {
                while ($ts = wp_next_scheduled(self::CRON_ALBUM, [$album_id])) {
                    wp_unschedule_event($ts, self::CRON_ALBUM, [$album_id]);
                }
                continue;
            }

            $slug = $this->freq_to_schedule_slug($freq);
            if (empty($slug) || !isset($wp_schedules[$slug])) {
                $slug = 'daily';
            }

            $event = function_exists('wp_get_scheduled_event')
                ? wp_get_scheduled_event(self::CRON_ALBUM, [$album_id])
                : null;

            if ($event && !empty($event->schedule) && $event->schedule !== $slug) {
                wp_unschedule_event($event->timestamp, self::CRON_ALBUM, [$album_id]);
                $event = null;
            }

            if (!$event && !wp_next_scheduled(self::CRON_ALBUM, [$album_id])) {
                wp_schedule_event(time() + 60, $slug, self::CRON_ALBUM, [$album_id]);
            }
        }
    }

    public function ajax_estimate(){
        check_ajax_referer(self::AJAX_NS.'_nonce','_ajax_nonce');

        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error('forbidden', 403);
        }

        $catalog = isset($_POST['catalog'])
            ? sanitize_text_field( wp_unslash($_POST['catalog']) )
            : self::get_opt('catalog_id', '');

        $albums = isset( $_POST['albums'] )
            ? array_map( 'sanitize_text_field', wp_unslash( (array) $_POST['albums'] ) )
            : (array) self::get_opt( 'album_ids', [] );

        $albums = array_values( array_filter( $albums ) );
        $albums = array_unique( $albums );

        if ( ! $catalog || empty( $albums ) ) {
            wp_send_json_error( 'Missing catalog/albums', 400 );
        }

        $unique_mode = isset( $_POST['unique'] ) && '1' === sanitize_text_field( wp_unslash( $_POST['unique'] ) );

        $cache_key = 'lightsync_est_'. md5( $catalog . '|' . implode(',', $albums) . '|u=' . ($unique_mode ? '1' : '0') );
        $cached = get_transient($cache_key);
        if (is_array($cached) && isset($cached['total'])) {
            wp_send_json_success($cached);
        }

        $have_count_members = method_exists('\LightSyncPro\Sync\Sync','count_album_members');
        $per    = [];
        $sum    = 0;
        $errors = [];

        $unique_ids = $unique_mode ? [] : null;

        $per_album_deadline_ms = 12000;
        $t0 = microtime(true);

        foreach ($albums as $aid) {
            $album_start = microtime(true);
            try {
                if ($unique_mode) {
                    $cursor = null;
                    $seen   = [];
                    $count  = 0;

                    for ($i = 0; $i < 2000; $i++) {
                        if ( (microtime(true) - $album_start) * 1000 > $per_album_deadline_ms ) {
                            $errors[$aid] = 'timeout';
                            break;
                        }

                        $page = \LightSyncPro\Sync\Sync::get_album_assets($catalog, $aid, $cursor, 200);
                        if (is_wp_error($page)) {
                            $errors[$aid] = $page->get_error_message();
                            break;
                        }

                        $resources = (array) ($page['resources'] ?? []);
                        if (!$resources && !$cursor) break;

                        foreach ($resources as $res) {
                            $key = \LightSyncPro\Sync\Sync::extract_asset_id_from_resource($res);
                            if (!$key) continue;
                            if (isset($seen[$key])) continue;
                            $seen[$key] = 1;

                            $unique_ids[$key] = 1;
                            $count++;
                        }

                        $cursor = \LightSyncPro\Sync\Sync::derive_next_cursor($page);
                        if (!$cursor) break;
                    }

                    $per[$aid] = $count;
                } else {
                    if ($have_count_members) {
                        $n = \LightSyncPro\Sync\Sync::count_album_members($catalog, $aid);
                    } else {
                        $n = \LightSyncPro\Sync\Sync::estimate_album($catalog, $aid);
                    }
                    if (is_wp_error($n)) {
                        $errors[$aid] = $n->get_error_message();
                        $per[$aid] = 0;
                    } else {
                        $cnt = (int) $n;
                        $per[$aid] = $cnt;
                        $sum += $cnt;
                    }
                }
            } catch (\Throwable $e) {
                $errors[$aid] = $e->getMessage();
                $per[$aid] = $per[$aid] ?? 0;
            }
        }

        $total = $unique_mode ? count($unique_ids ?: []) : $sum;

        $payload = [
            'total'     => (int) $total,
            'per'       => $per,
            'unique'    => $unique_mode ? 1 : 0,
            'summary'   => sprintf(
                $unique_mode
                    ? 'Unique images across %d album(s): %d'
                    : 'Total across %d album(s): %d',
                count($albums),
                (int)$total
            ),
            'errors'    => $errors,
            'elapsed_ms'=> (int) round((microtime(true) - $t0) * 1000),
        ];

        set_transient($cache_key, $payload, 60);

        wp_send_json_success($payload);
    }

    public function ajax_batch() {
        if ( ! check_ajax_referer(self::AJAX_NS.'_nonce', '_ajax_nonce', false) ) {
            wp_send_json_error(['error' => 'bad_nonce'], 403);
        }
        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error(['error' => 'forbidden'], 403);
        }

        // Sync gate check
        $gate = lsp_gate_check();
        if ( is_wp_error( $gate ) ) {
            wp_send_json_error([
                'error' => 'sync_disabled',
                'message' => $gate->get_error_message(),
            ], 403);
        }

        $tok = \LightSyncPro\OAuth\OAuth::maybe_refresh();
        if (is_wp_error($tok)) {
            wp_send_json_error([
                'error' => 'token_expired',
                'message' => 'Adobe token expired. Please reconnect.',
                'needReconnect' => true,
                'helpUrl' => 'https://lightsyncpro.com/docs/how-to-connect-to-adobe-lightroom/',
            ], 401);
        }

        $background = isset($_POST['background']) && $_POST['background'] === '1';

        if ($background) {
            if (ob_get_level()) {
                ob_clean();
            }
            
            $cat = '';
            if (isset($_POST['catalog'])) {
                $cat = sanitize_text_field(wp_unslash($_POST['catalog']));
            }
            if (!$cat) {
                $cat = (string) self::get_opt('catalog_id', '');
            }

            $albums = [];
            if (isset($_POST['albums']) && is_array($_POST['albums'])) {
                $albums = array_map('sanitize_text_field', wp_unslash($_POST['albums']));
            }
            if (empty($albums)) {
                $albums = (array) self::get_opt('album_ids', []);
            }
            $albums = array_values(array_filter($albums));

            if (!$cat || empty($albums)) {
                wp_send_json_error(['error' => 'Missing catalog or albums'], 400);
            }

            foreach ($albums as $album_id) {
                delete_option("lightsync_sync_cursor_{$cat}_{$album_id}");
                delete_option("lightsync_sync_next_index_{$cat}_{$album_id}");
                delete_transient("lightsync_tick_lock_{$cat}_{$album_id}");
            }

            update_option('lightsync_background_sync_queue', [
                'catalog'       => $cat,
                'albums'        => $albums,
                'current_index' => 0,
                'started_at'    => time(),
                'source'        => 'manual-background',
            ], false);

            self::add_activity(
                sprintf('Lightroom → WordPress Sync Started (Background): %d album(s)', count($albums)),
                'info',
                'manual-background'
            );

            wp_send_json_success([
                'background'     => true,
                'albums_queued'  => count($albums),
                'message'        => sprintf(
                    'Background sync started for %d album(s). This will run in the background.',
                    count($albums)
                ),
            ]);
            return;
        }

        ob_start();

        $cat = '';
        if (isset($_POST['catalog'])) {
            $cat = sanitize_text_field(wp_unslash($_POST['catalog']));
        } elseif (isset($_POST['catalog_id'])) {
            $cat = sanitize_text_field(wp_unslash($_POST['catalog_id']));
        }

        $albums = [];
        if (isset($_POST['albums']) && is_array($_POST['albums'])) {
            $albums = array_map('sanitize_text_field', wp_unslash($_POST['albums']));
        } elseif (isset($_POST['albums']) && !is_array($_POST['albums'])) {
            $albums = [ sanitize_text_field(wp_unslash($_POST['albums'])) ];
        }

        $album_id = '';
        if (isset($_POST['album_id'])) {
            $album_id = sanitize_text_field(wp_unslash($_POST['album_id']));
        }

        $album_index = 0;
        if ( isset( $_POST['album_index'] ) ) {
            $album_index = absint( $_POST['album_index'] );
        } elseif ( isset( $_POST['index'] ) ) {
            $album_index = absint( $_POST['index'] );
        }

        $cursor = '';
        if (isset($_POST['cursor'])) {
            $cursor = esc_url_raw(wp_unslash($_POST['cursor']));
        }

        $start_index = 0;
        if ( isset( $_POST['start_index'] ) ) {
            $start_index = absint( $_POST['start_index'] );
        } elseif ( isset( $_POST['next_index'] ) ) {
            $start_index = absint( $_POST['next_index'] );
        }

        $source = isset($_POST['source']) ? sanitize_text_field(wp_unslash($_POST['source'])) : 'Manual Sync';

        $settings  = self::get_opt();
        $saved_cat = (string) self::get_opt('catalog_id');

        if (!$cat) {
            $cat = $saved_cat;
        }

        if (empty($albums)) {
            $album_ids = (isset($settings['album_ids']) && is_array($settings['album_ids']))
                ? array_values($settings['album_ids'])
                : [];
            $albums = $album_ids;
        }

        if (!$album_id) {
            if (!empty($albums[$album_index])) {
                $album_id = (string) $albums[$album_index];
            }
        }

        if (!$cat || !$album_id) {
            $junk = ob_get_clean();
            wp_send_json_error([
                'error' => 'missing_catalog_or_album',
                'junk'  => $junk,
            ], 400);
        }

        $batchSz = (int) self::get_opt('batch_size', 60);
        $limit   = 200;
        $remaining = self::usage_remaining_month();

        // Read sync_target from POST if provided, otherwise fall back to saved option
        $target = isset($_POST['sync_target']) ? sanitize_text_field($_POST['sync_target']) : (string)(self::get_opt('sync_target') ?? 'wp');

        if ($target === 'hub') {
            // Hub uses fetch_assets - we'll push at album completion
            $out = \LightSyncPro\Sync\Sync::fetch_assets(
                $cat,
                $album_id,
                $cursor,
                $batchSz,
                $start_index,
                $remaining
            );
        } elseif ($target === 'shopify') {
            // Shopify-only: just fetch, don't import to WP
            $out = \LightSyncPro\Sync\Sync::fetch_assets(
                $cat,
                $album_id,
                $cursor,
                $batchSz,
                $start_index,
                $remaining
            );
        } elseif ($target === 'both') {
            // Both mode: fetch raw data first (for Shopify), then import to WP
            $fetch_result = \LightSyncPro\Sync\Sync::fetch_assets(
                $cat,
                $album_id,
                $cursor,
                $batchSz,
                $start_index,
                $remaining
            );

            $raw_assets_for_shopify = is_array($fetch_result['assets'] ?? null) ? $fetch_result['assets'] : [];

            // Now import to WordPress
            $out = \LightSyncPro\Sync\Sync::batch_import(
                $cat,
                $album_id,
                $cursor,
                $batchSz,
                $limit,
                $start_index,
                $remaining
            );

            // Preserve raw assets for Shopify push
            if (!is_wp_error($out)) {
                $out['assets'] = $raw_assets_for_shopify;
            }
        } else {
            $out = \LightSyncPro\Sync\Sync::batch_import(
                $cat,
                $album_id,
                $cursor,
                $batchSz,
                $limit,
                $start_index,
                $remaining
            );
        }

        $processed = 0;

        if ($target === 'hub' || $target === 'shopify') {
            $processed = 0;
        } else {
            $imported_count = is_array($out['imported'] ?? null) ? count($out['imported']) : 0;
            $updated_count = is_array($out['updated'] ?? null) ? count($out['updated']) : 0;
            $processed = $imported_count + $updated_count;
        }

        $remaining = self::usage_remaining_month();
        $caps = self::usage_caps();
        $cap = (int)$caps['month'];

        if ($cap > 0 && $remaining > 0 && $remaining < ($cap * 0.1)) {
            $out['warning_low_quota'] = sprintf(
                'Only %d images remaining this month. Upgrade to continue uninterrupted.',
                $remaining
            );
            $out['upgrade_url'] = 'https://lightsyncpro.com/pricing';
        }

        if ($processed > 0) {
            $usageResult = self::usage_consume($processed);
            if (!$usageResult['ok']) {
                $out['hit_cap'] = true;
                $out['done_all'] = true;
                $used = (int)($usageResult['used'] ?? 0);
                $cap = (int)($usageResult['cap'] ?? 0);
                $out['error_message'] = sprintf(
                    'Monthly image limit reached (%d/%d images). Upgrade your plan to continue syncing.',
                    $used,
                    $cap
                );
                $out['upgrade_url'] = 'https://lightsyncpro.com/pricing';
            } else {
                $out['usage'] = $usageResult['after'] ?? null;
                $out['remaining_month'] = self::usage_remaining_month();
            }
        } else {
            $out['usage'] = self::usage_get();
            $out['remaining_month'] = self::usage_remaining_month();
        }

        if (is_wp_error($out)) {
            $junk = ob_get_clean();
            wp_send_json_error([
                'error' => $out->get_error_message(),
                'junk'  => $junk,
            ], 500);
        }

        $junk = ob_get_clean();

        $next_cursor = (string) ($out['cursor'] ?? $out['next_cursor'] ?? '');
        $next_index  = (int)    ($out['next_index'] ?? 0);
        $hit_cap     = !empty($out['hit_cap']);

        $hasMoreThisAlbum = ($next_cursor !== '' || $next_index > 0 || $hit_cap);
        $finalAlbum  = ($album_index >= (count($albums) - 1));
        $nextAlbumIdx = $hasMoreThisAlbum ? $album_index : ($album_index + 1);
        $doneAll = (!$hasMoreThisAlbum && $finalAlbum);

        if (!$hasMoreThisAlbum && !$doneAll) {
            $next_cursor = '';
            $next_index  = 0;
        }

        $payload = is_array($out) ? $out : [];
        $payload['junk']       = $junk;
        $payload['catalog_id'] = $cat;
        $payload['album_id']   = $album_id;
        $payload['source']     = $source;
        $payload['cursor']      = $next_cursor;
        $payload['next_cursor'] = $next_cursor;
        $payload['next_index']  = $next_index;
        $payload['index']    = $nextAlbumIdx;
        $payload['done_all'] = (bool) $doneAll;

        try {
            $imp  = is_array($out['imported'] ?? null) ? count($out['imported']) : 0;
            $upd  = is_array($out['updated'] ?? null) ? count($out['updated']) : 0;
            $meta = is_array($out['meta_updated'] ?? null) ? count($out['meta_updated']) : 0;
            $skp  = (int)($out['skipped'] ?? 0);
            $fetched = is_array($out['assets'] ?? null) ? count($out['assets']) : 0;

            $albumFinished = (
                (($out['cursor'] ?? '') === '') &&
                ((int)($out['next_index'] ?? 0) === 0)
            );

            self::set_opt([
                'lightsync_last_sync_heartbeat_ts' => time(),
                'lightsync_last_sync_source'       => (string)$source,
                'lightsync_last_sync_status'       => 'running',
            ]);

            if ($doneAll) {
                self::set_opt([
                    'lightsync_last_sync_ts'     => time(),
                    'lightsync_last_sync_status' => 'complete',
                    'lightsync_last_sync_counts' => [
                        'imported' => $imp,
                        'updated'  => $upd,
                        'meta'     => $meta,
                        'skipped'  => $skp,
                        'fetched'  => $fetched,
                    ],
                ]);
            }

            if ($albumFinished) {
                $album_name = self::get_album_name_cached((string)$cat, (string)$album_id);
                // Map source to sync type
                $sync_type_map = [
                    'extension' => 'Extension',
                    'manual-background' => 'Background',
                    'auto' => 'Auto',
                    'manual' => 'Manual',
                    'Manual Sync' => 'Manual',
                ];
                $sync_type = $sync_type_map[$source] ?? 'Manual';
                
                self::add_activity(
                    sprintf('Lightroom → WordPress Sync Complete (%s): "%s"', $sync_type, $album_name),
                    'success',
                    (string)$source
                );
            }


            // Handle Hub sync if target is 'hub'
            try {
                if ($target === 'hub' && function_exists('lsp_hub_enabled') && lsp_hub_enabled()) {
                    error_log("[LSP ajax_batch Hub] Starting Hub accumulation for album $album_id");
                    
                    $touch_key = 'lightsync_hub_touched_' . md5((string)$cat . '|' . (string)$album_id);

                    $this_tick = is_array($out['assets'] ?? null) ? $out['assets'] : [];
                    error_log("[LSP ajax_batch Hub] This tick has " . count($this_tick) . " assets");
                    
                    // Debug: log first asset structure in this tick
                    if (!empty($this_tick)) {
                        $tick_first = $this_tick[0];
                        error_log("[LSP ajax_batch Hub] This tick first asset keys: " . implode(', ', array_keys($tick_first)));
                        error_log("[LSP ajax_batch Hub] This tick first asset rendition_url: " . (isset($tick_first['rendition_url']) ? 'YES (' . strlen($tick_first['rendition_url']) . ')' : 'MISSING'));
                    }

                    $prev = get_option($touch_key, []);
                    if (!is_array($prev)) $prev = [];

                    $merged = [];
                    foreach (array_merge($prev, $this_tick) as $item) {
                        $asset_id = (string)($item['id'] ?? $item['asset']['id'] ?? '');
                        if ($asset_id) {
                            $merged[$asset_id] = $item;
                        }
                    }
                    $all_touched = array_values($merged);
                    
                    update_option($touch_key, $all_touched, false);
                    error_log("[LSP ajax_batch Hub] Total accumulated: " . count($all_touched) . " assets");

                    if ($albumFinished) {
                        error_log("[LSP ajax_batch Hub] Album finished, preparing to push");
                        
                        $album_name = self::get_album_name_cached((string)$cat, (string)$album_id);

                        $touched_data = get_option($touch_key, []);
                        if (!is_array($touched_data)) $touched_data = [];
                        delete_option($touch_key);
                        
                        error_log("[LSP ajax_batch Hub] Got " . count($touched_data) . " assets to push");
                        if (!empty($touched_data)) {
                            $first_asset = $touched_data[0];
                            error_log("[LSP ajax_batch Hub] First asset keys: " . implode(', ', array_keys($first_asset)));
                            error_log("[LSP ajax_batch Hub] First asset has rendition_url: " . (isset($first_asset['rendition_url']) ? 'YES (' . strlen($first_asset['rendition_url']) . ' chars)' : 'NO'));
                        }
                        error_log("[LSP ajax_batch Hub] lsp_hub_push_lightroom_assets exists: " . (function_exists('lsp_hub_push_lightroom_assets') ? 'yes' : 'NO'));
                        
                        $sync_type_map = [
                            'extension' => 'Extension',
                            'manual-background' => 'Background',
                            'auto' => 'Auto',
                            'manual' => 'Manual',
                            'Manual Sync' => 'Manual',
                        ];
                        $sync_type = $sync_type_map[$source] ?? 'Manual';

                        self::add_activity(
                            sprintf('Lightroom → Hub Sync Starting (%s): %d assets for "%s"', $sync_type, count($touched_data), $album_name),
                            'info',
                            (string)$source
                        );

                        if (!empty($touched_data) && function_exists('lsp_hub_push_lightroom_assets')) {
                            // Get selected Hub sites from settings
                            $hub_site_ids = (array) self::get_opt('hub_selected_sites', []);
                            
                            $r = lsp_hub_push_lightroom_assets(
                                $touched_data,
                                (string)$cat,
                                $hub_site_ids,
                                [
                                    'album_id' => (string)$album_id,
                                    'album_name' => $album_name,
                                    'source' => (string)$source,
                                ]
                            );

                            if (empty($r['ok'])) {
                                self::add_activity(
                                    sprintf('Lightroom → Hub Sync Failed (%s): "%s" - %s', $sync_type, $album_name, ($r['error'] ?? 'Unknown error')),
                                    'error',
                                    (string)$source
                                );
                            } else {
                                $hub_synced = (int)($r['synced'] ?? 0);
                                $hub_skipped = (int)($r['skipped'] ?? 0);
                                $hub_failed = (int)($r['failed'] ?? 0);

                                if ($hub_synced > 0) {
                                    self::usage_consume($hub_synced);
                                }

                                // Build appropriate activity message
                                if ($hub_synced > 0 && $hub_skipped > 0) {
                                    $msg = sprintf(
                                        'Lightroom → Hub Sync Complete (%s): "%s" (synced: %d, skipped: %d, failed: %d)',
                                        $sync_type, $album_name, $hub_synced, $hub_skipped, $hub_failed
                                    );
                                    $level = $hub_failed > 0 ? 'warning' : 'success';
                                } elseif ($hub_synced > 0) {
                                    $msg = sprintf(
                                        'Lightroom → Hub Sync Complete (%s): "%s" (synced: %d, failed: %d)',
                                        $sync_type, $album_name, $hub_synced, $hub_failed
                                    );
                                    $level = $hub_failed > 0 ? 'warning' : 'success';
                                } elseif ($hub_skipped > 0) {
                                    $msg = sprintf(
                                        'Lightroom → Hub Skipped (%s): "%s" (unchanged on %d site(s))',
                                        $sync_type, $album_name, $hub_skipped
                                    );
                                    $level = 'info';
                                } else {
                                    $msg = sprintf(
                                        'Lightroom → Hub Sync Complete (%s): "%s" (no changes)',
                                        $sync_type, $album_name
                                    );
                                    $level = 'info';
                                }

                                self::add_activity($msg, $level, (string)$source);
                            }
                        } else {
                            self::add_activity(
                                sprintf('Lightroom → Hub Sync Skipped (%s): no assets for "%s"', $sync_type, $album_name),
                                'info',
                                (string)$source
                            );
                        }
                    }
                }
            } catch (\Throwable $e3) {
                self::add_activity(
                    'Hub sync exception: ' . $e3->getMessage(),
                    'warning',
                    (string)$source
                );
            }

            // =============================================
            // SHOPIFY SYNC - Only if target includes Shopify
            // =============================================
            try {
                $o2 = self::get_opt();
                $shop_domain = (string)($o2['shopify_shop_domain'] ?? '');

                if (
                    in_array($target, ['shopify', 'both'], true) &&
                    $shop_domain !== ''
                ) {
                    // Accumulate assets across batches for Shopify push
                    $shopify_touch_key = 'lightsync_shopify_touched_' . md5((string)$cat . '|' . (string)$album_id);

                    $this_tick = is_array($out['assets'] ?? null) ? $out['assets'] : [];

                    $prev = get_option($shopify_touch_key, []);
                    if (!is_array($prev)) $prev = [];

                    $merged = [];
                    foreach (array_merge($prev, $this_tick) as $item) {
                        $asset_id_key = (string)($item['id'] ?? $item['asset']['id'] ?? '');
                        if ($asset_id_key) {
                            $merged[$asset_id_key] = $item;
                        }
                    }
                    $all_touched = array_values($merged);

                    update_option($shopify_touch_key, $all_touched, false);

                    if ($albumFinished) {
                        $album_name_shopify = self::get_album_name_cached((string)$cat, (string)$album_id);
                        $sync_type_map_shopify = [
                            'extension' => 'Extension',
                            'manual-background' => 'Background',
                            'auto' => 'Auto',
                            'manual' => 'Manual',
                            'Manual Sync' => 'Manual',
                        ];
                        $sync_type_shopify = $sync_type_map_shopify[$source] ?? 'Manual';

                        self::add_activity(
                            sprintf('Lightroom → Shopify Sync Starting (%s): "%s"', $sync_type_shopify, $album_name_shopify),
                            'info',
                            (string)$source
                        );

                        $touched_data = get_option($shopify_touch_key, []);
                        if (!is_array($touched_data)) $touched_data = [];
                        delete_option($shopify_touch_key);

                        if (!empty($touched_data)) {
                            $r = \LightSyncPro\Shopify\Shopify::push_assets_to_files(
                                $touched_data,
                                (string)$cat,
                                $shop_domain,
                                [
                                    'album_id' => (string)$album_id,
                                    'source'   => (string)$source,
                                ]
                            );

                            if (empty($r['ok'])) {
                                self::add_activity(
                                    sprintf('Lightroom → Shopify Sync Failed (%s): "%s" - %s', $sync_type_shopify, $album_name_shopify, ($r['error'] ?? 'Unknown error')),
                                    'warning',
                                    (string)$source
                                );
                            } else {
                                $shopify_uploaded = (int)($r['uploaded'] ?? 0);
                                $shopify_updated = (int)($r['updated'] ?? 0);
                                $shopify_skipped = (int)($r['skipped'] ?? 0);
                                $shopify_failed = (int)($r['failed'] ?? 0);

                                // Count usage for Shopify-only mode (both mode already counted from WP import above)
                                if ($target === 'shopify') {
                                    $shopify_billable = $shopify_uploaded + $shopify_updated;
                                    if ($shopify_billable > 0) {
                                        self::usage_consume($shopify_billable);
                                    }
                                }

                                self::add_activity(
                                    sprintf(
                                        'Lightroom → Shopify Sync Complete (%s): "%s" (new: %d, updated: %d, skipped: %d)',
                                        $sync_type_shopify,
                                        $album_name_shopify,
                                        $shopify_uploaded,
                                        $shopify_updated,
                                        $shopify_skipped
                                    ),
                                    $shopify_failed > 0 ? 'warning' : 'success',
                                    (string)$source
                                );
                            }
                        } else {
                            self::add_activity(
                                sprintf('Lightroom → Shopify Sync Skipped (%s): no assets for "%s"', $sync_type_shopify, $album_name_shopify),
                                'info',
                                (string)$source
                            );
                        }
                    }
                }
            } catch (\Throwable $e4) {
                self::add_activity(
                    'Shopify sync exception: ' . $e4->getMessage(),
                    'warning',
                    (string)$source
                );
            }

        } catch (\Throwable $e) {
            // never break sync UI
        }

        $payload['lightsync_last_sync_ts'] = (int) self::get_opt('lightsync_last_sync_ts', 0);

        wp_send_json_success($payload);
    }

    public function ajax_cron_check(){
        check_ajax_referer(self::AJAX_NS.'_nonce','_ajax_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('forbidden', 403);

        $albums = (array) self::get_opt('album_ids', []);
        $schedule = (array) self::get_opt('album_schedule', []);
        $last_run = (array) self::get_opt('album_last_run', []);
        
        $rows = [];
        $now = time();
        
        foreach ($albums as $aid) {
            $freq = $schedule[$aid] ?? 'off';
            $last = $last_run[$aid] ?? 0;
            
            $interval_sec = 0;
            if ($freq === '15m') $interval_sec = 900;
            elseif ($freq === 'hourly') $interval_sec = 3600;
            elseif ($freq === 'twicedaily') $interval_sec = 43200;
            elseif ($freq === 'daily') $interval_sec = 86400;
            
            $due_now = ($freq !== 'off') && ($last === 0 || ($now - $last >= $interval_sec));
            
            $rows[] = [
                'album' => $aid,
                'freq' => $freq,
                'last_run' => $last ? gmdate('Y-m-d H:i:s', $last) : 'never',
                'due_now' => $due_now,
            ];
        }
        
        wp_send_json_success(['rows' => $rows]);
    }
}